/**
 * @brief Class Backlight - controlls the LCD backlight of a computer.
 *
 * This module contains the class backlight which will define the interface
 * to a LCD backlight controller. Below the module are the backlight drivers
 * which implement the interface to the specific hardware or kernel interface.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/class_backlight.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "systems.h"

#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "class_backlight.h"
#include "driver_backlight_sysfs.h"
#include "driver_backlight_pmu.h"
#include "driver_backlight_x1600.h"
#include "driver_backlight_gma950.h"

/**
 * @brief Private class data of the object
 *
 * The structure holds all private data for objects of this class.
 */
struct class_backlight {
	struct driver_backlight *driver; /**< driver object of the currently active backlight driver */
} modbase_backlight;

/**
 * @brief Initializes the class backlight.
 *
 * This function initializes the class backlight. The main job is to call
 * each known driver one by one and see if one works. The class object
 * will be initialized with the first driver object that returns success.
 *
 * More important and valuable drivers are probed first so that the class
 * walked on a fallback path. If the best driver doesn't work, take the
 * second best and so forth.
 *
 * @return  always zero
 */
int
backlight_init ()
{
	struct class_backlight *base = &modbase_backlight;

	base->driver = NULL;
	if (!base->driver) base->driver = driver_backlight_sysfs_init ();
#ifdef WITH_BACKLIGHT_PMU
	if (!base->driver) base->driver = driver_backlight_pmu_init ();
#endif
#ifdef WITH_BACKLIGHT_PCI
	if (!base->driver) base->driver = driver_backlight_x1600_init ();
	if (!base->driver) base->driver = driver_backlight_gma950_init ();
#endif

	if (base->driver) {
		register_function (QUERYQUEUE,  backlight_query);
		register_function (CONFIGQUEUE, backlight_config);
		print_msg (PBB_INFO, _("Initialized: %s\n"), _(base->driver->name));
	} else
		print_msg (PBB_WARN, _("No backlight driver available - check your Kernel configuration.\n"));

	return 0;
}

/**
 * @brief close the backlight object and frees all ressources
 *
 * This function does all the cleanup after a day of heavy day. In fact it
 * delegates this job to the open backlight driver and calls driver_exit().
 */
void
backlight_exit ()
{
	struct class_backlight *base = &modbase_backlight;

	if (base->driver)
		base->driver->driver_exit();
}

/**
 * @brief  Queries the backlight driver for certain parameters
 *
 * The backlight class provides a generic interface to the backlight
 * drivers. Each driver has several properties that could be read.
 * The caller has to put all properties he wants to know in a taglist
 * and call this function. It will then get all the wanted information
 * from the backlight driver, fill them into the taglist and return
 * it to the caller.
 *
 * @param   taglist  parameters to query from the driver
 * @return  taglist  same taglist with filled in parameter values
 */
void
backlight_query (struct tagitem *taglist)
{
	backlight_handle_tags (MODE_QUERY, taglist);
}

/**
 * @brief  Set certain parameters of the backlight driver to a new value
 *
 * The backlight class provides a generic interface to the backlight
 * drivers. Each driver has several properties that could be written.
 * The caller has to put all properties and their new values in a taglist
 * and call this function. It will then set all given properties in the
 * currently open driver to a new value.
 *
 * @param   taglist  parameters and new values to set
 * @return  taglist  same taglist as called with. They will be unchanged
 *                   unless an error occours. In that case the tag of
 *                   the faulty parameter was marked as error and contains
 *                   an error code.
 */ 
void
backlight_config (struct tagitem *taglist)
{
	backlight_handle_tags (MODE_CONFIG, taglist);
}

/**
 * @brief  Set or read parameters from a backlight driver
 *
 * This function did the work for backlight_config() and backlight_query().
 *
 * @param  taglist  parameterlist to read or write
 * @param  cfgure   mode that decides if the parameters should be read or written
 * @return taglist  depends on mode
 * @see    backlight_config(), backlight_config()
 */
void
backlight_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct class_backlight *base = &modbase_backlight;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_BACKLIGHTDRIVER:
			if (cfgure) tagerror(taglist, E_NOWRITE);
			else        taglist->data = (long) base->driver->name;
			break;
		case TAG_BACKLIGHTMAX:     /* private tag */
			if (cfgure)	;
			else		taglist->data = base->driver->get_brightness_max();
			break;
		case TAG_BACKLIGHTLEVEL:   /* private tag */
			if (cfgure)	base->driver->set_brightness(taglist->data);
			else		taglist->data = base->driver->get_brightness();
			break;
		}
		taglist++;
	}
}

