!{\src2tex{textfont=tt}}
!!****f* ABINIT/wrpawps
!! NAME
!! wrpawps
!! 
!! FUNCTION
!! Write a PAW pseudopotential file formatted for Abinit
!!
!! COPYRIGHT
!! Copyright (C) 1998-2005 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  filename= output file name for Abinit
!!  funit= output unit number
!!  pawarray
!!    %gnorm(l_size)= Normalisation of qijl for each l
!!    %shapefunc(sph_meshsz)= Normalized shape function
!!  pawdata
!!    %rad(big_meshsz)= Coordinates of the radial grid
!!  pawps
!!    %coredens(core_meshsz)= Core density of the atom
!!    %tcoredens(core_meshsz)= Pseudo-core density of the atom
!!    %dij0(lmn2_size)= Part of the Dij term calculated in the psp part
!!    %phi(wav_meshsz,basis_size)= atomic partial waves
!!    %tphi(wav_meshsz,basis_size)= atomic pseudo partial waves
!!    %tproj(wav_meshsz,basis_size)= projectors on partial waves
!!    %rhoij0= Atomic initialization of rhoij
!!    %vhtnzc(rho_meshsz)= Hartree potential of the ps-density
!!                           of the nucleus + core electrons
!!  pshead
!!    %atomic_charge= Total atomic charge
!!    %basis_size= Number of elements for the paw nl basis
!!    %core_meshsz= Dimension of radial mesh for core and tcore densities
!!    %creatorid= ID of psp generator (here creatorID=1 !)
!!    %l_size= Max. value of l+1 leading to a non zero Gaunt coeffs
!!    %lambda= Lambda in gaussian type g(r) (not used here)
!!    %lmax= Maximum value of l
!!    %lmn_size= Number of elements for the paw basis
!!    %log_step= Logarithmic step corresponding to radial mesh
!!    %mesh_type=  Flag defining ther radial grid type
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %prj_meshsz= Dimension of radial mesh for tProj
!!    %pspcod= Psp code number for Abinit (here PAW->pspcod=7 !)
!!    %pspxc_abinit= Abinit s code number for the exchange-correlation
!!    %rad_step= Radial step corresponding to radial mesh
!!    %rc_sph= Default PAW sphere radius
!!    %shape_type= Shape function type (necessarily 2 here)
!!    %sigma= Sigma for gaussian type g(r) (not used here)
!!    %sph_meshsz= Dimension of radial mesh for spheres
!!    %title= Title for pseudopotential
!!    %valence_charge= Valence charge
!!    %vloc_meshsz= Dimension of radial mesh for vloc=vhtnzc
!!  un_log= Unit number for log file (comments)
!!
!! NOTES
!!  File format of formatted PAW psp input for Abinit:
!!  --------------------------------------------------
!!  (1) title (character) line
!!  (2) znucl, zion, pspdat
!!  (3) pspcod, pspxc, lmax, lloc, mmax, r2well
!!  (4) basis_size, lmn_size
!!  (5) orbitals (for l=1 to basis_size)
!!  (6) mesh_type, rad_step[, log_step]
!!  (7) r_cut(SPH), r_cut(PROJ)
!!  (8) mesh_size(SPH), mesh_size(PROJ), mesh_size(VLOC)
!!  (9) shape_type, rcomp
!!  (10) pspfmt,creatorID  (here creatorID=1)
!!  For iln=1 to basis_size
!!      (11)  comment(character)
!!      (12) phi(r) (for ir=1 to mesh_size(SPH))
!!  For iln=1 to basis_size
!!      (13) comment(character)
!!      (14) tphi(r) (for ir=1 to mesh_size(SPH))
!!  For iln=1 to basis_size
!!      (15) comment(character)
!!      (16) tproj(r) (for ir=1 to mesh_size(PROJ))
!!  (17) comment(character)
!!  (18) core_density (for ir=1 to mesh_size(SPH))
!!  (19) comment(character)
!!  (20) tcore_density (for ir=1 to mesh_size(SPH))
!!  (21) comment(character)
!!  (22) Dij0 (for ij=1 to lmn_size*(lmn_size+1)/2)
!!  (23) comment(character)
!!  (24) Rhoij0 (for ij=1 to lmn_size*(lmn_size+1)/2)
!!  (25) comment(character)
!!  (26) VHntzc(r) (=Vloc) (for ir=1 to mesh_size(VLOC))
!  ===== Following lines only if shape_type=-1 =====
!  For il=1 to 2*max(orbitals)+1
!      (33) comment(character)
!      (34) radial mesh index for shapefunc
!      (35) shapefunc(r)*gnorm(l)*r**l (for ir=1 to phi_meshsz)
!!  --------------------------------------------------
!!
!! PARENTS
!!      uspp2abinit
!!
!! CHILDREN
!!      date_and_time
!!
!! SOURCE

 subroutine wrpawps(fname,funit,pawarray,pawdata,pawps,pshead,un_log)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: funit,un_log
 character*(fnlen) :: fname
!These types are defined in defs_pawps
 type(pawarray_type) :: pawarray
 type(pawdata_type)  :: pawdata
 type(pawps_type)   :: pawps
 type(pshead_type)  :: pshead

!Local variables ---------------------------------------
 integer :: ib,icormsh,iprjmsh,ii,il,ilmn,ir,jlmn
 character*8 :: strdate
 real(dp),allocatable :: ff(:)

! *************************************************************************

!If (t)coredens use a large mesh, adjust flag icormsh
 icormsh=0;if (pshead%core_meshsz/=pshead%sph_meshsz) icormsh=1
!If tproj use a large mesh, adjust flag iprjmsh
 iprjmsh=0;if (pshead%prj_meshsz/=pshead%sph_meshsz) iprjmsh=1

!Open the file for writing
 open(unit=funit,file=trim(fname),form='formatted',status='unknown')

!Write the header
 call date_and_time(strdate)
 write(funit,'(a)') trim(pshead%title)
 write(funit,'(1x,f7.3,1x,f7.3,1x,a,14x,a)') &
&      pshead%atomic_charge,pshead%valence_charge,&
&      trim(strdate),&
&      " : zatom,zion,pspdat"
 write (funit,'(3(1x,i2)," 0 ",i5," 0.",19x,a)') &
&      pshead%pspcod,&
&      pshead%pspxc_abinit,&
&      pshead%lmax,&
&      pshead%sph_meshsz,&
&      " : pspcod,pspxc,lmax,lloc,mmax,r2well"
 write (funit,'(1x,"paw3",1x,i2,31x,a)') &
&       pshead%creatorid,&
&       " : pspfmt,creatorID"
 write (funit,'(2(1x,i2),33x,a)') &
&       pshead%basis_size,&
&       pshead%lmn_size,&
&       " : basis_size,lmn_size"
 do ib=1,pshead%basis_size
  write (funit,'(1x,i1)',ADVANCE='NO') pshead%orbitals(ib)
 enddo
 if (pshead%basis_size<20) then
  do ib=pshead%basis_size+1,20
   write (funit,'(a)',ADVANCE='NO') '  '
  enddo
 endif
 write (funit,'(a)') ": orbitals"
 write (funit,'(1x,i1,37x,a)') &
&       2+icormsh+iprjmsh," : number_of_meshes"
 write (funit,'(1x,i1,1x,i1,1x,i4,1x,es16.10,1x,es16.10,a)') &
&       1,2,pshead%sph_meshsz,pshead%rad_step,pshead%log_step,&
&          " : mesh 1, type,size,rad_step[,log_step]"
 if (iprjmsh==1) &
& write (funit,'(1x,i1,1x,i1,1x,i4,1x,es16.10,1x,es16.10,a)') &
&        2,2,pshead%prj_meshsz,pshead%rad_step,pshead%log_step,&
&           " : mesh 2, type,size,rad_step[,log_step]"
 if (icormsh==1) &
& write (funit,'(1x,i1,1x,i1,1x,i4,1x,es16.10,1x,es16.10,a,i1,a)') &
&        2+iprjmsh,2,pshead%core_meshsz,pshead%rad_step,pshead%log_step,&
&           " : mesh ",2+iprjmsh,", type,size,rad_step[,log_step]"
 write (funit,'(1x,i1,1x,i1,1x,i4,1x,es16.10,1x,es16.10,a,i1,a)') &
&       2+iprjmsh+icormsh,2,pshead%vloc_meshsz,pshead%rad_step,pshead%log_step,&
&          " : mesh ",2+iprjmsh+icormsh,", type,size,rad_step[,log_step]"
 write (funit,'(1x,f13.10,25x,a)') &
&       pshead%rc_sph,&
&       " : r_cut(SPH)"
 if (abs(pshead%rcomp-pshead%rc_sph)<1.d-10) then
  write (funit,'(i2,a,34x,a)') &
&        pshead%shape_type," 0.",&
&        " : shape_type(SIN),rshape"
 else
  write (funit,'(i2,1x,f13.10,23x,a)') &
&        pshead%shape_type,pshead%rcomp,&
&        " : shape_type(SIN),rshape"
 end if

!Write partial waves and pseudo partial waves
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== PHI ",ib,&
&      " ===== #phi(r), for phi(r)/r*Ylm)"
  write(funit,'(a)') " 1  : radial mesh index"
  write(funit,'(3(1x,es23.16))') &
&      (pawps%phi(ir,ib),ir=1,pshead%sph_meshsz)
 enddo
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== TPHI ",ib,&
&      " ===== #tphi(r), for tphi(r)/r*Ylm)"
  write(funit,'(a)') " 1  : radial mesh index"
  write(funit,'(3(1x,es23.16))') &
&      (pawps%tphi(ir,ib),ir=1,pshead%sph_meshsz)
 enddo

!Write projectors
 do ib=1,pshead%basis_size
  write(funit,'(a,i1,a)') "===== TPROJECTOR ",ib,&
&      " ===== #p(r), for p(r)/r*Ylm) )"
 if (iprjmsh==1) then
  write(funit,'(a)') " 2  : radial mesh index"
 else
  write(funit,'(a)') " 1  : radial mesh index"
 endif
  write(funit,'(3(1x,es23.16))') &
&      (pawps%tproj(ir,ib),ir=1,pshead%prj_meshsz)
 enddo

!Write the core density and the pseudo core density
 write(funit,'(a)') "===== CORE_DENSITY ====="
 if (icormsh==1) then
  write(funit,'(a,i1,a)') " ",2+iprjmsh,"  : radial mesh index"
 else
  write(funit,'(a)') " 1  : radial mesh index"
 endif
 write(funit,'(3(1x,es23.16))') &
&     (pawps%coredens(ir),ir=1,pshead%core_meshsz)
 write(funit,'(a)') "===== TCORE_DENSITY ====="
 if (icormsh==1) then
  write(funit,'(a,i1,a)') " ",2+iprjmsh,"  : radial mesh index"
 else
  write(funit,'(a)') " 1  : radial mesh index"
 endif
 write(funit,'(3(1x,es23.16))') &
&     (pawps%tcoredens(ir),ir=1,pshead%core_meshsz)

!Write Dij0 and Rhoij0
 write(funit,'(a)') "===== Dij0 ====="
 ii=0
 do jlmn=1,pshead%lmn_size
  write(funit,'(100(1x,es23.16))') &
&     (pawps%dij0(ii+ilmn),ilmn=1,jlmn)
  ii=ii+jlmn
 enddo
 write(funit,'(a)') "===== Rhoij0 ====="
 ii=0
 do jlmn=1,pshead%lmn_size
  write(funit,'(100(1x,es23.16))') &
&     (pawps%rhoij0(ii+ilmn),ilmn=1,jlmn)
  ii=ii+jlmn
 enddo

!Write VHntZC=Vloc
 write(funit,'(a)') "===== VHntZC (Vloc(r)) ====="
 write(funit,'(a,i1,a)') " ",2+iprjmsh+icormsh,"  : radial mesh index"
 write(funit,'(3(1x,es23.16))') &
&     (pawps%vhtnzc(ir),ir=1,pshead%vloc_meshsz)

!Write (eventually) shape functions
 if (pshead%shape_type==-1) then
  allocate(ff(pshead%sph_meshsz))
  do il=1,pshead%l_size
   write(funit,'(a,i1,a)') "===== SHAPEF (l=",il-1,") ====="
   write(funit,'(a)') " 1  : radial mesh index"
   do ir=1,pshead%sph_meshsz
    ff(ir)=pawarray%shapefunc(ir)*pawarray%gnorm(il)*pawdata%rad(ir)**(il-1)
   enddo
   write(funit,'(3(1x,es23.16))') (ff(ir),ir=1,pshead%sph_meshsz)
  enddo
  deallocate(ff)
 endif

!Close the file
 close(funit)

 end subroutine
!!***

