// ---------------------------------------------------------------------------
// - TlsAlert.cpp                                                            -
// - afnix:tls service - tls alert message class implementation              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Integer.hpp"
#include "TlsAlert.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the alert level code
  static const t_byte TLS_ALRT_WRN = 0x01U; // warning
  static const t_byte TLS_ALRT_FTL = 0x02U; // fatal
  // the alert description code
  static const t_byte TLS_ALRT_CLS = 0x00U; // close notify
  static const t_byte TLS_ALRT_UMS = 0x0AU; // unexpected message
  static const t_byte TLS_ALRT_BRM = 0x14U; // bad record mac
  static const t_byte TLS_ALRT_CFL = 0x15U; // decryption failed
  static const t_byte TLS_ALRT_ROF = 0x16U; // record overflow
  static const t_byte TLS_ALRT_DFL = 0x1EU; // decompression failure
  static const t_byte TLS_ALRT_HFL = 0x28U; // handshake failure
  static const t_byte TLS_ALRT_BCT = 0x2AU; // bad certificate
  static const t_byte TLS_ALRT_NCT = 0x2BU; // unsupported certificate
  static const t_byte TLS_ALRT_RCT = 0x2CU; // certificate revoked
  static const t_byte TLS_ALRT_ECT = 0x2DU; // certificate expired
  static const t_byte TLS_ALRT_UCT = 0x2EU; // certificate unknown
  static const t_byte TLS_ALRT_PRM = 0x2FU; // illegal parameter
  static const t_byte TLS_ALRT_UCA = 0x30U; // unknown ca
  static const t_byte TLS_ALRT_DND = 0x31U; // access denied
  static const t_byte TLS_ALRT_DEC = 0x32U; // decode error
  static const t_byte TLS_ALRT_DCY = 0x33U; // decrypt error
  static const t_byte TLS_ALRT_EXP = 0x3CU; // export restriction
  static const t_byte TLS_ALRT_PVR = 0x46U; // protocol version
  static const t_byte TLS_ALRT_SEC = 0x47U; // insufficient security
  static const t_byte TLS_ALRT_ERR = 0x50U; // internal error
  static const t_byte TLS_ALRT_USR = 0x5AU; // user canceled
  static const t_byte TLS_ALRT_NRG = 0x64U; // no renegotiation
   
  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty alert - fatal close notify

  TlsAlert::TlsAlert (void) {
    d_levl = TLS_LEVL_FTL;
    d_desc = TLS_DESC_CLS;
  }

  // create a message by record

  TlsAlert::TlsAlert (TlsRecord* rcd) : TlsMessage (rcd) {
    d_levl = TLS_LEVL_FTL;
    d_desc = TLS_DESC_CLS;
    decode ();
  }

  // copy construct this alert

  TlsAlert::TlsAlert (const TlsAlert& that) {
    that.rdlock ();
    try {
      // copy the message
      TlsMessage::operator = (that);
      // copy localy
      d_levl = that.d_levl;
      d_desc = that.d_desc;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // assign a alert to this one

  TlsAlert& TlsAlert::operator = (const TlsAlert& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // copy the message
      TlsMessage::operator = (that);
      // copy localy
      d_levl = that.d_levl;
      d_desc = that.d_desc;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // get the class name

  String TlsAlert::repr (void) const {
    return "TlsAlert";
  }

  // clone this object

  Object* TlsAlert::clone (void) const {
    return new TlsAlert (*this);
  }

  // reset the alert

  void TlsAlert::reset (void) {
    wrlock ();
    try {
      // reset base message
      TlsMessage::reset ();
      // reset locally
      d_levl = TLS_LEVL_FTL;
      d_desc = TLS_DESC_CLS;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the message info as a plist

  Plist TlsAlert::getinfo (void) const {
    rdlock ();
    try {
      // create a header result plist
      Plist plst = TlsHeader::getinfo ();
      // set the message info
      plst.add ("TLS-MSG-TYPE", "TLS ALERT", (t_long) gettype  ());
      // here it is
      unlock ();
      return plst;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // decode a tls alert message

  void TlsAlert::decode (void) {
    wrlock ();
    try {
      // get the alert level
      t_byte levl = getbyte (0);
      switch (levl) {
      case TLS_ALRT_WRN:
	d_levl = TLS_LEVL_WRN;
	break;
      case TLS_ALRT_FTL:
	d_levl = TLS_LEVL_FTL;
	break;
      default:
	throw Exception ("tls-error", "invalid tls level during decode");
	break;
      }
      // get the alert description
      t_byte desc = getbyte (1);
      switch (desc) {
      case TLS_ALRT_CLS:
	d_desc = TLS_DESC_CLS;
	break;
      case TLS_ALRT_UMS:
	d_desc = TLS_DESC_UMS;
	break;
      case TLS_ALRT_BRM:
	d_desc = TLS_DESC_BRM;
	break;
      case TLS_ALRT_CFL:
	d_desc = TLS_DESC_CFL;
	break;
      case TLS_ALRT_ROF:
	d_desc = TLS_DESC_ROF;
	break;
      case TLS_ALRT_DFL:
	d_desc = TLS_DESC_DFL;
	break;
      case TLS_ALRT_HFL:
	d_desc = TLS_DESC_HFL;
	break;
      case TLS_ALRT_BCT:
	d_desc = TLS_DESC_BCT;
	break;
      case TLS_ALRT_NCT:
	d_desc = TLS_DESC_NCT;
	break;
      case TLS_ALRT_RCT:
	d_desc = TLS_DESC_RCT;
	break;
      case TLS_ALRT_ECT:
	d_desc = TLS_DESC_ECT;
	break;
      case TLS_ALRT_UCT:
	d_desc = TLS_DESC_UCT;
	break;
      case TLS_ALRT_PRM:
	d_desc = TLS_DESC_PRM;
	break;
      case TLS_ALRT_UCA:
	d_desc = TLS_DESC_UCA;
	break;
      case TLS_ALRT_DND:
	d_desc = TLS_DESC_DND;
	break;
      case TLS_ALRT_DEC:
	d_desc = TLS_DESC_DEC;
	break;
      case TLS_ALRT_DCY:
	d_desc = TLS_DESC_DCY;
	break;
      case TLS_ALRT_EXP:
	d_desc = TLS_DESC_EXP;
	break;
      case TLS_ALRT_PVR:
	d_desc = TLS_DESC_PVR;
	break;
      case TLS_ALRT_SEC:
	d_desc = TLS_DESC_SEC;
	break;
      case TLS_ALRT_ERR:
	d_desc = TLS_DESC_ERR;
	break;
      case TLS_ALRT_USR:
	d_desc = TLS_DESC_USR;
	break;
      case TLS_ALRT_NRG:
	d_desc = TLS_DESC_NRG;
	break;
      default:
	throw Exception ("tls-error", "invalid tls description during decode");
	break;
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // create a new object in a generic way

  Object* TlsAlert::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new TlsAlert;
    // check for 1 argument
    if (argc == 1) {
      Object*    obj = argv->get (0);
      TlsRecord* rcd = dynamic_cast<TlsRecord*> (obj);
      if ((rcd == nilp) && (obj != nilp)) {
	throw Exception ("type-error", "invalid object as tls record",
			 Object::repr (obj));
      }
      return new TlsAlert (rcd);
    }
    // too many arguments
    throw Exception ("argument-error", 
                     "too many argument with tls alert constructor");
  }
}
