// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ACSliceInfo_h__
#define __ACSliceInfo_h__

/** \file 
 *  Semantic information about an AspectC++ 'slice'. */

#include "Puma/Array.h"
#include "Puma/CObjectInfo.h"

namespace Puma {


class ACAspectInfo;
class CT_ClassSliceDecl;
class CT_Intro;
class Unit;


/** \class ACSliceInfo ACSliceInfo.h Puma/ACSliceInfo.h
 *  Semantic information about an AspectC++ 'slice' declaration. 
 *  A slice represents a fragment of a C/C++ language element. For
 *  example a 'class slice' is a (possibly incomplete) fragment
 *  of a class. */
class ACSliceInfo {
  CObjectInfo       *_object;
  CT_ClassSliceDecl *_def_node;
  Unit              *_pattern;        // inline member patterns (in 1 unit)
  Unit              *_base_pattern;   // base class intro part
  Array<CT_Intro*>  _members;         // non-inline slice members
  Array<Unit*>      _member_patterns; // non-inline member patterns
  Array<Unit*>      _member_units;    // non-inline member units (ah files)
  bool              _in_advice;
      
public:
  /** Various types of slices */
  enum Type { SL_UNKNOWN, SL_ASPECT, SL_CLASS, SL_STRUCT, SL_UNION };
  
  /** Construct a slice info object.
   *  \param oi The object info of the slice declaration, e.g. class info. */
  ACSliceInfo (CObjectInfo *oi) :
    _object (oi), _def_node ((CT_ClassSliceDecl *)0),
    _pattern (0), _base_pattern (0) {}

  /** Desctructor: delete all pattern units */
  ~ACSliceInfo ();
  
  /** Get the name of the slice. */
  const char *name () { return _object->Name (); }
  /** Get the object info of the slice declaration. */
  CObjectInfo *object () { return _object; }

  /** Set the slice definition syntax tree node.
   *  \param node The slice syntax tree node. */
  void def_node (CT_ClassSliceDecl *node);
  /** Get the syntax tree node of the slice definition. */
  CT_ClassSliceDecl *def_node () { return _def_node; }
  
  /** Add a non-inline slice member.
   *  \param i The new member. */
  void add_member (CT_Intro *i);
  
  /** Get the number of non-inline slice members. */
  int members () const { return _members.length (); }
  
  /** Get the ith non-inline slice member. */
  CT_Intro *member (int i) const { return _members.lookup (i); }
  
  /** Get the ith non-inline slice member code pattern. */
  const Unit *member_pattern (int i) const { return _member_patterns.lookup (i); }
  
  /** Get the ith non-inline slice member unit. */
  const Unit *member_unit (int i) const { return _member_units.lookup (i); }
  
  /** Set a flag to determine if this definition is part of an advice decl
   *  \param a true, iff this definition is location within 'advice .. : ..'. */
  void in_advice (bool a) { _in_advice = a; }
  
  /** true, iff this slice decl is located within 'advice .. : ..'. */
  bool in_advice () const { return _in_advice; }
  
  /** Get the slice type (e.g. class or union) */
  Type type () const;
  
  /** Get the definition of this slice or return 0 if there is no definition */
  ACSliceInfo *definition () const;
  
  /** Get the base class pattern of this slice as a unit */
  const Unit *base_pattern () const { return _base_pattern; }

  /** Get the inline member pattern of this slice as a unit */
  const Unit *pattern () const { return _pattern; }
};


} // namespace Puma

#endif /* __ACSliceInfo_h__ */
