// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __VoidPtrArray__
#define __VoidPtrArray__

#include <stdlib.h>
#include <string.h>
#include <assert.h>

namespace Puma {

class VoidPtrArray {
protected:
  static const long default_init_size;
  static const long default_increment;

protected:
  VoidPtrArray (long is = default_init_size, long incr = default_increment);
  VoidPtrArray (const VoidPtrArray &array);
  VoidPtrArray &operator =(const VoidPtrArray &array);
  ~VoidPtrArray ();

  void append (const void *item);
  void insert (long index, const void *item);
  void prepend (const void *item);
  void remove (long index);
  void reset ();
  void *&get (long index);
  void *&operator[] (long index);
  void *fetch (long index) const;
  void *&lookup (long index) const;
  long length () const;

private:
  void** data;
  long size;
  long count;
  long increment;
      
  void grow (long wanted);
};


inline VoidPtrArray::VoidPtrArray (long is, long incr) {
  count     = 0; 
  size      = 0;
  increment = incr;
  data      = 0;
}


inline VoidPtrArray::VoidPtrArray (const VoidPtrArray &array) {
  count     = array.count;
  size      = array.size;
  increment = array.increment;
  
  if (size) {
    data = (void**) ::malloc(size * sizeof(void*));
    ::memcpy(data, array.data, count * sizeof(void*));
  } else
    data = 0;
}


inline VoidPtrArray &VoidPtrArray::operator =(const VoidPtrArray &array) {
  if (data)
    ::free (data);
 
  count     = array.count;
  size      = array.size;
  increment = array.increment;

  if (size) {
    data = (void**) ::malloc(size * sizeof(void*));
    ::memcpy(data, array.data, count * sizeof(void*));
  } else
    data = 0;
  
  return *this;
}


inline VoidPtrArray::~VoidPtrArray () {
  if (data)
    ::free (data);
}


inline void VoidPtrArray::append (const void *item) {
  if (count >= size) {
    grow (count);
  }
  data[count++] = (void*)item;
}


inline void VoidPtrArray::prepend (const void *item) {
  insert (0, item);
}


inline void VoidPtrArray::insert (long index, const void *item) {
  if (count >= size) {
    grow (count);
  }
  for (int pos = count; pos > index; pos--)
    data[pos] = data[pos - 1];
  data[index] = (void*)item;
  count++;
}


inline void *&VoidPtrArray::get (long index) {
  if (index >= size) {
    grow (index);
  }
  if (index >= count)
    count = index + 1;
  return data[index];
}


inline void *&VoidPtrArray::operator[] (long index) {
  return get (index);
}


inline void *VoidPtrArray::fetch (long index) const {
  return data[index];
}


inline long VoidPtrArray::length () const {
  return count;
}


inline void VoidPtrArray::grow (long wanted) {
  do {
    size += increment;
    increment *= 2;
  } while (wanted >= size);

  if (data)
    data = (void**) ::realloc(data, size * sizeof(void*));
  else
    data = (void**) ::malloc(size * sizeof(void*));
}
   

inline void VoidPtrArray::remove (long index) {
  if (index < count && count > 0) {
    for (int pos = index; pos < count - 1; pos++)
      data[pos] = data[pos + 1];
    count--;
  }
}


inline void VoidPtrArray::reset () {
  count = 0; 
  if (data)
    ::free (data);
  size = 0;
  data = 0;
  increment = default_increment;
}


inline void *&VoidPtrArray::lookup (long index) const {
  assert(index >= 0 && index < count);
  return data[index];
}


} // namespace Puma

#endif /* __VoidPtrArray__ */
