/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

//BL_COPYRIGHT_NOTICE
#include <winstd.H>

#include <new>
#include <iostream>
using std::cout;
using std::cerr;
using std::endl;
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <list>
#include <map>
#include <algorithm>

#ifndef WIN32
#include <unistd.h>
#endif

#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "FArrayBox.H"
#include "Utility.H"
#include "ArrayLim.H"


static
void 
print_usage (int,
             char* argv[])
{
    cerr << "usage:\n";
    cerr << argv[0] << " infile [options] \n\tOptions:\n";
    exit(1);
}

class IVLess
{
public:
    bool operator () (const IntVect& lhs,
                      const IntVect& rhs) const
        {
            return lhs.lexLT(rhs);
        }
};


#if defined(BL_FORT_USE_UPPERCASE)
#  define PROCESS  PROCESS 
#  define BILIN    BILIN
#  define HACKD    HACKD
#elif defined(BL_FORT_USE_LOWERCASE)
#  define PROCESS  process 
#  define BILIN    bilin
#  define HACKD    hackd
#elif defined(BL_FORT_USE_UNDERSCORE)
#  define PROCESS  process_ 
#  define BILIN    bilin_
#  define HACKD    hackd_
#endif

extern "C" {

    void PROCESS(const int* lo, const int* hi, const int* dlo,
                 int*       tf, ARLIM_P(tf_lo), ARLIM_P(tf_hi),
                 const Real* T,  ARLIM_P(T_lo), ARLIM_P(T_hi),
                 int* listi, int* listj, int* listk, int* length,
                 const Real* delta, Real* area, const Real* sVal,
                 const int* imeth);

    void BILIN(const Real* src, ARLIM_P(slo), ARLIM_P(shi),
               Real*       dst, ARLIM_P(dlo), ARLIM_P(dhi), const int* r);

    void HACKD(Real* dat, ARLIM_P(dlo), ARLIM_P(dhi),
               const Real* dx, const Real* problo);
};

void
main (int   argc,
      char* argv[])
{
    BoxLib::Initialize(argc,argv);

    if (argc < 2)
        print_usage(argc,argv);

    ParmParse pp;

    if (pp.contains("help"))
        print_usage(argc,argv);

    bool verbose = false;
    if (pp.contains("verbose"))
    {
        verbose = true;
        AmrData::SetVerbose(verbose);
    }

    std::string infile;
    pp.get("infile",infile);
    
    int writeFlame=0; pp.query("writeFlame",writeFlame);
    int imeth=0; pp.query("imeth",imeth);

    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    DataServices dataServices(infile, fileType);

    if (!dataServices.AmrDataOk())
        //
        // This calls ParallelDescriptor::EndParallel() and exit()
        //
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
    
    AmrData& amrData = dataServices.AmrDataRef();
    
    const int finestLevel = amrData.FinestLevel();
    const int Nlev = finestLevel + 1;
    
    Array<std::string> plotnames = amrData.PlotVarNames();
    int idT = -1;
    for (int i=0; i<plotnames.size(); ++i)
    {
        if (plotnames[i] == "temp") idT = i;
    }
    const int nComp = 1;

    Real yrange=.015; pp.query("yrange",yrange);
    Real zrange=.075; pp.query("zrange",zrange);

    Array<size_t> fragCnt(Nlev,0);
    Array<std::set<IntVect, IVLess> > levelIVset(Nlev);
    Array<Real> Area(Nlev,0.0);

    Real sVal=1090.; pp.query("sVal",sVal);

    Real areaCheck = 0.;

    //for (int lev=0; lev<Nlev; ++lev)
    for (int lev=0; lev<1; ++lev)
    {
        const Box& dBox = amrData.ProbDomain()[lev];

        // Get state
        if (verbose)
            cout << "...reading state data at lev=" << lev << "..." << endl;

        const BoxArray& ba = amrData.boxArray(lev);
        Array<FArrayBox *> state(ba.size());
        Array<Box> boxes(ba.size());
        for (int i=0; i<ba.size(); ++i)
        {
            state.set(i,new FArrayBox(Box(ba[i]).grow(1),nComp));
            boxes.set(i,state[i]->box());
        }
        amrData.FillVar(state,boxes,lev,"temp",ParallelDescriptor::IOProcessorNumber());

        std::set<IntVect, IVLess>& ivSet = levelIVset[lev];
        if (verbose)
            cout << "...computing flame surface area at lev=" << lev << "..." << endl;

        const Array<Real>& dx = amrData.DxLevel()[lev];

        const int jmid = int(0.5*(dBox.hiVect()[1]-dBox.loVect()[1]+1) - 1);
        const int jhi = jmid + int(yrange/dx[1]);
        const int jlo = jmid - int(yrange/dx[1]);
        const int klo = dBox.loVect()[2];
        const int khi = dBox.loVect()[2] + int(zrange/dx[2]);
        IntVect ivlo = dBox.smallEnd(); ivlo[1] = jlo; ivlo[2] = klo;
        IntVect ivhi = dBox.bigEnd();   ivhi[1] = jhi; ivhi[2] = khi;
        Box subBox(ivlo,ivhi);

        if (verbose)
            cout << "...computing area over box = " << subBox
                 << " using " << (imeth==0 ? "jbb's" : "jfg's" ) << " method" << endl;

        for (int i=0; i<state.size(); ++i)
        {
            //const Box& box = ba[i] & subBox;
            //const Box& box = (lev==finestLevel ? ba[i] & subBox : ba[i]);
            const Box& box = ba[i];

            if (box.ok())
            {
                BaseFab<int> tf(BoxLib::grow(box,1),1);
                size_t bigLen = tf.box().numPts();
                Array<int> myListI(bigLen);
                Array<int> myListJ(bigLen);
                Array<int> myListK(bigLen);
                int myLen = 0;
                Real myArea = 0.0;

                //if (lev==0)
                    HACKD(state[i]->dataPtr(),
                          ARLIM(state[i]->loVect()),ARLIM(state[i]->hiVect()),
                          dx.dataPtr(),amrData.ProbLo().dataPtr());

                PROCESS(box.loVect(),box.hiVect(),dBox.loVect(),
                        tf.dataPtr(),
                        ARLIM(tf.loVect()),ARLIM(tf.hiVect()),
                        state[i]->dataPtr(0),
                        ARLIM(state[i]->loVect()),ARLIM(state[i]->hiVect()),
                        myListI.dataPtr(),
                        myListJ.dataPtr(),
                        myListK.dataPtr(),&myLen,dx.dataPtr(),&myArea,&sVal,&imeth);
                Area[lev] += myArea;

                // Idiot check...better not have counted any of these before
                for (int n=0; n<myLen; ++n)
                {
                    IntVect iv(myListI[n],myListJ[n],myListK[n]);
                    if (ivSet.count(iv)==0)
                        ivSet.insert(iv);
                    else
                        BoxLib::Error("Double entry");
                }
                fragCnt[lev] += myLen;

                // Double check by refining and doing again
                //if (lev == finestLevel)
                if (lev == 0)
                {
                    const int r = 2;
                    const Box fBox = Box(state[i]->box()).refine(r);
                    FArrayBox fdat(fBox,1);
                    const FArrayBox& cdat = *(state[i]);

                    BILIN(cdat.dataPtr(),ARLIM(cdat.loVect()),ARLIM(cdat.hiVect()),
                          fdat.dataPtr(),ARLIM(fdat.loVect()),ARLIM(fdat.hiVect()),&r);

                    const Box rBox = Box(ba[i]).refine(r).grow(1);

                    BaseFab<int> tf(BoxLib::grow(rBox,1),1);
                    size_t bigLen = tf.box().numPts();
                    Array<int> myListI(bigLen);
                    Array<int> myListJ(bigLen);
                    Array<int> myListK(bigLen);
                    int myLen = 0;
                    Real myArea = 0.0;
                    const Box fdBox = Box(dBox).refine(r);
                    Array<Real> fdx(BL_SPACEDIM);
                    for (int d=0; d<BL_SPACEDIM; ++d)
                        fdx[d] = dx[d]/r;

                    PROCESS(rBox.loVect(),rBox.hiVect(),fdBox.loVect(),
                            tf.dataPtr(),
                            ARLIM(tf.loVect()),ARLIM(tf.hiVect()),
                            fdat.dataPtr(0),
                            ARLIM(fdat.loVect()),ARLIM(fdat.hiVect()),
                            myListI.dataPtr(),
                            myListJ.dataPtr(),
                            myListK.dataPtr(),&myLen,fdx.dataPtr(),&myArea,&sVal,&imeth);
                    
                    areaCheck += myArea;
                }
            }
        }
        
        //if (lev==finestLevel && writeFlame==1)
        if (lev==0 && writeFlame==1)
        {
            BoxArray ba(ivSet.size());
            int cnt=0;
            for (std::set<IntVect, IVLess>::const_iterator it =  ivSet.begin();
                 it != ivSet.end() ; ++it)
            {
                ba.set(cnt++,Box(*it,*it));
            }
            std::string gFile("flameGrids");
            if (verbose)
                std::cout << "Writing grids to " << gFile << std::endl;
            std::ofstream osf(gFile.c_str(),std::ios::out);
            osf << "1" << endl;
            ba.writeOn(osf);
            osf.close();
        }
        if (verbose)
            cout << " ...................done" << endl;

    }
    
    cout << "Check area: " << areaCheck << endl;

    if (verbose)
    {
        for (int lev=0; lev<Nlev; ++lev)
            cout << "Level " << lev << ": Time, Area = "
                 << amrData.Time() << ",  " << Area[lev]
                 << " based on " << fragCnt[lev]
                 << " surface fragments" << endl;
    }
    else
    {
        cout << "File, Time, Area = " << infile << ", "
             << amrData.Time() << ",  " << Area[finestLevel] << endl;
    }

    BoxLib::Finalize();
}
