/*
 * main.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy$
 * $LastChangedDate$
 * $LastChangedRevision$
 */

#include <config.h>

#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <unistd.h>

#include "download.h"
#include "execute.h"
#include "frontend.h"
#include "install.h"
#include "log.h"
#include "package.h"
#include "suite.h"
#include "target.h"

static const char *helperdir;
static install_simulate_handler *simulate_handler;

static void install_execute_progress_update (di_packages *packages, char *package, di_package_status status)
{
  di_package *p = di_packages_get_package (packages, package, 0);
  if (p && p->status < status)
  {
    log_text (DI_LOG_LEVEL_DEBUG, "Updating %s to status %d", p->key.string, status);
    p->status = status;
  }
}

static int install_execute_progress_io_handler (char *buf, size_t n __attribute__ ((unused)), void *user_data)
{
  di_packages *packages = user_data;
  char buf_package[128];

  execute_io_log_handler (buf, n, NULL);

  if (sscanf (buf, "Unpacking replacement %128s", buf_package) == 1 ||
      sscanf (buf, "Unpacking %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_UNPACK, buf_package);
    install_execute_progress_update (packages, buf_package, di_package_status_unpacked);
  }
  else if (sscanf (buf, "Setting up %128s", buf_package) == 1)
  {
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_CONFIGURE, buf_package);
    install_execute_progress_update (packages, buf_package, di_package_status_installed);
  }

  return 0;
}

static int install_execute_target_progress (const char *command, di_packages *packages)
{
  if (simulate_handler)
    return simulate_handler (command);
  return execute_target_full (command, install_execute_progress_io_handler, execute_io_log_handler, packages);
}

di_slist *install_list (di_packages *packages, di_packages_allocator *allocator, di_slist *install, di_package_priority priority, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;

  list1 = di_slist_alloc ();

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->priority >= priority && p->status < status)
      di_slist_append (list1, p);
  }

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package (di_packages *packages, di_packages_allocator *allocator, char *package, di_package_status status)
{
  di_slist *list1, *list2;
  di_slist_node *node;
  di_package *p;

  list1 = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);
  if (!p || p->status >= status)
    return list1;

  di_slist_append (list1, p);

  list2 = di_packages_resolve_dependencies (packages, list1, allocator);

  di_slist_free (list1);

  list1 = di_slist_alloc ();

  for (node = list2->head; node; node = node->next)
  {
    di_package *p = node->data;
    if (p->status < status)
      di_slist_append (list1, p);
  }

  di_slist_free (list2);

  return list1;
}

di_slist *install_list_package_only (di_packages *packages, char *package, di_package_status status)
{
  di_slist *list;
  di_package *p;

  list = di_slist_alloc ();

  p = di_packages_get_package (packages, package, 0);

  if (p && p->status < status)
    di_slist_append (list, p);

  return list;
}

int install_apt_install (di_packages *packages, di_slist *install)
{
  size_t size = 1024, len;
  char *buf = malloc (size);
  int count = 0, ret = 0;
  di_slist_node *node;

  strcpy (buf, "apt-get install --yes -o APT::Get::AllowUnauthenticated=true -o APT::Install-Recommends=false");
  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    count++;
    di_package *p = node->data;
    len += 1 + p->key.size;
    if (len >= size)
    {
      size *= 2;
      buf = realloc (buf, size);
    }
    strcat (buf, " ");
    strcat (buf, p->key.string);
  }

  if (count)
    ret = install_execute_target_progress (buf, packages);

  free (buf);

  return ret;
}

int install_dpkg_configure (di_packages *packages, int force)
{
  char command[128] = "dpkg --configure -a";

  if (force)
    strcat (command, " --force-all");

  return install_execute_target_progress (command, packages);
}

static int install_dpkg_all (const char *command, di_packages *packages, di_slist *install)
{
  size_t size = 1024, len;
  char *buf = malloc (size);
  int count = 0, ret = 0;
  di_slist_node *node;

  strcpy (buf, command);
  len = strlen (buf);

  for (node = install->head; node; node = node->next)
  {
    count++;
    di_package *p = node->data;
    char buf1[256];
    build_target_deb_root (buf1, sizeof (buf1), package_get_local_filename (p));
    len += 1 + strlen (buf1);
    if (len >= size)
    {
      size *= 2;
      buf = realloc (buf, size);
    }
    strcat (buf, " ");
    strcat (buf, buf1);
  }

  if (count)
    ret = install_execute_target_progress (buf, packages);

  free (buf);

  return ret;
}

int install_dpkg_install (di_packages *packages, di_slist *install, int force)
{
  char command[128] = "dpkg -i";

  if (force)
    strcat (command, " --force-all");

  return install_dpkg_all (command, packages, install);
}

int install_dpkg_unpack (di_packages *packages, di_slist *install)
{
  return install_dpkg_all ("dpkg --unpack --force-all", packages, install);
}

int install_extract (di_slist *install)
{
  struct di_slist_node *node;

  for (node = install->head; node; node = node->next)
  {
    di_package *p = node->data;
    log_message (LOG_MESSAGE_INFO_INSTALL_PACKAGE_EXTRACT, p->package);
    package_extract (p);
  }

  return 0;
}

int install_init (const char *_helperdir, install_simulate_handler _simulate_handler)
{
  helperdir = _helperdir;
  simulate_handler = _simulate_handler;

  target_create_dir ("/var");
  target_create_dir ("/var/lib");
  target_create_dir ("/var/lib/dpkg");
  target_create_file ("/var/lib/dpkg/available");
  target_create_file ("/var/lib/dpkg/diversions");
  target_create_file ("/var/lib/dpkg/status");

  log_open ();
  return 0;
}

int install_mount (const char *what)
{
  char buf[PATH_MAX];
  int ret = 0;
  enum { TARGET_DEV, TARGET_PROC } target;

  if (!strcmp (what, "dev"))
    target = TARGET_DEV;
  else if (!strcmp (what, "proc"))
    target = TARGET_PROC;
  else
  {
    log_text (DI_LOG_LEVEL_WARNING, "Unknown target for mount action: %s", what);
    return 0;
  }

  if (simulate_handler)
    return 0;

  snprintf (buf, sizeof buf, "%s/%s", target_root, what);
  switch (target)
  {
    case TARGET_DEV:
      ret = mount ("/dev", buf, NULL, MS_BIND | MS_RDONLY, 0);
      break;
    case TARGET_PROC:
      ret = mount ("proc", buf, "proc", 0, 0);
      break;
  }

  if (ret)
    log_text (DI_LOG_LEVEL_ERROR, "Failed to mount /%s: %s", what, strerror (errno));

  return ret;
}

#ifndef MNT_DETACH
enum { MNT_DETACH = 2 };
#endif

int install_umount (const char *what)
{
  char buf[PATH_MAX];
  int ret;

  snprintf (buf, sizeof buf, "%s/%s", target_root, what);
  ret = umount2 (buf, MNT_DETACH);

  if (ret < 0 && errno != ENOENT && errno != EINVAL)
    log_text (DI_LOG_LEVEL_WARNING, "Failed to unmount /%s: %s", what, strerror (errno));

  return ret;
}

int install_umount_all (void)
{
  int ret = 0;
  ret |= install_umount ("dev");
  ret |= install_umount ("proc");
  return ret;
}

int install_helper_install (const char *name)
{
  char file_source[4096];
  char file_dest[4096];
  char buf[4096];
  int ret;
  struct stat s;

  snprintf (file_source, sizeof file_source, "%s/%s.deb", helperdir, name);
  snprintf (file_dest, sizeof file_dest, "%s/var/cache/bootstrap/%s.deb", target_root, name);

  if (stat (file_source, &s) < 0)
    log_text (DI_LOG_LEVEL_ERROR, "Helper package %s not found", name);

  snprintf (buf, sizeof buf, "cp %s %s", file_source, file_dest);
  ret = execute (buf);
  if (ret)
    return ret;

  snprintf (buf, sizeof buf, "dpkg -i /var/cache/bootstrap/%s.deb", name);
  if (simulate_handler)
    return simulate_handler (buf);

  log_message (LOG_MESSAGE_INFO_INSTALL_HELPER_INSTALL, name);

  return execute_target (buf);
}

int install_helper_remove (const char *name)
{
  char buf[128];

  snprintf (buf, sizeof (buf), "dpkg -P %s", name);
  if (simulate_handler)
    return simulate_handler (buf);

  log_message (LOG_MESSAGE_INFO_INSTALL_HELPER_REMOVE, name);

  return execute_target (buf);
}

