#!/usr/bin/env python3

# Copyright 2024 The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""
Debusine integration tests.

Test signing service.
"""

import base64
import subprocess
import tempfile
from pathlib import Path
from unittest import TestCase

import requests

from utils.client import Client
from utils.common import Configuration, launch_tests
from utils.integration_test_helpers_mixin import IntegrationTestHelpersMixin
from utils.server import DebusineServer

import yaml

from debusine.artifacts.models import ArtifactCategory
from debusine.client.models import ArtifactResponse, FileResponse


class IntegrationSigningTests(IntegrationTestHelpersMixin, TestCase):
    """
    Integration tests for the signing service.

    These tests assume:
    - debusine-server is running
    - debusine-signing is running (connected to the server)
    - debusine-client is correctly configured
    """

    def setUp(self) -> None:
        """Initialize test."""
        # If debusine-server or nginx was launched just before this test,
        # then debusine-server might not be available yet.  Wait for
        # debusine-server to be reachable if it's not ready.
        self.assertTrue(
            DebusineServer.wait_for_server_ready(),
            f"debusine-server should be available (in "
            f"{Configuration.get_base_url()}) before the integration tests "
            f"are run",
        )

    def download_binary_and_extract_file(
        self, package_name: str, file_name: str
    ) -> bytes:
        """Download a binary package and extract a file from it."""
        with tempfile.TemporaryDirectory(
            prefix="debusine-integration-tests-"
        ) as temp_directory:
            temp_path = Path(temp_directory)
            subprocess.check_call(
                ["apt-get", "download", package_name], cwd=temp_path
            )
            subprocess.check_call(
                [
                    "dpkg-deb",
                    "-x",
                    next(temp_path.glob(f"{package_name}_*.deb")),
                    package_name,
                ],
                cwd=temp_path,
            )
            return (
                temp_path / package_name / file_name.lstrip("/")
            ).read_bytes()

    def verify_uefi_signature(
        self,
        signing_key_artifact: ArtifactResponse,
        signed_file: FileResponse,
    ):
        """Verify a signed file against its UEFI signing key."""
        with tempfile.TemporaryDirectory(
            prefix="debusine-integration-tests-"
        ) as temp_directory:
            temp_path = Path(temp_directory)
            (certificate := temp_path / "uefi.crt").write_bytes(
                base64.b64decode(
                    signing_key_artifact["data"]["public_key"].encode()
                )
            )
            (signature := temp_path / "image.sig").write_bytes(
                requests.get(signed_file["url"]).content
            )
            subprocess.check_call(
                ["sbverify", "--cert", certificate, signature], cwd=temp_path
            )

    def test_generate_and_sign(self) -> None:
        """Generate a key and sign something with it."""
        result = DebusineServer.execute_command(
            "create_work_request",
            "signing",
            "generatekey",
            stdin=yaml.safe_dump(
                {"purpose": "uefi", "description": "Test key"}
            ),
        )
        self.assertEqual(result.returncode, 0)
        work_request_id = yaml.safe_load(result.stdout)["work_request_id"]

        # The worker should get the new work request and start executing it
        status = Client.wait_for_work_request_completed(
            work_request_id, "success"
        )
        if not status:
            self.print_work_request_debug_logs(work_request_id)
        self.assertTrue(status)

        work_request = Client.execute_command(
            "show-work-request", work_request_id
        )

        [signing_key_artifact] = [
            artifact
            for artifact in work_request["artifacts"]
            if artifact["category"] == ArtifactCategory.SIGNING_KEY
        ]
        self.assertEqual(signing_key_artifact["data"]["purpose"], "uefi")
        self.assertIn("fingerprint", signing_key_artifact["data"])
        self.assertIn("public_key", signing_key_artifact["data"])

        # Sign something real, if we know how to find it.
        if (
            subprocess.check_output(
                ["dpkg", "--print-architecture"], text=True
            ).strip()
            == "amd64"
        ):
            package_name = "grub-efi-amd64-bin"
            file_name = "/usr/lib/grub/x86_64-efi/monolithic/grubnetx64.efi"
            grubnetx64 = self.download_binary_and_extract_file(
                package_name, file_name
            )
            qualified_file_name = package_name + file_name
            signing_input_id = self.create_artifact_signing_input(
                qualified_file_name, grubnetx64
            )
            result = DebusineServer.execute_command(
                "create_work_request",
                "signing",
                "sign",
                stdin=yaml.safe_dump(
                    {
                        "purpose": "uefi",
                        "unsigned": signing_input_id,
                        "key": signing_key_artifact["id"],
                    }
                ),
            )
            self.assertEqual(result.returncode, 0)
            work_request_id = yaml.safe_load(result.stdout)["work_request_id"]

            status = Client.wait_for_work_request_completed(
                work_request_id, "success"
            )
            if not status:
                self.print_work_request_debug_logs(work_request_id)
            self.assertTrue(status)

            work_request = Client.execute_command(
                "show-work-request", work_request_id
            )

            [signing_output_artifact] = [
                artifact
                for artifact in work_request["artifacts"]
                if artifact["category"] == ArtifactCategory.SIGNING_OUTPUT
            ]
            signature_file_name = f"{qualified_file_name}.sig"
            self.assertEqual(
                signing_output_artifact["data"],
                {
                    "purpose": "uefi",
                    "fingerprint": signing_key_artifact["data"]["fingerprint"],
                    "results": [
                        {
                            "file": qualified_file_name,
                            "output_file": signature_file_name,
                            "error_message": None,
                        }
                    ],
                },
            )
            self.assertEqual(
                list(signing_output_artifact["files"]), [signature_file_name]
            )
            self.verify_uefi_signature(
                signing_key_artifact,
                signing_output_artifact["files"][signature_file_name],
            )


if __name__ == "__main__":
    launch_tests("Signing task integration tests for debusine")
