// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using Test.Cryptography;

namespace System.Security.Cryptography.X509Certificates.Tests
{
    internal static class TestData
    {
        internal const string PlaceholderPw = "Placeholder";

        public static byte[] MsCertificate = (
            "308204ec308203d4a003020102021333000000b011af0a8bd03b9fdd00010000" +
            "00b0300d06092a864886f70d01010505003079310b3009060355040613025553" +
            "311330110603550408130a57617368696e67746f6e3110300e06035504071307" +
            "5265646d6f6e64311e301c060355040a13154d6963726f736f667420436f7270" +
            "6f726174696f6e312330210603550403131a4d6963726f736f667420436f6465" +
            "205369676e696e6720504341301e170d3133303132343232333333395a170d31" +
            "34303432343232333333395a308183310b300906035504061302555331133011" +
            "0603550408130a57617368696e67746f6e3110300e060355040713075265646d" +
            "6f6e64311e301c060355040a13154d6963726f736f667420436f72706f726174" +
            "696f6e310d300b060355040b13044d4f5052311e301c060355040313154d6963" +
            "726f736f667420436f72706f726174696f6e30820122300d06092a864886f70d" +
            "01010105000382010f003082010a0282010100e8af5ca2200df8287cbc057b7f" +
            "adeeeb76ac28533f3adb407db38e33e6573fa551153454a5cfb48ba93fa837e1" +
            "2d50ed35164eef4d7adb137688b02cf0595ca9ebe1d72975e41b85279bf3f82d" +
            "9e41362b0b40fbbe3bbab95c759316524bca33c537b0f3eb7ea8f541155c0865" +
            "1d2137f02cba220b10b1109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5" +
            "c90a21e2aae3013647fd2f826a8103f5a935dc94579dfb4bd40e82db388f12fe" +
            "e3d67a748864e162c4252e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a6" +
            "79b5b6dbcef9707b280184b82a29cfbfa90505e1e00f714dfdad5c238329ebc7" +
            "c54ac8e82784d37ec6430b950005b14f6571c50203010001a38201603082015c" +
            "30130603551d25040c300a06082b06010505070303301d0603551d0e04160414" +
            "5971a65a334dda980780ff841ebe87f9723241f230510603551d11044a3048a4" +
            "463044310d300b060355040b13044d4f5052313330310603550405132a333135" +
            "39352b34666166306237312d616433372d346161332d613637312d3736626330" +
            "35323334346164301f0603551d23041830168014cb11e8cad2b4165801c9372e" +
            "331616b94c9a0a1f30560603551d1f044f304d304ba049a0478645687474703a" +
            "2f2f63726c2e6d6963726f736f66742e636f6d2f706b692f63726c2f70726f64" +
            "756374732f4d6963436f645369675043415f30382d33312d323031302e63726c" +
            "305a06082b06010505070101044e304c304a06082b06010505073002863e6874" +
            "74703a2f2f7777772e6d6963726f736f66742e636f6d2f706b692f6365727473" +
            "2f4d6963436f645369675043415f30382d33312d323031302e637274300d0609" +
            "2a864886f70d0101050500038201010031d76e2a12573381d59dc6ebf93ad444" +
            "4d089eee5edf6a5bb779cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0d" +
            "e0e19bf45d240f1b8d88153a7cdbadceb3c96cba392c457d24115426300d0dff" +
            "47ea0307e5e4665d2c7b9d1da910fa1cb074f24f696b9ea92484daed96a0df73" +
            "a4ef6a1aac4b629ef17cc0147f48cd4db244f9f03c936d42d8e87ce617a09b68" +
            "680928f90297ef1103ba6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb" +
            "02133b5d20de553fa3ae9040313875285e04a9466de6f57a7940bd1fcde845d5" +
            "aee25d3ef575c7e6666360ccd59a84878d2430f7ef34d0631db142674a0e4bbf" +
            "3a0eefb6953aa738e4259208a6886682").HexToByteArray();

        // This pfx was generated by new X509Certificate(MsCertificate).Export(X509ContentType.Pfx)
        // and was choosen when the padding was 01 and caused a false-positive on decryption.
        public static byte[] MsCertificateExportedToPfx_NullPassword = Convert.FromBase64String(
            /* [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="This PKCS#12 blob only contains public info.")] */ @"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");

        public static readonly byte[] MsCertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIE7DCCA9SgAwIBAgITMwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQD
ExpNaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzlaFw0x
NDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNyb3NvZnQgQ29ycG9yYXRp
b24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/
re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01Fk7vTXrbE3aIsCzw
WVyp6+HXKXXkG4Unm/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhl
HSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMBNkf9L4JqgQP1
qTXclFed+0vUDoLbOI8S/uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZDC5UABbFP
ZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQU
WXGmWjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsTBE1P
UFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMw
NTIzNDRhZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9k
dWN0cy9NaWNDb2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBM
MEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRz
L01pY0NvZFNpZ1BDQV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEA
MdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/oozxTbqVOfsN
4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjofOYXoJto
aAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUo
XgSpRm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdKDku/
Og7vtpU6pzjkJZIIpohmgg==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned1PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJAJpCQ7mtFWHeMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNTMwMTgyNjM1WhcNMTkwNTMwMTgyNjM1WjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
pfYZTHjzei9U3QxiIIjESsf9z3Bfl8FAQLIU+OeICN3upnDvTgeWM/Jw7LwiuHhu
XvSawPwQ8ONvUeSG/wfyjYyTB7VBpVnNi6oTR6E1WSuiu0iT3qlDHVwArTI5DvIM
FzP3/AT1Ub5SvwVbWiR2za6wuUIsryyLz5+zCwGr+J/Xbmta/H9IT9NLwmDJCZQe
4Q4hCWhf7FKdXWt59y9PofVnE7R8CKNfUKr6GA+gy+SEtM/cHgqox5PErnV9b14U
uVROnRUyo1bFwTOdoW3zf5S4VZ4pFPJHNYACHEMiE0eNgfJf+QeyPUPN50neEAbf
kQYkeEET8dW6JlDFrAI4wwIDAQABo1MwUTAdBgNVHQ4EFgQUK+C/eGYPlV+KaTvj
tF6lJaKmo3EwHwYDVR0jBBgwFoAUK+C/eGYPlV+KaTvjtF6lJaKmo3EwDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEAZUjvDMhGc45TLRHKO5rsyifN
g7qb3dO5vtD/JWeo+wyMYcBHIANIIxYrkT0dRBQWQerVDBvsAESahM3f0SdszGac
6y1qxQWxfjxRiCwrEQ7JVZkmspYLbOxaS1T2IZUo3D7VJReyna6r11EKy7i49Toa
KmrhTLBsHV+MUgPRtupiOOu0fXqfxpXE7XEvi0hyv8PKli+Oww2Zyt1jTTvv2RTA
eJRqTUNUbWEDesXAOh5CY6Xjfg7Gt6IYQHt0JMw29pXB3TV2uyXuvFNsc725cPbW
JCuC9TGQRUAUj+LZ43tTrfaZ7g5L80/eRrvlx5MIJSsX8cev8pZYx224WRtk/w==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned2PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJAM6YQ4PrC9jaMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNTMwMTgyNjQ4WhcNMTkwNTMwMTgyNjQ4WjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
7vkM6zrhXJFtrV63lUb4fsjZG2JYvSRGYv4Y/rwe7VLVdTYvMjosyvKCHJ4Frmtb
YU4jJeB+859mQAd3IOBEhgUJuJ6gC8cOJAwUFJNUabeuafXG2zw/U+396csRKr11
iBUpvooFJR7KLWrqPKXhK5yESV1k7OzSSZs4owmyIvSaGQO2T63S39OYJhq8ZUlO
+MznaOQGp2J+JWncZo9XCpiotZwdNtw5k+F1g3NAz4/+Vkah/SfQhcNCfJyfVDCX
IwBS+gz9i1BIw6s+SLYtkp167yyizmVIWoXtkgCPaeG0FzBPAhL9GDLTItJ/V/O5
F9SjbvS+4rUIuPSn7NdodwIDAQABo1MwUTAdBgNVHQ4EFgQUq4v4TrvYrsbKDRGF
bMnj3++P9B4wHwYDVR0jBBgwFoAUq4v4TrvYrsbKDRGFbMnj3++P9B4wDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEAS4ZKEGVfIrHMmYAZe9p3jm15
85OIPLlM4q6QjRccLJ4t2+xYLRU9PNa2Qmz8l+SFOy9XN9yJp79lSi1gN4lJ8ZdI
kwu+zPEzwsJnb6f3zRu2RQUeAkaShDCEdxpyKHEky1KVG2nOa3cKp+pqzN4DQ3Rp
cJCjcP5ncNJ0bbCZTS7w0oVvX5JhBWIigw3CN5rL2rf83CTPPBzUype0bt97sBSs
dxIPtH9l/q9OgdaCrPE8KBqcwXsfNlFwYGjkqmN/v7WaysBRdblHcoWmry3YsaK2
/tZo6lmYOHpdqL0OdDwlldToY7QdL1coICfHas5Ony49OHTCUZz6G/AS+3a3gQ==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned3PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJANzv9IQvr0bwMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNjA0MTMzMjIxWhcNMTkwNjA0MTMzMjIxWjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
wy+py+hFxSmCGTZmHrQm1Yobzxf34l+J8VD33ObGV1qIFFulxz8pnUU4gKf6FQNU
wAbezJ78Eqsjt4c7mwnGTdavWSZyDJ136bQzn52wsTOGRfUBe1vt+SMy7h8Nhhf3
ejRHQVsZKNfiGOekmjBKFLliavo6I8j80UsmpvAJ+TWnYpVQBf/EzBQ21ddIF6jD
nl2ZhcvWHvS63utWwXW68xkDXsLvjiat22YScRKnkkNAIvbBY4rvV1KwahUPaMTS
zWywr6caHxlKp7McZ4MJVIqUAeZUn4KYgSksi2IsfPA7qi8WpSaKGsOZFBD79DJC
wqzdLLBzEtg6okzgC5nFtwIDAQABo1MwUTAdBgNVHQ4EFgQUgKAUBaaA1XD8KqGg
1XEr74W4lrkwHwYDVR0jBBgwFoAUgKAUBaaA1XD8KqGg1XEr74W4lrkwDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEArNBpG8oCDKX9ERbMgvgm3qWk
FKmx+h58aiZVoMwfBsf2njZ6BzRoEOvluMDPe+pt8hhST5yaOsMUYIqrn+s692I9
17JRfrFhCp+4GT8oe/ZnSNTPm2zOzm0VXFkfDF53YGzdGTWXLH+pJpw4drCNoBoA
yloyF/JJGJ2ZMbnwuDtsPbpjup8qHLiQYjxj4hUWyXU+nbytGK/i8z8HHc7acOpd
9+MXEcKwUkthXzG0M/0bzz4GwWZ6PPmbI5EEqFGBzMef58/mbHDigl9/o3kUlJtB
tcCZhP5KEu6XKKc1GcTqbyA0vi92YyyZViUa36hhVrNqPxtpclir+lcnNgnlqg==
-----END CERTIFICATE-----
");

        // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Unit test password.")]
        public const string PfxDataPassword = "12345";

        public static SecureString CreatePfxDataPasswordSecureString()
        {
            var s = new SecureString();

            // WARNING:
            // A key value of SecureString is in keeping string data off of the GC heap, such that it can
            // be reliably cleared when no longer needed.  Creating a SecureString from a string or converting
            // a SecureString to a string diminishes that value. These conversion functions are for testing that
            // SecureString works, and does not represent a pattern to follow in any non-test situation.
            foreach (char c in PfxDataPassword.ToCharArray())
            {
                s.AppendChar(c);
            }

            return s;
        }

        public static readonly byte[] PfxSha256Empty_ExpectedSig = (
            "35306b63fbfe8eaf8e32937432e366d255461b02fb9e77368df319b99a8273cc" +
            "dad52a21e8ae9c5df97ffa877d6c3ef6eb8a27ac7bbc93efdddd5dda9cf8d30e" +
            "c52a76c7b2f9e478efe272d6e6f474033fb854e5685c44898367af860d5c893c" +
            "56f89a69b84cd53434457de3296f80fbbe4eb938b4066b51638e35bce1e1c308").HexToByteArray();

        private static readonly byte[] PfxData_RC2ContentEncryption = (
            "3082063A020103308205F606092A864886F70D010701A08205E7048205E33082" +
            "05DF3082035806092A864886F70D010701A08203490482034530820341308203" +
            "3D060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E04085052002C7DA2C2A6020207D0048202907D485E3BFC6E6457" +
            "C811394C145D0E8A18325646854E4FF0097BC5A98547F5AD616C8EFDA8505AA8" +
            "7564ED4800A3139759497C60C6688B51F376ACAE906429C8771CB1428226B68A" +
            "6297207BCC9DD7F9563478DD83880AAB2304B545759B2275305DF4EFF9FAC24A" +
            "3CC9D3B2D672EFE45D8F48E24A16506C1D7566FC6D1B269FBF201B3AC3309D3E" +
            "BC6FD606257A7A707AA2F790EA3FE7A94A51138540C5319010CBA6DE9FB9D85F" +
            "CDC78DA60E33DF2F21C46FB9A8554B4F82E0A6EDBA4DB5585D77D331D35DAAED" +
            "51B6A5A3E000A299880FB799182C8CA3004B7837A9FEB8BFC76778089993F3D1" +
            "1D70233608AF7C50722D680623D2BF54BD4B1E7A604184D9F44E0AF8099FFA47" +
            "1E5536E7902793829DB9902DDB61264A62962950AD274EA516B2D44BE9036530" +
            "016E607B73F341AEEFED2211F6330364738B435B0D2ED6C57747F6C8230A053F" +
            "78C4DD65DB83B26C6A47836A6CBBAB92CBB262C6FB6D08632B4457F5FA8EABFA" +
            "65DB34157E1D301E9085CC443582CDD15404314872748545EB3FC3C574882655" +
            "8C9A85F966E315775BBE9DA34D1E8B6DADC3C9E120C6D6A2E1CFFE4EB014C3CE" +
            "FBC19356CE33DAC60F93D67A4DE247B0DAE13CD8B8C9F15604CC0EC9968E3AD7" +
            "F57C9F53C45E2ECB0A0945EC0BA04BAA15B48D8596EDC9F5FE9165A5D21949FB" +
            "5FE30A920AD2C0F78799F6443C300629B8CA4DCA19B9DBF1E27AAB7B12271228" +
            "119A95C9822BE6439414BEEAE24002B46EB97E030E18BD810ADE0BCF4213A355" +
            "038B56584B2FBCC3F5EA215D0CF667FFD823EA03AB62C3B193DFB4450AABB50B" +
            "AF306E8088EE7384FA2FDFF03E0DD7ACD61832223E806A94D46E196462522808" +
            "3163F1CAF333FDBBE2D54CA86968867CE0B6DD5E5B7F0633C6FAB4A19CC14F64" +
            "5EC14D0B1436F7623174301306092A864886F70D010915310604040100000030" +
            "5D06092B060104018237110131501E4E004D006900630072006F0073006F0066" +
            "00740020005300740072006F006E0067002000430072007900700074006F0067" +
            "007200610070006800690063002000500072006F007600690064006500723082" +
            "027F06092A864886F70D010706A08202703082026C0201003082026506092A86" +
            "4886F70D010701301C060A2A864886F70D010C0106300E0408E0C117E67A75D8" +
            "EB020207D080820238292882408B31826F0DC635F9BBE7C199A48A3B4FEFC729" +
            "DBF95508D6A7D04805A8DD612427F93124F522AC7D3C6F4DDB74D937F57823B5" +
            "B1E8CFAE4ECE4A1FFFD801558D77BA31985AA7F747D834CBE84464EF777718C9" +
            "865C819D6C9DAA0FA25E2A2A80B3F2AAA67D40E382EB084CCA85E314EA40C3EF" +
            "3ED1593904D7A16F37807C99AF06C917093F6C5AAEBB12A6C58C9956D4FBBDDE" +
            "1F1E389989C36E19DD38D4B978D6F47131E458AB68E237E40CB6A87F21C8773D" +
            "E845780B50995A51F041106F47C740B3BD946038984F1AC9E91230616480962F" +
            "11B0683F8802173C596C4BD554642F51A76F9DFFF9053DEF7B3C3F759FC7EEAC" +
            "3F2386106C4B8CB669589E004FB235F0357EA5CF0B5A6FC78A6D941A3AE44AF7" +
            "B601B59D15CD1EC61BCCC481FBB83EAE2F83153B41E71EF76A2814AB59347F11" +
            "6AB3E9C1621668A573013D34D13D3854E604286733C6BAD0F511D7F8FD6356F7" +
            "C3198D0CB771AF27F4B5A3C3B571FDD083FD68A9A1EEA783152C436F7513613A" +
            "7E399A1DA48D7E55DB7504DC47D1145DF8D7B6D32EAA4CCEE06F98BB3DDA2CC0" +
            "D0564A962F86DFB122E4F7E2ED6F1B509C58D4A3B2D0A68788F7E313AECFBDEF" +
            "456C31B96FC13586E02AEB65807ED83BB0CB7C28F157BC95C9C593C919469153" +
            "9AE3C620ED1D4D4AF0177F6B9483A5341D7B084BC5B425AFB658168EE2D8FB2B" +
            "FAB07A3BA061687A5ECD1F8DA9001DD3E7BE793923094ABB0F2CF4D24CB071B9" +
            "E568B18336BB4DC541352C9785C48D0F0E53066EB2009EFCB3E5644ED12252C1" +
            "BC303B301F300706052B0E03021A04144DEAB829B57A3156AEBC8239C0E7E884" +
            "EFD96E680414E147930B932899741C92D7652268938770254A2B020207D0").HexToByteArray();

        private static readonly byte[] PfxData_TripleDESContentEncryption = (
            "308206980201033082065406092A864886F70D010701A0820645048206413082" +
            "063D308203B606092A864886F70D010701A08203A7048203A33082039F308203" +
            "9B060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E04083A55E19A8875CF7E020207D004820290C39707B6E15BC075" +
            "62B296E37D44A4C3654F398528F51988932D4E78C96AC7E783A21DD306AEB4BB" +
            "7490551F5CCB21DBF883B3D01D3CCFEB0148CC79C8C15AD7266BC0C0D23563FF" +
            "A879DDB05ACC5B1F9A9BA5F8E32ECE05458A55DC91882750B0573315D843524D" +
            "72E400C14E24B530A228B477DF60F01A1CBA22085350C72B8E871AEF219D1457" +
            "3A660D71064D4226814B8D360A84FDF9B24578E61D62119C893DF1CB97C733BC" +
            "D9CCA8C3FBA40B3E35F305AFCF1E24E720379FC94FB7103A27F817D98AF70D12" +
            "0A357EB5840720F48AE6ED59990E80541CCA08B2B3F9FAE0F7554F491616E72B" +
            "E2383478709947F5BD5AC8D2F74550A2C4260DAA789C66559DBC25D0D5657DEF" +
            "FF491287E9E982C870732E6C1494F25FED513D2DF246EDABA1DEF8FE05A1C283" +
            "676A0722961FBAD4B47E8D27D08E4129FACE86CAAB657A1899C7F9286DD534AC" +
            "3AE6B1A100C90207B9A39857C1D9B7A061A5E8496F0E099F1323FA56DE85BF34" +
            "96DBBE6FFEAAC0321F65A40BAE63503BB704553F1027700D15D12A20EC3A8FB9" +
            "8D62D902FBDBD1BA9984242A6413CF243FE1C5A5A42083EA6E821D302CC4BE4D" +
            "23C0D92247C027A6D5AE3D645F378916ACBD068F52A7772571F51F4AF46652E6" +
            "D9821A24267F6A8CF9D69B3EF5FF33702B71CCDA87A3704DB815685F83E22C64" +
            "B60D9B659EEA517BCACDD638DB2EF171692AA17176891901F2BB97F574908CCA" +
            "943743C1E8D2520FE7B5E04C10E748A666EAAE81EF5DB3577F669E049BE57CCF" +
            "7FFCF7583A9ABE0BC44F0D33EBE7F3BFC2F1F4012F588D0FC8CB0FFDF3BF1A43" +
            "A962B0F9059FAAC0BD68B23275584180EDDE1F1703BFE977456079FF0890BFE5" +
            "ED9CA93ABECF335C02DEBB7DB1A1E60B5B7746F6F152299A1DB983ACE6D9D5B9" +
            "DBAF68A3DEDCCC013181D1301306092A864886F70D0109153106040401000000" +
            "305B06092A864886F70D010914314E1E4C007B00430035004600430030003800" +
            "350032002D0038004200410033002D0034003700460041002D00410045003400" +
            "33002D004200440039003300420044003200350041004300440042007D305D06" +
            "092B060104018237110131501E4E004D006900630072006F0073006F00660074" +
            "0020005300740072006F006E0067002000430072007900700074006F00670072" +
            "00610070006800690063002000500072006F007600690064006500723082027F" +
            "06092A864886F70D010706A08202703082026C0201003082026506092A864886" +
            "F70D010701301C060A2A864886F70D010C0103300E0408812EAA0D26B0151B02" +
            "0207D080820238A471B31EBE90A6B540889297C09BFC73185C2C2676665DBAD2" +
            "BC5759464079976B4C82DCB425289BCFABF9CD11AABF18B3CFA1879A958919E5" +
            "05465E5B2DBCA5039990650F7D7490D322F2CCCD444ACE4625BE3C361FA5C096" +
            "615ACDA3123A478340332961655C973D74EE64F32015194558F4B8148DED4671" +
            "7EE0381E8A53B582F176DB914475FB57C2648DCC709D6153825F33F082B91EAB" +
            "BB2236343CA919065B9D9112D301E6700CE8104035469EDF7874246373351E13" +
            "DC1DAB74DF1DE1E076FE03869F494C0427DE55CA7C17536DEF40D0E3F1B7E5FF" +
            "5A2050175624A1F2D9EF4069186424CC956B84997768E589E38D3D836592CF27" +
            "881DB2EE51AB682B4F4C58ED14585C51B72CCF156BE3CD2EEC538AD15A598E84" +
            "0CFCDF6CD4A151B7EA4BE5388EA903298B4D1A86E7371300A6242DD7A654E430" +
            "2B4F849650230F467042CD328A4FD4B33803213C807C37F98024A718B40AD0BD" +
            "3AC6279E47FBD27E50D743BC8F595B78F8805D8143EBB3119B919DFFBFAF606F" +
            "A858C77C5E0526646B6AACC536A3B2FEA1E0A67CBAE166A8F85913F3600C55F7" +
            "850EF48619B9E93FB15F21994BC9ABA8FF7156B68B1DAD472ED2FA3917E1F84D" +
            "AD4EB34992493B7084F6490FB326697A0A52BA08BF7F537B3BDC09B421A0D47F" +
            "DCB8BD0271963EC4CE4F9C21A45AC382E8AD7ED8D69824BD5C00BCEAF56D88B3" +
            "F2D0D5A81CF1C14884040A8E522428154A9C752C0E64204FA849689739E5138F" +
            "0F96C471C5FE026EA7EA68AA42DF0A693213ECF06555804A191637A8F7858A30" +
            "3B301F300706052B0E03021A04146B4EDA4227EAFC85EDA331AC88761DD06D54" +
            "60FD0414C8A2B10C67EAECB3D48BEC69616133185721613D020207D0").HexToByteArray();

        public static readonly byte[] PfxData =
            PlatformSupport.IsRC2Supported
                ? PfxData_RC2ContentEncryption
                : PfxData_TripleDESContentEncryption;

        public static byte[] StoreSavedAsPfxData = (
            "3082070406092a864886f70d010702a08206f5308206f10201013100300b0609" +
            "2a864886f70d010701a08206d9308201e530820152a0030201020210d5b5bc1c" +
            "458a558845bff51cb4dff31c300906052b0e03021d05003011310f300d060355" +
            "040313064d794e616d65301e170d3130303430313038303030305a170d313130" +
            "3430313038303030305a3011310f300d060355040313064d794e616d6530819f" +
            "300d06092a864886f70d010101050003818d0030818902818100b11e30ea8742" +
            "4a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42b68056128322b2" +
            "1f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc916a1bc7322dd353" +
            "b32898675cfb5b298b176d978b1f12313e3d865bc53465a11cca106870a4b5d5" +
            "0a2c410938240e92b64902baea23eb093d9599e9e372e48336730203010001a3" +
            "46304430420603551d01043b3039801024859ebf125e76af3f0d7979b4ac7a96" +
            "a1133011310f300d060355040313064d794e616d658210d5b5bc1c458a558845" +
            "bff51cb4dff31c300906052b0e03021d0500038181009bf6e2cf830ed485b86d" +
            "6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d689ffd7234b787" +
            "2611c5c23e5e0714531abadb5de492d2c736e1c929e648a65cc9eb63cd84e57b" +
            "5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b2bd9a26b192b95" +
            "7304b89531e902ffc91b54b237bb228be8afcda26476308204ec308203d4a003" +
            "020102021333000000b011af0a8bd03b9fdd0001000000b0300d06092a864886" +
            "f70d01010505003079310b300906035504061302555331133011060355040813" +
            "0a57617368696e67746f6e3110300e060355040713075265646d6f6e64311e30" +
            "1c060355040a13154d6963726f736f667420436f72706f726174696f6e312330" +
            "210603550403131a4d6963726f736f667420436f6465205369676e696e672050" +
            "4341301e170d3133303132343232333333395a170d3134303432343232333333" +
            "395a308183310b3009060355040613025553311330110603550408130a576173" +
            "68696e67746f6e3110300e060355040713075265646d6f6e64311e301c060355" +
            "040a13154d6963726f736f667420436f72706f726174696f6e310d300b060355" +
            "040b13044d4f5052311e301c060355040313154d6963726f736f667420436f72" +
            "706f726174696f6e30820122300d06092a864886f70d01010105000382010f00" +
            "3082010a0282010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb" +
            "407db38e33e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb" +
            "137688b02cf0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bba" +
            "b95c759316524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1" +
            "109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd" +
            "2f826a8103f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c425" +
            "2e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b2801" +
            "84b82a29cfbfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec643" +
            "0b950005b14f6571c50203010001a38201603082015c30130603551d25040c30" +
            "0a06082b06010505070303301d0603551d0e041604145971a65a334dda980780" +
            "ff841ebe87f9723241f230510603551d11044a3048a4463044310d300b060355" +
            "040b13044d4f5052313330310603550405132a33313539352b34666166306237" +
            "312d616433372d346161332d613637312d373662633035323334346164301f06" +
            "03551d23041830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f3056" +
            "0603551d1f044f304d304ba049a0478645687474703a2f2f63726c2e6d696372" +
            "6f736f66742e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f" +
            "645369675043415f30382d33312d323031302e63726c305a06082b0601050507" +
            "0101044e304c304a06082b06010505073002863e687474703a2f2f7777772e6d" +
            "6963726f736f66742e636f6d2f706b692f63657274732f4d6963436f64536967" +
            "5043415f30382d33312d323031302e637274300d06092a864886f70d01010505" +
            "00038201010031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779" +
            "cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88" +
            "153a7cdbadceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b" +
            "9d1da910fa1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17c" +
            "c0147f48cd4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba" +
            "6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae" +
            "9040313875285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e66663" +
            "60ccd59a84878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e425" +
            "9208a68866823100").HexToByteArray();

        public static byte[] StoreSavedAsCerData = (
            "308201e530820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c30" +
            "0906052b0e03021d05003011310f300d060355040313064d794e616d65301e17" +
            "0d3130303430313038303030305a170d3131303430313038303030305a301131" +
            "0f300d060355040313064d794e616d6530819f300d06092a864886f70d010101" +
            "050003818d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff" +
            "1c189d0d888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55f" +
            "f5ae64e9b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b" +
            "1f12313e3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea" +
            "23eb093d9599e9e372e48336730203010001a346304430420603551d01043b30" +
            "39801024859ebf125e76af3f0d7979b4ac7a96a1133011310f300d0603550403" +
            "13064d794e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e" +
            "03021d0500038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb93" +
            "48923710666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5d" +
            "e492d2c736e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca" +
            "225b6e368b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237" +
            "bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedCerData = (
            "0200000001000000bc0000001c0000006c000000010000000000000000000000" +
            "00000000020000007b00370037004500420044003000320044002d0044003800" +
            "440045002d0034003700350041002d0038003800360037002d00440032003000" +
            "4200300030003600340045003400390046007d00000000004d00690063007200" +
            "6f0073006f006600740020005300740072006f006e0067002000430072007900" +
            "700074006f0067007200610070006800690063002000500072006f0076006900" +
            "64006500720000002000000001000000e9010000308201e530820152a0030201" +
            "020210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05003011" +
            "310f300d060355040313064d794e616d65301e170d3130303430313038303030" +
            "305a170d3131303430313038303030305a3011310f300d060355040313064d79" +
            "4e616d6530819f300d06092a864886f70d010101050003818d00308189028181" +
            "00b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42" +
            "b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc91" +
            "6a1bc7322dd353b32898675cfb5b298b176d978b1f12313e3d865bc53465a11c" +
            "ca106870a4b5d50a2c410938240e92b64902baea23eb093d9599e9e372e48336" +
            "730203010001a346304430420603551d01043b3039801024859ebf125e76af3f" +
            "0d7979b4ac7a96a1133011310f300d060355040313064d794e616d658210d5b5" +
            "bc1c458a558845bff51cb4dff31c300906052b0e03021d0500038181009bf6e2" +
            "cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d" +
            "689ffd7234b7872611c5c23e5e0714531abadb5de492d2c736e1c929e648a65c" +
            "c9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b" +
            "2bd9a26b192b957304b89531e902ffc91b54b237bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedStoreData = (
            "00000000434552540200000001000000bc0000001c0000006c00000001000000" +
            "000000000000000000000000020000007b003700370045004200440030003200" +
            "44002d0044003800440045002d0034003700350041002d003800380036003700" +
            "2d004400320030004200300030003600340045003400390046007d0000000000" +
            "4d006900630072006f0073006f006600740020005300740072006f006e006700" +
            "2000430072007900700074006f00670072006100700068006900630020005000" +
            "72006f007600690064006500720000002000000001000000e9010000308201e5" +
            "30820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c300906052b" +
            "0e03021d05003011310f300d060355040313064d794e616d65301e170d313030" +
            "3430313038303030305a170d3131303430313038303030305a3011310f300d06" +
            "0355040313064d794e616d6530819f300d06092a864886f70d01010105000381" +
            "8d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d" +
            "888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9" +
            "b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b1f12313e" +
            "3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea23eb093d" +
            "9599e9e372e48336730203010001a346304430420603551d01043b3039801024" +
            "859ebf125e76af3f0d7979b4ac7a96a1133011310f300d060355040313064d79" +
            "4e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05" +
            "00038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710" +
            "666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5de492d2c7" +
            "36e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e36" +
            "8b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237bb228be8" +
            "afcda264762000000001000000f0040000308204ec308203d4a0030201020213" +
            "33000000b011af0a8bd03b9fdd0001000000b0300d06092a864886f70d010105" +
            "05003079310b3009060355040613025553311330110603550408130a57617368" +
            "696e67746f6e3110300e060355040713075265646d6f6e64311e301c06035504" +
            "0a13154d6963726f736f667420436f72706f726174696f6e3123302106035504" +
            "03131a4d6963726f736f667420436f6465205369676e696e6720504341301e17" +
            "0d3133303132343232333333395a170d3134303432343232333333395a308183" +
            "310b3009060355040613025553311330110603550408130a57617368696e6774" +
            "6f6e3110300e060355040713075265646d6f6e64311e301c060355040a13154d" +
            "6963726f736f667420436f72706f726174696f6e310d300b060355040b13044d" +
            "4f5052311e301c060355040313154d6963726f736f667420436f72706f726174" +
            "696f6e30820122300d06092a864886f70d01010105000382010f003082010a02" +
            "82010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb407db38e33" +
            "e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb137688b02c" +
            "f0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bbab95c759316" +
            "524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1109d772285" +
            "847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd2f826a8103" +
            "f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c4252e2aae9d18" +
            "1f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b280184b82a29cf" +
            "bfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec6430b950005b1" +
            "4f6571c50203010001a38201603082015c30130603551d25040c300a06082b06" +
            "010505070303301d0603551d0e041604145971a65a334dda980780ff841ebe87" +
            "f9723241f230510603551d11044a3048a4463044310d300b060355040b13044d" +
            "4f5052313330310603550405132a33313539352b34666166306237312d616433" +
            "372d346161332d613637312d373662633035323334346164301f0603551d2304" +
            "1830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f30560603551d1f" +
            "044f304d304ba049a0478645687474703a2f2f63726c2e6d6963726f736f6674" +
            "2e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f6453696750" +
            "43415f30382d33312d323031302e63726c305a06082b06010505070101044e30" +
            "4c304a06082b06010505073002863e687474703a2f2f7777772e6d6963726f73" +
            "6f66742e636f6d2f706b692f63657274732f4d6963436f645369675043415f30" +
            "382d33312d323031302e637274300d06092a864886f70d010105050003820101" +
            "0031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779cf029cbc76" +
            "689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88153a7cdbad" +
            "ceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b9d1da910fa" +
            "1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17cc0147f48cd" +
            "4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba6752adc1e9" +
            "b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae9040313875" +
            "285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e6666360ccd59a84" +
            "878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e4259208a68866" +
            "82000000000000000000000000").HexToByteArray();

        public static byte[] DssCer = (
            "3082025d3082021da00302010202101e9ae1e91e07de8640ac7af21ac22e8030" +
            "0906072a8648ce380403300e310c300a06035504031303466f6f301e170d3135" +
            "303232343232313734375a170d3136303232343232313734375a300e310c300a" +
            "06035504031303466f6f308201b73082012c06072a8648ce3804013082011f02" +
            "818100871018cc42552d14a5a9286af283f3cfba959b8835ec2180511d0dceb8" +
            "b979285708c800fc10cb15337a4ac1a48ed31394072015a7a6b525986b49e5e1" +
            "139737a794833c1aa1e0eaaa7e9d4efeb1e37a65dbc79f51269ba41e8f0763aa" +
            "613e29c81c3b977aeeb3d3c3f6feb25c270cdcb6aee8cd205928dfb33c44d2f2" +
            "dbe819021500e241edcf37c1c0e20aadb7b4e8ff7aa8fde4e75d02818100859b" +
            "5aeb351cf8ad3fabac22ae0350148fd1d55128472691709ec08481584413e9e5" +
            "e2f61345043b05d3519d88c021582ccef808af8f4b15bd901a310fefd518af90" +
            "aba6f85f6563db47ae214a84d0b7740c9394aa8e3c7bfef1beedd0dafda079bf" +
            "75b2ae4edb7480c18b9cdfa22e68a06c0685785f5cfb09c2b80b1d05431d0381" +
            "8400028180089a43f439b924bef3529d8d6206d1fca56a55caf52b41d6ce371e" +
            "bf07bda132c8eadc040007fcf4da06c1f30504ebd8a77d301f5a4702f01f0d2a" +
            "0707ac1da38dd3251883286e12456234da62eda0df5fe2fa07cd5b16f3638bec" +
            "ca7786312da7d3594a4bb14e353884da0e9aecb86e3c9bdb66fca78ea85e1cc3" +
            "f2f8bf0963300906072a8648ce380403032f00302c021461f6d143a47a4f7e0e" +
            "0ef9848b7f83eacbf83ffd021420e2ac47e656874633e01b0d207a99280c1127" +
            "01").HexToByteArray();

        private static readonly byte[] Dsa1024Pfx_RC2ContentEncryption = (
            "308206EE020103308206B406092A864886F70D010701A08206A5048206A13082" +
            "069D3082043706092A864886F70D010706A0820428308204240201003082041D" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E04084AF212" +
            "89D5D7E2E702020800808203F0DECCF218AC91F26BAB026998AB77C7629D20DB" +
            "E2FB7022A3C4A1CECD743C0F932E944AE229DAFB61AD76C4DEB6995DF4F4BA01" +
            "2DBAD5C63A4C846E0807FCA0BC4A162CDFBAB4B3C4D304F473B3ACC1D268436E" +
            "F537DAE97ECC3C634C8DF2A294CC23E904A169F369021A0C024A03DE98A65B0F" +
            "3F14D6910525D76AD98B91E67BB7398E245CF48A4D2A5603CFCCF4E547D7EDAB" +
            "669D9A8597C6839119EB9FD932D1E4BA8B45D3317186CDA2EFF247BCFD64A5CA" +
            "ED604BF7033E423CC21CEC6454FE3B74E03A26C51A1C3519CE339FBE9F10B81D" +
            "DF6A0AAB4F8166D90B6F52B3439AB4B5273D0A506E3E01869F8FEBD1521EF8E5" +
            "BFB357FA630E3C988926EF3ACC0A0F4176FE8A93337C1A5C6DEAB5758EC2F07C" +
            "11E8B2495ECDE58D12312CCCA2E8B2EE8564B533D18C7A26A9290394C2A9942C" +
            "295EBB0317F5695103627519567960908323FFE6560AD054C97800218A52F37A" +
            "DDE4E7F18EF3BF3718A9D7BF57B700DBEB5AB86598C9604A4546995E34DBABBB" +
            "6A9FB483A3C2DFE6046DFD54F2D7AC61C062AF04B7FBAC395C5DD19408D6926A" +
            "93B896BFB92DA6F7F5A4E54EDBE2CFBB56576878150676ADB0D37E0177B91E0D" +
            "F09D7B37769E66842DD40C7B1422127F152A165BC9669168885BA0243C9641B4" +
            "48F68575AA6AB9247A49A61AC3C683EE057B7676B9610CF9100096FC46BDC8B9" +
            "BAA03535815D5E98BA3ABC1E18E39B50A8AF8D81E30F2DFD6AF5D0F9FC3636AB" +
            "69E128C793571723A79E42FC7C1BD7F39BD45FBE9C39EEB010005435BEC19844" +
            "2058033D2601B83124BD369DADB831317E0B2C28CE7535A2E89D8A0E5E34E252" +
            "3B0FCEC34FF26A2B80566F4D86F958F70106BF3322FA70A3312E48EAA130246A" +
            "07412E93FDE91F633F758BC49311F6CBBAEC5D2F22AFCD696F72BC22E7DE6C00" +
            "3275DFEC47E3848226FE9DBA184EA711E051B267C584749F897EFE7EAFD02F1D" +
            "BF3FD8E882474CA1F45509EF2E7B82F35B677CB88ED42AF729848EE2B424B0CE" +
            "2E9AAC945BABA550C20D5B25075A30FE70D8CAA5A527A35F1DF17BCCB91930C1" +
            "7120C625667120E0806C2B51EDFF540F928BD555FB48DBCB83CCCE0C385E78C8" +
            "65BE715AE6F8BE472E5FC187EBE3FEFD8D7FE62D4DB2EE61F42D24D81FAA9179" +
            "0FB17E8EBC8E219B6F9E039F5AB3BC4870821D474B36C8F8D0583D9DC06E4383" +
            "D03424420B8C8B26276877166A0F51E22F0D8FA60A070CFBD47EAFBC717C879C" +
            "B5A1EA69C4C2A38F26A1EEF96A0C32BFCECCE4EA97E90A425066B1DD0891353F" +
            "766EB9F2BFA2563A815DAF3639EBB147E1E8757A6BFAB902C4A8F037AD47E03F" +
            "AF2E019FCF6CA7430BDFEA4B45B28ED746BB90E09BEF7B370A75E7924BBA0920" +
            "25FE654A9A197A5B8BBBE43DC7C892FF14E75A37EB97FC489AB121A43E308202" +
            "5E06092A864886F70D010701A082024F0482024B3082024730820243060B2A86" +
            "4886F70D010C0A0102A082017630820172301C060A2A864886F70D010C010330" +
            "0E0408ECB4D1550DA52C6302020800048201509322DC0193DD9E79ADAFD38827" +
            "AD6DE9299327DDDF6E9DF4FB70D53A64951E4B814E90D2A19B3F4B8E39A2F851" +
            "A3E5E9B9EB947DD248A3E5F5EB458F3323D4656709E97C6BD59238C4D1F26AB6" +
            "7D73235FAE7780D98705957B6650AC0DE3E2D46E22455D0A105D138F16A84839" +
            "14EDDF5C518B748558704ED3AE4A8C4914F667BBDE07978E4A4FC66194F6B86B" +
            "AB9F558EDE890C25DFB97C59653906CC573B5DEB62165CFF8A5F4F8059A478EB" +
            "F6FED75F1DACDC612C2E271E25A7083E15D33697270FD442D79FFCB25DB135F9" +
            "8E580DC9CE14F73C3B847931AF821C77718455F595CA15B86386F3FCC5962262" +
            "5FC916DDB4A08479DCB49FF7444333FA99FBB22F1AEC1876CF1E099F7A4ECA85" +
            "A325A8623E071EEA9359194EEE712F73076C5EB72AA243D0C0978B934BC8596F" +
            "8353FD3CA859EEA457C6175E82AE5854CC7B6598A1E980332F56AB1EE1208277" +
            "4A91A63181B9302306092A864886F70D01091531160414E6335FA7097AB6DE4A" +
            "1CDB0C678D7A929883FB6430819106092B06010401823711013181831E818000" +
            "4D006900630072006F0073006F0066007400200045006E00680061006E006300" +
            "650064002000440053005300200061006E006400200044006900660066006900" +
            "65002D00480065006C006C006D0061006E002000430072007900700074006F00" +
            "67007200610070006800690063002000500072006F0076006900640065007230" +
            "313021300906052B0E03021A0500041466FD3518CEBBD69877BA663C9E8D7092" +
            "8E8A98F30408DFB5AE610308BCF802020800").HexToByteArray();

        private static readonly byte[] Dsa1024Pfx_TripleDESContentEncryption = (
            "30820736020103308206F206092A864886F70D010701A08206E3048206DF3082" +
            "06DB308202AC06092A864886F70D010701A082029D0482029930820295308202" +
            "91060B2A864886F70D010C0A0102A082017630820172301C060A2A864886F70D" +
            "010C0103300E0408C5A45CCF58AA36E8020207D0048201505F9A735F0ECC3DB8" +
            "43A79253BA3369D331D96D153D2C160C9DCDF0AF92A2AB58D64EF48634B70510" +
            "7787365B69AFA79940932DCA1168EC2B035BEFEF317EAED68C34E4FE980D7296" +
            "587C8E19E9D9ED87803008FE8A5D033050832FE5AB9BBBD6952E6C66D916CBA9" +
            "DFEC4C01731AF8FC8D561585CC8F883578DE69E7C73F67A88229E81497556C8E" +
            "F5ED1A0C5D3F523F0552EA42E8036C60B771AE8733C973A14DC2E00AA4A1E344" +
            "3BBF4F7B03B164456D979B13EEBB377B67FE5E21776CEB3EF94D40D5D2FA6A18" +
            "19743642C55CB1DD0C4330D8C9FB25ACA8A008E9A82F18623E3C66A704B7F0C7" +
            "2FC37D71AD5ED411B75C36CB9EE2C43D98F475E4ED6D1DBAF5921E5AA074097B" +
            "3878FBA0177F19CD5E55C2564053A53507AC2833CF556DDC7C20B20E057EB465" +
            "E4028FED7BAC0CE76063CEC30E44F89E153E6834D8B32282978171C254008281" +
            "5C06CEBF305A9FFD31820106301306092A864886F70D01091531060404010000" +
            "00305B06092A864886F70D010914314E1E4C007B004100430037004500350036" +
            "00340042002D0043003900310044002D0034003800330035002D004100440035" +
            "0037002D003000420037003900300037003800320036004100340035007D3081" +
            "9106092B06010401823711013181831E8180004D006900630072006F0073006F" +
            "0066007400200045006E00680061006E00630065006400200044005300530020" +
            "0061006E00640020004400690066006600690065002D00480065006C006C006D" +
            "0061006E002000430072007900700074006F0067007200610070006800690063" +
            "002000500072006F007600690064006500723082042706092A864886F70D0107" +
            "06A0820418308204140201003082040D06092A864886F70D010701301C060A2A" +
            "864886F70D010C0103300E040856D7BE7AEDC9F41A020207D0808203E06A8DE7" +
            "017FF8B50B3833039A4DB904A31C34990D4EBDBD5B9D658CB00ED60A6C4B90B1" +
            "BE3048A0A5B847F3554625F59A9FF61C029A58C841774A88293DAFF59647C551" +
            "B10DE3FFA99ADF7FF681359DC4C49C1B65DCB76312D88AB0D324482023A523E3" +
            "0D24B094CEEF4E07C8FBCD33F28B9D098155A5DB8388BECF344C4A7F52213111" +
            "B678DC1DDDDA17124EB2C47ED370C0E6EAE9B0A1C8286D8545A8990623D9D973" +
            "D9E6341884DE3D8DAA256D3514605BBEFDA3323C6DCFE22EEA43D3A4FC74848C" +
            "AB33518CEE63D5A4BB9BA41DB0B21C105D895E1BED12BCECE7BF85A13166C01E" +
            "11CB08EF41CA1B905ED4051EE398C1F256CB98435FA92811D8D2D3F02BB1AAD1" +
            "1D5580CD870E21C9E1690D586446CBDBFFA00FE010327187DA78563916206042" +
            "1DB4B8A10ECCC3B8B077C976367894FE1BB5A5CAC7BE00132595FA75B574FC23" +
            "6DF9DCADBCE06ABA44BA7C385509C04279CFAE13B95FCE5ADBC0BBD41C492C7F" +
            "D4BC2474751C48C6216EEE901545F27D03515F738A55EF6CB7B0B4E8CD04373F" +
            "49487ECE6433C7535835D12C0970B73C148638FE19E95C621647B55A2986A317" +
            "D518EFDE4676DEBB9D03C15611EE4F6D59EF01C3E1301080B2DCF836B01DCB68" +
            "96CCA61D89D4F0CF3421198F4F3AD8175EF8CAE102DE8FD66399B11C8538F60D" +
            "7E9CF57E6DEE7AC7742C7681B52517F4EE70B23D1DBD42AA33AFE02A900CA261" +
            "2AF63FF59C9DE612D4880535709751A60FD567C457C76A0C325E6AB695E20154" +
            "C2C3B347CFD704CFCD970B10BFC961B3E4ED39004C6C906A2EFF1DE6BAD59990" +
            "F669B784E7FBF85BBD6FCB6CC74F0CE94E7D368F69EFA60B18B6702B0A540A66" +
            "EB797061074C14927E597B9097161328971AD1F2A889768A94410785A4AC01D6" +
            "62FB84A6A8D426EDF37E84D170FE448D9B3424276A2F807070A0B93C21E3D685" +
            "045231F6CA9D7DDC183DC89F88B188BC2579BD9E4154B11CA0C4224EA8F1E92B" +
            "D2DA2F02ECD4BB4FD75CC02343CAF1C73FC2009B416BF39B4652EBDB680C7B77" +
            "DB226C611A823A03B7359E52D0C60347534867DB4F69815F69863F75B0556D97" +
            "D6AF654A5C1D20964320D3716B40DFED2BEFA504B32910698240C517999DB2EB" +
            "43F29BD5613C5872CD804DE8C60AFD164E426CC6D08E78C528651560994C61DE" +
            "2A4A553B1273E3D3BA4BC9F3C29CD4FF3A5F1853FF6FE1B807035F84080A1543" +
            "1F191A52BA2186923D2D0C70751AB617A89549575052A757DABD683B85685E82" +
            "E86CDDC75FB3CBE8331306DFA311EBD627A21A8101D12486F7B0FEE116A980CD" +
            "D595EF241061FA85C2DFB4EEF4D1C5F8A176C88D92A076FF57C371116BFE8C36" +
            "819E20916A942B2B4D002FAB3EC4F9582F3A9A19C500D6028E783BFED5303B30" +
            "1F300706052B0E03021A04143F1E5723BEBF22306EBF0DB2739B5E54BD5AB4C9" +
            "0414F83662B2693DF36BA3583ABA17BDF6590C903841020207D0").HexToByteArray();

        public static readonly byte[] Dsa1024Pfx =
            PlatformSupport.IsRC2Supported
                ? Dsa1024Pfx_RC2ContentEncryption
                : Dsa1024Pfx_TripleDESContentEncryption;

        // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Unit test password.")]
        public const string Dsa1024PfxPassword = "1234";

        public static byte[] Dsa1024Cert = (
            "3082038D3082034AA003020102020900AB740A714AA83C92300B060960864801" +
            "650304030230818D310B3009060355040613025553311330110603550408130A" +
            "57617368696E67746F6E3110300E060355040713075265646D6F6E64311E301C" +
            "060355040A13154D6963726F736F667420436F72706F726174696F6E3120301E" +
            "060355040B13172E4E4554204672616D65776F726B2028436F72654658293115" +
            "30130603550403130C313032342D62697420445341301E170D31353131323531" +
            "34343030335A170D3135313232353134343030335A30818D310B300906035504" +
            "0613025553311330110603550408130A57617368696E67746F6E3110300E0603" +
            "55040713075265646D6F6E64311E301C060355040A13154D6963726F736F6674" +
            "20436F72706F726174696F6E3120301E060355040B13172E4E4554204672616D" +
            "65776F726B2028436F7265465829311530130603550403130C313032342D6269" +
            "7420445341308201B73082012C06072A8648CE3804013082011F02818100AEE3" +
            "309FC7C9DB750D4C3797D333B3B9B234B462868DB6FFBDED790B7FC8DDD574C2" +
            "BD6F5E749622507AB2C09DF5EAAD84859FC0706A70BB8C9C8BE22B4890EF2325" +
            "280E3A7F9A3CE341DBABEF6058D063EA6783478FF8B3B7A45E0CA3F7BAC9995D" +
            "CFDDD56DF168E91349130F719A4E717351FAAD1A77EAC043611DC5CC5A7F0215" +
            "00D23428A76743EA3B49C62EF0AA17314A85415F0902818100853F830BDAA738" +
            "465300CFEE02418E6B07965658EAFDA7E338A2EB1531C0E0CA5EF1A12D9DDC7B" +
            "550A5A205D1FF87F69500A4E4AF5759F3F6E7F0C48C55396B738164D9E35FB50" +
            "6BD50E090F6A497C70E7E868C61BD4477C1D62922B3DBB40B688DE7C175447E2" +
            "E826901A109FAD624F1481B276BF63A665D99C87CEE9FD063303818400028180" +
            "25B8E7078E149BAC352667623620029F5E4A5D4126E336D56F1189F9FF71EA67" +
            "1B844EBD351514F27B69685DDF716B32F102D60EA520D56F544D19B2F08F5D9B" +
            "DDA3CBA3A73287E21E559E6A07586194AFAC4F6E721EDCE49DE0029627626D7B" +
            "D30EEB337311DB4FF62D7608997B6CC32E9C42859820CA7EF399590D5A388C48" +
            "A330302E302C0603551D110425302387047F0000018710000000000000000000" +
            "0000000000000182096C6F63616C686F7374300B060960864801650304030203" +
            "3000302D021500B9316CC7E05C9F79197E0B41F6FD4E3FCEB72A8A0214075505" +
            "CCAECB18B7EF4C00F9C069FA3BC78014DE").HexToByteArray();

        public static byte[] CertWithEnhancedKeyUsage = (
            "308202BD308201A5A00302010202106144CAE1C8BF3049BB9BF62C08B74F1330" +
            "0D06092A864886F70D01010B0500300E310C300A06035504031303466F6F301E" +
            "170D3135303330313232343735385A170D3136303330313034343735385A300E" +
            "310C300A06035504031303466F6F30820122300D06092A864886F70D01010105" +
            "000382010F003082010A0282010100C2B02AED5493FDACE7F0D91118D3433F56" +
            "A1320202CCABA9C4A78C8F6B60BC896F6E579E39453C2392C9C12AFC57C0D81E" +
            "ACAA92186921A52FE3EC36769780EE60776CB3EEF1327197C05C1D539C105DBF" +
            "DD3762C941E892F838DDFA87D0DD5642A1AC526E46CF4FF6F450ABD970189DC6" +
            "90ABE886DB65F3B34F7F2F2C60A59A2F85E531C54D4B182878D14D79A9B88EB3" +
            "29F14FFCEBBFA0DFDC97D3609C55F0A0D1D242E53B7143ACC64ABB7D985A1E43" +
            "9EF3EF0A0F8074574CE48069E0499F9C23F4E496373C0F18EE7AFCA08BCE5993" +
            "78637AAE51D2C57793FA899047BEA820D7E3FF1B935A32023BA13234F7D725C7" +
            "A8F3C9C16757E19F4710DEE2C0B2E10203010001A317301530130603551D2504" +
            "0C300A06082B06010505070303300D06092A864886F70D01010B050003820101" +
            "0062A51D36C1E2945D4A7BC80385961084C2A600391D8A8D4F14572A0E994062" +
            "6C4071B7533C2FB8894B55AFCEEE6C59E697244D43D136336DF04A595BB7C09F" +
            "A2885C43926AA61BECA482EA5A4F62279C123206C8908F299DC41A62C80ECE98" +
            "A36D1F12FEC80919E8DEB209C5B02EBD6D98BE00A84CB61781DC3C1585D79C91" +
            "0490365DEB5B96899B1469B930CB1CE6874E256B45573F43D7DD16BD4F590B9A" +
            "EEAA2E64987B64C6A96DDA5E6CA2EF96357B78ABFAF815DF4D021A15980EC7D9" +
            "ECE244B1ADDF8878FC15236101BFA8E804ADA7C36DE088A16E3BEAABB65D1A91" +
            "BA7010C7E3AA701A03B843A93AC1A291A791B3DBEEB511CBF73E2252691CBA33" +
            "E2").HexToByteArray();

        public static byte[] CertWithPolicies = (
            "308201f33082015ca0030201020210134fb7082cf69bbb4930bfc8e1ca446130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232343735385a170d3136303330313034343735385a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100c252d52fb96658ddbb7d19dd9caaf203ec0376f77c" +
            "3012bd93e14bb22a6ff2b5ce8060a197e3fd8289fbff826746baae0db8d68b47" +
            "a1cf13678717d7db9a16dab028927173a3e843b3a7df8c5a4ff675957ea20703" +
            "6389a60a83d643108bd1293e2135a672a1cff10b7d5b3c78ab44d35e20ca6a5c" +
            "5b6f714c5bfd66ed4307070203010001a3523050301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b060357080902010302010230150603551d20040e300c3004060262133004" +
            "06027021300d06092a864886f70d0101050500038181001be04e59fbea63acfb" +
            "c8b6fd3d02dd7442532344cfbc124e924c0bacf23865e4ce2f442ad60ae457d8" +
            "4f7a1f05d50fb867c20e778e412a25237054555669ced01c1ce1ba8e8e57510f" +
            "73e1167c920f78aa5415dc5281f0c761fb25bb1ebc707bc003dd90911e649915" +
            "918cfe4f3176972f8afdc1cccd9705e7fb307a0c17d273").HexToByteArray();

        public static byte[] CertWithTemplateData = (
            "308201dc30820145a00302010202105101b8242daf6cae4c53bac68a948b0130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232333133395a170d3136303330313034333133395a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100a6dcff50bd1fe420301fea5fa56be93a7a53f2599c" +
            "e453cf3422bec797bac0ed78a03090a3754569e6494bcd585ac16a5ea5086344" +
            "3f25521085ca09580579cf0b46bd6e50015319fba5d2bd3724c53b20cdddf604" +
            "74bd7ef426aead9ca5ffea275a4b2b1b6f87c203ab8783559b75e319722886fb" +
            "eb784f5f06823906b2a9950203010001a33b3039301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b0603570809020103020102300d06092a864886f70d010105050003818100" +
            "962594da079523c26e2d3fc573fd17189ca33bedbeb2c38c92508fc2a865973b" +
            "e85ba686f765101aea0a0391b22fcfa6c0760eece91a0eb75501bf6871553f8d" +
            "6b089cf2ea63c872e0b4a178795b71826c4569857b45994977895e506dfb8075" +
            "ed1b1096987f2c8f65f2d6bbc788b1847b6ba13bee17ef6cb9c6a3392e13003f").HexToByteArray();

        public static byte[] ComplexNameInfoCert = (
            "308204BE30820427A00302010202080123456789ABCDEF300D06092A864886F70" +
            "D01010505003081A43110300E06035504061307436F756E747279310E300C0603" +
            "550408130553746174653111300F060355040713084C6F63616C6974793111300" +
            "F060355040A13084578616D706C654F31123010060355040B13094578616D706C" +
            "654F55311E301C06035504031315636E2E6973737565722E6578616D706C652E6" +
            "F72673126302406092A864886F70D0109011617697373756572656D61696C4065" +
            "78616D706C652E6F7267301E170D3133313131323134313531365A170D3134313" +
            "231333135313631375A3081A63110300E06035504061307436F756E747279310E" +
            "300C0603550408130553746174653111300F060355040713084C6F63616C69747" +
            "93111300F060355040A13084578616D706C654F31123010060355040B13094578" +
            "616D706C654F55311F301D06035504031316636E2E7375626A6563742E6578616" +
            "D706C652E6F72673127302506092A864886F70D01090116187375626A65637465" +
            "6D61696C406578616D706C652E6F7267305C300D06092A864886F70D010101050" +
            "0034B003048024100DC6FBBDA0300520DFBC9F046CC865D8876AEAC353807EA84" +
            "F58F92FE45EE03C22E970CAF41031D47F97C8A5117C62718482911A8A31B58D92" +
            "328BA3CF9E605230203010001A382023730820233300B0603551D0F0404030200" +
            "B0301D0603551D250416301406082B0601050507030106082B060105050703023" +
            "081FD0603551D120481F53081F28217646E73312E6973737565722E6578616D70" +
            "6C652E6F72678217646E73322E6973737565722E6578616D706C652E6F7267811" +
            "569616E656D61696C31406578616D706C652E6F7267811569616E656D61696C32" +
            "406578616D706C652E6F7267A026060A2B060104018237140203A0180C1669737" +
            "375657275706E31406578616D706C652E6F7267A026060A2B0601040182371402" +
            "03A0180C1669737375657275706E32406578616D706C652E6F7267861F6874747" +
            "03A2F2F757269312E6973737565722E6578616D706C652E6F72672F861F687474" +
            "703A2F2F757269322E6973737565722E6578616D706C652E6F72672F308201030" +
            "603551D110481FB3081F88218646E73312E7375626A6563742E6578616D706C65" +
            "2E6F72678218646E73322E7375626A6563742E6578616D706C652E6F726781157" +
            "3616E656D61696C31406578616D706C652E6F7267811573616E656D61696C3240" +
            "6578616D706C652E6F7267A027060A2B060104018237140203A0190C177375626" +
            "A65637475706E31406578616D706C652E6F7267A027060A2B0601040182371402" +
            "03A0190C177375626A65637475706E32406578616D706C652E6F7267862068747" +
            "4703A2F2F757269312E7375626A6563742E6578616D706C652E6F72672F862068" +
            "7474703A2F2F757269322E7375626A6563742E6578616D706C652E6F72672F300" +
            "D06092A864886F70D0101050500038181005CD44A247FF4DFBF2246CC04D7D57C" +
            "EF2B6D3A4BC83FF685F6B5196B65AFC8F992BE19B688E53E353EEA8B63951EC40" +
            "29008DE8B851E2C30B6BF73F219BCE651E5972E62D651BA171D1DA9831A449D99" +
            "AF4E2F4B9EE3FD0991EF305ADDA633C44EB5E4979751280B3F54F9CCD561AC27D" +
            "3426BC6FF32E8E1AAF9F7C0150A726B").HexToByteArray();

        private static readonly byte[] MultiPrivateKeyPfx_RC2ContentEncryption = (
            "30820F1602010330820ED606092A864886F70D010701A0820EC704820EC33082" +
            "0EBF308206A806092A864886F70D010701A08206990482069530820691308203" +
            "4C060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E0408ED42EEFCD77BB2EB020207D00482029048F341D409492D23" +
            "D89C0C01DEE7EFFB6715B15D2BB558E9045D635CADFFFEC85C10A4849AB0657D" +
            "A17FE7EC578F779BA2DC129FA959664DC7E85DFD13CAC673E487208FE457223A" +
            "75732915FFCF3FF70F557B0846D62AD507300EA1770EDED82F7D8E6E75075728" +
            "A29D3BF829E75F09EF283A9DDEDDFBABC2E25698DA8C24E4FE34CD43C87554BF" +
            "55B1D4B2B0979F399AEC95B781C62CBE9E412329F9A9BCABF20F716A95F1D795" +
            "7C379A27587F6BBFA44A0B75FAAC15CA3730629C55E87990EE521BC4657EE2A4" +
            "41AF099A226D31707685A89A28EB27CA65512B70DEC09231369AA1A265D4F5C3" +
            "C5D17CB11DB54C70AB83EA28F4740D1F79D490F46F926FB267D5F0E4B2FE096D" +
            "F161A4FF9E9AC068EFCA999B3ED0A3BD05D8D1E3B67CF51E6A478154B427D87D" +
            "C861D0FE2A7A42600483D7B979DC71E8A00D0E805E3BB86E8673234DC1D14987" +
            "99272754A5FD5FEC118CF1E2B2A539B604FED5486A4E4D73FAAFF69023263B84" +
            "6870D6B8DB01E31CB3A1E4BA3588C1FA81C786745A33B95573D5381AB307827A" +
            "549A36AF535FD05E1247BB92C6C6FCB0E76E87F2E4C8136F37C9C19BE3001F59" +
            "FC5CB459C620B8E73711BF102D78F665F40E4D1A341370BC1FB7A5567C29359C" +
            "FFB938237002904BE59F5605AF96E8A670E2248AB71D27FE63E327077144F095" +
            "4CA815E0284E2FF5E1A11B2946276A99B91BF138A79B057436798AF72FD86842" +
            "881C5A5ECDA8A961A21553CC930703047F1F45699CEFEF26AAB6B7DBC65C8C62" +
            "4CA3286094596F2AA48268B9F5411058613185507332833AFB312D5780CEFF96" +
            "6DD05A2CB6E1B252D9656D8E92E63E6C0360F119232E954E11DE777D2DE1C208" +
            "F704DDB16E1351F49B42A859E3B6B2D94E1E2B3CD97F06B1123E9CCA049201E6" +
            "DB7273C0BDE63CC9318182301306092A864886F70D0109153106040401000000" +
            "306B06092B0601040182371101315E1E5C004D006900630072006F0073006F00" +
            "66007400200045006E00680061006E0063006500640020004300720079007000" +
            "74006F0067007200610070006800690063002000500072006F00760069006400" +
            "650072002000760031002E00303082033D060B2A864886F70D010C0A0102A082" +
            "02B6308202B2301C060A2A864886F70D010C0103300E04081F85B7ED57F6F934" +
            "020207D00482029051A5ADA683AAE06A699761CCF05CB081A4398A7B1256A250" +
            "84DBE1115BFAB07A5A9146BC22F2E4223FF25BCA1836AE218691815F20A27A1B" +
            "98D1FC78F84AFA7E90A55954EE5BEA47FFA35928A990CB47346767F6F4212DBC" +
            "D03FFF1E4D137979006B46B19A9FC3BC9B5036ED6F8582E2007D08DB94B2B576" +
            "E154719CAC90DFB6F238CA875FCBEBCF9E9F933E4451E6A2B60C2A0A8A35B5FD" +
            "20E5DDA000008DCCE95BBDF604A8F93001F594E402FF8649A6582DE5901EDF9D" +
            "ED7D6F9657C5A184D82690EFCFB2F25BFCE02BC56F0FF00595996EBF1BA25475" +
            "AB613461280DD641186237D8A3AB257BD6FB1BDC3768B00719D233E0D5FD26D0" +
            "8BA6EAB29D732B990FB9423E643E4663ABBA0D8885DD2A276EE02C92778261C7" +
            "853F708E2B9AF8D2E96416F676D0191BD24D0C8430BD419049F43C8E2A0C32F8" +
            "62207B3DA661577CE5933460D0EF69FAD7323098B55FEF3A9955FE632FBCE845" +
            "2BB5F3430AE2A9021EBF756CC7FDFC3E63581C8B0D7AB77760F447F868B59236" +
            "14DAA9C36AEBC67DC854B93C38E8A6D3AC11B1EE1D02855CE96ADEB840B626BF" +
            "C4B3BFD6487C9073F8A15F55BA945D58AD1636A7AED476EBDB5227A71144BF87" +
            "45192EF5CD177818F61836717ED9EB0A83BEEE582ADEDD407035E453083B17E7" +
            "C237009D9F04F355CEAB0C0E9AD6F13A3B54459FA05B19E02275FE2588258B63" +
            "A125F549D1B44C827CDC94260A02F4A1B42A30E675B9760D876685D6CA05C258" +
            "03BDE1F33D325CF6020A662B0F5DCCC8D77B941B273AC462F0D3E050CEB5AEF7" +
            "107C45372F7063EF1AB420CA555A6C9BE6E1067966755584346CDDE7C05B6132" +
            "E553B11C374DB90B54E5C096062349A1F6CB78A1A2D995C483541750CFA956DE" +
            "A0EB3667DE7AD78931C65B6E039B5DE461810B68C344D2723174301306092A86" +
            "4886F70D0109153106040402000000305D06092B060104018237110131501E4E" +
            "004D006900630072006F0073006F006600740020005300740072006F006E0067" +
            "002000430072007900700074006F006700720061007000680069006300200050" +
            "0072006F007600690064006500723082080F06092A864886F70D010706A08208" +
            "00308207FC020100308207F506092A864886F70D010701301C060A2A864886F7" +
            "0D010C0106300E04089ADEE71816BCD023020207D0808207C851AA1EA533FECA" +
            "BB26D3846FAEE8DEDB919C29F8B98BBBF785BC306C12A8ACB1437786C4689161" +
            "683718BB7E40EB60D9BE0C87056B5ECF20ACCB8BF7F36033B8FCB84ED1474E97" +
            "DE0A8709B563B6CF8E69DF4B3F970C92324946723C32D08B7C3A76C871C6B6C8" +
            "C56F2D3C4C00B8A809E65A4EB5EFECC011E2B10F0E44ECDA07B325417B249240" +
            "80844F6D7F1F6E420346EA85825EB830C7E05A5383412A9502A51F1AC07F315A" +
            "DE357F1F9FB2E6427976E78B8FF9CD6C2F9841F2D84658AC8747694EFD0C451B" +
            "7AC5B83D5F0780808417501666BB452B53CEB0698162D94541DE181A7968DB13" +
            "9F17A1076EDEB70B38B8881DBC6DE2B694070A5A1AA71E4CDFBF7F4D5DBCF166" +
            "46768364D3C74FA212E40CBE3BE7C51A74D271164D00E89F997FD418C51A7C2D" +
            "73130D7C6FCAA2CA65082CE38BFB753BB30CC71656529E8DBA4C4D0B7E1A79CF" +
            "2A052FFEFA2DEE3373115472AFD1F40A80B23AA6141D5CDE0A378FE6210D4EE6" +
            "9B8771D3E192FD989AEC14C26EA4845D261B8A45ABC1C8FA305449DCDEDA9882" +
            "DD4DDC69B2DE315645FBC3EE52090907E7687A22A63F538E030AB5A5413CA415" +
            "F1D70E70CB567261FB892A8B3BAFC72D632CD2FDCC0559E01D5C246CC27C9348" +
            "63CCFA52490E1F01D8D2D0AF2587E4D04011140A494FFA3CA42C5F645B94EE30" +
            "100DE019B27F66FFC035E49A65B2A3F6CB14EB1E2FFF1F25B5C87481BD8506F3" +
            "07E0B042A2C85B99ECA520B4AAC7DFF2B11C1213E4128A01765DDB27B867336B" +
            "8CCF148CE738465D46E7A0BEA466CD8BBCCE2E11B16E0F9D24FF2F2D7C9F8527" +
            "79ADBB818F87E4AFF7C21A9C2BC20D38209322A34B0B393B187C96583D3D73D9" +
            "440F994B2F320D3274848AB7167942179CFF725C2C7556CCC289A5E788C5B863" +
            "E6FCDD5E4B87E41458BEB3F43D14C7E5196C38CA36322F8B83064862178D5892" +
            "5AEF34F444A31A4FB18431D7D37C65ED519643BC7BD025F801390430022253AA" +
            "FCEA670726512C3532EA9F410DB8AA6628CC455E4AB3F478A6981DB9180B7A2A" +
            "24B365F37554CE04B08F22B3539D98BF9A1AC623BBF9A08DBEC951E9730C1318" +
            "02B2C40750AAE6A791B3219A96A5BAC7AE17A2F7EA02FF66D6FB36C2E6B6AB90" +
            "D821A6322BF3E8D82969756A474551DB9EAA8C587FC878F996F5FA1E1C39E983" +
            "F164B0A67897EB3755C378807FFDFE964C5C0F290784A08E8C925E85775A9B89" +
            "2E278F68C3C1DE72622AC10EA56D88C909EF4AC9F47ED61376737C1E43DBF0F8" +
            "9337F0684FA0B96E7A993EC328A6A5FBCDCB809ACBFDAE4ECE192A45480104ED" +
            "12820238AB6AC9C88CC9A82585FD29A81A7BC5BC591738A4D49A86D06B4E18BD" +
            "C83DFFAA60D8A0D4F70CC63D4E83812CB6753F3744545592D04223793E5B3051" +
            "25AAD8807A753D235769BD0280E2DE808B0CEE2B98B0F5562FF9EF68161A6B7E" +
            "08C8B105766EBCFC44AC858B1A89E34C099B194A8B24D1DBABC13909EFAF5B9A" +
            "9E77AEAF7DD9BE772FA01AB9518EB8864AE6D07D7DD7451797541D2F723BC71A" +
            "9C14ED1D811594E2C4A57017D4CB90FD82C195FA9B823DF1E2FFD965E3139F9A" +
            "6E8AAC36FA39CFA4C52E85D2A661F9D0D466720C5AB7ECDE968FF51B535B019A" +
            "3E9C76058E6F673A49CDD89EA7EC998BDADE71186EA084020A897A328753B72E" +
            "213A9D82443F7E34D94508199A2A63E71A12BD441C132201E9A3829B2727F23E" +
            "65C519F4DA2C40162A3A501B1BD57568ED75447FEAF8B42988CE25407644BFA0" +
            "B76059D275EC994BB336055E271751B32233D79A6E5E3AA700F3803CCA50586D" +
            "28934E3D4135FA043AF7DFAB977477283602B1739C4AF40E3856E75C34EB98C6" +
            "9A928ADE05B67A679630EFA14E64B2957EDD1AB4EC0B0E7BC38D4851EBF67928" +
            "33EACB62FB6C862B089E3066AE5EAAFD2A8B7FC712DE9BD2F488222EEB1FB91B" +
            "4E57C2D24092818965621C123280453EDCFA2EC9D9B50AFA437D1ED09EC36FD2" +
            "32B169ED301E0DB0BABE562B67130F90EBC85D325A90931A5B5A94736A4B3AAD" +
            "B8CA295F59AF7FF08CCFADE5AFBBC2346BC6D78D9E5F470E9BDFF547F2574B10" +
            "A48DD9D56B5B03E9E24D65C367B6E342A26A344111A66B1908EDAECD0834930D" +
            "A74E1CFE2E4B0636A7C18E51A27AD21992A2DCF466BAACAC227B90B5E61BED79" +
            "9C97DEE7EDB33CCAF5DAD7AAD3CACCDE59478CF69AE64B9065FCB436E1993514" +
            "C42872DD486ABB75A07A4ED46CDF0E12C0D73FAB83564CF1A814791971EC9C7C" +
            "6A08A13CE0453C2C3236C8B2E146D242E3D37A3ECF6C350D0B2AB956CB21057F" +
            "DC630750A71C61C66DE3D4A6DB187BEE2F86DEB93E723C5943EA17E699E93555" +
            "756920416BD6B267A4CFAC4EE90E96A6419302B4C0A3B9705509CA09EE92F184" +
            "FD2817BA09BE29E465909DB6C93E3C1CAF6DC29E1A5838F3C32CCB220235EF82" +
            "9CD21D1B3E960518A80D08AE7FF08D3AFB7451C823E9B8D49DAF66F503E4AE53" +
            "99FECFC958429D758C06EFF8338BC02457F6FE5053AA3C2F27D360058FD93566" +
            "3B55F026B504E39D86E7CE15F04B1C62BBFA0B1CA5E64FF0BD088D94FB1518E0" +
            "5B2F40BF9D71C61FC43E3AF8440570C44030F59D14B8858B7B8506B136E7E39B" +
            "B04F9AFEAF2FA292D28A8822046CEFDE381F2399370BDE9B97BC700418585C31" +
            "E9C353635ADAA6A00A833899D0EDA8F5FFC558D822AEB99C7E35526F5297F333" +
            "F9E758D4CD53277316608B1F7DB6AC71309A8542A356D407531BA1D3071BA9DC" +
            "02AE91C7DF2561AEBC3845A118B00D21913B4A401DDDC40CE983178EF26C4A41" +
            "343037301F300706052B0E03021A041454F0864331D9415EBA750C62FA93C97D" +
            "3402E1A40414B610EC75D16EA23BF253AAD061FAC376E1EAF684").HexToByteArray();

        private static readonly byte[] MultiPrivateKeyPfx_TripleDESContentEncryption = (
            "30820FD502010330820F9106092A864886F70D010701A0820F8204820F7E3082" +
            "0F7A3082076306092A864886F70D010701A0820754048207503082074C308203" +
            "A9060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E04081DFDCF1BC66474CA020207D0048202905D9D7D0A63C6AF2A" +
            "5746A7601547EA1687B31CB19EE4858857728FF10A6D08C57B2DA97308F95E4E" +
            "682DAEF4954CCAA26B41E6BF1EB28CB9FB8E7A65B5CE8F3D2A3A9022390BEC0F" +
            "0197DAFB37DB8C029C31944311C96CEE14A1863830DF6A171EA39525AFB8E5E9" +
            "654279F3A9575A70D9BF4DE4D8B43C57FEE8BFFEE25F4912E00FD0F4A0A910D4" +
            "5B01CD3B0255D2BB720D982E94CCA5E5CF8D82A9B820FFAF34758FA455A90CC8" +
            "17D1C1AD6407AC5B287C1D77FAEC59624C5604DEB0F70CB9EB61B29C3F1E46BE" +
            "2C3BDB052DFA8120C2F84F98AB43E6AEC558F0C3575761159DE0A9EFE1CECDBB" +
            "76D644C39A3894C77D3230C32D097EA456A0EE97696F573BEBD3833DBFF37A53" +
            "83E80724E15FA0F35B32FD035A97CD2E1A2511DAA6D0A1B224250E71008C5F79" +
            "75FE97EBF3F27864789ABF85BC9EC506CF5825D51FBAC8E0AEEEB9C7E9F6F7AF" +
            "12284E0922CF3123864C5C627F98E2F11F69872E2DC4B838C3D8A283489ACFAB" +
            "FAEDEB3432AA721343FC75800E8FA8EE58C72F5439D4191ADEE8101FB8939C18" +
            "9CAF4453C46B6F2CD9E4BCD5B6DC643BEF35C1999D2AAC14E35877CF0B784BDF" +
            "EE841478C7B031EDEC4B2BB50C30946B25462204059FC963464A3342C69166A2" +
            "0221F46A231477A7DF73003CC546287C9B4AA80484B14DD65ECE1BB5F1AB48E3" +
            "B331642455325B51E111A0F80AA89E083C29D2AC672B28C768696D7648A8DCCF" +
            "DBED389CD303A8B9D6E50A6B4A46DE9D703B44759AC3826DC511F38CF914E224" +
            "C6E8D63550AE8C3DD49968BED6A24AD16685CFE3028BC68BA2B28C83FF49C5B2" +
            "56D70F9CF794F7DF1CD9EB24E5FB32C5337AD5B3797C6A38A612E9FF0497295E" +
            "D8F4386BA8026C8EF965DA756AFC1BFEABE30B5111533D9CC1D4DC0AC3AAB6A1" +
            "5B42007A6999C6CB3181DF301306092A864886F70D0109153106040401000000" +
            "305B06092A864886F70D010914314E1E4C007B00300031003500350038003400" +
            "350032002D0034003900350031002D0034004200380035002D00390037004300" +
            "36002D003400430044004100410032004400340045004600360046007D306B06" +
            "092B0601040182371101315E1E5C004D006900630072006F0073006F00660074" +
            "00200045006E00680061006E006300650064002000430072007900700074006F" +
            "0067007200610070006800690063002000500072006F00760069006400650072" +
            "002000760031002E00303082039B060B2A864886F70D010C0A0102A08202B630" +
            "8202B2301C060A2A864886F70D010C0103300E040804A1143A86597BF6020207" +
            "D0048202904FE5224D130BF081597DC200F8AF9DCCFC33532DDF2A3B3B3D49E4" +
            "DA4DADEF8289EC041382C56CDB204637ECCD7CBEA9B6B1DDBCE4F7EFF5464179" +
            "AB975EF4DEA32C5E97ED12698F641B8BCF06F3FD91263BE4DAFB51E16236F39A" +
            "822677465ADAAF363CCE47F68C3DECDE44261EFEDC7F6830551C55FD3851276C" +
            "6CC03CBF67E9EC201CC3C287CDF1C6459A0E8A01B44D6FAC4128D02664501433" +
            "5201E98CFDC90A4D25BE14CAF927FC01C033F2A48A3AE2A545EC8CB1E5A7929C" +
            "BBFBD026642890F95EB3060576DE13D545B862BA87F25F75981ECCE9C23C2404" +
            "E183B03F32CC1F3B1B6763F3A1EF6E02F25DFF0575516EC9473D718803ABD19A" +
            "DE87025B458AE6C8D00FAF2C3CA8982B588E68EFF507630082A6237A2A4ABB13" +
            "462CE9D30469250FC34A80CE54E93F48CC977D25845A1FE058D329FA172A456C" +
            "CAB46CFDC6BE3B75BA2FE369A8F72EE502226B37E9D0D43AFB46FD4995AC960C" +
            "7F3189C0BA19409B2D9ED43716BD45FC10BCB831CF972B6A9511D6C13AF205E5" +
            "7729D392DC9F0FA8C2602FF50B94E095F9CBBB8C1191682C3927F4DE717096D6" +
            "C04B9A3C9D7785FD048547AFC2BA9FBA3D4F590EE23FC491305D1BB24B339B71" +
            "E3C217A47F3B4BDF952B7BDF2865639E8B112AEC8BA0AE4D45DBB9D8FD1661E0" +
            "69139083F5C538A0081D319265BED9079CC11BABAC92456BC7770D128FA7D32C" +
            "D50C69FA5253B9BEAA5EDA3F829F14122FC0936FE2A6D7CC3D8B9E140B139FF0" +
            "33AEA5CF58353216602B6213A5B4E09B66FF95F56484F46A4378F4A6EE911597" +
            "148FDC6230CF7BE99D4FE71298BE3CD21C7F50AF10102019BE7961B4AE3A4288" +
            "429B24C62896F01914E586F508311016CD8F402F38E8D6EA82E3C8BCA7A21064" +
            "B8D3DF80C1587667A45431E92C14E919A83270DE503181D1301306092A864886" +
            "F70D0109153106040402000000305B06092A864886F70D010914314E1E4C007B" +
            "00330037003100410033003000390037002D0034003900340045002D00340033" +
            "00460036002D0038003600370044002D00310036004400310033003900310031" +
            "0037004600460034007D305D06092B060104018237110131501E4E004D006900" +
            "630072006F0073006F006600740020005300740072006F006E00670020004300" +
            "72007900700074006F0067007200610070006800690063002000500072006F00" +
            "7600690064006500723082080F06092A864886F70D010706A0820800308207FC" +
            "020100308207F506092A864886F70D010701301C060A2A864886F70D010C0103" +
            "300E0408DE5DDA46C47766A4020207D0808207C815E126FB76D95BC41340EC9B" +
            "7C2B94D1AE77D6345E34B16530769596DE511E87E7E00590A967E4B8B6AD284B" +
            "3BB3BF1E132A35018F644F5A04F307F993EAEBB8ED5AAB497602570451C404F7" +
            "07EB6AA955A29363807B8C9762B9FA9D4F7720FFA941E6E7A456193E04ACB52B" +
            "691F44DF2BF55D91A3A26752120DDDB61044A3FFFADE602337E5419916091B31" +
            "0E2C40ED2905E32351A6C4FA77CB3D0BC802FC852986FEE41247A41ED49E2F27" +
            "3516276D8A385C2332C171FB9BF9AB8EE17EF9914BCEDA9C062E3C235B165916" +
            "10272829707F077577822C72ED516E8890671AD92CB266AC60B1F124E0A49EB1" +
            "E65C38DC192AED49DC9E62B1435FA0AA2E25BE8455392327E5E0833CC189E47C" +
            "406D11A2BC7DA9207E70F719C7E7DC636D119831AEB6740BC3397A9629505E71" +
            "3623B18168BBE42179AF7C3865A7FA195EC79A4C1707B4755852A7D57E8FF797" +
            "41883CE2F0D088C9F0ADB725D79C16D9FA474F540A6B9E461369D1AFF887DC1E" +
            "0A000D5B9ADC6A39753CD9B9D98337417408AA6B5B3B9264D0BE7CF5B085100D" +
            "81A7B22EE30036FB5A4D18DED4053C120930AD3004B118BF371A5D5288B71096" +
            "6D45F41DB5CC9446AB7AE8C91D7CBB5C162793F5335F1D4EAF46045CD85C9CF5" +
            "CF7CBE7797F181CE2BD1E45CF747697F91450975B3CB797A58833F91F332C75D" +
            "FB17795C6C7C65EE5584B6C7A8B2C18133E2541FF145C46B823F5074CED24187" +
            "CD1872F38ECF36EFBDFE090B8F1675A3D4B810DDD2A267F5F49892796E5B1335" +
            "6CF71BD0DAF467C8A7A5FC9BC18E566FCD891DA69C8C0C1FFD5F534FA01EB47B" +
            "166AFDD13ACE0BDE1FCC2F38F7F1371AEE2010B08F163C2310B3F922423B15F5" +
            "50928EE09E9501CE887B532688FB598C1A1CD6C674587C54B4912EE8334268AC" +
            "83BBFA02FB226288ACDAD2F53144076D1AE2A039834BBE8DA2DDA61FFBCA744D" +
            "56591786C784159EE1D79A239472303EC73F7972819D762DD35B2BC2FB9C852B" +
            "637E91149269F64CDB5243962DEDE1724D37A4D1AB9B928DA19E5BB118BBFBAC" +
            "82F02A1D732A65E57C1F3E62FCE6B3D3861E6737582122CE37A99E9D44403EDE" +
            "CB94AE00350758A8449641B3F0C97D630C3ABAF1A8A65ED22E847078FDA55026" +
            "013B2B23F50006CC884F4818060B20D1EB572CB81D03E896EC29E358A94EC263" +
            "39D58F2CFED79ABD7D50276ADAF978A463A38306420EE2C420F77B145FC0DCBF" +
            "B6E324303BDD43E59466EB4FD98BB2494061A9E67DFC7CC4AA909375FAA446AD" +
            "D7A95890A4969A017A02C3C84F9FF60F12F8F041E1AB3D0F1C4939134E894D0E" +
            "58D9BEBF95732D7420F24E25A0DAB3FA7BC8F2016E8C78F65A50B5B86B8060CE" +
            "34C36E59FF7DB623342BC20F5D99AD4FA064687B941CCFEB2B335A836DAE7320" +
            "8435C4ED8C3629BC08D64C0BEEC00007D033A8E43371FEC83FCE50A87E3E4C5A" +
            "5C8ABCD5AE53887A679E8A3065D820836237EA3D681AF268DB1FDE204CF2B293" +
            "37D14D9B3933AEA923A454E91455109831DCEDC9ABDCA2A2970B35A9D3EAAC90" +
            "9D0F39FAA9433E4857D84CC84395E4B92C48A6B207DA1FD4B0BF936BBE739106" +
            "8EBF80B579C00FE3937C468B86137568FA4BD85DEF102FBF8FFCBDFF9B1BD23F" +
            "A5647EF97A6F666AF3635C650382BC9186E3481D05AFFFC0B0F83CC761ADDFC1" +
            "6044488A083F5CA0DD604118423F8F84BA643D146A0E801C072D6B0CC1B19D5B" +
            "71E849707DA5AC8098C19377A7AA96DA53CC0DDA0650A3F200135236DA9D80A9" +
            "41BF14F1C61DE7C80E2E2281EA238EBF4E0B1FEFA1AFF4C37822EF252E784F75" +
            "635166E7C0D8DB07759264FA5DCE273B4D9773A42D020BA3BEBD6CA825F4E8E3" +
            "E743CB88DFF6FAFABF40FF1BD8E7212687C9E283ACC9CED6C8DF1CA0C670FEE1" +
            "F5621A1F1A2794043A3D3BD1B2FE2855B4C0C189A8127E4653D78D4D75F4E51B" +
            "3944B167C826022A7FF46865BC816F8346BD6704CCF4CF8C87FF32EE08A06386" +
            "30D7D8579593805B26E15C682D7AA9AD3C7D5A7BF8894BEA20530473F06E5D89" +
            "8986BDB1FCBAE6013B10875F39980FC53CA3D7ACB3286CE194A6E04F8C62F2B2" +
            "92485AD53BB9E9387563782AD618C0C94CE8514EA49A9C4456F6D6ED97920A85" +
            "B835919BEEF4252159423980078ACE601F9D58399FD9824AAAFD5C6AF92BF410" +
            "27C9DD077E007F0FC4D14F9324531716F6B2A54B7E8BDE6222FF68D9A9571F31" +
            "45D6916ACC13F52F2CAB6421F80BA630E0EDD980E214F3D185959868858109D2" +
            "613239A9891FC7F5C19A0E661F41C27D62DD502D7FDC6311824330354D9A7C76" +
            "8C98562BE80213A71C3E054EBC5305915BCE59E2F4D34CF4174FD2F04C60E04E" +
            "7D69896ECB6F65D7F15F22C6B7BE0ABA3A34A3C02FD79B29AE57557B00E3B19F" +
            "41C9544435D4288DA02AE28B7D615596806C16D8EEC5E5C6F4AD540BEA51079E" +
            "7772083FA1CE4C36DBBFAE822E56032076C1DF3831457326BC0859E2C358EDE6" +
            "A465DC7662CA13A2A29607EFE230E9234DAED39C50F64D34D1902F9AE2619550" +
            "8F714CBC6E6FC02DDEB716676CCF8674E04A933B255ADF36ED13C7E44857DD72" +
            "68B9A0EF46DACEF7758DAE51D60A9A9E070CAD66E2B8FB67ECB31CDC0D4D3D00" +
            "AE12C3679E83C2A15FBD7D008525C4C3F7172F33F70BBDA18BBAD562AB1501BE" +
            "72FF0D71E1E2F65E8A6095670CCD809152F86B868A4087C1723A9EBA29044E61" +
            "204980C5116FDAF83B565B09216F348ABBAD5A4867EB87CAC6E76E934B8C32DD" +
            "277017226C810551C9DFFC7F2F713DE8A24BF892B07DA8651AD57CDA303B301F" +
            "300706052B0E03021A0414AC44D277F1D7291B5A520398F597F6E3016174BA04" +
            "140F226E1BFC1FED6205670D7C624A3A9C1C20470B020207D0").HexToByteArray();

        internal static readonly byte[] MultiPrivateKeyPfx =
            PlatformSupport.IsRC2Supported
                ? MultiPrivateKeyPfx_RC2ContentEncryption
                : MultiPrivateKeyPfx_TripleDESContentEncryption;

        internal static readonly byte[] EmptyPfx = (
            "304F020103301106092A864886F70D010701A004040230003037301F30070605" +
            "2B0E03021A0414822078BC83E955E314BDA908D76D4C5177CC94EB0414711018" +
            "F2897A44A90E92779CB655EA11814EC598").HexToByteArray();

        internal const string ChainPfxPassword = "test";

        private static readonly byte[] ChainPfxBytes_RC2ContentEncryption = (
            "308213790201033082133506092A864886F70D010701A0821326048213223082" +
            "131E3082036706092A864886F70D010701A08203580482035430820350308203" +
            "4C060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E040811E8B9808BA6E96C020207D004820290D11DA8713602105C" +
            "95792D65BCDFC1B7E3708483BF6CD83008082F89DAE4D003F86081B153BD4D4A" +
            "C122E802752DEA29F07D0B7E8F0FB8A762B4CAA63360F9F72CA5846771980A6F" +
            "AE2643CD412E6E4A101625371BBD48CC6E2D25191D256B531B06DB7CDAC04DF3" +
            "E10C6DC556D5FE907ABF32F2966A561C988A544C19B46DF1BE531906F2CC2263" +
            "A301302A857075C7A9C48A395241925C6A369B60D176419D75E320008D5EFD91" +
            "5257B160F6CD643953E85F19EBE4E4F72B9B787CF93E95F819D1E43EF01CCFA7" +
            "48F0E7260734EA9BC6039BA7557BE6328C0149718A1D9ECF3355082DE697B6CD" +
            "630A9C224D831B7786C7E904F1EF2D9D004E0E825DD74AC4A576CDFCA7CECD14" +
            "D8E2E6CCAA3A302871AE0BA979BB25559215D771FAE647905878E797BBA9FC62" +
            "50F30F518A8008F5A12B35CE526E31032B56EFE5A4121E1E39DC7339A0CE8023" +
            "24CDDB7E9497BA37D8B9F8D826F901C52708935B4CA5B0D4D760A9FB33B0442D" +
            "008444D5AEB16E5C32187C7038F29160DD1A2D4DB1F9E9A6C035CF5BCED45287" +
            "C5DEBAB18743AAF90E77201FEA67485BA3BBCE90CEA4180C447EE588AC19C855" +
            "638B9552D47933D2760351174D9C3493DCCE9708B3EFE4BE398BA64051BF52B7" +
            "C1DCA44D2D0ED5A6CFB116DDA41995FA99373C254F3F3EBF0F0049F1159A8A76" +
            "4CFE9F9CC56C5489DD0F4E924158C9B1B626030CB492489F6AD0A9DCAF3E141D" +
            "B4D4821B2D8A384110B6B0B522F62A9DC0C1315A2A73A7F25F96C530E2F700F9" +
            "86829A839B944AE6758B8DD1A1E9257F91C160878A255E299C18424EB9983EDE" +
            "6DD1C5F4D5453DD5A56AC87DB1EFA0806E3DBFF10A9623FBAA0BAF352F50AB5D" +
            "B16AB1171145860D21E2AB20B45C8865B48390A66057DE3A1ABE45EA65376EF6" +
            "A96FE36285C2328C318182301306092A864886F70D0109153106040401000000" +
            "306B06092B0601040182371101315E1E5C004D006900630072006F0073006F00" +
            "66007400200045006E00680061006E0063006500640020004300720079007000" +
            "74006F0067007200610070006800690063002000500072006F00760069006400" +
            "650072002000760031002E003030820FAF06092A864886F70D010706A0820FA0" +
            "30820F9C02010030820F9506092A864886F70D010701301C060A2A864886F70D" +
            "010C0106300E0408FFCC41FD8C8414F6020207D080820F68092C6010873CF9EC" +
            "54D4676BCFB5FA5F523D03C981CB4A3DC096074E7D04365DDD1E80BF366B8F9E" +
            "C4BC056E8CE0CAB516B9C28D17B55E1EB744C43829D0E06217852FA99CCF5496" +
            "176DEF9A48967C1EEB4A384DB7783E643E35B5B9A50533B76B8D53581F02086B" +
            "782895097860D6CA512514E10D004165C85E561DF5F9AEFD2D89B64F178A7385" +
            "C7FA40ECCA899B4B09AE40EE60DAE65B31FF2D1EE204669EFF309A1C7C8D7B07" +
            "51AE57276D1D0FB3E8344A801AC5226EA4ED97FCD9399A4EB2E778918B81B17F" +
            "E4F65B502595195C79E6B0E37EB8BA36DB12435587E10037D31173285D45304F" +
            "6B0056512B3E147D7B5C397709A64E1D74F505D2BD72ED99055161BC57B6200F" +
            "2F48CF128229EFBEBFC2707678C0A8C51E3C373271CB4FD8EF34A1345696BF39" +
            "50E8CE9831F667D68184F67FE4D30332E24E5C429957694AF23620EA7742F08A" +
            "38C9A517A7491083A367B31C60748D697DFA29635548C605F898B64551A48311" +
            "CB2A05B1ACA8033128D48E4A5AA263D970FE59FBA49017F29049CF80FFDBD192" +
            "95B421FEFF6036B37D2F8DC8A6E36C4F5D707FB05274CC0D8D94AFCC8C6AF546" +
            "A0CF49FBD3A67FB6D20B9FE6FDA6321E8ABF5F7CC794CFCC46005DC57A7BAFA8" +
            "9954E43230402C8100789F11277D9F05C78DF0509ECFBF3A85114FD35F4F17E7" +
            "98D60C0008064E2557BA7BF0B6F8663A6C014E0220693AE29E2AB4BDE5418B61" +
            "0889EC02FF5480BD1B344C87D73E6E4DB98C73F881B22C7D298059FE9D7ADA21" +
            "92BB6C87F8D25F323A70D234E382F6C332FEF31BB11C37E41903B9A59ADEA5E0" +
            "CBAB06DFB835257ABC179A897DEAD9F19B7DF861BE94C655DC73F628E065F921" +
            "E5DE98FFCBDF2A54AC01E677E365DD8B932B5BDA761A0032CE2127AB2A2B9DCB" +
            "63F1EA8A51FC360AB5BC0AD435F21F9B6842980D795A6734FDB27A4FA8209F73" +
            "62DD632FC5FB1F6DE762473D6EA68BFC4BCF983865E66E6D93159EFACC40AB31" +
            "AA178806CF893A76CAAA3279C988824A33AF734FAF8E21020D988640FAB6DB10" +
            "DF21D93D01776EEA5DAECF695E0C690ED27AD386E6F2D9C9482EA38946008CCB" +
            "8F0BD08F9D5058CF8057CA3AD50BB537116A110F3B3ACD9360322DB4D242CC1A" +
            "6E15FA2A95192FC65886BE2672031D04A4FB0B1F43AE8476CF82638B61B416AA" +
            "97925A0110B736B4D83D7977456F35D947B3D6C9571D8E2DA0E9DEE1E665A844" +
            "259C17E01E044FAB898AA170F99157F7B525D524B01BD0710D23A7689A615703" +
            "8A0697BD48FFE0253ABD6F862093574B2FC9BA38E1A6EC60AF187F10D79FF71F" +
            "7C50E87A07CC0A51099899F7336FE742ADEF25E720B8E0F8781EC7957D414CF5" +
            "D44D6998E7E35D2433AFD86442CCA637A1513BE3020B5334614277B3101ED7AD" +
            "22AFE50DE99A2AD0E690596C93B881E2962D7E52EE0A770FAF6917106A8FF029" +
            "8DF38D6DE926C30834C5D96854FFD053BDB020F7827FB81AD04C8BC2C773B2A5" +
            "9FDD6DDF7298A052B3486E03FECA5AA909479DDC7FED972192792888F49C40F3" +
            "910140C5BE264D3D07BEBF3275117AF51A80C9F66C7028A2C3155414CF939997" +
            "268A1F0AA9059CC3AA7C8BBEF880187E3D1BA8978CBB046E43289A020CAE11B2" +
            "5140E2247C15A32CF70C7AA186CBB68B258CF2397D2971F1632F6EBC4846444D" +
            "E445673B942F1F110C7D586B6728ECA5B0A62D77696BF25E21ED9196226E5BDA" +
            "5A80ECCC785BEEDE917EBC6FFDC2F7124FE8F719B0A937E35E9A720BB9ED72D2" +
            "1213E68F058D80E9F8D7162625B35CEC4863BD47BC2D8D80E9B9048811BDD8CB" +
            "B70AB215962CD9C40D56AE50B7003630AE26341C6E243B3D12D5933F73F78F15" +
            "B014C5B1C36B6C9F410A77CA997931C8BD5CCB94C332F6723D53A4CCC630BFC9" +
            "DE96EFA7FDB66FA519F967D6A2DB1B4898BB188DEB98A41FFA7907AE7601DDE2" +
            "30E241779A0FDF551FB84D80AAEE3D979F0510CD026D4AE2ED2EFB7468418CCD" +
            "B3BD2A29CD7C7DC6419B4637412304D5DA2DC178C0B4669CA8330B9713A812E6" +
            "52E812135D807E361167F2A6814CEF2A8A9591EFE2C18216A517473B9C3BF2B7" +
            "51E47844893DA30F7DCD4222D1A55D570C1B6F6A99AD1F9213BA8F84C0B14A6D" +
            "ED6A26EAFF8F89DF733EEB44117DF0FD357186BA4A15BD5C669F60D6D4C34028" +
            "322D4DDF035302131AB6FD08683804CC90C1791182F1AE3281EE69DDBBCC12B8" +
            "1E60942FD082286B16BE27DC11E3BB0F18C281E02F3BA66E48C5FD8E8EA3B731" +
            "BDB12A4A3F2D9E1F833DD204372003532E1BB11298BDF5092F2959FC439E6BD2" +
            "DC6C37E3E775DCBE821B9CBB02E95D84C15E736CEA2FDDAD63F5CD47115B4AD5" +
            "5227C2A02886CD2700540EBFD5BF18DC5F94C5874972FD5424FE62B30500B1A8" +
            "7521EA3798D11970220B2BE7EFC915FCB7A6B8962F09ABA005861E839813EDA3" +
            "E59F70D1F9C277B73928DFFC84A1B7B0F78A8B001164EB0824F2510885CA269F" +
            "DCBB2C3AE91BDE91A8BBC648299A3EB626E6F4236CCE79E14C803498562BAD60" +
            "28F5B619125F80925A2D3B1A56790795D04F417003A8E9E53320B89D3A3109B1" +
            "9BB17B34CC9700DA138FABB5997EC34D0A44A26553153DBCFF8F6A1B5432B150" +
            "58F7AD87C6B37537796C95369DAD53BE5543D86D940892F93983153B4031D4FA" +
            "B25DAB02C1091ACC1DAE2118ABD26D19435CD4F1A02BDE1896236C174743BCA6" +
            "A33FB5429E627EB3FD9F513E81F7BD205B81AAE627C69CF227B043722FA05141" +
            "39347D202C9B7B4E55612FC27164F3B5F287F29C443793E22F6ED6D2F353ED82" +
            "A9F33EDBA8F5F1B2958F1D6A3943A9614E7411FDBCA597965CD08A8042307081" +
            "BAC5A070B467E52D5B91CA58F986C5A33502236B5BAE6DB613B1A408D16B29D3" +
            "560F1E94AD840CFA93E83412937A115ABF68322538DA8082F0192D19EAAA41C9" +
            "299729D487A9404ECDB6396DDA1534841EAE1E7884FA43574E213AE656116D9E" +
            "F7591AA7BDE2B44733DFE27AA59949E5DC0EE00FDF42130A748DDD0FB0053C1A" +
            "55986983C8B9CEAC023CAD7EDFFA1C20D3C437C0EF0FC9868D845484D8BE6538" +
            "EAADA6365D48BA776EE239ED045667B101E3798FE53E1D4B9A2ACBBE6AF1E5C8" +
            "8A3FB03AD616404013E249EC34458F3A7C9363E7772151119FE058BD0939BAB7" +
            "64A2E545B0B2FDAA650B7E849C8DD4033922B2CAE46D0461C04A2C87657CB4C0" +
            "FFBA23DED69D097109EC8BFDC25BB64417FEEB32842DE3EFEF2BF4A47F08B9FC" +
            "D1907BC899CA9DA604F5132FB420C8D142D132E7E7B5A4BD0EF4A56D9E9B0ACD" +
            "88F0E862D3F8F0440954879FFE3AA7AA90573C6BFDC6D6474C606ACA1CD94C1C" +
            "3404349DD83A639B786AFCDEA1779860C05400E0479708F4A9A0DD51429A3F35" +
            "FBD5FB9B68CECC1D585F3E35B7BBFC469F3EAEEB8020A6F0C8E4D1804A3EB32E" +
            "B3909E80B0A41571B23931E164E0E1D0D05379F9FD3BF51AF04D2BE78BDB84BD" +
            "787D419E85626297CB35FCFB6ED64042EAD2EBC17BB65677A1A33A5C48ADD280" +
            "237FB2451D0EFB3A3C32354222C7AB77A3C92F7A45B5FB10092698D88725864A" +
            "3685FBDD0DC741424FCCD8A00B928F3638150892CAAB535CC2813D13026615B9" +
            "9977F7B8240E914ACA0FF2DCB1A9274BA1F55DF0D24CCD2BAB7741C9EA8B1ECD" +
            "E97477C45F88F034FDF73023502944AEE1FF370260C576992826C4B2E5CE9924" +
            "84E3B85170FCCAC3413DC0FF6F093593219E637F699A98BD29E8EE4550C128CA" +
            "182680FDA3B10BC07625734EE8A8274B43B170FC3AEC9AA58CD92709D388E166" +
            "AB4ADFD5A4876DC47C17DE51FDD42A32AF672515B6A81E7ABECFE748912B321A" +
            "FD0CBF4880298DD79403900A4002B5B436230EB6E49192DF49FAE0F6B60EBA75" +
            "A54592587C141AD3B319129006367E9532861C2893E7A2D0D2832DF4377C3184" +
            "5CB02A1D020282C3D2B7F77221F71FEA7FF0A988FEF15C4B2F6637159EEC5752" +
            "D8A7F4AB971117666A977370E754A4EB0DC52D6E8901DC60FCD87B5B6EF9A91A" +
            "F8D9A4E11E2FFDAB55FC11AF6EEB5B36557FC8945A1E291B7FF8931BE4A57B8E" +
            "68F04B9D4A9A02FC61AE913F2E2DDBEE42C065F4D30F568834D5BB15FDAF691F" +
            "197EF6C25AE87D8E968C6D15351093AAC4813A8E7B191F77E6B19146F839A43E" +
            "2F40DE8BE28EB22C0272545BADF3BD396D383B8DA8388147100B347999DDC412" +
            "5AB0AA1159BC6776BD2BF51534C1B40522D41466F414BDE333226973BAD1E6D5" +
            "7639D30AD94BEA1F6A98C047F1CE1294F0067B771778D59E7C722C73C2FF100E" +
            "13603206A694BF0ED07303BE0655DC984CA29893FD0A088B122B67AABDC803E7" +
            "3E5729E868B1CA26F5D05C818D9832C70F5992E7D15E14F9775C6AD24907CF2F" +
            "211CF87167861F94DCF9E3D365CB600B336D93AD44B8B89CA24E59C1F7812C84" +
            "DBE3EE57A536ED0D4BF948F7662E5BCBBB388C72243CFCEB720852D5A4A52F01" +
            "8C2C087E4DB43410FE9ABA3A8EF737B6E8FFDB1AB9832EBF606ED5E4BD62A86B" +
            "BCAE115C67682EDEA93E7845D0D6962C146B411F7784545851D2F327BEC7E434" +
            "4D68F137CDA217A3F0FF3B752A34C3B5339C79CB8E1AC690C038E85D6FC13379" +
            "090198D3555394D7A2159A23BD5EEF06EB0BCC729BB29B5BE911D02DA78FDA56" +
            "F035E508C722139AD6F25A6C84BED0E98893370164B033A2B52BC40D9BF5163A" +
            "F9650AB55EABB23370492A7D3A87E17C11B4D07A7296273F33069C835FD208BA" +
            "8F989A3CF8659054E2CCCFB0C983531DC6590F27C4A1D2C3A780FE945F7E52BB" +
            "9FFD2E324640E3E348541A620CD62605BBDB284AF97C621A00D5D1D2C31D6BD6" +
            "1149137B8A0250BC426417A92445A52574E999FB9102C16671914A1542E92DDE" +
            "541B2A0457112AF936DA84707CADFEA43BFEDAE5F58859908640420948086E57" +
            "FFD1B867C241D40197CB0D4AD58BB69B3724772E0079406A1272858AAA620668" +
            "F696955102639F3E95CFFC637EAF8AB54F0B5B2131AB292438D06E15F3826352" +
            "DEDC653DA5A4AACE2BB97061A498F3B6789A2310471B32F91A6B7A9944DDBB70" +
            "31525B3AE387214DC85A1C7749E9168F41272680D0B3C331D61175F23B623EEC" +
            "40F984C35C831268036680DE0821E5DEE5BB250C6984775D49B7AF94057371DB" +
            "72F81D2B0295FC6A51BCD00A697649D4346FDD59AC0DFAF21BFCC942C23C6134" +
            "FFBA2ABABC141FF700B52C5B26496BF3F42665A5B71BAC7F0C19870BD9873890" +
            "239C578CDDD8E08A1B0A429312FB24F151A11E4D180359A7FA043E8155453F67" +
            "265CB2812B1C98C144E7675CFC86413B40E35445AE7710227D13DC0B5550C870" +
            "10B363C492DA316FB40D3928570BF71BF47638F1401549369B1255DB080E5DFA" +
            "18EA666B9ECBE5C9768C06B3FF125D0E94B98BB24B4FD44E770B78D7B336E021" +
            "4FD72E77C1D0BE9F313EDCD147957E3463C62E753C10BB98584C85871AAEA9D1" +
            "F397FE9F1A639ADE31D40EAB391B03B588B8B031BCAC6C837C61B06E4B745052" +
            "474D33531086519C39EDD6310F3079EB5AC83289A6EDCBA3DC97E36E837134F7" +
            "303B301F300706052B0E03021A0414725663844329F8BF6DECA5873DDD8C96AA" +
            "8CA5D40414DF1D90CD18B3FBC72226B3C66EC2CB1AB351D4D2020207D0").HexToByteArray();

        private static readonly byte[] ChainPfxBytes_TripleDESContentEncryption = (
            "30821306020103308212C206092A864886F70D010701A08212B3048212AF3082" +
            "12AB308203C406092A864886F70D010701A08203B5048203B1308203AD308203" +
            "A9060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E0408B43CDBD7536FD7E2020207D0048202904EC18175B7B959AD" +
            "8BF52CCEE41D754471CD84C99404E07672BA2F1BF60C59AFE7D32B64EF1EDB60" +
            "326F06991D20797C598F2BD192C258C0ED726382C1B7B8A12781CA64AB72320F" +
            "CAFD864D9E3CA580BA961DEF71477BA4463F8E56C9FDE471B7E0D252988256CE" +
            "8780A6E843B46D9D829EF20321F8B207C1410D85E04F0CDD9CA2E7C25ED661F2" +
            "820C76495A900A83D5485D0CA93E6B89D3088C727EF6A702F63C648C9271C292" +
            "FCF716E755F23EB65376E1F413E12946E3A43B6AB419E5076ED523CE4DC6DF9B" +
            "2C8E461A3243708764B8655AA2763A55011A558778E2A31072B33DDD4D04A86E" +
            "89C416F4CFA65E6FADF390F2EE7DC8764428F85C3BB7553478A03F4807949722" +
            "77077B65A799E862C7E67A4EEF0AAEC1C36ED7F1408E3B0B53DB24A76CE80966" +
            "2DD472121B64740EDC3306AED1CD9E200011999B31A3EEB56F7B34880FD5918C" +
            "7116607366E6C2514BAD741BBF25B10124B01F848162F405572C3781049E9AED" +
            "A5294FB5AC6171E885311E7CB7001A8B83A86964B6D407096309322051ED60AF" +
            "1E2D76A534D44ECC0895BF4984FA2F05625F2AF2B05B82794007705DE256914E" +
            "F66AB1DD8726CE09E7950B8CE37D5138951620EEE7DF1F6957B62610BBCC53CA" +
            "DE6FB33DBA5709DAC3D83E92028391132535B2C8A1B96E153FBF18D008A9F495" +
            "E022B8B1DD77FE3456388D450E7026246F0246408B96F995C60930F912463E34" +
            "45B6803F5BDBAC95C6CFF02893C8D57FFE091E7C520B971D24F6050D0ADA8E52" +
            "F9F64D68DA38891A36AAF0231B35873CD90894393835623D6D3E898361AD17E3" +
            "BC99FDBBDEF9F82763740468DE692A3FA094E1683E651DAEAD3267DBD030C930" +
            "5C3A4E4AF5AB61B483BFFA4998C5F708FE4CF8D3435EA38B1512D56310ED58CD" +
            "65AAADB6A64C7C623181DF301306092A864886F70D0109153106040401000000" +
            "305B06092A864886F70D010914314E1E4C007B00340031003700380032003500" +
            "340041002D0036003300440042002D0034003400380045002D00410030003700" +
            "41002D003200320033003600420038003900450041003800350045007D306B06" +
            "092B0601040182371101315E1E5C004D006900630072006F0073006F00660074" +
            "00200045006E00680061006E006300650064002000430072007900700074006F" +
            "0067007200610070006800690063002000500072006F00760069006400650072" +
            "002000760031002E003030820EDF06092A864886F70D010706A0820ED030820E" +
            "CC02010030820EC506092A864886F70D010701301C060A2A864886F70D010C01" +
            "03300E040894BE093240AADB8A020207D080820E989C5F83126495753056D861" +
            "A825CCB15575B04D71869B2A3685B26E535C4748F62EE23ED1C6ADC3D35DC39D" +
            "0EB75C11742BBB7F36A7BA52D63EEA6F1398BBF1361EB98CC06C9EF405040888" +
            "35B4063525FB2AA0715E88E62D44746404681304F38E5E2CD27BBB14571885CA" +
            "DEAE50AB9F3956B3332B0C4CED90F0C8A7421532D938015861AC6EDF0F3007E7" +
            "FFB9786DC56E299D5D9277BB7A29902863CD606CED8494932607AA202ECACB5F" +
            "BB23AB7E824FEBED60CD76B462417ED099CAF56CE946FA5B5906341618ADA2EB" +
            "3C739E332EE43ED6D16A971D2F0AF47941BABB3251250108BA6E4C0909B27808" +
            "6CAA4509DFE267F9ED352330576EC22D28F11607B9805D5A82CF5A4B45C702F1" +
            "A55B52D9A0E1A70543746ED457E39CAA42D66BF1EA8EE7C4DEE500E8B35849F4" +
            "797583651465B5088C8F83665D05087DB9E68A5D6AFB0C489BA255A1E639E540" +
            "977C53CA9F250E2C5F2B5822A7F706BD3C45DC8BCB546CB1184D7CB088701D0B" +
            "53845736CCDC4BAD654108459415CC5DA8718E15E319473E5EF8107709B4EF68" +
            "395E164987D467D39EAD4A049EB314F30505A072D52FB9902AF3FDC76F607D4C" +
            "5068F4AD2CDB0E734E5EC18B72A44DD009EB02AB3C9531CF2BFBD6E3B40B3A4C" +
            "2AE0AF8B04F6D2038102869A3952FEB7F4B82A03BC180D201FAC040785E02DA7" +
            "261E1A2D74BBE95D13DA0C3F07744D2A159E4BBACB8FF4D7CC6306D0DE3E1D12" +
            "6A9BDE0DFB9ED71D2C0B3827B0087D1FFBAE639AFD6EBB3C3C7138602C582F01" +
            "404D997BB66EDC324CF43EB0315C13C8917EB88C318351E77CD5168EB8D05F04" +
            "1F375ED6149E797497DDB6B549CDC7E24375E90746143F95474FC5A72E1B097C" +
            "85C61BA7C5CCFC573B0DC308EA08BC1E21930DE314934CC4A7BBCB6C83165B1A" +
            "2D7A34C94B165879FABCEA245919A9CE9681DB0FA6B4278B6E757047661C6604" +
            "CB66E95BE6C5BE1F43CD6CCBC6CD9687310DB42C64E6BAE6224A219AD34E4918" +
            "81007A7A69BCD4FC7077D0B469E90F4FE7FA584D8F7384C2013C94502AB0D61E" +
            "24341FFC2894FBB0C807D8F9228E15DDE47E20C909B6BA802C1C3C4FF4E6FFE6" +
            "7EAAC94290681A250FD681305C77FF881142C04FA1262D0B29E233AEB0DBAF7A" +
            "70C8EAB8257BF553CDCCB5E3D06617410E6CABF19B24F6943597FA353D8861BE" +
            "BF728864E17A54B37BE81FBF25FA5C3F5EDDADBD633274D226A6DD9BB2D427DC" +
            "B283A29A691BC89BADFF80A5B852EA64521704974CDDC0599E94EF1BC259EC23" +
            "3A1310F791878294DB5F8D7B2C890B6EBAE3E517EB1138D0A00AC762BD27D273" +
            "2D66BDF971C8E87ED10CE4880D0460C9EFEBF58DE7C65AE59BF6FBAD9B5B32A2" +
            "49DD6B9B3F9821F584004EA9DFB54C94B54B7A756D531F565A84A60747A47E6D" +
            "D2A5A0F2F16BA15D4F5AD0CA110D26E629AD7CF4E6812948138FEB7DF3104AD1" +
            "CFDC9F6FD6F64EB6A009D27E4D63C9A212ABF61CABEE2E290691913AE20E4966" +
            "D0B8FF6E97342A6297A07A72796D7466AAA8E9336F8F7796EEB4CE624E83335C" +
            "F3DC78C97A67BD868C9EE8C8CA3B756ACF93B7798DF470FAB7DE524C751631BC" +
            "4040483C2C53CC691E13122C1BFDC36D83F0FEFD1DDC1ED9042613406FA476A0" +
            "F97969213DE68568FA15F04F1AED1481BBDAC0BD9FDFF6FAB9C2E130024CEB6F" +
            "358809FBCB94EC1454E46D2E369923D1B93F18F8B584B6CC2820E26C605C7BB1" +
            "F50595DD9F0E724DB23359E9B7045DE35E46ACFE07AF27E1D111551689CC0352" +
            "AD474F0159B8645408512B03E5EEDE3302513FD665666FB7D1D210F426E18E1E" +
            "2B5616EBAB322A306338C784CAECA267F3592D217CE17D318A0A68A6D19CE60D" +
            "90BEAC809F8F3DCEDB427B51CE604CBE814597A71311C9E650D64FA1BFECB801" +
            "D8EE3C1607E02F641C31212AC6ABA56922A912A6E0C5284C547CA80A94B59C46" +
            "443510EDEAB9388D422014C472394EAE7DC125C3A6C8B707AC3552EE52153B65" +
            "F7FC989093721329910D4B5C92398FB13644A8C784380E82E73AF03A478C7875" +
            "1A7E8E9B6E0D1ED589190ADF388C7A82BC451BAC4AD363C5B6889EED91DF2FB8" +
            "A8DCC6810D188570CADDBA85BA935E272026AF23611D1FEE694E4475BF9E68E0" +
            "8E8A88F2202A153259206FADAE4346D3BAEB59387715FC5D05E7D5F915184A38" +
            "81609998068E7EB6A843A719A5ACD41A82364358F4B6867F6B9EA0175703A431" +
            "00742F88AC0769B03071D7C3BF345C95806EC951F2B7C55AF7FE52174E37219B" +
            "59309BAA09EEBA299F78E53F375D549C560E5D3F1848113A2E2C8B6CE65892C1" +
            "F55E3822864E6D95E055BA5CBB0098BB922C3188E3EB56B9EAA93030B2E8587E" +
            "9EF411F725D57F1A12D6F911413D91C78334963679083C308BCAD4C8DBE43116" +
            "BCD5C74AA6B22FE9904329DF6E8EFF458BA7659ED9A043302009D6BBED2EC674" +
            "EE661AB7789058CE56918A7F31A4827667338C7B12A958C239300FEE3D151F46" +
            "D595707B76707B247BC37457DC1498818F6B56D9CABB72682F65BE500F74D00F" +
            "A7B1144FD05FBAEB12DD83014F1A01367543761DA8FFF0B78072E5DCC14A561C" +
            "5A08F509FEFF2DA20C5F342BC7E7C498417BF758B5DB26732DE45034FE952AC7" +
            "6FDB80A526066505610E8753BC5521BB47333C2EBBD42B959010EA18FB5F8194" +
            "A8BCD6FC6E5B8059F8C14DE002E3F78AC453C4E14FCA4DF09EFDBC992ABAEA34" +
            "A7E6648B6E7F4B7D7676B2DFDFB780A275456ABC2E0C6409FBD017A10E08D3D7" +
            "D8F37CF3C451BC73CD28C77F3BE2E1812D5FA6DBB842B60FBA1A5B718BC43A18" +
            "2D9BAFC7C2899D8BCBBED51B2B252AC90EAADEFEA04E801F3AEB7766D92C70A5" +
            "D4D05546F698B191B080135CE040A626F91589088E14E4AC494B2C4EB5CE6D1C" +
            "0B827E412001F6C366A9FCD29C1663D4003AF2BE77733DA806787E4AC39DBE07" +
            "9F13344B958A38EB6A0D7DB28F51241D1820316D33A23233766D57ADC6C7383A" +
            "16FDC152C4100CFB459D37915478DE03A6BCF64CF513EB650D8390077C7EE12E" +
            "0CC2EC447B1DECC5C675F0A0912FAD96727942E3F76490A319B65689D25212E6" +
            "574AFE8AC6E558FA17D324E8ACAE80DA5803CC17C10BEB887F5B8783C37D1EF8" +
            "4382D5DEEBCF3D2CBCD4420C1B5B304C56F60EF06A58C632CBB9435932E0C605" +
            "C46291952EC08261C5847B2785E5E3FCCE6756BCE869E5E564FFA6D92F494099" +
            "BCC08F2BB84ADAE91F000D134E918DE947DBB217EE5D24A42E68C1A6F14B3C37" +
            "0F0E0E829FE29F91DF01B205BA209736A979D06DA905E52569D19F460CD3C9FB" +
            "AC7D428554D7BE5766D246DAF927F32EEE76D88B29D24EBB8F20C0EE01661579" +
            "568B20F08523893DA596F0418EF2565CFC90A2AE97860D77F43806B1493FB1B9" +
            "873AF46DDCE8112CA98141AEB2A3CDCE6FB90545CDA7215CB5AFD0E1FA1CFA01" +
            "9B8A95E552F7B4C174F4B5635430500DFF11851F5084233AE8D9CBB3416C7AF6" +
            "128DAFA764C2032E4E6FD45C2C8ED83436CA673C7F0A1310B50A59E99A4DA328" +
            "A179D29128C74B14FEACF2C6D0B0E079B7A700CD4D5D92E50CFEBB5633055C87" +
            "5F57D2063CE4A5A4BDE5043F9F069C8051A5F2803D242A67F61FAAE19CB01827" +
            "7132484137ED45B02FA919C399AA6F7DC4EC9D116C7A12803F8A6589AD801FD0" +
            "154387C09072D48D4D9F4DD70E3319B61AE35F6B8BDF77C796C12FCFCA7D3F5B" +
            "AB3C639D79AF8CA579944D2EA9F1D3F8B1E872C48384FA089EE74E044255BEA7" +
            "79C79FE7826D7BAADD6CCA0B53912762CCE890A419E94B9CEC3152A46D1D1158" +
            "84671CAF82DB53B3E1D23934C58AADC97E84589F2E0745DB62B757C4061416BB" +
            "5DA650CAF0D687BAE96DF8521AE185B3122C13E342F7FFC985B4ADB2BE8E412E" +
            "A4ABF9D39DDFAB7FACE1BC319F725F9982944F8B34B4F563BC4E138C99B7EA49" +
            "05960529F79E4124B52508E7F20547FDBE7DE03692A5B249AD0E0739E34CE987" +
            "9423CEEECBAFF243EA4EECAF7E7DA782306FA771AC2D4CF3EDB33255382CA410" +
            "F8DA4B467012CF4322BBA86138A75D1252C1C08A5E24C8BFB57DF48C860B93A4" +
            "55C6A428DEEA8EDB91272BFA03C73E80987BB94F8C0CB504552C8FB0C7A3003A" +
            "62F678A4424C0B8AA1A9AAEC747786C051FCD407F1CE56A28F3ED28B2BCDADE0" +
            "5724FE94FBF67EA6BA4030832D4B8DB97DEA9496CFEDC85E71AA35524DB4331E" +
            "742BE36CC7EE1DB92AA57467F23AB8AA73FC23B32E5749C8A43915A8E5A1AC3D" +
            "3E031CB786B4070195D88C98E0075B713BB71D77D6F9E50FB65F18B0552B1F83" +
            "B4158B304014C94B266C52781521388E3469FF78E8128EB26D8FDFFF75FEF45B" +
            "C0D2241851E976F3B0A75E6A9BED401095A093AEB7F6F8EFC8338F3CEC40B71B" +
            "DB86433E7E0617FBB354D37ABBC4C998E8E4E3E9514C2BE06D65109772DCE808" +
            "CDB9AE319F35E681B83D29E01BACFB0E02619DC8C1D390C35C87299D40D8396C" +
            "0344242A963987ABAFA357634E8398C0CAFFCB5AB5F5EAF395B38CBBF87F6AB1" +
            "0331837BB08FEEC14A16B22B1B8DECF5C79DB8F27D4DCE805FB15E8823E35529" +
            "136BDB216A3E48EA042DF59237C65048DA7CA3E06FDE9EA8EF04124A8DBF4ED2" +
            "720A4B9F99B93659B220072A4B88EC093E3E69CE964837E413A145EF08E9265A" +
            "946653112C46CBA62D2425AF9A56B6F88A30802BF1174E854EEA5585AC4CAC24" +
            "2A5BFDF5585659C2FF9EDFAF71594DB4A9D81188350AD5F30CBA9FED3B023ACC" +
            "9424D1FBAB4FE8B1562544C90EA2599E8FAB028A244EAF3FC6FF72EE0029B277" +
            "6E12786BD3783E51CF2AFC8AE484C205576F6E4EB216F7D4573D0EBCFA67BC3E" +
            "999DE7390EE74F5E73B570DF80D8145DE6FE96A2D4D12D758A6B134303E7F361" +
            "500377765CE0F5D3ACDB9ED3FEEC304CD10C0C19872A426A077B0D57D4CC8969" +
            "81EAD835C2EAABDA8A6537E19B034638F1B57F544D8B5CAF9AB8754E45CBCCFB" +
            "C1AC47A4AF3E225D96FCC828979B4A6923294E1E9F75C3F35354C9BE8C9FF95C" +
            "805FF2E108FDC596965FC4837D0EC02C40F1C343B21C96110FB4AAB46A840984" +
            "8AF62B81FA593FD49DA006B3B44CB16F7DAF32603742990FEE02569EEC8CC042" +
            "EAEBAFC01F28DF096A3CD961D8F4975FD19CD8FE4F64D46D14A53BEFC60B3CC4" +
            "8CD8DBADEDE41EFBF9DDDC5C583F0E66F577192D3B31D2254B6A6991B1BD684B" +
            "6DECD4E8F232767A46BBBD83E9BA5CDE1EAA09BBEF497FAC94BE5DD435E364AA" +
            "8A19E9413C47214C2D285A6B20303B301F300706052B0E03021A04147E4F196F" +
            "6D0A4EF84A2C08674CDDF6899F7672A50414206C1AAAFFB1E282745C91F6BF5F" +
            "0E87036AC61E020207D0").HexToByteArray();

        public static readonly byte[] ChainPfxBytes =
           PlatformSupport.IsRC2Supported
                ? ChainPfxBytes_RC2ContentEncryption
                : ChainPfxBytes_TripleDESContentEncryption;

        internal static readonly byte[] Pkcs7ChainDerBytes = (
            "30820E1606092A864886F70D010702A0820E0730820E030201013100300B0609" +
            "2A864886F70D010701A0820DEB3082050B30820474A003020102020A15EAA83A" +
            "000100009291300D06092A864886F70D010105050030818131133011060A0992" +
            "268993F22C6401191603636F6D31193017060A0992268993F22C64011916096D" +
            "6963726F736F667431143012060A0992268993F22C6401191604636F72703117" +
            "3015060A0992268993F22C64011916077265646D6F6E643120301E0603550403" +
            "13174D532050617373706F7274205465737420537562204341301E170D313330" +
            "3131303231333931325A170D3331313231333232323630375A308185310B3009" +
            "060355040613025553310B30090603550408130257413110300E060355040713" +
            "075265646D6F6E64310D300B060355040A130454455354310D300B060355040B" +
            "130454455354311330110603550403130A746573742E6C6F63616C3124302206" +
            "092A864886F70D010901161563726973706F70406D6963726F736F66742E636F" +
            "6D30819F300D06092A864886F70D010101050003818D0030818902818100B406" +
            "851089E9CF7CDB438DD77BEBD819197BEEFF579C35EF9C4652DF9E6330AA7E2E" +
            "24B181C59DA4AF10E97220C1DF99F66CE6E97247E9126A016AC647BD2EFD136C" +
            "31470C7BE01A20E381243BEEC8530B7F6466C50A051DCE37274ED7FF2AFFF4E5" +
            "8AABA61D5A448F4A8A9B3765D1D769F627ED2F2DE9EE67B1A7ECA3D288C90203" +
            "010001A38202823082027E300E0603551D0F0101FF0404030204F0301D060355" +
            "1D250416301406082B0601050507030106082B06010505070302301D0603551D" +
            "0E04160414FB3485708CBF6188F720EF948489405C8D0413A7301F0603551D23" +
            "0418301680146A6678620A4FF49CA8B75FD566348F3371E42B133081D0060355" +
            "1D1F0481C83081C53081C2A081BFA081BC865F687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F4D5325323050617373706F727425323054657374" +
            "25323053756225323043412831292E63726C865966696C653A2F2F5C5C707074" +
            "65737473756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E" +
            "636F6D5C43657274456E726F6C6C5C4D532050617373706F7274205465737420" +
            "5375622043412831292E63726C3082013806082B060105050701010482012A30" +
            "82012630819306082B06010505073002868186687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F70707465737473756263612E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D5F4D5325323050617373706F7274" +
            "2532305465737425323053756225323043412831292E63727430818D06082B06" +
            "01050507300286818066696C653A2F2F5C5C70707465737473756263612E7265" +
            "646D6F6E642E636F72702E6D6963726F736F66742E636F6D5C43657274456E72" +
            "6F6C6C5C70707465737473756263612E7265646D6F6E642E636F72702E6D6963" +
            "726F736F66742E636F6D5F4D532050617373706F727420546573742053756220" +
            "43412831292E637274300D06092A864886F70D0101050500038181009DEBB8B5" +
            "A41ED54859795F68EF767A98A61EF7B07AAC190FCC0275228E4CAD360C9BA98B" +
            "0AE153C75522EEF42D400E813B4E49E7ACEB963EEE7B61D3C8DA05C183471544" +
            "725B2EBD1889877F62134827FB5993B8FDF618BD421ABA18D70D1C5B41ECDD11" +
            "695A48CB42EB501F96DA905471830C612B609126559120F6E18EA44830820358" +
            "308202C1A00302010202101B9671A4BC128B8341B0E314EAD9A191300D06092A" +
            "864886F70D01010505003081A13124302206092A864886F70D01090116156173" +
            "6D656D6F6E406D6963726F736F66742E636F6D310B3009060355040613025553" +
            "310B30090603550408130257413110300E060355040713075265646D6F6E6431" +
            "123010060355040A13094D6963726F736F667431163014060355040B130D5061" +
            "7373706F727420546573743121301F060355040313184D532050617373706F72" +
            "74205465737420526F6F74204341301E170D3035303132363031333933325A17" +
            "0D3331313231333232323630375A3081A13124302206092A864886F70D010901" +
            "161561736D656D6F6E406D6963726F736F66742E636F6D310B30090603550406" +
            "13025553310B30090603550408130257413110300E060355040713075265646D" +
            "6F6E6431123010060355040A13094D6963726F736F667431163014060355040B" +
            "130D50617373706F727420546573743121301F060355040313184D5320506173" +
            "73706F7274205465737420526F6F7420434130819F300D06092A864886F70D01" +
            "0101050003818D0030818902818100C4673C1226254F6BBD01B01D21BB05264A" +
            "9AA5B77AC51748EAC52048706DA6B890DCE043C6426FC44E76D70F9FE3A4AC85" +
            "5F533E3D08E140853DB769EE24DBDB7269FABEC0FDFF6ADE0AA85F0085B78864" +
            "58E7585E433B0924E81600433CB1177CE6AD5F2477B2A0E2D1A34B41F6C6F5AD" +
            "E4A9DD7D565C65F02C2AAA01C8E0C10203010001A3818E30818B301306092B06" +
            "0104018237140204061E0400430041300B0603551D0F040403020186300F0603" +
            "551D130101FF040530030101FF301D0603551D0E04160414F509C1D6267FC39F" +
            "CA1DE648C969C74FB111FE10301206092B060104018237150104050203010002" +
            "302306092B0601040182371502041604147F7A5208411D4607C0057C98F0C473" +
            "07010CB3DE300D06092A864886F70D0101050500038181004A8EAC73D8EA6D7E" +
            "893D5880945E0E3ABFC79C40BFA60A680CF8A8BF63EDC3AD9C11C081F1F44408" +
            "9581F5C8DCB23C0AEFA27571D971DBEB2AA9A1B3F7B9B0877E9311D36098A65B" +
            "7D03FC69A835F6C3096DEE135A864065F9779C82DEB0C777B9C4DB49F0DD11A0" +
            "EAB287B6E352F7ECA467D0D3CA2A8081119388BAFCDD25573082057C308204E5" +
            "A003020102020A6187C7F200020000001B300D06092A864886F70D0101050500" +
            "3081A13124302206092A864886F70D010901161561736D656D6F6E406D696372" +
            "6F736F66742E636F6D310B3009060355040613025553310B3009060355040813" +
            "0257413110300E060355040713075265646D6F6E6431123010060355040A1309" +
            "4D6963726F736F667431163014060355040B130D50617373706F727420546573" +
            "743121301F060355040313184D532050617373706F7274205465737420526F6F" +
            "74204341301E170D3039313032373231333133395A170D333131323133323232" +
            "3630375A30818131133011060A0992268993F22C6401191603636F6D31193017" +
            "060A0992268993F22C64011916096D6963726F736F667431143012060A099226" +
            "8993F22C6401191604636F727031173015060A0992268993F22C640119160772" +
            "65646D6F6E643120301E060355040313174D532050617373706F727420546573" +
            "742053756220434130819F300D06092A864886F70D010101050003818D003081" +
            "8902818100A6A4918F93C5D23B3C3A325AD8EC77043D207A0DDC294AD3F5BDE0" +
            "4033FADD4097BB1DB042B1D3B2F26A42CC3CB88FA9357710147AB4E1020A0DFB" +
            "2597AB8031DB62ABDC48398067EB79E4E2BBE5762F6B4C5EA7629BAC23F70269" +
            "06D46EC106CC6FBB4D143F7D5ADADEDE19B021EEF4A6BCB9D01DAEBB9A947703" +
            "40B748A3490203010001A38202D7308202D3300F0603551D130101FF04053003" +
            "0101FF301D0603551D0E041604146A6678620A4FF49CA8B75FD566348F3371E4" +
            "2B13300B0603551D0F040403020186301206092B060104018237150104050203" +
            "010001302306092B060104018237150204160414A0A485AE8296EA4944C6F6F3" +
            "886A8603FD07472C301906092B0601040182371402040C1E0A00530075006200" +
            "430041301F0603551D23041830168014F509C1D6267FC39FCA1DE648C969C74F" +
            "B111FE103081D60603551D1F0481CE3081CB3081C8A081C5A081C28663687474" +
            "703A2F2F70617373706F72747465737463612E7265646D6F6E642E636F72702E" +
            "6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D532532305061" +
            "7373706F727425323054657374253230526F6F7425323043412831292E63726C" +
            "865B66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D53" +
            "2050617373706F7274205465737420526F6F742043412831292E63726C308201" +
            "4406082B06010505070101048201363082013230819A06082B06010505073002" +
            "86818D687474703A2F2F70617373706F72747465737463612E7265646D6F6E64" +
            "2E636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50" +
            "415353504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F" +
            "736F66742E636F6D5F4D5325323050617373706F727425323054657374253230" +
            "526F6F7425323043412832292E63727430819206082B06010505073002868185" +
            "66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E636F" +
            "72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50415353" +
            "504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F736F66" +
            "742E636F6D5F4D532050617373706F7274205465737420526F6F742043412832" +
            "292E637274300D06092A864886F70D010105050003818100C44788F8C4F5C2DC" +
            "84976F66417CBAE19FBFA82C257DA4C7FED6267BC711D113C78B1C097154A62A" +
            "B462ADC84A434AEBAE38DEB9605FAB534A3CAF7B72C199448E58640388911296" +
            "115ED6B3478D0E741D990F2D59D66F12E58669D8983489AB0406E37462164B56" +
            "6AA1D9B273C406FA694A2556D1D3ACE723382C19871B8C143100").HexToByteArray();

        internal static readonly byte[] Pkcs7ChainPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----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-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MCcGCSqGSIb3DQEHAqAaMBgCAQExADALBgkqhkiG9w0BBwGgAKEAMQA=
-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyDerBytes = (
            "302706092A864886F70D010702A01A30180201013100300B06092A864886F70D" +
            "010701A000A1003100").HexToByteArray();

        internal static readonly byte[] Pkcs7SingleDerBytes = (
            "3082021406092A864886F70D010702A0820205308202010201013100300B0609" +
            "2A864886F70D010701A08201E9308201E530820152A0030201020210D5B5BC1C" +
            "458A558845BFF51CB4DFF31C300906052B0E03021D05003011310F300D060355" +
            "040313064D794E616D65301E170D3130303430313038303030305A170D313130" +
            "3430313038303030305A3011310F300D060355040313064D794E616D6530819F" +
            "300D06092A864886F70D010101050003818D0030818902818100B11E30EA8742" +
            "4A371E30227E933CE6BE0E65FF1C189D0D888EC8FF13AA7B42B68056128322B2" +
            "1F2B6976609B62B6BC4CF2E55FF5AE64E9B68C78A3C2DACC916A1BC7322DD353" +
            "B32898675CFB5B298B176D978B1F12313E3D865BC53465A11CCA106870A4B5D5" +
            "0A2C410938240E92B64902BAEA23EB093D9599E9E372E48336730203010001A3" +
            "46304430420603551D01043B3039801024859EBF125E76AF3F0D7979B4AC7A96" +
            "A1133011310F300D060355040313064D794E616D658210D5B5BC1C458A558845" +
            "BFF51CB4DFF31C300906052B0E03021D0500038181009BF6E2CF830ED485B86D" +
            "6B9E8DFFDCD65EFC7EC145CB9348923710666791FCFA3AB59D689FFD7234B787" +
            "2611C5C23E5E0714531ABADB5DE492D2C736E1C929E648A65CC9EB63CD84E57B" +
            "5909DD5DDF5DBBBA4A6498B9CA225B6E368B94913BFC24DE6B2BD9A26B192B95" +
            "7304B89531E902FFC91B54B237BB228BE8AFCDA264763100").HexToByteArray();

        internal static readonly byte[] Pkcs7SinglePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MIICFAYJKoZIhvcNAQcCoIICBTCCAgECAQExADALBgkqhkiG9w0BBwGgggHpMIIB
5TCCAVKgAwIBAgIQ1bW8HEWKVYhFv/UctN/zHDAJBgUrDgMCHQUAMBExDzANBgNV
BAMTBk15TmFtZTAeFw0xMDA0MDEwODAwMDBaFw0xMTA0MDEwODAwMDBaMBExDzAN
BgNVBAMTBk15TmFtZTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAsR4w6odC
SjceMCJ+kzzmvg5l/xwYnQ2Ijsj/E6p7QraAVhKDIrIfK2l2YJtitrxM8uVf9a5k
6baMeKPC2syRahvHMi3TU7MomGdc+1spixdtl4sfEjE+PYZbxTRloRzKEGhwpLXV
CixBCTgkDpK2SQK66iPrCT2VmenjcuSDNnMCAwEAAaNGMEQwQgYDVR0BBDswOYAQ
JIWevxJedq8/DXl5tKx6lqETMBExDzANBgNVBAMTBk15TmFtZYIQ1bW8HEWKVYhF
v/UctN/zHDAJBgUrDgMCHQUAA4GBAJv24s+DDtSFuG1rno3/3NZe/H7BRcuTSJI3
EGZnkfz6OrWdaJ/9cjS3hyYRxcI+XgcUUxq6213kktLHNuHJKeZIplzJ62PNhOV7
WQndXd9du7pKZJi5yiJbbjaLlJE7/CTeayvZomsZK5VzBLiVMekC/8kbVLI3uyKL
6K/NomR2MQA=
-----END PKCS7-----");

        internal static readonly byte[] MicrosoftDotComLegacySslCertBytes = (
            "308205943082047CA00302010202103DF70C5D9903F8D8868B9B8CCF20DF6930" +
            "0D06092A864886F70D01010B05003077310B3009060355040613025553311D30" +
            "1B060355040A131453796D616E74656320436F72706F726174696F6E311F301D" +
            "060355040B131653796D616E746563205472757374204E6574776F726B312830" +
            "260603550403131F53796D616E74656320436C61737320332045562053534C20" +
            "4341202D204733301E170D3134313031353030303030305A170D313631303135" +
            "3233353935395A3082010F31133011060B2B0601040182373C02010313025553" +
            "311B3019060B2B0601040182373C0201020C0A57617368696E67746F6E311D30" +
            "1B060355040F131450726976617465204F7267616E697A6174696F6E31123010" +
            "06035504051309363030343133343835310B3009060355040613025553310E30" +
            "0C06035504110C0539383035323113301106035504080C0A57617368696E6774" +
            "6F6E3110300E06035504070C075265646D6F6E643118301606035504090C0F31" +
            "204D6963726F736F667420576179311E301C060355040A0C154D6963726F736F" +
            "667420436F72706F726174696F6E310E300C060355040B0C054D53434F4D311A" +
            "301806035504030C117777772E6D6963726F736F66742E636F6D30820122300D" +
            "06092A864886F70D01010105000382010F003082010A0282010100A46861FA9D" +
            "5DB763633BF5A64EF6E7C2C2367F48D2D46643A22DFCFCCB24E58A14D0F06BDC" +
            "956437F2A56BA4BEF70BA361BF12964A0D665AFD84B0F7494C8FA4ABC5FCA2E0" +
            "17C06178AEF2CDAD1B5F18E997A14B965C074E8F564970607276B00583932240" +
            "FE6E2DD013026F9AE13D7C91CC07C4E1E8E87737DC06EF2B575B89D62EFE4685" +
            "9F8255A123692A706C68122D4DAFE11CB205A7B3DE06E553F7B95F978EF8601A" +
            "8DF819BF32040BDF92A0DE0DF269B4514282E17AC69934E8440A48AB9D1F5DF8" +
            "9A502CEF6DFDBE790045BD45E0C94E5CA8ADD76A013E9C978440FC8A9E2A9A49" +
            "40B2460819C3E302AA9C9F355AD754C86D3ED77DDAA3DA13810B4D0203010001" +
            "A38201803082017C30310603551D11042A302882117777772E6D6963726F736F" +
            "66742E636F6D821377777771612E6D6963726F736F66742E636F6D3009060355" +
            "1D1304023000300E0603551D0F0101FF0404030205A0301D0603551D25041630" +
            "1406082B0601050507030106082B0601050507030230660603551D20045F305D" +
            "305B060B6086480186F84501071706304C302306082B06010505070201161768" +
            "747470733A2F2F642E73796D63622E636F6D2F637073302506082B0601050507" +
            "020230191A1768747470733A2F2F642E73796D63622E636F6D2F727061301F06" +
            "03551D230418301680140159ABE7DD3A0B59A66463D6CF200757D591E76A302B" +
            "0603551D1F042430223020A01EA01C861A687474703A2F2F73722E73796D6362" +
            "2E636F6D2F73722E63726C305706082B06010505070101044B3049301F06082B" +
            "060105050730018613687474703A2F2F73722E73796D63642E636F6D30260608" +
            "2B06010505073002861A687474703A2F2F73722E73796D63622E636F6D2F7372" +
            "2E637274300D06092A864886F70D01010B0500038201010015F8505B627ED7F9" +
            "F96707097E93A51E7A7E05A3D420A5C258EC7A1CFE1843EC20ACF728AAFA7A1A" +
            "1BC222A7CDBF4AF90AA26DEEB3909C0B3FB5C78070DAE3D645BFCF840A4A3FDD" +
            "988C7B3308BFE4EB3FD66C45641E96CA3352DBE2AEB4488A64A9C5FB96932BA7" +
            "0059CE92BD278B41299FD213471BD8165F924285AE3ECD666C703885DCA65D24" +
            "DA66D3AFAE39968521995A4C398C7DF38DFA82A20372F13D4A56ADB21B582254" +
            "9918015647B5F8AC131CC5EB24534D172BC60218A88B65BCF71C7F388CE3E0EF" +
            "697B4203720483BB5794455B597D80D48CD3A1D73CBBC609C058767D1FF060A6" +
            "09D7E3D4317079AF0CD0A8A49251AB129157F9894A036487").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComSslCertBytes = (
            "3082087930820661a00302010202136b000003f4e3a67a2348550c330000" +
            "000003f4300d06092a864886f70d01010b0500304f310b30090603550406" +
            "13025553311e301c060355040a13154d6963726f736f667420436f72706f" +
            "726174696f6e3120301e060355040313174d6963726f736f667420525341" +
            "20544c53204341203031301e170d3230303832383232313730325a170d32" +
            "31303832383232313730325a308188310b3009060355040613025553310b" +
            "30090603550408130257413110300e060355040713075265646d6f6e6431" +
            "1e301c060355040a13154d6963726f736f667420436f72706f726174696f" +
            "6e311e301c060355040b13154d6963726f736f667420436f72706f726174" +
            "696f6e311a3018060355040313117777772e6d6963726f736f66742e636f" +
            "6d30820122300d06092a864886f70d01010105000382010f003082010a02" +
            "82010100c69ea6351173387784f57cd1af5de2b7e7752f954fc2fc66d045" +
            "66be4922566ded721fdda0e23d6e542c4d5b663546a9410dc27818fbef40" +
            "c2ceff2e3c9a6944fdcd5e62228a7e12b9a2beb6b52118d24e176991f1b0" +
            "4465789607cb06cb2315cc1698ef140bb8962f0139854e21ea2ac0610723" +
            "4034594aae4cf6e63d3f3c039a6f9e535721fce2347c055579ef73d04d12" +
            "7c8f8c69c4889be5674a85aec9fe631fe9836aee734f21b0bdf88c16b462" +
            "8ad735052849d98b497f4997733477827de4584ea01bca09572b40dcbc34" +
            "ccd42af6c42214b8c6c3b5a708d8d40c8916bfec4e1d8da184de4342b40d" +
            "8134189587b381c707d42719de822de64ff277c10203010001a382041230" +
            "82040e30820104060a2b06010401d6790204020481f50481f200f0007600" +
            "f65c942fd1773022145418083094568ee34d131933bfdf0c2f200bcc4ef1" +
            "64e300000174372e743d0000040300473045022100930438131ebc6c1a6d" +
            "b83517ebbfcfecb5dd5d4899edc83d5539a79f59382005022024609430a9" +
            "cf09344c0f88b848adf62cbfd421ffe756b076f7263f2ff0cb00ed007600" +
            "4494652eb0eeceafc44007d8a8fe28c0dae682bed8cb31b53fd33396b5b6" +
            "81a800000174372e745000000403004730450221009bd86214f6f66215b8" +
            "ca63e61f87dd1cf443af22dbeecec5ab2c6c16fc40b0ad0220267015da53" +
            "5c1b557e8c8f9119a8c896d557a31277d9a87e9aee2ce6fdae979f302706" +
            "092b060104018237150a041a3018300a06082b06010505070302300a0608" +
            "2b06010505070301303d06092b06010401823715070430302e06262b0601" +
            "04018237150887da867583eed90182c9851b81b59e6185f4eb60815db9f8" +
            "108691d07802016402012330818706082b06010505070101047b30793053" +
            "06082b060105050730028647687474703a2f2f7777772e6d6963726f736f" +
            "66742e636f6d2f706b692f6d73636f72702f4d6963726f736f6674253230" +
            "525341253230544c53253230434125323030312e637274302206082b0601" +
            "05050730018616687474703a2f2f6f6373702e6d736f6373702e636f6d30" +
            "1d0603551d0e041604148c75c331eb0496672786a7168161398fbc17bb38" +
            "300b0603551d0f0404030204b03081990603551d1104819130818e821377" +
            "777771612e6d6963726f736f66742e636f6d82117777772e6d6963726f73" +
            "6f66742e636f6d8218737461746963766965772e6d6963726f736f66742e" +
            "636f6d8211692e732d6d6963726f736f66742e636f6d820d6d6963726f73" +
            "6f66742e636f6d8211632e732d6d6963726f736f66742e636f6d82157072" +
            "69766163792e6d6963726f736f66742e636f6d3081b00603551d1f0481a8" +
            "3081a53081a2a0819fa0819c864d687474703a2f2f6d7363726c2e6d6963" +
            "726f736f66742e636f6d2f706b692f6d73636f72702f63726c2f4d696372" +
            "6f736f6674253230525341253230544c53253230434125323030312e6372" +
            "6c864b687474703a2f2f63726c2e6d6963726f736f66742e636f6d2f706b" +
            "692f6d73636f72702f63726c2f4d6963726f736f66742532305253412532" +
            "30544c53253230434125323030312e63726c30570603551d200450304e30" +
            "4206092b0601040182372a013035303306082b0601050507020116276874" +
            "74703a2f2f7777772e6d6963726f736f66742e636f6d2f706b692f6d7363" +
            "6f72702f6370733008060667810c010202301f0603551d23041830168014" +
            "b5760c3011cec792424d4cc75c2cc8a90ce80b64301d0603551d25041630" +
            "1406082b0601050507030206082b06010505070301300d06092a864886f7" +
            "0d01010b050003820201004a69de19b185c500841e607746e90c40e37298" +
            "e16ea7c6d6f5517c2efbac51f7f5789213fcb61ba98093bfb17a73e47d78" +
            "273e21c8fd47dbe992065ea83971b6704766dd561f530d15d58acde25d91" +
            "13322a3ca2b1d6bcf454879738a5c350f675e019dd5f67637d0c9265279e" +
            "d528d2e532f5f131f80133251ab58e3bcb018bdd75ba608f348a53a27931" +
            "2e68ae08aa26cc521f9db5b30743d94c7eb49c10563a07379a9cb0d8f48c" +
            "c5efc960bf331d0a9ba43e059f07cfdeb2c3a09e9ef2c0611c5dd6b7af7c" +
            "ecfa99942291d1971eac307fa598570e150d3d2aa07e2e66303d6c1c52e1" +
            "d8dc08b4b1f79898f442ea41e520b1240ecad11eef25fcc4e7a8f76bb2b4" +
            "6c1bf084f14d3ab84c5ee27b5623280ccceb086d700f07139b792202cfca" +
            "76da55075097d7b1e4011a1008fec3f48750d15b06290da418c9b7460fd8" +
            "0e0e2ca0b8b360ba60f411512ecc38a301d2943b48c193cc25cf160007ad" +
            "1c30c7cd98a4571ca68295727a3bf95ab76e1020c45ee16e661530c88bf5" +
            "26e56b4d038eb8882940d78e82038484a3ea581c86bd0d7959d10e185174" +
            "ec060030f3ad6541c4797a301aed40d42cd30fd21d7f0d58b6801e383c19" +
            "0fd4bf9263f72c1ed260370b1add02cf6489898b8c3ae078cb04712d1d07" +
            "3e84756b54b0281649eb9a05e05ef1e939184f9a9244471277b2dd11385d" +
            "af125671c3331768048c27aed2").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComIssuerBytes = (
            "3082055a30820442a00302010202100f14965f202069994fd5c7ac788941" +
            "e2300d06092a864886f70d01010b0500305a310b30090603550406130249" +
            "4531123010060355040a130942616c74696d6f726531133011060355040b" +
            "130a43796265725472757374312230200603550403131942616c74696d6f" +
            "7265204379626572547275737420526f6f74301e170d3230303732313233" +
            "303030305a170d3234313030383037303030305a304f310b300906035504" +
            "0613025553311e301c060355040a13154d6963726f736f667420436f7270" +
            "6f726174696f6e3120301e060355040313174d6963726f736f6674205253" +
            "4120544c5320434120303130820222300d06092a864886f70d0101010500" +
            "0382020f003082020a0282020100aa6277cf9a63b20684f39036f499f314" +
            "51abea950a3b4606fd11411ffe5b0658c9386e08fc4f4448cd3aa4f7bd1e" +
            "a2e295b8be5120c5bfb270635d780c43c029cd64490996daafcefd055f2b" +
            "2a91e8016e2e189b2c9cd0017f69f5ee3f53885cba056cbe2215671482f2" +
            "2cd2be5b6337ccaf6085e8966b6b8008a86ebe009c6b9570fce41812b11d" +
            "1bb2c11331673334e625c9625b58827576f2fef23f3b16dfaa4283e3326d" +
            "9b8e4326f0bd0e1fa1a73aaf2cc88ae6ea3ff9a5d2258f92aa1a08129cfe" +
            "ac4ac7c3eb8094ab8716d12349e7a4bbc791dfe679343f414aa73a26d2ea" +
            "6f46e33873e6e5d491ae0b789e78a5ef96e373d8f79565e905bf4f5cff52" +
            "a7f9cf08afa74d0999c071a3527aa53bd79b015403e3b662b05a279c3026" +
            "8eb64d56a117177a7b95a107ac5331b6d62e0fcd4174ecf101b2fd45bffc" +
            "31e146423136431eb9aa055f847f91b18bae0fd754c3fdf064086ad39c8e" +
            "ea7934ec033d73e01b36d46811c75970b0877cc0dc6e45ca36ce43267702" +
            "a9700de8b857544442c3fbac1b632608c2d2231f7f930b7c6f08549a2b4e" +
            "5dce9fa53ed2985bd102dbf183ce3052483863f1b1fbed23d33e92b5278d" +
            "d04273d79d236871ba595e0752a6964dbf7c4e6f742205c0538016d8604e" +
            "97314f894e4863d8edf9e5c2d90eb20bf6694cbd4b01c9cbdd06bf3a02eb" +
            "1cdd308b0d4a1460f9d5644f4344a1ed0203010001a38201253082012130" +
            "1d0603551d0e04160414b5760c3011cec792424d4cc75c2cc8a90ce80b64" +
            "301f0603551d23041830168014e59d5930824758ccacfa085436867b3ab5" +
            "044df0300e0603551d0f0101ff040403020186301d0603551d2504163014" +
            "06082b0601050507030106082b0601050507030230120603551d130101ff" +
            "040830060101ff020100303406082b060105050701010428302630240608" +
            "2b060105050730018618687474703a2f2f6f6373702e6469676963657274" +
            "2e636f6d303a0603551d1f04333031302fa02da02b8629687474703a2f2f" +
            "63726c332e64696769636572742e636f6d2f4f6d6e69726f6f7432303235" +
            "2e63726c302a0603551d20042330213008060667810c0102013008060667" +
            "810c010202300b06092b0601040182372a01300d06092a864886f70d0101" +
            "0b050003820101009f2bbe92675bda7b8aade8ff9d4d050eedb60d1541d1" +
            "e615dc0360f9f422569c48f99daeda2b3ca8c0abd0ba95b8c8c1fd7c6371" +
            "b6c87a889b3046a38e7d9602e3f82204efe036c06fc2bf2e0d6eedd67628" +
            "0d81873e9be7a7108cda661f4051eae7bebf4e6798bb5459636f42e30f31" +
            "601964000f260c97d184c0a67a193b70de4526dc96463d9c663fe13a8238" +
            "e53603042857a4e94b64a218886d60898d7abe10918bace63f3130bfeb64" +
            "d79e8de9c192566e388d343faecd6c6b4252623cd46989e0a057590b839f" +
            "c6722442f5080384ce1663f334f105763719b206de133e137061d304f2b8" +
            "476f05e38a88302b47455e7954c5f9ddebfa3f785175d25b").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComRootBytes = (
            "308203773082025fa0030201020204020000b9300d06092a864886f70d01" +
            "01050500305a310b300906035504061302494531123010060355040a1309" +
            "42616c74696d6f726531133011060355040b130a43796265725472757374" +
            "312230200603550403131942616c74696d6f726520437962657254727573" +
            "7420526f6f74301e170d3030303531323138343630305a170d3235303531" +
            "323233353930305a305a310b300906035504061302494531123010060355" +
            "040a130942616c74696d6f726531133011060355040b130a437962657254" +
            "72757374312230200603550403131942616c74696d6f7265204379626572" +
            "547275737420526f6f7430820122300d06092a864886f70d010101050003" +
            "82010f003082010a0282010100a304bb22ab983d57e826729ab579d429e2" +
            "e1e89580b1b0e35b8e2b299a64dfa15dedb009056ddb282ece62a262feb4" +
            "88da12eb38eb219dc0412b01527b8877d31c8fc7bab988b56a09e773e811" +
            "40a7d1ccca628d2de58f0ba650d2a850c328eaf5ab25878a9a961ca967b8" +
            "3f0cd5f7f952132fc21bd57070f08fc012ca06cb9ae1d9ca337a77d6f8ec" +
            "b9f16844424813d2c0c2a4ae5e60feb6a605fcb4dd075902d459189863f5" +
            "a563e0900c7d5db2067af385eaebd403ae5e843e5fff15ed69bcf9393672" +
            "75cf77524df3c9902cb93de5c923533f1f2498215c079929bdc63aece76e" +
            "863a6b97746333bd681831f0788d76bffc9e8e5d2a86a74d90dc271a3902" +
            "03010001a3453043301d0603551d0e04160414e59d5930824758ccacfa08" +
            "5436867b3ab5044df030120603551d130101ff040830060101ff02010330" +
            "0e0603551d0f0101ff040403020106300d06092a864886f70d0101050500" +
            "0382010100850c5d8ee46f51684205a0ddbb4f27258403bdf764fd2dd730" +
            "e3a41017ebda2929b6793f76f6191323b8100af958a4d46170bd04616a12" +
            "8a17d50abdc5bc307cd6e90c258d86404feccca37e38c637114feddd6831" +
            "8e4cd2b30174eebe755e07481a7f70ff165c84c07985b805fd7fbe6511a3" +
            "0fc002b4f852373904d5a9317a18bfa02af41299f7a34582e33c5ef59d9e" +
            "b5c89e7c2ec8a49e4e08144b6dfd706d6b1a63bd64e61fb7cef0f29f2ebb" +
            "1bb7f250887392c2e2e3168d9a3202ab8e18dde91011ee7e35ab90af3e30" +
            "947ad0333da7650ff5fc8e9e62cf47442c015dbb1db532d247d2382ed0fe" +
            "81dc326a1eb5ee3cd5fce7811d19c32442ea6339a9").HexToByteArray();

        internal static readonly byte[] Rsa384CertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIICTzCCAgmgAwIBAgIJAMQtYhFJ0+5jMA0GCSqGSIb3DQEBBQUAMIGSMQswCQYD
VQQGEwJVUzETMBEGA1UECAwKV2FzaGluZ3RvbjEQMA4GA1UEBwwHUmVkbW9uZDEY
MBYGA1UECgwPTWljcm9zb2Z0IENvcnAuMSAwHgYDVQQLDBcuTkVUIEZyYW1ld29y
ayAoQ29yZUZ4KTEgMB4GA1UEAwwXUlNBIDM4NC1iaXQgQ2VydGlmaWNhdGUwHhcN
MTYwMzAyMTY1OTA0WhcNMTYwNDAxMTY1OTA0WjCBkjELMAkGA1UEBhMCVVMxEzAR
BgNVBAgMCldhc2hpbmd0b24xEDAOBgNVBAcMB1JlZG1vbmQxGDAWBgNVBAoMD01p
Y3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsgKENvcmVGeCkx
IDAeBgNVBAMMF1JTQSAzODQtYml0IENlcnRpZmljYXRlMEwwDQYJKoZIhvcNAQEB
BQADOwAwOAIxANrMIthuZxV1Ay4x8gbc/BksZeLVEInlES0JbyiCr9tbeM22Vy/S
9h2zkEciMuPZ9QIDAQABo1AwTjAdBgNVHQ4EFgQU5FG2Fmi86hJOCf4KnjaxOGWV
dRUwHwYDVR0jBBgwFoAU5FG2Fmi86hJOCf4KnjaxOGWVdRUwDAYDVR0TBAUwAwEB
/zANBgkqhkiG9w0BAQUFAAMxAEzDg/u8TlApCnE8qxhcbTXk2MbX+2n5PCn+MVrW
wggvPj3b2WMXsVWiPr4S1Y/nBA==
-----END CERTIFICATE-----");

        internal static readonly byte[] ECDsabrainpoolP160r1_Pfx = (
            "308203D00201033082039606092A864886F70D010701A0820387048203833082" +
            "037F3082029F06092A864886F70D010706A08202903082028C02010030820285" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E0408E4D7F5" +
            "F71CA4D5380202080080820258F19503250C93322C81CCC92C57AD76AD4DDF79" +
            "10DBAB6A63D6AAF3F470AC5283BDEB532086D3379B7A3D68D17FAC483EBEA029" +
            "1D2B5F862885E048A3034580A41F238AA836B9E94C12B0656B51C72355AED1DD" +
            "19FE771E3768095D6467FC8742BE0BC5D65360CD875D35C23D272842F64791A1" +
            "53F96AFBD3D7660EC016BF9D59B2B68C2A34D93B133697D6B77DB27649BEEABC" +
            "0B68D35DB3779DD4C871C9C26799E6ABB5E9048DDC44C6E6310F3A023AD09E97" +
            "1AB1DF38FDF3091FB35125EA3A14F5D72A00EC4C637951F026DE79C0E30E0244" +
            "808FB46EFD4EA9C67411DC2B13842B273F405F6D58D45D1D2D47BC1362ED9884" +
            "C2C44EA334A3B02C7E661F489DED798B63D64F90916596BADC87C68C868FCECB" +
            "6F246410186BBB2F2DC2ED24BF215AA54570072E21970CF856398BB1FD8C2F61" +
            "0788231C51D45CE471A235464147A405799F8CBE39AA30A8BFD2534C738AE330" +
            "8771394D871429EF2D6AB6381F793D7CBC0374D4E529B8B6AA37BE04FBE71A9A" +
            "A7954814C0C8841740539ED97DB56C3CBE5851C9D875E4B6023AE095B62C9DC5" +
            "36E06DA40C4B874776CBABDDAB50BDD5ECF9D997EEB1483D3AC23E6F37DD4CBD" +
            "64163E7A115BCE44554C53DD860B83CBE3B35F1E26B87185C602E4FFB3A99458" +
            "0D6A9334F74AA29B3609926FE86F197C955CBAEC2A41EE1572A4295F62D4D9CA" +
            "050CD933BC0BA43D7744EAFA9D6B8253241EB36C605DC334A6470BC709F13985" +
            "8AC238FD7F3C14EDDAB6E29996FE966A96EAC23CF17063C89315734D76CCB21F" +
            "94A7E4A44A5F6DCB93CEEEFB539664296F8F609CFE293200FE4B5EE57AB3A1E7" +
            "A3483DC6243081D906092A864886F70D010701A081CB0481C83081C53081C206" +
            "0B2A864886F70D010C0A0102A0818B308188301C060A2A864886F70D010C0103" +
            "300E0408CD30F3C5A9918832020208000468BF29F33058622BA159CDD3DE2B49" +
            "CBDD736BF1483FF4D43BACCC93B544A513D5E47DB4FECADBB4E3277A6B90345D" +
            "7E73F507924A615D968F807834D3796EFB0A3EF214A75883E3AB90086DA2418B" +
            "0B2D657DEA39A8600172B6975FFB39E88863DB11283A5CEA1FCB312530230609" +
            "2A864886F70D01091531160414472193B362B056F6D6928EFF4C43FF1EFEB173" +
            "4E30313021300906052B0E03021A05000414B703685D5039D8EEF1A46F772F31" +
            "F177FDE874EC0408B4EF89F18902CE9502020800").HexToByteArray();

        internal static readonly byte[] ECDsabrainpoolP160r1_Explicit_Pfx = (
            "30820501020103308204C706092A864886F70D010701A08204B8048204B43082" +
            "04B03082032F06092A864886F70D010706A08203203082031C02010030820315" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E0408C55E7D" +
            "72B26355D202020800808202E8B8BCB9180C8751F860C5F77C04CC921A89D836" +
            "E8BC06DA99238EF8D65FB3473AF97D930B93471B28F08F483062BCEDB44287FA" +
            "E813B6CA812393475F50BB2D0AD233A8CE0832DECE54EF78A1F289A7802A9900" +
            "BAA2B7392BCF6C963D54D8DD63898D5D2FA472E77194763822157664417D93D8" +
            "BF740A4A5683FFFDF8B4CC7E074C6DCFF73E21368D743F0A1CE33484B02A4502" +
            "F7340A853F82F606B10DEA7239EF56C8DBDAED8DD3C956DD4D3E42CA517318E6" +
            "0DF70721469C512E120C500956D960A59EEB4A3A6541891319ACA68AB99462EA" +
            "CB59B52F6C6BFCF24EEA4E19BDDC5DDE1B6F36549DD589E1B10CAA02D4A53C36" +
            "1F1B9F101F9375394CF9BC8D941F94CC4F128EC861AA6B4182E1F81D2C0BADEA" +
            "2E6596BAC8819DE293ECEEC7C54BE97CD3917E0497F2559D44EF3484618DCB9B" +
            "85820762F392925BB36BD482DF544B614CDF9C6728BD92E858C1DF61435E7A03" +
            "DED27DA460603A61BE7DB93870E9530EB51384663E42A49C342891B8E757ED49" +
            "18A590D59734EA1C95E992CD93B80EBD3C246999305C204A813E0DCF26E64446" +
            "15A79E74042C7EAD4FEF0E68AA9DF627B5F4C037BF728015B6BBFA046CAA805C" +
            "BE7B1867262B36172E8759FAE7B965FF9B98D3539A48525E10A57A84C55AEFAC" +
            "3ED9025AB8B0680E0896DDD776C0AFC1A95BDD5DBE0ECCEB860B3CD3D6A2A493" +
            "2BC7774246A6936AFABA9BA8292841F9D6417AFFB00872E9B4ADF11889AEF20A" +
            "FCB8EAEBADAF38A2A240D36940B1585B37F7CA6A887EE1FBA199990FC104CD1F" +
            "A983CC2CE91156559EFCFC0F25B7C24B161DF4B4436F14428C4AE06F49FCC328" +
            "D00891A44AFAE5412FD330E23CFAE6107B4C965DFDB6335E8EFDF2C40767846B" +
            "C95ABF784DE155EED92DAB7A265DC17BC3ADA68D04E378D2E4E8924B05937092" +
            "E779EB04899E4FB19AAE7AA5FCF8D7A33BA464E4BB1FFB4E4D4CD71152F42B68" +
            "F5AB298B10DEB653C7F77F66B761DFD61E4E2DDD13B0B15639473BF5C3B8A31D" +
            "3D2334287F61E1A06F44CD3F2E74F59F43876F0D923082017906092A864886F7" +
            "0D010701A082016A04820166308201623082015E060B2A864886F70D010C0A01" +
            "02A082012630820122301C060A2A864886F70D010C0103300E0408A92CDE5D9A" +
            "F5278D0202080004820100A302E03B1BDF59D4ECD6F0FE7745F7C0DCE2CCEF0E" +
            "B26A7B92839B60288B05445BA1C91109D7E632E0C7B742E2D7D0947573AFEF1F" +
            "FAFCF8135DA3B5EE26A8E3AB7F415A8A19112724F850F024D3527F1FE2A303B1" +
            "34A644307AC6816E59E08540FA782351B27E37775AF3CD904E50A1A76C7C4F34" +
            "7EE78A1ED51FF71D00954130369012750746A280983E883E59AFDBBCCC7D1AA0" +
            "ECDCF2079ECFA4645E156ACC5FD6913763FC93C2E0C572042D00FE4EEB5E75DE" +
            "28C21FA1A7356C4071572DF23CC23833EA26705C0AA080636E27512B5F5755FE" +
            "A0514A31833D52C48A743BCDC0B58257FEDD23EE4EDBC06B574019E792B44BD6" +
            "3F3770875A25075183AF2C3125302306092A864886F70D01091531160414141C" +
            "1C8591A700DDE70FAC750C1539B2DFECAA3C30313021300906052B0E03021A05" +
            "0004143706E218219A899C700BD7AE3E8650FD1B2885AB0408E77FDD798BCADE" +
            "3C02020800").HexToByteArray();

        internal static readonly byte[] ECDsabrainpoolP160r1_CertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIB+TCCAbagAwIBAgIJAIJ8gBL6U36GMAoGCCqGSM49BAMCMHAxCzAJBgNVBAYT
AlVTMRUwEwYDVQQIDAxOb3J0aCBEYWtvdGExDjAMBgNVBAcMBUZhcmdvMRgwFgYD
VQQKDA9NaWNyb3NvZnQgQ29ycC4xIDAeBgNVBAsMFy5ORVQgRnJhbWV3b3JrIChD
b3JlRngpMB4XDTE2MDYxMDE0NTYzOFoXDTE2MDcxMDE0NTYzOFowcDELMAkGA1UE
BhMCVVMxFTATBgNVBAgMDE5vcnRoIERha290YTEOMAwGA1UEBwwFRmFyZ28xGDAW
BgNVBAoMD01pY3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsg
KENvcmVGeCkwQjAUBgcqhkjOPQIBBgkrJAMDAggBAQEDKgAEQk2dep8HoNJcbCal
ie5QIMYsNnphtOo9WUCgrrzEG3wfrxz39HcAXaNQME4wHQYDVR0OBBYEFPprBFD9
qDQynQJmJUpVKv9WR8z5MB8GA1UdIwQYMBaAFPprBFD9qDQynQJmJUpVKv9WR8z5
MAwGA1UdEwQFMAMBAf8wCgYIKoZIzj0EAwIDMQAwLgIVAN3U12PFcEe4HHi+Rio+
xk3lf6EbAhUAqdeGDOXgpHEoWEmzOQ6nWWQik1k=
-----END CERTIFICATE-----");

        internal static readonly byte[] ECDsabrainpoolP160r1_ExplicitCertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIICijCCAkigAwIBAgIJAJVtMTsUqcjsMAoGCCqGSM49BAMCMHAxCzAJBgNVBAYT
AlVTMRUwEwYDVQQIDAxOb3J0aCBEYWtvdGExDjAMBgNVBAcMBUZhcmdvMRgwFgYD
VQQKDA9NaWNyb3NvZnQgQ29ycC4xIDAeBgNVBAsMFy5ORVQgRnJhbWV3b3JrIChD
b3JlRngpMB4XDTE2MDYxMDE1MDg1NVoXDTE2MDcxMDE1MDg1NVowcDELMAkGA1UE
BhMCVVMxFTATBgNVBAgMDE5vcnRoIERha290YTEOMAwGA1UEBwwFRmFyZ28xGDAW
BgNVBAoMD01pY3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsg
KENvcmVGeCkwgdMwgaQGByqGSM49AgEwgZgCAQEwIAYHKoZIzj0BAQIVAOleSl9z
cFncYN/HrZWz2BOVFWIPMCwEFDQOe+KigOt04r5hutp0XZfo98MABBQeWJqFlUI0
EhNPqi297JXI2GdeWAQpBL7VrxbqP2pPYpOMRjHrWve9vNvDFmfLR3oajsM4+UdB
ZpyXYxbaYyECFQDpXkpfc3BZ3GDfWZHUUClAnmD8CQIBAQMqAARz9ueEHonciPIW
lTsd673ZaNpP9GMoFfHns3DnUC0pC1Grh+6sZcPIo1AwTjAdBgNVHQ4EFgQU65OI
c9u4x/ZfIRyjcSaZTUKSsuIwHwYDVR0jBBgwFoAU65OIc9u4x/ZfIRyjcSaZTUKS
suIwDAYDVR0TBAUwAwEB/zAKBggqhkjOPQQDAgMwADAtAhUAxMT7z8lLv7hgWmGh
5siYmHkAExoCFFaaS2r7/kdkXsauyr37q6ewD6s+
-----END CERTIFICATE-----");

        internal static readonly ECDsaCngKeyValues ECDsaCng256PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "448d98ee08aeba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a91".HexToByteArray(),
                QY = "0ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff121586809e3219af".HexToByteArray(),
                D = "692837e9cf613c0e290462a6f08faadcc7002398f75598d5554698a0cb51cf47".HexToByteArray(),
            };

        internal static readonly byte[] EcDh256Certificate = (
            "3082013c3081e3a00302010202142659312913bdce7c9e4f6ad7bc6ddc47e844" +
            "c96c300a06082a8648ce3d040302300f310d300b06035504030c04726f6f7430" +
            "1e170d3230303931303134323033375a170d3330303832393134323033375a30" +
            "0f310d300b06035504030c046c6561663059301306072a8648ce3d020106082a" +
            "8648ce3d030107034200044337f39c7ab746015018de0e23f94a802eae96ade8" +
            "9858cfe482e813cd2bdf40a1ac886f2c1d6135c7dec49ec1ee892d1f8f96e75c" +
            "aff8420d2efed269629e78a31d301b300b0603551d0f040403020328300c0603" +
            "551d130101ff04023000300a06082a8648ce3d040302034800304502206c3019" +
            "8d724bb0501282fcd48aa383362ffe6da299e0701e83a5ab8bb0dcd292022100" +
            "8638a48e53393babfe2236c330211d76e5df95bc1fe2a11b1b4d9350f4268a8c").HexToByteArray();

        internal static readonly byte[] ECDsa256Certificate =
             ("308201223081c9a00302010202106a3c9e85ba6af1ac4f08111d8bdda340300906072a8648ce3d0401301431123010060355"
            + "04031309456332353655736572301e170d3135303931303231333533305a170d3136303931303033333533305a3014311230"
            + "10060355040313094563323536557365723059301306072a8648ce3d020106082a8648ce3d03010703420004448d98ee08ae"
            + "ba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a910ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff"
            + "121586809e3219af300906072a8648ce3d04010349003046022100f221063dca71955d17c8f0e0f63a144c4065578fd9f68e"
            + "1ae6a7683e209ea742022100ed1db6a8be27cfb20ab43e0ca061622ceff26f7249a0f791e4d6be1a4e52adfa").HexToByteArray();

        internal static readonly ECDsaCngKeyValues ECDsaCng384PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "c59eca607aa5559e6b2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bd".HexToByteArray(),
                QY = "d15f307cc6cc6c8baeeeb168bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901".HexToByteArray(),
                D = "f55ba33e28cea32a014e2fe1213bb4d41cef361f1fee022116b15be50feb96bc946b10a46a9a7a94176787e0928a3e1d".HexToByteArray(),
            };

        internal static readonly byte[] ECDsa384Certificate =
             ("3082015f3081e6a00302010202101e78eb573e70a2a64744672296988ad7300906072a8648ce3d0401301431123010060355"
            + "04031309456333383455736572301e170d3135303931303231333634365a170d3136303931303033333634365a3014311230"
            + "10060355040313094563333834557365723076301006072a8648ce3d020106052b8104002203620004c59eca607aa5559e6b"
            + "2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bdd15f307cc6cc6c8baeeeb1"
            + "68bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901300906072a8648ce3d04010369"
            + "003066023100a8fbaeeae61953897eae5f0beeeffaca48e89bc0cb782145f39f4ba5b03390ce6a28e432e664adf5ebc6a802"
            + "040b238b023100dcc19109383b9482fdda68f40a63ee41797dbb8f25c0284155cc4238d682fbb3fb6e86ea0933297e850a26"
            + "16f6c39bbf").HexToByteArray();

        internal static readonly ECDsaCngKeyValues ECDsaCng521PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "0134af29d1fe5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb693081dc6490dac579c0".HexToByteArray(),
                QY = "00bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653eff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0".HexToByteArray(),
                D = "0153603164bcef5c9f62388d06dcbf5681479be4397c07ff6f44bb848465e3397537d5f61abc7bc9266d4df6bae1df4847fcfd3dabdda37a2fe549b821ea858d088d".HexToByteArray(),
            };

        internal static readonly ECDsaCngKeyValues ECDsabrainpoolP160r1_PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "424D9D7A9F07A0D25C6C26A589EE5020C62C367A".HexToByteArray(),
                QY = "61B4EA3D5940A0AEBCC41B7C1FAF1CF7F477005D".HexToByteArray(),
            };

        internal static readonly byte[] ECDsa521Certificate =
             ("308201a93082010ca00302010202102c3134fe79bb9daa48df6431f4c1e4f3300906072a8648ce3d04013014311230100603"
            + "5504031309456335323155736572301e170d3135303931303231333832305a170d3136303931303033333832305a30143112"
            + "30100603550403130945633532315573657230819b301006072a8648ce3d020106052b8104002303818600040134af29d1fe"
            + "5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb"
            + "693081dc6490dac579c000bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653e"
            + "ff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0300906072a8648ce3d040103818b0030818702420090bdf5"
            + "dfb328501910da4b02ba3ccd41f2bb073608c55f0f2b2e1198496c59b44db9e516a6a63ba7841d22cf590e39d3f09636d0eb"
            + "cd59a92c105f499e1329615602414285111634719b9bbd10eb7d08655b2fa7d7eb5e225bfdafef15562ae2f9f0c6a943a7bd"
            + "f0e39223d807b5e2e617a8e424294d90869567326531bcad0f893a0f3a").HexToByteArray();

        internal static readonly byte[] EccCert_KeyAgreement = (
            "308201553081FDA00302010202105A1C956450FFED894E85DC61E11CD968300A" +
            "06082A8648CE3D04030230143112301006035504030C09454344482054657374" +
            "301E170D3135303433303138303131325A170D3136303433303138323131325A" +
            "30143112301006035504030C094543444820546573743059301306072A8648CE" +
            "3D020106082A8648CE3D0301070342000477DE73EA00A82250B69E3F24A14CDD" +
            "C4C47C83993056DD0A2C6C17D5C8E7A054216B9253533D12C082E0C8B91B3B10" +
            "CDAB564820D417E6D056E4E34BCCA87301A331302F300E0603551D0F0101FF04" +
            "0403020009301D0603551D0E0416041472DE05F588BF2741C8A28FF99EA399F7" +
            "AAB2C1B3300A06082A8648CE3D040302034700304402203CDF0CC71C63747BDA" +
            "2D2D563115AE68D34867E74BCA02738086C316B846CDF2022079F3990E5DCCEE" +
            "627B2E6E42317D4D279181EE695EE239D0C8516DD53A896EC3").HexToByteArray();

        internal static readonly byte[] ECDsa224Certificate = (
            "3082026630820214A003020102020900B94BCCE3179BAA21300A06082A8648CE" +
            "3D040302308198310B30090603550406130255533113301106035504080C0A57" +
            "617368696E67746F6E3110300E06035504070C075265646D6F6E64311E301C06" +
            "0355040A0C154D6963726F736F667420436F72706F726174696F6E3120301E06" +
            "0355040B0C172E4E4554204672616D65776F726B2028436F7265465829312030" +
            "1E06035504030C174E4953542F53454320502D3232342054657374204B657930" +
            "1E170D3135313233313232353532345A170D3136303133303232353532345A30" +
            "8198310B30090603550406130255533113301106035504080C0A57617368696E" +
            "67746F6E3110300E06035504070C075265646D6F6E64311E301C060355040A0C" +
            "154D6963726F736F667420436F72706F726174696F6E3120301E060355040B0C" +
            "172E4E4554204672616D65776F726B2028436F72654658293120301E06035504" +
            "030C174E4953542F53454320502D3232342054657374204B6579304E30100607" +
            "2A8648CE3D020106052B81040021033A000452FF02B55AE35AA7FFF1B0A82DC2" +
            "260083DD7D5893E85FBAD1D663B718176F7D5D9A04B8AEA968E9FECFEE348CDB" +
            "49A938401783BADAC484A350304E301D0603551D0E041604140EA9C5C4681A6E" +
            "48CE64E47EE8BBB0BA5FF8AB3E301F0603551D230418301680140EA9C5C4681A" +
            "6E48CE64E47EE8BBB0BA5FF8AB3E300C0603551D13040530030101FF300A0608" +
            "2A8648CE3D040302034000303D021D00AC10B79B6FD6BEE113573A1B68A3B771" +
            "3B9DA2719A9588376E334811021C1AAC3CA829DA79CE223FA83283E6F0A5A59D" +
            "2399E140D957C1C9DDAF").HexToByteArray();

        private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows_RC2ContentEncryption = (
            "308204470201033082040306092A864886F70D010701A08203F4048203F03082" +
            "03EC3082016D06092A864886F70D010701A082015E0482015A30820156308201" +
            "52060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
            "0103300E0408EC154269C5878209020207D00481A80BAA4AF8660E6FAB7B050B" +
            "8EF604CFC378652B54FE005DC3C7E2F12E5EFC7FE2BB0E1B3828CAFE752FD64C" +
            "7CA04AF9FBC5A1F36E30D7D299C52BF6AE65B54B9240CC37C04E7E06330C24E9" +
            "6D19A67B7015A6BF52C172FFEA719B930DBE310EEBC756BDFF2DF2846EE973A6" +
            "6C63F4E9130083D64487B35C1941E98B02B6D5A92972293742383C62CCAFB996" +
            "EAD71A1DF5D0380EFFF25BA60B233A39210FD7D55A9B95CD8A440DF666317430" +
            "1306092A864886F70D0109153106040401000000305D06092B06010401823711" +
            "0131501E4E004D006900630072006F0073006F0066007400200053006F006600" +
            "7400770061007200650020004B00650079002000530074006F00720061006700" +
            "65002000500072006F007600690064006500723082027706092A864886F70D01" +
            "0706A0820268308202640201003082025D06092A864886F70D010701301C060A" +
            "2A864886F70D010C0106300E0408175CCB1790C48584020207D080820230E956" +
            "E38768A035D8EA911283A63F2E5B6E5B73231CFC4FFD386481DE24B7BB1B0995" +
            "D614A0D1BD086215CE0054E01EF9CF91B7D80A4ACB6B596F1DFD6CBCA71476F6" +
            "10C0D6DD24A301E4B79BA6993F15D34A8ADB7115A8605E797A2C6826A4379B65" +
            "90B56CA29F7C36997119257A827C3CA0EC7F8F819536208C650E324C8F884794" +
            "78705F833155463A4EFC02B5D5E2608B83F3CAF6C9BB97C1BBBFC6C5584BDCD3" +
            "9C46A3944915B3845C41429C7792EB4FA3A7EDECCD801F31A4B6EF57D808AEEA" +
            "AF3D1F55F378EF8EF9632CED16EDA3EFBE4A9D5C5F608CA90A9AC8D3F86462AC" +
            "219BFFD0B8A87DDD22CF029230369B33FC2B488B5F82702EFC3F270F912EAD2E" +
            "2402D99F8324164C5CD5959F22DEC0D1D212345B4B3F62848E7D9CFCE2224B61" +
            "976C107E1B218B4B7614FF65BCCA388F85D6920270D4C588DEED323C416D014F" +
            "5F648CC2EE941855EB3C889DCB9A345ED11CAE94041A86ED23E5789137A3DE22" +
            "5F4023D260BB686901F2149B5D7E37102FFF5282995892BDC2EAB48BD5DA155F" +
            "72B1BD05EE3EDD32160AC852E5B47CA9AEACE24946062E9D7DCDA642F945C9E7" +
            "C98640DFAC7A2B88E76A560A0B4156611F9BE8B3613C71870F035062BD4E3D9F" +
            "D896CF373CBFBFD31410972CDE50739FFB8EC9180A52D7F5415EBC997E5A4221" +
            "349B4BB7D53614630EEEA729A74E0C0D20726FDE5814321D6C265A7DC6BA24CA" +
            "F2FCE8C8C162733D58E02E08921E70EF838B95C96A5818489782563AE8A2A85F" +
            "64A95EB350FF8EF6D625AD031BCD303B301F300706052B0E03021A0414C8D96C" +
            "ED140F5CA3CB92BEFCA32C690804576ABF0414B59D4FECA9944D40EEFDE7FB96" +
            "196D167B0FA511020207D0").HexToByteArray();

        private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows_TripleDESContentEncryption = (
            "3082041D020103308203D906092A864886F70D010701A08203CA048203C63082" +
            "03C2308201CB06092A864886F70D010701A08201BC048201B8308201B4308201" +
            "B0060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
            "0103300E04081DE327FA4C98D20D020207D00481A873FF8BEF537D3D09147413" +
            "9C8DE1AE756D8932D5724A2CA5BA03060FB6FD8A62EBC0074958924ABEEFE40F" +
            "A9F990DC9F554F1F4A1796CEF9267804B5D810974186136D40A864B124395321" +
            "8A28EF057EB581403D4BEB59D78CFE7A3E0BB447FB84C3A430346720E9D420AA" +
            "5CF510047B37FC605C1C9ACDCDCD29BDAE4A322FAB6DED4CF88DDCDD2415BC68" +
            "5D755B2F7C4FAC7CB8432103748E946E8FB57A799A227E82834063AF1B3181D1" +
            "301306092A864886F70D0109153106040401000000305B06092A864886F70D01" +
            "0914314E1E4C007B00450039004100350045003100300044002D004200370035" +
            "0032002D0034003700390034002D0039003800370031002D0037004100440046" +
            "00450037004500460044003200350033007D305D06092B060104018237110131" +
            "501E4E004D006900630072006F0073006F0066007400200053006F0066007400" +
            "770061007200650020004B00650079002000530074006F007200610067006500" +
            "2000500072006F00760069006400650072308201EF06092A864886F70D010706" +
            "A08201E0308201DC020100308201D506092A864886F70D010701301C060A2A86" +
            "4886F70D010C0103300E040864425808A9BD6600020207D0808201A855AC3396" +
            "F8C4B26F71605DF7D89868094B973AA2FC3F94CBF0CD876EC8DA9D828BE05496" +
            "5B90C91D8997A0B69908994F07E75BE4BB0CA9257BFC2C26EA3788BD85CEBBAF" +
            "BFAB13EBCD1389DEE25666C76228D0845BDC3D03F4D1AED8BD6D10D2753AED6C" +
            "BFF979E5B11F646EB26554324C7CD4BFDB0D9406EAB9BB177DDD2BEF5DDFF25D" +
            "34A2256CF1444128B28DDD9758FEEC747E8BD453D69B6DD63FA3AFA65853D0B3" +
            "D0C93BF95F9F6C4F8A92E3C6A5850BC3DC27C10F3596F73FE7EEDA3FF81C4493" +
            "CD6D8564D5D3148AFC60029115ABFC4623A2662237F1FA4CF7E9E9FD26E18CD7" +
            "1C6FA3D4020110E6CE1C62A9CB1ADF23AE8DEA8265C6A50325FB37F05F9AA408" +
            "ED019F2A253ECAA24737511B8298BE651A97AE0BF67155FCFDDC8CF4700A8D81" +
            "FDB27FD8162BCA60F25F265A3007A651F4DAE2539B4879689C594FB2FFCC5F70" +
            "77DBD31E90844B7BB383BF6D71905988102A615096A8B056F405A87B15FCFC71" +
            "F8EC9BDB83465D8DEC2A45ED973A06C7F155E2144FDD3F51A2F0D181684D8D5E" +
            "0203C7DC33F2A27371496026FD49E83B67D1C16553440B56F215421863A14EA1" +
            "98B1777F303B301F300706052B0E03021A041463C16FE8BB5114509BA9D023CB" +
            "383E5DCE6904080414A1E1FBAD8DD3BDE73063DC6828C83C4B247E8F41020207" +
            "D0").HexToByteArray();

        internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows =
            PlatformSupport.IsRC2Supported
                ? ECDsaP256_DigitalSignature_Pfx_Windows_RC2ContentEncryption
                : ECDsaP256_DigitalSignature_Pfx_Windows_TripleDESContentEncryption;

        // The PFX in ECDsaP256_DigitalSignature_Pfx_Windows_RC2ContentEncryption washed through OpenSSL
        private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_OpenSsl_RC2ContentEncryption = (
            "308203BE0201033082038406092A864886F70D010701A0820375048203713082" +
            "036D308201FF06092A864886F70D010706A08201F0308201EC020100308201E5" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E040888F579" +
            "00302DB63A02020800808201B8F5EDB44F8B2572E85E52946B233A47F03DF776" +
            "BC3A05CB74B4145A9D3AE3C7FD61B330194E1E154B89929F3FA3908FEE95512A" +
            "052FDDE8E1913E2CCFD803EE6D868696D86934DCF5300DC951F36BE93E3F4AA2" +
            "096B926CF8410AF77FFA087213F84F17EB1D36B61AF4AAD87288301569239B9A" +
            "B66392ADA3D468DC33F42FCEC3BEE78148CA72686BB733DB89FC951AE92FD0F7" +
            "D5937DE78B1AF984BD13E5127F73A91D40097976AEF00157DCC34B16C1724E5B" +
            "88090A1A2DA7337C72720A7ED8F1A89C09AB4143C3F6D80B1925AB8F744069F6" +
            "399D997827F7D0931DCB5E3B09783D1D8555910906B33AD03759D292021C21A2" +
            "9EA2F29CF9BA4D66E4E69AA9FDCCCB4D49A806DBB804EBEBAED7AE0DD4AD2133" +
            "1482A3CC5DB246CE59998824B7E46F337F8887D990FA1756D6A039D293B243BB" +
            "DCFB19AD613A42C5778E7094EA43C3136EF359209790462A36CF87D89B6D76CF" +
            "BD8C34B8C41D96C83683751B8B067F42017A37D05B599B82B70830B5A93499A0" +
            "A4791F5DAB2143C8DF35EC7E88B71A0990E7F6FEA304CE594C9280D7B9120816" +
            "45C87112B1ED85124533792ABEF8B4946F811FB9FE922F6F786E5BFD7D7C43F6" +
            "48AB43C43F3082016606092A864886F70D010701A0820157048201533082014F" +
            "3082014B060B2A864886F70D010C0A0102A081B43081B1301C060A2A864886F7" +
            "0D010C0103300E0408F58B95D6E307213C02020800048190E0FB35890FFB6F30" +
            "7DD0BD8B10EB10488EAB18702E5AC9F67C557409DF8E3F382D06060FB3B5A08D" +
            "1EA31313E80A0488B4034C8906BD873A5308E412783684A35DBD9EEACF5D090D" +
            "AE7390E3309D016C41133946A6CF70E32BE8002CD4F06A90F5BBCE6BF932EC71" +
            "F634312D315310CE2015B30C51FCC54B60FB3D6E7B734C1ADEBE37056A46AB3C" +
            "23276B16603FC50C318184302306092A864886F70D01091531160414F20D17B7" +
            "9B898999F0AA1D5EA333FAEF2BDB2A29305D06092B060104018237110131501E" +
            "4E004D006900630072006F0073006F0066007400200053006F00660074007700" +
            "61007200650020004B00650079002000530074006F0072006100670065002000" +
            "500072006F0076006900640065007230313021300906052B0E03021A05000414" +
            "96C2244022AB2B809E0F97270F7F4EA7769DD26F04084C0E2946D65F8F220202" +
            "0800").HexToByteArray();

        // The PFX in ECDsaP256_DigitalSignature_Pfx_Windows_TripleDESContentEncryption washed through OpenSSL
        private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_OpenSsl_TripleDESContentEncryption = (
            "308203530201033082030F06092A864886F70D010701A0820300048202FC3082" +
            "02F83081FA06092A864886F70D010701A081EC0481E93081E63081E3060B2A86" +
            "4886F70D010C0A0102A081BC3081B93024060A2A864886F70D010C0103301604" +
            "108460E312803BC7452A54A463A320AA10020207D0048190246FA090D57DD7DE" +
            "35F4EBFBE6FC75852637125874D59B7BDD61AAA617C7AC7F72CB08621A875061" +
            "A0E1D362F4E2571F67BA5970264C79FD3D49CB12986E92E016326B710D7EC853" +
            "70266F357DC6CA76E4878A29B93D90CA8C65D632A71ED9E5AEA78D48EE135DCA" +
            "A68EAA5D229522873483523CB61F4644EA9DA46DBAA5D98B63EA494CF006B748" +
            "7F1BC1CB11473AF63115301306092A864886F70D010915310604040000000030" +
            "8201F706092A864886F70D010706A08201E8308201E4020100308201DD06092A" +
            "864886F70D0107013024060A2A864886F70D010C010330160410C73C2C250B71" +
            "DBDD1775F6E0DB924D64020207D0808201A81FF585B0354DFA72B861025E8114" +
            "D5329CD4BEA43DAC69FB5B8D46372731184C9F96D56C53FA9A7201ACD1F56397" +
            "0344B4656F5F90A3267F29C8D233D9544FD3596931F223908EEB33343565D11D" +
            "296A5C32372C10FA71218D199C461A5B9011A996A02308C572F914599AA0D138" +
            "F162B8AD7DCB67BCAC82FD858944426F657907310F63095868C05CED94243858" +
            "95F2707B36D82C919AC40AC0DC0026D9159ECDB9CD48F71BAD82B234B0A2A631" +
            "4DEE0221DE6DE4D27EF0FE654633AE90DAF86FF661FB0DA5C262FC5B38DE26C6" +
            "5971057A22C081C5D95A25EAB12C655008254F7AC0B35BD1974BCF9EE78A4172" +
            "879FFE771022F98F2BD67732BF0D318CD774BC3D89D7861C0D42AC24D0323F04" +
            "7AAE0D64277FE037D8B12BCE2C51F1739CD39CBDB1AABD86374F213B1602DF6D" +
            "F3A0AEFE14456FD1703D7DC90E111204FCAF1FBCBD94CACA097FF1B258CC0C23" +
            "E4CD3BBFB66988A17279D048D8D344888C395F6F52D303FBEFF032985511E795" +
            "3C345EE9B3B9043280A3CB1ABFF9127DD4431E8E88A8CCCF753BF822B0BBD9A9" +
            "4B5C5C547D8B535B455F647180409C50F8379B46A6A96C5A9A96303B301F3007" +
            "06052B0E03021A0414D38DEF52C1BA0A1BA96F04916183C4CAE0F9D8490414C2" +
            "B3F3C6F9C1C07BA92E77D8016E51A7D38C04FF020207D0").HexToByteArray();

        internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_OpenSsl =
            PlatformSupport.IsRC2Supported
                ? ECDsaP256_DigitalSignature_Pfx_OpenSsl_RC2ContentEncryption
                : ECDsaP256_DigitalSignature_Pfx_OpenSsl_TripleDESContentEncryption;

        internal static readonly byte[] EcDhP256_KeyAgree_Pfx_OpenSsl = (
            "308203420201033082030806092a864886f70d010701a08202f9048202f53082" +
            "02f1308201e706092a864886f70d010706a08201d8308201d4020100308201cd" +
            "06092a864886f70d010701301c060a2a864886f70d010c0106300e0408ad4c9e" +
            "aef9eed46a02020800808201a0525f00a9eb7ac25c488a8ee0ac1dd6d866d390" +
            "3ecdd1366ad1927c8aae0aaf9fcee84ef7af78398eb66527027a0ce34b6bc4c3" +
            "208815343a8eb5d14aca1c683a83b41307ab94f4acc4d2eb3cd74bcc32b934bd" +
            "dd167466854b094f578e726d54834b2bc343182b491f282805a0833753af05c3" +
            "59f7c00106f0b3024c5cd6de88276a282400bc34017076dcef29ffc4f960313e" +
            "fe9e28f2513c251311cf491aa2ea74d3d078f561c954c19e5ecea20983965647" +
            "81492ffbd15210bb52f1099cb70c5cf43b20ac59afccb43197654c069e8f106b" +
            "68acea7006f833273658d2fb6f87d46e51663fcc94b32134338f8e307ce9cb23" +
            "a39f39308897e8c862e96f2c8a2b26d0985338bb9b20cc31632d2bfbc9eb7049" +
            "37ee8cd6db47a5bbaf15d82bf998dd71154d30b3c6fa946243bde7941506f0a5" +
            "8e1949876b7d4c0c5031b92118dd82411e474327b2f3f31a64225f2636cd20e4" +
            "b7b5438f80b464eb4c337a4c28ce3a5d169291d47cc1bc730383c6b4a0bee781" +
            "c0708d49e8f25b95f3b5690d3c473b461af676dfec0f30b7091165d1a23b5267" +
            "e9f37995b126820a158a9a70c23082010206092a864886f70d010701a081f404" +
            "81f13081ee3081eb060b2a864886f70d010c0a0102a081b43081b1301c060a2a" +
            "864886f70d010c0103300e0408f0bddd30f5305af602020800048190b8f2d218" +
            "4c9c6e1d5ac465415dc26e0fa3b0ae4c1d328a0909ce24291c1f1839b1ef25de" +
            "6f5429d86d87a29c61bb61fb5040a7b3ee4e03af4e72f75af2664e37ba8eb5b7" +
            "161d6b99404649e4e31d3ccfc16ae7dfbda3455a5ca18d11de6e1ee880227343" +
            "6b73e14dcd687e59dd9fbf3b8d51b2b65f7fbfa7ed9068446c744ecdd52409ef" +
            "df4c040aad52c1dbd80ae6ff3125302306092a864886f70d010915311604146e" +
            "ae9d3e34f7672106585583aa4623b6cc5ae2f730313021300906052b0e03021a" +
            "05000414d82cbf4010f6e9e061f5837b6dff6a4f9cbcef0504087dacc651dd12" +
            "f86202020800").HexToByteArray();

        private static readonly byte[] EcDhP256_KeyAgree_Pfx_Windows_RC2ContentEncryption = (
            "30820405020103308203c106092a864886f70d010701a08203b2048203ae" +
            "308203aa308201cb06092a864886f70d010701a08201bc048201b8308201" +
            "b4308201b0060b2a864886f70d010c0a0102a081cc3081c9301c060a2a86" +
            "4886f70d010c0103300e04089ea31f5e7b87dca9020207d00481a86addd4" +
            "d0131ed9c34f608feb8b5d7cebbe532399969c2a9c3303c3a754e11b5b0c" +
            "5427490b299b2678001bd0401f7b2a50e3ed70aac4c7e8cb286d7732318d" +
            "f770b3f0ed70240018ab8a51cec13cd07fc756ff0807c1b747477719c6ab" +
            "89b9333e860db5fa4be13aae617586b8f2d66debd0f10b0ce52fae47c370" +
            "345b854b58d48efb3bf3cdc2cccd20a0e4cec392d0cc34ccbf1ce9f2425b" +
            "75a6779310ae46b0f22c617a02c9743181d1301306092a864886f70d0109" +
            "153106040401000000305b06092a864886f70d010914314e1e4c007b0039" +
            "0035004600330030004400460035002d0032004300330044002d00340032" +
            "00360044002d0038004100360032002d0045003500370042004500410033" +
            "00410041003800390041007d305d06092b060104018237110131501e4e00" +
            "4d006900630072006f0073006f0066007400200053006f00660074007700" +
            "61007200650020004b00650079002000530074006f007200610067006500" +
            "2000500072006f00760069006400650072308201d706092a864886f70d01" +
            "0706a08201c8308201c4020100308201bd06092a864886f70d010701301c" +
            "060a2a864886f70d010c0103300e04086dd7b320c50b9e75020207d08082" +
            "0190cec985992355b05c38120fac763e2e44d5921a075356eb7451e29527" +
            "1d197af96d45b5f8038164ff9b16cb0232142263b9482a4fce016d109ce8" +
            "f4db6e2b06319d6cea46b02bc133af9c86c50480f046af9d161426b77303" +
            "c4c905048384fdb11c5efea2561cb8b1c77e1c04a8f708ca8baa37f4f888" +
            "a597b9478f43287bf69e8eb6b0bf2fd5a8a1a0cc90cfa17db2af68a0d0a8" +
            "4161d6bb1d55df5a6b66e627c22dccaf5936fbb0169b0115725733f585ae" +
            "8f009f9c1b57fb25fd793dba88bcaf7a9e8e2c55a095365afdb2fba3588d" +
            "06d4431b0f5858cb052fba98d6e2541143821f663c06e835b816cfcb4c40" +
            "ee810b1256ded3b3a568f224cb469c4995d43be2beaaa685f3b37e42111d" +
            "c6e7013119c8b2fa4199eb7b73475f9b60604643253ac2f28ce1faa53c07" +
            "199b54fd60a4eba53589f0bf7b0a4873b88981258c1a2ff517ea4c56a324" +
            "18503862ebce2ad54e38c21b9dfc80ba1f6483e7d2cb21c74c82addf200c" +
            "44ec9602d1efd3958a09d2850c44120eddd2f26c062abfa21cc2bbf743a9" +
            "5f66285bc43d07023f7e1211303b301f300706052b0e03021a04144cd1a9" +
            "a7c08bafd5f9b98e28918201f4e2fc43fa0414fcb0ee0d7e9f504e754223" +
            "b786ccc551a220ba26020207d0").HexToByteArray();

        private static readonly byte[] EcDhP256_KeyAgree_Pfx_Windows_TripleDESContentEncryption = (
            "30820405020103308203C106092A864886F70D010701A08203B2048203AE3082" +
            "03AA308201CB06092A864886F70D010701A08201BC048201B8308201B4308201" +
            "B0060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
            "0103300E040848F7290AA46B2312020207D00481A8A86B9A112AA5CCA4841E58" +
            "5179D13261F3458710DD49D60F7EE45C3E80B9FEB74DEB88069A123E115F1782" +
            "F100E2E871411334C3BB7288891F5ADAE79E99AC3A0A9C57A8267A0F4639C622" +
            "8DB26A64E0C6AC6071ED39C9419024E2A53100416C95F00FAFBF0435EFFEFE44" +
            "7940C9556D6A25807195C0F3B62B694A418D4C2010B2DD9C5D1F237B48F72809" +
            "B7D99FE7E698DD8AF99FEE5B674D83962713AB882F1864C5D432EBAC033181D1" +
            "301306092A864886F70D0109153106040401000000305B06092A864886F70D01" +
            "0914314E1E4C007B00300039004400310043004400360038002D003200350030" +
            "0045002D0034004100340041002D0038003100420046002D0035004100390043" +
            "00410032003900420035004300380036007D305D06092B060104018237110131" +
            "501E4E004D006900630072006F0073006F0066007400200053006F0066007400" +
            "770061007200650020004B00650079002000530074006F007200610067006500" +
            "2000500072006F00760069006400650072308201D706092A864886F70D010706" +
            "A08201C8308201C4020100308201BD06092A864886F70D010701301C060A2A86" +
            "4886F70D010C0103300E04087C40341C9DF40C5D020207D080820190644D3F66" +
            "DAC98F21BB3D0E20BEFC7847CCF6DFAA5272D13B9930027705DA8CEC12FDAD28" +
            "A184506A1DEC64C78442A0D4E320A32BF7A6F6C6C7F5A0C27FEF78D45C770978" +
            "3DAF447B845E60935002F46C9B78A85ADDFB25D776017BCBBE839404F43E63E2" +
            "25427874DE824EAA8FC87172D83B74C5676B5FFCCC4CCE7AECAD272167166A7D" +
            "B72793E41635234E3B3D2D25A72F59C8580C73E0FF08370146FC208FE3C7DA97" +
            "34527EAB978CD8B6F6EB758E7B0A066889E7A7CCDAC45AC271C90D673860C164" +
            "5975B5755EBD94203F9F0F2E1F79F3089A43FA6BEF8F0B4FFF8A0946C31550CD" +
            "9E00B0C0D8A11FC527CA819EA8B5109B8374C0C258AC6C0979B4090B225F6743" +
            "5CA7729BFAD1ED500C63CE9B0041462217FAE93DFF6248E7F43E0E4AC6912750" +
            "6B41CE1AAACD541ED7D41ABE738132D10BE54A16707B70C554A1D011BAC2AF76" +
            "FD3012EFBDDC820D8DB6801BB53742AD7A0CF3240B42CABA6EAF0BE79CF13DBD" +
            "DFD06156C4FDFDFD5C408F9ECCF0F154E563438533603C0A9A531C98F0894721" +
            "B0FC90D1A0FCA39A80C4BFF8303B301F300706052B0E03021A041469ED25801A" +
            "54E840547E4DD26ABDBBE7FBE329320414B77DAD6004C21BEF736BC9AC28CE55" +
            "C61FF7C3D0020207D0").HexToByteArray();

        public static readonly byte[] EcDhP256_KeyAgree_Pfx_Windows =
            PlatformSupport.IsRC2Supported
                ? EcDhP256_KeyAgree_Pfx_Windows_RC2ContentEncryption
                : EcDhP256_KeyAgree_Pfx_Windows_TripleDESContentEncryption;

        internal struct ECDsaCngKeyValues
        {
            public byte[] QX;
            public byte[] QY;
            public byte[] D;
        }

        internal static readonly RSAParameters RsaBigExponentParams = new RSAParameters
        {
            Modulus = (
                "AF81C1CBD8203F624A539ED6608175372393A2837D4890E48A19DED369731156" +
                "20968D6BE0D3DAA38AA777BE02EE0B6B93B724E8DCC12B632B4FA80BBC925BCE" +
                "624F4CA7CC606306B39403E28C932D24DD546FFE4EF6A37F10770B2215EA8CBB" +
                "5BF427E8C4D89B79EB338375100C5F83E55DE9B4466DDFBEEE42539AEF33EF18" +
                "7B7760C3B1A1B2103C2D8144564A0C1039A09C85CF6B5974EB516FC8D6623C94" +
                "AE3A5A0BB3B4C792957D432391566CF3E2A52AFB0C142B9E0681B8972671AF2B" +
                "82DD390A39B939CF719568687E4990A63050CA7768DCD6B378842F18FDB1F6D9" +
                "FF096BAF7BEB98DCF930D66FCFD503F58D41BFF46212E24E3AFC45EA42BD8847").HexToByteArray(),

            Exponent = new byte[] { 0x02, 0x00, 0x00, 0x04, 0x41 },

            D = (
                "64AF9BA5262483DA92B53F13439FD0EF13012F879ABC03CB7C06F1209904F352" +
                "C1F223519DC48BFAEEBB511B0D955F6167B50E034FEA2ABC590B4EA9FBF0C51F" +
                "9FFEA16F7927AE681CBF7358452BCA29D58705E0CAA106013B09A6F5F5911498" +
                "D2C4FD6915585488E5F3AD89836C93C8775AFAB4D13C2014266BE8EE6B8AA66C" +
                "9E942D493466C8E3A370F8E6378CE95D637E03673670BE4BCACE5FCDADD238D9" +
                "F32CA35DE845776AC4BF36118812328C493F91C25A9BD42672D0AFAFDE0AF7E6" +
                "19078D48B485EF91933DDCFFB54587B8F512D223C81894E91784982F3C5C6587" +
                "1351F4655AB023C4AD99B6B03A96F9046CE124A471E828F05F8DB3BC7CCCF2D1").HexToByteArray(),

            P = (
                "E43A3826A97204AE3CD8649A84DB4BBF0725C4B08F8C43840557A0CD04E313AF" +
                "6D0460DDE69CDC508AD043D72514DA7A66BC918CD9624F485644B9DEEAB2BE0E" +
                "112956D472CF0FD51F80FD33872D2DCC562A0588B012E8C90CE7D254B94792C6" +
                "E7A02B3CCAA150E67A64377ACC49479AD5EB555493B2100CB0410956F7D73BF5").HexToByteArray(),

            Q = (
                "C4DD2D7ADD6CA50740D3973F40C4DEBDBAB51F7F5181ABAE726C32596A3EDD0A" +
                "EE44DAADDD8A9B7A864C4FFDAE00C4CB1F10177BA01C0466F812D522610F8C45" +
                "43F1C3EF579FA9E13AE8DA1A4A8DAE307861D2CEAC03560279B61B6514989883" +
                "FE86C5C7420D312838FC2F70BED59B5229654201882664CEFA38B48A3723E9CB").HexToByteArray(),

            DP = (
                "09ECF151F5CDD2C9E6E52682364FA5B4ED094F622E4031BF46B851358A584DCC" +
                "B5328B0BD9B63589183F491593D2A3ACAD14E0AACDA1F181B5C7D93C57ED26E6" +
                "2C9FC26AF37E4A0644ECE82A7BA8AED88FF1D8E9C56CC66385CDB244EB3D57D1" +
                "7E6AD420B19C9E2BEE18192B816265B74DA55FA3825F922D9D8E835B76BF3071").HexToByteArray(),

            DQ = (
                "89B33B695789174B88368C494639D4D3267224572A40B2FE61910384228E3DBD" +
                "11EED9040CD03977E9E0D7FC8BFC4BF4A93283529FF1D96590B18F4EABEF0303" +
                "794F293E88DC761B3E23AFECB19F29F8A4D2A9058B714CF3F4D10733F13EA72B" +
                "BF1FBEC8D71E106D0CE2115F3AD2DE020325C3879A091C413CD6397F83B3CB89").HexToByteArray(),

            InverseQ = (
                "7C57ED74C9176FBA76C23183202515062C664D4D49FF3E037047A309DA10F159" +
                "0CE01B7A1CD1A4326DC75883DFF93110AB065AAED140C9B98176A8810809ADEC" +
                "75E86764A0951597EF467FA8FD509181CD2E491E43BE41084E5BE1B562EE76E9" +
                "F92C9AB1E5AEAD9D291A6337E4DE85BDE67A0D72B4E55ADCF207F7A5A5225E15").HexToByteArray()
        };

        internal static readonly byte[] BigExponentPkcs10Bytes = (
            "30820311308201F902010030818A310B30090603550406130255533113301106" +
            "03550408130A57617368696E67746F6E3110300E060355040713075265646D6F" +
            "6E64311E301C060355040A13154D6963726F736F667420436F72706F72617469" +
            "6F6E3120301E060355040B13172E4E4554204672616D65776F726B2028436F72" +
            "6546582931123010060355040313096C6F63616C686F737430820124300D0609" +
            "2A864886F70D010101050003820111003082010C0282010100AF81C1CBD8203F" +
            "624A539ED6608175372393A2837D4890E48A19DED36973115620968D6BE0D3DA" +
            "A38AA777BE02EE0B6B93B724E8DCC12B632B4FA80BBC925BCE624F4CA7CC6063" +
            "06B39403E28C932D24DD546FFE4EF6A37F10770B2215EA8CBB5BF427E8C4D89B" +
            "79EB338375100C5F83E55DE9B4466DDFBEEE42539AEF33EF187B7760C3B1A1B2" +
            "103C2D8144564A0C1039A09C85CF6B5974EB516FC8D6623C94AE3A5A0BB3B4C7" +
            "92957D432391566CF3E2A52AFB0C142B9E0681B8972671AF2B82DD390A39B939" +
            "CF719568687E4990A63050CA7768DCD6B378842F18FDB1F6D9FF096BAF7BEB98" +
            "DCF930D66FCFD503F58D41BFF46212E24E3AFC45EA42BD884702050200000441" +
            "A03F303D06092A864886F70D01090E3130302E302C0603551D11042530238704" +
            "7F00000187100000000000000000000000000000000182096C6F63616C686F73" +
            "74300D06092A864886F70D01010B050003820101003BCAE7E02D3A828435110C" +
            "8A65197FF1A027EC5ACA37EBE29B6E7093A4BDCA9BDA8E244DC05846AA9F186D" +
            "2EBBDF6474BB09ECF5A3C11F3A7E56D9D489C3D4AE2DCF5D52ABFCDFED6D4623" +
            "AF7C7D2E52A189BC4A0BFC5EB96EC158A96B292DF6E4ADCAE5233A7E1598444E" +
            "23F732526B71172266E45706F90EFAB0945A75D446F0A6547C788DD81AD6F4D1" +
            "E7FD0E8884083AF52003D9CD38B3A140F2E552CF3FBF0B4C771E5745C6DA6F26" +
            "DCFD0FEB87B9FDD2F4724A09DE1FB4C55E439F43C6E37A866BA19494B210D294" +
            "699B3C957C6DD22E9B63DBAE3B5AE62919F0EA3DF304C7DD9E0BBA0E7053605F" +
            "D066A788426159BB937C58E5A110461DC9364CA7CA").HexToByteArray();

        internal const string BigExponentPkcs10Pem =
            "-----BEGIN CERTIFICATE REQUEST-----\n" +
            "MIIDETCCAfkCAQAwgYoxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u\n" +
            "MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp\n" +
            "b24xIDAeBgNVBAsTFy5ORVQgRnJhbWV3b3JrIChDb3JlRlgpMRIwEAYDVQQDEwls\n" +
            "b2NhbGhvc3QwggEkMA0GCSqGSIb3DQEBAQUAA4IBEQAwggEMAoIBAQCvgcHL2CA/\n" +
            "YkpTntZggXU3I5Oig31IkOSKGd7TaXMRViCWjWvg09qjiqd3vgLuC2uTtyTo3MEr\n" +
            "YytPqAu8klvOYk9Mp8xgYwazlAPijJMtJN1Ub/5O9qN/EHcLIhXqjLtb9CfoxNib\n" +
            "eeszg3UQDF+D5V3ptEZt377uQlOa7zPvGHt3YMOxobIQPC2BRFZKDBA5oJyFz2tZ\n" +
            "dOtRb8jWYjyUrjpaC7O0x5KVfUMjkVZs8+KlKvsMFCueBoG4lyZxryuC3TkKObk5\n" +
            "z3GVaGh+SZCmMFDKd2jc1rN4hC8Y/bH22f8Ja69765jc+TDWb8/VA/WNQb/0YhLi\n" +
            "Tjr8RepCvYhHAgUCAAAEQaA/MD0GCSqGSIb3DQEJDjEwMC4wLAYDVR0RBCUwI4cE\n" +
            "fwAAAYcQAAAAAAAAAAAAAAAAAAAAAYIJbG9jYWxob3N0MA0GCSqGSIb3DQEBCwUA\n" +
            "A4IBAQA7yufgLTqChDURDIplGX/xoCfsWso36+KbbnCTpL3Km9qOJE3AWEaqnxht\n" +
            "LrvfZHS7Cez1o8EfOn5W2dSJw9SuLc9dUqv83+1tRiOvfH0uUqGJvEoL/F65bsFY\n" +
            "qWspLfbkrcrlIzp+FZhETiP3MlJrcRciZuRXBvkO+rCUWnXURvCmVHx4jdga1vTR\n" +
            "5/0OiIQIOvUgA9nNOLOhQPLlUs8/vwtMdx5XRcbabybc/Q/rh7n90vRySgneH7TF\n" +
            "XkOfQ8bjeoZroZSUshDSlGmbPJV8bdIum2Pbrjta5ikZ8Oo98wTH3Z4Lug5wU2Bf\n" +
            "0GaniEJhWbuTfFjloRBGHck2TKfK\n" +
            "-----END CERTIFICATE REQUEST-----";

        internal static byte[] EmptySubjectCertificate = (
            "308202A73082018FA003020102020103300D06092A864886F70D01010B050030" +
            "1F311D301B06035504031314456D707479205375626A65637420497373756572" +
            "301E170D3138303631353134343030365A170D3333303631353134323030365A" +
            "300030820122300D06092A864886F70D01010105000382010F003082010A0282" +
            "0101008BFA2727E0D93BEF2A992E912A829FF7A374992F2BD910291BF2BD747D" +
            "5CCCF997276ABC2F1CACEAD3F964333F5FF9D7F116A0AC392E711866CCEB0E48" +
            "80716367613E4ABD26FCB946E0A2C6AB84ABFD1EF377CD4F6C497D49D9B99CBE" +
            "DA878CA4E962307DE110345D2B22CB95A2CC3E0AB94D505CF43FA3B0343B4957" +
            "AF361E3604507732150254D77162909887509D5990499C039E5C3871326E09C2" +
            "DE132786032B4CD9F460CE35DD0650BE77B0AD9963BC498773CAC858AD15000E" +
            "A97793A906D5052A381AA2EE84BF2734833BF12DE932CF67A6D567E627898D96" +
            "2FEA2B4F55DE992C205DD67B5B2A59CDD25C04070363D48C8ED854BE013C5D1E" +
            "3E57FF0203010001A30D300B30090603551D1304023000300D06092A864886F7" +
            "0D01010B0500038201010001BF6D51DA7C45965CB2B6B889E0B1875DFEFDBAC1" +
            "558978A53E37BB796E10CBA9AFB698168F55DBD2EC4D26E9FABB7D40D55A2FCA" +
            "57F7BBC8509D1B88EB753468F3B57A42718081F00430115A48035CE72CBF9294" +
            "2837DF2C2FDC38CE213F2258C5E071A4E18ED70DFC281CD376E84ADC92922416" +
            "43389C87098AFAD976F811AA95B48B69DAF7CB31C79953BBFD1C96839561CE12" +
            "435F83CEA9F2CB9A94235B0B21FB0D591CDC41676FE927E41CC3A776FAF97146" +
            "BF14B08041CB1A90AC7339E7BD1DAF9A600479754F42C88D418B5449F5D34050" +
            "4E543013489C47297C83440C3EEF49C9A9D96E398949F03ACB0D5F72E7B4E9D3" +
            "895D82CAB526DB74AC5629").HexToByteArray();

        internal static byte[] EmptySubjectIssuerCertificate = (
            "308202D4308201BCA003020102020900F68AD256DB542CF9300D06092A864886" +
            "F70D01010B0500301F311D301B06035504031314456D707479205375626A6563" +
            "7420497373756572301E170D3137303631353134343030365A170D3333303631" +
            "353134343030365A301F311D301B06035504031314456D707479205375626A65" +
            "63742049737375657230820122300D06092A864886F70D01010105000382010F" +
            "003082010A0282010100A67F5898CCA5FC235EEB2FF14BF0BF490BB507C4D552" +
            "76E0D86CAA72BFDBDE7E7F38EC3184B18D32AEA7F5A1EEF0D2D24B7B8ED340F5" +
            "00703D3758B7E3824848CD9A4CDF15F73EBB1D4A02ED8182673138822C148463" +
            "B6126D14BE03C9A4DF62D35109BF7A8CDDFF6AE5A55C75496C13376C9B0096A0" +
            "5F398703FFBB6B69F7EA79B1F1F955F07CEBDAF3FB87D6E6F9C33678C49EC5E5" +
            "7EF10AEB14E009C83DC3DD3A2707F36D1A8723849DCD020CFBB0D38972B15F12" +
            "53209E015915A2275ADFB0164DB5A6C79BA53EA5782B001D92764D21694E5992" +
            "33B4D2C7FE260F925194C372EE473812B4F82381B4027E1F7F52E72A6ECC5BEE" +
            "656FE161E7681A06A9BF0203010001A3133011300F0603551D130101FF040530" +
            "030101FF300D06092A864886F70D01010B050003820101008D239025B8266EFF" +
            "AEB1B76EA159ED7BF6A6F2E5A23266DF07767DEDF26B43983EC8717AC2E6B99E" +
            "A21625D787D39BF94110D5A9B37CF0E6FB3062090D280B889A0E8DA5952E46D5" +
            "107F9BE829119282ADD53414CE0BA47C1E04D82BCC50118BB9CB3905E22FADD5" +
            "1905F23373D7F5E9E32C5C7D27B68CD15D1E079153B66B3DBE148D4DF2BF8EA1" +
            "8E68D838BE1C86E68E4CAF09A90F7428A4A50922BCB01010A1B585227930CEE0" +
            "027D561804C2F023AEBAA63F312F49A37B74BED2DB8B6D8AA1A6E60320589EB5" +
            "B14EB93CC47DAF7F07D67C25FD275493C0AF0CC23E8AED47963DCCE7C2ADFD56" +
            "5B8A77118C2978E15D1511572C4496CD1180094D43AE50FD").HexToByteArray();

        internal static byte[] T61StringCertificate = (
            "2d2d2d2d2d424547494e2043455254494649434154452d2d2d2d2d0a4d494943" +
            "4544434341626f4343514348477057667a4678366e6a414e42676b71686b6947" +
            "3977304241517346414443426a6a45614d42674741315545417851520a644756" +
            "7a6443317a5a584a325a5849756247396a59577778437a414a42674e56424159" +
            "54416c56544d52597746415944565151494641314e59584e7a59574e6f0a6458" +
            "4e6c6448527a4d524177446759445651514b4641645959573168636d6c754d52" +
            "5177456759445651514c46417446626d6470626d566c636d6c755a7a456a0a4d" +
            "43454743537147534962334451454a4152595562574669595856735147317059" +
            "334a766332396d6443356a623230774868634e4d5467774e7a49314d5467310a" +
            "4e5441785768634e4d6a67774e7a49794d5467314e544178576a43426a6a4561" +
            "4d42674741315545417851526447567a6443317a5a584a325a5849756247396a" +
            "0a59577778437a414a42674e5642415954416c56544d52597746415944565151" +
            "494641314e59584e7a59574e6f64584e6c6448527a4d52417744675944565151" +
            "4b0a4641645959573168636d6c754d525177456759445651514c46417446626d" +
            "6470626d566c636d6c755a7a456a4d43454743537147534962334451454a4152" +
            "59550a62574669595856735147317059334a766332396d6443356a6232307758" +
            "44414e42676b71686b6947397730424151454641414e4c41444249416b454131" +
            "366a590a51424d417459646372394d3534426677684b695852675855566f7031" +
            "4f5a4e735a77714d714872543557674d50386b3565344a63444175417455434f" +
            "6d7761670a52697647463735684a5449375963764b69514944415141424d4130" +
            "4743537147534962334451454243775541413045415377736f42336436506c72" +
            "65433864710a4f38507370457171576a4e69464358724b317169547865584336" +
            "536844674d445733464670316c6d33564747564637365078504a792f4c6c6249" +
            "39366d5a59590a734f65735a513d3d0a2d2d2d2d2d454e442043455254494649" +
            "434154452d2d2d2d2d0a").HexToByteArray();

        // The SubjectPublicKeyInfo value from the test certificate in
        // https://tools.ietf.org/html/rfc4491#section-4.2
        internal static byte[] GostR3410SubjectPublicKeyInfo = (
            "3063301C06062A8503020213301206072A85030202240006072A850302021E01" +
            "03430004408495687560021A407508CD138C31892CFDE505037A435CF46D2B0F" +
            "E74F327E578FEBCC16B9958803D09A7C85AE0FE48DEAA6BB7E56C7CBB0DF0F66" +
            "BCCAEA1A60").HexToByteArray();

        internal static byte[] GostCertificate = (
            "308204f5308204a4a00302010202103b0f900958220fa74f1fd37f3ddcd2" +
            "65300806062a85030202033081f73118301606052a85036401120d313034" +
            "37373037303330353133311a301806082a85030381030101120c30303737" +
            "3037333239313532312f302d06035504091e260443043b002e0020041d04" +
            "350433043b0438043d043d0430044f00200434002e002000320033311b30" +
            "1906035504081e120433002e0020041c043e0441043a043204303120301e" +
            "06092a864886f70d01090116116d6e733132373035406e616c6f672e7275" +
            "310b3009060355040613025255310f300d060355040713064d6f73636f77" +
            "311c301a060355040a13134665646572616c205461782053657276696365" +
            "311330110603550403130a464e5320527573736961301e170d3132313030" +
            "323130323032315a170d3432313030333130323032315a3081f731183016" +
            "06052a85036401120d31303437373037303330353133311a301806082a85" +
            "030381030101120c303037373037333239313532312f302d06035504091e" +
            "260443043b002e0020041d04350433043b0438043d043d0430044f002004" +
            "34002e002000320033311b301906035504081e120433002e0020041c043e" +
            "0441043a043204303120301e06092a864886f70d01090116116d6e733132" +
            "373035406e616c6f672e7275310b3009060355040613025255310f300d06" +
            "0355040713064d6f73636f77311c301a060355040a13134665646572616c" +
            "205461782053657276696365311330110603550403130a464e5320527573" +
            "7369613063301c06062a8503020213301206072a85030202230106072a85" +
            "0302021e010343000440286aa49187f798c61d83382a0deae4dc3c644e88" +
            "e8bf04098644d2710c263ef310c3de5492fc5fa54264b0a76b0f9060b65a" +
            "df9125e101b0154ea0f4bed550dea382020630820202303606052a850364" +
            "6f042d0c2b22d09ad180d0b8d0bfd182d0bed09fd180d0be204353502220" +
            "28d0b2d0b5d180d181d0b8d18f20332e36293082013106052a8503647004" +
            "820126308201220c2b22d09ad180d0b8d0bfd182d0bed09fd180d0be2043" +
            "5350222028d0b2d0b5d180d181d0b8d18f20332e36290c5322d0a3d0b4d0" +
            "bed181d182d0bed0b2d0b5d180d18fd18ed189d0b8d0b920d186d0b5d0bd" +
            "d182d1802022d09ad180d0b8d0bfd182d0bed09fd180d0be20d0a3d0a622" +
            "20d0b2d0b5d180d181d0b8d0b820312e350c4e43d0b5d180d182d0b8d184" +
            "d0b8d0bad0b0d18220d181d0bed0bed182d0b2d0b5d182d181d182d0b2d0" +
            "b8d18f20e2849620d0a1d0a42f3132312d3138353920d0bed1822031372e" +
            "30362e323031320c4e43d0b5d180d182d0b8d184d0b8d0bad0b0d18220d1" +
            "81d0bed0bed182d0b2d0b5d182d181d182d0b2d0b8d18f20e2849620d0a1" +
            "d0a42f3132382d3138323220d0bed1822030312e30362e32303132300b06" +
            "03551d0f040403020186300f0603551d130101ff040530030101ff301d06" +
            "03551d0e04160414e0aca23183615a27ac05b888102fd46009b6fae43012" +
            "06092b060104018237150104050203020002302306092b06010401823715" +
            "020416041488a9bed7a922afee5c420aff55b347b034370d8d301d060355" +
            "1d2004163014300806062a8503647101300806062a850364710230080606" +
            "2a85030202030341003c272993f0a9a4108d721f0280e7d48bfca7ebd16e" +
            "028bedeefab9a2be80466fe278fdc50db1b9530e1796b23271b4df2cddd9" +
            "4769c8a21a8f66c6d4bc181713").HexToByteArray();

        internal static byte[] CertificateWithUtf8 = (
            "30820315308201fda003020102020900e48b784d93645921300d06092a864886" +
            "f70d01010b0500301b3119301706035504030c10d0bad0b0d180d182d0bed188" +
            "d0bad0b0301e170d3231303931343230353635365a170d333230383237323035" +
            "3635365a301b3119301706035504030c10d0bad0b0d180d182d0bed188d0bad0" +
            "b030820122300d06092a864886f70d01010105000382010f003082010a028201" +
            "0100c394711b3839a612aaadafde855ccc3479bc1dbe253a29e508cc46464efb" +
            "ab09b16b85fb67e4be0ab8b09fa6ff73cb3ec5d8f8b7d2869a79c414d025a43e" +
            "4c158c9711638dbfb070ef92b3633037d5e633a870ed024ab9017c97e26f02de" +
            "2273dd92c0837a95ff12229981cd661b140afd841671d3397fd44aea05878dd0" +
            "9362bd2de8da17225dd3caf0181aadf2c9e13faefbbce3ce80ef355dcc15c1b9" +
            "c6e86bb1cddf32b3113e7a72ad3799cc67f64ea34c0518c22727972802084801" +
            "e6565b2323e87fd20019a9db151e87b2b1db004583e15dd64c5393439a0ba5ed" +
            "3b9fd262c8d8bf0263dcba092ca5591d4d34f990daa9ee41811cec6229f2121b" +
            "582f0203010001a35c305a301b0603551d11041430128210786e2d2d38306161" +
            "326162746c6f3363301d0603551d250416301406082b0601050507030106082b" +
            "06010505070302300f0603551d130101ff040530030101ff300b0603551d0f04" +
            "040302028c300d06092a864886f70d01010b050003820101001c0512d3d99267" +
            "4ca3a1fdf942052064ffbda824aaeff07d9869eefa86e592102afca0dba4e267" +
            "e54753552fc23da97b18f739ea437794aac5bb2ae7413ae54ce921b4b96fe420" +
            "f3fd86f922adcab202589823c4095fc6e7904e61be1716faec7e09ce2d1bf431" +
            "fa9fc2d7284689d2c940f7d027e16b9c3305f9cd6d5dc6bfee9089d85d342f5b" +
            "d07920c51899288674a1732708a7e3a39fb11c152417d50a3bb9b09975852baa" +
            "39767c289baea9330902ea38388932964c410cd22049415712223fb5710a21ef" +
            "153ac22391a1603547ffb3f9b328c59e59b5d64568b9c451df233fe9a581116f" +
            "05be8c67b92bc7df9984f30535ad53817cb4abcd77cb521856").HexToByteArray();

        internal static byte[] RsaSha3_256SignedCertificate =(
            "30820303308201eba00302010202147bcedeaa2286e1308106525ad7efbe" +
            "fb88e96995300d060960864801650304030e05003011310f300d06035504" +
            "030c06706f7461746f301e170d3233303531383138303135305a170d3233" +
            "303631373138303135305a3011310f300d06035504030c06706f7461746f" +
            "30820122300d06092a864886f70d01010105000382010f003082010a0282" +
            "010100ef7fd4e037107937b690472066d58048580cfa3822dfb91b776e76" +
            "dc1de93b7f72d613d836c423909f139f909f7e5b77c610c114f2accee09f" +
            "ee9f431524480f31e246c42e31807cdd3cc831f3ef6b601df4f81ecfede1" +
            "96f90f9ea25749a9cb7af32bdc4857a9f272a87acd5ca73b288ed8817f71" +
            "6c1297166c76b097be22f4705f0678ce3057b05580418649b381a7bcb44f" +
            "9cfd429cc788487d3632de477c1ac6d430d8beeeee258e094388b8b3a9c2" +
            "7a4b48f2d049beb56de950585f3c6c6308321e53cb37894c1b2a0b55a731" +
            "e7861f6342f3f451ea0977dd4ef65f95b3218702722df9aa3672888ee040" +
            "5eb4f4969e416150fa5460637f0071d7e444190203010001a3533051301d" +
            "0603551d0e04160414ab91d8c96f342fb48b7903361835b1a4e255534d30" +
            "1f0603551d23041830168014ab91d8c96f342fb48b7903361835b1a4e255" +
            "534d300f0603551d130101ff040530030101ff300d060960864801650304" +
            "030e05000382010100bf3f9103c076fc81d3d0d7b770280d2cbe56eb568b" +
            "24a4ea526eaec44408f3d6581968c45f20dc62c24da618b8453dd6874314" +
            "82de6cb7e66826edc8daf10cb5273ae1fda4b290b6b0fa46431fdb92a03a" +
            "71a0174d1f780f3de4b2d2a8067e230e77e8e8b8bc88a6e43f1fdfcd2280" +
            "16de21af0a2e46f3401c5a598adc1532041b2b211087358fb15e71becd53" +
            "43c6fe33b374cef8030556a42a97fdf7a1387cbbeaa4ee74a5525321355c" +
            "d06582c8be0bc642e98e36fc94019b83e1fa3827db6b914e3cefb1127c89" +
            "4160a40665fc6663e70f42c25de76dcdd8c6ad96deb53560e870ab826a86" +
            "ad82f8f331c63b3b5f256af0b702cf2413f6a1dc37a57d1d69").HexToByteArray();

        internal static byte[] ConcatenatedPemFile = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIFcDCCBFigAwIBAgIQB6krbZc11OZ5l2/FnU3CpTANBgkqhkiG9w0BAQsFADBG
MQswCQYDVQQGEwJVUzEPMA0GA1UEChMGQW1hem9uMRUwEwYDVQQLEwxTZXJ2ZXIg
Q0EgMUIxDzANBgNVBAMTBkFtYXpvbjAeFw0xODA2MTMwMDAwMDBaFw0xOTA2MTMx
MjAwMDBaMB4xHDAaBgNVBAMTE2VjaG8tYXBpLmFtYXpvbi5jb20wggEiMA0GCSqG
SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCFTKdiYc8w7wt59nCfHzpT9xy8mDk8spkC
ECPzIC1Qim9T7dSRsT7tcUJIOHMPXxrlxyySSK1qB7LLdoDFuhW/CtUdD4c6t73y
ryNHQzhPZ7fQvb8jMWf5VWPTNsS1FBAKZdTe6n0pjIAS4nypxfF+eXMaQrHiH4Ib
iV+aZP7Men40j/YucEeii8ukmfmlQ8L351BUZmCD1FZlXD+fLb5YgbZjC+c6TB0K
WI2oe3qK0zFKGigaFvNBoZl1A+v0V7AFWZ+tYKfCvyVBuwase5pK4770GKNfqXaX
a/q1p5N1M3D6qa6j/U01IOtn9gJqB+PvVKBVZ/TcfBJVHtdDj+aPAgMBAAGjggKA
MIICfDAfBgNVHSMEGDAWgBRZpGYGUqB7lZI8o5QHJ5Z0W/k90DAdBgNVHQ4EFgQU
PmgdPNeivSJ4aVavb6e7hsUZfg0wHgYDVR0RBBcwFYITZWNoby1hcGkuYW1hem9u
LmNvbTAOBgNVHQ8BAf8EBAMCBaAwHQYDVR0lBBYwFAYIKwYBBQUHAwEGCCsGAQUF
BwMCMDsGA1UdHwQ0MDIwMKAuoCyGKmh0dHA6Ly9jcmwuc2NhMWIuYW1hem9udHJ1
c3QuY29tL3NjYTFiLmNybDAgBgNVHSAEGTAXMAsGCWCGSAGG/WwBAjAIBgZngQwB
AgEwdQYIKwYBBQUHAQEEaTBnMC0GCCsGAQUFBzABhiFodHRwOi8vb2NzcC5zY2Ex
Yi5hbWF6b250cnVzdC5jb20wNgYIKwYBBQUHMAKGKmh0dHA6Ly9jcnQuc2NhMWIu
YW1hem9udHJ1c3QuY29tL3NjYTFiLmNydDAMBgNVHRMBAf8EAjAAMIIBBQYKKwYB
BAHWeQIEAgSB9gSB8wDxAHcAu9nfvB+KcbWTlCOXqpJ7RzhXlQqrUugakJZkNo4e
0YUAAAFj+wW/FAAABAMASDBGAiEAo5ajttPYYb/u06ZYvQ1A+wXljlscciiJQO2J
q+aZmwQCIQDTytC4r5crkEOvnIu/SVEQF83XnnXoqa1Hc8GZubzjSgB2AId1v+dZ
fPiMQ5lfvfNu/1aNR1Y2/0q1YMG06v9eoIMPAAABY/sFv6UAAAQDAEcwRQIgc1ob
hd3vnx2CPLjFqNy/98zvyfh6LkZhSRJgp/kOv1oCIQDPB9E24+ifg9btt7F4ae0e
v7x2QubFqHTV0mbbPIdRmzANBgkqhkiG9w0BAQsFAAOCAQEAnxhNKdhvKOmcY1xZ
f0C2BBfsezzIm1MlxSThk8UXhMgNdnFAjhb6PUneR7ea/ls/KuyLhVvE5A1i/z5Y
P3jiwq8qfCa/WQeRpZ4wxCqOWwK0hWR1iDZeL7z6+YSmOkrJru2TpOMf9DaExaVs
jVgzC6N0FAOgosicCUojJGZKHDwgh/2UoXdLSKuvXJcOGijZ+v1/7BjBwlaYecEN
0Gx1UkhBoJLFjUgAuEdiGLN8SyZce2geddK+ekTfAGmAvXe1ILFQid/CIoqVUO8b
EpQT6CLezn1LfucP7FsHjUqUeknlsMi3KSvCPccM8VWwoM7PM2krWqSYveDC4dVx
4xynhA==
-----END CERTIFICATE-----
-----BEGIN CERTIFICATE-----
MIIESTCCAzGgAwIBAgITBn+UV4WH6Kx33rJTMlu8mYtWDTANBgkqhkiG9w0BAQsF
ADA5MQswCQYDVQQGEwJVUzEPMA0GA1UEChMGQW1hem9uMRkwFwYDVQQDExBBbWF6
b24gUm9vdCBDQSAxMB4XDTE1MTAyMjAwMDAwMFoXDTI1MTAxOTAwMDAwMFowRjEL
MAkGA1UEBhMCVVMxDzANBgNVBAoTBkFtYXpvbjEVMBMGA1UECxMMU2VydmVyIENB
IDFCMQ8wDQYDVQQDEwZBbWF6b24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
AoIBAQDCThZn3c68asg3Wuw6MLAd5tES6BIoSMzoKcG5blPVo+sDORrMd4f2AbnZ
cMzPa43j4wNxhplty6aUKk4T1qe9BOwKFjwK6zmxxLVYo7bHViXsPlJ6qOMpFge5
blDP+18x+B26A0piiQOuPkfyDyeR4xQghfj66Yo19V+emU3nazfvpFA+ROz6WoVm
B5x+F2pV8xeKNR7u6azDdU5YVX1TawprmxRC1+WsAYmz6qP+z8ArDITC2FMVy2fw
0IjKOtEXc/VfmtTFch5+AfGYMGMqqvJ6LcXiAhqG5TI+Dr0RtM88k+8XUBCeQ8IG
KuANaL7TiItKZYxK1MMuTJtV9IblAgMBAAGjggE7MIIBNzASBgNVHRMBAf8ECDAG
AQH/AgEAMA4GA1UdDwEB/wQEAwIBhjAdBgNVHQ4EFgQUWaRmBlKge5WSPKOUByeW
dFv5PdAwHwYDVR0jBBgwFoAUhBjMhTTsvAyUlC4IWZzHshBOCggwewYIKwYBBQUH
AQEEbzBtMC8GCCsGAQUFBzABhiNodHRwOi8vb2NzcC5yb290Y2ExLmFtYXpvbnRy
dXN0LmNvbTA6BggrBgEFBQcwAoYuaHR0cDovL2NydC5yb290Y2ExLmFtYXpvbnRy
dXN0LmNvbS9yb290Y2ExLmNlcjA/BgNVHR8EODA2MDSgMqAwhi5odHRwOi8vY3Js
LnJvb3RjYTEuYW1hem9udHJ1c3QuY29tL3Jvb3RjYTEuY3JsMBMGA1UdIAQMMAow
CAYGZ4EMAQIBMA0GCSqGSIb3DQEBCwUAA4IBAQCFkr41u3nPo4FCHOTjY3NTOVI1
59Gt/a6ZiqyJEi+752+a1U5y6iAwYfmXss2lJwJFqMp2PphKg5625kXg8kP2CN5t
6G7bMQcT8C8xDZNtYTd7WPD8UZiRKAJPBXa30/AbwuZe0GaFEQ8ugcYQgSn+IGBI
8/LwhBNTZTUVEWuCUUBVV18YtbAiPq3yXqMB48Oz+ctBWuZSkbvkNodPLamkB2g1
upRyzQ7qDn1X8nn8N8V7YJ6y68AtkHcNSRAnpTitxBKjtKPISLMVCx7i4hncxHZS
yLyKQXhw2W2Xs0qLeC1etA+jTGDK4UfLeC0SF7FSi8o5LL21L8IzApar2pR/
-----END CERTIFICATE-----"
        );

        internal static DSAParameters GetDSA1024Params()
        {
            DSAParameters p = new DSAParameters();
            p.G = (
                "6BC366B66355545E098F1FE90E5469B567E09FA79D817F2B367B45DECD4301A59C81D6911F7691D370E15AC692C04BC11872" +
                "C171A7FE654E963D7DDA575A9E98CE026FB7D3934A258608134A8EC5ED69A2AEDC89401B67ADDE427F17EDAEB72D7AF45D9A" +
                "B1D59E1B13D4EFBD17C764330267DDE352C20E05B80DB3C109FE8B9C").HexToByteArray();
            p.P = (
                "C16D26C74D6C1627799C0918548E553FE58C7881DA484629CAF64311F4B27CFEF6BDB0F21206B0FFC4999A2FED53B43B9EE2" +
                "910C68DA2C436A8018F4938F6472369F5647D005BCC96E22590CC15E3CD4EA0D132F5DA5AF6AAA0807B0CC4EF3404AF542F4" +
                "546B37BDD6A47E641130837DB99397C845635D7DC36D0537E4A84B31").HexToByteArray();
            p.Q = "D83C0ECB73551E2FE30D51FCF4236C651883ADD7".HexToByteArray();
            p.X = "C02678007779E52E360682214BD47F8FAF42BC2D".HexToByteArray();
            p.Y = (
                "690BB37A9145E05D6E7B47C457898AAEDD72501C9D16E79B1AD75A872CF017AA90BBFB90F1B3B7F5C03C87E46E8725665526" +
                "FD34157B26F596A1F0997F59F3E65EFC615A552D5E7569C5FFC4593D5A0299110E71C97E1221A5A03FE9A6935AEDD88EF0B3" +
                "B2F79D3A99ED75F7B871E6EAF2680D96D574A5F4C13BACE3B4B44DE1").HexToByteArray();
            return p;
        }

        public const string RsaCertificate = @"
-----BEGIN CERTIFICATE-----
MIIDNjCCAh4CCQCu2+oEr9yAxDANBgkqhkiG9w0BAQsFADBdMQswCQYDVQQGEwJV
UzERMA8GA1UECAwIVmlyZ2luaWExEzARBgNVBAcMCkFsZXhhbmRyaWExEDAOBgNV
BAoMB0NvbnRvc28xFDASBgNVBAsMC0VuZ2luZWVyaW5nMB4XDTIwMDYwODE3MDk1
M1oXDTIwMDcwODE3MDk1M1owXTELMAkGA1UEBhMCVVMxETAPBgNVBAgMCFZpcmdp
bmlhMRMwEQYDVQQHDApBbGV4YW5kcmlhMRAwDgYDVQQKDAdDb250b3NvMRQwEgYD
VQQLDAtFbmdpbmVlcmluZzCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB
AL/yBbH8MoxLuApvUhWMMV+5h+v15xY9or+TxyAkwwtP0mLKMWPeHZm2S4OT8JCn
fJmz7J+7J4Nf3VB0HYDN+41uEEwNX3Caps30Kt17dcJm3/sG9uBB7eiQdbj2HNSS
nzxYK7d2mKnWwZlHg0I7/ZR5TMM8eCdJpESsHRpF5dPGgdWFgUzbdi73Yyk5/PFM
lauHPM0d4TWHoWnY0yh7Y08PMY/4MH7HoMVt+mbpV2d3DRC8WS0jhU+mbCqce4St
BWTCmR+ObuHFdMSVx7o88MWmdteZXCX8N6ohPwAl9W02pOa5Vq37xwuUZ1VvT9G2
ndi+VRRgN51HCmG1qB1LhtsCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAIBtwgcLn
7LT+gfqqDsaZcMjpsHMu2TamR9inoRdxwoJKnG7dIe8tWL+nGxsN38DjjcZhb3y0
Ilqi0e7LYqb0QyXAuS+Su2uJjypgvNd4stj+4Pl1EfU1rpzed4CA1O0pap5m0U++
2YmWNNBmSZxcUi2ge09ZqrKm78a7Vtrpy8bNcixb+szrPSUFWh7WOXBCABusZ/OY
MnzZBXtQtQzDCtJb6IxevxAGod1XxInXQaB+nDnG4MD3v4MZQYgTI76+cAPxS6nx
pzJ/tfFzq7OAGBkrxcdmzqb1/caPWINKzwbDAEuNX2yeThP8eVrbRvGciy7LNzvR
EA0/67lEfPmRow==
-----END CERTIFICATE-----";

        public const string ECDsaCertificate = @"
-----BEGIN CERTIFICATE-----
MIICDzCCAbWgAwIBAgIUeohSwWS4OT31fY5xvUxTEv07e8IwCgYIKoZIzj0EAwIw
XTELMAkGA1UEBhMCVVMxETAPBgNVBAgMCFZpcmdpbmlhMRMwEQYDVQQHDApBbGV4
YW5kcmlhMRAwDgYDVQQKDAdDb250b3NvMRQwEgYDVQQLDAtEZXZlbG9wbWVudDAe
Fw0yMDA2MDgxOTE4NDVaFw0yMTA2MDgxOTE4NDVaMF0xCzAJBgNVBAYTAlVTMREw
DwYDVQQIDAhWaXJnaW5pYTETMBEGA1UEBwwKQWxleGFuZHJpYTEQMA4GA1UECgwH
Q29udG9zbzEUMBIGA1UECwwLRGV2ZWxvcG1lbnQwWTATBgcqhkjOPQIBBggqhkjO
PQMBBwNCAAQyB0wIKKfk5lmK4Z907qQnPsRRXh3TrU/VPMCTHxuBwoZqFBSE7gGm
JWLTnGwZ0MGMACP+N1HK4dU1S9VNoNw9o1MwUTAdBgNVHQ4EFgQUPzDLKQI9EfTa
rnMMO0/4p8rNwZgwHwYDVR0jBBgwFoAUPzDLKQI9EfTarnMMO0/4p8rNwZgwDwYD
VR0TAQH/BAUwAwEB/zAKBggqhkjOPQQDAgNIADBFAiAwq18UD2DCufaMGTsz4JOQ
1vRqLI4hMLsIUQNyYgzbcgIhAJsB3qgv5WGyshlav98MPORcdCmYfkIvUCal0oPX
Wesb
-----END CERTIFICATE-----";

        public const string EcDhCertificate = @"
-----BEGIN CERTIFICATE-----
MIIBPDCB46ADAgECAhQmWTEpE73OfJ5Pate8bdxH6ETJbDAKBggqhkjOPQQDAjAP
MQ0wCwYDVQQDDARyb290MB4XDTIwMDkxMDE0MjAzN1oXDTMwMDgyOTE0MjAzN1ow
DzENMAsGA1UEAwwEbGVhZjBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABEM385x6
t0YBUBjeDiP5SoAurpat6JhYz+SC6BPNK99AoayIbywdYTXH3sSewe6JLR+Pludc
r/hCDS7+0mlinnijHTAbMAsGA1UdDwQEAwIDKDAMBgNVHRMBAf8EAjAAMAoGCCqG
SM49BAMCA0gAMEUCIGwwGY1yS7BQEoL81IqjgzYv/m2imeBwHoOlq4uw3NKSAiEA
hjikjlM5O6v+IjbDMCEdduXflbwf4qEbG02TUPQmiow=
-----END CERTIFICATE-----";

        public const string EcDhPkcs8Key = @"
-----BEGIN PRIVATE KEY-----
MIGHAgEAMBMGByqGSM49AgEGCCqGSM49AwEHBG0wawIBAQQgIgCUm3/m+ftoRITe
YV98jl/5auSR4u+J8K0v3fa9HbehRANCAARDN/OcerdGAVAY3g4j+UqALq6WreiY
WM/kgugTzSvfQKGsiG8sHWE1x97EnsHuiS0fj5bnXK/4Qg0u/tJpYp54
-----END PRIVATE KEY-----";

        public const string EcDhPkcs8PublicKey = @"
-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEQzfznHq3RgFQGN4OI/lKgC6ulq3o
mFjP5ILoE80r30ChrIhvLB1hNcfexJ7B7oktH4+W51yv+EINLv7SaWKeeA==
-----END PUBLIC KEY-----";

        public const string ECDsaPkcs8Key = @"
-----BEGIN PRIVATE KEY-----
MIGHAgEAMBMGByqGSM49AgEGCCqGSM49AwEHBG0wawIBAQQgIg/wHlL4nNyuLFbK
00Ga7N5LbP/2YCgEXqBiB3el8syhRANCAAQyB0wIKKfk5lmK4Z907qQnPsRRXh3T
rU/VPMCTHxuBwoZqFBSE7gGmJWLTnGwZ0MGMACP+N1HK4dU1S9VNoNw9
-----END PRIVATE KEY-----";

        public const string ECDsaPkcs8PublicKey = @"
-----BEGIN PUBLIC KEY-----
MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEMgdMCCin5OZZiuGfdO6kJz7EUV4d
061P1TzAkx8bgcKGahQUhO4BpiVi05xsGdDBjAAj/jdRyuHVNUvVTaDcPQ==
-----END PUBLIC KEY-----";

        public const string ECDsaEncryptedPkcs8Key = @"
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIHgMEsGCSqGSIb3DQEFDTA+MCkGCSqGSIb3DQEFDDAcBAh5RmzN7/AXZgICCAAw
DAYIKoZIhvcNAgkFADARBgUrDgMCBwQIQHgahqqSQKcEgZCvEBMgW8a7IXmT+weI
0mlM4AcTELDkE+SEKpYC5qVF4ZDyrw4OmnVLkSPiu0GUwgJIopazWOQfetMdgC5Q
n5pYHoRm9bek0s6TK9eoaTIA+M2T0MMNM0fWXWcYaT5B2/4Uv+mMEYgIRncFe1c1
FEixDW6ObZIXVBbxl+zK1KwtCpdewXE4HRX/qpBrPhB8z2s=
-----END ENCRYPTED PRIVATE KEY-----";

        public const string ECDsaECPrivateKey = @"
-----BEGIN EC PRIVATE KEY-----
MHcCAQEEICIP8B5S+JzcrixWytNBmuzeS2z/9mAoBF6gYgd3pfLMoAoGCCqGSM49
AwEHoUQDQgAEMgdMCCin5OZZiuGfdO6kJz7EUV4d061P1TzAkx8bgcKGahQUhO4B
piVi05xsGdDBjAAj/jdRyuHVNUvVTaDcPQ==
-----END EC PRIVATE KEY-----";

        public const string RsaPkcs1Key = @"
-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAv/IFsfwyjEu4Cm9SFYwxX7mH6/XnFj2iv5PHICTDC0/SYsox
Y94dmbZLg5PwkKd8mbPsn7sng1/dUHQdgM37jW4QTA1fcJqmzfQq3Xt1wmbf+wb2
4EHt6JB1uPYc1JKfPFgrt3aYqdbBmUeDQjv9lHlMwzx4J0mkRKwdGkXl08aB1YWB
TNt2LvdjKTn88UyVq4c8zR3hNYehadjTKHtjTw8xj/gwfsegxW36ZulXZ3cNELxZ
LSOFT6ZsKpx7hK0FZMKZH45u4cV0xJXHujzwxaZ215lcJfw3qiE/ACX1bTak5rlW
rfvHC5RnVW9P0bad2L5VFGA3nUcKYbWoHUuG2wIDAQABAoIBAEavE5XVr6+meqGt
GOdCdzQvGHS2W2D/VZ2DCAM4RnM1893ZY5LJStE+JlTP9/jtFJ9teKfhvc1NUiy8
ddjnAcm1TF8VVZ4b9W1GizqAqn7qb3T7vZIb9UZ9XDy+tSM601Tfi0nGbLWuliCi
Cx4rBVjVyoTjEcQ2BD4du6HfN6FihzCu+DV6Dsm4Ar6orWRCYJ0v1doMYdGzOaoJ
SwcWCtAYkJ+krfYz37xlrFJfppMgYy67q040XBV7PuyjkGtt1GcNO98pw0puiCAs
VP+u+vnngNCjSqXlVSAZL1ISJylUxKkNRynABNBTfdGeRWCuhwc/M0vuEeX8A5Ce
NlDA1MECgYEA5iGjpQkyeXcsSP6zqE4LhA0L20YoCsnYnyout/Hn4C4AtUH3Yt2o
MvpRne7JxmGxmhSJkud+z6LfbaAKLU/s86wslOeclwlMkFz+CK75AFCDNecNgFCR
jENNBFIou329+/Rx7+fbGsgHfMwz2Cdv98hKCOmX3W3pZRCW+gVLMtECgYEA1YWG
q1v4M2QdB3f8hKiElicvlyNwVW7rHgbcsUiDPdGSLg/fREZeA3C4bnV+Mi8LlRDJ
+PoZyqML+xWhQDGWK+7r0iWuvFPsbCwk0BRTCYue0p7pCsUEjS0vVQPk0O1fLlOH
UdLtyyuYegsAtn5XSMN7+LeM0zyP+OWVfqWkEesCgYBSiYQYv+izedOPRpKG7Z7h
uJAlD89ytxwTUdy5qnBAjh9A4yzn75nQ1siI/Uiu9wDswyroXlC0BbVeqwSbZcwV
RQ4kRcF6xiIIsOGHmcHCpB27KmhEOiFJjiXEQ/dJ73pBMFXg9mY1/8H3t4FsqBBX
bSVoduc5yp7n2YBcoaNr0QKBgCPloPBqM94f9KluyKtc0X2U9PFJ6fbTAQA5Ux0S
/c2E0DiiPnzx/5hAeSFI64BwXFghTHNpSLDCnJ8H0eZC7+ZO8qKP50KOMW82NLIu
2I8ARCFQygkfelZpxE1crDlbzuaw8E0XUxcXKzlJZENKFk6LXuo/oZNZ2TKVFn8G
RgElAoGAMsD758wEk2bVTJFNISizHLHQ4yyCYgJSjmUqtP/GqK0qvo4Kz5TSm/2D
H7b+ao0a48b+i4EyuDTVT/CeJ3RoQzXvayc9WuipV/ewRBYRsEVzZwFO6D4aD+jI
a+7OpVqFLMtAySKTg0vPwcRF4OxQ/4YBE1e+ZGnx+vVTpkE3gjw=
-----END RSA PRIVATE KEY-----";

        public const string RsaPkcs1PublicKey = @"
        -----BEGIN RSA PUBLIC KEY-----
MIIBCgKCAQEAv/IFsfwyjEu4Cm9SFYwxX7mH6/XnFj2iv5PHICTDC0/SYsoxY94d
mbZLg5PwkKd8mbPsn7sng1/dUHQdgM37jW4QTA1fcJqmzfQq3Xt1wmbf+wb24EHt
6JB1uPYc1JKfPFgrt3aYqdbBmUeDQjv9lHlMwzx4J0mkRKwdGkXl08aB1YWBTNt2
LvdjKTn88UyVq4c8zR3hNYehadjTKHtjTw8xj/gwfsegxW36ZulXZ3cNELxZLSOF
T6ZsKpx7hK0FZMKZH45u4cV0xJXHujzwxaZ215lcJfw3qiE/ACX1bTak5rlWrfvH
C5RnVW9P0bad2L5VFGA3nUcKYbWoHUuG2wIDAQAB
-----END RSA PUBLIC KEY-----";

        public const string RsaPkcs8PublicKey = @"
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAv/IFsfwyjEu4Cm9SFYwx
X7mH6/XnFj2iv5PHICTDC0/SYsoxY94dmbZLg5PwkKd8mbPsn7sng1/dUHQdgM37
jW4QTA1fcJqmzfQq3Xt1wmbf+wb24EHt6JB1uPYc1JKfPFgrt3aYqdbBmUeDQjv9
lHlMwzx4J0mkRKwdGkXl08aB1YWBTNt2LvdjKTn88UyVq4c8zR3hNYehadjTKHtj
Tw8xj/gwfsegxW36ZulXZ3cNELxZLSOFT6ZsKpx7hK0FZMKZH45u4cV0xJXHujzw
xaZ215lcJfw3qiE/ACX1bTak5rlWrfvHC5RnVW9P0bad2L5VFGA3nUcKYbWoHUuG
2wIDAQAB
-----END PUBLIC KEY-----";

        public const string RsaPkcs8Key = @"
-----BEGIN PRIVATE KEY-----
MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQC/8gWx/DKMS7gK
b1IVjDFfuYfr9ecWPaK/k8cgJMMLT9JiyjFj3h2ZtkuDk/CQp3yZs+yfuyeDX91Q
dB2AzfuNbhBMDV9wmqbN9Crde3XCZt/7BvbgQe3okHW49hzUkp88WCu3dpip1sGZ
R4NCO/2UeUzDPHgnSaRErB0aReXTxoHVhYFM23Yu92MpOfzxTJWrhzzNHeE1h6Fp
2NMoe2NPDzGP+DB+x6DFbfpm6Vdndw0QvFktI4VPpmwqnHuErQVkwpkfjm7hxXTE
lce6PPDFpnbXmVwl/DeqIT8AJfVtNqTmuVat+8cLlGdVb0/Rtp3YvlUUYDedRwph
tagdS4bbAgMBAAECggEARq8TldWvr6Z6oa0Y50J3NC8YdLZbYP9VnYMIAzhGczXz
3dljkslK0T4mVM/3+O0Un214p+G9zU1SLLx12OcBybVMXxVVnhv1bUaLOoCqfupv
dPu9khv1Rn1cPL61IzrTVN+LScZsta6WIKILHisFWNXKhOMRxDYEPh27od83oWKH
MK74NXoOybgCvqitZEJgnS/V2gxh0bM5qglLBxYK0BiQn6St9jPfvGWsUl+mkyBj
LrurTjRcFXs+7KOQa23UZw073ynDSm6IICxU/676+eeA0KNKpeVVIBkvUhInKVTE
qQ1HKcAE0FN90Z5FYK6HBz8zS+4R5fwDkJ42UMDUwQKBgQDmIaOlCTJ5dyxI/rOo
TguEDQvbRigKydifKi638efgLgC1Qfdi3agy+lGd7snGYbGaFImS537Pot9toAot
T+zzrCyU55yXCUyQXP4IrvkAUIM15w2AUJGMQ00EUii7fb379HHv59sayAd8zDPY
J2/3yEoI6ZfdbellEJb6BUsy0QKBgQDVhYarW/gzZB0Hd/yEqISWJy+XI3BVbuse
BtyxSIM90ZIuD99ERl4DcLhudX4yLwuVEMn4+hnKowv7FaFAMZYr7uvSJa68U+xs
LCTQFFMJi57SnukKxQSNLS9VA+TQ7V8uU4dR0u3LK5h6CwC2fldIw3v4t4zTPI/4
5ZV+paQR6wKBgFKJhBi/6LN5049GkobtnuG4kCUPz3K3HBNR3LmqcECOH0DjLOfv
mdDWyIj9SK73AOzDKuheULQFtV6rBJtlzBVFDiRFwXrGIgiw4YeZwcKkHbsqaEQ6
IUmOJcRD90nvekEwVeD2ZjX/wfe3gWyoEFdtJWh25znKnufZgFyho2vRAoGAI+Wg
8Goz3h/0qW7Iq1zRfZT08Unp9tMBADlTHRL9zYTQOKI+fPH/mEB5IUjrgHBcWCFM
c2lIsMKcnwfR5kLv5k7yoo/nQo4xbzY0si7YjwBEIVDKCR96VmnETVysOVvO5rDw
TRdTFxcrOUlkQ0oWTote6j+hk1nZMpUWfwZGASUCgYAywPvnzASTZtVMkU0hKLMc
sdDjLIJiAlKOZSq0/8aorSq+jgrPlNKb/YMftv5qjRrjxv6LgTK4NNVP8J4ndGhD
Ne9rJz1a6KlX97BEFhGwRXNnAU7oPhoP6Mhr7s6lWoUsy0DJIpODS8/BxEXg7FD/
hgETV75kafH69VOmQTeCPA==
-----END PRIVATE KEY-----";

        // password is `test`.
        public const string RsaEncryptedPkcs8Key = @"
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIIFGTBLBgkqhkiG9w0BBQ0wPjApBgkqhkiG9w0BBQwwHAQIozMG90LwJXECAggA
MAwGCCqGSIb3DQIJBQAwEQYFKw4DAgcECK51Ahgtvr/LBIIEyO38HrNbUgAI95h6
JolbqSkwpPeDfQsxmRlWsqtlin8K63U5gCZS/I1nkkkC13xXHB6WOQ0DrYIGv5pk
ceKiGYNki5UMxw7voXBw2F4pih07iBiATW0uKQnlNxdxbaxH4AEVyxaYTmfI6een
cPyVpgjk8BqZWIrzrcCNq0MfSkI7X1u2VRpni79xETMkdBDeNA5KqwwmeDs8EpRR
laWj2u/s+fpYkR0G1HgrJswMX38Ybp5GBu9i+JJUleDY/ByFRwaEmaMyArdhcI5V
Fp5Bl4vFFsrB0gXf8tApKN3Bmhou2gpO5VbRp3j1iHU27JCa8yBr9KOyrZ8UZrIz
bBcwjHTDOUOp1hwRVu1RUij+b33Bf4IPSobesGovIUgIe0iV+luOZGgitITb5Q4Q
Rup+q4v/M6E2JLry2zDz/FHOmtk83Og+lp9kczHjpGbj7OrGqOyT8YGqpLbmgD6q
vsuus8nCTEpAWG0rmI3RMsOR/nHdQkhgRb8U74vMIFhKfupUVAhLRvB9reKx44p0
k5TUS+YYxh2LdZLquUeYcjMKFbRjSnzO/6N5vo8/Y/AV646bTSzrySmNaOUYArvl
A4l/GMZOFjODfjj0/p2AGE4q5kC9cKmPS99TfqD+l4IBncw9TD/tzi+c/+R+WSy9
evT2cwQE26byo7703B7mn5Y1e0068gitPUhx5mudEloVN8Hg7N+opZpMWLrCAMEx
6kyXbWSdnrSLqv/LTSKXq2WoBk1tJ+/vfsOYdvXHU7oSje8ZoCj70OHsk3f9q8eH
b6R/+hvOJBg4UTtljXJ2BpF8yq34GPLuINYtBuMh/wo7iKQzz9AxdlfSn/EaR8Bz
MnFJlQo76Ie0LSuiHMqMunF9z6TweHMQI37IocBqpOMDRo+jz+Q0AQTe/H8PhQk0
tGN68tPCpgkuY4apjWcNv/6dxm9/ZpR0ZvbA2DXSnxK4p1BQUEeUX1bTrb5DKL6a
KqRPokfoUbdyikGc80+TOk9SJ95if1+2XwT8gN+NLSagCRnJPWdZaOQqInYcaaZ+
fXWPymGWEUjfIa9L71YMTOaVJ73oooC0mGFBKxMlGNamhRSqXUnX8OMJ5Zzflajw
HkgsooaGaL0CjHn7h1zFdctHyJTjRLamKe4f0drBcREz9U3sU5DQ31+hScqtJp+C
k/MZgiRRJudfi2tjNFDa28iP9U0wygPZ0RvD9IXBpYZH7KroYMtT2ysK0q16EJc3
UqhwwEpCj6rPcVktEh/sYmeS6HL19VL03hwdxkKdmQ/7PjI8/vCy96AeIxgpBEJm
ziWCZffAXkbYwSNL2YIHronuvyt4sfjCctnoQUIBoCOQIwPF6yfNKbpL3ihPg55I
EsaecMRl/FDCbew/NqXY1ujA/ntpSMCcE4SkKKIGmzinlINw3BdfLU8bVzlPknCY
3W5ZH9Z5FV4SsEXmrRWFcAKb5kd1hvEl7XtOWpRvOtDbk9WrCko/xY2fgDXyYm25
fl2Y9C69W24gloqGX1EOOCv1sOnUBaOoS0QnGaTPoo7aos+TNah6vB3qjylwxFFE
YTdGR9lY5wKHcJeFATIIcQyL+cziHNJQCwM839Q3uc0Sj6JvVm0sCHd/VUgzhfzX
16v5sI3D+15h+GTFjA==
-----END ENCRYPTED PRIVATE KEY-----";

        public const string Ed25519Certificate = @"
-----BEGIN CERTIFICATE-----
MIICFzCCAcmgAwIBAgIUXX/A8D+9ewGf98c9FjYwAAhRIsswBQYDK2VwMIGAMQsw
CQYDVQQGEwJVUzEcMBoGA1UECAwTRGlzdGljdCBvZiBDb2x1bWJpYTETMBEGA1UE
BwwKV2FzaGluZ3RvbjEUMBIGA1UECgwLRGV2ZWxvcG1lbnQxFDASBgNVBAsMC0Rl
dmVsb3BtZW50MRIwEAYDVQQDDAlUZXN0IENlcnQwHhcNMjAwNDI4MTUwMjMwWhcN
MjEwNDI4MTUwMjMwWjCBgDELMAkGA1UEBhMCVVMxHDAaBgNVBAgME0Rpc3RpY3Qg
b2YgQ29sdW1iaWExEzARBgNVBAcMCldhc2hpbmd0b24xFDASBgNVBAoMC0RldmVs
b3BtZW50MRQwEgYDVQQLDAtEZXZlbG9wbWVudDESMBAGA1UEAwwJVGVzdCBDZXJ0
MCowBQYDK2VwAyEAJ0WQ+SKMG8q3g9alCNzq0t3vjAeN295rq6IdaFj0NmSjUzBR
MB0GA1UdDgQWBBTUdW0AAtdSTqb1Oc6J2Pfvsk4bpjAfBgNVHSMEGDAWgBTUdW0A
AtdSTqb1Oc6J2Pfvsk4bpjAPBgNVHRMBAf8EBTADAQH/MAUGAytlcANBAOjstGTq
0fmzySS+DjSNfvkB9IXmaGd3XGIM8SWTKCnFi7L+mXhDBgd9wLO3H+EulBiuKtIo
mf0l9zwCygC+aA8=
-----END CERTIFICATE-----";

        public const string OtherRsaPkcs1Key = @"
-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA05yPUFo642M7gUYBYZgp9Itk/yPcrLcisUImPvQAsrrCeQDR
kKxo1rNApfEZAK5UffEL7kbYFRY7v0mhwWY4K2g17FJ5Bd7W54Sj5ziDzk5LKvvN
dxjZIIDi69z+AZYKcPWu4D6Znbi8isDeWv8mUTeFcA8Ine4YuvsAJ6b6IHGuz0Kr
qCZeE6319wjA6Bh14UziIPxOm1LLJ8Vlwq7dtngk034kUyliO9kWJg7aNatthn5U
q3nSUUCarhjfMFu0GhBy0TFRfsQmKNLOBMtrtx058aVbzeR3XO6pYhVsdF1cwhoP
SXPCYpgc9ZslH9u9RjQoP/0+vRuT3jBfHFKtvQIDAQABAoIBADufE93+3jKtBdoB
gGgf+Eo3cChW0Vk0bCjnS2FXXE7/QcXYDjOl8A/2B1P53yKK+7FUVhk3irA+SG03
8MRN6auJPBAumHyn8YLfo1KFYNWix5j/wz84GA0JY2YzKLoHrT9waWozGRMQNscd
WkjnNMOTUhxlj/b3W65rA/soF7dF7Vohb4Ab251apJi/k/IBkv6gVUUH6adhPSxG
piEY4AZ0Vpngf14nI5gesStFLBhJqBtHbMeicdQbHJzg9OjNpQzNGCQzCx7EIteM
LJ/DZ+r6S9qm/m5qvgxALc0SMTi58RgIf0JwWpTrUuEEOfO0qgjKZoi8As8yFR27
cU0n+MECgYEA7m5U81M+d/nV116Ln1+s9t0NbaR0Vzk68WsHok54lSUTItcsuCxo
CpCp7B0ARi8KzG3W9Tv+NPS64JZ7iYj/ST6w02IjgpbldpJXmPl4wV57uUOPsTq8
hReeGEcXGN5wYTCo6jiWNmiLR4CNScmFjxcjwcXSGwac5+Sn/MttT5ECgYEA4zRR
JPaDqluVhGhu+uK5Jhb1tqeTlruSxorRuN2GAApaf354+SdfQffD73zqCpXDdtbq
uT7/Qu6tniR1bEFpLKSGeH/ayLYNfxTy9m9aC6ccxTmEhbDSntY/HHo50CtWJOWz
F5O4R6aeE5rqWfHiB+Ya9dDhdVPor5wUyPUOfW0CgYBMGw2kqaKb+zRFzZj1oz17
gu3BXKgCG3N0Efza0v9sY+wqx9Iva4U+MhT7F/q5bFSfEkR1/NNUpfVssLv4F7Gc
+JTKtF2vVmkiIu4xFxhzaKxHY4hfQudf+DzvdOmrd30ZmMWiFbPk5BPpG+B9eATY
usMgG/vHwqGc54CzkV9v8QKBgHlveP9ckrr3AE8o0khd7b+h/eqGXqft0WE0ySsZ
m4lh/0StgFMK7CsCFkNmbGED8tkNvZ8NQLmxgDJKIkieHWyy8vxsua8VPtlxhPqa
QXKA+yuetmoOPESRFmJOIaBVyVEnRDWRyqjhMRQhdKhmU/0My9QetKJVGsThk2pl
MD1xAoGBANrkobUHP4sUIGenNfWld1As4tmKXRDTv2HHVKOCvIfF8XY5jMQBf96W
Tx7VNbqFa2IEKZW07Q3TmLgD4cplE+XxIKGRGIie3PECiQdO+EieEtPe+nFLDUrb
w4qGwpDllP58gBySGj3BQS8PG8d9fuOlCSfOQIW10dE8U9H0wwlE
-----END RSA PRIVATE KEY-----";

        public const string OtherRsaPkcs8EncryptedKey = @"
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIIFGTBLBgkqhkiG9w0BBQ0wPjApBgkqhkiG9w0BBQwwHAQICs3NqD8hVe0CAggA
MAwGCCqGSIb3DQIJBQAwEQYFKw4DAgcECBLUM63bM9vSBIIEyAk8lJcgxlkclPR0
9BlUXGQA/ll6zCQa1DDiXdrLxTKuuedwOv0YF3X/iiZlhpey0E8zjKo13HQt7P+w
rpYVvA+qHH+4k0oc/ZC4ZD/ml4AIKCpQGs0AX8gQXJ9z8heFbCGAIYe2O8sx+Teh
V9YNbhzXiZ521+pPpMgzlrfeDFVimjCEw9ESNuHl1nJaHLzeC+S6TL5uXNJCy3BF
XwNYH53GfSIWwWcxiDKdHu0uI0pTUgCUKRgm4qWxqQvbO+dd+pBvqa/nrd5Yk8Sa
vSUmsSHBHiMnYoEs2guD3BD6YIzgzGQ3dR9LWCJNCh7BavnYoGFPCre79RiA+Wdy
8Sxd+irf12JESZdQPsUSAiMBHtZ3gpsvO1bA+2xwSqE0EcltHdrDyUoLzUAx+2yU
0uvJxRlchYsNLAkDcezOWA1W4uisNBQCXxklSLIVRk7zKvAtZ5vLm0fvybCYoHyH
xPTPs3k16xBF5lsquGF/XHcxfKjB2s8oMDRU/7ZnOSjCOyZV/Ey07ZkjhU5GbyxI
ninu5h7q2K79GWOm3B4LaEzl3IYFCFifJqod1UyI08fv3l2x5GdGlEL9Hdvl1nAW
Ty4JnhvvJR4VdGGOczJNDv5/ZDirN4wpW5Q5QUnbedIZjiTugxHC81Q1JxMXfFPs
iDLCVXAU4oZwfkQYJYbwvR/5Zu973zFRIjsp2cJgdN8XsS5mFUD69cnGnZaHEMJy
o+Mki2VMF6QiimVS4RezJFQcdglrY4Uy16J4indJFGIjJez78M1+Wdr0iDk5akOr
pln+Dj5yim3IG5+hfY5Q/jkbuPMSUkfsLhGAHrxKCLcJSj6+FYV4/Mk7lpC7WPND
xkVTYae0XDOAKGzXZ2fvS7ORk5ZJDn0RKoHEv31QXpIP4fyzM54x0tknnxlnI++J
erLOgEnl0r9F0uM4Lw/JqYRhfjrroMUgGxDWSHwGtHkanhjQNAKTY4zLp830HZY/
yii7+Tt7q29/P4R0cHUcnqdz77re/zYZK7ru63UNZwnE6hybG4c9vneCVOT6bmxG
Acxu2L6nVpUhxhhDQ/K1d0JgFNRg15rQjA/LN5MTfIDimPZAdGhDPey0NLLl2BVG
jc9JRT7KopOh2eypWcXecNCHqoDAKcHdAi71OIZRKEQS8BGVk/LY55iwdgKqkqLF
+iuWtEfgDD9aekOBKLdTQqn8PvvHc3+QOFIoDzQ7iJLh7Uca6MUqK0uwBieDnK7B
8b9NLs2Uz1Rr4UqK+r+06QUQ/kOldgeLguouOt9fz+ugZNxfvDGXSbkzMXQbsSJ4
EFs7DHD/76GXsgN6p95LeCdR1jGgFV/HlD+pdNUW5A+P3hs02UqvkWzeC4ZOBnLX
/oLjja5XWuxuYkyk0yE4UP0CzJMnrVwzKeqesrWKSP42miwSxPzSY4AY8pZnYY4Y
iOGN5vueX9hkfRmsrrud6rujPuS49ho+UkJuygmR9ElGktvzw1NbouhoaVsFyrRV
cEL42Y0WZVuUrNcm3X8ng5HkT2L5D9Rys3WV0GaaxximLzGzuhIsfG3cCiLq+ZT5
XOmtog9dZs8lAIM58M/a/7q7dP0+FGxQ41/JslOjbWY5zypViy3aZssrkVF+7xZ/
oWZazn8MBIuVRT4TcA==
-----END ENCRYPTED PRIVATE KEY-----";

        public const string DsaCertificate = @"
-----BEGIN CERTIFICATE-----
MIIDWTCCAxWgAwIBAgIUFRQGA90GHC74cNK/hNzQDi7XJFYwCwYJYIZIAWUDBAMC
MF0xCzAJBgNVBAYTAlVTMREwDwYDVQQIDAhWaXJnaW5pYTETMBEGA1UEBwwKQWxl
eGFuZHJpYTEQMA4GA1UECgwHQ29udG9zbzEUMBIGA1UECwwLRGV2ZWxvcG1lbnQw
HhcNMjAwNjE5MTkyODIwWhcNMjAwNzE5MTkyODIwWjBdMQswCQYDVQQGEwJVUzER
MA8GA1UECAwIVmlyZ2luaWExEzARBgNVBAcMCkFsZXhhbmRyaWExEDAOBgNVBAoM
B0NvbnRvc28xFDASBgNVBAsMC0RldmVsb3BtZW50MIIBtjCCASsGByqGSM44BAEw
ggEeAoGBAJyiyioeXx1O98gRCMEjlPKMpr79KrcDkoroghtuXO1U6Cx34pBRjOQm
QLDPqSOriEo5VuG6SJc/ppfZx9TrSrzqB26hKTUmiaOKmwpfIfzpi72wgsZeMOtU
7JQ+FThfGyS8VxGh6G0h7xw26B/9ALxRw25zO1cy9ZJs0EY3hsHzAhUA/4dpclsc
k8K+SkWBTcPfU+x7wTUCgYB4LP6UvrvIiiFPxhk7AEGMMr0MhcJ3hhsgKWukUqIY
sJKBM5MpKCnej5BHvnLXdKodIxygcKR4dJX7BRv69L+2RJk+UrYL1qBco5HpUslu
mA0e3gNdwRLoOoGD14dn1LD1LdESsyMgwfHHJ0RRkYwacgCVXsvHv/eAkA8qq136
dwOBhAACgYAHltgzkK3zD8yGdcGY0YgvN5l3lna1voLmcK+XtmehjMVy7OSSFICN
KybLBOvO8paydhCb1J0klkLPAoAjgP2cEd+KueeRyJpx+jD1MsjIEXIn5jtjXdUH
d0JJmHWAyHdNzmhXrXC7JLnj4ri7xMAV3GZGDpAnYvvL0LiXzFyomqNTMFEwHQYD
VR0OBBYEFF1l4ZrF3ND05CjGd//ev0dJLCB7MB8GA1UdIwQYMBaAFF1l4ZrF3ND0
5CjGd//ev0dJLCB7MA8GA1UdEwEB/wQFMAMBAf8wCwYJYIZIAWUDBAMCAzEAMC4C
FQD6plYf60MDCvMjf1yQ8SBaFX3YYwIVAKqRQklh2b0Qhv+US222hb8xySJV
-----END CERTIFICATE-----";

        public const string DsaPkcs8Key = @"
-----BEGIN PRIVATE KEY-----
MIIBSwIBADCCASsGByqGSM44BAEwggEeAoGBAJyiyioeXx1O98gRCMEjlPKMpr79
KrcDkoroghtuXO1U6Cx34pBRjOQmQLDPqSOriEo5VuG6SJc/ppfZx9TrSrzqB26h
KTUmiaOKmwpfIfzpi72wgsZeMOtU7JQ+FThfGyS8VxGh6G0h7xw26B/9ALxRw25z
O1cy9ZJs0EY3hsHzAhUA/4dpclsck8K+SkWBTcPfU+x7wTUCgYB4LP6UvrvIiiFP
xhk7AEGMMr0MhcJ3hhsgKWukUqIYsJKBM5MpKCnej5BHvnLXdKodIxygcKR4dJX7
BRv69L+2RJk+UrYL1qBco5HpUslumA0e3gNdwRLoOoGD14dn1LD1LdESsyMgwfHH
J0RRkYwacgCVXsvHv/eAkA8qq136dwQXAhUA216Tqp4OvdUBNv8QLv8Z5QPopGQ=
-----END PRIVATE KEY-----";

        public const string DsaEncryptedPkcs8Key = @"
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIIBoTBLBgkqhkiG9w0BBQ0wPjApBgkqhkiG9w0BBQwwHAQI+PhdT1Kk/SkCAggA
MAwGCCqGSIb3DQIJBQAwEQYFKw4DAgcECGV1ZmaiQtz2BIIBUA/6pNqTkXpkOLlI
22Lh0cm5+/foDRh3qTrAOSHHHV0Dz1xYvYMa9MFzONatLf55Rpb2ZPji3hXwUQfn
gOJeTBRTaMNz5LaKJiOIWj0qDckhgKt9cmgiBzVTvXO4pERp1uz5zcvaUOKj2TSv
ljxishj76MYQftIGMMkJQKf4OsHubCopuKUbzTPgJt0FuF4eT37+tiEMgbYrmA6p
REPE0vT1aY+LYdJLV/Dax/l4lMvYmQYOWs9TCLPlI5RZQxxte6zbcA13ESg/qLE3
4Mx8xgXrPvCxp3h8KBKNMaJR1xzpr7UQOpkI9qja++3cJAl6O/0mdeqZct0V9Z8P
a3+wyUWo58z5sOPNdJHIMV6qw6m3w+IQoCJC7EbV0+Pyo5eSU5zbgm7YWZ9Yx6l8
g1mCP4Q6Tqe6LjKfBsZAmYWSfKqoTKRjC3ocJMt53tIDpB5jFw==
-----END ENCRYPTED PRIVATE KEY-----";

        public const string DsaPkcs8PublicKey = @"
-----BEGIN PUBLIC KEY-----
MIIBtjCCASsGByqGSM44BAEwggEeAoGBAJyiyioeXx1O98gRCMEjlPKMpr79KrcD
koroghtuXO1U6Cx34pBRjOQmQLDPqSOriEo5VuG6SJc/ppfZx9TrSrzqB26hKTUm
iaOKmwpfIfzpi72wgsZeMOtU7JQ+FThfGyS8VxGh6G0h7xw26B/9ALxRw25zO1cy
9ZJs0EY3hsHzAhUA/4dpclsck8K+SkWBTcPfU+x7wTUCgYB4LP6UvrvIiiFPxhk7
AEGMMr0MhcJ3hhsgKWukUqIYsJKBM5MpKCnej5BHvnLXdKodIxygcKR4dJX7BRv6
9L+2RJk+UrYL1qBco5HpUslumA0e3gNdwRLoOoGD14dn1LD1LdESsyMgwfHHJ0RR
kYwacgCVXsvHv/eAkA8qq136dwOBhAACgYAHltgzkK3zD8yGdcGY0YgvN5l3lna1
voLmcK+XtmehjMVy7OSSFICNKybLBOvO8paydhCb1J0klkLPAoAjgP2cEd+KueeR
yJpx+jD1MsjIEXIn5jtjXdUHd0JJmHWAyHdNzmhXrXC7JLnj4ri7xMAV3GZGDpAn
YvvL0LiXzFyomg==
-----END PUBLIC KEY-----";

        // This string is for tests that are sensitive to the exact formatting
        // and line breaks of the PEM.
        public const string CertRfc7468Wrapped =
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBETCBuaADAgECAgkA9StU5ZnBmM4wCgYIKoZIzj0EAwIwDzENMAsGA1UEAxME\n" +
            "dGlueTAeFw0yMTA5MTUyMjAyNDNaFw0yMTA5MTUyMjAyNDNaMA8xDTALBgNVBAMT\n" +
            "BHRpbnkwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAQZ+baUXzzLi+p3cZEf4f23\n" +
            "L/2Dbn5UB/uMCB7L71rWf3UwuCA3Is5uPci/3PQYLNwDkP3m3ZzxyzVCgFVqqYFg\n" +
            "MAoGCCqGSM49BAMCA0cAMEQCIHafyKHQhv+03DaOJpuotD+jNu0Nc9pUI9OA8pUY\n" +
            "3+qJAiBsqKjtc8LuGtUoqGvxLLQJwJ2QNY/qyEGtaImlqTYg5w==\n" +
            "-----END CERTIFICATE-----";

        public static readonly byte[] PfxWithNoPassword = (
            "308205DB0201033082059706092A864886F70D010701A0820588048205843082" +
            "0580308202F106092A864886F70D010701A08202E2048202DE308202DA308202" +
            "D6060B2A864886F70D010C0A0102A08202AE308202AA3024060A2A864886F70D" +
            "010C010330160410286B4EFF0202AFE16583E50C3EB8F38F020207D004820280" +
            "C3452B3850AE69FCC0AC426FF3A0421477813259C128D643452219EEF71EBCAB" +
            "5E7054B7A195E3F945222864CA37D8F67DDFA9136A93CD7FEAD86F00D4179F1C" +
            "557253253C6235295499729C564DE2CE30E131C0D9B3E1BDBB211F8FA9E78B7B" +
            "088C63137DF44CF50C293E082E7C57A8D0CB0404D1F5B9D1491F4EF9045181D3" +
            "8D528C61F49EB3F1CF11ABB60270CBC10AC4BAF115A5AB52EA22FE4406743695" +
            "7DDC1BFEE0C6BDC097BDF092AC6D11CABAE497FC10564E7E7797BC6028CBD75B" +
            "1A2339329D439F6557B3CEB77489467FC8990EE832D48E2FF65A7BCB20E2DBBD" +
            "F81C762F688E2EE43822CF9DEDE11914DD982FAB2AC141496D912396F6F67E3C" +
            "D04D0617F8EC2BE6D35AF9860C384DB8C21FD0B00494FA3188983E6200DE90B3" +
            "E6E662C5B07AB202A1B9C3F10F03B88E677EC7BDC2873AA4DFF873DCD714AD39" +
            "42C33E63442A855C709F58063D836F8CA77DB9D208F3DB2552D7AC611409E8BA" +
            "942BE520D9B4951AB844892D123DF550BA4F22D255069AD77E6C14D730128FAD" +
            "3736551CDBEED2022417A948B1BD2567C39DFB561F251C45D2F4B5711B4AC82A" +
            "09226403432FEEEEAC5F54590CC2EC7925768651ACF42C32A114ED133718FB1A" +
            "9FB0DCC35C10640226F7587BD4EA67C10EF963113D03988C82B3A0B43B1B60ED" +
            "BE0A7CBCB6422CD9695BF206A190E8EA6F87E2FABB33549081EFA7A8B04CA044" +
            "9DF52E133556781511CDFCBDC55B487DA51C8D476FB635896FD0C71754B2EEE9" +
            "A395F355F4A56CCB8CE75D65F73AE8DE80215EA7CA9129FD90EC48ECF26828A6" +
            "0E471FA5F34471AD1C21AD200E16C4E4E99FA55B5CF75BC4BF68D1A975E8276F" +
            "5EBD47EEB21DB06092F51AEE6BDAA2CDBC9387620A2B3983247E9AA93252D49A" +
            "3115301306092A864886F70D01091531060404000000003082028706092A8648" +
            "86F70D010706A0820278308202740201003082026D06092A864886F70D010701" +
            "3024060A2A864886F70D010C010330160410127070F8C0D10011CBE0CC8489A3" +
            "0C71020207D080820238C151B340B678AE8CAD6C6B11E0661E91FD1F4107F0DB" +
            "687F039362D88AFF382E3557A75157D8A6D93F0777AD41D7520D32916677699A" +
            "EC3DAEE462344BF18410EE07E83811EC26569FF9CD8D13A77F387D6E7C5C21C1" +
            "6BC9936DB1B8AB614A8B4B6F9975E4A0A5DFCE29EF14E833FB2526805901A782" +
            "724AF6BA2A80E93A4C4BA07B1C1319169E200A4B7AB100AE2CA135512919120A" +
            "1D1AB57EF6DED00144F87D051391676D205196ABB4B211698BF137436D6E39D6" +
            "719737B66AD2E76D5766D36E87108D79145021C77328A9F2ADDF44EC2A95EEA3" +
            "86DCA32FB53D0AA92FB5C5BB7B49CB1F1755ABF195C7274702681C616C21BB05" +
            "817B5FD344FC25CC6145A4FFE36F4D5BC131434E6C44BD14769EB08FBDA0D1A0" +
            "6DCF2D061FA4A2FCC45A30125680507AACEE7903ABA0C1D36395925A82741797" +
            "CF93A11F249D7E7D8228F8F6AFD03FB317D1F2BDB319C0AEF15E19E9DB1067B1" +
            "A6CC12CA458C33BAB31C3F275C45A956F71CFF939F393EC7D20E13B397E64263" +
            "702B54DD228D0E1275B39A77B3B1A28EFD5C7DF2643CAF7AAF8574988CDF4112" +
            "E057F715331F6E75462E6C948BFB92C5BC81B84FBB47FB97AEB3D8C228388B94" +
            "CDFA0E2A48F05A32EA9F2CDFAE2B0CEFF815531B148C358ECE1D23F7B793A1FF" +
            "ECE491E990BDF231756600B87FA2F7F3AD2C2AA2F6DB42FC6D62766F28F60436" +
            "FFA4993C87BE6631D7CE6C06C5B7AE7218450C504ADB401B9FBA5FB2FC6A8289" +
            "B42D51B4E1AF159AE7F3A63BA8644C8C5A99F108FC25A27DE54E268AF8A259D9" +
            "F6E3303B301F300706052B0E03021A0414AF311074EBABE699402460BFFFE14E" +
            "4D7314FCB4041469835268466D1390373566F7034C4736346CD17D020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs12NoPasswordRandomCounts =
            Convert.FromBase64String(
                /* [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Self-signed cert created specifically for inclusion in public-facing unit tests.")] */ @"
                MIIvdAIBAzCCLtQGCSqGSIb3DQEHAaCCLsUEgi7BMIIuvTCCLrkGCSqGSIb3DQEHAaCCLqoEgi6m
                MIIuojCCAvkGCyqGSIb3DQEMCgECoIIC6DCCAuQwXgYJKoZIhvcNAQUNMFEwMAYJKoZIhvcNAQUM
                MCMEELD+7LV5Y9tyUiJnNeZVLwQCASowDAYIKoZIhvcNAgsFADAdBglghkgBZQMEASoEEBzHfelA
                4f5vP3LmQVtFodcEggKAfy/+/5lLpe/Ti1qZB1kbVrY1nheb2TVTfqLyjqUxsQlo4ElbAdgvlkJi
                NpmGu//pniqKxODkZujRMTty8VfdnLpe8gVGCltSiDSXF3ttcsv7yKYkJiU5GN4cWwz8qud6P6cn
                zEHtnJJ4AkNJT/Lkjy7ktTbww+ha/1IbteKWFW7tarKkjrx2PxqiizLKhydPvLrr7v0tqgv9K2/G
                mFvJF9i/ynVHS2NtD11kHysBj3DjMTSEddBp7th1gXNevQ4NcxNL9hG0g4tCYS/Tq6WRMJip5PSb
                7tsYxtWJU4vb3JYKbadVnOwiP7VAUOEU8YJUiF6lBtWvNTwpmRg8S/Jg4EWqOX32DjSl5jx/E+q6
                ljMANkInnAI7uPa87GACaaCfKI7sPtlwjz2JQtj5HUrjGu5ff0yY2fl98FneuO5n6atD1X7460vc
                KdeDAhXsWh/oBGrmREYvPFI/9vE8Lm8i5FKzZZTY+XiTsQMeLeFLmzikqCubfcQhtYOJLCZUb3LD
                HALSKFdtKpBJAp0rrh6NkR5ss3jFQmclctJeYYFGPfHhM6g75jqR/3X8JUucduJ8+2k25B5qqiQ2
                xMVYwLdCTU8bW2KIt3VCROEO1n+kLM94Byju1f57vn0ZkRhKtVS2AvmxdxAD2mrVE6Mj6Hw2VWHi
                Bnqer8BrmOhVOoKuRj49Nz5PFGoncdayJJe6h9/HDB+RofLe8tKPbLDHbDm1jmJaZoBHbypESW5f
                KUKHhjsH1awzNMs180o71hsiL4NuUqUk9nEAvx3QqolHIuGe2YnLlRBWzOB0lCqmMBV2D91O1SIX
                RfZ0T+0AA87jGvjK+sCxVNQvqwgEYZTQAn11HDCCAcsGCyqGSIb3DQEMCgEDoIIBujCCAbYGCiqG
                SIb3DQEJFgGgggGmBIIBojCCAZ4wggEHoAMCAQICCQDG4/UcO7qKXjANBgkqhkiG9w0BAQsFADAR
                MQ8wDQYDVQQDEwZwb3RhdG8wHhcNMjMwNDE4MjAyNTA4WhcNMjMwNDE4MjAyNTA4WjARMQ8wDQYD
                VQQDEwZwb3RhdG8wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAL3Hf+x7UDSJ6CzFBJ+riwvJ
                TepV4Zjy/NY+09FbR9L6rmzF8fp6IyT9c5osmCK/Q81e3fptYFeSowcm4v628tM8Aw9L8+H8OO+w
                xfwiQeyRXCFCwl+rRgKVuvkvtupejL6aZdaZmvF1GP/Jho13zY1ILBMzgtw+MXnZVsWi4iYxAgMB
                AAEwDQYJKoZIhvcNAQELBQADgYEAE8FaAa0klFrcONIsUk6A9M45lo/iXZjnxuq/X05tDlUd/nDW
                Am6bYrSxTC/o3LcYkRg+GVy8Gs1AaeeG/MFXe2zm9N7DuOqKtslKvmjt8L0L1E3zygUc4DUl87y8
                p0tn8bKI/RoflMNU/mHr8U3pKg9SaENLZq0TlBXLrr8IGyQwggL5BgsqhkiG9w0BDAoBAqCCAugw
                ggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBB6JnCtAnPpOcMgsMFOBUggAgE9MAwG
                CCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBAFove+QNh2kcK4jBcdRQZqBIICgEZS8QJsHwDdktJ6
                4TalaTiYmQ734FJDwHA8v0qhMEh7gWGf/0X2dRi2s27lU78g0C3YtJNnEFRtICTpco6mJu1r/+nq
                9QG6OTdwfiB8vKPpkQaLVQr4bl9rA0lAOP9Lqyv9cjFg97GLJCrsPkeykV0FafBRyeQdMjDR60yO
                X2AcV3XFPrzphibsAazyC0IymSfUvytHK54PI4z9RWGwFncBZjItkbNPxfVoMRnfBpLOYpz/nfFu
                eky6qItXtXkz0rfC4qNPEpPEd4E2ecoRWJxCK6ItS2R5zfrTeQ0LqlkptBvcxC2k3TfSl+TVTee7
                g0+z8bc9JypL9qr/M4Ir8qLbdGT4aAigRrS8+s336p3J4qJkdFreS+FtHHlmms6E28g2h6HwynCG
                Mc/z8TikKm/ltJ4+FiB6g+NywPgJbMFBjsaNF+wPlGMRO83M+3zysQCPrhtdo2gsjnJEqBN2kXu8
                4k05UAHTpsrZn7FmAG7Kt553y/xJqY8U0rwswlcy402CKtS26kQ7Vv0Ic3YfsaNypNwYo1h5b/tl
                t/8c0Sws8JoEj4RW/dflaMmZ5qDdZ1Smiwgx3l4VC9lpm2AmTlvV3MBdsnAKMfQuilqK14rOftaG
                lzKIEI34F/Z7qUKCTL6LgTsHxt6L/aHxeraXSMplLb7E6rRlau7BYZBNbtZ6hHRzqBIui9m7iQ23
                3wYxpK7etlFz6fAHyFFxsawMrjgFq24L3dugCa2wuUPesNwcSn2Zte6FS+JUOgaOqWAsSwhDPn+u
                Fws/t23vpKm1cBtNAE4kl6IWI4/QBfxnAHabNE66LZfFUSFXsk+wKJO8PKsfSLZ6oVZSkRcOHQ4W
                EN4wggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAj
                BBDJY4TwjOXIl9C20wMX0JQSAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBBegG+K9bCn
                WqkCVKlZOYRrBIICgO1ic3SIALh0ujMqyrqazrx+E53ShenbUAx5fgSka7pINpFjtFikScuN6bAc
                fs2lQTgH0+alWGr+HLoHa0GsO4Z/TPYZqswD8NFoVFKz5gK2zXMfFrObp/0HVj8hLMfOKiMH6tWo
                yL0gDDn3t1kz9v2eglf8ifjS1jFANpX4s/hpEFdCfzqjsQwy7ToZdn2BB2CbUy9A3GFoi/z1O+7N
                angPPr4r6PE1jOH63TFIRwxmH6kLGGc253UxQI4P+v5g/hh19dRByjIQvk2TyWYRugkAx/B4RNzD
                NAWYFvae79uWhKQrfzUWhGhtfmWxqcZnwDb69aCdzBBXOKz7n2ZCK6Ex8qK2gm29foqjMQnX8YQZ
                EB5dRX1r+jIBHwBQFyz4fofByVdjb4SA7g7RAmoCRYsJYupz9Gnesg0De3iADbTx97oco60PXdjL
                OHhGqjrMpgAKSVS+tULc33CSUHq+Zsevs/hSB0Aqh6YH+3kWp/LdRB72FUHQSpt5/aFm84M8guc7
                lD11Fcgpj/y1zUmyFlyvRgvGMeFhkGTJPkkSpFXq+M09xAuljtQ+QCTF1Q/jx9cI9glconr+umd2
                VB0Mq/ppRrbiCGFNLOOfHk6na1Z2FAv8bywZ701WyMaZRvgaYca8ueKEh77MQsymG/5A1kKueiRW
                9Gx6tP0OJDWUCaOeGMJQFdP/e52FC+EdKkAZajngWThSX/ZO2Mp8oxufHRGcGHIdzFCzh5iaVxb9
                7pG3rV6NS9wSz8eAMixmEmf44P8rOhiHbyxoaCB+6iYluHtQuw2fU7JlIPFrK/o++rGtDq+GHQw2
                zXUIbrpw1MzbtUgW/hTRlqjasXtlrgDzSZYwggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqG
                SIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBAQSmfTMz55MV7GR5lEF9SXAgE9MAwGCCqGSIb3DQIL
                BQAwHQYJYIZIAWUDBAEqBBDLBwbNgTYNIoSJTI9zbdvuBIICgHux5dOkqx9pQCQ8nDYiqUJAkfl3
                SEF+FRJb2xUErUshDNwFeC8NHkJ6R/KkBAtqRtak44/EuN8/RPrZa4huIFxu71hABETWit+fyirU
                HnqAS/mz66pe3IxRr5qFY5/WxUCsYj3lYjfVyLKrCwBZhtYyrXQJg+0qWYs6+16ij7pifosDb90e
                UgD5U9jqwMIwAdjiTxQ6BSq9AeIlhdI9Y1z13kfcXXhWpNu6EWlcYB2VNPZ4r32CRJe+fR9AWXCb
                iajbIItWjhiwRlVLao8KcFMAq5N5CaDEZj3RwVb7Cf24Pb6j+dF+EotSW2Wem3ZAPeKOTI0Ykk2f
                MtZPD5znGZBJtbBM6iuoqm3+axrnK6f8XAyKdKEHnA3lqGtQR8GimqwCFkBv/gUVk9eVCvL6yErp
                zoJpA6r0fLo1NQGSW17TR0NMyfyPThqRrUtTZlDCWzgJbfNWs/oaAplX63naXsUtS3duYQSTHO95
                4e89OrrSa/eXSYn30j767GEkIUGBFaPcFqV86FMBiVbDOmvEYBK6ORdBqYrjLAD88EYX9e/pcmgn
                svWLwWlYe+uMAGMOxsuvQcp3vqlyPCxuYrh6LdKVBrK4m9wQAIyF3niDrGOJpmVztKdDaB51jDav
                nK16vc5JpG3BlnnpjBEEYJGUS18ms6RTH4iLpk/Ui4A32sdhM44P/nnUoEtaDrKoUxMNS6UcDaEf
                5N7eJ+ccEDGym3tGxVCT7i6uKrQKEyKf7FI0mf9NgPk1Lz/DBCyvnRVgx7qEdhmbhEUZgQ7ccmBj
                V4cFR7+NfnxUABCyJyZu+gLFqaDfALhQvwmCIrAgOgx8FyeMw8MWdFFUk0WV7K0nUh4wggL5Bgsq
                hkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBA8NwJFoLrq
                bxCmSinxHNnGAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBB3zPuXtX75pJJA2Rvq+DyC
                BIICgF/qdMfgy4cf03s8T2UbLt1bsLvsJgptXjgrGYWfc6lgidaWxGPf+66HwqaCA4sTPH/U98jO
                Ok8ljcVApAAlAONsQKlOYm1G/V8AxgF1RD02GQo148+kHq4swNcdXoAcowuvEihsGKk1jszXH2L0
                iahiAMwRimEglTaGBJd1cJLhVe45A7RHPl8hKZECvvSx2kB1h26pqBIYRoGSamAwABRKYc43YlBG
                eczB8Ry+sZUwrhntjSFeGa61TGM9jWr04GjnnTWN+IxRXz9Epe0NWFPMcOXg+gIPFf6RyULYdSO0
                kNb3jolrz7nYKO4qTHwuqYrQGJC/3sfqc+s2SCNZnIhrQf3Q2L3jhqxnc/u7gdxqXGwFCxkeG/q+
                TW8Yt4ncqETFqmQs8J6IMPncuoRHmYT+9JZjBnCakFjsrwOBQD69K1VMJ1ZbfZY66YAA+tLUuDB5
                khn5zmeqbyjRDaG+sZZtvHYhob+Hbz0o1Uv9vXQ3t2mVENHPj2LI6Fk+AGNKp5O4DYBVsrak8huL
                hc7W9Z9/SAAUFfkf8P4fp18WgTnWr8Wv7UBSHmwTWwzjEERCI+KENH/zODdlwQH2I8/bsS8vNC91
                WWXrBIF1BU1jsHSYbqkDvOugll8bY9LB04isZfcUucdIJH9U7FRbAR3hXeC47C9C6kZvKVqwW5ag
                KcFKE212LRK19NQ8UBNwQdxL3WIlT/32AHVVzkBfL5mUqkwTQWaZrD+6+wrVKsG28ECrJvYUQlQg
                L4+uErK5RWgfTOJHgrlWB0bd+AJ7oZ2agys9eL8Nh9/pzR8vp3xrrwofFRZxc+8rzHRvcJhjpUAQ
                uWK1DJtx8SBLWXJh6QLXiGIwggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBR
                MDAGCSqGSIb3DQEFDDAjBBCRnVLg3B7r7n0iuf43dPAcAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZI
                AWUDBAEqBBCNlDLKYmIdyzfRqK8rUN3IBIICgGM4XE2PVGrFipIfzqKwU17TmIDKKPPBbk24RCo9
                0Q9yRIH3yyLoV8rd3xONA9eUU5JePhwDEgDS3z1Vl6E4qAVXZ86E+APODs2ZyhYO3IA7NmUjvlH1
                s01OjmVIQdn/GHhfbYl5vseuedFtYmKGMmyUxCOVtYBro1P5RKZF0boeA4Dq+9tl6awVptEN18yL
                a24HmWKfleY9uYQZ26sX3LFmzhCZI6gSpdcg9Fv+WzoFZn/XbhY9g4IK7OMzEYnqw9JGBQRl/kNR
                1sSkxPiB8CHWfeNkQz6hfUrubFI9+rZCzMn+j1LC7ERFKff4bX+qBU766+jB7DK0vL2Q6QqLYvGR
                IrYGn6B2X2X7sOA8CYIvFKVawy7Ec9ZLi3cxriFgZMdmkB9mzzenq+wB8gkFqnnbw+CjTWQTwB4P
                SZEUsls1uHgoClg+UMvpNXJ0YosdfmrOswSwG82kPF8R6ZT89o7E/+8SGKz7khF0U9Tdn+ZygoL6
                OxuMln0Jwg//hp0QejHYxVxRy9D5GHVH4FuLpei8Jwpisz3Rvb5+59SUU3VKlkxuPD2R7OJuQVkL
                N+GXeEOJ59gOI38FKef7p/AJU8PdTt4nxCTjLpbBsNkG9NcY9ORpEY3FCjEJK6jLiid04FE94A1F
                0Xbsr6fWFIbCrYG+pZiashoiboAyUTEiwFuNoI0ao1sNaOkEkFiEDjxvlGZ7KH8Cv+wxZ8qDpcUR
                Uhd2VEmpN1/RbWlXfqgGANFo/jipLgNXOCCMrBCpkkPnscoKoWAQNVSbEjcelAslXiCqumR+EBr6
                Hx3W0prA2SyGlhHDjJGA1c3a9oPpJ2oDX7+RNThRAnhiQ7xNeGPHgHowggL5BgsqhkiG9w0BDAoB
                AqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBDYU4dscAnvUzG7FUBYdA2c
                AgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBAwsNB30zk+oAIl92c/+T0cBIICgA3tpZTk
                auJrxGyE7auf2lrpB1nCtHGHsxHpm0TOj+7ovX8/TZcR/j/JTQrAs85/MHRmZu9MZ/PNRTMpgcP/
                9qtZQ451OaRtiamanzKXklZdCQYEKsgT18PlLDLtmGJu7vlDJqwwCRTl9hix7seVXpM7f3zeh2eA
                z941NN7MREVEtGgvTP0SL9w6S8V3KhU0h35roH81A8dh7TLjisAMKkfMePLg7AzGX++npX5Q59zm
                Pf+eZCk7oqhGpZNCruLXgF363h94MoTweifZjtZp2kBD/V3258Ix3f6Z+SiGvTqxyhSYiVUAFHDP
                zmLsj/lJw8Rwk7GcGjgbeF+hrkvvl95mtNCl29ocbKn+eJjmkF2g5YGUtdk+JZd0vZ6f1DhvpO0b
                ONq96Do5ltTUM0YmyFUilAJx+vYXtG1t2iXjN+ASGfYGI81jbksK8HPB7Bg80ILREl7D0EsKgB1o
                cUfPT7FqW56EJQpTyaDarKSkHXMNu7g1SQCe4X55qOJPpXgJb6mNaa1Kf2f8y1n0qtd6FYNHBpcF
                3dFxcDV/WLQRqpgIRk8EB1nEoT+yEJfMfQn/KCMM2IfW7+yrsfS/KUSKnN5B6JUBO2+OzlpJDQ2u
                rB9oKFXN4Jx/x8wJHgmQRf6R111w7166mddBgJfwKs/1hphyoCefr9nuemyP0RE0ukU5r+aOv2mR
                CS2DmPpPT6VufeeruhxsL2+fcuVfF7hjX5HUbMhbgIyiDnzIrkEZxU0qqJSwbLSzHrmLqWuSIkXI
                AKbQXVBhZoIxnMIafNVJ+SXfaqyl8O/xn8hi1XcZbNs46cSOfbtGdzfqxwmPfwRewVaXN4UisOeM
                xMA0jalkD8wwggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3
                DQEFDDAjBBDJYnZTe7lgNIfACZNNAJX2AgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBDN
                DMvGFo4NtfB+zBihcVVXBIICgP0SMgHhWsdkjRny48Ha7ji8ysQ1KZKU6umNfjMZ5A78Ic8Zsk0K
                AXHHtX1zDwFOkx+/laRpYlNH8lMYyo4elf8oBfsXzvYfF9ROz08L0EqHIL0gWm4XTShkHGyrB5RW
                XurVNRKIy6lA/OAB2lf1rFGxEAgOGvoROc3pIJGH/KQa9XRUhD/NK1U3CmhoF5GcHOUoxsW5zmxT
                CHR8gZIG+U3W7kF9H87ZFuM9VNk9zbQr9pBKzPWr8JxrA7MikjTZli+rIyOgeuuMeE+pDw7d2pS7
                /rbnUY/X2l4LvR3vjaGCTxctx/0inpyY5gmN1DSybq1sypTz0QpqFUJY9kAKI+7BA+KsCmWPjFpo
                HrUfjQ2vqmbcSFyKItGBMEXT5tJb9+rL5HNvFmxTt8LCfa7/0KAONLCe6+iaY/rLwA5a71RC7WKp
                oYyedIMkZi6Bv097gXJa+H9l7VPVJCsO71PPKWHZEbm/F4xXiaf7s91/abt2kHGGq5IfFx6mtuJE
                0+RqoyZwBL2tOob+BBUTW8p9/nBmGkTjY9Eqt9Kp8tZrPcFOwrD2PqSN93n8KmFDE9/B5Aq3ejis
                ctNf+aq6gAw7/QmvM6VFIZJbq3b38xax4W/e+uUb4xgpi4hsj/w0xh7yxBjYv8HtW8+mBJzUpArj
                BoRsnyn/SHJHpnEOM+nCi+p2/qF1lvHDCNfoW+/6utwX95ic1ogMeH3TCvGUmGf+uc3L79OC5HlD
                yCkA/yCd7YD65T8EmjZhwa2RPx4DTErQ1jJPkJPhM+k5kSxcXUQjF2A7jo6Av6FGFESmu1N22fTH
                2yK9osG4vbvhKYEBtYAzi9GFZPqwIh2uG6YTdhbNfKAwggL5BgsqhkiG9w0BDAoBAqCCAugwggLk
                MF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBA8SVWRj2wPS4KfI+loUA3XAgE9MAwGCCqG
                SIb3DQILBQAwHQYJYIZIAWUDBAEqBBBr5TnL/rG4n7kGGunNQWOWBIICgIu2UuhmU1gAQGAXvgcJ
                zPa8FhjQE1OoGfh5oySL2uqn6VH4PTc87/j/p+yGwFVFe5hfe0YwqhSiDrLEXYMNb8PisHILlQzW
                fDHCwLSYX6oUPLGjUOnuf9uGv9/WwzxjHVPt0lKawYNLTTSZ2blURC4TxIFSBLauUVEAsfQRHghf
                vcNcMQr1kUlIsRS1mTc7qiOsJTG9I2HAxEakp0+O6uxRS27QHxRHthFnXV5ebfpvTuuFIUOBUl+i
                lBr7p6xsTJ7waFUrMFIFoFFZv1iJwzW90ldW2KEzuR1duv3WWUrlHrsmXKA3wT9bHNF4bhLHseXg
                lyw5rVF5+/2DicQWHYa2e7oRsw0NtMbZLscW5NhMh1BShFMP85k2LFY9D9K47UxXX0oPQ2D4bMSR
                YuiYTERKgvCFgjZSIW9FKttkpJdL3MixJHSrNo8pdVcR/CZMc30d/JKp/BPT4DNmpL2K3iTYnYL4
                LckgbBcKGa4y3wkyQxfZNwcD0fIk75s/snSuYrZOgbeHznfVprrf9T9/Zo3nDYBGcMGg+p3lwB9F
                +/cCtJLGdxIusy+ZVrqwuWvPRRG71mvh1DHbkBF9k60ms2ihSc1eJjmwC/HETa443k57RBYoLUqb
                flPpxRVUIjJu6Foyz1TO2d8b/RMKivhJp4CjFMYnX5iPYy4prICTS9vDFfuGukmlF3AReoD93WYl
                4O+tT4X5bnJTAGe/uh1I1WmdnZwQ01+O/xqHMFKIFmJcvKN7wyHyydPkSdoVKf8+h6xZamm9WptR
                oTcExFwUxR9JUqKjCwcomFIZkaqh7ZSP3FExj6tbPur4h1v52DwCHVK0cDotkn3di4qR6PYX0kMw
                ggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBCd
                Z1J37v5rAgCtcJsxya59AgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBCYWX5/999vcvvu
                FZE5J2JWBIICgOtuuzHGjQYlFZ0O5cG8Xytr7DuLa/wJBtwRTFMpwNVsRAjp0L/vZkG+Fi6QpQDd
                4dXqItfgHPdOjmvNbA2UPbvuYQ4kMKfCP1ELxTbDVunTWfu4agPfjKvJ25eGR+Sn8ny+sEluE6n1
                MPu92QKuTGKvBTwoXIY3NJwA2V6Hg3EEGH10jHAn1xvvboHgOvZ7XYqE+ZSVlL+g+L6Yjkk3UyrF
                NXaCnqWP3rn6IV48KIj3d/iR6Q7RFkNmzhF2prmJuhlGz56kpoIbdVUvRGmPccSq8iMALMtZolaN
                JFD3zH7e0Af4xkOqEaw3Y91g/u3NxdaGBVnOecPqyVv7uMBw3k+v6xRn7gNpS9USR6cAgrvm+fo7
                KHVy/M8c0WrsyS1dHCYohe1OPONbKZfVdoABu9qFVoj/O2rawMtKSMyVlGLSPNrTFmz0K2ka40w6
                7xnvoRGHcHtRVSRnJXqPfpfgZlZzdcddbe0hQd7nmx2XPshAFr11Oxv295Ew+6RJYtSouyq10vY+
                a7qWlAsN0NVufk1Ych8ltxe9b/EpnjfPh1vKh9g9DjrhIbSIjGvH3NxAT7I87BfbWb+NqiEJd9Lf
                89nSM0ynXCq5ALYesO/s+OIFJ1npLEr0cE+yyejX+Io/CvMoV1W5loYF5k8FiHdKxCCRabP/biLF
                +oeaOCyPWlDNIpYywqRtZca4C/WDYxS/ukQmrerH6UR4GQ3+oOBQpYglworiu6dZ081vmxzRhE5E
                5vJRD1ymIo6GvocOA0i9MyC2NlTf3M0b/byn43n1LrEkrHZgVb0ZqCF3WkIjnsosfkNwmbtICKD4
                vykU+kaBy58/lrqw95+/4QEhjN8cU14wggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3
                DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBASEnWW6rKuQyZM2kxWjR1YAgE9MAwGCCqGSIb3DQILBQAw
                HQYJYIZIAWUDBAEqBBAGLCV0XI3bVBD6CZSK/AfBBIICgJsaPg9gul3iekO8dPhzuiaC2h65zsxb
                RPYjMF+gAGn4MzjCTqZcy6pscUOa/efe4KsvRFw0rpmHWW0CN9bXqnim0h5TtWkivwlzJBNjlZ1+
                Yrnj90sj+rpASQXJNdS1+61zBrRpRFwC/WJYsNj5NLh4la6hDipYET9dQY2pNDwUhULTfjmNTkZB
                Hp2Ju2Pxh/8p1+usfCpcxpEQ+byWswibPr17ggfYtmel1fxDHKPlfgqwfU3bK2t43hl3QfVz4Y9u
                ceU4VM0owGyyMbEXMjb8D6sd/at/XrASRFH3aieJQbA87ffj281/NYn1k1OVt6ZqvpOkyX0tu2nk
                4CrgY97gn+CMgZrEskvrXHqD8lA8y4NGv/YUGjRwoqgepZETnt7FxGZXAVdrvukxExAtFJSnmaV0
                VEzlkfYzzDFOqGJDkBLCLTrc5yJ6BBTwIlFzYuxSb2YE88hinhoKuMdJZFQ0KXVI0iddL5xoWEpl
                aa7BI4B2v/r7n2uo8RtHRBhjhBQ5TmGsNog5yJt0Y2XStfEgD75po8jS4/SjTPaxuMlsBdiK9ZQ8
                kjEiIDqx1V+zGNjsgQC4wSbApCoo8OYiwbeP1HE7Mi9s/XXWwyJDA7uZxN8So0F2+yniyH6rYAnk
                50vJGEZX2QoTw99kz0dSvBhgxACzP4fq5olI3700f2wvHojlG2qkaaGHAeqpLKD65613hOCkgJfT
                nVHPcONbbU4ECi60003krVSazYNPJghR+7PdPKl0wajVzdiYvGBa6mbgN1gkG0jqA26yeP60oUR+
                YWXjcTCpAnQHoksnuyvESjKdZZJRhFt3HgyFwtcXl2dJ/AU0Mn343guo8ZlAUCAwggL5BgsqhkiG
                9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBCMQWIRFEXSbPX/
                jRCzhjGwAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBCHQ5CNps6DXhgk/tC17MpGBIIC
                gAERI1Hgiu8K+df+O7pKcw3oaWWdIVECK5mLSIDMEFg/bPvcwTYtQeC49VhOtFVlYGlAfDmGsISX
                f0PRGGOPz3Cc2obMhA8UtPJkt/LV0Mzkyf4+apaW9DSfQFI9dEZzdVGym7oxjPoHatc7nCuAcFN+
                GMP9SMMSXx7swabccBmfDCRMBrsp2F3RuYsd2w5Cvq8BrrnczXZIFgUlztM2UxOJT35cGjIHnNEP
                fpZYftLjbB+uLuFfMfd3zzqwdNtwGfCQ15YAaXF88/klN5gWr+8btcOlRNfsEFv8gvZ9mjm8AmmK
                biGVPPaFvDQG5WlRu1XdK2LnRG1hfzIZ7fO2w/lVJIUj6a+pzzzT+lCSZILztOcfAJOar6qYYN5F
                10fNISl8/AZ8YdiwEOEwS+Z/AKoYe0XdFAhs6YMC+3aqtFTM/523bjCIqrjM9S9khS8yL0oZXCBK
                du+YN2tZbslVpWc80R4n+JZXAMWjp+0KKW5tg0VdURBs6ftqpPCk7ZohMD8TGmh8PRG3Z2hVohL3
                aXTLfE0HrPzjYjwJHYuD/kR4UARp7B4D2en2RLmsUFu/dG+/G8kZ0yDRgCYqx4tfMjfIShDCzKk7
                u9ccX/HOyWNc4VYlILTkYepIPU1qezHTBbZvXlcZIUwqKoT8Joz2uJ3Dsx1f/9tXSTjGOIFHYq8A
                QFtp/edOC3gwX4W/Ep4F2DKlmh80rrAy3a59wxsZzPINXWRCk5u8i1DaGDVW0jaJt9uRrv3GJPFD
                ZzO+EI7lK8C/aH9Vj7JE6isz3JjW5ml39vk4BQyUi9h3Y86ZjUMqKtY+bPm6gge4ET0qmAU2VBIw
                nj9l3Cejjdp7E2NjZOswggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAG
                CSqGSIb3DQEFDDAjBBDwOs7VEjNL2wOWOEByveYiAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUD
                BAEqBBANcxniDlBevXtVGETyVpalBIICgAzqfhOnh6utR+5JQa+onB5DQWQCwC9xgE728LO4zrIg
                NGGBtA1Yv+tvMIzAmwppYyyUeJebANC8xICh1R/Jw1zVxildGsDe+CcIqCSV9XzwLiORQU9uq8El
                8hOSKp+jxQU41NhK70CqUCT2icOjDikUGzDGxXsY8VnKZyddWb10Mrnyafqp5PORkC3KL1cAsfnN
                2hJEVbVFwAezurxYShMsPnhrCUzq+RBMxds7KWn6+kcgkKN4tjbqHOCGv8ufljrl3I44LfVb8tsB
                ymKdYnqYV0q2Bn72DYWAHYzn791J5sCWcPBPd1VMmYi7U+ERkmmY8fyfCF7tNSS3jjj6r++FJ5DS
                nsfVjmiuZrFNstZkINL814fgt59qGy0LpdWCNkiR00Ntv6CQG7S5qaHxTOEmbs4E3H53x+QO/FCa
                7UtI01zSV8FmBtZDgOmXTkAX8mgNqfCEqMs9ezyZ8P/5K9KnspmVVZs4hpdmimp2Cnuf7ASD3TN7
                iNGgk/RNu6F1tniyhUqPIiMLe+DZDnNewmPaf55JaD5KTHUt4V13M790HvBtjFA39KfuZF0YsKsl
                CdRW4h4/r9FO7gOflakBrqlOfhcaz/SdV9+nO9nIe/IhMe73r6dcOsd2tKX1dqpQaD6FDQE5/eR5
                gcFsUxoCAy0XWTg1H+Y+nspAyMjIAQ5ai9ymLsclIoeA2upO3nlY3xSMF1fngOngTJcSCPTklVkE
                SKvIoebknFgBTpBgSwYf5752UmrnyylqXdajPjG5tfOuRSe2evoEJAsLy0WZzuanJ6AJizXP4Aqc
                wHXpI+ZTW4SXCekbiGOK/69D2DO0VamzljGRsu+v6oiaEgpsIjYwggL5BgsqhkiG9w0BDAoBAqCC
                AugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBB2L1pvvp0GM+oyvQaukcrRAgE9
                MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBC1LDhdsGRaHR0kgO37xDl8BIICgPuaW4/Gr5d3
                ZHHkh9VKu8ETi5zscFLAtTcFpb4MJ2c3UnKQG+RjFMezp0svRSW84//X0pvzlfoKxng8VoNBCUwR
                N/aJsEqC5UnBqiHWKL2wkiVl4gziRUZu+eosgm/9Gk1xd1Qxs7nwyTdMYfgnvfXof+MCs8IjSfB9
                //RJLPqVRu9mK1ZHXY7sb1iiXxoVjPhOvDF2L2uUt47ADSQSNp1iYhAA7g++mXiSDBTRa3Vy83cd
                Tvq0bVx4NHgKUXss+b9kA1cM23DqQaqpouqDlXxTW2mC9490rCUQLyRTCj4xfrdgPypntFfBHuQJ
                Uj891OEHiHzTnptw2anXjk8DfSkdAy3q38MxHCSxHKy/m8FgUUBkGGoL7wb7MD+vV6Mi1lyziNCD
                mCPKIwaJe3+JEhdCO7GVZfVz/lsfhSZBHHkomfXe0LFwGFbvRxSyb9xiAF8NpIkFyazTc0odsaZ7
                JsfO4QWkVudzbTM1ByFCNetvZJPtfy2jZhSwXrKnF/tTEBFh5VA6UOkGEpvKmUXW+Kkisg8rZ/95
                G186Nc7BC7Y7OIlNG7ZBeVwjOUi9yD3YawQ/w4OC0TxKM+CtcJuepV2fCu6oMRueT7wxR4k/sl/k
                XZr3AREf6s/JAtUJ4DtGcHnlOKuEnSxrwrAgChjxIjkf6IjSpgp9cUHg5NpTRAk9isnkYYlHK2bf
                xPoiJ9Z7hosT56xRuFf9RRvNLjw0IiVKY/SnZILvMK5Vj/IAONgeDTXMGPaxn6/JCDghD0Hq4i/C
                lciZ54XoyrbJLUd78TSsf8OA1RyNpoAFl00NidyjaNjvJfHtlWE+6nNXt3pQ7CpxPA4bKH3rnJ+V
                qm+41t8wggL5BgsqhkiG9w0BDAoBAqCCAugwggLkMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEF
                DDAjBBD4bw4y5yha/aU2REAUVi1HAgE9MAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBBSBk3R
                ONt1SqljlDVYuUrlBIICgM1rgQ9V7mZSMAk8fCCpVgWmsRIsAIBkqrFfInr4sDQgg6XZSXBySVXD
                Jx6cKY4UR6z4731n9Up+7FLA+wC74W+fyPjWTuJPZcR0YGtbj1N3RKf9mOuNxqir/mamf8EMW0ix
                zX53l3DWn26U8F5nwE2DsJh894I9/BSCRNwMpJF4d4G4nf52oAe04/iD21/1a4TShmnlt4+wwccX
                9em8Z71ZUE2FAQyTOQVBburLKFLCxGHZnFlZ7ZL9APuXPuvcOXwq3aa/K7Eq+fqliVqZORnWrfo2
                ORKzU3rNkobgFiyJCNDV+YRswoIdqjI+P/RlQJvJUBko7CJEWgRzTwi2PkYbxrmYxdBK4EV6O2Ws
                DBmmY8zDfOqDMyzUn1Z56kZZqyg2O0fRFWBB+ouleLaX2QqbXfBEkjZYDcLfT7X7iaqWSMQK+S4u
                1BhkjoJuoHF3zP7BYLomcNBINVz7ZpP5cPLYwhKnC167cywc6OBOQ5PwGbSWcbM/HIvSxZvX/Djc
                iinyo9BaxJXXfjX2yW2uDovMATlcNiQQO/jjCuTcfQXtLarQNDBjztsuhTkErEiUpKU+GD1cuqzV
                Nlf47d/Nz++9xix9A9NPbCw4LZ2ZWPbelC2cfnok+o4kiwFg/iLul/rwIa291OwCW5WsAH8XbGtg
                FjYsfGZ7WwnAw5T/Qkmxh0kWPMSsWWbpHJsxT+RzRbRbDWx/0ky3AjHbr75KQqCIOw06vHxBrf/U
                wPLUVabMq7+InwPmKu7aHn+lQVulbLF0WbctOhgftsKnh8etipi1Fvbxn1hhqcv5ZRk8wHf0ZKEe
                zK6jiI9f/ZOwmYk/nWxdcdmywxIP3+mgJAB+YaowgZYwTzALBglghkgBZQMEAgMEQAnxDpUZgkDr
                nJ5wJc7Dkwj4LYOne2DnHpz/51y7BYiWx/Ib2Nq07pRd6KcOv5bCLtRg1LtASYZwGVwnrK3Agx8E
                QEjnyQlMNis7fiLOs+ZthpmtwptebOMdcoB5JdmK4Q1QsnkVetK04cL1hh87unT4vAbwn02BSg47
                mSdzI1U6Td0CASo=");

        internal static readonly byte[] Pkcs12NoPassword2048RoundsHex =
            ("308209CF0201033082098506092A864886F70D010701A0820976048209723082" +
            "096E308203E206092A864886F70D010706A08203D3308203CF020100308203C8" +
            "06092A864886F70D010701305706092A864886F70D01050D304A302906092A86" +
            "4886F70D01050C301C040887A8C4D7DB154CC402020800300C06082A864886F7" +
            "0D02090500301D060960864801650304012A041028529AA9F7079F5031C52A23" +
            "8A23CD4780820360ED47A5097F8A6C7B6EDD1989CD19C67B8A59DBF6CD22F7B8" +
            "36A4B8D9BAAD54666FA9F022CDF366BC1B4A47604C4A0B67C09F2DBB8F75D185" +
            "B62DF310817E24FE71A47246831C58670A27A5693CAFC52EB730B7BC0CF00145" +
            "A9E4D6784D3A49C524546415EA6448D991C5C83FACAA4C1FECF198DA5B8CA9B1" +
            "09346B723238359CC18D1CC28B185AA3860B082086F9FD909F3BA3D94C9238BB" +
            "CF8AF18C0A56E6A88E7E0F4F15675770C812F805604743B1E8E683FFEB3FA4C9" +
            "9D2C6B00A858050DCFE12C0DB3471B678657C7E28C074277165D714FC4A6E193" +
            "C118C676FD2B2F97B36D8EDAFC3332609F192EF98D3ACAF1EDBCB03A2DBB9B0C" +
            "8AEF748A5BF66F1EDD9C7C8A81638D644F9ADC99DDD6BC7FA7FDEDFF044D47C1" +
            "6F62280D80526D05F17D5DBB44209A3F27CF88B881227D5685DB6BAA22914D0C" +
            "E0B01D9997F9A429B4294940BA75FBB08C7C7EEFCD77B7BAD0DBF3E418F4B551" +
            "ACAEC537DDA914DD0BE933181A86B433640C1E10609D96D093544D119727D1F8" +
            "49060303189EA33B8FF8950493909B0963EA4BC8999BE78D2D2F1315F96B24C1" +
            "DE6A925C0DE3C862788B2AFC56BF343E63E1274D0B8652C6C49033E1B21AA283" +
            "0AB2A1406291D215B799D8CB5CB6CA0D3B0458C51EBFB01207E96DA04FDE7C24" +
            "F32E48A0B4E48EDE150B13D8B2B1D3B736BC4BA54F800DA9F980F7CB9F23B7DF" +
            "8C343F980ADB0752E2978D4C1069337A631E15D945D8EB858FA6D1548C230269" +
            "21D0413F13FD4C94A1BB8897040AC3CBF13DF42FFCA7D0CF76FF08C68893E45A" +
            "061700E4BC3B005FDF4E8A04323B98D07ED037E4A341E49E43F2D2BABB0CE6B9" +
            "4D260BF720A7B864DFB12B22D28F4992C91A16E30346252191EB60A06C23BB6E" +
            "1FEAABFFCCD57C0A2516E8015E9B329C97D99704E653EBE04BBF26199B11A2B7" +
            "E30FCBCD2C57A1389B0747E1B9896C6CED2C86D721DEE424120E0347A05B57F9" +
            "19FA1FA334512C1EA43205E5B1A56BD1F490C5EC3CFEA17CF1E8EC4686DD6150" +
            "0C420D03E6E807491B24ECB6E64F210BC8029399C153E62705E80FF358E1A705" +
            "DE3EC20DD3AB6651EB51E7F7F2729ECB9BEF94B9E1A685555DC075453072AA06" +
            "5C8C821033DB91D1DE64CB085E09721580FB01AEE2C105E62177D88E25B4EA54" +
            "4605636CA3261107DF7CB8FB2661CBA0B855726F8F3F087B95EB257B523B007B" +
            "45B9C71E030FBDCF3082058406092A864886F70D010701A08205750482057130" +
            "82056D30820569060B2A864886F70D010C0A0102A08205313082052D30570609" +
            "2A864886F70D01050D304A302906092A864886F70D01050C301C04083D689F19" +
            "C613307C02020800300C06082A864886F70D02090500301D0609608648016503" +
            "04012A04107F45F9CB7E9F7A54479C6B260310BF0E048204D0C37C420A86E386" +
            "22F3EA79EFDA069F9A0542207ACC6EC2B7D497D06C2F0E20FC362F637C032D86" +
            "72B9A0A393F0DE872B19668D37C35CBA4DA6407D80A1A61998BEDBB01713D41D" +
            "741C5D4A03B877B2CB6F66295B3C7E3955AEE9F8BA250086355CA8D340850825" +
            "28F3544F9DFFD19D764DA15E6D178E7D8D843E0E8FB0FBF5CD9D6203644FD6D7" +
            "88754AF0372BEC11938406A27D76EFEC6974F92982F2F2D212256DF3122CE631" +
            "7C079B14D9B14D8E8AE31EBFB76754DB6D5497DC3F7DEDB73DC2E65F722BB3DE" +
            "D1D217628DDC42EFD646501B8B8E085996D87F11E3B29B85FBAD0AADF71A391D" +
            "92F11DA38286F7C57D12F3923D334D3EE186098B88A77869514DE08FF1D09675" +
            "4D6FB2978CDFCE051F7A0AF5E27BC8B0DDFD9230FB292EDF4F8893BA7C1CF8C1" +
            "B91B527E13B93D0B32E7A130D829D928281655FF93A60220194777EA4903B577" +
            "EE7D12FD585D69ECBE52192BC9A0731444122C13ACD45926DFE13E7F151BBC71" +
            "2787DE41A1319B974DF20F0E28A2B8C247553C6A98B52F545723ED67F36DB626" +
            "7BEDBE8198AEADF2AE0B177F3F2B61A53AD5EBD99F10BBF8F578AEBC780EF766" +
            "F512118263A7775BBAB8D1E7D2B8E55892A59BA4E4DA74C63A90319589096AED" +
            "516AC29F6DD666BE867C6E411F0A63D8EBF0A4F8F591D8E16F106DE0A63F001D" +
            "373B0A72B3BFBFE27B1463F8A01CF5A4BC81988DCEE84A76ACFF3F27BB205708" +
            "848F0555686E0A650280EFC7517BB433742961DE1E0B08D43EC64BBB6B2847A0" +
            "01FAA219D470E553A8BA05479DDB4B3C0FE50BE45C7126084AD249B521733477" +
            "473B687BBB75D47C3AEF41C72BF7612FB4ED8D9EF51CBEAD26279AE07C3A8151" +
            "7E65FC8A77712D8EEE8994F1632157C3F1403DB0FD2A95C0A89D323BD9205AF0" +
            "E228DABEA7EEA182B8D257BEF9D704CA0E9FAB1027F49BCB8D75C2969447EC4D" +
            "59EE74B762092664AB68DB5390C8E76AD91399DA5473222DF32C96D519FDB21E" +
            "3D2EB1C6D98B3A3525D1E92504E29A0DFCE59B0B982D0A8EB845A8BA4DF22CEF" +
            "CEEED25773E51C5C1FC0F96A697E24995342FF82D7573479D342B7E41AD2FBEC" +
            "BB6F728F8D3D964C09F9FD39239B58AA91269DEA87755C6CAF9F6F876A1EBCBF" +
            "9983DF2D6BBFF81D180435B65DE95C7BD41062F123B7CCF31897CC7028F6DDED" +
            "EA747F069AECA55637B9848674AB1CC4A61944C274B500E37924F7E394DCCECB" +
            "93052BCA45AD66851E9CDD4194470BC2F91667C6E1F696B89A219C45CDAF1752" +
            "CC8D95875EF343E90765CA366CDFC55F6D72D2590FBC47BFFA9F45BAD5C4D50B" +
            "A89F117F20A934D84C972A15AC4DE8F7D105B1E0AE214BFCA16BC44C4D4A096E" +
            "699057ECB1E3D9D6933CC8AB9D9E93A5194BB39FE289E6FC121A564FC049C99D" +
            "6B6BFE5B11E8292CEF1E4E80453F805D5BA49BB26EBD84847F1CF238335005FE" +
            "2552A972DF84618E59201798DC1FAB60188DCBF18C6B30CEC921D40B76B9BBD7" +
            "458819427D0444A2421D69CE1E23D335EB6C327F8145DD3FF25EE740F7896A24" +
            "AEAB3E65FA9209C18BCE8A1D28EEF053611170E3B1CB43F583D6084AD40C0035" +
            "E053CC6133E67B18F12D508BA43B79E1B2D1EA0CC2B91002BFA2D286DB6CDEB3" +
            "796449B33D6DA4E47CD326B8C123228E68D822DF1B8A50520EAB8A2319227415" +
            "4F4EA2C604A2274AB881D6B4EC36E0C3F5104A0BBE51E061E4F09A425F797BA7" +
            "545D45CB4567A39D923125302306092A864886F70D0109153116041444BBC698" +
            "3FD4DB3AD40E15D39784227F2A2E60A330413031300D06096086480165030402" +
            "010500042053B736831F5398BCA4DD020714A2E9B46FA9BE273F81DB5A13D27F" +
            "482282DE4B04089ECC39D621BAA9ED02020800").HexToByteArray();

        internal static readonly byte[] Pkcs12NoPassword600KPlusOneRoundsHex =
            ("308209D20201033082098706092A864886F70D010701A0820978048209743082" +
            "0970308203E306092A864886F70D010706A08203D4308203D0020100308203C9" +
            "06092A864886F70D010701305806092A864886F70D01050D304B302A06092A86" +
            "4886F70D01050C301D04082BF1679DE99B425702030927C1300C06082A864886" +
            "F70D02090500301D060960864801650304012A0410E60AE5FBE308F1D6E6820E" +
            "18F85D357380820360E334B918A3971F37CFCEAAEFC215BD0773462CEF830E5F" +
            "73D28190E78FA023A2EB377952BEE5927A0FBAF841C96059E4B0111880DCD6CE" +
            "619B4B07DCE651642F29AE978AFD15EABE55D5E2AC94439294F15F4E4A304D88" +
            "54B04555E30D251D7BD6B53F0584C5293E78F9427861EBF90BA0E41C00A4210D" +
            "5208DD27C45E5C964988F00206399AA256D19103E391765F0E4E89319AA8FDAA" +
            "F79856ECF1E6F5DDC9629E778446A0DB25FCF541D4815E66967124BB751ADF2F" +
            "10DD658BCD76CD7854561E161FDF4300F84C5E04B91B3FE52436BBE2D7DFA2D1" +
            "5D9C6E853D4F221F833FF85A985DA2010050FDC17B6013DF16B71B1FE181401B" +
            "447A7CD78ADB894F310119600EBD79872CEE05254F141FF4AE317B4B19089943" +
            "1FF692246DDC9D042C517FBD389EE2AC9A8DDD7214BBC010FD7A90A325DE4224" +
            "D833A55B8118C5BBB003304956C98A4CC1A91EEF29BBE6DCC4F5353EE53065B4" +
            "2B0011E291B81C9A8D43CF436F2A47DEB40C21547CA2E416A61740E9DC5107A0" +
            "1913612DB720770788894EC2531BE2676F3CB9AE5C73D5AF35261F41D474A565" +
            "0DCCE0A9A63EA6A4BE80E0726EF4C1A5805DC8888330D2973402BD0BB4017BA9" +
            "D83355DB97A9EDD04D727A1A3FF796267E56494E2A89B0705EEF88805BB23D98" +
            "05207982C2B43CD3FB83E8C65EAAB0DA8E5FB85D9821CACA20BD21311BF0E3EC" +
            "4F309375D7AC89E9B32B1B2EFB51D45FD5F88AA066A05A775311857A192B7DFB" +
            "200FF7F276C284655E30164543A53B3463BF6F8B037452544BA86764B4ACE00D" +
            "F2C26B18D9F457153585432201F578F74BFE9A4F8010C3B4973C865CE175485A" +
            "3EEE47538C73EDF5E3DB34D814B6525351D03648948561760CCE3F4DFE423ACA" +
            "78E1F82C78476528BD35C8D02C17AD41E9796F048BFC4CCC15FE11139C1A104E" +
            "469368C654FEFE69139014ED6B30E6CAB30DC7FCF1490BC91C50F1716CF348FA" +
            "7EED577F7AE25219163BC1BF378271F9E676CB99227BBAE64FF8D9BBA2FBA69C" +
            "6A045CD70675809F9C3EC1DD30A4542C36E4B5B52F982E8369300C7D036144E8" +
            "004804F186F2F54385BBD5ADE939F2CEABA8D785DF62D54348CA76D94669C350" +
            "07BE6C9A9A9CB263A9A098157B39C6B8E1368D9D7F9D9524D5CD3B9FA1EFEB9D" +
            "44AFA24F2536C3A70E19DF07B7B0A68385D8572753A251648534B953717CAEBE" +
            "19EABF8E36F039A2503082058506092A864886F70D010701A082057604820572" +
            "3082056E3082056A060B2A864886F70D010C0A0102A08205323082052E305806" +
            "092A864886F70D01050D304B302A06092A864886F70D01050C301D04088064D4" +
            "32AC75560902030927C1300C06082A864886F70D02090500301D060960864801" +
            "650304012A041028618CDEA80BF226578A2DA0DEFB48F9048204D070F86CDE3E" +
            "25FB88321EB1D0B9E8BA8736C1954E7EC2AD3BB3B4DBFAF6ECE268FF693008CA" +
            "DF785473709F6ECE9AA8F93019C5EEF6E557DD5587C3B3A85B77A3FB1871F58B" +
            "0316B4A4210BC2856E5B4AF46B3BFD114AD8FEC7195CAA4D8FF578EC0A945A63" +
            "1000981A56525F8AFC6F140F5DEE2193C7DE62579890ACAAF056A32CC6E7CCFE" +
            "2839DAF60FD881C0D34BC450DD7F0F9378A695F27F547A73044444163670294E" +
            "578FEAF5627865F297147ECBD16E45C128697063D16FF4A773ECFE2978124C06" +
            "A7728C6495CC35C957A635F213D237C8D64B425ECEE868CD1B901CCBC3AADC1D" +
            "CDFBB433195EDA5CBC6C11D048D7DB7C5A0443828483DADA0C63FC4FB3234A3A" +
            "6F749428F371121389BE52DD5F9532B8CB86E096ABAEF67865ED7A6D9AA0AFB3" +
            "98361237819188AE5592D65B450A9F99191D1E4007FA4C7067E55B495A224435" +
            "E6FBB891D5CC84D929166B807C8114E26D6125C7B19B0ECAF55676F61127A75B" +
            "56740D00BA582A4412301E3220EE58FC1A55B578EFB90F398B9940CC23581A29" +
            "804244A0C37D4661113DF6B43FD77CE00B0E9C4AACB28292FB20DCB12FA461B1" +
            "07E853C057867C5ADC6C8D21A812B05A20ADD09F6E6ABEFDC45DD1CF7C64D7AA" +
            "0408CA24D1B7014F2C960E8ADD356FD13B75448287677B23F48F7B527380709D" +
            "4ABB09A5EB3100068F55CC4A194166BE75F32C52F3031F69EECD36A5400C0769" +
            "E5C5DFAB81F1480C916872169EC34B49646C9AAC76909EB8B49CFB9ADF07ACB7" +
            "5B051AFF623955F1CB0D5E11154576497306D92CF1A19C02D0B10D219C956DBB" +
            "47DA7F24C554D0AA6D71C5DB4E966606B863445F39DD955AA062DF78BCD0C509" +
            "000BA379A4BA1FEAB5A9D3EB5FF40B3C85C9450B1E06F79A2670DE7EA799B741" +
            "28B67A2B1062391F78B00958497948F8F1133096EEE10E761080CC55928D1F13" +
            "AF835DC7098181CA2D27DEEC292C40CB969AF15BC8F82E31A224BB24F04BDDD1" +
            "49C57688FC1867E6CA995A4D8B2D72FEA76ADB9E36775EC6994EC346150B26F7" +
            "40491EA804C176D9C8852308AF4849A88F2D1C93E948C848D22B0122CFB5C7C7" +
            "E848DE044480DCBB6E282285BD00CC8445E6992190B13764337CD226613F2CF5" +
            "EB5750A7FBE079B68FD3950E8D3CAC9E579BAC61AA7D52F9F4194C4DA59F07BE" +
            "40EAB8F3104D0C70E92AA0273DD3C0CE1901BC9780B5762C833974D193539627" +
            "61AD01E3BF23C6C7941372CD042BB05A897E18519C497236DF21ADBFD5507414" +
            "E50725AAE16A5078464719FA61C426BA4DF9B42EC2D84824F2505E5745446BF4" +
            "CDD8444FEC1E505665C46339054D8CFD6A54CA9C1275C96BBFAF456BEB77491F" +
            "71D2D002B2F7805D44DFDAE37C0C4FAD680A7B0DB89AD006208FE01284050AE3" +
            "3A21CF566FD2627B1A86A1C2402658083B08998D37234B00CAD3C1F91A54CDA8" +
            "57A777BCE878E9902297E4A8128AA2D7D58AD5F491717F9B3C10775F55783641" +
            "D052BBF2B69B15D5CB78488731C091E5B7D3EF768D8CA1F2089F30ACBE014566" +
            "F566A69CDC09268F2D25B9E9E60FF08742EDE0E85735FB4D4FAD8464B163542F" +
            "CE3BBEA4F6DC2869F48B8212013226BDC8B520F74349E85244C7DE0D84F9C4AC" +
            "DF8B7C5929267A2CF2AAB77FA7C46BC6A983FC70991002EDCBCDC647ECF289E1" +
            "BA930216325D5919290260BA192E8F3E1EF136D4BA18DABF7F2A6F87701EEF98" +
            "B66CEF645CA90BF8B75AEC3125302306092A864886F70D010915311604143809" +
            "23C6F251CF50C47C7670DEE626FD1708A70E30423031300D0609608648016503" +
            "04020105000420AD0EB570ACFB8357A8E99B17672353CFBA69C76FFE5B6BC113" +
            "05577F12AE24040408D04E60444B79672302030927C1").HexToByteArray();

        internal static readonly byte[] Pkcs12NoPassword100MRounds = Convert.FromBase64String(
            /* [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Self-signed cert created specifically for inclusion in public-facing unit tests.")] */
            "MIIDygIBAzCCA4QGCSqGSIb3DQEHAaCCA3UEggNxMIIDbTCCA2kGCSqGSIb3DQEHBqCCA1owggNW" +
            "AgEAMIIDTwYJKoZIhvcNAQcBMF4GCSqGSIb3DQEFDTBRMDAGCSqGSIb3DQEFDDAjBBCNparJkj/3" +
            "Uk8N7n0KCMeQAgEBMAwGCCqGSIb3DQILBQAwHQYJYIZIAWUDBAEqBBAcqpBrSDFcXYAWVWKcsEi9" +
            "gIIC4P/ANdPYWI1vBH1U5sZGMIwLjY96pYaBelyZd0ZfKA8QfGHVNP9+E9hplBKGvRfIMiqmFutj" +
            "RO4v7Ls8HZEk0hwBt9+6zXPWDJLxBDfSMHUd08+ZAH1yzEqq8aBMyIRVHOQkJFuFuCQJ9Ke5HzVi" +
            "39S1rgHpnKYFvy+xZAhgI9OO1YxuFt4P9nhlEV/JCoyEQ/2iY99kKc3z7ArrV7BBFhfYGKhWQCBu" +
            "kAmNBKweRldNWgDuW21WJEl5sByOmyDwpiK55Zxy1K1aIY8DYJTtIzzcX4CILaj6tClMH1G9w4jW" +
            "BkQI2CG4vCsMl/28BbIP9EyH2C+gBAxvc1N32y3NSvO0/GPVenmQFF9KBMc4FVy4Z21syMKzUkBi" +
            "PtIbDkcQbGAfyPgFk4SXCgn8OpIIvOOGI50/r+Hj14qex9VIrlwAAWCH8Y+YjwqFAQJYHQpb47zp" +
            "B1fTwJFOrsXrBgLUzJLZKLR43yW2E9u6b8RsTuFHjh985naCHLuWPYOXS1zduBpHKpwoPUyCwD2r" +
            "DAokCvA7RCsSXroUkpJarN4CAqsEB8COnzV1Dl2xcAYMerJxrTCKX6WIQUYo0/qeCoqTT38lDAlE" +
            "7Ydjyx12iVM6eWejAdjORvlVtCQQtCxz8fZpdFGbMP8rf35A8hu++e4u0CLHnhTx38zPIm6H6YfN" +
            "qj5h1Kz0xLzqnRfa7EGfDEERSHOy/DqNY4nUNG2DTjGOHy1QJelToG7Vo2L7CCZV+leX0nwLNExf" +
            "hKEp+uQCiYSJe9iDm9fS9VymED79OJbr2bxdq3MggEGksLZv/H0ZT8Wsue0vq9jQ6J6YIEM+DKYr" +
            "Zt2l4WgTBEKbpqmRvOqYRh9O8Sp+3IRNPzMC2ehzlYXqoPbtG4vxpoRsAMCM/W2x61jbsBSaNSFA" +
            "eaUwcnKswRg30UonHUAIOJkqtadI57WE/Rat5eHVyya9f7ZN8bTFZjx0BQs6Bo8PK9yfqoidSN8w" +
            "PTAfMAcGBSsOAwIaBBTt8zpgzygINykjoAwr2GKEywYFwgQUA+L1vfCVASwiE++gTfRgIScMGycC" +
            "BAX14QA=");

        internal static readonly byte[] Pkcs12OpenSslOneCertDefaultEmptyPassword =
            ("308209CF0201033082098506092A864886F70D010701A0820976048209723082" +
            "096E308203E206092A864886F70D010706A08203D3308203CF020100308203C8" +
            "06092A864886F70D010701305706092A864886F70D01050D304A302906092A86" +
            "4886F70D01050C301C04088FC6C10979E6D39802020800300C06082A864886F7" +
            "0D02090500301D060960864801650304012A04109EF63F4518B8635FE54FA2D5" +
            "2EC0C19880820360B9DEB42B1F8B8F4E58C9CB35F871A99821775275D604A14B" +
            "498689CCFA1D8855C1BCB7C07528145C3D77903BCC27E9133041782018D6C1DE" +
            "F094328772836DB17E5271F0A0B7D8A9BAAFA1086E95483DA5CE8519B4B911ED" +
            "88FF0FC47AEB5924180D8B35083C5B3B44D2C8B7DBB2ECF5FAC710862B090D5F" +
            "801932A0E56286DF0CAA10D7CB8243B88D7C2F619D06B233DF6D16060D392F28" +
            "2C564ED685ED8227619694ED34050B76C8E28ED38B69C0AF9672060545C81B99" +
            "EB6463BA8A786422322180C31D64FCEA249DB78C51CA5B84DBD8522A8AF531EC" +
            "930A0A03BFA5CC7F84EE9F98819784D8E22237D524369F02D620B5AAD9C336C4" +
            "737285305DC53A8C383453E0D278450CAE3DB2807DE4B0D5C6BAE4BE40370ACC" +
            "5CB8D03644E553B0E98FDAAAAC00FDFDD102CF8ED916E22E686F30B575660B4B" +
            "3C5BE7A1D3CC1D5217A8614F3BC80727671B13E5776988F7784A7E886C23C733" +
            "A0C6A633D7FE8D7EC9ED9B2706B40373940BA8F9BD1BB7F8CD700384D06EC809" +
            "21EE29DC45CD7D6EBFA1BBDECFAAA26AFDACB9E8C86450EF404BD3B60716E2E5" +
            "1E943E1E1E595113C0EE3687493F252110ADCD4A44562BB3E2A6C682500C6591" +
            "F72542BF6953BDB94CA977925B636DCEA6A6BA634CE8E4A44387CD20566DD120" +
            "E3096E99ABF83D88D12F6CC4117E422460414DD0F89982A080FBCA7669568865" +
            "0E9A07087EAC27753AA5E0AB126093CC9391B7510541820C6DBFF07B790F6DD9" +
            "699530BCF5286C9537A9024C0778BBB854EA5A9034B293A78F4FBD2F3C572538" +
            "2C703E8CCDFC7C54CB28005E4C511F8C8EEE84E65415CD46680547A7B7BB72BE" +
            "CE5432EB1C6EFEF308092C35667BEB7509594F75A6D06D35C5C12FBF1C331E59" +
            "6A5BC1E4DA753E319254068B7EC7824BD8389E5379773D14BF664D1A75991CD3" +
            "E4AD815CC9A131FA66765530E55362983D4AC316D055BE68E8052101EB36980F" +
            "2E72F5A3442BC1302F1E9A721C7A8D7B56A6DB36477712FAB04F1C17F073F8B5" +
            "279B441E35BBC7986E67C1D9C8AB6DB30AC045ADDEBACC29BE1AB6FC57EA41C0" +
            "5F7CC21BD1AACC1D7B9F2B5155893CC11D512A6DFA5FCE72BEA259D381B57019" +
            "6ABE1C186B1B445B1B0B03DA596C3A5AC0252D026CB3F5DE42C06E81AA7A7029" +
            "F249EFDA7581862AA29FE1F4E8ECD9F1BFB673119ADB0D369A28A2E3962BC265" +
            "3BB1453BBDDE09CF3082058406092A864886F70D010701A08205750482057130" +
            "82056D30820569060B2A864886F70D010C0A0102A08205313082052D30570609" +
            "2A864886F70D01050D304A302906092A864886F70D01050C301C0408CB7AC3CE" +
            "BA6EA01002020800300C06082A864886F70D02090500301D0609608648016503" +
            "04012A04100E253ECAD79E0856D73A2CE9A3D48101048204D0D617F9EFF71270" +
            "B09B3287BD6014E3DE6C8AD23BA19156ADE125B0324CA30B758DB8EEA9689A25" +
            "1586E6E50A0C3A5DD975A2BAA6B571FF1C9C78E361F0ADBF4E9B4F7BB57B2728" +
            "621E3B113C5ED339ABB89AD31F803AE2C976C1F0B96ADD7D2FD54E27C490D3AE" +
            "220202531960B57B029379188C3613D6B1AD6F03526E1BB23AEC273C8F6E017C" +
            "01358356EEB8D1273B5B114129DBAD60E367D70C92B65D38D0EB56130864F708" +
            "3A43B7E3118362D21CD032D21CE94E6266809A69087313767E0D9BE736333AB2" +
            "9D737E313302C6BEC5124023A22C395EF4A0AD165D344B329A09E5E4593F4848" +
            "323FAE51B283EF0DBDFB951BDE62A0C7FC2E6419A412E86680F6414E154DA560" +
            "2B4B46B836AABB3DEA03B1FB12110DCC6BA1C0FFE902CA892E4827955E1488B7" +
            "AACA0FA0EBE8B8ADDF425CD9AA65B407503D0089D74B809310C737923386B86D" +
            "DA357654C4A60285C31BF9C2BA22862D7774EE2FD8AA92008F678CBA8260F04A" +
            "561081FA8818CC384307AD0AB4C3BFF86D1BEECC136D0595CC6E816B51C0888D" +
            "EB4251FDFA5C02FC6D9EE0C60114406FB7391A149756497A2D88207075E2A528" +
            "E021B68820F68F89F4F6A5709A18D440E09B5BF239D4ED36E8C3358DC7890033" +
            "D9A0761A0A062734B709144A5B1CE9296480E66E0800B62FAA881F70AB89823C" +
            "D513BAC42346758EB3EC982B7F8B4808DCCEB64CF854A18FB0E2D0D3B29FDF63" +
            "B60C21A614621183F643BA56BBD0713AA5444EABD1415FF9CC2ACA80B70256CA" +
            "CB059949760257128AFCAD51E8B9F574DCEA9BBA42344BACA4D96F7EFAA5AF64" +
            "EF01119D34E971D4837C3633C202FA47892C59B37A88310F9CD2C523F0A4F952" +
            "BF925A80F2DF0A813283739416DA1885EB3DE5AF04FE6C828C1AEEACBBDA7B82" +
            "A00F3505A56C88FBA0537B1B4F7B7F54C1551A5F5279DEB9DFCA2C2EEEBF1D75" +
            "F60DE8777703B7AB67411E5DD4478E7430E34CF1D0E0FBA7AFB124D40FA3E04E" +
            "5D737D9C1CFF0F8B1CE7E2D4F99C340BB4360FCC285610A44247021BF1CD338A" +
            "DBD4C9CFE0C1685AED1A4EE01CD34A9BD446F0E69F16EB395A6AF8E4710A13A2" +
            "75D2B102AE38DC0204A6331155E70A0B73440759FCB89CC5E0A2012082696A16" +
            "7B97FD90FFE53349D9883FACECC2B7237B28468C6637ABDC47EB595B82662703" +
            "C9A8BB3AD8B7DEFFA323EBA092C8BEC5CCE3C5BDD10AABAA872A7620B4888351" +
            "EDAC82D0E7DB5585C5B8119E300F8CF8B24C62EB397B9069038AFD69FBF1305D" +
            "83FF499E203873F45559868DCFD813A5FB5BF4EC695A7463C6B0A2A8487B09FE" +
            "B58BDC0BBBAA377909DC88F03FC3E058D80E22E54840E8803BCB78440C44E6E0" +
            "1A17519F78C2C89D965C9B984C96FCFE5DB22ABE51241F88A15BF0DF20936072" +
            "EBCDFA465182F41A0025E877A308045D09FF988F122AE2639EC14601ACA00E59" +
            "81C3E1B6C8123B8EBB2CF9936D9CF3C4A0A8A694E26E9FF6EED0B0544BEA6F08" +
            "EEF5FFA27DC0FA3D169F51C42DCBC65C3BA935F6EEF2D78DD352206C0F5E0F92" +
            "AB8A4284D67088C0587443242035C23786BD0A3C48452ECB44F4E07422DDC849" +
            "099587763991C299C5BFE8BD14D6904516B31254C1176F191962B81FB5BF16AF" +
            "B0E25980A0D069D9FABB11A053FA5D8CC71F7F825A2C96A87D7A6F21FC6D8DA5" +
            "059F848F6AB5B85A04D1257218EE480444148F4B0A1BEA8CB4622A96EB9B4EE2" +
            "A76F14B1CE25EDCFA23125302306092A864886F70D01091531160414B524C1AA" +
            "CC976B19833DCC9739FEE5B240AD3A0030413031300D06096086480165030402" +
            "010500042029A4ACD27A78AA6BF3B9114E19D1315E76BD450EDFDFE88F4E5E98" +
            "2DDBFC734B0408D154718FB757EF5102020800").HexToByteArray();

        internal static readonly byte[] Pkcs12OpenSslOneCertDefaultNoMac =
            ("308209180201033082091106092A864886F70D010701A0820902048208FE3082" +
            "08FA3082036E06092A864886F70D010701A082035F0482035B30820357308203" +
            "53060B2A864886F70D010C0A0103A082031B30820317060A2A864886F70D0109" +
            "1601A082030704820303308202FF308201E7A00302010202146543F3FA4FA3B8" +
            "75A1BFDDB3CC23ECBFEA736A00300D06092A864886F70D01010B0500300F310D" +
            "300B06035504030C0474657374301E170D3233303432303137323530385A170D" +
            "3234303431393137323530385A300F310D300B06035504030C04746573743082" +
            "0122300D06092A864886F70D01010105000382010F003082010A0282010100D6" +
            "C77692CB15D98788AF8C00AA0C626AE45015C8300364C0303F09A1D48DFF5599" +
            "A3B8DEC49365E28274E2E33288ECE042322CF8B2697E760D0FB4E8894C209859" +
            "9D7BCB39DE74596638D11C87F8245EB245AA0FFDDABC45F804158776B6A1980F" +
            "13F4C87F2BDFADEFDA91C3628544BF5F328E5F1215B5F277F6C7B89413720B22" +
            "9FA9DFD132A7CC187F88154D6C091841461D2E7C35F75C1A40BB99EDC3C3873D" +
            "51C2F68F804F58C0A4FCBA8001F6B92825581B4A9391C6A770614F6BD82B39AF" +
            "13094EE7E49DCAB7FA2C751533B358C874148BAB88B915FC99BB8C8F907A4317" +
            "3FA14F3B26EF4FE5B014D107DF73E5A2FC0E9ED651CB376A78CB1177E5D83B02" +
            "03010001A3533051301D0603551D0E0416041479B60678BEFC1E3784E8A780B2" +
            "30A7E826DFD627301F0603551D2304183016801479B60678BEFC1E3784E8A780" +
            "B230A7E826DFD627300F0603551D130101FF040530030101FF300D06092A8648" +
            "86F70D01010B050003820101000ED62D8DCEF486F04C6047AB0C2CB25FDF1923" +
            "7E46D1AD9C16139B8C0A0236EDB46582AE45EDF2E242947A81AEE2AF4BA9AA9B" +
            "3297FDD76B1C7F235288802DA590213B4A5E7986B828E072665D2855860B86F8" +
            "F625F2BF6111938C54563DBE8AAD21DA2A883F901CC2D7CD08DB113F3468384C" +
            "388A6B07E4C301839D4F0E8BD38920FEEA4DCAC3ED817BE1D0F3BBAC45E772C5" +
            "41B0819DA4F0EBAFB952C082A57714758394C07416EAB0B53ACBEFD2F2E2138E" +
            "B91CE02F4F8E008C976CDBED10340B79B9F9FF8CC94C65EAC274C4A08E1347B0" +
            "21268E581CDF1730AE9DF3B959508262DED10B040A1D8346F0F544A2BBBA7454" +
            "C2B2A7218B21B64D82B8C04A0F3125302306092A864886F70D01091531160414" +
            "D222A8E9C8CF876416F9A2132202326C1AE63C203082058406092A864886F70D" +
            "010701A0820575048205713082056D30820569060B2A864886F70D010C0A0102" +
            "A08205313082052D305706092A864886F70D01050D304A302906092A864886F7" +
            "0D01050C301C0408E0AF915D08BBD72002020800300C06082A864886F70D0209" +
            "0500301D060960864801650304012A04100CA536131D2DDA95F84830C4E47E84" +
            "4A048204D0075FAF61AAE642CAE276A7AED857DEB08094880EA949F763F54DA3" +
            "B1C0CED0D383B3B305F5D4C033B33BCFA08385F58BE862048F739B59C30AB3C2" +
            "1B8C877D2056D27CD1A015AB8EAF86725CB5F2892C667CC02DE0AB215D4859F4" +
            "A4DFF4C6B8187B4DB5DD2BE8217482148D0135E1B6B7784E7C8515FF4155D662" +
            "5E742D45D3DDDA357BB7FAA90601773C96468F2E023242CD6B087CD0C504C124" +
            "8D747BF0465F33EA2904682678CFE57B4FC9DD8162230F7C6980B399D793B796" +
            "08668B7FEF17430E128AADFAF37F7610924845327A631F51B4AFDBA3D08C2408" +
            "8C6059FE0495003E210F53E616CAAD669954A7E517DFF203746DA1A6693BD23F" +
            "3DFF096DF050DB68AF7E4213F2A40D03C274C3E1F224B66A3481001C445E99DF" +
            "F5EAE97A9E7C34D035A89DDAAF982794E2BAB630EB61F96399B891F460463052" +
            "CD6176E6561DDC60C9FE476D125758AD3EAF9747818A4F963181928E79FC667D" +
            "396EFFD95CFF25DEAC00E0B1A4092E0F0AF9132116A78C75B3306469F64B536B" +
            "FD6130B756899A7FBC79EB11BB0F124A8BF08AD818923ED9D1F93A79C1C1DEC6" +
            "58B8859794A3C02F9299901F4303CFFBECD354BB77FB5BF63146B8F85C4C0F81" +
            "85A7A6C8A28C51C33340DBB2204AFC70D4D79AD2C35514E4BEB413D519A12288" +
            "29E7195A730A1A65FF9585E32F91F232DA23BDEB2DFDBEA5377AB9E053198899" +
            "B140F816979D53F738601782C7CC24E881CF337F5CF50C24EBE2ADAD74FCD523" +
            "2E620B8F144252BEE0B0FCC08AB239526B1DBE7F6C3D1FC8184F60B8B30C64F9" +
            "B42F5B674C475A4D088C81C8D9102118273C6DB632CF7E127E037D6B317DE6FA" +
            "C7B2752F1173851E3C099F40F207B0EF57BF80C73C224B70D0D975C15A8A2193" +
            "2E34EF1113C48EB8258968B37D2358216E6302192C6FB6A68B325D7B731CDABC" +
            "4D1FE7A4EA25AF60F5B224FB8BC0A3DB8ABED63FC1B91FB43AA4ACCDB32B46F8" +
            "1441CBE62B1C38A4FC1E76E2419579890A10906044E18CC1D06A1866EAE98B78" +
            "2BC5503BE5E24638C79FD804809F69D8019B77F498A36CEAF1582DAE245CE949" +
            "BE493751FDD232A02AEF72DAF9B527AA29BF8C0B0384DB29BECE1A1E3B81E681" +
            "00C047A22E73ADD5ED5B19BD922279BF8E0D4834B69408B5A9844B8421D6420B" +
            "3BA243BEFBC2AFA74DA94D34B58BD827D59025438EF4E982E0BFA133F5EDB181" +
            "FB5E3723CAF8BF0AFDDF20DF75A433850DFBBDAAB3FA7B61962A66A0AEEC1476" +
            "B12A016CCC2B12C16AA8547D46B33F3B6008A65C165125BAD61345D9777E40DF" +
            "232190FB47934E3B935B963DDE4717F4F4D1E932D1D7020226C0A2472327E16A" +
            "F8E16DCEF37C52E58A65FA44842E5A323DD768322DD0F22BA1CDF9E54A3C3ABB" +
            "7010A9E4C3D856719CD025C1EC7AE2DBCB8DE8ED68AC542797978C47310BB3B4" +
            "D34A4244B3D0D01338A2EF8A43DF1180C01BA03AB9A382C888912BBBE541F8A8" +
            "7C0CDE2EC2405F67AC22049410E2AFD9A8E4B8A836055C399A5422DB375A7520" +
            "99A947C933DEFFC26532F399D08978443C6D040FD29C5842B6E0A8235D44012C" +
            "6783AA66ACA31FCCA155AD6429BEDCABA71398B287AFFCE96AC8ECC36CD222C6" +
            "7140E144656A0D8FCBDDBF5E96390DBB7C7C093EEBD8DFF0BDB68568BFE77D7A" +
            "A7A3B6C94CD9737B8FF3C4F9684062E175AE9EB8DAB5B8956EAFB3A5EC84C9C7" +
            "376F5F4B588E60875F3654A15899FD3734455366083125302306092A864886F7" +
            "0D01091531160414D222A8E9C8CF876416F9A2132202326C1AE63C20").HexToByteArray();

        internal static readonly byte[] Pkcs12WindowsDotnetExportEmptyPassword =
            ("30820A06020103308209C206092A864886F70D010701A08209B3048209AF3082" +
            "09AB3082060C06092A864886F70D010701A08205FD048205F9308205F5308205" +
            "F1060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
            "010C0103300E0408865AFCE404C9F4C7020207D0048204D89F814ACB055451B7" +
            "7FF858162304D57C3B3393283C3DE8F81B4A2264501B47665EDF7A830B064DA5" +
            "DBCBB95A1DE88E07E50681AA732A800FC006617317B0D461450EB0A8CAF73F7A" +
            "26B96510A47BE05034301B33954B6DC2E581D00369DACC98C189FE2ABCDEF057" +
            "3E5A04D0199EC35F0D401CF314F714D6720220315E13ACE3B982258A94642407" +
            "88BF9899E4B631B577293E8C8BA288D459DD6B158AB1C8C585D8ECAD23E3F49A" +
            "99EE4B64027DB0A5E5241EC4F2B7D614A2F213F50CB97BCC8C64E1194CE6103E" +
            "E21CEB1F83C8A8C4D3CB4F0E76CC8482F34AB8F3D42386E061C72C4A9D040076" +
            "09C86331483F0E19A3C94BD2848E708C4B6D07DFF70F839BD7D39A269B4985C2" +
            "025FC6230D08999B5121C71FD9E11E5CDB694C6C5C1EEC5D92E9296A8BA59902" +
            "82666A494BDFA916EE9FDA34EA64A0877868DDB61A5F63E3EFC54DE52629A2E3" +
            "FEB9D302331C4693711AA3B3C8DBEC5E44F4EF28E40C6B2C6F3C6861F270C87A" +
            "210F5BAD2C05E43D2D93D5D349D7A60A5002E11F78E402294E77C946DA9F8F59" +
            "F51E6BFFD1E322D95E3E49CEC25FE338F579B910B70268EAE76763A368125663" +
            "E0B14F84C502A70EDF05D9C4A2257742E2F52C80E01A6A9FFC21E2E231DD94A0" +
            "93D2819AF8261C7A2DB6659CC3FB237B8992FDA7242836A9F549555FDF55208D" +
            "1E5E2B13B18D776A606941C9DE54B560101289141BD864160EB2C382DEC0AF28" +
            "B9037FF80ECBD0189F1A2D22CC395637AF046D62E90220EE6630957422EFAD6F" +
            "2F6FDF425B8B9F713368CCA5D3CE9C12DD6EF61B12074FAFAF87F3E59A42D8FF" +
            "41BC8CAF09B5E79E3AFAD9832DED992F7A8441307379DA4225EBA0ED9E357664" +
            "615FB53B349EB5AB1C2EF450B6338442D938A0D9C52CE38862015F11F1F805C5" +
            "426E4AA33E6FF964A4E7F39F8088DB8A98821E737D1D974A70E3CEEC960B733F" +
            "68A8C1A243F438DD6D4DD25924251242711343E6767359AF38655455D3058110" +
            "73CC04DC656DFDDF7C3E30614D994C315E2CFAF63307C0327753817BDAA08C92" +
            "5307B136DAC0BB5FA128AC7879501D00269E7C7B779DD5624F81C28040A2FE42" +
            "0FED94C6086AB945D4A81F6F9547B73BF03A0F6E8E36FA629B7730A42D574EA1" +
            "D6DD37AD7624AED6A5667500A3CC2CA45333A8E2872B390950EEBAFA7D98C3DC" +
            "D16CD6F932C21D3C302DE8B161B85AC67A524855664A3473A541EAA9336E2FF6" +
            "B4978BCCBE0C4D7D874EC424AA450B5C2802BD1DA7FF07493F947D2F0E202D3D" +
            "6D43F57DA836EAED898B280BB374A23AD10E0F4399F549A76B166F80D0CF4ACA" +
            "AEA53B7A7A13E85162FB48F54A7930D564E99F6345B030106D625FF92EC96E89" +
            "921442A0F248C1681757D1C8638EEA75A1A5A2CA74F4DD8EE1EF114A56E040FD" +
            "6DB11B4C6B542D8294C08B0C65B02DB59D883F52B3A5F0165E37EEA7999BBEF3" +
            "669510A206A533F5A82DC52D4F6B54977D5F8AE493AE49BA10663862CF1B8675" +
            "D42F8DC6BA3D748CFAF4DD75C870D782CC00F4D463EAF06E2BAFED47748225F5" +
            "1756D089EDD71C090CD87225EE9D501848E4682460A23ABE5A4D0589ADA5105B" +
            "14439ECAA937517FCDB35CA6BCEBD1190E71A793979F070B37A9AB2D003A00CD" +
            "38D8EF194207FE53FD95D7C43F7C58718F0CE80588DB13B6828A1984D9DEBF0C" +
            "1072FD2171AF46156A1685CD841830212B7C78CC1A40C9F92DB0017DCE853E1A" +
            "83A9FDF792AD2935750D74298EC8051C3181DF301306092A864886F70D010915" +
            "3106040401000000305B06092A864886F70D010914314E1E4C007B0037003000" +
            "4200410033004500340036002D0033004200410036002D003400360033003000" +
            "2D0039004200380043002D003200410039004200380034003300390041004200" +
            "320032007D306B06092B0601040182371101315E1E5C004D006900630072006F" +
            "0073006F0066007400200045006E00680061006E006300650064002000430072" +
            "007900700074006F0067007200610070006800690063002000500072006F0076" +
            "0069006400650072002000760031002E00303082039706092A864886F70D0107" +
            "06A0820388308203840201003082037D06092A864886F70D010701301C060A2A" +
            "864886F70D010C0103300E0408196DD4C6F976C7AE020207D08082035049FA3E" +
            "9C159024F1821379145C29EFEE5AFEADC5F364CD86547A125BF22CDD99010BD6" +
            "FD3A5CF0F028DC3E37E1DD509F83F2C13F59AFE2D3908B33EDC93488DBC0D091" +
            "6A209039CC4A9B8E37F614FA6E877969796EE55B58699A65A98449F8F0ABA078" +
            "50497075821CEA2BAE375092254407B1CAB99E8AC3AB1F582D13217E651E328A" +
            "DBE6054A3FA4E0A6911B57A5F2651FC772D5A180FDA6846C089C6F0CFF7D3EFD" +
            "22D04551B59F58BB605D9F6B99EF7F027082E200052A19AB9606DCD522386A98" +
            "BF17FB599252052A69440DAC7D57187000D58E8902911A51422DE3BE79001A42" +
            "0EBEE4D263A265D49D41432D6040D4D28734DD38288DD84ED32EBE281C361400" +
            "7DA2D4751F423538EF6608F017C3E2B0A0E143ED066AA65A933D62F5872EC26B" +
            "1C5CF29E264146DD6EC4A2A90DA88FCCC9E0E20F27373CAF09E9CBFB0D057094" +
            "A9FB914BC194941A873D03139FFCA79405300FB46CBEA0070115271268C2196F" +
            "C39DE444E483D6E36A973DC95775AAB3C81F766065A76EDDEA274C38F103B218" +
            "5B57FBBB82F1316A6C023A51BF873DFE71E62259AA2FB6A5C4E4152D45D70284" +
            "E08927B559F1706505277EC088BF91D3ACCC4E541AF1B7701DE69483B1CB64F5" +
            "DFDCF259761E080D5C58ACD1BA8E3E13C52E4346D4743F4D7C5FC9FD1BD64D52" +
            "0539B7E627A33ED2F5840A9D878AF88B32C5D5B8A8279662895A6DD129E8E511" +
            "C5123885514D3E3FFC61C000DABC1648BC816746B3FCD00F967BF8F12B1FB18A" +
            "E87A9972DD2B9D83A69E06FCACC37EF3CB6CEFE37D6EC68C40ABCDCFF9A91716" +
            "6BFFFA54A54016089E4C008B0FCB9EDDBF2F426E6232D04FBA7C41F0BD044D52" +
            "70B4B82D865D75BF8475970459E1BC28E9207189586944D84D1CDBB74EF25773" +
            "9A2F070A9871B1072822A6614732A9D90B36E6B1F8B9DB190E0B65059601AFF2" +
            "D19BF35C1C900E6D575DDBC1DA505FFBC19CE98E90741A898A207D254E236A89" +
            "1826B66AB6E1D26AFAE6F59A70477F17D490DD1CD5673FEB65BB4C52CAAB02BC" +
            "ADFDE6E483DB58C24BC3E596E451BB70FB425CD2010861D60CE726F9067659CA" +
            "907134542C069E52AD359E2C300FE19AB08E343614AEBE1B5584D258F1713927" +
            "4FB671EBCBA8B11CD4E35564B518A840D0DE69B54E39F762F3821E5C6EF5C2FE" +
            "9CD9D62EB1031AD5D191B95F2B303B301F300706052B0E03021A04144BB56BD6" +
            "06B405745EC53A2F4AE4598F557ABFF60414C984B699CDE84E8ADC55A7291E5D" +
            "FF09EAAD2394020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs12WindowsWithCertPrivacyPasswordIsOne =
            ("30820A06020103308209C206092A864886F70D010701A08209B3048209AF3082" +
            "09AB3082060C06092A864886F70D010701A08205FD048205F9308205F5308205" +
            "F1060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
            "010C0103300E0408D10B4BC509135A33020207D0048204D8BC69D1EFA1BFF345" +
            "14685D3FD6F4407BA4106DDBAD150E899E2C6BAE0B291797542F07376E797891" +
            "16D8124259D24C3E753781187F3CF49FCF1994554A28E12A2AA76046692D3673" +
            "00A6FA70D09BD1A0299FF481A2060879A4B34FD35ABA4C1EE87AADD95A42CA75" +
            "50433F2BCF46548580AE0FBF839AC5C6BA984C9ED00EE8C164F701985B116C3D" +
            "4771CFBC95B41CE2C61EA951A0B2B8297ED1A10E94F80640E3BC49AB004DD110" +
            "7DB967921976B49AE1DC70AC7F3F1D85A4AD84D911E63BAC9DA9A620EE4CD74C" +
            "512D240DD4937AD6718432C1CC39B00354945F80BC59B3A9C3A2CE80D17E4388" +
            "88BD9A0C6665E1C4D31C676DB4135386704366E1650E457803E784C37658BEBD" +
            "64C23338E6D5416587FD8460E2B04B2F9C884845481B93B6E3A829DA9C364ED8" +
            "FCFE8DDF2053B8F8A6A3978B6F1760758C0124C87F04BF9D411DC3478E0A45CF" +
            "A0BF342255F798D34C059B401B29962DCA862577C43D7BDD41A69BA220FD58E9" +
            "5CD79B0F6FC43730E59E7C8709732B88C68F5BCD7156A75359D7AE38A71AC69A" +
            "605BEC35038DEB4167EB989FA4D1B4ECBD25981C0C4D5D3109CF8C12863DF59E" +
            "904DA26B0B03AF0CCB1D83F07A02338F413B6177DFAFBB4ABEC97FCF7804655C" +
            "0697FAD4182358064A55AAC998B035D559C4496575676413152A38F079568272" +
            "2E80AA090D9D974F8CCB59159F184B448ADCC3FE9E3032D64DFEC9405ED27030" +
            "09A5BA3AA1A6A9D190DE7D068BBBE5F5A9220788B8052A81C1AE4343E1E17FE0" +
            "0B8EB2D4537A97A9726E901FCC8F5B660AE84FF7EF1D2633FCE243DE56053C64" +
            "52B1B0C8E73112B34C4DEDF9ADE21BFF9217B341017945D73B6B2670926AA4C0" +
            "FF1EEE23EE56C8650CF0861A338F645B4859DB37281B4D655BE084D5FEAD1B8D" +
            "15C77FF3EA96CBBD098C4E05F718FA81BBD8D423D491DA5A42C7919D14BC0C66" +
            "D7F2FFF8102F2207D7866B3952A15B3EF96E59B8B0AFE59668AE4BBF8A75C0F4" +
            "39F3096C3B9B172074C37CE8C0335BA90CED4C78A35EAE6BEC26FCEC69AA6470" +
            "518A9131E0BDCE14604CF7C1DFB7C6CBC90C0147EC64417418F877045F0C947E" +
            "16168035DC404E2EBD1C6B9BA0347F011AFAF908FF0A1D8FEB85DE87BA2F8CC5" +
            "BBE29F4AD3FA4F2F16BB282DC801B817DBA5D00A71A83ADCD228DC03DDC477C1" +
            "1693D7C9F179D1ACB1C13F6CFFBFBF7504A31BDF719B35E7A25B116EFAE68F96" +
            "6AF7A9B545DCD3C9FE9FDA171E28E46A5FA749F81FD8D87FD52D4795F4E28554" +
            "3B3C2289264EA9F5B1DBF0797F7795638E2BEA075F3AAF04C119329F43D9788A" +
            "98E22784CE45D3F0659CFF73C55379156A56F75D3CE6BBA7F11F949CD61F9ED6" +
            "9D84ADC7A41D67E64C2FCF532CFE4BA2D31400A8DBE70C4344790FFB9F353245" +
            "8967D45FE4C6BAA588534AB8A43BB1417DE16D7413DB778698BDB75906BA8E22" +
            "7C9221B2B44D358908BBE0C0F2F004BBCA5EA8986169C1EB59B6C8C9C254037F" +
            "D44375DF38AE1AF5BC23485CB6FDCEC1C0CF0B8692B4E0A00EA549D4E4697924" +
            "0CF5A90137D4E12ECC7F1966E6817CB1F06673EA7D51F462289E08B080257281" +
            "30D738CAABFDA23821DCD9CF25452857E3EF237B661338E44A0C8C474944F265" +
            "0FE693A3B677EC99A647CAD574E85605800B1BF305B779DF5ACFE21329E1B4D8" +
            "09B9CC96363574E387403DE8B57A636D731108EF9501075547E412D22475C053" +
            "AC21B74B24ED7AE7C23E1235E93F58D73181DF301306092A864886F70D010915" +
            "3106040401000000305B06092A864886F70D010914314E1E4C007B0037003900" +
            "4100300039004200300036002D0044003100440030002D003400430046003600" +
            "2D0041003800460044002D003800420046003100360043004400320039003700" +
            "340032007D306B06092B0601040182371101315E1E5C004D006900630072006F" +
            "0073006F0066007400200045006E00680061006E006300650064002000430072" +
            "007900700074006F0067007200610070006800690063002000500072006F0076" +
            "0069006400650072002000760031002E00303082039706092A864886F70D0107" +
            "06A0820388308203840201003082037D06092A864886F70D010701301C060A2A" +
            "864886F70D010C0103300E0408CC0493121813F19A020207D08082035041716C" +
            "8BF56E445181DCC9560D5AE27F850E9F87928D642B9C3264C2991F3BC05818FF" +
            "30EF2D88C20F7E8F3453BF925BBCBBDE8273F5EDAA719D979B468DB9A02B38FA" +
            "FB3A852F1CB828B69AD64B92DB29B0275255E0E9E2A1D82208B9388ED4075E7F" +
            "18EFB6FEE20A3F1A440245BBC01C71DEA4AEC041C6BC2B7B108E85C1636F8C98" +
            "2D2F860EF6B87C8C2705C14F4EBD351A36112BB26EF542EB4CD89DAEA112817D" +
            "667B44AC99FBCEFC2164AC9D15A5CE589A61E0AD01F5C4467BD0835E23B41D48" +
            "8122061A2E547CFCF38ACBEAEC0DA94290BB0CD52EFE15F5E0AAA89000E04437" +
            "2B63787DC5C88392258C58C12BA1E63EB6A8D69ACF662BABF63714D99410C419" +
            "D167820182988F5EE810A4C3F76D95057F0772F80E4E6B5F69CC58226DAC747A" +
            "34311F509FC06F8055A83895E08520A88F74AC118C8F5FFA94DD6DDA74FAF779" +
            "2B2A9B105079654A209D981E4C9954B3EC64E45681629DFC08E03B3D5FDF9461" +
            "2A82365F2B959FADC808244ECC80A31C931E0E74DB4AF6864BD7AA90C33EE7E3" +
            "432C41C756479927E1D80E241C59CF7BD5F1C2C3ED3376B2C1595C8C28A30937" +
            "4E092FC99863A02D3C88E55AE43C3DDF83D8729D5CCAD761AF84C16E44F575AA" +
            "67AE08CE9F8BC09C0A67C4C53B964A747C2505BDCF20690D8094F5AD58667CE4" +
            "23BEE3383D9FCCC522F17304EF55B27B2DFF3CC4347537B7FCA30ADC16274EF2" +
            "2DB9BFDECD5F44D55F8246D21EB64BBCA141BC5A8B62F999CBF06553490B6265" +
            "68C3F61B060B0A04AA7A758D5C33155AA453564BA9CABE511C6D4C351A49F914" +
            "3B80877542D9E65DEE2DE4AC2BC7A4997EE657FA4B885E3606D1F84D6977E758" +
            "EFA84F7097A18330A3A222E7EAE434AD8D472127C7457CEC10602C4E0A4F0B72" +
            "1968D8E0019EB5D77EF5B1EFF02D4A6CEE9BB28EDDD72D623FE9295126082631" +
            "BE3A95DB414758B414448CB401EF0763BCCE74C83D93D45C2512977C35DEE3E9" +
            "8A9D7AB96CAA0EFE7DDB8BB7AA77427151B88087FC97579A39D8A2AEF93741A6" +
            "2C02E951ED1C3A172FC366253000AD457CE3194DCAC4DE13297D138A2EB007D3" +
            "667F72FD0CE169799D85B85E474D06A28EBFB4C166911EAD43A70DB60A7FEC30" +
            "4A6931591F13F3D029409548002D873551ABB2873BA3E16411E7E55CB693088C" +
            "2B5DBB5F6861679BD1C2817A73303B301F300706052B0E03021A04145FA3D2F2" +
            "D00A66EA4DFB51D5DE4D316276BCCDC00414FDBF473E8C4EEE8CC4D8DF166E6F" +
            "6409A8F0EEDE020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs12WindowsWithoutCertPrivacyPasswordIsOne =
            ("308209CD0201033082098906092A864886F70D010701A082097A048209763082" +
            "09723082060C06092A864886F70D010701A08205FD048205F9308205F5308205" +
            "F1060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
            "010C0103300E04083EE6D2E1BADA3F12020207D0048204D8B84EA7E2CF71A281" +
            "7828C34AA3005FEE52146EA713B5C3AA7958278B35FDF64AE8B998E3D87EA25B" +
            "54B5319CF4390C812F7EE71D10CD3C70851AD81C690DE0326D0FAD71FF3D5038" +
            "223988DAE31EC0060B6C2C5C9C74B08BE8EE3B78FED2A04AF83D38AD11AE8DF7" +
            "8F7E5E1E447CDDDD0D9DF2AB10C4B8A90AF6E1152B6A7BE2BEDFFB4BCD8FDF3B" +
            "D100114E75760396304BBA8C2AA19C430C9913028B422D5A74AE1429940D9B55" +
            "DB9225604C5C4FD9B162A921E62BBCBC268470B84BC3BBB383A2C9BE7E178C0B" +
            "2593D617D1466B95E57E90E30ADB7E08F6B0753B8B9981C49537B9B80D98BEFF" +
            "350017130F360095156BD5F2F8963572F1B78A9B08B67642BB09FAEE48877259" +
            "3383A830F3EB080210F39F5C28EA7892B8E6D8D6831A1DB33EE2B7A5BE758C65" +
            "8837A6D1CB4383B47FDE181BB1F043C403567EDB66EFD95F585BC8DB248CD0B1" +
            "909763DC9E3F46203C6FC61004962FA6799248E5E7F5B7A131D720F24056BD42" +
            "30D1F5E6827B3025A66DF97436A02D441678797BAEB5135931B02A182C045B16" +
            "9E8F86CB5871FD3DFC3FB573392230B70414A18BC792AC988993AE9AA2DAC5DE" +
            "E09C87F60AA13637EC006ACA2525078E353497E1B242A5FA4BE57C764B24C4B0" +
            "F5956D6067CD150ACEC57D4CAE982E782449588D5184961952A2871EC6FBCEB7" +
            "E38DAED63C897808D7C2D499428E1EE04BDA40A287552493954A45941E684028" +
            "2C964AB94D5A81E833EF28B3B7DF3698CE84595415001DE52ED85A6DB49D4329" +
            "562C73F2AD51118A9E409FF96677C117B0528E725F55433798DB598F2B5B558D" +
            "77F7703D1BFE6C24CBBD1CDC5961FFC45C139FE9855F93BA52A0183FF1C18742" +
            "73CEBA04CDFD3270ADF3E20590E65E019DD6DE4FBD7C62DCEFDEC21103DC1C65" +
            "078B76E9CC1DE6D33C4C6688A14CE938ACFEB25C8CD134C1BE439BFFCF8B9399" +
            "8E8DBD7CFBD86D6CED0E1D1D8712913EABB7DD493209C61C39B757B329130562" +
            "0E38EFC575C2FFC43A86300D5665C0B56DEE483762A623DFDEFF4627BCBD00CD" +
            "3DC181AB22CBD3D602DFD379A223F6AC2134E47507501EEEAF33B06001C45B67" +
            "C6DE14026B88CA0BBD44F941729FD3B9A078D801B98AAEC83A2431C1A7A1EB7B" +
            "99051B4447256B2EC42A4E2917D6E7BE65FAB7866EB797F6ED4A988C57D4DCB9" +
            "8C9236348A45BA717796E7715E734E2B585176100C5FFF37778EF76EE952CCE1" +
            "5540E9DEC3E84E740694BCBE6E4108414319DC8E760852313C86B4EE03B8DF16" +
            "8D00615FE20442C4687400E2436135C938E9167027799ED8890CC08788BD2A28" +
            "41F0F2C1CF8CC2CB2950D3B2C93394017300C714BA941D356F8FA95FB5F82262" +
            "B703F77A48E6EBE95DB2E7325F9248758578735EEE3BC0CC169FDA777A9FB591" +
            "9A0007E0C191668F65714C359A872C14527B1A455305CE275135C7EB9F2E6AFA" +
            "5F7FBB4994D0C2E8BF42EA9BC3581A7D70D3829B7D8204F2BC5AD9D24D7F3A09" +
            "E0A0933D6532ADF03C500E11865C3138834AC110C513239E83606921929853F8" +
            "D87FD51EEF9288178891AA5BC8A496E9BE44D4BFB83A3D435625DF3FF752E7FE" +
            "FB8509D28E3A5E8BD5115FF0BBC2AB54DFB9DC8250478CC7E7A0624060B446A6" +
            "4777501DBE0FF81EAAAC1F108BDD7E122FAE1CE7F326EBC697A016CDD1E364AC" +
            "39FE901DF5D9E7A49225017DEA5374091232A47BE1147753CD60199268C70CA5" +
            "ACFF5023B35FE76C92738BEBB1533FCB3181DF301306092A864886F70D010915" +
            "3106040401000000305B06092A864886F70D010914314E1E4C007B0037003900" +
            "4100300039004200300036002D0044003100440030002D003400430046003600" +
            "2D0041003800460044002D003800420046003100360043004400320039003700" +
            "340032007D306B06092B0601040182371101315E1E5C004D006900630072006F" +
            "0073006F0066007400200045006E00680061006E006300650064002000430072" +
            "007900700074006F0067007200610070006800690063002000500072006F0076" +
            "0069006400650072002000760031002E00303082035E06092A864886F70D0107" +
            "01A082034F0482034B3082034730820343060B2A864886F70D010C0A0103A082" +
            "031B30820317060A2A864886F70D01091601A082030704820303308202FF3082" +
            "01E7A00302010202140A49BFCB1DA91FFFDDECE6FBEA3EE7098A4FE4B6300D06" +
            "092A864886F70D01010B0500300F310D300B06035504030C0474657374301E17" +
            "0D3233303432303137323434315A170D3234303431393137323434315A300F31" +
            "0D300B06035504030C047465737430820122300D06092A864886F70D01010105" +
            "000382010F003082010A0282010100C53F2D0A6732510BD381F04F9EBFA1914E" +
            "197754762CB256EB13B690D5F9A89E27D32FB92F706CB2615285BAECE16C93E1" +
            "B6E2E205818A69EE99C2FF456F6EF117F1B414E43F71C0E78BEA69565683C9B6" +
            "1CBB1247055FB88C758288CD70B22A1FC72E0B0EDE8A8A7B7AD50271B915DB06" +
            "F2F7F6AC149439FA12049CFED26C916A80FA53E8DFCFFD7F2FD4FBF8A70B2BEC" +
            "11342C977636C40EEB88F56B07337B2174EF51AD505B4283B876E98B93E8DEAF" +
            "5CF4415BC828CDF39C34340921E16437BC7E19EEA44B73B26B594CBDA5729601" +
            "C758F3476A221AE043E74975BD90D154643E319F9C253DCA9263A1C2526F7096" +
            "A75C524A24B02F6DB13ECA5C43F9370203010001A3533051301D0603551D0E04" +
            "1604143939D5CBE561A3E16AF98C385C4DE21C7745A7DB301F0603551D230418" +
            "301680143939D5CBE561A3E16AF98C385C4DE21C7745A7DB300F0603551D1301" +
            "01FF040530030101FF300D06092A864886F70D01010B05000382010100B4CA91" +
            "1663AA6EB8B55B8BDEDDD4A07B1B0D0C79A819035BCAC086A01C5DFC64D2749C" +
            "49FD67B20095288E1C54BC3A74FB7384DA4EF82CF5E66114373093DF1729DFF5" +
            "95520975BE15C319B747F2209623245113C39B4A61909A6962A4CDDE832F6EE7" +
            "DE15308DACAC4A6CFB1C0B0DA61AA9FCEE70441DC554F14A170BEFAD2BFDE032" +
            "99C781BC115305E56E0BFF667B518DB0F16FCFA0B730B61EE01CF65EFE499C6A" +
            "F5A67F9E20F7DFCD96A4B86ECA84F3E60AE94931B7A06C238BDE733BFAF131A0" +
            "BD7D69D92AC75A4FA2EE300C1119DD05BF44BA05BDF1923DCAC9E372E5F2C155" +
            "9EADDD2DB07B9BE2A984D2463E12262F058170C60BFC2646713C347D04311530" +
            "1306092A864886F70D0109153106040401000000303B301F300706052B0E0302" +
            "1A0414B7DC97F46A80902C720E8D237F10ED6FCC44ADB90414695EB196470870" +
            "037CD616FBE42CC7469BA795DC020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs12MacosKeychainCreated =
            ("308209B60201033082097D06092A864886F70D010701A082096E0482096A3082" +
            "0966308203BF06092A864886F70D010706A08203B0308203AC020100308203A5" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E0408365868" +
            "CD387F42DB0202080080820378CB4B33E06FC54306174EC0AEB21E7DAB66B368" +
            "4BFD1E6A2A01F894F38EAE1899CA23205BCD50A9525B125AD532084E6BD4AB4A" +
            "5F80AD6FAFADF47AEDB18C9533E527833B972D8F8719141F9982E47AF3E7560E" +
            "C34A3F248393659197480475AFE35DC804DA34C14043C67E0EB969FCEE09A933" +
            "AC1DC8407F667744C395AB4FFCAFBE4E5553204228B32340692C68BEB99C4519" +
            "DDF88D30E8CEF133AAFA77BE355A333C1BC395DFA9581B7F78A59B2089289CF1" +
            "207266F8B237141013CA65EEF7DB2681ADC4FA7E36ABFAB2D424AAF83FED6039" +
            "FB74BAA00D7D743E6DE29184656754EF99E927960717B414EA3390D62E3773E5" +
            "BC82D8B0477ABB2338685951CB6471665E6EF8A7AF869F7A204DB5DAA1C34153" +
            "EE9BEFADDE03082ECC0496EB245FE6DAD783F286AE114BAFC0DE20D435F1833C" +
            "1FEE5021161D327127198D3BF568297C302DC2C37769DE8E7A499AE6CEF5186B" +
            "168544996F5F4159676A2B9B887A335FDEDB14699C03629E204DF1FB5396B00B" +
            "B40E20F8898BFCC46E400E32085C827346FC8E69F463938C9224E16EF2E892AD" +
            "C37488389501A79A35EB573E915D62ADB2897543989D5105594A3CF3C38E0472" +
            "FE8A66FB507498C09FE2BB191935C46A00C9DAAD4FC842DC3218182548D36B0E" +
            "EA4114E5997F7FF6210352FE8FA2B0E5D0871DB8B65AF97F3748BD413FC55DD3" +
            "B522BBA38113801D03577200126F18868314C11A0F2FB1BD5A4C60A7DAE49FBB" +
            "B14A4B256B367289FAA891A1CABD58951531224B5BCDA6B61A17C7033BAB3D80" +
            "E08CC0385C98E7B388E072060030B60796BF56A71B31163AB9827F3E3ED043F4" +
            "F3DA619D8AB27E7887983152AF110EE552B5BC722FF0F021CEC1206A5C6D2B12" +
            "3C5DF369DCC5A1C3C722B3A6767881F5142EE63D0A67866E8ED447884BD4A349" +
            "A6DD2641AAE38624CE400E3D71A6420C0685F22422608FA143921EF661252626" +
            "F54D743214531AD1F8D91B547E0EC24B9B01A97FD7FB2FF8DCBA530C221DE1F2" +
            "0067196F2C1FAFF892E8E49803B33BAA291FC516E3FD5933E5DD0D2016F724EE" +
            "E4E8644D7D93D80C9A6BDA84A10F79F50B52E6F76EA9274BCCE40D09A85F6072" +
            "654645F4FA044719BEE24A04C2380A6596E17F1DF7213AD98019561E21DBC14A" +
            "64D4AF49676113FBBAB9DF2C51E71048B505D1741B325B7AD2D202DC19D541B6" +
            "DEDC7B57279C08EBFF5F588086295CE78840D8504D91577C35E9B49F164DAC31" +
            "90704B2A7E3082059F06092A864886F70D010701A08205900482058C30820588" +
            "30820584060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A8648" +
            "86F70D010C0103300E040814D5FF817FE6B0EE02020800048204C8E933D4E9FE" +
            "25B62F989659649395ABCA2E20D6784D7F60A36074EFF5E6BB6C319D7E595E1A" +
            "F676D24709D8C0A2DB9BB5AD2A0FC0342C77F098C2BA51CB4A1E1B3E26CBC64D" +
            "6655704BAF173EA6C38740A9DEDCCF378D7C95588F3F767A9BFF8B0B6DFE668F" +
            "1475A860F86E782469F21486C001BFF476319C6E5F9FEDFF8BCBFB4F7012A68D" +
            "C85EE96590FC71DD8ABB0C967724564082533DADE23522D73D56707E22211A71" +
            "EDD5F62041A79500840D808C61644B1746865D84A6021C20A0818DC173B16263" +
            "01F462E54115E15726C1C897CEC39466117AE2AB5C15EAAD645F04E59E7F5E4D" +
            "360AC5B4CE7F06C197C3FE699CDAF1B27A197897F97D0FD62D5F9F8FD3CD0D66" +
            "13D01CD2DE930C147A7CFE4787763DC0BD864185F9DD9D50A231079B92ED32A4" +
            "D290B85142FACCFBC2FE1A9823FDB1E4D5AAE57826F39DBEA82E2A8D7BC8D6CD" +
            "B9FC1407BE84447ED48EB17FD3529091FB78398D56CB8C797CFB2574ECDD9732" +
            "023315A77D3BCE8EA9DC29FA97F23A09BD5C546A524BDB3C7F4870AC7CACE935" +
            "F13C1466ADEF7E66A405B28F4A59589149524A429E03854E62297CE26A4DB720" +
            "9BFA48AB52F3646FB5421494C287C30EA2E2311EAF42C561CA144A5C7C592004" +
            "88C8D85E9A89EB36C94CB18A2AC306ADC55197C21106137CBBA023DEB14561C8" +
            "B6C2B789BB542816CE48F98CC68D65B1DFC406E2E99A80816D069EA1BF2117DB" +
            "50D4ACA51B19AFF8669347DCAFB227D827137E633AD37727C352E1CFE184EC5F" +
            "D567B865C63BDDC66D16BDAAB053EDD9210EE167F28D9E885262DD8B142A0143" +
            "1356A942FE56D9B49E343F9459597C02E2EEEC4F59BBAA06BD9D6E9906ACC2AF" +
            "85129F72F78FF97D2FCD0B6A6CC6DD6E2A5A92903B5A35C04DE62259A128CD8E" +
            "A162057FE51667D2AA4D661AE36728A424D359409678956EFF604CB254E0A1C6" +
            "822C1CC349AA77541D101FC5A0D487C33FBBC318E14FB1B13B74F6C41E9C53EA" +
            "ABCFF7FBEF0D370E85A5918BC170AD62BFEFB8B46273901FB9615A808B912183" +
            "25E79F666DCBF60B2FFEA25680E1AF19EDC0768AB6C1BAA67304F2FD25AED2B4" +
            "BBFEB8BD4FABE09AC3B1298B4CAED0772C5B8EDE2063E0A65C424AE2BA344B56" +
            "9E8DAA789B5799480E293EFBD625F8C3482F81C25ED9A44B41DE951D3B0565C7" +
            "4889C29C101B9921952D2D7776212AF33C6F60C55A3BCA457C2045C6F4B0ADE9" +
            "F4B63CE3148F16153C4425373A335EC789BF3EACAB052BA091A75ABDF4E01B7C" +
            "F33805702A53A11E5FFD0752E2B6E14CE2840B94FFE53405EE0EE5C43C19D001" +
            "7147891D4FC5DE5B949A8C602481D8C1F9F8B2CC3BF2271BB12B306452EC55BE" +
            "EE81E13710746EEA3534F106F14DBE9FBD4E00C2FA2A331D2C8FC54607F6A772" +
            "D8EE0B3E349448B7AB4E5CC783E72C3A3AC82AAD33F7CBB24403CE167A14798C" +
            "E23284BC8A3F90BEBECDC4FB142037CBFD0BE595A6DCB2FEB4B80CDD4136374A" +
            "6347898251859BEE1DBF18F3972493474E1DA9B60943D49978A8F4A73470BEB6" +
            "E534D3E74726CD08BB6E9A04594FC8AD7F586F52AF8313D41D4DF45826A300CE" +
            "2E6225CC0FD912F6D56A46A6BE4DB2CFCFB16CBDAEF928F6ABFCC212B12E3A51" +
            "33864B1280EDA8E003F9FDB30A06058B35B31B133D67743C934C9CBF3CBDC06B" +
            "959340EBACF1D7A781AC1318635697C2D27C00EDCF067A6CCBCADEEFD2CEF3C6" +
            "B4E750318182305B06092A864886F70D010914314E1E4C007B00370039004100" +
            "300039004200300036002D0044003100440030002D0034004300460036002D00" +
            "41003800460044002D0038004200460031003600430044003200390037003400" +
            "32007D302306092A864886F70D010915311604143939D5CBE561A3E16AF98C38" +
            "5C4DE21C7745A7DB30303021300906052B0E03021A05000414AABF612B1D42C9" +
            "7B2A777B2B544302FDEE1129EF0408B1FF31BA6E2D1A2D020101").HexToByteArray();

        internal static readonly byte[] Pkcs12BuilderSaltWithMacNullPassword =
            ("308214060201033082136506092A864886F70D010701A0821356048213523082" +
            "134E3082134A06092A864886F70D010706A082133B3082133702010030821330" +
            "06092A864886F70D010701305F06092A864886F70D01050D3052303106092A86" +
            "4886F70D01050C30240410978E6F227D033AAB5C51682EFFBA56570202271030" +
            "0C06082A864886F70D020B0500301D060960864801650304012A0410905BE763" +
            "FCE5DCA6ADF7F8E1261A5237808212C059073968AE0766D615903AA16B242ED1" +
            "765D4776FCBCAD0EBB59DB954E9D0365C5183E758854B80D5CCBEFD131264AF3" +
            "D864832A2C25B8B93DFEB8EF2D4CD204B2ADA80B2B41C4C58E5B48B55EC65DA0" +
            "FB083D61288687A896374B6C851B6251C0E661751F7EDF47478B9041EC7F5B54" +
            "DB9FB6FB8F4FA58CB9F8AF80D7E33B4D0F36880E2D348B5A84BBEBA738B3A10C" +
            "7CF9026327A1D47E695515D2EF2B73ECB46B42DA27917607E3504D7DFA40A000" +
            "CB265E4FD9C9DA026104220E3ED100EBCD434AAE29FF89A883B25111DAE8501F" +
            "68AE219BA7725B7EA2600841F879E9D57664940D74C36FB25E29ED104DFDDF2A" +
            "3F104B6A89A878A85C50133B68C22C00FD9791170287828EE50CDA25EC656C83" +
            "7932D5904090337F092B24853D7F4E27108654C5764F04817D8205F661527326" +
            "9E7DA0B4982108A91E5344B54A0FC07056326483B0D6D445601888D84B22C8CC" +
            "589338DED377ECC841B08B02FA5ED7A1E8841B5BB2E0C5C3CB750EEB49D1BDFE" +
            "D10516924049FDB0FDE9B20768ADF2DA0425AC8B4DE6EF89CD6AE5270513D7D6" +
            "B78B97FBFFDFC2CAEFC428DEDAB3D44C9917AE8F688F7E3EED875A5203821E22" +
            "DBBD45A12E6B3C296F8B37B7782AAE09CE48837E772BAFB0B6063644FF3B3DBD" +
            "861115C3B91C14669E5E92CCA38879F6260D387B6884C368F7246DC74387C404" +
            "7086ABA92BE2D3783106C3B6BBDAFB7F9788E1315751E411A5DCCFCBB58B67E8" +
            "ED207FB88B5E92302B97256C3CCA280D3670A7DEE71639EEFC0546D651F2F400" +
            "A45C8B7643DCD814349CA3099A6CF5384CDAB9A9668FED88C40EC7ABD1E0079A" +
            "F4CFAC8799506D5330A5A97B2988073893A98611327B9806D5B7E0B2BB46FB0E" +
            "A2304A2F6A21357D0A3282A149E6303B8A06D9ABEB0518EE665D05DFD1207CE2" +
            "4AAF471334D61524FEB2BCAF02D09C400E6EF7748BC71DC7F2098E82EB18A367" +
            "1061F2BDFDB063D1823791C182B22AC24E7128287C036CBFD5998B2BA25B1601" +
            "937EF2403AAF3B582662D50558D12F3BFE0C4A4174EE2856171FAAB8DD94A065" +
            "8D5016CD4F6224B06E9E7A179BECA8FA90F9419982CB5799AA024F8CD4C33CB4" +
            "2BD2BAD207D8EE12F739028B04D7015DA37DEDE30A2855FA99A5D33F68460E36" +
            "C7E28CF4C87A13B2E92EA349D394ABE6A9C0BD0A2E9BFBDF8961DAAD2F5BB547" +
            "7E088EEDFE2858E816815594DF0D6DF581E41F8E758089344CBF536FCAAE2859" +
            "2E170F4A11455DB0D3D7D2D5B8E2D951603B9FAF3AA1F76F9CEB0A4988D0EE6C" +
            "F0A6FC537C74F6F1A66C38790BBDEF0DE139DD931F1DA04806474931CE5F5F3B" +
            "6B419D2DE5B390F94BB1EBCEBAD49E56A7333C6803C0B30FE9A354B81112A93A" +
            "57CA90CE91E9FB92163E87A223D02C917A85F7AED5B4074750B79ECD116D86F0" +
            "0C7E053C1D113B3281423F8A3CA78FE67E3DE3E6CD7D6BDAE0E4A69EE3B3EA76" +
            "7EB2A9AB24124AE29B00CFA484B8CF26E6C7228CB95261F96E6EB8A0B0EB153D" +
            "B6BF8B5D1352CEC2C7FA6C4F5DB10AE2023F85C6A6225127E769205B047AF02C" +
            "3F6F18F1B1A590B26F29F1632E47B8231BD40C9BA6AC7140B5493CF3CE58F85B" +
            "0ACD857B8EE850B82522757FC12B142C1C8A9E76F68C38C779593207B146D598" +
            "64C676CDAA4FE83A3FE6B2974B89D4D04CE9E1ED3A95A1402C1242422A6782E4" +
            "EC846091719BAB4A4BA6DF294B3E35764D3E41F2057061354694AB5432B02470" +
            "EF92768C838B8E472B5CD48E4BDCACC08157AE2BAFC50C309C30A9B52E12B221" +
            "961F251E84C8A69342D14B23AAE57E5295EEDB55A2B3AEE0494CD59CB9B392DD" +
            "A1EC7C72282336385561505B1F772BF592C7A3ED6FCE4CCB8249EF83960B841C" +
            "9664ECA0409D1DFDB9D926D331011B3FC906968B364CD0C6D0B8523C24204234" +
            "CAB129D1A1CF17FAFF85E16B215052E886EFE988813DF5AEF7A525798023021D" +
            "D821D867427BB605D7DBB4A522078EC9E3D6E863CB54EFD79B291058EE264945" +
            "6D61B5A93ED32B507F0DDFD60BEB3258BBB8577FAA4763E06F77F135C8C62DE4" +
            "CCB766EF06DC5A7AAA3E8AC42008DEBE1F70901FAF8B6937FD18FFA9B9DD856D" +
            "2136E0814817DE4CC48B129CA09C5A58D7FFCF8C1978C54DFEE521DF01AD683F" +
            "682066B142B86CF8BAF94E42369BFC4A5842D9E68C3CB2DF69AD0595D60D998F" +
            "DD26984428BEEFB4EF36376D4EA6940CA9A286A7AC285F00FD2AE8740E7BA3E2" +
            "1E85CF48F03231CBF7CBA3836A5ACAB14FDDD15F0F90EA21C593FF6E134721E3" +
            "FA2704E4473CDBB58A00F88535D5668F9B30A2C11FF44AAA4715BCF6DBC85676" +
            "345435564F43A10C1D8BD14434C06A1A75FEC14F77A9102337443EAD5494F2C7" +
            "96D8A940F000C78269BC362A08948324057D448966AE0BF05D980DA1C8C8390E" +
            "94C56145A62AA07392006B7D36DB3C0541387E826DA682BBAA5436A5DF631653" +
            "78D3A1DA28E2B525DAA104457C02A9E2CCCAD37537396478C0441B426197767B" +
            "4A37A45D2B31C76DD2DEBB9940D61065F1488DC371B6388BC59054AFEDD80D07" +
            "3BF1DC05F10BCD1A519743258CC9354EC43AE806CE9133EE3CD78E2695E0D7EE" +
            "017D16A839C66011536EE17C4FF635739828BB6617DC1C74E57702C4BD91B362" +
            "2E1DB8D94AD98682B3A07CE713B0C3392081D028E3E28A66DC949513F4CA5028" +
            "6A9169A2CE5B16B541440399BA50A98434D62A0C19FD40CF2D6859DC6673EE85" +
            "60B7BADADAB9C61B7A7A0CE3CFD261C572B8D6C3C4FAE1B426E8B6CEBECF6D5F" +
            "8FB9DF07DA90D0250606B6924363AB365E081C89898C84881965AB4D897FEA21" +
            "EFF53408F637A844618288404F251009BEE40D1FBA009AA5F30489FBE4E51467" +
            "9346544A82001365FE1F45C5332225F2CDCF9859CF286A0D3C41BF20E421CDAE" +
            "96723EBDDD5C7431CA0CADEB4550857DD2828C054F757AC2D62921A7DD6A8FC3" +
            "1EFB1E64440FC33DE646AFAE0050CA5BC2799E24CF0C9493396B0EB011393CC3" +
            "B8C07AF714B3EAD4FD85959177DF63E84CEC40A449EDE23C3670492E9CE28AFA" +
            "4FE4F5AA916E4EA3CCDC2A381EC5675FEB06B541AE428A913CC9EB2C4ED70B04" +
            "CB7B0A03776B313326C7698E6EED937C81653421D500D66B70545671C4D0FF62" +
            "01B6E6318B177C4FA29B408CAA3776D57B4C0424F7DD0350FBFB60CE27FE8A41" +
            "6FB5F61E43DF1538617B993D4229643B7AF9F2F8DB8F5197FBDACBFC8B228019" +
            "2A12AC4053B05CF307DCDEC654A0C795A11BFF7E8386F057FEE687C97A2476D8" +
            "BF1A7850D8A97FCCF8DB5BCA0818813A05F68B84D5EE3583F8D57E5C919F2F86" +
            "800AA4A6A4DDA2A4C6F7028CD9205B7B3A0105CA5394D9B02D74EAE6EC792534" +
            "247201A00C32C1C732F863E6538C810D046025CE81B6E7DE5F721F609A3EDBE0" +
            "FF0EB459309BEAEF029907C7F16CDB809937057BE54D1FB1DB18B289765F6863" +
            "FD398ECECA0524C20709306BE27CC94B3D2C705F7D676BA7CF00DFCFC170F0C0" +
            "2EC263AFE2EFC0E6B1B1E7ADA062B2D6DA2D067833B31BE50120351B3EDF8CBF" +
            "7BAAA08AFE2A2C3914A1A1AC537AAF4061507066426270E98E6CFBB877BC2AA4" +
            "BD3C60EBB630F9788A960BBECE743AC1AE7FCFF28FDE43F7AB14C8216217353A" +
            "E25FCB031C2DF336FB09C22D118FF9751323AC9C82A352447F1E6C7AB1E39547" +
            "868A5CF5CB44C8108298620E8524A04310EB5C4463D3A50E25AF1ACA506FC1AC" +
            "C642C5D5DE0536EC82C8DB9CB17FC612275AD03BF8FF030DD34C3B0B9BC24929" +
            "CABC0249DA09CA034A9280CBC3EF09D4C4EB09924F569728499587A1DDA9B185" +
            "7FD8B5CB61BE96047A4A5A28C8B7EE4F9F976E0A3CF25620EE783F6BA53719E3" +
            "69C3EBF94BC8B229251C34388B01657F0EA45B5FDF1B0EDC25C3F08463ED559A" +
            "392820F79CA86CBC0CA7034BF996E98E4FC0255DEE55EC693BD7E20FA123EEB1" +
            "BD5FF110E8D85348036342A3330573C03E9F3949ACEB03AB5010173871BF1DAB" +
            "AF12D10596AC8E22C48B45CE0E69C6881A4CD268401993631F47A1BEA6C90DF0" +
            "CD26D86D32CAC9307FD864CF44841095F636D38233CA73812E9AB12DFCADF391" +
            "D9D8733A6D039399871B479B734DCE9A358F5FAFCB39AB43215F8A0635EBC0A0" +
            "943EEBA6E7726E57E2F041467FED30D543785B47476DFC0A1BB36C61C726C4B7" +
            "1720428889B9F8CA2A82168469EF7E1EEC52E61B8676FCEBFE49E1DBF13761DA" +
            "A66DD693BA7E311B042E3BB978DFC50F7177244F8AA32CC09234B7F83367967D" +
            "17856EDF91D1D1357B1B4BAC1191BD692AD06BF215CE76B9530B4993E5613D7A" +
            "D7FD354D89FAFF2D397E0AEDC3A21D1DA8965CD2C39CA833D218F1B7BFD02643" +
            "885881BF15C24916ED1979DFFAAD9B9B6F4A6F4D03C84588CAE47AC7EE9CB9C8" +
            "6DCA2DF1D2F92AF46CD5F20E3F7D262C8996FE0CF224FD8D7F848FC2D03E1BFF" +
            "01E23DAC01A2A27575CEACD93DB8B330EC4047B82F303177233BFA8094E8861E" +
            "0637C9ED617CEFCF60938B78354E80F7E405474BDC35548553D8AF005DE611BD" +
            "B52425A99CA1475C63ED881AC909F9B35C83B8572F4A0B51E4E6A66D0381F1D0" +
            "09D71577B22D58314023FC3DAD9E10D95D3E340B77A962B93B13E4F595B4BA82" +
            "FEF8CB6A7E3F92FBF811CC41D75AD49FBAC4EEA9066F2DACF868CDA63851E1E8" +
            "8C939CA85A70D0375D4C9788AFAF31150DEC9AA7401577239C094A8A6F242103" +
            "06FF01F613E50249E7CB850109A8CC7FF8A07B8DDA5EE2FB964779F7EB7FD093" +
            "47CB0065F2A6B473B974BAFC40A3733D13FECF8C1F4DCCE5FD1FE8791A3B0979" +
            "3B84F5A05A80A8B8A7F37197CF4CE48152D052431636863AF42F3240D5B8D4F8" +
            "4BB2027B756B54CE93C0902A2DABE8758D118D5E6DD5F60AA44BEE1150ED7551" +
            "4610402F58FFAB5C2E6FA177E7C250D8409F6917A5C37A7C051FF62A74EB03CC" +
            "74017451A96CF9356E08C75DE4C86AFBF6D2A170BB09251EA914D65FBE1686E3" +
            "82F96A7A21840E7AB172D611704D7E32B0C2F4829C8E7576E9BE3332112868A0" +
            "1479B6C114ED7983FB9F30B44F299F58AE2925587BCD66F5B9F3E221CE1DF9DA" +
            "749D6397854FE5D972805C693FB18943BAA8FBC55B8CF24747217F803F74B2C5" +
            "E7637AC7D576E23F644E3E6CF24113E64E1CCFABC56C4BA104BA69A14A9A69F8" +
            "B6B1DA8AFD9C8F6730F5BADD3F63F557E63B7BF98DC3DA725AF8485047F61559" +
            "03BAD87ED8CDA9EC21BE96B60326F9465119E3F069FA147A25FC11343D531954" +
            "CE81C15B31838B8EE1CD25B11AE4A8A7FFC68DA57A828318BBED76596A550C68" +
            "72982E099C070812E20C099EBF882B86AB722F02F0EFEAE73CD885E6C36F02F8" +
            "1C8DF15AE3C01F0370B02706B0BF4AB292E1488450CE70EDA02292323979415C" +
            "91DD1391FEFD5EF4EA539BBFAF1E361826454A2E6CC32324D6C3A5F5D7D88372" +
            "3258E59706A650F19B5A6A1C2EDACD23EE3CCCB1F09B9910DC3256DA7A851B8E" +
            "A110319F11CCFE7EAD2DAD4D73312A64D6A15D9EBC5DDC912562FB494194B5C7" +
            "72290595529184953D1DB67035CE1DBC6C05E15131E2F4114A6D6A0B54866DD4" +
            "DCFC00475FCC5C44B3494D23A703388314E20EAFF4C4A034EEA686D589A7E128" +
            "CF12599CE46BF7F6E1415CCEB27E3DCCD6715949DED6C4CAC4479F442D4554D0" +
            "D0181F637098C9F72B0751F508394C06682A22BE7D587788D010D24786C5E156" +
            "A07785A78F737D924580CEA11B845548711310D9D29B04364965252CBCC68A99" +
            "B0E8FF248B8F706CC61EB5F0B1142F17FA96E3DC4F43449419ED9C348EBE07AE" +
            "66ED2CF037AB4C7B814C73C9F7E7A0B82BB4CA5D301AEE6D1B82858568EA263D" +
            "3E5536614A7C77F3AECDF941424B09792246A9738AAFED3939EC319569ED4875" +
            "A1B09FDD2167950BF95DAD59F8DCE0FE450E765D0B703FFEB330D032884071B2" +
            "F3DE1FF0F5E43C7F180E3F0B45744802E8113311E58C2776A4484A8643CF2567" +
            "2638C63B40DCCF80F767D26A5E4A830E15397787FA550FFB0D07872EDD38BBCB" +
            "7555C9D7637A503C26E4DA6BB41052C0E1DA2A7B7F33C74EFCC529EFEE43977B" +
            "8373A2210EBDB9A7126AD0A6742D7AAA7A73E9BE546344FB35043209E3C8BFB0" +
            "527AB4456E6D9F873B4C238DA0491D930EA654F71910ACD69973F01713817B0C" +
            "1165F22BDFDA789DBE3D01A3375DEC8C40A0A6EB977B10929CEF89B61274D332" +
            "ABFBDA987CFCF14EA59C3FA5D7138AA8C32E1CCAEC3D229E1B4A46A2A9F02EBE" +
            "5EEAA9D88122BF053F142E42E8A0E1268A19B77CD4B2A45F68B74EE644D3CCBE" +
            "FA586076CA0F8FEDAAC1CA11058E4B9896ED529F613E246F2A4CE29ACE49F0D8" +
            "FF6C04150DD03F526DFB7AE7DD751FA746C378AB0D658BB4A808955C9C8C3B36" +
            "79426C1941F15835D5FD179BAB3D658D4A0462F617DDF5FCD23EF2C899A52D46" +
            "60144C01A542DCBB5C15A8086B6E97262090127E5DB9FB999E45B869833E9641" +
            "A7795DC55B0D7971A0EA423F306475E80561384169333D7D26299100FCD24B3B" +
            "5AD6A215B8323CAD902A6C1B97A19863F4DDF460780D188391FEFDF013EB4693" +
            "D3CFA097C480A10502F21CF5C1C2C45E9CDA63604008EAE8C11112167F539F16" +
            "236A4B50D76E8C143AF38ECB60CE07572E9F31C5517D1CCC20AD644B3361EF25" +
            "84ED7840AB11F901556CEA40435FD9B491EAD820DBC283D4BFBA3B00A8CF2F0B" +
            "99627225326C9C7DF70C89E112CF06C9B9B968E28A148B61BE10B7C8670925EE" +
            "77EBEC504F822614D0B0F3F0CDCA3EA9308197304F300B060960864801650304" +
            "02030440699B89444D5AB2D939E849803579B6C1651E206FB208C67F879AFFD6" +
            "32E080D8D15BDD48651241FF03180ADE528193DB946145CFAA069790AF3405E7" +
            "81BA7445044055FC4F56A8FA8F63539E4AD754A11A1CE1BCF972388D24023EB4" +
            "0E8B8907D86A9F4715D3591307E9DFC832B7756EB214BDADAB09F9E1E61AB184" +
            "3EC3297C0D4302022710").HexToByteArray();

        internal static readonly byte[] Pkcs12Builder3DESCBCWithNullPassword =
            ("3082088E0201033082084A06092A864886F70D010701A082083B048208373082" +
            "08333082082F06092A864886F70D010706A08208203082081C02010030820815" +
            "06092A864886F70D0107013024060A2A864886F70D010C010330160410F19120" +
            "18896CFBB4653C4B7C06E67C7F02022710808207E02D736F0AE772F48FF976AE" +
            "91E8614648678AB5A318CF7D577713881F80B0D6FBF645446359584803814B99" +
            "1A8BFF573CBD1D52AC5237EC7A42ED303474221F91529FDE33DDC2126EB4AAD7" +
            "65CCAC633209A0AD43DCDB5C1D4ECFEA856EFFF607F03D66B29D433F268C7E74" +
            "993F5F2384F9E0D08C6657CC7AD32549CB8D893A1EE37A24E398A44987D23E79" +
            "B3E4F3F48B106FD6193812634F999CDEB1E5E3485D642B4DA850BE1C210F9321" +
            "3BB1E01AB38465C5B23FB80AAA6E1421A2F20BE775F658BB52F57B4CAB79197D" +
            "E65860D656CE8135E235A3C7E06B511F1BE398C03416BC137E063DC8CF8B7100" +
            "628DC62B87A70A282C11BA08A08AB0E03814A9B1940145AC0EC3327396AD6984" +
            "4BE0B4883DAFC5A611804FB628B598DD006ACC039A86B23100B102E1F4B3CCFF" +
            "06FA1E9841DBA8913B9D1D15D6A16330A94F55B04D9779A90EDC91230A664C41" +
            "9AC825594D9F3F39A08074E06E5B671469E3AC9D20D906FA9B99CDAB6696A952" +
            "49F7B3CCBB8EC1BB2714C248AE0C3D0934F949D68995A785F782910BB8457B66" +
            "1B4EADA3724023BE1DE63144C3661055986A2E4D2F59A2DC018EB8E6178E9DAE" +
            "14027352738A4D3579C55F48D0C373C9280689DC509EF602C69EF10566043EF9" +
            "F1A0852A618103B6C2039574696D64C8492963B54C5BCBD4B124FF5F93B04DF2" +
            "F5C1FB577A2BF8D24D29D7C6630071EAB5A3CCBF7DA9B4F7CC39673CDD25ABB8" +
            "9A805838B32A002CEEC02404A78BEC048DE5D8AD470FAAF89F31EA7E42ED1DD1" +
            "20B66246AAB7BA1FEA3DCAF7A78BCF679A3FAD5029318D69EDA8F081FC7216A3" +
            "EB610BA3F6EEFE78660CDB365055110446F73D0FB468D4F4FFF01C394C9FDD20" +
            "47792192AD9A46458E3BB63BA4703A7D8D4E56776D48312C5033497795A396B8" +
            "60115AA3FCEB1E3CAED4FE9211787A8B4B0BC1671A3104DA4A7925A1E3D661B4" +
            "35760FD3DDF71A5902269288512C637C94DD1DB6B8551EE951019294342A50BC" +
            "4310BACBEF7313D82CE7D351FC2E9114BA421FF1EC206BFE1DCA5E772E90BB92" +
            "E7DEB1CE899CD9328C09BF299F7ADFC9D43AE4721FD36BAA93E50EA6B8B6DA5C" +
            "7A188AEC11EAC6EE755D60BA5FE98ED8B8D2360972C114DB4A17CEB5BF8EC1DE" +
            "8EA0D6A4B26F40376B2C5D66657DCE4A2A2F3E519C65035E51981A6A74415AFA" +
            "9BEDB27B2A6B175DAA953919580EB1ABF52D15E2CCC725AB0BE5F09864FBD0E3" +
            "6EB76E8ECCA70C1A850197BA84698A00EE1AE2EB95DD79F002EC2401DF3AE3F8" +
            "1C68FAD724C59E51AB60AB22C60A2E6E0CCE2E9BA5F9E2E503946CFA77CF6713" +
            "E9B98253C42F3A68A6487D2CA4ADF6D0CC8A13962F5569AAE34C3500414CB363" +
            "DDED7F3D3BD5BF6A0E1B229136F23362FD016595AD0F825E30BEEB9E2F95124A" +
            "6033D3BF9CE8758AFD964B10437DEFD03FB28BD46D3153FACF19D6405D260B1B" +
            "257B138D0BDF3AC006CE3C22D1626BE10E8613F6912536D07EBC6142D3EE8A3B" +
            "9D7321D35DB2A78A1C3376894D5C2A1C3E41BD7BD8190007327505E76247C6D7" +
            "3B7E78E098F4504B205D35EC50987B601F76FE6363AC0E2FDC89853CEEF2BFC1" +
            "08E4CD798C20AA7B326A81736B5FDB9365B5272DDF0CB692230A5C69A37DA5A4" +
            "7241213B95C44B9733700A7D4216043860C649E979E9B8C304C1C02E7981A502" +
            "60233E8539A01A361FDE3B83A8B42259FCDAD89FA279CEFBA6F769C562E75E9F" +
            "58FCE65C6D11A7F701161092E78DDBAD9D3FD407DDF675A46348DF53E9EF7CD5" +
            "FB7CE0A3C1B7EA0138529D061CD3CE3A76FB40138ECCD2A29BED34E27E2494D2" +
            "D3787326C11AAE5933BAE7FFA6F5F21D33FED6CDDCDF5CEE3C1D7A8FADCD8EB2" +
            "E77A75D01A50B38E445D91A986F8D1C1100E37A9AA5E83823280236482425DEC" +
            "F49EE6D898E9D8FA3E0F502B50EF968AA14220E926D1ABDD225624E81AB0DACA" +
            "A781857C260B9E712F644A87D5AFE0EB3D73C9999073F3C3835473C3C4716990" +
            "DD57540B5A0A2D433F3A7F6CA74D046658FF85A9AFCE68DF76ADFC9CB682F80F" +
            "29F1A15257A9ABA43641C68A405A88F499ED3B29D097BBA995F1D29616CC47CD" +
            "D8FAAC992A8E88D7A4E033E1D37B5DB2F38DB1E37B2A6B1A094A621BBDB70B67" +
            "EB35E63425BEE4C0549D2B30FC9207FFE0BA8C3B39EC8562AB61D0A77F78CAD9" +
            "D5C6227C7EDD91316D5202D6AC8CB5503F4FA01BB76A47C59FBEAEF8064709A6" +
            "CC2DD5BFAF4C96535A1F699AAC58C213BC1DA0043252C5C2E2852F3B0326D428" +
            "A3EB3CEB1372D65271F23935DA069CDB2E2446B26BB98EE04516548482EF7D9E" +
            "2A855266521BED66156D491E7FA20D723054F55930D0E02A356659B7513CED09" +
            "8666D05B6C3EC9281048F57031992620BB3F582F5963AA167BC7F8C8DB07644B" +
            "8AE5984CBA7122A67275E024B3790178C8DE7F260D0D2FCB643B990FDC514F5F" +
            "88B9EE89D16B20E26FBB2FE315347E972A25B6DE4DED8D69526C506004C46918" +
            "13567486B8CD40E4D710069F137ED0E5EC4A493ACA9545F395935E145CB63354" +
            "4D436F9FB5BDF0B154E72867FCD9A2F8675D38FAD9CBD2B1A4EC01B78C229445" +
            "FD2B4232DA613A5BD9847CC2F19E624F6B1223E82F6F12CD97740E938C34FEE2" +
            "54F775437183BD93D52F96C4C1ACE5ECE2D70F1C42DAA699BB76AEEA5693C7C4" +
            "0B5DCB4E6D61E648E5335B06B2E0BFB55F2F338033DF38D8A2C8AD9760F0B21E" +
            "8DB0DF9AF8E682E6F62F0C70A9E921FA86A7C9EFFFF2A4302F94604D1F991AA4" +
            "F6B1D3B7E46BBE4126C874583281746CDEA73337A904110E94B333CB6BFFF28E" +
            "1D0A0516EA5F72876614328AD34B4C1148B0930A2E303B301F300706052B0E03" +
            "021A0414C1F4E43998F3594C85C97FC9C5777848189A67420414104FBC73DE62" +
            "EFB93E15B5A39DE7E0C1B46E20E102022710").HexToByteArray();

        internal static readonly byte[] Pkcs12Builder3DESCBCWithEmptyPassword =
            ("3082088E0201033082084A06092A864886F70D010701A082083B048208373082" +
            "08333082082F06092A864886F70D010706A08208203082081C02010030820815" +
            "06092A864886F70D0107013024060A2A864886F70D010C01033016041039A75E" +
            "DA3FFCB2391E25BF504AC4BBAA02022710808207E0A9EFBDBE9CCF94EBD69A0A" +
            "22DAD13EE5036E45C5A28F2C15888DE9A6E357D3A35E9C466E3101DD33B08AE3" +
            "B17103506D371A69D230C1F00DE33487E0B9DE8318DB88397D13A6B8B172CB2D" +
            "C57CE7D3D273878ACDEEA06F6D49D06757B7CED3F6958D98620AD7FB303D9CE7" +
            "5F0B0880C9F3D2762157B946963FCA3E4C7DD3CF1BD359CA27CEDEF45613D8C4" +
            "2ADD13B133997B508A53C56C1581A87C03F0A946C33226D6F40F523B7F920985" +
            "470B0B5E5EDD5401C4BE78072D3333CEDF47FEBF83EEFBC7E6F9D9D02AE95384" +
            "B68DF04EF7BA859B6D5AFC6FD9C918BEF427C49D5097E7645FC3BBFD9CA29EDB" +
            "09F89908937E2B51FFA2F806CA0EE6E09A1CDC9F73D3E8393F1B9DE437709FF9" +
            "59E4AEB2524C15269A871A4F6201D15846CA40F7F052AF72909878945894016F" +
            "6BFF0B966338D371BAEE16EF7E3151FEE91F3EB6AA48E91B7580FC1ADB3456B5" +
            "2382EBAB36548CA3521A583647CAEB75C50EBF485AECD50EDF6DF6B630031A5C" +
            "FBDB9340C05F5F94B2622B8A0DB23B2D658D6F5733DD65328B1EDB4779D318E7" +
            "CC704C7B7B78B950DEFAE77872CAE2BB1E25BA7BCAB8C04ED12F6EAF48808C79" +
            "7E9C04FE4F7444A7B58D7E8A6E5445729C44E9118158C62E2760BA6D014EF026" +
            "8EFD70AE83E45FC1EF05B9043C55E5AC4393508CAF2A36932ECDF77495C35036" +
            "4F4C92AE4DEF9475A7F274F946EAF3347D05BF49725C86CFF72579006636F096" +
            "FA49D5183100271F94C19D292DE0A7C78A8D2AA347396ACAFF36A123135FDB45" +
            "604E3B829C00DF082B7010B41DBD62416315C2AC384E7D700779C592487F69F3" +
            "917B391246A24F8E76258F8D3219A3A6D1313443CF78BEF667FBFDC507191625" +
            "1913B729E11E895B1840635882997E5C428DB006A875E561AC0758F80F702E33" +
            "DD985319B2D6BDB1EA0D403A761968494BC77CAAF4CE42B6E9124083134989A2" +
            "877077CAB2F8EDFC8EC13C255022C3AF77A40A3BB5CE1B76A57DC5648234346C" +
            "A072852934F310D3446CBD129858D00A64462E89E3C61D128F1C9590CE9A6284" +
            "CF29EC375DB56B79F13CB404AC4320BF491472FB0FB1D530D68C82E2029C7CE1" +
            "2AF38F419C163454EFFC7FE6EE590BCC3543E2B48AE9A43358E76C4B2D90AEC8" +
            "CFC82BFFCCE4DEAAA23E4944C3034B4C1341FCA68246F328B65341442571FEC6" +
            "F3671DA0284FF6789452C081DA021CDD3FDBAA34E82FE0C68B75321488018EBB" +
            "FE787787A89058696E4F92E23211085EADF706462391F1F631CD87CAD48D72FF" +
            "1EDE98CAE24DA0C776F342FECE74B1EEBCA4B8FCDEF4BC9AB952C04F79D5B235" +
            "680782399746091BFC93577B863CCF499D93736CAB5500A4C45DE78FAD5E57D2" +
            "E22D1A5F7C85FDEBE52F0431F980E90B44C0250079BCB09D4C2E3CCFB6E9930E" +
            "B9EA497E7035FDF826600F795E3CC9EB0CE66DC84131B8EE3D55829DF48F6739" +
            "0BD11F92B92D20C86B93FC9F9DDEE2C0D7DFF25275BFD25D7C2FF5BED8B4789E" +
            "BC5AE940D802C0A86BC0CBE2958BC93B7D80C3272945032AC9B0FAD174A636CD" +
            "68500BAB5A53A399A9630F8B523A1639A7B313F3F0517708EFD57E4E57319955" +
            "48887831D86087F6D4B5180559CBC461C29327D5161518DD88E0761BF0CE05A2" +
            "7FE504C84F6ACBA1C87A1284ACF18D5EE2B17B63549C35951CCC582BB8975EA7" +
            "B49E7C061933B53724D25EE9F8F04F2A36B8B1F293A1A76B822C3BB0546C029B" +
            "91F3FE656778236EA95B978DAF9987AD80757696DE2393149A260B5EF0C6437D" +
            "DA8A63B8A486E14F42B8D1A69725CDE49E68C867B5F87D9E3CBDB4D64045C0CC" +
            "50ED426A5F78CA1216AD394F3A4CD33FF6AC219AF8ED4555079511E49C288E96" +
            "1165A20134EC21A5100FA597133B2B81646E245E5C94C26247112982667CE49A" +
            "5633663629BC8280A4115100F2A3B4384469D3A0A7DE8D35D2FB984EE0190A0F" +
            "C70CE790CF27CD3F4D261F941BEA6478CE155C5A409204F1298FCA20B1D3F095" +
            "ADE129236F101BBE9D766DAFFD9D12296AAA618FE1FA8E0EB0B4D873AF67B1DC" +
            "A3AC63D935952FF833621D1E7E06AE0C5749E60941D9B29D3E68968DF21E31AF" +
            "DAC972B604CE613E98BBFB40BBEADA6AE85F716BD655066B1CCBEB11CB48C5A7" +
            "56B4B7C96BE9DCD3DC8B74CAADC4CA7038DA00B867B35185355EB0D2C2FC76D4" +
            "2842B307D92B707CB61F9BC8BF6AB1FC9EE8D0F5A8C3A1EECED5D6FB1D13A38F" +
            "B2D17E0513AA4F126B5D5FC9029E574179C447D04D5A22063B60E88C48A4F52C" +
            "949F49841DBF73A2F884239F18BBE35453BF67611036781876920157358303B2" +
            "BF9D17473EBAD888D5A612BDE9AC41778B33F660C7C2922B654CAB899F16F017" +
            "ABDE8660ABF43652ED8AB2B9177CB407E79B3B94FDD61407618B77FE980A727B" +
            "BBC8B3034B0FF138C6B32B0AC9B0A851FEA1A8383EE1C571A55E0ABBD63B55BC" +
            "3A5F87A3ABC9D69BFAC4FE5AFF937F536A66D9EECC8CEB491BFD3505564AA8DB" +
            "0EB624817F7D6E848E505C320349697F13026A024F8D5C74E0DE9065974CD0C1" +
            "D8D6D7C55C0721EFF5A960C6A44F7FCFC432998859C397C33171A3E539BB2FB3" +
            "C0939BFC3D63BA3D03EF18F11CCFD4F95AB2C99AC69CCFAB692659193FF81FB6" +
            "5A2A4ED12C75E2B3E12C5B76420FCB34C94CB5140AAE2F96F5CE0C6C2784CF30" +
            "88F10B7DE2B2FE8F486E7A5A8858891A98DC460C51730909688BC9E1DB92DD3E" +
            "B6BED869FC7D708FEA4BA6FC5DEADAF86DEBECC75A4FB0FB61FD19C05F371EBC" +
            "EA880808ABAA83CE2B447F7DE021900D6777A7377FB990B85DDCFE64A36C8FD5" +
            "09463C6E50BCA36EB3F8BCB00D8A415D2D0DB5AE08303B301F300706052B0E03" +
            "021A0414A57105D833610A6D07EBFBE51E5486CD3F8BCE0D0414DB32290CC077" +
            "37E9D9446E37F104FA876C861C0102022710").HexToByteArray();

        // Used for chain building tests (CN=Test chain building)
        internal static readonly byte[] SelfSignedCertSha256RsaBytes = (
            "308202BD308201A5A003020102020900EF79C10DFD657168300D06092A864886F70D0101" +
            "0B0500301E311C301A060355040313135465737420636861696E206275696C64696E6730" +
            "1E170D3231313031333231353735335A170D3232313031333231353735335A301E311C30" +
            "1A060355040313135465737420636861696E206275696C64696E6730820122300D06092A" +
            "864886F70D01010105000382010F003082010A0282010100E3B5BBF862313DEAA9172788" +
            "278B26A3EAB61B9B0326F5CEA91B1A6C6DFD156836A2363BFAC5B0F4A78F4CFF5A11F35A" +
            "831C6D7935D1DFD13DD81DA29AA0645CBA9F4D20BF991C625E6D61CF396C15914DEE41F6" +
            "1190E97B52BFF7AE52B79FD0E2EEE3319EC23C30D27A52A2E8A963557B12BEC0664ADEF9" +
            "3C520B587EC5DABFBC70980DB7473414B4B6BF982EA9AA0969F2A76AA085464AE78DFB2B" +
            "F04BDE7192874679193119C2AABEC04D360F61925921660BF09A0489B7C53464F5FC35B8" +
            "612F5B993D544475C20AC46CD350A34551FEA0ACBD138D8B72F79052BF0EB3BD794A426C" +
            "0117CB77B4F311FFD1C628F8E438E5474509AD51FA035558771546310203010001300D06" +
            "092A864886F70D01010B050003820101000A12CE2FC3DC854113D179725E9D9ADD013A42" +
            "D66340CEA7A465D54EC357AD8FED1828862D8B5C32EB3D21FC8B26A7CFA9D9FB36F593CC" +
            "6AD30C25C96E8100C3F07B1B51430245EE995864749C53B409260B4040705654710C236F" +
            "D9B7DE3F3BE5E6E5047712C5E506419106A57C5290BB206A97F6A3FCC4B4C83E25C3FC6D" +
            "2BAB03B941374086265EE08A90A8C72A63A4053044B9FA3ABD5ED5785CFDDB15A6A327BD" +
            "C0CC2B115B9D33BD6E528E35670E5A6A8D9CF52199F8D693315C60D9ADAD54EF7FDCED36" +
            "0C8C79E84D42AB5CB6355A70951B1ABF1F2B3FB8BEB7E3A8D6BA2293C0DB8C86B0BB060F" +
            "0D6DB9939E88B998662A27F092634BBF21F58EEAAA").HexToByteArray();

        // This is nearly identical to the cert in Pkcs7SelfSignedCertSha256RsaBytes,
        // but we've replaced the OID (1.2.840.113549.1.1.11 sha256RSA) with a dummy OID
        // 1.3.9999.1234.5678.1234. The cert should load properly into an X509Certificate2
        // object but will cause chain building to fail.
        internal static readonly byte[] SelfSignedCertDummyOidBytes = (
            "308202BD308201A5A003020102020900EF79C10DFD657168300D06092A864886F70D0101" +
            "0B0500301E311C301A060355040313135465737420636861696E206275696C64696E6730" +
            "1E170D3231313031333231353735335A170D3232313031333231353735335A301E311C30" +
            "1A060355040313135465737420636861696E206275696C64696E6730820122300D06092A" +
            "864886F70D01010105000382010F003082010A0282010100E3B5BBF862313DEAA9172788" +
            "278B26A3EAB61B9B0326F5CEA91B1A6C6DFD156836A2363BFAC5B0F4A78F4CFF5A11F35A" +
            "831C6D7935D1DFD13DD81DA29AA0645CBA9F4D20BF991C625E6D61CF396C15914DEE41F6" +
            "1190E97B52BFF7AE52B79FD0E2EEE3319EC23C30D27A52A2E8A963557B12BEC0664ADEF9" +
            "3C520B587EC5DABFBC70980DB7473414B4B6BF982EA9AA0969F2A76AA085464AE78DFB2B" +
            "F04BDE7192874679193119C2AABEC04D360F61925921660BF09A0489B7C53464F5FC35B8" +
            "612F5B993D544475C20AC46CD350A34551FEA0ACBD138D8B72F79052BF0EB3BD794A426C" +
            "0117CB77B4F311FFD1C628F8E438E5474509AD51FA035558771546310203010001300D06" +
            "092BCE0F8952AC2E8952050003820101000A12CE2FC3DC854113D179725E9D9ADD013A42" +
            "D66340CEA7A465D54EC357AD8FED1828862D8B5C32EB3D21FC8B26A7CFA9D9FB36F593CC" +
            "6AD30C25C96E8100C3F07B1B51430245EE995864749C53B409260B4040705654710C236F" +
            "D9B7DE3F3BE5E6E5047712C5E506419106A57C5290BB206A97F6A3FCC4B4C83E25C3FC6D" +
            "2BAB03B941374086265EE08A90A8C72A63A4053044B9FA3ABD5ED5785CFDDB15A6A327BD" +
            "C0CC2B115B9D33BD6E528E35670E5A6A8D9CF52199F8D693315C60D9ADAD54EF7FDCED36" +
            "0C8C79E84D42AB5CB6355A70951B1ABF1F2B3FB8BEB7E3A8D6BA2293C0DB8C86B0BB060F" +
            "0D6DB9939E88B998662A27F092634BBF21F58EEAAA").HexToByteArray();

        internal static readonly byte[] EmptyPkcs7 = "300B06092A864886F70D010702".HexToByteArray();

        // This is an RSA-1024 certificate, "CN=outer" which has an extension with
        // the unregistered OID 0.0.1 that is a PEM-encoded ECDSA-secp521r1
        // certificate ("CN=inner").
        internal static readonly byte[] NestedCertificates = (
            "3082041930820382A00302010202084062D86F6A371DD7300D06092A864886F7" +
            "0D01010B05003010310E300C060355040313056F75746572301E170D32343031" +
            "32353031333630315A170D3234303132353031343630315A3010310E300C0603" +
            "55040313056F7574657230819F300D06092A864886F70D010101050003818D00" +
            "30818902818100AC24F75F85C4C38E6DBF57DE889AED7598DD01202FACC00EA7" +
            "4EC449DD420E7CB49992E1BCCC69B3EAB8B2AEECA5B2BCFC1295DC82B83EDB71" +
            "C1764191EA0F73D1BAB26D03B95F322EF8299B1DADAECCAADEBA052BD3BC2549" +
            "D83AD1C2FB2DC556370306AC3CBCE09F3669448EDEF8FCA12164D793D5B456A7" +
            "418393899E00590203010001A382027A3082027630820272060200010482026A" +
            "0D0A2D2D2D2D2D424547494E2043455254494649434154452D2D2D2D2D0A4D49" +
            "49426D6A43422F61414441674543416767312B7233664B4775457054414B4267" +
            "6771686B6A4F50515144416A41514D5134774441594456515144457756700A62" +
            "6D356C636A4165467730794E4441784D6A55774D544D324D444661467730794E" +
            "4441784D6A55774D5451324D4446614D424178446A414D42674E5642414D540A" +
            "42576C75626D56794D4947624D42414742797147534D34394167454742537542" +
            "4241416A4134474741415142513143437A4448737A724E6839445456697A6E75" +
            "0A4B38516F496B5649324138494E676B4B6F6B3649704158484C533058767333" +
            "4E43745152396973486C5A376C707844366C6B72376449793363396756427063" +
            "630A7131734242796B73534854745A4766337A2F6E526E5A3961684463755444" +
            "2F486E71302B326A33476461423557594B6733336E65337730423258385A7377" +
            "65390A46774F2B335A6954734D6D647038312B6B4D6C76463775453236597743" +
            "6759494B6F5A497A6A304541774944675973414D494748416B4942454B6C4654" +
            "4978560A434274324839636A30595042493632457734752F665A4B4A53657272" +
            "7848707244476C67546B36635075585345723569395A397731505A3872654A51" +
            "7A7367690A6C736268363653584B4F6C67466945435153587972457736633565" +
            "734B587A6269484F7762564E4756305A4430424A6761316533667144507A4170" +
            "75527734510A75656A636749684E723577504E7A4E696A464D57697944716D79" +
            "6E586D4B6E627933457050396D370A2D2D2D2D2D454E44204345525449464943" +
            "4154452D2D2D2D2D0D0A300D06092A864886F70D01010B05000381810032833C" +
            "38E5F66D64880C191D247CE8B819899B20F284DD59CBDE1731A4D0FE09A95A52" +
            "D5A7D049CA108BEB9673FD207C842B324DFD8086C9E1CDAB931D7403730E0521" +
            "69943C58ECC3DA6E11C6ED4F16455152D6FF4D104C88976F9BA88120B0889563" +
            "1378357F297A6B3E444296C06C636A589973250F2A096C39C1EDE5C1C9").HexToByteArray();

        // This is a PFX that uses mixed algorithms and iteration counts.
        // PFX {
        //  SC[0], encrypted, PBES2: AES128CBC, SHA256, 2000 iterations {
        //    cert, keyid=1
        //  },
        //  SC[1], plaintext {
        //    shrouded_key, PBES2: AES192CBC, SHA384, 2001 iterations, keyid=1
        //  },
        //  mac: SHA384, 39 iterations.
        // }
        // "Placeholder"
        internal static readonly byte[] MixedIterationsPfx = (
            "30820A320201033082099206092A864886F70D010701A08209830482097F3082" +
            "097B308203FA06092A864886F70D010706A08203EB308203E7020100308203E0" +
            "06092A864886F70D010701305F06092A864886F70D01050D3052303106092A86" +
            "4886F70D01050C302404101215FB90500DC30882E7D84625F97923020207D030" +
            "0C06082A864886F70D020A0500301D06096086480165030401160410558E3FAE" +
            "FE92ADEC4CF1405E7DE7C5118082037095F887CE32E748EC0D647062286EA9D0" +
            "C3C04213ED9431FFE0F6028105D94D088684A55C997630D16EF7CE7D49729C30" +
            "1F750D8864B988ABBAF551A8819ED7D17CBB810CA3DC39EBD804B3C388C73BA9" +
            "320783DF3AABEBAD9F44E4ABFF931AAA95AE874E348B49F35749614D9EE94F5E" +
            "EC7A5C597FE8CB6CCDB7A0721EE5836C8F839D83E1577A753F8B2B7E0C2C53BD" +
            "D365322E211A62DE061FAC3ECF770CEBCF37AC2604A1CF318C4DB1BA83C85779" +
            "3BE57C898EA80D52E0F83B17F877210C1A5E819F5C35DE6DB5EBACE1BFB3101C" +
            "C2FE9FF4983BAB7B93CA5D466B855D973FB5D72D8DD109B61FBA238320C886C0" +
            "8FC141168C43B946AAC28C114B20A667D189E473E9779432EF64C965459AE9C4" +
            "6430B94E19BCEB4733D3038D9F65B85D79B7A1DA954887BAE1DB0A999135A09A" +
            "25C38616DBADB9BE50F729925F23C107F899341F76B4B2161CCAF8C499DF7646" +
            "F7DB16A21EFD90097D75EEF893B8B8115F6A8816B2D9628278F8ADAFD0E4AC7F" +
            "14BED13FFB6DD7FEDE1EA1990CACE772DBFDACC2A4547F910D3F2B238FE86A4F" +
            "1BAEE8AC282F918D3F82B8149DD3A99B78D433AB0F5592E7B7B25FBC4A726520" +
            "A49CF2FDA60D747F19031B34F911C442E20BD91600A32F9A4FCC1E3DF330171A" +
            "4E0721CD01A93CC20C2B770990029C30306945AFC7E3ABD9516D8C6F359D904B" +
            "5F32F3BAEC8770A8BE15C89E4B836B1F20D29A7D3D3BB7491EFC9B88A161B598" +
            "02819D2EC695000C2F83B1A363217CC2B0A70A4A226993BD32BE302DCD157EEE" +
            "3F9DBFBDE0375A03DEA2761941F49D42A8F5029DC3B65E661B61DFBE5F2527D6" +
            "C59927B5A0857FE7B9836A63D202895C1FA6FDE40E70107659CE3E4C7DDDBC9E" +
            "04B85A46104EFDF6692BF898C565480F5A5163231167DB69F873D1341CDF8DF0" +
            "7BD4473F58162A3EB653C6567C2E3120B0034B9E35D22838BBF02968651FC1F8" +
            "873C405237B7F29297ED47CADEABF248386CE94D215C8D5731A503623C9CA916" +
            "C5896B8F713A044ABBECA764D36D074DF523D989228B89589463DD3615A7E844" +
            "E7E9E91BE0BFAD79218CD7B52A313458BBA8FC660B5DF8513464FE749EF3C201" +
            "5C15B2EF15F63D5FDA726ED8E04B3B5E7DB8610367AB08EDF120DDEEA7382AC5" +
            "1C778B8069E09F2EB68D19154DD10129A1E0DA0717AE1E108B2496C69904D851" +
            "18B37E06FA95E1F5E75371D8C7EC3BC80C817F931744428BF03C1F51A0164C0A" +
            "3082057906092A864886F70D010701A082056A04820566308205623082055E06" +
            "0B2A864886F70D010C0A0102A082053930820535305F06092A864886F70D0105" +
            "0D3052303106092A864886F70D01050C30240410B5D9DF064271CEFFDBC3C067" +
            "F6DFEB74020207D1300C06082A864886F70D02090500301D0609608648016503" +
            "0401020410DEC794EDB555B89CAFB3671FAB4A52E1048204D050B2A2E9E3D3C0" +
            "84D5BFCAB1C7C27951C0B72FB45B52089605DBF5E60550521DEAB0A66EF2C182" +
            "3AE2F4EC21A1B8BF65D240DF20759C66AB0C6311CABB30F9DE1290B81CF69AAE" +
            "7B6F42A4E9BE7BB6F09058C7D0D6FBE6F24EC2E3457680907F000C5B457B1E7A" +
            "E394A7A70E6289C3B009F522399D5CC41014F12A336926F3AB8E2A3AF5496BC3" +
            "27B073206F20F131AFCB627A1E67B9B457C44DB6A6C10EEBB8856E0BAAF99D8A" +
            "9D24D7C90AC5EA9EB14C66315E77F158948988BB729D8E0796741CAE29894DC9" +
            "6614C2B06911013C168C7A4E6C46F09D1AB0D7933729FC88BA47A41BFAB0AB53" +
            "C60FEAF6A93688E67039184B598BDF1CC95C3967F9ECC649745E265974713102" +
            "E271ECFA6A067F10751C2A4A70A94FA39F37E944996C843809D82990A9D44C18" +
            "9EE7B9DC8534F9B821A5C56A61DFAAF470444883200B3FFFAA23DCE84375937D" +
            "D0149869B2683773F650B70003EB025A04E170DFD97D4F86D2913E91F757BC01" +
            "5F1F85F497B1400052E0868ED6844E2390F71B036B524E824B3ED6381BCE2600" +
            "71A5F6EB12E9C1C44BBE0664217C4A44EA07E440A33010A0CA46647E7FFA4F58" +
            "74F1EFB9DD330DA2CD50AC01C489639520A1A27B603F44831EF235F0540817F2" +
            "196AF45D3B9957D272B5549BEC5507F6BA443A37F54FD6D6AEBEBBB7AAE15F7F" +
            "8C1C46D68CEEC2E95426D177A390E7C50B0122BDDA9EFC104294840F6D5374AD" +
            "90952842C0EDF796AF48C6FDD4AA02C81AFCD606CD0E94F3DF2B06B649A47D9D" +
            "7C8AE23E1A776E1D149CA25E4929238778B19E33DE05DE577B763305B4884B8A" +
            "7DE31BB9712477C9CDC138B52FB2D59CFE42A1CDAEB0E205C70B38CC4B88E0A6" +
            "ABCC41D29D11947FF5B03633A5E164ED22F10536AAD07DB2EB52A2C696EBB135" +
            "B5220839D1E5A7DB6C2B8DEB09C883129BD5253F68169F9D5F5A2F3AB17C4921" +
            "10636978B8573A5B4E1FFF5A4C3E75E2F03AD71AF874C544474E1B41969416D6" +
            "D8FFFD595428EC7928BD17C652F67D9B6B150C6D4A9352C405BED162492A5FD5" +
            "7EB6BA5F77AAC2BD7E4EEB6B0BD2E4329E2CA8A425F88F4743B25F259E292C04" +
            "483472CA79FF52271A830AA6A27A52C3531E2B2503592C017A7CC00F91F63F73" +
            "4E3E56746475B8B338440B7D5FAC87A90831EE78A2DE4FD6F60F1C66B31A520C" +
            "44B73B09D5419451C4A32E8E1A5BB17E44B9FAABFB07021747093DBFE248BACB" +
            "E2BB6C7F145DE7397A2B2AEAA083EE57F46C8DF85FA2DFF4582C2E3CE3CB2E91" +
            "706395A63BA96343B0567E41A33FC964BC8C03CEDE5E3E1D7A8B285F745EEAF5" +
            "CD1382C86DA82922DA1772158F8BCCFF70DA87A64602033560566F33A3793A4E" +
            "5C404D2C69274EEE9E82B5B97B0760FD66067888711C572E84DB382491792CB2" +
            "C7BFA472E6B0D70D529701C2A0B730F5E1E0A980EAD56EA323E0008C70D62F53" +
            "5B9E0533F9A4B7CFA22319274E68C8B4737E5DAB5B1956C235E7EA548E24E23F" +
            "F9FCA61D11DAFC6B90E0BE8E96A66B6973D5F025C0619D283CC92C3224000FEE" +
            "F9BD002E7EFAD4C737C4CFEB42858DBDFBC489B1131AABDD1868C58EDCAD35C2" +
            "AE1A42BBCF0A2A90E0557A7A5F79F2D92D19E39D505994163CC94F5EA56009F0" +
            "5E9ABCAF24807130F90FCA606D5448C103489BB53090EC603394705B472132E4" +
            "FAD50491069F44040A0D66F7D3D5C86593D61C9D37CDE3BBE5651EA2E104B324" +
            "3272E665CDBB139E063112301006092A864886F70D0109153103040101308196" +
            "304F300B060960864801650304020304405EDC86442FD573401BD2DF0F95356A" +
            "1C1454F401231B7F772179626ABCB220C8096AC0ED6C27CACED7D94615768B61" +
            "8BDDCF4B8A0996E019BD418423F79F173404406B32D0D889B85234D716C87F3D" +
            "EEBD62B5DC14984FDB9EA9FC765B340F54D3E5203C6A9F4F23913B22605A32BD" +
            "3D8D120CFFFE4ACB83BA4D488C67271E38CD40020127").HexToByteArray();

        // This is a PFX that mixes encrypted and unencrypted certificates.
        // PFX {
        //  SC[0], plaintext {
        //    cert, keyid=2
        //  },
        //  SC[1], encrypted, PBES2: AES128CBC, SHA384, 2000 iterations {
        //    cert, keyid=1
        //  },
        //  SC[2], plaintext {
        //    shrouded_key, PBES2: AES128CBC, SHA256, 2001 iterations, keyid=1,
        //    shrouded_key, PBES2: AES128CBC, SHA256, 27 iterations, keyid=2
        //  },
        //  mac: SHA384, 39 iterations.
        // }
        // "Placeholder"
        internal static readonly byte[] TwoCertsPfx_OneEncrypted = (
            "30820CAB02010330820C0B06092A864886F70D010701A0820BFC04820BF83082" +
            "0BF43082029406092A864886F70D010701A0820285048202813082027D308202" +
            "79060B2A864886F70D010C0A0103A082025430820250060A2A864886F70D0109" +
            "1601A08202400482023C30820238308201A1A003020102020900BD698EB46606" +
            "F1E4300D06092A864886F70D01010B0500305E311E301C060355040A13154D69" +
            "63726F736F667420436F72706F726174696F6E31173015060355040B130E2E4E" +
            "4554204C6962726172696573312330210603550403131A506C61696E74657874" +
            "2054657374204365727469666963617465301E170D3234303531303138303033" +
            "375A170D3234303531303138313033375A305E311E301C060355040A13154D69" +
            "63726F736F667420436F72706F726174696F6E31173015060355040B130E2E4E" +
            "4554204C6962726172696573312330210603550403131A506C61696E74657874" +
            "205465737420436572746966696361746530819F300D06092A864886F70D0101" +
            "01050003818D0030818902818100A6638EFEFA9A1B364574D9A7BA4A85017E73" +
            "61831606B717DCF8FC0F5B982583D5460BCD216E99FBC15ABF62B5C30FDC7CF7" +
            "D13CDCF9E0A0A25C26AF0AC14D116569FAA6496CB87ECB0A3B87AAF624010630" +
            "4E7F0DDB63FF7EC95396F2CF4E57A1F0356414C7D1032433831C327AC33DD264" +
            "FC7B1BA567FBE360AF35446B63110203010001300D06092A864886F70D01010B" +
            "0500038181001E2C2AC3FC484C42FBFC3A0D027E438BDBF6FDD4609BF1E11BA9" +
            "CDCF50D121BAEDFA5361619B469D48F1CBA203F361A2E7D662A88239A500D056" +
            "CB38D215197A3F5FFACA5AADDAC875D380C4435C0E8633243174623BF59836F0" +
            "8DBA917C5B4A2270579574566FFA29A7A5FF0415D34E8CCFFB9EAD4BA8EC90BF" +
            "B3A2F7041F6B3112301006092A864886F70D01091531030401023082031A0609" +
            "2A864886F70D010706A082030B308203070201003082030006092A864886F70D" +
            "010701305F06092A864886F70D01050D3052303106092A864886F70D01050C30" +
            "24041078790CAC0DA9B3E9B0EE236CA8073CF3020207D0300C06082A864886F7" +
            "0D020A0500301D06096086480165030401160410256DE9ED9807FD7D30597DB5" +
            "51A5815080820290D8156A7DDCD344FE96B8D8BBA4303A373108D725BFC30071" +
            "DBF716A7C1FD1B598BAE1FBC9A77CCBB7319646E83B747B59330760B6EBC29B0" +
            "91E591FF0030ECB28626BA1594FEF6F0A01B60F2548AE1577FD05E7CAC5BCFA6" +
            "2422E71F551FC2A8ACE488E871C03E1E02A9DB4ADA9DB335466EE1A6E7AE6B9C" +
            "AE118AF4008879690C446F0D4A03740E31E4879B163E58FFA46394DF57CD98AE" +
            "FA3C44E94BDD36D31A53EB2C9A74EA9F45718370106F205A81837A05952E7C17" +
            "3460A2400195D658671BE62E76AAD565D848109BE41B6F06CA87A7B7676B9A5A" +
            "525B6ADDE80A68A87FE82C8547D611F6DEC5C2AD617D354216CC7E724DA2F7A7" +
            "BC67C336A68F7A8ED4F555E53330DDA1318332E170458D10E7A1CA60D87F61C5" +
            "89B8BC4CDCAC26BA341B96BD20096B89977750B1E2BACDEE23130825B827F5CA" +
            "E73373AD2258201B4DED998F003E1084BE969F5B1EE33B8443BE01C509927876" +
            "1D69A1E48662EE91DF5211176DA5495AD54CB50EB2A2A3E077E93946958CAFE9" +
            "5F2FB3450B6269F3541BB21B6F129288115E177594D6EC0DD516B8882100C9CA" +
            "DB9874064B3A1F051FBB0B43257F0644A05C6D416C0652696E89DB6E43CE5E92" +
            "94D05711CDBF7B304E076D73FEDA97A7E2DFF398761DE6425730CB576D26F49A" +
            "2CCD93BDD0E410F70D4EF8DB6BCF221BD3E53C472CEA9C40E09A284923992BD2" +
            "12B32E36BBB06EC3000261E8EC9D9E389F663DBAAFEBE0CF671E771A246D033E" +
            "AFFDF400B94D8926E23FD660E0A8304C01D490EF54F868506EE8AA255F95E00A" +
            "7E1AA1D18998B9AE8E0AA0472CD152A674536F5D2882502247D2B056137BFF97" +
            "08CD8EC61A3339CCB1DF43DC60A0A75F9260905BB3D339580EA5B5B7BAA92374" +
            "4A8850389A2E68B98FE6EF78D78207ACC858A97DEC55C0D03082063A06092A86" +
            "4886F70D010701A082062B04820627308206233082030E060B2A864886F70D01" +
            "0C0A0102A08202E9308202E5305F06092A864886F70D01050D3052303106092A" +
            "864886F70D01050C3024041077CF0CA72DCB1B541E6F481D3A51470A020207D1" +
            "300C06082A864886F70D02090500301D06096086480165030401020410503141" +
            "7A0A42803706E84B8C7470CD2D048202807E3108B51EA4D80D2E7CF358E08C6E" +
            "A30087AE0E9F80CF1B022A0DE0BBDA715733141C6877276864DE98D439D61650" +
            "63ED517FBF206450AE6B29835F89914003B8D77F0F15D6FB073EF27F90E9D851" +
            "ABE569D3DE6968E8F61EA184B484B4284C2FD3803D9AA4DE083F18A64B2F9320" +
            "156004E49C4A179CF8800799451563200CD998CBA779032302CEA4227A78CD10" +
            "8FC8BFA6C114F005A2937948F14DDDF05AEB543F4D2FB3C56FA7C1F14B7ACA57" +
            "0CA8554A6EA4AC1C0FD356E3D1DD568F97D6EDBFD3B09C87BC2013A88C442993" +
            "4C73F442A5D5DA3563FCB85EBEC41085843D14EE88E582151E6855741106CA1D" +
            "D775A5793E8A0B4E036427E6D2FC9F0AA394085997E42875E4EF5742099141CD" +
            "70B4AA887C40E3EDDB407C39F3C474F95D394D98DAA68BD1005F1223B6E58B52" +
            "AA552FD25652AA871F03A53A58922953DB120BAFB44C29DCF9C9C2F96DEE3BEE" +
            "7E0F7FF8F04C98ED631F6A845B8B89F30EFB956846BA87E9B3F6FCCB49FE0F21" +
            "10139E855389C8EC5120983A09D5F4D655E64058ADD1E8C44F3B6A4936E6788B" +
            "E939061221424D71666765D99FE6D883C9E8079992D585F73B0CD57947D38A54" +
            "FE7CBE93BE117675A3E52F708B2A6BA6C390BD3BE01100243E7B7B5CB8A9F501" +
            "CCCCD082F9867B4DA0A9FF3DC5BB0F0974D4EDEF9FF89F6EB355BCE07B00A32D" +
            "AEE443D46EF19A96E6EB3EA4C0B141606A1FDB200D1BADBC11954951383E8935" +
            "4603C10E241353FED1502D8111E42DA45198D1586CB0938947F3856CCC855F84" +
            "231F131D636DFBB46394A3EBEDAC9A4DAD0DCB32163C57A9B5DD21BC56CA969A" +
            "F574E38787DFA6EC2BA063E60375585ED5B9E8D5B6D1BC162D63036C52FF4473" +
            "E0A4FD96AA055ED4014C3B0E4F7A098F553112301006092A864886F70D010915" +
            "31030401013082030D060B2A864886F70D010C0A0102A08202E8308202E4305E" +
            "06092A864886F70D01050D3051303006092A864886F70D01050C30230410DE7A" +
            "16E6BCE889C8A5F823E7670FB9E302011B300C06082A864886F70D0209050030" +
            "1D06096086480165030401020410683B6EE3E1287242ED73707ECE1F271F0482" +
            "028018435588B8BEA17EB47E7947D4FE82D66F7C26850B840C0F06964DC99C56" +
            "8391707178494F177EFB65E17D20D3C39E80A00342783B33164F371902D76658" +
            "78A79802FE7D9EB31D21137B56692D1B7E2B38F54B00CEE506867968420CB4FA" +
            "9EE135BA960C0D40EC22714F5D8576BAF03269D19384C246A60ED4B9F0F4268E" +
            "2E135F7BE41536D844718B7339EE14A1CE261B8A27C2AA2AAA90C98F9437372B" +
            "0448579A0C392B347C7E95A1A7617A47218F63D0C52C88A3F93AC47BB9493E2C" +
            "67DE061F28E90FED463E392C6743C8E6F60524A2455C5E46218E357AFB31553D" +
            "0AAD18A2784719D281B12560CB904527D912BA2B40790D9068661C01AA4B03E0" +
            "B5316D6FEBBC38087B4DD46CF2CC0C98B6F488AA9940C6FA16BC8BB1853CD0B0" +
            "E41D85F382F5BE36E5A821C54EBB8C34DD7CD7B970315FF65B657AED6CE2598A" +
            "6DF4E96370815660B3711FDE0A2CA0AB5C2B234C95B699922CFC4E95B781F783" +
            "F20373E8E23EB6D518C797A21B401A797DDAF8A823602ED38135F3FF3C555C7E" +
            "7FA7B800F4535B781B4F6B21BF2147D8C20388CDD48DB3C17413DC4FBD347368" +
            "809762C4851C28CAD52B4F916CAF53E2D5E05ACAC3E85BA1F63525F57F456059" +
            "56B7830AFBC50D0CE517C282AE5DB5638635155073C567A82A7ACDEE6860D18A" +
            "1CAACE341845A41470F47E7A440F21A3DA96CF986181E1F044B28FB3EEFA0E41" +
            "82B93AA2FD50A23B045DABCD774519879CB824B9A58A1859ED553578508B5CF7" +
            "85BE3D2A67F0FC508B94A738162455CD11A4BF0C3736914D8C5A99F9D5ACA959" +
            "B1C4D5872BA852DD3F8B2B8C231017AEF373B3B5E7B65B6DF7115DB99D5EB11F" +
            "E80E8CB3E1D3CE6F50DD7B255F103DC290B7287D6AD7AB12D5242E13092DB9E1" +
            "ADE03112301006092A864886F70D0109153103040102308196304F300B060960" +
            "8648016503040203044044A762FCCD979F096F1C0DE0C4C584E42C1848DB1138" +
            "389DAC33048D6B81DBBCA7E031E8620B37C14F85257DA4A6F57FE58E939493C5" +
            "928343622B80009D8C5B0440E74D33A5F7DA48801A4EF4FD65D0F442F26C845F" +
            "A418D3E0D78FD0285A92A74B433661F516C6955EC40FE46DAB813B6AE940C2DE" +
            "FE8F8F5E32E6B491C999D598020127").HexToByteArray();

        // This is a PFX that has a lot of duplicated attributes:
        // Integrity: None
        // SC[0] Unencrypted
        //  Key (3DES, "Placeholder")
        //   MachineKey (empty)
        //   MachineKey (empty)
        //   LocalKeyId { 1 }
        //   LocalKeyId { 1, 2 }
        //   FriendlyName { keyname000, keyname001 }
        //   ProviderName Microsoft Enhanced RSA and AES Cryptographic Provider
        //   ProviderName Microsoft Software Key Storage Provider
        // SC[1] Unencrypted
        //  Cert, CN=Cerificate 1
        //   LocalKeyId { 1, 4 }
        //   FriendlyName { (CAPI CSP), (CNG KSP) }
        //
        // Note that this test cannot be built by Pkcs12Builder, because that type
        // always unifies attribute sets.
        internal static readonly byte[] DuplicateAttributesPfx = (
            "308207760201033082076F06092A864886F70D010701A08207600482075C3082" +
            "07583082043006092A864886F70D010701A08204210482041D30820419308204" +
            "15060B2A864886F70D010C0A0102A08202A6308202A2301C060A2A864886F70D" +
            "010C0103300E0408F7A34F4CC26F79890202100004820280BD3E4386BE9699C0" +
            "F53C27D4971B17E984BE885F3E1D3B1B75B666EF3F53BED590A70AA071C05057" +
            "7CF587B92AF7F84F0D6E79475CA0D46C5F86A6D548ADE9538A955B7033154F2D" +
            "A356B1DD930576A0A1D6CC87FF9055BB00CECFF5E61040B709FC19C29046A4AA" +
            "BD4B8D1CA8F09B119ED6A8FBDB985E3F22E531D5DB13E292278D73A6C4E05498" +
            "79642858CAA162BB21E9A17B4B14341B388665E0F5B90EB1EFB0C4B211B9B49E" +
            "F8C7F435F9B2D32A319D43B9133039844E7ED8C0BA663E8681C8EFDF707356DC" +
            "A1B78551839300323F046A497DD4C468232993946343764F151AB3EFFFC7FD27" +
            "9739CBE00337399AF2341E7842DA1CCDD98B12A7831A4DE9827F8F1CCB5A0F4E" +
            "ECB412D208983CA5D801D720B3F1E118C20BB8A1853770435177253EF59A62A2" +
            "43E53ABC531F310E245CC1A0626E5456ADC08924F15E1408B2FD30BFDD4A4F32" +
            "01B1983DE0F7F42E7EEF2E8EA6D9B0EAB98174A4B4D0410FD04167670FDFC20A" +
            "1EFC58AB2A41ABD3EC42D3071F31EA5D0A6B93EC070E1D543F0FC7BB8B88361A" +
            "D904E81ADD0C3B0261F1406EACF956F19055FC1C2832F25209DFBD35C8EDD8B4" +
            "091B626E8C07D58F8537C519C90E23E94E8E61DCF3862C1DFB63010D2D909037" +
            "6CFB21042041B550FE62122E3473B88E479B42153FB17077C4BC1318715BAB99" +
            "597226F0C24524FB844CEAA4EC8DD164321DDFB74509FBC4844C205FFC27B067" +
            "C9E4A78B8B12F4643F3A4C754E84F244F84D7A075F290C10A3B544264E317BFA" +
            "41647EC4F50D6B1B2A691B5F0575B9492484019E88355CADADBC0A30FAEED71E" +
            "4392E37BE497900408A85C711BF68B27A84433B0F546DF2FC2CA3FD22C4367BA" +
            "BC074313B982E5012B26863FA98148E5DBF43D26423369C13182015A300D0609" +
            "2B06010401823711023100300D06092B06010401823711023100301006092A86" +
            "4886F70D0109153103040101301306092A864886F70D01091531060401010401" +
            "02303906092A864886F70D010914312C1E14006B00650079006E0061006D0065" +
            "0030003000301E14006B00650079006E0061006D006500300030003130790609" +
            "2B0601040182371101316C1E6A004D006900630072006F0073006F0066007400" +
            "200045006E00680061006E006300650064002000520053004100200061006E00" +
            "640020004100450053002000430072007900700074006F006700720061007000" +
            "6800690063002000500072006F00760069006400650072305D06092B06010401" +
            "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
            "0066007400770061007200650020004B00650079002000530074006F00720061" +
            "00670065002000500072006F007600690064006500723082032006092A864886" +
            "F70D010701A08203110482030D3082030930820305060B2A864886F70D010C0A" +
            "0103A082020F3082020B060A2A864886F70D01091601A08201FB048201F73082" +
            "01F33082015CA00302010202080828E93144879296300D06092A864886F70D01" +
            "010B0500303C31223020060355040B13194475706C6963617465204174747269" +
            "62757465732054657374311630140603550403130D4365727469666963617465" +
            "2031301E170D3234303730323231303531395A170D3234303730323231313031" +
            "395A303C31223020060355040B13194475706C69636174652041747472696275" +
            "7465732054657374311630140603550403130D43657274696669636174652031" +
            "30819F300D06092A864886F70D010101050003818D0030818902818100E6EB18" +
            "5FCE3E6A5A6F73D260B408FA9DD06DC99D546AE9A1F1F8792FDF818E8A2759FF" +
            "D0FCD8A88301117CA992ACEC9AA4E429655AB7A73EE20994155FE97572471D06" +
            "2C06295FF4EE218090DF64AAF787BAD7F511DF329F2F685FFC3B819F95F17811" +
            "E9C158D97D832208C27214C958D844432481981B03FE8C9E0E8C1A5605020301" +
            "0001300D06092A864886F70D01010B0500038181009243F8BCFB5B21BC5BFB45" +
            "83DF87F0D1FE7680C62D1F35698F1A25A0D0517F17B977F039985457A85058DC" +
            "7375D1ED845EB30B20FF7FC3E8970871F03AEDE3658FF964B1EF1BFD1EEB7CF5" +
            "E4A219F1B00DDED1F00BE42F132A2409D0FE78BA131634F5059B06E1B905AE18" +
            "6897E1E10716282E0BE25D460AE93483E9BC0329E13181E2301306092A864886" +
            "F70D01091531060401010401043081CA06092A864886F70D0109143181BC1E6A" +
            "004D006900630072006F0073006F0066007400200045006E00680061006E0063" +
            "00650064002000520053004100200061006E0064002000410045005300200043" +
            "0072007900700074006F0067007200610070006800690063002000500072006F" +
            "007600690064006500721E4E004D006900630072006F0073006F006600740020" +
            "0053006F0066007400770061007200650020004B00650079002000530074006F" +
            "0072006100670065002000500072006F00760069006400650072").HexToByteArray();

        // Uses Placeholder
        internal static readonly byte[] SChannelPfx = (
            "3082062A020103308205E606092A864886F70D010701A08205D7048205D33082" +
            "05CF3082038806092A864886F70D010701A08203790482037530820371308203" +
            "6D060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E040823A7915B5E0EECA9020207D0048202905D79973EDC12FE3C" +
            "839C839008F3E79933BAA119C8015AF35C0DA932704D0C45990F2A60FFDFD26A" +
            "4CC138A3A04673A79CE067AD6D1376608B1A0E61B5F9284A9D9D229ABB17EEFC" +
            "870EB8CD61E6C054FA0AE0202B0FF04452BF3487B38FE256F406CFCA94547EC1" +
            "BD44DA25A857E90990EAB58EAAD26322FDCD8810E4019B81670AD455FF905675" +
            "DABDAC3331AA662AB08D1DCDE0B56CB0F3B8D53F5ABDB613772174B3959EBE75" +
            "EF085404D60DC6161E576B641E5BFB60400C462BA5F9F69CDD2F4F48BA5A3C64" +
            "509FCFF53EF2A7C5AA471762C9BCE99B2AD8C0E415795A816BED896B46C66FE8" +
            "E737829614E0FCF2E3BDC68D24710DDC86FEA5329F8355EF1A51330981303DE8" +
            "38F4CD00D3187CE52F03ACB9BD5A62F98FA1395DA14E3D5FA50F8B466488C0A0" +
            "4780074FD330CA3539067A8A194CC63C2D0D35B3A61A6F8EC711907DB0D8E3D0" +
            "6912F8202746264C5A9C0A9CBD7AACB1176519D901100C299126E0B30D869C14" +
            "B68AA04E8EF3B144447976581BDD63E83DCDDBC8C5C185E2EA598CCAFC137BB0" +
            "F46D053F900D46381915322BFF04F6EADE31F14D6781FD98764613BB679ECE45" +
            "01821EBA6C0AF5E603151903B2A7F19BCBF623401E14C3456DA20128BD53DE70" +
            "1422EFF06FB0DA901D68D55389BE369D76F65B5D4AA112B6636C12F4A7806BE1" +
            "A8DAABCD827D65AEE41B9F2AB9E978E936309F54CC77A5E693161E84D4031DB4" +
            "26440E61F01464CBE564C97BD0F6E23D4E803C9CBB65A79BAEECED26FD03FA41" +
            "4EB2DEE34044DE434D18CE643C2AA1A12A7336522020498642BBC2F2B8132E84" +
            "87227D130B37BA4376D57EF4CF57ADC56F9AB38F60235DAC7C7EC7ED1A537C25" +
            "E3BDF6E5D1B647817CA8D730E4FC0A6013CB4FF8920F76CBD287A50D40818638" +
            "D7E44168E65E26513181A3301306092A864886F70D0109153106040401000000" +
            "302106092A864886F70D01091431141E12004E0061006D006500640020004B00" +
            "650079306906092B0601040182371101315C1E5A004D006900630072006F0073" +
            "006F00660074002000520053004100200053004300680061006E006E0065006C" +
            "002000430072007900700074006F006700720061007000680069006300200050" +
            "0072006F007600690064006500723082023F06092A864886F70D010706A08202" +
            "303082022C0201003082022506092A864886F70D010701301C060A2A864886F7" +
            "0D010C0103300E0408543FA90B580A66D0020207D0808201F8DE9CCF9441B460" +
            "478D35A09462569A805A1067DA2BA99BB5E7AC02FCE133BD4DD5246861D614DD" +
            "903F02D1D342CC5CB3A939D5DABAB1157007B30BB9C9099F29A62FA812EA5DE8" +
            "460B3312D8490EE1DAA69E7E7946488B28E33CDBF5378DD288427ED276FAC685" +
            "12E3CC1D4D5F6BEB297EF639A331970B5D33C723E7F66C64FEF8907DB8CDB3C4" +
            "BE64B5DE333E0D900FC2F5AF6E782114F1A6704D58698C4EF9C796F4A054D985" +
            "5570D3E9B268A321E7A1F655B77EEB45C198EB2A7B6D11A30E816ADB8CF5FCDE" +
            "29ADDA29CA15E807862E607DB8B0E4A67A2DC4C167EBB0EE4ECAB3CB3F844CC5" +
            "528B569128F27F6ECE1495E7ACFC34FC114A8906A872BAF40CDB8425431FD985" +
            "DC0BAC262D7E8FB4F71DF5151E242C4C160252FC56DBAF16EF1C082A2E1C32BB" +
            "992E3A12F21D86259A37FF6528D1FEB98DC64E79D0901F9873372D0C0659C917" +
            "A2F1CFA7176EEE82292A58EEFAFAC104FA61AC124114C4067EB1211C9590714C" +
            "A0F67B315E597AFD1BE546C61E58490C093842A926BB1CF3C4D27395C520A707" +
            "294234316A3E1837E6521C2565845E244345790FD53DAAD00673596D66438CCD" +
            "82E5EA088AC6D2928DACFD3BBC7B40FC55BBD502B6410B575AADE4D50F6DBDB7" +
            "94A24BAAACF49C2CCF025D454A1B7541B159CBAC7F12D184B7ECA8F4A7B72805" +
            "EE10774B5501BDEBDE4A61D0A3B22C9B0F303B301F300706052B0E03021A0414" +
            "95BAC18221F16265AFDD6688C9FA59BE80616F830414C3365AE07D90C0CD656F" +
            "15661D39D56F307AB983020207D0").HexToByteArray();
    }
}
