// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    public class Kmac128Tests : KmacTestDriver<Kmac128Tests.Traits, Kmac128>
    {
        public class Traits : IKmacTrait<Kmac128>
        {
            public static Kmac128 Create(byte[] key, byte[] customizationString) =>
                new Kmac128(key, customizationString);

            public static Kmac128 Create(ReadOnlySpan<byte> key, ReadOnlySpan<byte> customizationString) =>
                new Kmac128(key, customizationString);

            public static bool IsSupported => Kmac128.IsSupported;

            public static void AppendData(Kmac128 kmac, byte[] data) => kmac.AppendData(data);
            public static void AppendData(Kmac128 kmac, ReadOnlySpan<byte> data) => kmac.AppendData(data);
            public static byte[] GetHashAndReset(Kmac128 kmac, int outputLength) => kmac.GetHashAndReset(outputLength);
            public static void GetHashAndReset(Kmac128 kmac, Span<byte> destination) => kmac.GetHashAndReset(destination);
            public static byte[] GetCurrentHash(Kmac128 kmac, int outputLength) => kmac.GetCurrentHash(outputLength);
            public static void GetCurrentHash(Kmac128 kmac, Span<byte> destination) => kmac.GetCurrentHash(destination);
            public static Kmac128 Clone(Kmac128 kmac) => kmac.Clone();

            public static byte[] HashData(byte[] key, byte[] source, int outputLength, byte[] customizationString) =>
                Kmac128.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                Kmac128.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                Kmac128.HashData(key, source, destination, customizationString);

            public static byte[] HashData(byte[] key, Stream source, int outputLength, byte[] customizationString) =>
                Kmac128.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, Stream source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                Kmac128.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, Stream source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                Kmac128.HashData(key, source, destination, customizationString);

            public static ValueTask HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                Memory<byte> destination,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac128.HashDataAsync(key, source, destination, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                int outputLength,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac128.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                byte[] key,
                Stream source,
                int outputLength,
                byte[] customizationString,
                CancellationToken cancellationToken)
            {
                return Kmac128.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }
        }

        protected override IEnumerable<KmacTestVector> TestVectors
        {
            get
            {
                // Generated test vectors.
                // See KmacTestDriver for how they were generated.
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "52");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "63");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "54");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "64");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ff");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "01");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "fa863bffe6ea9dde");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "6e54951c247e6608");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "e37c286ffb19260e");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "f31dda46d3cb2d67");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "4f93b758e4536f08");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "7f6f9b7a2aac2449");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "d699c86e7c08f8d31a87661f6089d3ae937c0f4f8184dbfa");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "5eef59fda36f3d2b6288c95eb399619d562e6dd058c4ea37");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "9dc402fb76e41e17a0cbe03296f8b2b9415ed04cdb19f784");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "ae42d2d96270d517e0f2cc018f9ae5cacdf654bfd59e69b3");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d4f887991205847bc5e70e1607e6d07acaadbcb078bc7504");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "91b70dee1889ecfd863763b346e6c96673773bcd21c9db81");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "4021497fe5396bb803d18e812a565e78493c01e6f1a24ee147b215ccdca342fe0e3217b6b68fc06d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "a61b9978e09d13cae290a1b35bfdeb1d4f8e3158409f8b225e33b3ab8f4c0a27471cc4b559041e4f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "dba6ff1463e5615aef632ad5c3babad83a6318a2f40f3ad82b5c92b37678858dc2dffa207ac8e168");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "32486d43d447963c0ba77499a10b2f73f46aa8b0e02e3d8f8f6359c2f96c4df0f70d142c4df60d9a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d2463cde8ba97f879ef2bd1a7c8ae8aabfa6f9457e3e958df683b0a423dbb9d7d3ffc7e554d0cf41");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "23600d09083ef7897c4e8bc62c9b7a29c5b72edc46cdfaecb553f605e48a990dd8db87adc44c0053");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "19d1973758677165448168790f2dfe1d931fca0518da6686a61083cd2ba722d6e3e2be0d65876ab546ce996715dc3db054b4e620a4bf34c7");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "8a2bb6eeea3bce00d1da040fecd42486c69a8832ae1dfbcaec182fdc939554d86599d0246aa6790f318e976eb19bfb78422de8c6ddc85180");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "e241e737df9a0b01b02b9313e1318252ddb3663f128120e5efd10d4f9761cc672c0f55cba094d379a5182a27a28bdfaaf8c82d787966e64c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "93aab4ece42dc48bb535e885c50bd91c376907f602c5b977008c2a913b56c35ea72c28bc561945a4de4809c935c4319265c0bcce893a7d2e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "6479f634eadb7333695fd94d27b5442cb502c0bcd19341da5321bb9c2d04b990c3f7d39f95a2f963fbbd06e1c5573aa185817664977bd164");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "23821a925a91eacdb1769b906ccc7509dd24a02dd9058bba4bebeecaff0f7244b401b38061426182bae381f3b3ad71aa532beb12485f04f4");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "d3c54bd94526ed06223a303ace704386855200170706a83ab327c791e112694b59930d543c0c185330e710f9e506fe80876e3983098749d62119f7fcecd39f7321752e272813d28f");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "511fd961731f30a7a1ef7c5884e35f28986ba46088afa0fb6c336629556d20ebf7c97bcd1bb5f4e6c9f1a1822610d2ed6c37e67faf44427b5bae0c300c2f61105e6fb8facfdc79f2");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "92f4ad1920c986c507e4aec772e80c3ef1d8155a5a2398a31b0104874a96d82f1bf84e00cf97bf27de22c7b9fbe508d56f9eb3a6060480b284d422d25983eb7c6cc7e8c85ba3afa6");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "1b716a70bb37253b66b90031fbb7a1b5ab20a8ece65cb3bcdc96134c1cc0e51d9dc2dafecb983c878caaae18a78e9acf3ca4812f5f26a6d048b161d6b0d507e05d9e6150c9a00893");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "e1b5ac41f6537e6dbb99cf060ea7660c6f17386508abeb6c0c122adfd57f0263327e03bcd44bf6a822f9015c6e54997d33d1974ad853931be1d447eeb19e4cb786ebd6247e681753");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "ada7bcb3d36162aa35c56c69d96600a66c2674e2f94ceeb62550c80618ae9c056f1a1ae5f4aec6030c4498df8400bf19bd9d8f67313a20fb0d72b22f4c8996272ba396d80528118d");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "3aa51d1a167a9b4e6710d6547d6b54d53f6eb130ee1f801c1fb836cc4a400907b9de3d8933624412e8b8f958894f0fa4ba9dbb2fe011f4604c015ad420eed953aa344067feedead98d5f30b8ac3008dbd27dc442e7156d68");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9a7d43a600e5bcea71557d33373499c371f8641efd70a6f4912e3c45a8a753525d09f7446237a752b0c5e4af1705ccfa4f6bbf9a59a5bbf72c0f2a5154762f1f93796e03dc39cf86af00e4cbeb388fc5761fc0d1942b8bad");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "effd6b5d314e137db6278fb1f23da22557bd6e5e50d01e9017847722a31389ab22d162d6d6b992f836144f40c52b21f3484c475ea393d0ead3d52e74150516b753047ec1855a74af459ba6367f37559af660b14d35f71dd3");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e1252d3fdb925a42f4a4b4e4fc6be5de22f4614678cca70072de8b9bbf4f0f592b860d1530089aa29bc17bfa7ab27978060c5f1725f3a6d70bdbe419feff564b8116029fd3d33fa05571db4735df7819a912dfb98a5fc256");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "df43da4204bb77bc8adeddc0d740c5002736fb719f14dbb8509becfe4a80ee044b44a107e82cc6c1453e546a60a7673bb36cffad6c01e6fec205c04914c0aa408d3749b7bd653b1d914f52f86d1292b937074bcd354cbcff");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "11fd7e011e61fd32281a585476c8c7b8a92ec8b4ef19e1e37cb43e1469494b38145783285950a467fe3e027b483d8e31e3ab7d99ceff666674cc39af7c42abcd6721d2fd0a0d10b04dc67b806dd10b8c0f6e8aee761d89fc");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "463aa7cb64219e12cb90d26938c75c290586b82ad1f880e24762b4b4d196426e03f56220cfdc25da82584da7ba3e6e062f7894147a5e212f3aa7a96adf9d4fdf8bde08ff08a0b1c608ed2fde9287189ece9220b2e297d44bd9d2f802427cac2f62498606cd0e2784");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "a4fb92359ffefa17be9570339bfd99a45d23563926b953e400902c750e293b2f5ce3e926eb98ca866cc0f6f8c9328c6ba1466812e8bbadb29e188e904c4223e7ffa0d74fa209649b5d598f640a8f45d35b9abf15914f6551e8fc027434d84c9e99bcaa5ec985ecd8");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "8ed741e46389fa6281951790ab7a18ef3c7f630fde7ec1f1c8978eb0736165e6b6d9fce92368609d5ac080b38ae80d660ee2c782f078c697ae232d0a554171b513940af456326aa6a38cba39842086fef739ae2ec3eb92cb47ad9ce5b58eaaa7ebc4dd0c83ea5ca0");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "599a9c567ba9ed6f30e359c8ada93a27c027eff6443aed43831b7fef4baec14edb76bb12e062c7313e0c454b05ec0c1e5ef43372159521b8ef0397270d95d2a7eb5d5f13a5d248fb837048e393ede033269437cbc02786128f452fdc07e4f93fe9f98348e4afe48a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c4130e4772da6391d80c28ba7ec22f4a82d31b66cc502062b4f713dcf14fe96520ea854d4c1dc4286af5618eae9b64ebe1339d9b0a83574adcc5d4e79d66915d6d41e16f6ae0e2fcc9eb6a8f9778ff1ce68ac55248f67b232fb9ef90af335f5f33b861140b2441e8");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b186f49c93ed425ab7803f3fbd562c8a63b33a84daa6b17d7b7c9b860343bbaa81756f205c76025d55866e4dee4a7c16aed21c9c1fcc533312fa5c464584df1b11eb41acbbbb9c511ec3b77fb885c9512e4a46352e99832397ac9d26b5c8769f465d226d0ab19588");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "40278a025f8d71e6d1a8d9d15df251d320078c070ff8d12ce1e284394e1090bd5901f101a12f78a3f4920222d65c7be3f3fe26e4731b0f98d8b552f45fbdcd27d9d12245b571eb64647fbbeb366598dfb81b580f587707314a1d9acb8ba55eafd41115bb12d903a0708d5ccd130161db0f42e96bae53961e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "96fa7c32d7820eae7d508e7d5fe09e653a7e2d1a8d39fd87e299eb8a511930465686f94bc784e9bbf95b04c6f989445c8d87b6a81474c3376140eec812f50ca3f66a37e934c8b0d0af27c4d78b2242ebdcaf82d0caee0584994b9ed9c399d3087b090d5f61ac11a6cf2e9386b5a819da6c908d8b4a99fade");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "da0c28b3a361e3973baad5a8f0d7feaba3e42efb416cb8e939258339e355604df8d16a28ebfb853fae64278671e9e60115a08f61817c4f133d28c671509ed85d154ac212d12a09b6f36222d744584b076a9c8a0b70dd106cf46b9604f67e04eaca2b683fe19c8b3a8aa52024f78d738d9736e3b3ec29f0b1");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "8a7d8a7834be453c00e442ae80d300dea9cc346e1d3aaa9d370d7573fbb7f7ee1e3f9ec6d934bfcdb9efc5c68b512be77c865dff8637d872718a2eccb6ed5513bddd2975d6d38c8cc58558b0e0eedcc1a4b93c4a758a52b06e1e9f5b91add33d2f51367f844042750d4fdd0ad6972b79862989fa9b3a46e1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "98e3f41329c287a92273f1c5002c5c30dddb30cfa4840b7951f1d6f4d05d563807929d42ca6af808b092f758f08707ca178fb5ee93289c6ff3a82bbbedeaf3fb6af5e892cf6c021725b0cef13e8b3c4ecc18c8f94da08ed10363a803d65be358924f90533788681a6b9d353f8f9899ae8f7a29505ca6d24c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3485cbeaa48799830338b6dc68a7084bcd47ad1b97aa2200e464db63f1a67c4dfcaebbe8fabff1b63fa8e8dd0d04b36d08bbeca56efaa888b09d5cabc88992b17c3f478bd0e7afe6616a3d3c75fdbc12f69ed9b3376b1ce5a7fac5a989143f35fad131fba50f24de1a5041711b120e064c6730134cdb01f4");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "40eddcfb4dc8a7d852a77f094c1fbd1e0300d41f7654bf15245a8bc292129a01dc21d198dd63c20fead4b01b262af6c97b4838654063d35c3587bf951c44d4f9344a0e7062e535df24ae901bd02b73c359d77147d0d2f12f703e221748d79cb6ce92b140cefb2257bc0482d8a5922047a4414720d850687e5f9e2351881c3c5574d2982b9ddd9edb");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "e6f4ce9a2487087df39344abc17c9ebede5379ac7afe7879e9f60f46499976a777a1da06e3300c1f3528870580660ec7c12522daa22274a8fb5ffa2d6efb549667c43d25830ae61f408ee1450a48355c525ad9e7623c69efbda64354b222a665fea8a404dcd3ca1a776ea809d14cf104478e53eb6dc0c36f714b461e1c4a64997eccc0bd8aa32bd1");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "eb03b7adcbee6bdf55d340ce0fa43fdde9624075d196c857c7ab3b2493f76e39a5d856e152f7ab5b20a33a364b317bc666874d3107f270375d690bfe2ca8a2bed27a4c7e877e0bbf75780a03c9d0e7209794d1fbc65d5e7cd7c839c2d9b6d31da6ffef3ef02aec16d2d793b964fa520583dffede4e2f1cf879299a6405e7045b582f09c2f234c078");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "c812a9f2a5b82581ef3c46ae8a7fe1a52a7d7c5b299ac031522ea11ff35812ab0b862eec37eb8d4ba387bfc4dcc26ebb295ff0f0953ab162cc7011194d482d3a34b1d26356d2541ec1a123bb97e91a099dad9031135b3d8a8290661bf745bb3e16df9cb60234f2deacd22c1f14b5ffe90ddb4bf30c009e6ae073b9ab37e15e506996f43e66258794");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "748f916d410a036c825aac712110f0fbbaa01ac8571e88589f5feaaf5fb522465a76e17013f7515dd6ff2fd8163aeb36b2bf415ae43747ff10bbb0d20edafa8ef96f9d5f6d86982b1f2519bb86806c284d9a369d37aa13d5ca205b52566847f6b8fe43b25c9af0fd2f46b572cd7f6a2b17e1b9514d3db149dc2f25074486a3a1fbd694da45db0db9");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1f4a83ea2520783f114f01e9d5cf78ced51dd12970175082c04d450289a2dc845dba1def51d49616968cd28611d424f9730d5bd30b61b887f0bd4f18765595eb35ecbde3bcb220961f572fea8d1d3fcb818b0b3ed34eb3b486cc53e24aa979c9e3e2671924ed0fe5d33b0bc309b2a23ae5f225e7a55e5a46bfe9226a808e8988e06a8d86092de40f");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "038dd62f2a1023301186a3d21445004f6c826d42a532ec0a91eab0dcbafdde1c09ed811e5681664ba9daf9b1009daae4fd07c821f454f5a6d2d631a964b97b28b6c20312eb7172f0e71312b82ffffa6c09e508ce73bff9ab357d84649b3e181912e03f5d379629423eb0ec8ec5a34d557fb31bb63dbacd46d3fbf0b38ef3e76c94e1bbaaf8c4934a9720713c2b4ad63099d2096b428423fd");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "4da3ab1a49e39d7417f91ad6f738e6b07e9dca07ba967efafc9182513185b669dedc1d614b1bfdf96728d26479a019d77a5a95cb6463dd30fb31373da1c0366af214b9f185566d5bfae58272b729533e60e9c3dfe9b45bb8b257ede5bdf898187ce2063ecd71565a136301febb97d9a86301da1d7fa5ccc74c6b5e90fcfda00102f7f96b4f350c35415e2b17e00283cb55458d5696fa2ddb");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "f0e2717c5bd3578205b951c4a10311c1491dd230af356341fac3b8998987503e9f4f09ed9787a4f1299022d0535cc2a131db9ff4256671790f17d6a97fb037e138d1915d0f855bde94787fd5dad15c24bf972a56505329d7bdcd1eabb984a6a82bb784e1c8d3564167158b615958fb4df5e4cc9cbd52c73e61c4bccfba25a1f7e9dd8d278dbda6a567c4116495347da0fd21aa90f3a2caac");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "422649f8b913f436b0c1c411b30de536dce3345b5aaa4e9f508cb10096c94e4c38eaa8a74f574a61b7198b49efafe4481e0740004a7a1c0502e46785da78d63fdf1b0d33a146da3f915239b3c813747d6a0334983ac9d7cb0ed1f9478d286f1926064bcd7393a9e9e4c1c2b102299d9d409b3670ee6932cfc6cb7b21b95effb21b4c55c9f53b1cd87aa3a1368bbda64e858e5307e7c73f52");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "05d1fe5bd3c64fd0f5affbcd3facb972de0343832671a70507f90f1070094415e0505db8a418275ee7593f0da50c07124f8e6f251fb6f548b1e789e7a0c11934cf993eb57a9cabf5f831bf941498fa58fdf3f1bb618a9f8ba1ddef856d9d672f4dcaefe6b4650c11f17fc8bd9b83bc75983338acf32dfc4c8727ff41b75bab6aefe5592b7710d15af97e81811d3f869d58ce6d6159e311fe");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f9b6ef22007b6d56f24e8aae33f6d500cf13774b4f13515806c553e9670d34a1b059791fbf033b7f1587c2e81e830206162e171623a4833b7dbbc7fd7df697df596811a27290f9fc3cda7d3bef86ec8d995cfa8706aacda5e081de3cb206faf72ca7334c77df865c51e602b85d1d977e33f9a73df97818e066ad03b3a9bc0010484673b37fe4b16128dedbefebb279ed124923af9475ccff");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "b019f539ec77eb2a8792b87d7ffac6076dbc65a163076b26304a27b027f7c8a09a5e6c1ff1c1727b4c140441d4d9710f6f877bbed407d7f671cbd617260e29a4a1bd209b462fe32a2df774d91989c03ec67fe484752192338d2e2cfb76789d31153ef3b975ae7268905801ad5f009f37ccfe17169d250aef58573075acdd47e1151ea87d943093135800cf7bc74f4e93f7f1c41ccbe09f173b740d2f57184d88d16620195b54aebb");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "2d38eb5d74eee636bddec290706364225704be2477ad9d542ed07d3a5d2a374e54c1fe02fdc73e474ff8b15a35cbfd5876af278675f4ac8c28ec51d45184c4c0600cdf024145f686730d2d037ceb01a7b4b8be4ca007985a4da3d1fffcb81bbed1b4063793e3ef055d73ce74df185c0bc6a9312a079d4d536b9e5591c586cf74ec90e69f1a44e0f9605f8a27680531c4571c9bb1968be1e7248afea2be9a959d35cbba94f0d38782");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "9c17e3167953edd781d2d5c19ff050b4d7db2519aed1adcb8d9af09e723799127cb6e8b20cda0a2f4fdeb3fbc85c70e4681fa170f347b14f9da6418145cffc5dc106708f906e22be46c8710c08212f7117772ab953e15356082d56075017a837f6eb353cf2e0e1a99b4cf6b4bd226ecd07db29203987ac1a356520aac6901f4a187fac9beeb9df4fc3e548bdfb09a0d958886acabf32abaf69cc377ac45fd186bec3b80088e4e625");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "755f4c5089bb006aaf0a7970b38cf3278b05ebcd3a864968d47858a5164ac51274521d9570e718f427cfeafdc2b72f48bf0da0af20cf8cdaad5a3fed08e80c51c02921990987be817998bc155219b3da8cf5f6bba9251daaf1aad54d2794dfcbad11d87969e1ed7b9b584b29b8de2bda68208ae995c9edcd8ccb939b00200533f8f4361db2b14964926e8f51b3cbcc0be2810dd2d4729e9e2e848680223c1b3297e433f1a82b15ce");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "568e24070f5210e789202c02384bdbc734465bb1fd6ff910286987f1d6bb6716c8b29f9dd59b880c3e9516d30d7fd9020f1459f812f99a851685ad6745ecd5ae8c6339073bab31ab6b6228bf345950061727778d87fe9147509198e4bac4b9bf5767cd1d8f24e4b4c82ae117c7d86f7f45e2729ed190e1b790136c7f8b0f2f03fdd530c99e915ec7d3203879efba84cf3556126092c90cd2bf27ef4890f18661daa18c908c60e52b");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5b2e598cada9d0ad10a0e10ddb3a2f7e2793573dab55f0b6c691e557c68a7be6a5198f7e4305f9d29167ed7c2ac31dfa85c945c327927f299a06fca7a5954c2d6a69a8ed5a2a2f17ccee4b4db75bc9b42614ab32f6758e0217834fdf0973209c9589e349566d383cad16c19e1b38d83000d06610e75a3d9f186be2efbd0d347c3063f9698e29aa713f2fa020a8290e05ce3ca6ac16a261a7ec43f6bf133dd75b9bd58e48d284504f");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "3b98d1d4b108a9473d4035bd72620a94bb901c78f9956ce392ac025d2380d008999b4d533341ecb316c9ed3e1ce9318afef99f3f343ac6135b7d90b22fe3e2e8805995153855b19cc80a7a0ef3d062a069cae5415f36e4544fa355c4a77d6b4b573f122d398fab9e97999939d1074fe344687f828f6bd9c7709b7a5bb653b81dda542796f5741f7e4429faf4177317d56dac05baf8f9f33de5033dfb26ce0eb4c7862d29d46c501664e78fa13252a69481d7705328ceff05");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "88a13173bfca0be5ea7796208bd2a412140dc8cb58077d79444ee8f4ba3e2ae0ab8c8ab8ab5f4b8a48566f81dca1e5abc2b1aba02ef4baad4d3e1b88bf556851a71d838c31bf8cb625b5df74ce2221a66add300dd2f2f14f160b24c96068abddbefaf9652f3578c136fee485f7ba16c31cc98eb57cff70f9db1e5074981d1c1bb5415211c6e50232d31530f73b5fe9c7aa28134d9c9163498917f6f510d27f2ba417fa01d3e5203b0bf85fd6d4aa0155a3a25430e2de1072");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "d640f0bfe90e6b1d3a6de18e82d412dfce41d92c7ab77922954ad45399223f4a8e1c60631f0ea8fa987add1fe79b9ab0a205c9db3ca609eab3e855c2ac6741810491064147c1f46bd9ab6099a7864d42a5b52815e5c1d09f6418f9257b20bc563922d5c7e3b246080e61b385f90946812ecd89e78901ef9ad6316af920a4db4e20f4c5233c6f08383e0b27c64ae1c649fae12acf72bf438f2497b220704133bc6ebeadb133d64cef8ecd4608fce8984a23949805c1c6db50");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "3d619b0bda2932436a2d301f76ef56281008355decd2028b3702414744b7adccb8a7d60d78dc70d532bdc45c806b81981661bf7a27318040b8c1aebdb76a6991d9d215345793b75724454715267e922df195f8d3c5fe978c751970a53b1d04aaa8c5d9e18fd5e0a76e4f89bab5ea1a8728c4a88877fdfe5023f60e2d535de28d26c730ad55611b01b8835353111b66d261f996e4a1e90738a003c24918aa0aadef2395a5a8cbecfe081f2b1e812012499f0ac9b62dbe7e17");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "e5b7e950bd79fb025084c0d4398439db1554b0c2b4d801ae257bc301a3a867b47f206e3f8aceb6aa497ed9c8978415451f3db4bc285c2c765645590244954d26ced531f8892501e0069f3214ad97ad6a6fdbcc266c4493363e16273075ff89de9f5b6e114a82a22a0fd11ab22873c545d613ed3d77924995b9d0aa494bcf4d056459752063deec6efce7c0fe774d07440d6ac472c87e744a9129c40d1b1820c9f7fb7d2d6c55fb200356cd931120a512e5655a0598a37155");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "42ca1ee3d2a5b99face4760adeaa83072ba6be9394a03cb4b1750692b84fff5d240ba1d91e0a76f6ec6983aceedf483addc20f970b124f045b755efa0103f322ff84c3bdddd4d76e54ccb20dff889196408f890a631d6265761d9378e58016707ad9bf1c47e646388f0756249e4b8bcfce0db59380e0efc997827b4f00c07f12a6048ffbfc85d5fb9f9161d2dd50708f44ce93268ded5d0752b8101b8fd3c3e19327c3c7a72830c4be7dc8588fcf3b0f794af11f8f395059");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "78ab48d16ecb5eec220a20dada08d6dcbdd7f2d5bac70161628f055d02ed492f834e4a2ba33ddb5fa178c280180f66b3f3cc0c32726f368b06d092b68f6dc8a0fef344794d59bd7fe9ea33a0ad8df6e978814a85e86ba841eba9990021a0b801e15ed3d6c48239f3972a544237743824ac0ad5253dbbe299a519b489f9b0adef5c73b7e4ae8bd3bee3758a48ffce0de875219a6e3d9bf661e55e32cedbb971ddef98ecb363cb4872821efd7d6988b01eec017ca21eeccc53483f1ee1edc3a7b02a9b911fd2e8dae8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "a532fa364fa7dfd7722f9f53490a293bfc5f833b98d1e2dbe93dd6a8d6fd894496ea1134d74c204b9d28e5c400d0b471eeb9e9bd11f6bf4211930022025a7d4f067140c30b8af15cf6af43d12e912d5bffe4a65a2ac115f8521a8299de7bab89ccac1d21bb2bcfcd1ce1221265cda5f080fb12f90488b458d230241a8eec775dad0c6deadcc4dd54141dfd85e60e77a7f0eb8804ca91809e5f2505e4150c48cf895ddb2cd56355c33d77ce72808b1068ea9c0df38ff58ffe5a39fddedf3c79299a0a58dc993260ff");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "c8852099b88d25d051baf28a2251669cca7eb9a778c044a53feefc293e4fe529de30b2f3711df277dede008a0e4ce1c334cad1c986ab804e9984d915ab6d354c5d99ad8df1488fcc6dc5f85e99d02a818eb1fd3a32cc9e29717e5867fa9ca20cb175efe34bbc77cddd7cbbe66afe45962907f5f605de393db6c8256a41126d46304c690490d087163b9b2fcf7451003a7292d84a5036c6508db578c8311f53d89c51d1bd109fd925878bf25be2a2332873abe4c0f876eec9969874983286f5d08eabab296e90e83e");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "f7a300b240cc4d51a6bf9b3942fb7e837394dcd38951d9008da4024ce43c5d8796c03ea24c50a44574d29bb62101d041b68f88a6261d12b045dc0e93deafe67bc8a52ec334e3648c364c989fcd4bcd156d8ed692567e624d4690e16e540ab8b04e5ed21eb0756505ee2406324ae0251a6c99274ce9abe796ca679f243922595b41e61a06973ad0b4ea49084346bd8824e20550da451d3f0894aec7eddb58a878d437814e3aa1fe1dce63b850a1f54efce8ffe7a38ae0f7aac27f505c8cbc7d92fa695b6cffcdf658");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "dfb9d9f0f36ce4edc4f0124288654dfa7a86142703d55e9a2529071e8b58fe1d0dcd0f48c50e3781cf2223454eb4245c82d278643ecdae89eff68c1f8aee77cc3180232804edb5073dd8362541a36ad77761b20f643e07ce0db1d9293dceb4e71f8d243b78171bb11f4cab9c6f46955585254193cd8713147150f51b4f5c92c855f45ad1dc028d6958d7a0de8f9aebc706c42d51c4a7930c38e3a8132b8868dcce01fd46388a5c092a78ec93d64fb794d55414d079bd1d13d49388cd56026ddccd6f6eacd1b970da");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "41b883f226281d68182d27042411abf24a5bf719dd802781dbfc5741d35e07351e0d5d993b23f16bc975097bfb03dd08caa726896fb2246f0c8bc9e2a342d3568fe1f7709c275d183c086d22aa552c57928d951483af6f8e3ffd5bf0056ad12380eae8b41bfe8643298bbc01ef09d18f1bb4cbf9d5e0ccd673b81d10e9c519a6ca19b3551518eaf2b44be19406730865cc1e2fcb8e79d54c08261e095d4f37eaedebc00f724a305467bd0172d2b277235bc6b6c08ac41d2ad38734dca64fa8e2489a45e62a1425fb");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "58d0d7a32048c578cad9a644df4579a2e2cdcb8c350390e4683758c59ebcc113cd4e9ecb7c103ac946c7849aeb9eeb7557f80678c4b71bbfe4728b66e001ce7d19f55e8d234d4f308af8aee279c71fe7f399fc56ac6d118445e132dd2c77bbc59cac869dc8c5d9aa4f2dd887698b5a76febf25c597b226a90ec8fbde9edd2f29047585956a5f130ac4de58ecf36b68cb867bd4737a39ae4f6bac6f3235c8ebe1e5f5e3dc24bab44654a7ccf43c6a999e44f35555cc097efd92dfbcf2329ab2c72062a9245510498c451d9dd043a4559846b3b0da305181fc");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "9ee796863f591c1081f2780dbe27fdb937937cd897079b50954c4f4682a0daf945d1408021ac2ad99e7c41f1a71b9fbaa67ee773b3de51941b197d4ad4ca64e3acdee30f8f830998b70caad400e288ed3fe853f266fa22472f810a5bfc2ce1ef8d0040a022ebe6cc39a6272f33a95bba45a2af1074517ca1dba1d79e95356e37f855d3637921e37f7d301e98f0547dac479a7a7f80dc11ed3c4987e9ef28162fb78535f8103216fd58c15fbd11fd067ca32ab4afa6b62921e9dd528e22607d4b27e09aefa83b53a93594435b20652ea6a2c9aba720a073c9");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "b74fd7a63cd5f40879d21e05e0785b763ae37275c6968f1ccd1165c3e5f0fbb71de50ddbe44e35c76d9233c15ae7dca2a830e1b58cadf3695dfa82fc41443b930e58d969e89d9d806f20675389cfcfc07c4b6f06611b308bd8c1218c573319ff0a4c2aaa16f46419a07e0621923032483c28e0ed577f227af86c7b3ceb1ce58bf25616f2393aa1a8c870f20a752f4e0abf24b2b63792290aa1f92ce6925312efc16aa07b1a781c4d56a5fb24ca294076add7d97d69ace7dc9f862cd3f0937518f4048fc9391f3a116d81e91f034764cb949cf5167d5e38bb");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "3664fcaf96b81f5e46bb2e800a06ef40e8930c5fa465b9d965d785eea327091ba75e47ffef96345753f4b6846e555ee00870bcf0674f3145bb835e007eeb1d7285e3189695fb20fe454e048ef1cd72b9b0e6743eb3c3a813f49adc0b90808cc701ba3296a9c7e95ab480eeb84c071ab62047e926a0df10b9ceec2bd1de33a8ec7980e3a1c823bf1ce697111b6d73e6b5fde8338796fd051f9a5fd637fde99cda41fe0e6f60f3e91d3aaa1455cfbe1d197563f8c3350c27c021cd2402e377228e25e5e72d6e0955fc35229f9796737894a4c36469cd0edac1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c6a9de60a1659d2fbbb01918a075313b62073f229643a40a0c51258b3f5379c1960fe0e56e25ade55948c5736338043533b68e98a65f045ca021ded8fe31ada2e69fa7ba6b8e77478689aad96025a4395efad33abe31ceb46ab6e3d636da3afd880e2018470ed098d894ba5e0f260998fcc1a898c0a29fb541e6e06bee2c5052de2fca7aac07ad7b202a710035e42e9f3730487346d017a4d5a75dd8304e40bfba9b0d5671699c655b7520a3009abfb2330e93c5083af935cce613288c952ac39e8f257ee8c978648c82d25d0f5d3f621799ac5f3ef26545");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "18ff3db3832931d66d4b45dcc336689a9c4b53868beb14d07fcbf63c9a29877924c61a9d988bf746ed4477c5f47c14503fb571e17e3e4a264acb69cb3c4aeadb004e4053db6fe6e9b03dd86d783b0ed48ed55cd0fe77ce280cadb60e056f867b34124a64f688ec8db9c0644718a88e6ce7934e3c2a25f258aabbc0611a1cce789f3239b8674510a4e1a7e7d1712c1321515d1dd120ade44dc9936a1ea5d201cd42b21f1a8bfa7f0202e1bbb06197cce0a8b5961e9af897ba1e68aa92d1e0100b093908de6d945858cdf102e15b7df9db58354526bf02447e");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "9ad69fe9cd3dfcbd1b806569ca2e7c09140aba775379a7d54020e706337f210bbf15c0e391e1c531a80a4ee526b71620e8873d67bb37d272da6c8aa10097c41a10274fd5a4580331bf451cfe764b64036cc93d39c46836ca8d80b187a20dbc8f2cb6a2bc78baa644598dd34853bb518e3c8e0809cbbb1d1ea7695ddac58e6bb3bf10ed50ac00a36eb9d41fc2cb721a8a99f7ae0dabf656f9c4afe280f6a38d44390efd4c6c802f8e43a15968c85347a897ba1e6d627cc29b363015a3551b8a8cbf32130338cb784f1e382088bcb9c97c17dc89c6a16d924ea564c7288a518a6fa67305de0465915c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "c3b4f159d9eb793f5080ba6594c8ee1c3c9c60dc65ffbea5c1b2646bc74a7fb84120525dba957ebd2c7b73ad148dba27036bb67799e5990f5d5ce90ccf1e3ba6a80eaf6d83cb3f9429ad4babb4843368683ce5c229c15c51ea332fe79562844934802bbe4ddd0437a51376de26694de80f54c2eb21e41f5378674f489820fae7115237f367cf0064b59bbc031407011f9f88a43304c7e846c5059586f0f70efbbf8d3c4553dd887799ebf89d1643651449e7a01abde87b078db79a6fe907cf13fbda420d20ae84f9ee74e127a8ceefbad8a8cf53462c74bbbb418f39e6e688c736641010dee43b10");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "dc2c0156c61aabaaecdbe0bf2d5b39c2205b28c26b5043cb38b8da4246a32f5231c48fd485720130feb410babcef8d4d8ddc2d34f71527add2126250e43fa0632a0caf91a9b3815d83009edc9c045289044a1c75cc4a934c19f4954180693c6bd40c206b43b371cdce25118cca88a94d4f37e752695b03678c890623ce1560c3ff70c57003f121e3a50a9e2b89f296de10ae24f9f147280919eb37be2b27e7449d60cbf76fb40bd0d80a9e4a5aea3d61ff664dafc188f039ffd4efc2066e1be76eb3712839bf59a839645f6f4bc109fed96b63ad18c9e0449dd060e2d5771c187ce2167c8a097277");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "1dfd3c6b3254e830935cba9c1d573bfc8a24c6ba65c7f2941e2ca31fe4a9be6035a2c049e69502497c9086469bc481b1db80c50ca9e2b58b4629925fd79dfd5274cb6f5f2b4ee131199f00b6bfd1b0b002f81860c006242aedc1f9d1aef95504d7e1266c4a5d91c9632fed496c8fe1225d28916da697d9829fba83fd46e886eb24b3bed1c875bee55eb8feeee0eea8374d96f6b07f24f488428a9d8775aa8dc147985772162e648f180501e63174ab2f690584f02e90ba65ff68b83fe9a7d42d8502f80644ce24d61b83ea60e3793806faee435f86f0c9d5abe0f775d3e3460d581747f37a2eda39");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "5d1752d4081f40c07328df7a53eb161b4d74cd21a30d9d2947957e165e7658c2d1bae2661fb4f833d7114b5980c8417ebf0d0f949cb2a650a7558950de505f60d96057e0783b53d2fce7bef139b5fe8bde184d040a4de7ea303413f66e23c2d7b15ab6be3593cf46f42b123eaa100405e9b7ec1a818bb5130738f64bf42bdffa4a7dbf9f51f2a9d4a78d27d55083c5919d78de045eaf3685c6ab52de7ddafc11e7fb73e1743b472b789663887ece29459c8026ea1745f70493c99002062afd8f16deb12d0727f49947d39faea59deb9e15f153314ce341d05ba5040b27b7c4eb3005a0d0c2572441");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "37e8bcc18fa122ac8b35ada070f15143a41b1e31116ca025ec6077466117953b9e01e27b43c7229e5995967dadba13b4106b25ad068e2b0d7483d761227d5633d77ab50deb2c6f73a753a17398eb396278b0dac7cfc9bbb2d16c8302816c18acbf078083194b084c4e52d47c410309dbec1ab4629454f8ce7a1fd3aba2802c9d527ff5d5321a4d9129466953ebb5e11a780687da25ef81cb5c838df3b34473083c2e6686aaacdc392ab44d2ab041392b7b1ef8ad542c530193092e3d8a902ce357c1e93d041b8adcaed4cf483472167edadf008cecfb3af1d5204b0fc072105fdcc39e959dc248e8");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "971c7bc53a92c33a4c73852d7516162c6705a6118daa27e5129a0ee47724686763bb584e9d02719146071166646c99e1b1ca4125eb6ec1657607046acefd07c6f195e4252ccce36bb6487a6a0fc743b704e2a1f553d3eef62a716b7d7459307faeb7431623dc905d8749c07c558ab0f7a1ce397efa4403ee4e3dde3568246ccb70560b7371b062cc823da927f483af7a2b91d7d7f8c87d405129b044e8c3cfdc935909f018a6fd12e10755269a83fe13d67645efe76533b8b5a6bc163253f130abbefc8fd3b4acba21257f693fa89b2613d06ef63455c6cd47e1d9f911ba92d1a98b5ec674b5c34b97e40c2668f5ed71c1be771786b5e0cb");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "adba9936ebc76e02a614c5bed1828e37b138688b8d3c2adf802dcd36d7264e932fdba4d0ffe37d76dafecc1b898af7f91462812ec6c7bf787f96646d3e4ded9cd2295552915f95402a997e02091c886754106c2df09cd5fa8bab17ec077178427c7e0158c4e78145bdf8a2b3e31258ea7760915c2afbad530fb8b513cd06381aff918371e62c19c17a0322ee75a0628efdd23a2fd2d5c5dab85e48e6852241d9049ad52922695ba9f5ac7d8df6f071775f03a76a05a66ab91e2a3b6b168f2a61954596cfb89be4ec5d1c2e33a4816e254d8c526c00107e2e410b5ca8be80dbf760281c724dcf7d7374075ed96b8cb561d53cfb257dbceda3");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "5ffb6db7c40bbd6f6dc8bebed1d6d3363df2defacc0321498eace541baab8fb622c5309a3f37cc1b493b185457b1c3f8245dcb2e56eec3364f721d31cdf68956fb0a2f18a544b679eaf2c1474be34f0d6e1dac9287a847608b30fc07593dbb6b9b5577c4a8f43f95f9a2ea013e14635d636e62ce491df79b843388d459a2eb33ffbb3cb28e268326e05ff9a04eff6be914b2d1c62d50c69831208364d73e6b69cc95837d74ca92046ef69c555b0a06646f31a4bd8a613cf60febffbc7aaf36121f4d8fb6f8ec857cd1bd2f5bcbc26ee0c2bd3aa52c806c95eddb576a92eed5198b2df415cb6365bcd52fb86bbf5ce21c83298719b342cf4b");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "f0a3efedb7dc1b0e6ff87f930096699fb975ee2dbb657fd96c4b0f726dbf72f504e221fdb3f930badfa1717a4181d2000b0177102b0afca8a923732917b033009548b36cfe65c02b33561c6e64ea4a6aaa196e5059c8b3c40e1b8724461e18a3c9ba7ac82f600b4a0a7ac1e9eee53f18151de4de9ad2bfc53cb8a3fd1297c5f1aabae0ce0dd576337e104759c45f38ee4915d7f0d1e2a5e3c183c37be647abced433ccd09b109b05fc1bcdebeba38352f32b5dac69f225db2e66a635e2f256e4e7daef47800d5341b3177cca4224415fdfd26f65b1113b2828eea13fd944e2b9a185dd075393f0ea24f4a5d1b8e604c298da58520a4110ea");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "1bd6c7e3f5a0eccbad954278352baae20dbbfd27a51c885d30f3d84b5ff7737a199dca2b32451b7940f1c5567840ad3a346d8ae491cb703ab85ba1457165a35321b7da21beba1c96fb5bbd3e18262ce42878d4c70d7b285b31e16976dcaaf5ddd2a55d284a28c2af00081678678acb828f45d58bb5789f7a64faee78af9f074519aadfdfaf75ca22d9bc870b0c8723c2bdfbab334b0cd5335653066b2a5cd6e5ad4aaa97c4cf5ade1bbfb03c74f9ae1a3cf08cdb07ec1af100d1302d974b33fbacb0221b128854b53a2a7aba5a7418e77e0279ba6e3a8e90da27209a7942aa56b5169038cd04e2aa01fe296ba23e42dd9d375a09e0340eff");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "8d6287dfde336c6abde16a40d5aa5c638e1289e9f6c655bb94b226f545192fd9ccff7d75ace1b529c4dba5a87ceb41c0052cba8d6d479eac66949e4b2ba63b423d733929d3aca2e3714e15684d322ac884eaf86775475a8be11eb951af461d80088c4331af7c26eee72bed821c8be1f94adb537f922ee38d6fe9753048c7dc2011f7e43e8b487571cfadf71a665e78cd0982a2c52cd9b7ce658ee5dc5173e669fb146e9df9fb5c006da35eaae080cf3434145bf10a8ccfba0950b98f41d2a31743f7bbe8a0b5ddec1172a2ddb0fe892e3dda06bc6af8aa7e33be3c43c7c0e850e9e594cf823f9e339b4963a96daded3b3740b5899ef198a8");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "b480c41335a232467117f5229e603882b6656f55bc82b427f3e871a425029355d35f09f6f3f5fe419c177c031b9aa98d7827091033380de52179eca9e6e2b883ae349c5ce549ea2ad7af24a5be8d9bcc9f3a7cc0de4340f3668cd708c68dc708bb158b1ed3c922dce2b62e21cd9e85e26e73653d3438183bf82f6cbb7d296a03bc4ae739aa26d6755d1336930caf4a900592657e3a9b4cc6a3a5e95c022c159e6bac86f68dce1c7edbd0ed9809433ef0314e98f789c453f5669760beffe5d0a832b2c064467f7f9b03ea540f8ecd30aa208fe74d985862e2e2f923493dd1f44f781bafc5def6f9f756df8a42aedff60c11780a48937e883a5868f8d126ebb3969bee626a159feda8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "19205a3f7d28d9b0ed04837b26377019f8b4048e115a0f288eacdf4d6a4f26dc9ada49854f413ff852be5958f1cd3eb1b15430d433363161fe3c9cc9b2eff0307571ffbc3a798ffaa4f3c1c2a5ef55324e07fcd34079dc840642d8a395e6faec272b35fbb04b126e0388fca9dc1704094c569fba472f825c9bd7edcef65cbd4504793c695de2f7ed40c1aa4aba2c36c0cd0e5c316c51c7b0a44dc9918196a7fde3bb94a54bba9fe547a95b1f99155918ad7a51984adb5c385aedc663440f68c48b708f322370b37d38df8e87b82517e3852f45712825546c76a902b20cce411e79dede2c307f7566eee631242f8c115c9906bcaa4c8cf5c93ac54a882be3c9d51e7e03a97966ca5a");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ef538b29b30c8018f88d37e723b0054f70b02549ec15b40f48d43bae035e58c85995de6db2a225c5ce55264d2d63b9035652170e92a43428588e347f9c15a054cbb716af5decbaebc5282de9d4b440e1abf8fc84dd2b040c01af18fc28c9838fd6b428729231c3d93d93e286612f969657adc7fcbc302ec8c7c14332eac68566cfcd7c56f50eb79c2b8b97ec645854bd7d86288a5b53dc494b8ffc99475489601d266f76933cba3132d7095c75f923b002841f033b58e172e5b0d62c4dcf02609224b07fb1f24f185115c06e546c506ea710a8d2e02c8e66876963264b7c6f4bd764ac74d23e1236fd091bd886d2b939ac942ca0da6d25d355adf88cf96462278be27c4536b501ce");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "85d97e9f4ce89f264bea48c21a44598187f57195b81faf24afd0242c20e6519c816656bd7946a5b9f3b438e680549edb3518918c8b951b10572928f036a867cd7d115ab6568e786c8b51397aac5c9e562b038f86df5b5c657088912907a699843777e1862b1aa034cde11fcf4561f07185dc5b4112730961e28bd73c6b25a28e537d9fc9e61e6d8accee97b92dfb6bdaebf858928f99f8fbe696df82f1a50eb840ca23bb50669511ce2a230359146d53d393613487aa21e4d1dcccdf2c6c72eab62e4621a1eb26fff8336f01e1c086516f165e07d0c572c4573459a164cd9e0721a292b0a4b243f31dc6f917238c33b581ed7efb56b923ee971c21794f7bb448d82dd67edcde9d81");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b4211d36e644c36ad442cd7039691e0207f1c57364213857078208cabb64698572b915b1523c15c530fc17dfb2eda7ede70cd685405a6634c0fefc3bd23c2eb0188e13a0482f81ad82a01bcc26e6b04c73e95b37337d409bbaa3d9a168d7595e63c59b70f0c125aa042447999de4974c9011aef5a6d67691a824dd53cf513d344a5d07e27995da789b4b67bc422c81c9d62d41e56edda2bd8a4415964abf3c9c16fc6960f8550acf238e98d732331fbecbc006ceea90be57e71301c5f481869e6b16915ce8f5020201bc34e95e3dd01f0d3530c9672c5faa4f2a75e16d35efb9bc90c2e1314aa3a3201acfbcd4c25a4e631e8bd627b5bf8e811c16fa113f78b3782c6526a63f183e");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "60ea0bcea48f4dac609364c1721260bfc885022eebb8189b627ae1fd779b58ea1011fec51e79d9fdb1e694f24a60fdafb3308df9ab56ec1bdf7d58a9a0db77686c3a9f416310852a3a96801cb41e1b172cb53b876730c416687b40a8db47a1fcb6108344e7bcb9c36a6ea04947f83ecae073074e4a3bc00621a3d8fdcfff6b7d2ab23ed5473df12e9808568d8687c923f973191993dc0ce8cf96e2edc6fa79e950d562723d505694ac270ea259ec0483e8935e522a81b83200aae3a94866f007127fb2c563be99af9493fdbd2448a5c38cf26e9e91eb641f987a1fb7b108b22416de54047713b219646b13ebd0589cfe121eb6572dcdb7171cd5b30ea7484d7ae3534f56583bcfcd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "7f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "24");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "e7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "7c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "62");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "45");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "c0fb4b29f96feac3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "756cdf497d81dd69");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "156fa7f709b4cd45");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "1df32fd3d18665a7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "622bd9a6f752e222");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "efe91cb453a6058c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "8d82062d6dee73f39724d4fbd018f73c0f8f038b8e3a2e94");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "40cf4824ef4421a198d9195b6869f80393b39aed8af00cfb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "ae96ab3d27d7748c9339e996bc7e1f7aee69a5a3b96114ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "596f90ab43badb795d101a1b739ec7aae88a06f8997859cd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "a016f1ba06c08ce2f9ec45a4ac30f8d818ed0d6801544e7e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e9c078013a071b5d18130be7eba0308ccccf1cdf6d243274");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "4f6e23b36225eb4df9aa9106684f2008cd1dc527ccb8aee59191c663936d8fd3d961bcf592309a4d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "bb47448d8fc37a8f997bb1aefa7ba378a5a6840e65ec8d656a5616be59aad8cfa0f4c1740b1dca98");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "3dc0880e97ffecee882b0bb274446220c662835757a7a36ca2da970f8a6d58eb6ab001f292222093");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "7b464cc80f7f647794c6b4cc0a9deb1138f2682be0b4a01cc101f583555a5a995b69193aeaa167ad");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "4312a7b217a4ac8282f9d816a6659fe4e7771c1416ea0616716b5e17a29d076b5809c5921ef1b28b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "53582543633c445f9626278b259b0227d1072375f3c94e1b4176ccda2a3b724199a43b99ca2c90ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "4637eafc02a041750420d03ea18bc8665bdc65f25e2faa6dcb197f01b6ec646869d894de9be972271c61774c87ab6e134e81d54fba8aeeb8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "2c7f97c9f599c9b6ef521be483a0a0f9cb9ad042b16dae13dc66d78e06f07a8494bd8573623417d9f445573264eb5013822696a1642a9281");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "14474689f7c24c64664cb73005443f9787aef4a8f20db2bd1dcc9cfca2b45113734e01dd79d3efa2e272d84149b36c94aa39d6a71e5bf75c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "0f3152c27d93ea6dd9c5f19981b1401f30b31b9c60d915479f774e4f66f8d3912eb0105abb1941b132b6c33da0cf21f473d622d512fd9509");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9935d232def5797482e9bbb5c79d62344c9c6173450b8ebde80055590750552b699a0c8ff317eb123e7eab7c18769011b05ba2bb7f469650");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3575a7ccb6ab6e59a9f3a29a0c50b345c66fbe452c4ef5788c45d741f71b076e790fb8fe7d9546b2e8eeaddaf9f23b5ccbff6a7bed1e285a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "47d02138bacf7b1658514e6963a841a4ec6163748a377a644cb5d6641ef19c039729a6a85c12a559744f36dd5b21e2be143834ddafa0724af27251d54d9cefb709a2dbb58fd60d1e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "55a2e13392b42c7507d7d2731d0aa24c26f339bf0981e15cfe7b0ffdf52e1c93fde15a5f952ac2165423bd702340e26f0c55ea536eec85b4e2a99a4b8faa91f782833438c40c295b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "9b3be5167d723a74880dad6af615568ceec9b6cedc42cbaa50ae19164fbd8409d25eda01a5dfaae2290145e742746e48eae5c04877d01c9ebf6885f1672a81e71c8c2065e8f5ae4c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "cfb2ee4fc5a51dd0986f1ab75af05dc5d84c1404876e3ae65a0b61d0cd3ad7ab52065e9f8eafee883bbd4949ba5f2ed495f44be34d49d455f9db242897cf80314a4a58a342feb3f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "dd20be32406ffdf91d94d7434475f1c3d3c9fb94962880c23291aaa1c95ad649f5a59da7e6ee720168c25e2fe4982235680279aed452fecbeaaac5e60b480bd4213442bbadca771d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "78f129642e9834b80aaaca7c9c66c0ad40e15adc0c59dc5556ccf8f0769f9f84a10bcfb67a2678785e3ada4409f8c9dc6810b357a34a00eed03d77fe151119b396236df7e4f9fae6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "9a714f58819db0c2b8736742fdc15e413f3d5579cf29b775dbebc5af04e2da97d1585413c535cdd536a2a2bc786b2c9377765e7d7da258c1e108bb3e02c759c6ae81c6ddfdc85a2e1868badd40a1628d335246cada176571");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "a8c6eb17f9e75759f3b4479b9ab28af879b74bd6fa610e888623e607e263850c67372141df8da6fa14208fb6c8044ef920b3033c382a005cb2cba449b226f77a5e50e8fba50e1105231a3adb52419c7706d72acf52d4baf8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "ddea297fa646d55deee37785feb1872be5aba04847315188928cd6cf4cc51cad608e0505afbef84a4bc3d0b8ef680611b4cee023c05d1f311df7836331cf73fbe63dc87cac3002aa799d98c2002f52c48399f95427dd3aaf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "19be6cd8e442ad4ee4cad0ee17ea96c488e909b4ed7eac1f703549314cf1ccd505db3204347d21f6f802e43544c118f4068ada1e69b288a25d9ec0167d1b0548b2d8841b110263cc387ded259fb3fdad2d4ea6076ec2a028");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "abec887539d63f9760ac2d84eb3dae31404a60fe8b1222b1d089b65cfae6491f58a4a771dc7fbdcb4923b5d470074f8908b1ba337f07cb8d8ccc60266cc50bb688886ee406aa3228ff8ecb924148a30ac8673f10fb3062a3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1a87b7fd251707eb159edd3a06aa04562c95284d92b91a9fa46ed36a428f0e1a0944f44818d78d408f37e5209c20090799a763eb562f84118c434c1e8d29ed098c44f73397cb631c2db513e8d2f03b6753a8fedee027614f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "3f0957d7ac49b2853e601bb55de7bbf2a0063b34ecc7ae0627bef34c9c28ac428517e3bc403d2fb88c34cec16337695002e282e2f0bc23e399e651db2d0dde606b422d710a189fb7ab90adf5aea663271f1c3058f996475d1380594b8815a28446806c6a23aacc3b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "195d29b016f97735923a452d90ff8ae415284a516b4943a108669fd61e0dd756f3b7178275515c098c610ffa35ef2204809b7a41ed4d728d8da9c1ab4da27b129fbf24c40c75e6bc6796dd1b7410a5a79bc7a953a4f4a0a99d097bf2d36163ef10749f1e537e16f5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "b3819893c96b94096e6a856c6b68d48ebe372deef19b219d50569e2bc9816f19fe90a6ed7d004e1abd1006153348765a87f8e985d5db4bbc5fba41b85a0fcbfb7b3b420967564589e8e57cc94aea6710361371e1f9ff0bba51a1bc07a50f8b8a8e731d2a96285995");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "e004993a56c8ad0d5414ce17fd1d494ec1c2251a860fb0401711a4ce4ff8b8a317886a9518cb76304b3bc40dfaa4429991f25ddcb6a143163142834b26a99e84c4fa14858ab8407597a8e946c3e667a910c67ce5eae449f119f1149006830d801dfe1500bd82fede");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c3ebdbb8a2a8f169ebfda9810d5bb23435f41f10fc9ed5cdfd2f24b7f23b41fb8953daebc98f64d5c181a5729bdce37ea69a30017ceb514bfa1b125a2aa277488e69c5e2fc3d6aec79ef7bb1f5eaca4c50c003e932d5ac0e990a55bbc89613bd1f9195d8df42044f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "09e0104b906fef6d343256a8e0b4809cbaaa2fcf47b20b04d5796ee61d739bf6fbdad225b5223f6bc66e79d7248ce371fb2ec799df0f3572e256cef5acfd91c22621319ed0baf170349f016c81194651a3593ff9a0e30ed3f8250c726842aaa6a16086aa2b9b93cf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "dc33ad38cb01f6b08ab1fb8f25bdd27fae84c16ca31de5e5e0f1d8c5bb4c86b43f1281df4933acfa7d9dab9b8b4481d09cdabd13d0b2d8b833b5d97dcf9d9b82c2b4bbf600b446c0f59dcb12f70bad7ad91474c7d09e18def6cbaa6f4c74be2d70dcec49c3e5ea519c9433ecea3f070a6bd338ff1329e3a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "92088272e78eb6e8f1dd6fc306c7191659db5e981531c534aea0fb3ea6981a287ef6692cdad8b0177ca8dc5e0eeb5a7e4439a58d4ea45c4061aa53eb18cfa9f28a69a1bbf45d5d2a92dda13056ddce4a748d9b9d2dfb454c396fc5c70507be250a7556035a4c4a83e053315968f9d2e58ba71cce36556639");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "060881ee599cbc587a125335820d6748801b24d3eec55eb244bed9b45d2366737b6e94a511b885be0aedc0e7292b21debcdfc19a9120c61c62505a2fe49e1a20edda1a58425662362d4cfefed679dc6318786c1c2977ddf2901108a76e8b1caa9a807730a641dfab1d1dc515156b80193c6125d148dc57e0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "e1ee58540227cb210b1e8921f51026414229d5eee51ce0e7b58d52581a255cb9b7a1e08ef858bd5e000fa196065a19ec68ddcb0aeb0637bac53b3aefa9d05cde5a87cbe5526d24e218a1e43117bae6a2c1b88317260263d4ed8407ef2d09b4dc13956503e6e2708af9c2a1d425e4aebff5cd0d6e73571a2d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "84f3eb2147f791ec26343cc8626ea1d83b371bf753671ff9218defd78064517d3e37475c57034c82dd367b60e48eccc70160b32ada79f3015a091bd50a4ed5cd92b590d7de81cee430a717b0eef55e4e629d91c8b0f18b8dca589450722830148f0511347dcf435e324efab341a948ea51a73868247e4bd0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "e81c47d82fe02160382554a3a9c04b59c00c47f7d6a05e7936c96b4464f9167e1918a7309958b6649a7b9c2b598570a2d6f01f9c165609d82bc6a563f03a732204d2ab47bbb8af15c0f8aa1695048cef0d3f18bf3a09e09480e2b78ba06304c5548d2a781de4772d0a9ad25dde190a358527d7742ea037a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "c2f0ad438738c21b2708e9e4fcd292d8042bc223311907ad527335c9072433191103d4dfa38fc687755d028bc14966e3a961a304f0ad669dac23b57b66fd1b4be6bdc62f5be927b5c31fe41cc03715e9c3c60846c33488617ca0a4cfda0e317bfe45ea451e9776d3db471f4ba8263d42dc5988b749e1f2aae44cf669b44f72beada21c9fb8da0e6a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "a156b11d4849a8e5e23864245afbf45316af10f0f67c0edefcf3f23affb6e695f0780c117628dc934c43096e56d2314840c4d0d442ea610020a2b0f753aa3040705f52a206c3eca2a5ada6767ae21b8a2910ae31fd5f5dde8f1608c45aa972cdee3a162ba3b5865d3a8e4a96834af97a94abe5ebc71ca750d8bfc0c16facace050612bdcc4c4e261");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "9a0f55cb03664e04ce135d81344f1999bd4d7e7290d6e4b77bed73c0a12831a9f4d1747355f186f0014b58d4572cadbb403596dc864d22f8a8cbe4fcbf84599850cbbe3bd4a3adfee2af14a260f2d43f2358275fa3476a4b00ec8cd4d507d0ffc3ac94a2b6c837634790df4fb14e932c1645c35603f03c55f6326c7aae08c05cae4a8460b9ef46d6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "6ffef1b8753fa4ced45c66b6c3c09a07cab14b9829f4027ae3bdbf2c9672dd58823d87bd35ecc898a541878302a24ae077f8f18da0624c12c9d5116aac14e9ed23a00ec2fb80b8a831cc599f1ebf8aa3bc229e1681378804a219c7315ffaf7eea0988a7cbea15cdd4b56f14cf177ea61aaa960db3676ce0f3fa810042a4bdb6702a1baf9dc0fcfff");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d41e8dd3421204d6e6432ae3bc8bf898e7af002bf0447708616ef7fc505f8e080af2d8ce461c40687ad1539cf3f85a580da2fe34db66d717c31eef70d77fe12f2a291f8bd26d8640e0cc26a0c83f3336cac9aab6a1a371fe055336955c0c40363e9809a39d8202150c9dab8bb448432fec0c74a9b37886794a648af295299549778bbad6bf862223");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "fc9659479c4c90ca6346e819a32c903a6fce9c8a99fe0c58706b82d0db272d478dc086f096380f496c27f601cb9d627eba7429df4bd9ef2e5f2d392d66f9222e52819ed02ea4a3e3c565ce79b94920ad2d5c1fcaa719f9a742287c7ac3c662c39c46f18301417e0398459930a47ff39770badd56c569147414fa295b947bd8d8390f7b3e0693ff0e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "23ac54aa78f1ce4552e0d736ed3b26459fd49f2d1be040d108c3abec091f5e3e89b33f163dbe67594785aec70d892e8ce659b0bb4ecc639e9b3da4579b5589afba322ce50daa2d88482008e14c0daffc5a43150fd28b7ac84b6620219a71ccd85c33c5206be77038998510123c07661d08033c04410dbfc424779dbcd385d813a2bb5ac888a2c830acb8a61fc0064267c4239c9e3632f07a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ddb52ca3aca18e8f172aed91682b9d4c58f42b70739c29cdb081b894fb0fd6d05eeae4df515ec5b7c38611b3834d92ef2f822f86c215396214fecfef9580638c048c9eda6eae895bc0b73465c7872e0ccab5495615ae6f05c12d1af8420a279cec08f1a85d841c07dc43b21bbd5b66a075d187b5a7aed0bfb8f9ea6f7a50639705933f5a8eb017514abd82c880ae45639e45fe5824a908be");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "57cc2cac04fb00065fe6cd0cdf8734f3d90ff76eaf3e3eaf4f22bc9ef7f2ff22b8d001c3c902814a86900fce29c93557caeb59db9535b4a102a4b9128767765d2a93355ec9bbf45a83aa03e8f9c4e45712c8897f0a477e6301735820e9babb0bfbf2f6eeee6dac6beaef4bfb57257fedd919878b15af915e155154150ca6d8b29f0343352794300924816bdfa172cca8c4e34ad09fbc545a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "da9a8418f213e96fd34ad1646703fecba768235b3c2bbcea3399f12cd0a607bc21ca18be5203233343b550b0afe951bd6b88aa063a82daa9985f61cd0fd4cb2eac377cfbe79aad9752af7e425329610e411294e37cefbfdb6d33e90aa7878e47a7bc4602b32e148065536d36b0a70b354b3c75a8e190bbf1be625fa3aca1a3179dd42fc8806a7926d734503a718f77242f7f867884784574");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "e5d90b94a0747d04fb10e7d7ed3e478e09c74b4c47ad630e56ff90e5224e3cad09b43218e748fed579d5c3eb563de6171d976b7fe83dd5ca6c462083cea6d24fb2282e95e2112bac03e810c30316c62688e6ad48a45253ebfbe66f4a601b147bbe311905981f719b20454dd25a4b367fad4ec8571efb06ba9d2d9c7829595c8fa1263a1383dc47e7a449f4c22ed0139d5ca15513d760ce38");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "ce23678a2ce05b4639b96f0639216adb56c2ac6a3a89a100f250ac57e845f23785fda461a3b774faf9c766a4f046018a2ae0bdf0e74a81a57fe47202c28a89406feb5ee915ebc3e40dfb8e11231ddefe916f8cdd36c2a3519ffede8d11491ed2a31d9ba666b1589858a106cc53d5684b25f6eed2839e97ebb2057e55d3bc9d27fda4695db44fbd3cdfb603db4e45bd21124bb10618b22941");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "b104f5fe436597ca2ccd3539f27b7d01d8d2da4b733cc430d5348d2e32f575b6e86d9c23f9b77b0e0e1cdd2e3fba9303c276c9ed48e4fe0051da4a5f70af2b28c9b35ec4dc290d5f452bb6008b057029ed0af83086a716223f52d30f7bd900045a56d7602dbc124565a40d9d28277788a4764a893e811e8afec6a521eef3fdc159b9cc12230dccb383e436481181d4f6249ff6a4569bec74a49b7680555dcd0de5dd21ec384ef8d9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "144d6d443af132ff7c154e4f1cd67a8f2f14bb4f5910b4fc7203b81f5ec64fa3e9b231384e856d138c9d0021ffbc4fdc6b82023f4656477c0066012dfae47d35753a98b47a1eefaecfc5ceb25a91362a66134c116ba96c271194ccdd8ab3fe1ecf7381a9ce6129f981693d0408ffb3a9abcb121eaa805d15800b0333481b2eadd405d878e1a61ac94270c2884e5aeb36d414439e3ae540941959944dc594287ed9720e789cf8f940");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "a66ba94c3dea9078bf0b6096aa65721387ea038a2a47aeb66ade6fa534c080b46ead83ff7c3cb4661ec79e82dcaf047d3dfe053627d5814efd08dad5b71c1b28a4708ab839e91b4ddefd25154f5fc151de4896dc393f66629d9e624e825b4c3b8f7d18323aee27534e9f771da6c4daa75dda520ebfc81b89627b706edc730b8880ad7d62d645236f4a3b3769f6127d159bdb84425b6e7c487fa5e774a9a2603e7b1f6564ff29812f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "73f947f1571710073a4907912d333334401932664512578b5cdc48097f4fe7bbe44b4f88d1f44fc1200a26cdc55a922461d87c4f2e75db791058291c5512dbbe42f71b6b6e7e3467fc0422f2a5ac325571e203e30d0d0b6503764212796dcbdb19f0f73303a182eb86fb7108e45968dcdeaa3c68993d6adb5c8763d5c4ddfa6ca4e03450ebce4a893d3c7ea4e75ed00838a2756c9ffc446a7248dc449be2facacafdf260ce967d93");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "9a7f7effe6324fd7ec155857e975c32e8ef634f51d39dd0a1502bf193de70c9e33d30fcc290986d02f10567ac88c0032e5f186b6a96cd190ca3af031626ab923494fac084d6208a1dd3047c60e5066e89fa47512fccaeebbe7339b7b65774d7e799383cb9f607435fb715c7dc290cc450217751dacf57e8da7339549029897a0addc5190bda9109096618be07a6c3f95182863cd2d9e03cb5099baee30bc2766c7862c6e08303187");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "f37a605682531e61547721d71ba4f2817874e3856f7564ccad74cd28d940886ab7e32462e5b8c9ae01e2788ec1f5e6941c0bd942b844d5fe9eb79d3fdc9d5d95d87a564cfcf7cf808fa1ba0a336e6e4715211b0575df5009efe609003abd61bae3cb6dfa64ccd5baa5e34dc7323a2e0f8dd36576d6e7751ea1c6e459845a20eac3adf6cf6ec1e640b49997546aeb3cdabc2b4905becfef59c3fb5e5699cded1af5709ad7f2a93059");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "7f53bbfef59882b596ef4a88c7d98e368af24eb87504c91620309741408c28514a42027ef5378957cf9982a3eea05fd7dbefb97eee953bbd90c5905d7f91d946f08547fdf0ba04036e9644f43638130d2dfa17a2e86e75b3face662a1176064d6a323e1ff03d26fa47bc8c5ea71c6014deb77f14f91562ca67a49ec0ecfc582af3607f2cb05a983d7f83f5dd15b0c6ce28a8afa6f701bae5d7a1f83a79538f1941e9dae9de88b4fbaafe9425ae7f8d024f1b97d2c29bcc4d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "9bee0c67cacbcddcb29e1c2e606a8a6adc06aa3f5ea0db67089197bc29668af1f465ea8d0165c19393d949b51328dc68d756800fe03e93fb106acf69c208f70a7be04a4838f7356e8a35374562b7ebf9e1bc47007b7de7f61a7d1356fd68f8b80eb2febc6441fb125009419717cd586358e0ccc6bb3da17fd9f03b50864fa8fff6ba1c63ff8c27032de9433436ce33b97dbefa8c149a645a307e3eb3dde90d03707e9ec7c798fc1da91725f5390632e104707519080a23a6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "105b10cb6d89e6ec2bf3a33d8744996fcc615f21566e5e4ae6cce584dc35aef73772677881821a0bc5db2d02f371ce3adfb5389d609e5a4c0f48943d212cba203f98fd69fbb0f5df58efe85bc3339c6c1e06ab9a5fcc9f146d9b42a2f220e325eb134cdfa058e3a086b02c6b5c6611ef5ed6bb91e4d04c16fc555caa4d96a1e8d764bec695fec0d0370db0546c56fc53a45e1c575998c12eef2b347aa8cb6877eb98f70c7e9600ba61352e134736c586c91bfd9f4585c60e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "3b477a8625d8d0e669d5add82c0eef26ddb31e4f9a5d331080e129d8904124fc08ab140020d893fa94702f597012d4cf67aba4e88f1efbf676ecefc18c3103cbe9c874cd4a41057876649778c0e07d3c8f4b56618f5c7e6627f972a5690586b4e1202e8f029c017ae13e2834dd608e14f35ddb15f3f89123dd9da3fb78af255e007742e5de5eb7ef7c400159e238b65b01ec50f89b82d67df553d54e5b301895b2edc2c7a74a81cc27089e1547febe82607371dc0ca14307");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "12d39445e3c31c1c3eed303bc2539be33f71b876a175b75045bc3c389eb1687731686f6de922a52728969910f6f220728b3859a3a27fa811b5fe1096fc0b174aeabe34104a37fa039c8c034f859786bf18e8f1d7f071104b62ce9a0091c6548a20361abb3e600ed60fb125aa0e7aa59e4c09f512c7a1dd7a7413c3f9bcd39eb90623a830a2b0454c6642e39b7bc8c78fd6fe8bae18034360a421bf669cc4742c69fe1809356584ce6dd0c795825d868ce2a8f3709c2d7c56");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "70f1ec27426783c12bad6851ac49cefe211717747616d66a5eb2063d8824f02ab37ee1d5299be36f83b5a14331f09493bdccd0835a8949e40868ccbcffcb8ed2d88dc78d4595fac6a1289ee6ef1957ec0455ac32eebbd0e561b5ebd76ff0dcbf3d5abbbd793452a57447960f5efc3a3d15f7f90510b9132cefde77a545905e0fb2e3b8e1ce67ea808ddff562d99fadd4f81dc27e19dc32a67f914baef22d63284b2c440166c6388b2ff8ea71bece10dfbfb8796269f8863a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "aa2652131b80cf2d756e7d4e70dcc09a3c83e86e5c10bd96fc04f1164dc4dfb427cceede8e71bdd7e100719c8377e97df63c55b721bd1c7d36bc34802f0e0d53a8f661891721e6a4dd65ae187ebb3430d02ea6969200033dc31faa438b66de9a0e6f98b499ddf2ac96a8d64b1fe3c77571b37ea07561b8659ff862d2e9f2fc6dab1136c1494bc6e6950a3d67c8095051f92d8a8f417c40a721d3063b3b3b146e4f21a3173bcf3c74af05f0ee4ce637e2fa34f79092312183be26983ccd13fc7fdd3b93e3a92d8e5f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "07fa66d82f4abc01972c4e664784a9d9647846262dbd94c1d7b048aeae857d8bb3e1fc7398b47278e024ee26e63b8eab5bacd31c8445f00ffcc28371dae56e5849908f6ea0a1ae4817cb8f938907fca3074fb893b4d7c2407f837283ad5a2abb05d5339d9fd296a47fb5ad5d81ce300a1131bb24024bb3e5a067935b5fec79f77c938fd17445daade95d7cde5cbdb240e469b17103572ab271b9409806235a3f74c7603b4938e6c1c97c0fad0ed2e49e01ad6f48ce001811dc54066915d53c33b7984d3d57b8fef0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "fb4d3af8244b182efb29543ebebe1c0b01a512125ac20bbdcceeff81984e6084435dd719b108d099756bbd1b7d41c3e8d20cedf0e002301d982672413c72b73669159f2ef963ce2d5f7201cad19c4d9cc99e63a59d6a15069ce5ddc2e57db3a3a9e32c9500e78f8c50cd6e48551a5157c888d0827c095e4d15423ae82cb19e2765abad77a732aae580fb145d98d0caea35998a45b6ac3b80dace8be4d49ada2d5f59599573b19763d145c2ce7dc901980dfb21cc27b6b73e256eab284d4cccf049a4bdf1a8023aa0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "8676c28e89b9dcd0463edd0ca67266376eac9a6c2c9ee55cf3b1803c729059a813209b4b7b9dc832faf23b5f2aad6af570a5ca089d1e3c0e7233e99b29e57e7fa308ead80ad60f626416c6acef58d2c3f6ca9f77f7651cac697fb2edcfa684e7d008c30bfa02ab391b87f40770b4afe4e1ea707e5bb9691e35293d42ebfc4f7155fdfde3b44cc98ff3ab8c886dac76569c7a142759ca3bd9e816407087a242bec9916227ad03de5613fcaec148e7909650f27081820a67944f48df71083ae274586de01e66c3f245");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "d76dad40e31662427795d3816b03bf917035e8d185f51360c36a8016de62915d10983c968aedbbbb1057d509a030cb9bf593d8d4cc25db6ceaaaf0c16f965ac6de2765e425e83a6e2140fc08b6b88563ad865c09c93999bb3abc0643dc813f86dfa0e0f65b36570ea2dd93bdf832b5fc5dd14071fed5df2b860084c55ebf0e2b45874e9298e271bfcbaf7bb49989e1b00e46904282954bd82302e00460ade47ad94114296e9436440100063d321eda421626fed8815b00d7bccc61ed11be61a88f56ea081b5497be");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "aea0d62315a033c8715327ef0e207664388eed7a10159169cbd198127d66bb7549c05048c2d15797293239c61bfc2609e8d470d9139a58a38861ac8aa7609a64a746b4fa8fdd63870ec892508e5e2ed1b8c0f8206ce51db96a62e9aac9d4ed6ca7826d5c8104c8e30cd139f503f34b322e8c6e61a44554249c9c895c6c8f39dd0584f2cdd9be7929fb43a2345ba26ea860a4cb140857b9d910299e4e1d5e1efa9b7b3da1c318d1e4937952c14da5e0f82ed7cc8d564d8f0aefef2b7efdc2dbb18b8b9ce9563d5c4a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "40c0e12c3bb964228a1361a800526cccebebd93ca62ff75698a9dffeb1e3f171ab78cc8ca175a02764acd1aaf16cdc6b84750ded6a1894e24484a8194202c741431bd0e826d885ba80d8076e380a1ad697b760da8f81d8b0a337c693cfeaad0c1828c5397de2def5d9a7d80575582e0e59938dec11294ddfd3c6ef78a38c2127b86e5cabc4421547d31a912a98a0311371810a938e968eff612a1069651f48c84801595a9c1d95eec91c90b425037034d51d8a9c3f66b8354fce598dddab4c8c155e9cfacccb02a1ac22361f4a7ff9d07f13cb07c1a59bc4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "062d6c82983b8e3b5b4c37fcaab6f15ad4c1c555522c0ee4745daf50a296b3b047d52b2ae9efc7b4cdbfa41cbccc2e58893667c987bc98a4b36ce41d9bc122456e82faa3ac7ae94e42d070188b5dbdb56cee34c56c225251b239a0f51476c2bcee827f14dbe30633b85e149e63c8d66062ea8abc09d59be1f84ad51d4b83f470c93abaf31ce689af9215258184e0cc84ca2106819a00a8ad188c235d108ba2a40d5b7ad591ca954499a947182d577b3ff080332dca523969069f487601b6c4286810efc02e901c2ec716d9b150bab005a66b9419c9518df8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "003ff0d417d7968f373b4bdbcb0172bfe05a1f32065872b9f9d5cfc7d44ce8ce3f42aa72f667ea394055e173be6ea3dcb908d4a341f5866a26cacda1e35e0481970394615740050efabdd70350f1681ad34b5cbf782b74fda6fe5704e578020101da564f81b4a7baf168f462bbd3919fdbd4d461fb9f3483c451a8f733c136f70e3cb23fbee5f6bfef6132875e8846f50bfadd0b17a1b7bca8eef1476412b110f73f5407c628b87acc9b1bee0945a20d890d3236977f663b6340f26093ac7038b8f43f22603a8f364525c1ec947f123a6556b899891ad5a2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "d9246a6efb37dd2a6ddf17c8c98b1e0879ad5d9c1c33ed0e372972b8f2890537aa7de07373656fea3e30d344cffd444ab9bbe922240afb57d3078238f8adaa38e71633102f7a9269e2d006b7c51b0276786cc8fac53c36bb07aafaf107b37ccebb16cfc306f6eee54c1947b10c15d6cca70972b17ea39f53e578a21f83fac14c76807823b2ae4aa1449d842697fad336256057604dfcbd589ccb66667728ce5823c0e1a0c8a74e4c68438643283152a5fd2108c3889dce1ef063778cbbcd97f0b2cc447bc034d39802372a9aab359951631a2c1c370b28e0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "17a86c65f2177f7c4d323bc6538cd87a125d5f0bf399b0d3483e0fa79cf55bb37530591ff66fe926868dfef7c24795ef6c970d826373b1a399368ff1d2dfd56be65d4d793bbb0c1c1d7f6a8ee53bab0c9e6f877453d6a54c56a08200781848c2781d08a20e2ef9a28b2976b1d1634cc9da383794a210b24eb2c3ce3ba4c12fce64550225fd6b175e829b4e607657352230952e1c1c9eb829fc00ac259dd52359e9a6a804a198ce1c2952f9f8dedb8948bb322d6473a6dfb2f15e2dbb562be74ba702239b6b64e0a617daec2cfbc323e09aa59b380ed5966b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "a5819dcc75113b3058809745b7f9981e370e992b8c2a89732f096e627c9b13ef5543944f9fdc7d5206f4f1cba39a434a314e097ccf191e91bcd71012efd5207c194143473e5314f8eeaa93436b430a5171ee6647d3041080a7f244674ca839d67b319347b1039c703127706bd222731fe6745ab989cce297addf6c88032703ccd8e11ee3d9ee1dea104498c5e3ceef4c100fd4e555d92ea3c3374e1bc7e1ff95093edc2bef709a26f2080f857d7b2964b2fc1b4f6b712b202736cfadc4ede936dfa78d92e1ebfa87f085b7ebdd600c15eed0a79fbdab9472");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f647428d4ac838bcc7bde62cb73495722b62560ea7c7299350710278a1b44780e72a8298c0f8dbe0465ac0a33234963660c00edc70b8328c9e22383dbd1090419069422eb83121580b016de70c266b3a54b3f9f93c93e2af80f55de50b8c3bf5c2947cbba0465993d65c7549d885569accdbbe2d35c11cf3539f7467b84bebe2f0f3cf9b51f9007a9365959e6327b2fa5e785af055afd6ef5d082602f2a439c1b64936464d9fd507003488b0a949de8fcec79ad2eeee24f8b0d89be8c6399a377830687952b18ead01636aede0716c0885a170d34c89d50843d3234de13a19b8a599b50a9e6358a7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "0c4c5084ac95ffaa648bac56361f8ef82642981631d470b610c8e638b4f6c9772228698f0e00ba96d8ce84abdcb1f132b29be493c94781e486678323dbee9856ab0c6a152ec91313431c72d8b4ade3d116792db8d7b53b75c4140c3b0cd71151ec1a9c55bf557a13d62185630ca52221535b369aaef8f17229865009b953800937b2c9aba8d7e468fb48118bb953722a2801faf37644c8769ffc4c5ec14c780ff98b832bf39b03c20291747a4d0210489bc055e0e3b05f2a54e8567303a40552a5ea57681768511390c8fed12ecfa74d31c6db894f9920e060e7bb1c28edc2f62026e5ca9f9166e0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "8e0b49f1d3ab942ad13e0adee92ea401f69b0e1ab7dd746843ceb624b38e85b3017d33ed99621d6faad2739307d84059de93cba1ee5453583ecb0eb4e2d83508f5456f463af44b02f0a303eec664cf3104dad98ed7c11b2ad1a632f2a7ff278974a33542e10554438294b1a4677393bcfc12fe75e49cc5391a25e8e394a28fb5a013410177bfda1af95a9b26c38b000e1076d71d40722d063dd80550adbbe177de1b8a97c5100695489c935a92bc7285fa218008afc271d9425014bf9b5dfea9ffadb794c0038a8683c94c7c8ad74d6391ff1685b47d89b3af0baa6fbd2b323ec438cae764ed9093");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "5b4ae6eebe8854ca8df3120c0423f5791a2bf52e57827877c1fa05e81bdd7d8f98348d1f6e7f0afbed973007ef8f84f4d65b139226bb38a0b19f9b265e4dd44991c553287957422b16c2270605034fa197e67e2b16338e1a3b521f1ddda3e09fa45d88435f8baee698a75ccd50289af3f7179de4a8ac62487e40ba8e1013a6fc1b276ca854eb3948c5080de5f428ce8090973a12f83aa574f7ebc6e5ee07a64fd981a3c57f41bfa0d68628f52a736ff23dbb73d6b54174e227ccc0f0205c19edbbfa92977a5168a5874b081bd6f6e29eb25f05f92f58ce49566e7e89203841a5cf3555b1e90d87f0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "04d89b2878c7ff8c7bce39c9bf703945b9fa3e30c430858dca06eb447cc6fa9f4c734d70c5fd715f4d292e470aa968474294b9ef82dc2f51b88b5832add286fd4b746b6f2cdeb93d11779626a4cb4d7d69a0bf7b1b6236efd635817e26a2e767e02ea39d6aebef2b8fd0ea85d61e715a5d48328a7c06a6086f848f529d1d6dede8482a059528a6a8f4c32a14ed632af3dae932bcf351afa816ccffe21b00c0086eca2ef596fa99bb0c26288dbde3c526cca4537a1dc39d1f713959ab0a7d4d02db1ac4c92f5965c93d2bf6c226f0f6efa82145cccc924f9c9b6afece994b04a796f2682d641874db");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1a6fb90a2e3cf2fed6f804421d6b642d64f026dd5c890458f004a6ea350a460522114583a9876e01cfbe74d546e84797b97b3acbdd686e62325cccc8f2b1be4b5a67434adbd61d34c0be792bb33880d102a266f143bf7573ca1c7e1500d5cffe6b038a5cb19b4c7731fdc7bf733fbf661e2190a43506694c141419299f22d4b74d95155a53bbfd7738bfce9fbaddaa7bad79515be5ba09738fc31715a64624ef9876b5551508a09da0519cf8d45d2a1cc9f9e9c09ef075f2d2431381260603d6d6039023d2cc5dee8430503d9daa81f2bbf9dabd2a36007d266d8d38757d51f99038f3a6dd6b8d14");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "4cd5dc92a5e502d39874156e6391c5904992c0d197596fd6b89515c8451d4ad3d262a891f05ffc90f1706e51f57a2b2d7d683cd04e7121ae87427782872b3bb9a0ce7eac3702d5fa86ecddb3ed7664522cede4a5fcc1d805bdd95aeedcb77c79d7bc3457b7a46a9030457ad2888cf48ffe34f2a5db273901058b98cff30656c0bc62b242c94adca51e84e9644eb35be22b2a262c3ec5bef31efa7a3d55f037ff9b060b777ed8bcd6f1f82b3dca513680bd6fd7748d455286d7203f2f8eb0c17ce181f3aad251c3ae0a451f8d85f530a99019c4ba03ea05096b4f6586982b59565a95146f07ede27e51fa188ba5b598d1637b55010fffe081");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "68c281cd02cb401db3433bbbbd5b2820c0bc39e6a622657d873feae036f96316b1656e3eeafe79799b69337eb575406e97e4cb77dab7d8db4154c8faa1837da6c7f0fb34b5dc3af41bcf3246666926760efbce4605459e6163122922f0282b7e8448446ef92626ddf5a4311c98e57a0727352e84864055be7f8bd97ad04010662dc85b0a50ae69f50271cf8ed1f15258312c8e29fb0510089acfa30a051c5a716df37a308adf0bba86d506311a9151b25d5f99bd120cee56ec5400832f4172e49fa701cbe150888925705b280d0b7d687fb79d7df9d9c2412c12a2d09dfd6ee336a82dfb6619246f3766e7d0eddc34a746042120eb2eddc6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "90fb1853c97f5b700a56b6e0d3088d3211e7ef4ed00a9cc0248d40ef77ef4be3d21fe52714c67f5191c790b2860c21e9d56b0498628b9259f6f7bdd1da0ffecc5f92d1be1e075c319deb9c1fd201c5bfe77af90add725ab857f71182a66fc460b9ef8e0e4a6f89dfbb623f99372f6a4e70b6695a3b398aeaded7ab5b1aee88a992a96b2eb46a86c8befb2cd15ca03b8df21e4d4cde15b7ef56cc49ad9de7cdd938febcd0367a3a740dfe779b4c119ce8e19309eafd8e96b904a90e5de192fd99ebacd8952155dda802fe069b71daf6689783944a0f3e3959ec2f1a14fc7341e0c91c7ae3f9e93a6942f3d813e8f081f5040d36a53fb7a738");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "6ff0b4a23f34a72f84b29bc250814a69b3c5f853592b0733651a575c85e8ceb4d64d6c1768612ce45480c50a5149283d1c3bdd46bdd5cac5d3f6e7b5109f0bfaf8ae93254dee6d72e2bf6ad48c0054fad42d462d3aca08dcb7ae9e16dec17b3aec835a4585e5401cb73bdc98db3ecd465d170419014067b3a16b1d63e6a2239e0f70f2cdb42084abc955b842612125b0e01edbbb143cf711b97c85e78e14c7b4de7876ddba9326de057f3dab90fcf8b14ae087f91e36ec46aa1b24c77e60acfc9bd544f6612283208d3d029b6b8bebdd8febe4bca4f95820027552143db04bbbeef5c85aadcebc82327dcfb6d6120be62a0f73963abe44f5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fd394ae805864f8a22829551228f73c874cdcae299c91db1d13e74e40a0be00b14755429623cf5a0c5b388528be7d5d86fdba6534d9ca431b2ff7cc2c372ee0de80ba3a415541ee3b4f6d46593ce9d2b489bb8a40e8a19a1581cc9817e5b2ca2dcc3a1be914d6a0220f9e549ddeb9fa3f40c774b74835b7698a85b905d9533ea87dafd343af38fa172bff618de4c34cff6f40c6835c7c392bd1ac53caeb1547363922d79d11dc83ca20f13583c3abb91272b016c2ce8950817618fa01259e03218d870151939c2c5444f05968ba1d968893cf68157cf4644141eb9ebc067793a2c98201b23230193bc51dedd8757195dc0e376dd1dc2d2ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0df2b0cd362cdc46b30aaed8594265cb7d9b142f49cd2caa9bd2a9b7ad8af66589165e411d76f923a53bc0751db35645007823f9f007456dd20d97351439e7054641ce8901efc86217b6e465059d69d050c1b786b554c260c209848fc7bec5acbcce60ca7ef69afb5fb8d948325851ca2dd1c1fcc4232b710672e180aaafd0aec218875869d003ef619d417d86d266345f48fe892639da828d3994a4a526ebdd44c0259b9ad17f8a82b32318a4675ec3311581ee9b28db1c749a4c434bc97ce1ba37261cf6ca35fd78fe24907f20568f22ab453244f5bc033c3a7c18828d9f34f0bbc04e067f96a9a75fe1a235dc3c4a5cb8436763d8e641");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "e07433e752e8e9eaf7d977670282f14016960bde09a995f1598f09eaf791405d267570115000e1f22ba4db42c96738708d6d9589cb28964117e2e7ac1bcb08b0dc4c8bc374c0b01831d33a016f24301142e744bff2a6c06079d5e573dba6e25e2e34bbbf6260b9accef2dbd2a41fd401cdf93c7d9b4f20b4ba74e44e00ce555588192755c0e0447e0a39a612bf1d99d8bfa9f301edf4f7cc70c75f1ba6ba9a93b4e0c86afcfb7e3902c6067a569c55e42dd9d2da50fae482a425aed99313ae35eab05bb6e2c12af142d9ce182610ab7caefb05316a06500758c9cb9833d943b1106f049e34e2428aa36dfce7e7b132581ebfa5bea4cef8130e822c36eb601ca0d358d5d1116d8383");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "12b4aec1f96cbed9cb4b6159ddf227014628ef8fdbbec79aeae3499791f497216e3ef2a1f6b68210fea6a9bf331eac5146c4656ee0f0f70fd4f30a2a71eedcead237bd488c1737b51f0f9acdbb6f7f0aec0b423e5685b40f9d8e96dc5c9dbf5ed4d19294aabccd3750f74b48a5a6cb64bcdab2863d80019805391ac0d5b7f48db678a8d59c586eecb39aecd9d92d5f25fc0a66b3d70eb4fdc6c79c0b970a76f3fd3628092fb68ed3c3460677e44aed6bf98a2c211d6d9f355044de4fe6d39e1fa95036caab80b6923bc6e33989903aa66955bb9096ebc96f6706a3437b797be09fe749c6002c7e91a70fa0c5b479821a59177055d6e9210bdd0501fdc97b1a334713f561f473f4d9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "c809dc8eeabccec4cca5f665034c123e6ba92661ac2f3c65116c2f35d60878e4c1ad11063edd50a5429186f6ac8f4cbef9eb179113d35c7a9034dbd444dbb53692bfc489308afd39d2252f29c237e8b9bc94bc743292a1ea766e9d4cf6a1da0b96667a075469c504a33f3b1132c3ee81ec466e2f02758919664f9e3e687221e091480440f60517f52ba5cad4a99e77d4dab4e30ecb45f64361325272309f131e8bd53fd4d713951b176a5cf442c7b65eea7654b7e261f02715522bf39b23543697a5b19bfcd5b076e3f492c87051157a96c3dcb2b4f08ef5425ec294410a9741c0f1fbc54104b420307ddd904611475c458ffaaa7c574e40f3068afc0cbe49eae7d485b565b37ecf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "b1a8780b194e07ab0d6ce08ca0ca96062d80fbdd93b842a285daa6a28ec6f66e936f924490866ab8d119dca9da55be96841490756a4e7042fb4bf4771d8d1dc312242e9507f3090e995df47f70d0a6749a83accb468306e7ae76c7dc7b94f15aea7c0fcc42c7baf508384707c7186fe47fa4a3e77c268e8440522d2e9c096f9608e225b80e4a527df8ea7a009c97f31a0f34a971fc4d358f80a69baebe363174b64e79fa8b9d4f898af00f7976deab854607d08d9020e3b8addfb7ffd9aa22d4556bb0f3e6a1f598194dc198576f2df4ed49f3ee0b623ca1ad086cc17fe49162dffb5ac460a651f67a7cda7507cae3b9868a0f4a87730f3ae774c91e2d137077c10e1506b63d3dad");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "5e1a78c9043c51db17dfab048511a3717443b89d93a1124bc0377840251d1bb9adab2723cac5820f7c8e578ac92020d1b2a5e7ad39dfe0770c3a940d8c3ceb92e10f5ee9755b87f0fd05fd64bf2af7e43a8b2d27f390f63bf0c8441e86fb02e57b14faf3bfa934f2444abca526524a3902b54db23bb744bb93eb67931b24d94d5717a9905c50aea9b8b06e4806bd768ad061dca2bffd1fd112634a32a5d16ada5d083f93042cc8070c963ac61c3e3ddabda8b4b629e4e448639ec74dc06e70d6969867f29191e13210eea88ececa93a1100c52ed383a772a6c494b6ae545a0ee51805ce0738b79d6c8b31a93b7f68597d682503be83b85a41dce096888e971d18b05b4615e089804");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3f21a5e3cf4aeebd1947b9e9d9abc15c3ecc12120882d2d1bf61e8097dfc7e6323fc746b47d2ea180c42c1c78c619224fb79cb2e93931bb6832aff04ef93d0e2ecfbead639d37c41b9c617bda3c6fa87a86ca656cfc38214e94e75b70c261a791b487eb9529ab507b8b4942285e7bf22387fd7ea7962ed8e3755b499e3906db79b9baa828540fe5043ec1f1c241fa20245dd02ada2d19ca3882706ce7731b6c9e246be841166964d37aeb1fb2ecd9ac5e34b63038cf6d2e767b2447f0312a6e5c032791079da02489aad228717dfe2a4aca080bb65df8d54005e79400a9d1a6fbeced02d38f0e82d9b4e9df8aa1b7ab2183b0967dc763ca22566ae1234aec0a525f7aa36441ab17d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "18");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "67");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "13");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "24");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "79");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "8f4b9a79ba89098d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "2df9afc296d95a40");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "3b9468afac25957e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "0a5283af4cedf8de");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "b4584639f0d6797e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "b40f75c04d651770");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "acb34a59e063be3dd929672965506961344cfcb1d60cfcf8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "4a5eb3c9bf01ffaeec3649b30c3b12f5c92510167182eefa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "0fb6210aae98d6f59184576271636e2c6065fefefb526031");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "6d94d2256954974ed3e6e6c511395529dcd43ec0525fb916");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "76804da5ae36c8dce40c8b868b7a761d0d6d44341295e5ef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "dc40eddbb1d46409e572a4f27f2b502a20fcb65c0d8795a6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "026e5695d6a76279d966679a05a7c575af97804384ac14a54c39d890a11af85f3356ebfa5f8e6551");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "acf14c681cc9980d7e78a321e0ebfed4fade1ddb3620a287ed4c6c828a5f818dd689c64aab790000");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "7fb1b1ce574b3eabc733f39edc5cb14c3d54a1bd2ea5a8cd5795ddd146892ab5e557a4b91a20eeda");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "d9f659bbed2ee1b04b04b36d8e74d189d31124eea3e99375b05d53f86173187fe05b81bfa9006001");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "3e0bd35baec2da760339eb3d3a38f24c21a219b2b2e2dfc6f29cb2b7f359f30b97fa15cc14929a5f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "dd44a4f733c5a839aa22ae0845fac99178550963d00f09e257362d7980a0660ed74200b834e0d55e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b6bcb2a5ebbf7e7c6a46ca66e41595bde5820abcc19919262e03bc328a4518bd35f1db65fe0f3240e7bfd483792cbb3d0c9545a8225eb4d3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "88894cb11ab28a1cb0b1e7dec10754934edf1067f8b9ddba31556d6f6985ddbc96c6b0ba6993d31c58f0aa58fefc4e59b87a6c279c28ae21");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "a986e18cabfe0c1ab890c1ce5271108a6183da0a69517f3f5d0df954fbc1d06227dc55ad166f49b5fce16af3b37fccf460608d9136a1c171");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "936f0faeb2d11554b145444b218cf9c4f795cbba649235706bbda854de65f355e7bb20aedaf46bd36e576601334b05dc15a9f6b6f6400b8d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "c0a055aa473d15de08a7c20531dd91d8affebbf6bc2b051a9361a6050decbd8ce513967c5e979d5a183b83f91f593b67d7c5445e9e1ab34f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "1036bf76f1bfb48e0da2280abc89761f54a39b165319ad5c8ed5025e25c44c097a2d3e7539c46ecb7408892303d32c4e29cd941c8a04ef02");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b1a88c7d80633903fafe42c898c5012c45006b6ceb313b3c1bc725b59d88b1335ac5991435422bff1bb4b2bab409b0aa9455dd409e007625acc56fa22f0f841a4ee2416efd67ee79");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "1b4b8633a9da5961e52f270511e60fd7201d018ed1a0e27bd911efe38c3734e8494bf99f99a3db8fbea447bcdcd4b8fde3e15f79adfc5c7d89bc036cf9765636a3c4e2059073dcde");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "99238b4aad82657149863fbd449f9fe69acc96e92dc93dc3e0a58bd004560036cb69e8ee8a5038449174a4a3d6baadbbcaa0581b37790956b8d24bf4e395e0d94252a382779076d0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b31b840d92ca23696ad02e19b26deebfd0b9939fcc44092b0ab53aef7ec0fb5cfb421faf0a83656cc89a5789f211cd06356de788ee75da46f1d338613b92b728be8af5e934dc32c0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "f771622c1e5cc4bb118c3151c532b0430c95bda19c4d795e03e0a4a9cbef87820e65530a824a89603cc83343eed8c505a57b56aff3d9802c99fe3fdc4c61401076ce107d54cd9e85");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "04a58910fdbbd569c13734ce9d9d6445ddf074108fe7275f14381833ff8b2b436b68eedf6aeead8088b4b65d40add7882620ae6e38e29f9634c8192262999c48e5c49e7f95768f0d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b02baa57fb6b0b1db571ce34840c6170fd8ecd131b5aa1df7b7739f5cd10cbd5454a39b0814a36114605d96f5f3728d40be55931fd502ec859cc65dc49feef8d8be0456a0089cc9ce0ba1fff3de1a4e3ee6d7fa38f90d31e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "10e81e63d0490c6a4617d6615191545975610fa1037dbd186710af55598c50e1a5121926bd54a000a5487e08755881c95621f43bfabe227df9ef6a6db369c69a951580cd4557752bd06dbae9a36a9269eee8283bd4c027fe");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "a0e8a55861260738da437317e03f6a074ae11588806412d24887f45f94ba6f481968a7d3b31bacc53fe069677a439a5f412bda273181e9311811dd3286f0047a2532660b4a87aa9bb723d6dbfd6c949426ee62e18475a0b5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "978dbfe8cd2aa5b4b440571e5ba5e9a44edb86fb016e449762a3d5289aed3e398536c3716b777b02023d9d099568562f28121d506571f22bba1ae0b4eefa3e2406c7b00c4b10d2d4e924af70048ad729b2ad83a117948cbf");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "f43e526f126e38141a535bb6f1dd348a0d049a00472298855701b96c83917c4fb8af3922f21e908773dc0c62be86e13f58a2dd5800923ed39823761010307186f2140a35e60765f4c616c477673bcbab27d70ab5a416cbbc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "3db197a9027ebf7b8dd6064d16bd001e3ce215b3e74bc209db9d19cf341acac612ac68be48fafb1689d71e743a3e9ebae1fda5929747a3ee21c7ed0d8bdf6a0469fad22707deb4ffdb2a6a84a7e3dabf56186393c4ab62dd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "85beae9624ee5dfcdb83f3a962a5989884e1e974051f1aead6089b1871b36ac4a94a82796b59cca55f5f3de6b48cc66f2f8b4a9001769b4f7de186798554e4b901a7daff2322f6dbecd2d94af7027f0208744347126e27be9ee1fda0d7ee76fb4579946c3259a6ff");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "1067c46d073adde489e6435db0c70994bc5c8cd2b3a596aa96b0b4b3c0dbf28341b9130c0b38ad662127617158f3af57dbd2bacf1ec742eb7bd3fbeab1908fb87f170d057d7ed150b8fa5b4f1d812bc44d54fb0c872442891d9cf0eb86631728ac4408cb979f7ba5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "18ed0c79edac1d3e42e86344ff9f4e72df0b61e1da37a35b77bfb41687de955066f581109b94b225ae8c7340f2735ae9d219e58e801e1db5cd66060c8eef4b973fca36e5dab70bcf1f24da189169abbc6a654e365508724270f34cf1505d38522907053ed357491d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "8864c531d64ca59e23d669755bdc73030542b06ee717790c4aee7ff6686d3ddcd0a8b002c5ef3c68aaa5fbc34706c6e956f85cd97b9d3bb5997d2cc8e80b86b3b69fde94fde61e114baf6346fc933322eeb199dc5f4c7e39263d9ec2d7dd8cfb637d42abadde47ee");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "94555b9c84d1ee33189bde28798b44270a06c636ad1cc06fd6c0a1bfc76b7cb67cc4984f6a3c042f4e5b73d9d755c24de1a27992d26428b5bd70aac0e346511d707253f580f6da1200e1c9f5babb9d5fa4e656b7edd166bf96d053c008d4916b6165742008915578");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6f9dc8337660644db2cfdf678da5afa7f2ac50fee5c7d1406fc0101353056d7db7e1f27e831832c4b87f50f02689c3856c82e0cba423fd1c42a854b77469c02b2b21722e214a2e72e54d52277c7fdc493d1ca99e403060d5a25a9fa6568920a325988957816c64c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6dd7bdda680c938bc0a2716a5e737044055f8daf708644328c92b575a4c3e8977eefb0bebe763915bc6fe940d39a9a15affedae577004b19f95ea4697f89203155bea71fe1a9085303d39fca92754181886acc6c0f4528957eabf17a9d85b60b491e3a61992ad1c76b08f789e6159f9d80847851ed6901ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "774521911a940b142ca35ad18b9202a051b911ac9a8cca287082c1a4a317926e859637f2617934f94eccdfd45b9f2aa007f78bfc65d91daff612225125b248aca6bebc86f7e5e8ed2c739c1da09f34cbe6102d661728c3cdf6730287848ffd7a3266b8d9973d07bda9b7528a2120454fc54870c4cfc1b74e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "caf34219949cb1f5b9b790165c582bb179ecfd2a9ccf9137fbe4b1203fb494aec9953411a2ecfcff4ac327f83b1add8eaa9257f7301bd34d8944cf35cff47e04318c8a25026206bc4d906f8dd63c20c4b52add1a4c3fea806d1f308bddfd3c53fb4b00d55e36740ffbc0133bd34a9347172ff0532bd085c3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "302c38c93d7858812b41cc0bae536e8e264d73b9c4a8e3b04a9376428dbd8d51f4b4a6d3e5761077a9a6df4d434f2358ad9c463401d5f9b0ec50bef5ad191b3f5a9bff62975410b7561614025ce4d43759055c3f544684fc2b036f2e3b3ce75ab53f6133f983d85bd0fdeda242243af47b7562a9261eb7f2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "ffc70c0bb914d9431a3986555a55e689bb26d4aa84bb0813216b85bf857f996e2e9d80b9966cf5bb2269db193316ef1a75017ed77c15774a4d0659b4f1f72f0a53db304f5a868fa1b6c73b14283c00b2959c1c2ebc2d0feaa07fbb04ae662fc08f8460108dc37248cb42e7d3e312234a761df1db386342a3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "47c8eb37b273cadfcbc7866ba6423c52099addca2a448fc27117d6e837e565ce8059a8ca54f01d1d1498037371766d816e591c538c37919c47a047c803ee727d647ab6f53e1f94764ce6a203c2150cf68dc42577dea8d795c543da14de263706ea2133c16ff1ea1838ea9b83be43ac3a025063024d3e8853");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "63e7ec0ef57692ec233a77ff61840ab3efd4bf13ef55acfa5daf6a7bbc40a327935b96a9ab03e8032ee795c1c9deaca5e68b9f79c674942a4bea2e495b6384217fe1abbcedcdbfdbacc7b63c674e12caa9bdef26782af542d9c09a3408dbf33edd8c373f505d89aa988e87ac65f9a0f7c98cb1b05d10141f607c0a4cb3db9a24e88d4cf95044ee09");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "d4e12ddd79b0fe4eae5cb6bde2088b1d6d57f764f15deed1558008852aa730c8bb23c81658108b7b96b38fad5f42fa99b07b35025c6cd9d9bc5e12539c2e0be7281e3f9cb54666237b930ce69f0a3079cba1035d0a4c68f963a5ca5dc267aa733e06af417d3880e0587ca600a9e40640df56efdfb3f61184e680991f624e89d0239e973061045121");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "51276676d1037e17ec2e3cea8f944bed76496f74146f9d8e39d3baaf0d0208383e0dcb515b87205fb566c59228681dc3e363f7113689ae807616647249c1d5d7a777851c8bc1d6ddfe670c78ca0a1f697c4c2ab6e12c85ab1f85540907c3791b5b8a81cc82da2e0bdb99338e04e40689761e96597211cce9c1c70ab4f7a0aed9b51e250e7d1fef15");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "f7eba6693b288dc4419ee591a63feabe0e91ba17262fd97e8c786f9ed0aaa079516296a7d84d788cd64310d00e83911a7a6c76769ed7e0e67cbd73d1d765900f9b3db563d49ddcf44859aeb325c61449e323b198d33f2da37df120d68a7af0cf89b7145ea7f486fd7b71dda4ad92a8aa6a3b67e81e43371e6a772e2f782a804be3c3ed87acc95d60");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fab7c2cb53ff1d8a4c0a149e035662fc8bf201b3aaebf91041231fc73790a17116dd63627147d06801d9e0f445c6abbf4a5f10b30fd82c50affa5b90dec320d63dd28fe6fdb44c33c684557744388252cfa6aa70dc7f2c5738808ca8cb2ee8d68bb31443065c6be9bdd8009c6245706dc945eb15660a87b32a3879233462d3a8b2b1fa14dfbe0b5d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "d7c3adaf29ddc4c886f4044d41b139b282d86aecb6c221237a64b03028c34525fec9f6dc8ed76b7f7ce0fbe412f73fc3e5ba620c95b58c8ad9d4d3c9735379e29d4f31e5a2068922e0475ac5c14ff5a0bc0d2059ef1e594cdb0f66b09f03ed7c2893da2cefe2f683b2307755931f7ad6ac7f5fc2c2a1eb8150c32aa82505e010069a5489e1d6365b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "460445d7a3bdbc999636cfa1c500c6d25d298d265cedf32057478fd70713eeb46e267d128cbffc962c4f0cf3dbde7e86c31274c8bdc0d266d0351311e74ae6b89dc56c35af2db0af2cb52ace294bbb5e43991485a7448f01af25973a5974bc0e4ad95940ada7e3d3651722cb885c7f23dac7217ca03b8767fa856cf47499740e605e066d83f08ed0abb9ad48cb1b52b2ba0b14b0ef123aef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "094b1ffdeb599290289688dfe70995766fed9519f237d6f492519ccfc250833596fb226c790b543238d7017b7b9a7beb10df8904d09ccf3afe01b297a3bf92b11f9198f627b7d9aee0aec6c6574056dfcde4f039891a7ee776cf807c712613680e2a70b202f021c23ecc600ad44282c14235a621f118ae1204b02c944ed1be160d73cc1fc3afe618432ca5d86496fc6188b73bf24596470a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "d5e03eb874b3a784482fd5861bc3a13981ddbf80be5a27c6d8d916eab21f138c8cc5476b5a4075637468d5decc108aee377e4268c6f9b84bd45eac82ce1baa8797d178605a19f8b753bf7913c025174addfee58b51667abe82a6c6f641a940ca43d55dc728432fece37ea12cd5185205742b712071d359f4528b5686ee64f838ac63010aced02fe7da9155c659bbd6cebd4b902654a1c954");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "437485af602d8b57fc0497d20787d7971ca86fe465b880c0a0f0ac183fb4dba3b91662ec57283c7862376d57855c6f87ef325bd17d4520984b85428f02e2b6a50769c279a3d7b3c38e15af45ad9d98ac2a74f3ba0ceecdcc7283194758a1d8d5e857c8f96f78e2e2e43dc0622068f02bacf9c154ba159fd01115f47b692d936aedbc410cb43603f87bccf7c1b8677f02be496d2e9eeb8a54");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "a63ecb7d222660054d559f558e29e5796e3afc9ee2f2d4dc84a9d9245ac715145f6b43276fadef74bee248b151991edcd870cd90245f0159af268fb89f9d5535cfb085275063e5a6cd360d844e54f20348e9788bf42075de55491f3514ce011df6c7bc8734451fede29d3cd0004890f7abcff5d88b84869a3ffc0a7838c671f737bbabf682e01c7cb803534d2c3b6fe70898e78dcad18b28");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "6d2dcd7fd473e6d62145c5dbacadc97fc6cd318c17eb69e998716070826a03ab3241aea091acd8a9e80ba304bc1dedd677e7fcbce24e7b4554819e8fd77e8c1d7a045c3d5c613be1da7c81ba781e8c1fac8c51fca88bb01f9139a6f87ba787cda6e283ac7bac541cf282053a33de6b83c63cda641b6e5f322fe0c4a2d6334c4a42395d0f139387a9b0338c09910be961094d02ff06a33e71");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "b545cae69c87e838f53ea62caa2c5d2d4f70f076b4e7e0fe854a43c5b1a95201c4c35cb04dc0ceff5c1cd83cf738c5bd765365815f0f994587656cd5fbf00cb3b45ae806b8597cc1fafe7bb7ae914ff187b6fbe46835fe0de2edbc4ed829025129a7c3fdbc7d5a0ed1c9971f0397db15006677beaf4650c6d218bb2e6c45f530b2b37e20b80f53d7fce4d64f7e51a70e5da7cd5096b3de04debc8f43267cac633ac0633af6740753");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "323bbcc530762c72bfb219e9b9b2171b16facbac28ff948b5d13ba0530040d884ecdf1f2683a3c788fbd831fc99497636a6172b66d350acf674896fb615ba4a2c74dfad03d4f8a5e2a11f9e48ae7aed55d517c4109726d9d9a6dc5a94ca887e179e29c4608404cc0db764acedf79857d73f6b877790e1950ba128fcac11ec04680f923d86c80a1e9c059cf1ba49238fdaab887e723508d3c1d50cc64047beaf8a55cff7a1c057157");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "492588312301f955187de4294ea01e01661cdac76ddb253b4719683bdfe2583029b0116a156e2d42be4725465546e615581983117ee88335d1d353a076aafdde5ce2ebd05a8240b18c1dfd466bb1d6a4eeec8c8089654a2ba19044be62376f6777b2529c2ac53c5776caa408ab4d108980cbdd7b64dcd7e6757afa490420bdbc94386625655135b2bd0f1749d840467d95347c3e1f9467f3d9b200b7bac65aa74551817689a14193");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "252f90f91b4fbdae350df7b9e3bd3dcce398e8dcf39b0e1149546f3dc5cc5145cca04d4f2b807c6f98c1833b1462a33bfca71bb7aad3b4791bf874fc9574387085f8e298a4d822227d233a28df5d53667110e474ad384c7831ddee378b06f5f5dc7af491432c0a0fd53282960a617058c60b21e5e1760beaa88cc6e598ac4b6a6a919cb557ee72b1a16eb661f4c7d9a5018da67244b2cecc6ece80b78f958d7c6126c79f5a12914e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "f0d74f35fa4eac19075a5ca64358e475149b4c85ec563617967a78665fd6cb6831813f5eea1a796c6a0318035c21981d8b8f9d8dcd2195936fb80f104ac2730b7574f815539abb215cef80c2dcefa07a9a2940152d964f6feb1595e56ff18dabf5d133aff434ebb2abdb10d3f5d992d5682cb865cb1adfa02f38a3150afcacfad50b86af31408f4409c4a262463a4ec5aca48462a8342f520f6cd0141fca0750d6119a54ae84b0e8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "dd6937cf2b00616c26bf180a21707ca33e569e99ed7bbf425ddc254c6bf6a7d8f4204eb753e8b2172ef97a87445044ea350ec5846714ac3b42bae398b8dff301ad86cc902b739a5cb4b67b2051aefdeffa805a6cf99e66d2aa68509a46c36c98a964e2d2ec31d9a6c434128a32fc9fd029b7bef5225f61f76d47a15a445c8cd0f9c9737c9195d8f2db55c29f193e2b42ec90d20a3b1b34010eb115978cd5661af0f317602da26e2d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "7184165cc4089186777cc3d103c49b562f3c8192dc8b2b944ae398cea6e6bad8d6d74fdd31ee390db126241d2215ca57d1011e3744ebbbed827f9be505d6f32983ff078c6030586524d2ccea554c48d831e9d754a3c82bfe0a119c65a5b4114801aca02ad8a7c4ddec017a5f540f4d6cc39b81764a9aeaf47a15b9d76492fbaddad3e9fe990629776b9199e872ba60bcde1a7c8ddd0ac4ca614456136413a19743f8c1497842d62cb8b6573e17cf208099b1a8e369745f54");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "7b61c8cb1f98d8f2ffa4a5425ec3c7e54be06c47094a01c794e3c892d60daa9dfb7cd7d3415bb9af0b4c9981aa04620838add770ca429ca05b62604be9446b9e2a5f8be8dc2f0ba5c5fcdf4c4faae9c771fa217e161d984c593fb041815f23077283eea1036f1784c5c0923665b95eb46445dd2f5fcb3923bb760d6b4eaaa56fef0f358e7dbb7de44c857cde6a125a77f07eb031b6db3dbaa1e09a5901a5d276de41c6c76be2942ee79b7924bed119e84719caf0469afbd8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "fa0c3221f3ed91fb7d6832e3fb6d80eae8988f436046b7be1650511cfb6122bec21ee86dfc675a9102e39264ba7a9baa3de46d3ef15cce129e10465500d0834ba540777d66822eb9b1e99b04ab7314ebe4cb7e1c19fd0656ecae40135b51ce933e1a722bf6b5c7e3495ea3f6c929a704af5ff9822a4daa0b51cc496e1803b112eb55a8b0ca674e6007174edf9dd55496bb8ab2ed06fa16b99814524784c8d897f7c5827b489261e25bbbe4f1904aca2fd038266596c0a0b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "8917c4d4387a5de68b9d0277d8f841fa2dcd76d996e42e3f1b397efbbb178dd2c9f5df506ae85b37938409e423bfe6c63abdf9dfdc50dd4dd94261615650160c8e7f78a4374b99e579ec80b505d3345a901e45355d019adecdb042b7074aac4c7b9dbcad241df93da7538af71af6b20ad71a3024b133ee3aff27ccfffdf259448ce34546172bad89030bc0a4af38c9def7b763be3ca05493eb9540057ccebf62485e062fa777ecbc858368cf93cea7583850a6e52e5cacca");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "a5b2039087f3321a4b57a74a3a3d58b0e5675165e341ab6145a29738cebfe34ea62d171c6a0579da1c9507551d38fb27aa606d47a7540729a88bfc3e42de58bfb6c2e55c807e720028b6ea96cc513937678775a48c6789705c4b74f9ef88b182567768f12ab9b0819ea298187436e061fdd439fe9e445d58abbb69ad4c91edc2cfc74963442c92f202db0066219b8201ab7df9c0e4e48285519951a9b1cab1ab2d169c21328eb697e85bb546e4de3b3e3550fad0a1e8ec5e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "ad56663bddee998ad7444b8f3bbe69ee05b61ef2e7774d7f3b3d55cc1c4d93ba5f7d95c75900b11c2eddd38ffd4d4e8a9bdf2a37308c39751e630dc280daa8e9c5cbf1c9b1a7023a351513690cd2e43c56dd7031cd74839578f96a8acac3b9c3b118474eeb3a70c2a376828ca263e49def5edd16c89e884bb6353c353f5dec95980ed9a9684493da51bc6be5c1b63b1869c9de1ac79a5418ac3191a3e24511ac76b6cb2015d72b81b889a42e982f9f8e811f88183c9345a5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "af9bfc35eb7c6c946ff35f810677c93f31872d4a0e5b478d10e562e27fc41e3a40abcf063a792b8796c5779a1504b61eff9bb5a8f0b9bbdf2a3b5584df7fbfd50d04ceea36ec25dafbdd1c3d1312a24da0f8ec96ea1c068be9a64f22553f63fcfaa394249038594b655bda55347beec88f0e60dec9b1bd638d21d51bc12fabc73ab7745b7dc21147f8b94718c54e08dceb8c5bc65887f51e5c2ee0330c62991a786cb8e7549d69427f15ba797b302fc3bd79357aaa4c756f901a984d623cdd34aae8056656e31c1e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "6b2ed3a3fe224c077e1b34ce209d971fa5f3a7464a946894d8e8cc57c3921055063efa426712638cb80635bb72c2e7be34aec518f95eeed3bc8a9da35936097656eb05773b1959cb73b80c4bf871343f0b429da989afb069252fd2e92d60c27ec5a186de9ce5e3aafd438c62cc9425f9cb329d47dd08f460789154c5865d296fb72c7899a5f8bfa2f69fbfc62e85c0b9202b55a4adb69e1eb21244eb02d37e2d62aaa2f363588a03d4f530b17bb4ced021cb7730c022f47044620f2d99f9ada0b0f953e1ec9a025e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "c6a7d098a89f887005fa94d6a2aa43123b684c2f30cbd287b1ef7d4a20f1354318da83a1e478fdcf47b36058abe839c48da1961abcf43806f0f00ea19fa19b6822bbf052d7039371d8b96e1a43dbef56871838ced84ea79e36e848fc2860db578ca33ecd36bcbbb3eebe7ef35d63d6a7ec200345901dc280e37d8684129280e458c3c033a6a2e30ad76766cedad2fefcef6bdbd58b422e729c121891042bb1d134939f5db01f6a4f8885972c9eb547a15900fba4406f0ce9ef893f2f74ebaa0c62359e5b4657c472");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "ef6541e59521d1bad440ad3007d515cce7b00cad5a34f925ec0ac5b5e6927f45d1883e79d1bd06aaaea4b55e7ce41df723051bcef95fbba21c2984ce95ddc7bc6622aa01b1ce1d8b0f53681f826bfb44346ce5419a5aae6b2555466688e2e8e63e1ab54905b60d4bb96562747fb5467d3e76b6b0db0e1b9d33bf1bb92661dd20861a81d973805cb9d3093eb07b59828e3c737451037faf51ee4ce48e0703de1c83ac0231ce95e7c31dd5aa756c2a6e7f65b2a27e1eaed16576fe5ce089ae95759289e6d4cc3bd203");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "82894779aff3270774d23ff694f807d7e8cad9a6b329521518af7277ed13136ead83ba04a13da1d415d30f2f5d20ac732bcb0cc9bd031122f27d438367611a0ad92f07ae8e4646d0ddec19d474b62c32832ed61a8ed4543775490dd4f95a432dd60019c39f85a7ad212150513f351bb8491deaf188f932470a5b8635a7110cf7d0a7001eecd30b786e87f15cad29f2a183fa52d0ad94aeae6ab89201efbb8e710c37840e0616387243a0901ccb82a453d318d206d7d8b3ed3ffa42037074d74bea5d4a3f7c2d6d4f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "52a200f823f5352e22130fe0ea06268a74b719e130a2f594cf18e6a7ad8d3841617cc50386fad359253718657993b70aa299a815984688ee96efa70854081cc92f96bfd5f2477cf4f7865eade47df5b05a553f8ea771aa7c26682823cea750311c289ec2fc3485f166b7f69fbe420a7b9d743f67a7fc951251d82dc39e14a11321a8736ccdc1697986f0dae5ce852b519ea3ab316e3a0d5c50d19a732db11e63dc6b0e5556a91e480492fde6face4c1616243494485c3411e1c3ffc897e25a777b1d3d47e48ea97b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "654fb8cfc44eac7f3f55e5802065be480ecc6efbe7fe9d6424927ba440bf6fd5838985510c2fc6e7653366a8f1f2afdf1ddbd346e789fb671d4b99734fc985fdbbe1c6d87aa53adc36b57c1cbddc3027a152db3ef727937348d2209bc5ce024e657a2d829b37f710333153960b80c4aadb082ae7af05fbfa6d05e65296e29f1765ca1c13cec56648adc447e02811d2d736b92f2a10f509d06039051e35549ba7b7c5f2e00a9c5f48e619da6bebb25f4c26971d717905f637363b68d45cbda31ed19c8c8a4cb3605a4342493b0e1a49008a1d2d8e85815617");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "e69d18f8facae8b4eea1e6417cd85a0a879ff72a31d7568bf1f096e5af985d26eab37d2f34380612a517cabd961dd4d8ba9ad7aa672e39f1d3e910162c9bcd5407fb8e8f7df62bb9250b6b621b15b9787d4eb5a83554a43856b8b406c8d49a6483146c0f34e5a597366fe4981391c7f44ba1dc7f27172dc3940b77194319e5b0d79c8a0814a381b45ab997e34b1e65d5156c987e09e336da6bd1724e32577cde4895f34ac0dc2f18a45c28c7c8e5421a4f8bd8b5c1ce6b37cfed9d382d77f609f75345931bb9f2cba1e8aa3048f05a6a3748a7a51c0f739a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "1f24347381bbc5693140b676a016af70bb4b60a856b2aa8d5f8ebc133b40a8e46a7a6b9058a1101d574a02ebb628b9bbb30b9643a7e62eb5bccbc8006d59d4014643c665d900f4030a7047d068787965d232c7594f5fb975ba23dd75523703e06b471ba98335b4ac7879707def13f914c7c44e2bf79863918db244fead474c9749b934e7eeee6f174b27bc4976feac12f160145fd6eb7932c91750b9a93d35d62768bedc0ec44dbed98f1ecca6ed4b01a1b59591b57307122f36b8d991589408db44d56effbd28a11b47a4d510c4628d853781a97fd40f21");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "540a84844223d6e19e12fe2d60bd750001d53d25fc6ad76bc887f4e78b7e071ab6e804d1827c520a5f18008363da2dede1adb573ba9e88207884764cdfef1a97cf29b309a3c32d13aac596c6d15670cbd593a1057e0a082a4d86f6b80330279bf77910f42332acf5f00f8274e5ceb6c31a32bc1a2e1f8c5771eb1202342b93a69c99800c79369afe8958d70122700a65c6ea19c8ff067c2b7fa3452540b397fbabcdc2d412998ca59e09bc65dfe156104f35fcba6573f48909c00242172dca125d33c2a18ddc6c2b866e1f29ca2276e57ccc670ec1d973d5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "26c31d328b00ed3d841319243f374aec18b1f24cef0982139259b5efc3e4340ec086eeca96d2ae76edc5de5bd662a102eaab85ec6db4c0a48568a2f93f6ef97f0886bdd119eff2d883e445d4c3d6d5adcb8abbed698db68237dbf36c2a2ea401887bb99598672ed771b7e08145d0f47cd259591a2e5ab70fc32302d728617220ef0f0af398dad2d841a5d670ed5328f12d69a6212f446dfe523e55437157cd54bb5a50901e96c5620d8bffa9f6d6fc355a86203754dde77f07a2e29fe1457a4fb2ec61a399dba5c4c158d60bf9650f40549f605d167e23e9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "acf75ce73dc3749d6a0ca0fd0f5a0b3d0cdebe632a5418e2d1f6563a176dc4427f63ed7abb29c69f41750657f442fa4c5d4971398f4d56ca3ddabd513fa52b4d29e9d86383555530b99d6aabccf67ba61bcf97bdb9479164030354ccbf3e06cb49e21c1016abe679a129aa73c7ca6be17706c3cbb4b575e940e0ba075f760a752619280ed2d69823410a068b1e32855d6a2e8640c869d48cff99a599941a127bee52d8f6463d921e5ac275fb1d9ff33cd5952494258f1c779f61034c5c52e9f6d7fbcf2306a9230c51ad501aebc005a75a5e9fd9cb7ef0a4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "ba034c49a834e62bdc5b4227bec6c3900b438ee372a8e49c669ea1c7ae31fa11d8f016b4b7a98ef538c12b610f53b37c1118f0fad98476ea173f57734ac2b68aed1e6c4e607b5d670833f471ff3dd711afc5d6841781d7ce75b91a4161aa05c9194ea715a98d0840765142478e5f840d9d0dd71f2674d4960b6582e8762a8036eb3a91e28cdfb9832265f14af53dabacd268385b43f910044397169cd0fff1cb26d510b80f9a062f27faef4c79102eca63c824d26fa32106b9693ed2d4f3d0ae7fccba16d6c90e4b946f0095e466eaa641803192c6ea5fc2544744448897b269075b16d87467d222");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "5e2ac84125444d77aa7bd7934f5ab38a2a63442a11ac74e0be13c53dd4a84ec8b7c13e8293539356f1972206a25d1ccf7103c1e486327bb301670e8e7c6105e7c0f416b2d6676cc6d6855b92a7a953709307e1cfedd0d1b2a4070776a9ce8acc4a2f1be7545642ab6733072f18f7904093e3227a96738b5e3e8abc5204b620727db504486611cb0bc6a80a30fd9f48b5b2adf20d2a82df219b4522a34beb4d87cb67ee8a17390591bd7be808de3cd46920655fdcc0f257d245c8c2e082098f16c7f778871fc682fae0a2a95af1be46a4533ac8bd54970ebccf39f14bb58253c3ed5ce7c95e9798e5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "9558f7d3b92c1da00e331a9207d8665e34f47144f4ec285492dd6800648821587b4b835f7c8db0f91c61405712defe1daaa606893a42ba1eddbec580afe9c3fce4523b2ce8b7abe37e5170e7c97420c9327dfeae8876e37aa25dd951d3e172e5911b1c4f4fc2d9f73d4a18fe9e93b2459e2871003c903d74710c98137c9d0c6508e11bb6309319b983d0dde45919fffc21ff318d0499d61bdb816792048b8db73ef185e77ec60aba9892af56de34070120a6d1f8240f5a97c312f36c2c690e366a34bcc00984fb4d5f006d40df7305347a6e1f78f9d09617a0a52fdcbc117e5a4c189c5ce98419eb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "c2854e777a2a023db4b1931b5c0a23492f8b0881af6fea64c24b83b7f417d83312015ce88da1e39aa779c9e8be0d99a08575705fb1d3beac92b112cd4737f36819620da465d4a0dd0815cd092ebb834a4ca6cc443cd543152a9854e4310a8130bc9e456b949a6541aee42815a79140bec75a59a8dfd307edce3d728446b43fe9bf5a2d9f236d7c5214edccb8ad78129ad3c1b8dd5eae66fbd14941af7a06de0bb542cfef27f84a4203af3a2cae19d026bbc56e5c620723fc487afee13369f17979dd3b26a3a1f287c0fa75e4af8ff6381a48d729958dcc18117f51c4bef525023ca7c58b82ebb637");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "182305399255cffc24c4e11cdeb3830e3e343cf3bfdb1aa183f2fe19ab7b04cf50d9923cf991cac42caf4c6ea71a18551019cfb127ea835e9d436d4edea051d1b83f0e31a399414df269227c272ccfc1514a76e15f31f9ce44f8fa19763ef749f5f0915965a2134a60c48392243b704859105fc13aa6ab530455d0ac1dc2bec1df9c6f99a83f6ecceac31cbb4efc7e4ae21b23973aae97ddb49a391ed99eb7b7ed98d69620c4491c7c6d91fe4efb8034120c57f071dc85526234b8ba697484bb716527c059531c8c8e6ab4943394a657f5476456b7b6eb23efba873baa08ac657e6aae59a43598eb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "7552c656b19c34ba53c28dc0e5a68917bdbdafabd289dda3880afe70cd2baaf64ba05c6d1fb50f5d695b668b2f3220cd671dfcd0ccaecbeaa9dd8910167e158d390d7102f5ddf302e16c1a867ecbbcb7e15bec87f0fb1acd07871f3c0f6da241c5c6c5f5438c734bc773f93e8e374987e38deab3bf83b9177c0efc78413fcf6d22cf10a5296acd96099f7359f4859deefedc4d6cbfee5f8ea67a58304799b3f18a40052ef750636ef58ec262c95a80665d97a75bff551d5a74339de17b06aed0f36e089e565d268221244acf3277fbfb13ee5455cd9dce1559a9dcb6d3f6b5ac3612357d5cb13f86");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "a657ccf830fc005bf13b0896a95389dc0e3b04de25e981a8e35a603d5a2ac4bd0a8cf810c2d8ad391078fd6b3eeb277991b87418d465d568d49da53c15e84a04faef9ccd7bc41fed12c544d408a9805d191df06e754b23b52bcdd100d5ad938d0b8b9966b9d0198f9c3c50922fdfab515a463e66177f4df077b2229da7e6c6f63b9720432738ab86e7d3da97859e91629d9f1f093796a516dafb7c72b9e1c21240d0566c7950a9088c90fa28b77982d4540c217caaee1e937419d1d3e622b632467a8eb6c2dd084930e5daba64348c9b5cc322dfb5bbcab599437628338bffbb7df3d5f68dfa1934229574a51c9174efb993f88042e64ddc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "f87c824036fa8cbe6d8537e39ced39663cd695b8ba79e935e46f743923a32a700869193de0cc3bc974f2d65482a3f1667008244edee65d76a8d5ff8816e240f37f2a0aa9c46720c353c538889a3a432d5c298a8218962aa16fe772e3860d7e7aa823ab09b932381da442fbe06f8592e3dd671bca0af2378cb960809e25de0079421e26178449d34dbbe4dbcbd898d0131e7eb76758db421d1e5dea00ddb715224cc8f6b73cba3e7680b9485ac453e829a4c0240220dea63079ac785eef8922afc5bc5f84b34360d2e0d0b121642f1ccb2105386e2b538d707906fac6f877520aea4d8e7813be93bddcdd4b0ef8204d18883fe3cabf69c4a9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "ec6bbec0465b0ceed6f2952d3a4f4ba61453333bf95b3a2566199c4ef9d9c1cab2be59e5555e63933fbfa1fb3e1e14b8bc48a2e8156f8fbeacfcd4f8d62e781b8dddc071be9d1533effbd38dba44725322450fdc1abc26a5289cfcc59fa995078fc61996a8e5be0173c65e4e9a64878c076b1b56041e9aa2edb9d0bb6ceddbaf7080b5a01f97df06d2dc0bacf84f1c3acf50aace450091354da7c7422021bf8d75baf8290f8c8b7ee5b67fd84d30f3e964f53e4061208bea4dac962eb592b68f2dc101f4e3ed273d973ce68fccf71a1f108f981e07536244ffaac0c2c4f97923a9e3400095a69e00fdcd34dad29ec74882ac6a29414c7c87");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "29a53f34f196d4b1ed4c19d542dfb000cfde5571ac00bf9eb2644c61866fa75fa76def915fd1d7243c49bfa53255bf6eaa4b34a8e9813591c2caba99afa2bdfaa57e19b872559fda59400cd67d863005cbb35e4b85a60459654b604a3587551dd58340edc14b7a34974e7a507be99ab5faddf6959293ca88b2b97cf75fc25946253a36394ccfab358867de2a2bc1e7b40c40b28e7b1128efc9f94534add0ea27e3a3a42a9281cb91607a7aebed442a481decbb3ddbea35f572c523caeeada00ed513f36eb7a6b4b635be1212b53874b14fc95c64caa748114af81575383a5e95bb862d5dc0379e01da5578306628cef1e5e2bb24ff9f8d18");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "f3e9f2ef579fedea0330ae691504abf71a11fb4048ca6c52cc9099589a6d5b4f36971ea686e002eea719e21c2d8d9b22ce5ded1d75ad980b2e989bf0dff93db4180650fca1a85b855501197cc30ec2da624579721138b80566fcf40b083d6c9eb88eac3acf6b6c346dc068512c0d35c6a5933ea24850ca31eb9a509e1c2c8bbfc26f34ff7789c5f1ceb4fde7d836140f48cf4a34fdcb1a2d5f8a6a7aaf1a0fa1b76058772fe271483cdaa93c4fe95feef88edcdf2c0458622251c5b718c150257e6cbe586b24567796ebd9ca8f06207db5466a731c565ea88bb299655074725c5f4dd2756d03c3e376983dcc7a1bec23692b74391840413d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "537ce62904310d51111551ba8bd6b8666db276d9921f4c8e9067ce0876f4c7fd52a80d661424f2d0d96f1da286219986c258146a9512a9d1f916326e01db4001f02a58f9a5ded2f0dbafc434e2c17fa4aaf00bae0a8ebc6f70ef8b89ac01d746b76d9da95b22b9d3b74e265642a5c03c6dae5bf41b1d4fb0e92eb0594be07745bbc7e1e805c9b8acc0a44f41132433599aef191e7c9d21df112b31b497a39caca505fd6efd550bd6e9838f7f3975e8e0639d1a72bb4af9af350f2924dc7eb0335bdf0849f0f236b710e82824d69ed3006109cf6058ce1c5e8e787d8f1b95d710203ff96fceffea4b7bf31a2a4bd180355356b6c9d8ec54b7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "0a4a88a7e22da9459e74199d18af0f2e4da9eb68eb4156f4c7e86624cad5d8d27c0c803ce7e05ca29501a2066e914a1a74ae59e1f6a4441d535566667150d568c7e304553cea40464ee8ff81d5259e8e682a09cb7b33f85d6f1b38fc852230b100874e4177c35399fec89cd5964d7c1d02b01e34b6cbf9391c467597f10338923372775cab505993a4436b176ef9fe577a77aac0b8ca138b4d6626df4d89c79914db7206d1d2fe6922375df32e83b65dc35004c98b0905ede29ef045cf661fe140434523cb99bf8b9d2c5f00e8dfd581cb4aceaf79d24a865ebda99b09ad94dc43557ab8f07fa79108c432d9a59796e119f7fd8b699174d0358049f00f963806157c72646c41daf4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "1c2c49b2df0537a9a42940d883b7ecf9c8f477dec0466995e8e7f8425979bfecde86060a3d6520f4e719186e1bc2326ddccdc227e1c4e07c2190fabaef1d2da44c32273d84b72ec8192ed6e5bf060e1e498584c114c18b880fa5cecf838156f3e4e3e21b9d3290ed881458bd5f9750015f7c2cc78aff4550eb386297103c061cb4b6d29bc896c632ea12b98cfdcbdbf60a9f061f2d6c5b13d2f23d500a15ed832cefe184353fcb88ecb1df97c385eb623094a7ee750d32fdc182df769e6052872192424e3f94e71f8f6a5c0db413e4d7e7ea469b49cd6d8ca1295e907ca3a39cccd0f88c5cffc9efdac476831ae3c2d726253c0b8ad14aa2ab1e80bf318b5c20cb88cb2cb930956a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "f2713aa0360cba515c6026141c6d606ec2f3ad9c14ab008d764efebc3937bd1e256e69f49a88aef47563bc2bc2e38ab5bf02e21e9606e85a627d651a0ab1d3db98468d1291c19ce88ce6a1a8932ee67cd7d687eb775def557d60df0809c2d69bc1580d9c78918f5ba1266a731d6c059c7fbcd6207f771ffd1bf89bf1a7f3fbbb723ae519a6aa46ed9d19f0a82cc808f2fb14210d842e03d2502a1b9efcab6d4544cfe3a2301db86021438c11ad285f98babf6ba008db6692ba4f322e3f4505c63b4e06c68bb0bcb6071d4f21ba46187fc7fdcda8a1af05206739252447a964bce7d0b92bac78447abc158fbe5385f9803564daadb641242221c658b82bedcd9b85a568c726d5d266");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "881e4a12dfe1648ee9bdeb77b50010ed2ca430ff811b0563c90f5f2d93e6aa8967b700ce5cddf1eda15232f13f3e830d0b0f7ff07101bc5ce9e29b10b67f22a7ca4bcc0f7be113657548756352a7bdb0843dd4e0405425534624ebe99087fd44bc09e4e559d995da5466ee7be3969e643ff57c9daf1a5306d8f92909bfb4bf41a51f404c452d435330c7e341ee87896ebb85461d975759cb8b434a1b56a61c06b91a54a6b811c2800c6943d152cb05b918344bca1301e7eb27c08a5d21b0ca65d354693814806f61909db519930aba185244a490aa600c33ad74e4fef6f9c0b3b3fc9bb1d299436eb32bdb4945d38eeb3f0a7c33145d68fac344013e284c84f7815a3b173c63fd1b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "e8518c59ddf212851a03fffb1ff222b7f4f35d7c0ddb92678c2b9df1554d4bbfe1ff0737f023e25aa2d1138ff18de98cbd572b03d3562d4f89e435aa2fe649fa76c33a873ec617066e8f117a3fd39352cafc9e5ad62744db36bf38848bb1a87cdb2290c0eb95427420c8b6bbd95af360a4f97c737e8d41913984d0981dd0a25fc60bd2dac698dafa3186699c7c51af395f1aecc9e035ed20b150b231617e8e4a2291dd90de68e9faf42742fffeb08ecb8934cb37b03e4cca27c9331061f27be9f588c297bd7cbd878ff0dd81f913062616518a68564581cc2563cee6b58a7da7c1fc8616eab336b83331d11bbbfd2b826c33bd4ec60af07dbefa5d611855c66abe31fdf6f79db3f1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "36191f2deadea101d0e9e33bf908dd7c77ef327f5c19156e110dfe7d8bf3164f723bde11717f26ef7567d1c70e8aa007d92ef5e3c63d878fde029becd34e70c2356ba85fd51fefc04c17e0399ec95913898d52e3b426517f0874b5e3a87fba131aa8b20f438723cc4b1a01c617668ae8a02c5bc0420d208ceb4dfc1c8e2c03ae1e944bd54a13658ba1f14d826e5a9edfb672e353476267a5cb2c959625fbb3695b085cc9b696c7ab7e171fcb35e5c55a991b7f7d6b19771673c66d8c9287a5bd433c458365b3138def2cd27cb66ff5a48f191dc1bd748900eef8aca4f59fb250151e62c7ab45fad3ea7f3b757f381bb64fccc12e4e716441f08b6dbcbe1b48c08e938fa632d2b01e");

                // NIST test vectors
                // From https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/KMAC_samples.pdf
                yield return new NistKmacTestVector(Name: "Sample #1", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "", Mac: "E5780B0D3EA6F7D3A429C5706AA43A00FADBD7D49628839E3187243F456EE14E");
                yield return new NistKmacTestVector(Name: "Sample #2", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "My Tagged Application", Mac: "3B1FBA963CD8B0B59E8C1A6D71888B7143651AF8BA0A7070C0979E2811324AA5");
                yield return new NistKmacTestVector(Name: "Sample #3", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "My Tagged Application", Mac: "1F5B4E6CCA02209E0DCB5CA635B89A15E271ECC760071DFD805FAA38F9729230");
            }
        }
    }
}
