/********************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2018 Michael Cornelison
   source code URL: https://kornelix.net
   contact: kornelix@posteo.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************

   Fotoxx image edit - Edit menu functions

   m_trim_rotate           trim/crop and rotate combination
   m_upright               upright a rotated image
   m_resize                resize (rescale) image
   m_voodoo1               automatic image enhancement with limited smarts
   m_voodoo2               automatic image enhancement with limited smarts
   m_retouch_combo         adjust brightness, contrast, color, white balance
   m_edit_britedist        flatten and/or expand brightness distribution
   m_gradients             magnify brightness gradients to enhance details
   m_flatten               flatten brightness distribution
   m_retinex               rescale pixel RGB brightness range
   m_mirror                horizontal or vertical mirror image
   m_paint_image           paint on the image with a color 
   color_palette           select color from private color palette file
   HSL_chooser             select color from HSL chooser dialog
   m_clone_image           paint image with pixels from elsewhere on the image
   m_blend_image           blend image pixels via mouse painting
   m_add_text              add text to an image
   gentext                 create text graphic with attributes and transparency
   m_add_lines             add lines or arrows to an image
   genline                 create line/arrow graphic with attributes and transparency
   m_paint_edits           paint edit function gradually with the mouse
   m_lever_edits           apply edit function leveraged by RGB level or contrast
   m_plugins               plugins menu function
   m_edit_plugins          add/revise/delete plugin menu functions
   m_run_plugin            run a plugin menu command and update image

*********************************************************************************/

#define EX extern                                                                //  enable extern declarations
#include "fotoxx.h"                                                              //  (variables in fotoxx.h are refs)

/********************************************************************************/

//  trim (crop) and/or rotate an image                                           //  combine trim and rotate
//  preview mode and related code removed

//  fotoxx.h                                                                     //  17.04
//    int      trimx1, trimy1, trimx2, trimy2;                                   //  trim rectangle limits
//    int      trimww, trimhh;                                                   //  trim rectangle width and height

namespace trimrotate
{
   int      ptrimx1, ptrimy1, ptrimx2, ptrimy2;                                  //  prior trim rectangle (this image)
   int      ptrimww, ptrimhh;                                                    //  prior trim size (previous image)
   float    trimR;                                                               //  trim ratio, width/height
   float    rotate_goal, rotate_angle;                                           //  target and actual rotation
   int      E0ww, E0hh, E3ww, E3hh;                                              //  full size image and preview size
   int      Fguidelines, guidelineX, guidelineY;                                 //  horz/vert guidelines for rotate
   int      Fcorner, Fside, Frotate;
   int      KBcorner, KBside;
   int      Fmax = 0;

   editfunc EFtrimrotate;

   void   dialog();
   int    dialog_event(zdialog *zd, cchar *event);
   void   trim_customize();
   void   mousefunc();
   void   KBfunc(int key);
   void   trim_limits();
   void   trim_darkmargins();
   void   trim_final();
   void   rotate_func();
   void   drawlines(cairo_t *cr);
   void   autotrim();
}


//  menu function

void m_trim_rotate(GtkWidget *, cchar *menu)
{
   using namespace trimrotate;

   int         ii, xmargin, ymargin;
   cchar       *trim_message = ZTX("Trim: drag middle to move, drag corners to resize");
   cchar       *rotate_mess = ZTX("Minor rotate: drag right edge with mouse");
   char        text[20];
   zdialog     *zd;
   
   F1_help_topic = "trim_rotate";

   EFtrimrotate.menufunc = m_trim_rotate;                                        //  menu function
   EFtrimrotate.funcname = "trim_rotate";
   EFtrimrotate.Frestart = 1;                                                    //  allow restart
   EFtrimrotate.mousefunc = mousefunc;
   if (! edit_setup(EFtrimrotate)) return;                                       //  setup edit
   
   PXM_addalpha(E0pxm);
   PXM_addalpha(E1pxm);
   PXM_addalpha(E3pxm);

   E0ww = E0pxm->ww;                                                             //  full-size image dimensions
   E0hh = E0pxm->hh;
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

   if (E9pxm) PXM_free(E9pxm);                                                   //  E9 invalid                         17.04
   E9pxm = 0;

   if (menu && strmatch(menu,"keep")) {                                          //  keep preset trim rectangle
      trimww = trimx2 - trimx1;                                                  //  (full image scale)
      trimhh = trimy2 - trimy1;
      trimR = 1.0 * trimww / trimhh;                                             //  trim ratio = width/height
   }
   
   else if (menu && strmatch(menu,"max")) {
      trimww = E0ww;                                                             //  initial trim rectangle, 1x image size
      trimhh = E0hh;
      xmargin = ymargin = 0;
      trimx1 = 0;
      trimx2 = E0ww;
      trimy1 = 0;
      trimy2 = E0hh;
      trimR = 1.0 * E0ww / E0hh;                                                 //  trim ratio = width/height
   }   

   else
   {
      trimww = 0.8 * E0ww;                                                       //  initial trim rectangle, 80% image size
      trimhh = 0.8 * E0hh;
      xmargin = 0.5 * (E0ww - trimww);                                           //  set balanced margins
      ymargin = 0.5 * (E0hh - trimhh);
      trimx1 = xmargin;
      trimx2 = E0ww - xmargin;
      trimy1 = ymargin;
      trimy2 = E0hh - ymargin;
      trimR = 1.0 * trimww / trimhh;                                             //  trim ratio = width/height
   }

   ptrimx1 = 0;                                                                  //  set prior trim rectangle
   ptrimy1 = 0;                                                                  //    = 100% of image
   ptrimx2 = E3ww;
   ptrimy2 = E3hh;

   Fcorner = Fside = KBcorner = KBside = Frotate = 0;                            //  nothing underway
   rotate_goal = rotate_angle = 0;                                               //  initially no rotation

/***
       ___________________________________________________________
      |                                                           |
      |     Trim: drag middle to move, drag corners to resize     |
      |                                                           |
      |  width [____]  height [____]   ratio 1.5                  |
      |  trim size: [Max] [Prev] [Invert] [Auto]  [x] Lock Ratio  |
      |  [1:1] [2:1] [3:2] [4:3] [16:9] [gold] [Customize]        |
      |                                                           |
      |  Minor rotate: drag right edge with mouse   [level]       |
      |  Rotate: degrees [____]                                   |              //  auto-trim removed
      |                                                           |
      |  Rotate 90° [image] [image]   180° [image]                |
      |                                                           |
      |                            [Grid] [Apply] [Done] [Cancel] |
      |___________________________________________________________|

***/

   zd = zdialog_new(ZTX("Trim/Rotate"),Mwin,Bgrid,Bapply,Bdone,Bcancel,null);
   EFtrimrotate.zd = zd;

   zdialog_add_widget(zd,"label","labtrim","dialog",trim_message,"space=3");

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labW","hb1",Bwidth,"space=5");
   zdialog_add_widget(zd,"zspin","width","hb1","20|30000|1|1000");               //  fotoxx.h  18.01
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"label","labH","hb1",Bheight,"space=5");
   zdialog_add_widget(zd,"zspin","height","hb1","20|30000|1|600");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"label","labR","hb1",ZTX("ratio"),"space=5");
   zdialog_add_widget(zd,"label","ratio","hb1","1.67   ");

   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lab2","hb2",ZTX("trim size:"),"space=5");
   zdialog_add_widget(zd,"button","max","hb2",Bmax,"space=5");
   zdialog_add_widget(zd,"button","prev","hb2",Bprev,"space=5");
   zdialog_add_widget(zd,"button","invert","hb2",Binvert,"space=5");
   zdialog_add_widget(zd,"button","auto","hb2",Bauto,"space=5");
   zdialog_add_widget(zd,"check","lock","hb2",ZTX("Lock Ratio"),"space=15");
   
   zdialog_add_ttip(zd,"max",ZTX("maximize trim box"));
   zdialog_add_ttip(zd,"prev",ZTX("use previous size"));
   zdialog_add_ttip(zd,"invert",ZTX("invert width/height"));
   zdialog_add_ttip(zd,"auto",ZTX("trim transparent edges"));
   zdialog_add_ttip(zd,"lock",ZTX("lock width/height ratio"));

   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=3");
   for (ii = 0; ii < 6; ii++)
      zdialog_add_widget(zd,"button",trimbuttons[ii],"hb3",trimbuttons[ii],"space=5");
   zdialog_add_widget(zd,"button","custom","hb3",ZTX("Customize"),"space=5");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");

   zdialog_add_widget(zd,"hbox","hb4","dialog");
   zdialog_add_widget(zd,"label","labrotmess","hb4",rotate_mess,"space=5");
   zdialog_add_widget(zd,"button","level","hb4",ZTX("Level"),"space=10");
   zdialog_add_ttip(zd,"level",ZTX("use EXIF data if available"));

   zdialog_add_widget(zd,"hbox","hb5","dialog");
   zdialog_add_widget(zd,"label","labrotate","hb5",ZTX("Rotate: degrees"),"space=5");
   zdialog_add_widget(zd,"zspin","degrees","hb5","-360|360|0.1|0");

   zdialog_add_widget(zd,"hbox","hb90","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lab90","hb90","Rotate 90°","space=5");
   zdialog_add_widget(zd,"imagebutt","-90","hb90","rotate-left.png","size=32|space=8");
   zdialog_add_widget(zd,"imagebutt","+90","hb90","rotate-right.png","size=32|space=8");
   zdialog_add_widget(zd,"label","lab180","hb90","180°","space=8");
   zdialog_add_widget(zd,"imagebutt","180","hb90","rotate-180.png","size=32");

   zd = EFtrimrotate.zd;

   zdialog_restore_inputs(zd);                                                   //  restore all prior inputs

   zdialog_fetch(zd,"width",ptrimww);                                            //  preserve prior trim size
   zdialog_fetch(zd,"height",ptrimhh);                                           //    for use by [prev] button
   zdialog_stuff(zd,"width",trimww);                                             //  stuff width, height, ratio as
   zdialog_stuff(zd,"height",trimhh);                                            //    pre-calculated for this image
   snprintf(text,20,"%.3f  ",trimR);
   zdialog_stuff(zd,"ratio",text);
   zdialog_stuff(zd,"degrees",0);

   takeMouse(mousefunc,dragcursor);                                              //  connect mouse function
   currgrid = 1;                                                                 //  use trim/rotate grid
   Fzoom = 0;
   trim_darkmargins();

   zdialog_run(zd,dialog_event,"save");                                          //  run dialog - parallel
   
   if (menu && strmatch(menu,"auto")) zdialog_send_event(zd,"auto");

   if (Fmax) {
      Fmax = 0;
      zdialog_send_event(zd,"max");
   }
   
   return;
}


//  dialog event and completion callback function

int trimrotate::dialog_event(zdialog *zd, cchar *event)
{
   using namespace trimrotate;

   static int  flip = 0;
   int         width, height, delta;
   int         ii, rlock;
   float       r1, r2, ratio = 0;
   char        text[20];
   cchar       *pp;
   cchar       *exifkey[1] = { exif_rollangle_key };
   char        *ppv[1];
   
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  cancel

   Fcorner = Fside = KBcorner = KBside = Frotate = 0;                            //  nothing underway

   if (zd->zstat)                                                                //  dialog complete
   {
      if (zd->zstat == 1) {                                                      //  [grid]
         zd->zstat = 0;                                                          //  keep dialog active
         if (! gridsettings[1][GON]) m_gridlines(0,"grid 1");
         else toggle_grid(2);
         return 1;
      }

      if (zd->zstat == 2) {                                                      //  [apply]
         trim_final();
         CEF->Fmods++;
         edit_done(0);
         m_trim_rotate(0,"max");                                                 //  restart edit
         return 1;
      }

      if (zd->zstat == 3) {                                                      //  [done]
         if (trimww == E0ww && trimhh == E0hh) goto cancel;                      //  no change, cancel
         trim_final();
         CEF->Fmods++;
         currgrid = 0;                                                           //  restore normal grid settings
         edit_done(0);
         return 1;
      }

   cancel:
      draw_toplines(2,0);                                                        //  [cancel] - erase trim rectangle
      currgrid = 0;                                                              //  restore normal grid settings
      edit_cancel(0);
      return 1;
   }

   if (strmatch(event,"custom")) {                                               //  [customize] button
      draw_toplines(2,0);                                                        //  erase trim rectangle
      edit_cancel(0);                                                            //  cancel edit
      trim_customize();                                                          //  customize dialog
      m_trim_rotate(0,0);                                                        //  restart edit
      return 1;
   }

   if (strmatch(event,"focus")) {
      takeMouse(mousefunc,dragcursor);                                           //  connect mouse function
      return 1;
   }

   if (strmatch(event,"prev"))                                                   //  use width/height from prior
   {                                                                             //    image trim
      width = ptrimww;
      height = ptrimhh;
      if (width < 20 || height < 20) return 1;                                   //  no value established, ignore
      if (width > E0ww) width = E0ww;
      if (height > E0hh) height = E0hh;
      zdialog_stuff(zd,"width",width);
      zdialog_stuff(zd,"height",height);
      event = "width";                                                           //  process same as manual inputs
   }

   if (strstr("width height",event))                                             //  get direct width/height inputs
   {
      zdialog_fetch(zd,"width",width);                                           //  full image scale
      zdialog_fetch(zd,"height",height);

      zdialog_fetch(zd,"lock",rlock);                                            //  lock ratio on/off

      if (strmatch(event,"width")) {
         if (width > E3ww) width = E3ww;
         if (rlock) {                                                            //  ratio locked
            height = width / trimR + 0.5;                                        //  try to keep ratio
            if (height > E3hh) height = E3hh;
         }
      }

      if (strmatch(event,"height")) {
         if (height > E3hh) height = E3hh;
         if (rlock) {
            width = height * trimR + 0.5;
            if (width > E3ww) width = E3ww;
         }
      }

      flip = 1 - flip;                                                           //  alternates 0, 1, 0, 1 ...

      delta = width - trimww;

      if (delta > 0) {                                                           //  increased width
         trimx1 = trimx1 - delta / 2;                                            //  left and right sides equally
         trimx2 = trimx2 + delta / 2;
         if (delta % 2) {                                                        //  if increase is odd
            trimx1 = trimx1 - flip;                                              //    add 1 alternatively to each side
            trimx2 = trimx2 + 1 - flip;
         }
         if (trimx1 < 0) {                                                       //  add balance to upper limit
            trimx2 = trimx2 - trimx1;
            trimx1 = 0;
         }
         if (trimx2 > E3ww) {                                                    //  add balance to lower limit
            trimx1 = trimx1 - trimx2 + E3ww;
            trimx2 = E3ww;
         }
      }

      if (delta < 0) {                                                           //  decreased width
         trimx1 = trimx1 - delta / 2;
         trimx2 = trimx2 + delta / 2;
         if (delta % 2) {
            trimx1 = trimx1 + flip;
            trimx2 = trimx2 - 1 + flip;
         }
      }

      delta = height - trimhh;

      if (delta > 0) {                                                           //  increased height
         trimy1 = trimy1 - delta / 2;                                            //  top and bottom sides equally
         trimy2 = trimy2 + delta / 2;
         if (delta % 2) {                                                        //  if increase is odd
            trimy1 = trimy1 - flip;                                              //    add 1 alternatively to each side
            trimy2 = trimy2 + 1 - flip;
         }
         if (trimy1 < 0) {
            trimy2 = trimy2 - trimy1;
            trimy1 = 0;
         }
         if (trimy2 > E3hh) {
            trimy1 = trimy1 - trimy2 + E3hh;
            trimy2 = E3hh;
         }
      }

      if (delta < 0) {                                                           //  decreased height
         trimy1 = trimy1 - delta / 2;
         trimy2 = trimy2 + delta / 2;
         if (delta % 2) {
            trimy1 = trimy1 + flip;
            trimy2 = trimy2 - 1 + flip;
         }
      }

      if (trimx1 < 0) trimx1 = 0;                                                //  keep within limits
      if (trimx2 > E3ww) trimx2 = E3ww;                                          //  use ww not ww-1
      if (trimy1 < 0) trimy1 = 0;
      if (trimy2 > E3hh) trimy2 = E3hh;

      width = trimx2 - trimx1;                                                   //  new width and height
      height = trimy2 - trimy1;

      if (width > E3ww) width = E3ww;                                            //  limit to actual size
      if (height > E3hh) height = E3hh;

      trimww = width;                                                            //  new trim size
      trimhh = height;

      zdialog_stuff(zd,"width",width);                                           //  update dialog values
      zdialog_stuff(zd,"height",height);

      if (! rlock)                                                               //  set new ratio if not locked
         trimR = 1.0 * trimww / trimhh;

      snprintf(text,20,"%.3f  ",trimR);                                          //  stuff new ratio
      zdialog_stuff(zd,"ratio",text);

      trim_darkmargins();                                                        //  show trim area in image
      return 1;
   }

   if (strmatch(event,"max"))                                                    //  maximize trim rectangle
   {
      trimx1 = 0;
      trimy1 = 0;
      trimx2 = E3ww;
      trimy2 = E3hh;

      width = E3ww;                                                              //  full scale trim size
      height = E3hh;

      trimww = width;                                                            //  new trim size
      trimhh = height;

      zdialog_stuff(zd,"width",width);                                           //  update dialog values
      zdialog_stuff(zd,"height",height);

      trimR = 1.0 * trimww / trimhh;
      snprintf(text,20,"%.3f  ",trimR);                                          //  stuff new ratio
      zdialog_stuff(zd,"ratio",text);
      zdialog_stuff(zd,"lock",0);                                                //  set no ratio lock

      trim_darkmargins();                                                        //  show trim area in image
      Fpaintnow();                                                               //  17.08
      return 1;
   }

   if (strmatch(event,"invert"))                                                 //  invert width/height dimensions
      ratio = 1.0 / trimR;                                                       //  mark ratio changed

   for (ii = 0; ii < 6; ii++)                                                    //  trim ratio buttons
      if (strmatch(event,trimbuttons[ii])) break;
   if (ii < 6) {
      r1 = r2 = ratio = 0;
      pp = strField(trimratios[ii],':',1);
      if (pp) r1 = atof(pp);
      pp = strField(trimratios[ii],':',2);
      if (pp) r2 = atof(pp);
      if (r1 > 0 && r2 > 0) ratio = r1/r2;
      if (ratio < 0.1 || ratio > 10) ratio = 1.0;
      if (! ratio) return 1;
      zdialog_stuff(zd,"lock",1);                                                //  assume lock is wanted
      trimR = ratio;
   }

   if (ratio)                                                                    //  ratio was changed
   {
      trimR = ratio;

      if (trimx2 - trimx1 > trimy2 - trimy1)
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;                            //  adjust smaller dimension
      else
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;

      if (trimx2 > E3ww) {                                                       //  if off the right edge,
         trimx2 = E3ww;                                                          //  adjust height
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;
      }

      if (trimy2 > E3hh) {                                                       //  if off the bottom edge,
         trimy2 = E3hh;                                                          //  adjust width
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;
      }

      trimww = trimx2 - trimx1;                                                  //  new rectangle dimensions
      trimhh = trimy2 - trimy1;

      zdialog_stuff(zd,"width",trimww);                                          //  stuff width, height, ratio
      zdialog_stuff(zd,"height",trimhh);
      snprintf(text,20,"%.3f  ",trimR);
      zdialog_stuff(zd,"ratio",text);

      trim_darkmargins();                                                        //  update trim area in image
      return 1;
   }
   
   if (strmatch(event,"level"))                                                  //  auto level using EXIF RollAngle
   {
      exif_get(curr_file,exifkey,ppv,1);
      if (ppv[0]) {
         rotate_goal = atof(ppv[0]);
         zfree(ppv[0]);
         rotate_func();                                                          //  E3 is rotated E1
      }
   }

   if (strstr("+90 -90 180 degrees",event))                                      //  rotate action
   {
      if (strmatch(event,"+90"))                                                 //  90 deg. CW
         rotate_goal += 90;

      if (strmatch(event,"-90"))                                                 //  90 deg. CCW
         rotate_goal -= 90;

      if (strmatch(event,"180"))                                                 //  180 deg. (upside down)
         rotate_goal += 180;

      if (strmatch(event,"degrees"))                                             //  degrees adjustment
         zdialog_fetch(zd,"degrees",rotate_goal);
      
      if (rotate_goal > 180) rotate_goal -= 360;                                 //  keep within -180 to +180 deg.
      if (rotate_goal < -180) rotate_goal += 360;
      if (fabsf(rotate_goal) < 0.01) rotate_goal = 0;                            //  = 0 within my precision

      zdialog_stuff(zd,"degrees",rotate_goal);
      rotate_func();                                                             //  E3 is rotated E1
   }

   if (strmatch(event,"auto"))                                                   //  auto trim margins
   {
      if (CEF->Fmods) {                                                          //  save pending edit
         dialog_event(zd,"max");
         trim_final();
         edit_done(0);
      }
      else edit_cancel(0);
      autotrim();                                                                //  do auto-trim
      m_trim_rotate(0,"keep");                                                   //  restart edit
      return 1;
   }

   return 1;
}


//  trim/rotate mouse function

void trimrotate::mousefunc()
{
   using namespace trimrotate;

   int         mpx, mpy, xdrag, ydrag;
   int         moveall = 0;
   int         dx, dy, dd, dc, ds;
   int         d1, d2, d3, d4;
   zdialog     *zd = EFtrimrotate.zd;

   if (! LMclick && ! RMclick && ! Mxdrag && ! Mydrag) {                         //  no click or drag
      Fcorner = Fside = Frotate = 0;
      return;
   }

   if (LMclick)                                                                  //  add vertical and horizontal
   {                                                                             //    lines at click position
      LMclick = 0;
      Fcorner = Fside = KBcorner = KBside = Frotate = 0;
      Fguidelines = 1;
      guidelineX = Mxclick;
      guidelineY = Myclick;
      trim_darkmargins();
      Fpaint2();
      return;
   }

   if (RMclick)                                                                  //  remove the lines
   {
      RMclick = 0;
      Fcorner = Fside = KBcorner = KBside = Frotate = 0;
      Fguidelines = 0;
      trim_darkmargins();
      Fpaint2();
      return;
   }

   if (Mxdrag || Mydrag)                                                         //  drag
   {
      mpx = Mxdrag;
      mpy = Mydrag;
      xdrag = Mxdrag - Mxdown;
      ydrag = Mydrag - Mydown;
      Mxdown = Mxdrag;                                                           //  reset drag origin
      Mydown = Mydrag;
      Mxdrag = Mydrag = 0;                                                       //  17.04
   }

   if (Frotate) goto mouse_rotate;                                               //  continue rotate in progress

   else if (Fcorner)
   {
      if (Fcorner == 1) { trimx1 = mpx; trimy1 = mpy; }                          //  continue dragging corner with mouse
      if (Fcorner == 2) { trimx2 = mpx; trimy1 = mpy; }
      if (Fcorner == 3) { trimx2 = mpx; trimy2 = mpy; }
      if (Fcorner == 4) { trimx1 = mpx; trimy2 = mpy; }
   }
   
   else if (Fside)
   {
      if (Fside == 1) trimx1 = mpx;                                              //  continue dragging side with mouse  17.04
      if (Fside == 2) trimy1 = mpy;
      if (Fside == 3) trimx2 = mpx;
      if (Fside == 4) trimy2 = mpy;
   }

   else
   {
      moveall = 1;
      dd = 0.2 * (trimx2 - trimx1);                                              //  test if mouse is in the broad
      if (mpx < trimx1 + dd) moveall = 0;                                        //    middle of the rectangle
      if (mpx > trimx2 - dd) moveall = 0;
      dd = 0.2 * (trimy2 - trimy1);
      if (mpy < trimy1 + dd) moveall = 0;
      if (mpy > trimy2 - dd) moveall = 0;
   }

   if (moveall) {                                                                //  yes, move the whole rectangle
      trimx1 += xdrag;
      trimx2 += xdrag;
      trimy1 += ydrag;
      trimy2 += ydrag;
      Fcorner = Fside = KBcorner = KBside = Frotate = 0;
   }

   else if (! Fcorner && ! Fside)                                                //  no, find closest corner/side to mouse
   {
      dx = mpx - trimx1;
      dy = mpy - trimy1;
      d1 = sqrt(dx*dx + dy*dy);                                                  //  distance from NW corner

      dx = mpx - trimx2;
      dy = mpy - trimy1;
      d2 = sqrt(dx*dx + dy*dy);                                                  //  NE

      dx = mpx - trimx2;
      dy = mpy - trimy2;
      d3 = sqrt(dx*dx + dy*dy);                                                  //  SE

      dx = mpx - trimx1;
      dy = mpy - trimy2;
      d4 = sqrt(dx*dx + dy*dy);                                                  //  SW

      Fcorner = 1;                                                               //  find closest corner
      dc = d1;                                                                   //  NW
      if (d2 < dc) { Fcorner = 2; dc = d2; }                                     //  NE
      if (d3 < dc) { Fcorner = 3; dc = d3; }                                     //  SE
      if (d4 < dc) { Fcorner = 4; dc = d4; }                                     //  SW

      dx = mpx - trimx1;
      dy = mpy - (trimy1 + trimy2) / 2;
      d1 = sqrt(dx*dx + dy*dy);                                                  //  distance from left side middle

      dx = mpx - (trimx1 + trimx2) / 2;
      dy = mpy - trimy1;
      d2 = sqrt(dx*dx + dy*dy);                                                  //  top middle

      dx = mpx - trimx2;
      dy = mpy - (trimy1 + trimy2) / 2;
      d3 = sqrt(dx*dx + dy*dy);                                                  //  right side middle

      dx = mpx - (trimx1 + trimx2) / 2;
      dy = mpy - trimy2;
      d4 = sqrt(dx*dx + dy*dy);                                                  //  bottom middle

      Fside = 1;                                                                 //  find closest side
      ds = d1;                                                                   //  left
      if (d2 < ds) { Fside = 2; ds = d2; }                                       //  top
      if (d3 < ds) { Fside = 3; ds = d3; }                                       //  right
      if (d4 < ds) { Fside = 4; ds = d4; }                                       //  bottom

      if (dc < ds) Fside = 0;                                                    //  closer to corner
      else Fcorner = 0;                                                          //  closer to side
      
      if (Fside == 3 && E3ww - mpx < ds)                                         //  closest to right side,
         goto mouse_rotate;                                                      //    and even closer to right edge

      if (Fcorner == 1) { trimx1 = mpx; trimy1 = mpy; }                          //  move this corner to mouse
      if (Fcorner == 2) { trimx2 = mpx; trimy1 = mpy; }
      if (Fcorner == 3) { trimx2 = mpx; trimy2 = mpy; }
      if (Fcorner == 4) { trimx1 = mpx; trimy2 = mpy; }
      
      if (Fside == 1) trimx1 = mpx;                                              //  move this side to mouse
      if (Fside == 2) trimy1 = mpy;
      if (Fside == 3) trimx2 = mpx;
      if (Fside == 4) trimy2 = mpy;
      
      KBcorner = Fcorner;                                                        //  save last corner/side moved
      KBside = Fside;
   }

   trim_limits();                                                                //  check margin limits and adjust if req.
   trim_darkmargins();                                                           //  show trim area in image

   return;

// ------------------------------------------------------------------------

   mouse_rotate:

   Frotate = 1;
   Fcorner = Fside = KBcorner = KBside = 0;

   rotate_goal += 40.0 * ydrag / E3ww;                                           //  convert radians to degrees (reduced)
   if (rotate_goal > 180) rotate_goal -= 360;                                    //  keep within -180 to +180 deg.
   if (rotate_goal < -180) rotate_goal += 360;
   if (fabsf(rotate_goal) < 0.01) rotate_goal = 0;                               //  = 0 within my precision
   zdialog_stuff(zd,"degrees",rotate_goal);                                      //  update dialog

   rotate_func();                                                                //  E3 is rotated E1
   return;
}


//  Keyboard function
//  KB arrow keys tweak the last selected corner or side

void trimrotate::KBfunc(int key)
{
   using namespace trimrotate;

   int      xstep, ystep;

   xstep = ystep = 0;
   if (key == GDK_KEY_Left) xstep = -1;
   if (key == GDK_KEY_Right) xstep = +1;
   if (key == GDK_KEY_Up) ystep = -1;
   if (key == GDK_KEY_Down) ystep = +1;

   if (KBcorner == 1) {                                                          //  NW
      trimx1 += xstep;
      trimy1 += ystep;
   }

   if (KBcorner == 2) {                                                          //  NE
      trimx2 += xstep;
      trimy1 += ystep;
   }

   if (KBcorner == 3) {                                                          //  SE
      trimx2 += xstep;
      trimy2 += ystep;
   }

   if (KBcorner == 4) {                                                          //  SW
      trimx1 += xstep;
      trimy2 += ystep;
   }
   
   if (KBside == 1) trimx1 += xstep;                                             //  left
   if (KBside == 2) trimy1 += ystep;                                             //  top
   if (KBside == 3) trimx2 += xstep;                                             //  right
   if (KBside == 4) trimy2 += ystep;                                             //  bottom
   
   trim_limits();                                                                //  check margin limits and adjust if req.
   trim_darkmargins();                                                           //  show trim area in image

   return;
}


//  check new margins for sanity and enforce locked aspect ratio

void trimrotate::trim_limits()
{
   using namespace trimrotate;

   int         rlock, chop;
   int         ww, hh, ww2, hh2;
   float       drr;
   char        text[20];
   zdialog     *zd = EFtrimrotate.zd;

   if (trimx1 > trimx2-10) trimx1 = trimx2-10;                                   //  sanity limits
   if (trimy1 > trimy2-10) trimy1 = trimy2-10;

   zdialog_fetch(zd,"lock",rlock);                                               //  w/h ratio locked
   if (rlock && Fcorner) {
      if (Fcorner < 3)
         trimy1 = trimy2 - 1.0 * (trimx2 - trimx1) / trimR;
      else
         trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
   }
   
   if (rlock && Fside) {                                                         //  17.04
      ww = trimx2 - trimx1;
      hh = trimy2 - trimy1;
      ww2 = ww;
      hh2 = hh;
      if (Fside == 1 || Fside == 3) hh2 = ww / trimR;
      else ww2 = hh * trimR;
      ww2 = (ww2 - ww) / 2;
      hh2 = (hh2 - hh) / 2;
      trimx1 -= ww2;
      trimx2 += ww2;
      trimy1 -= hh2;
      trimy2 += hh2;
   }

   chop = 0;
   if (trimx1 < 0) {                                                             //  look for off the edge
      trimx1 = 0;                                                                //    after corner move
      chop = 1;
   }

   if (trimx2 > E3ww) {
      trimx2 = E3ww;
      chop = 2;
   }

   if (trimy1 < 0) {
      trimy1 = 0;
      chop = 3;
   }

   if (trimy2 > E3hh) {
      trimy2 = E3hh;
      chop = 4;
   }

   if (rlock && chop) {                                                          //  keep ratio if off edge
      if (chop < 3)
         trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
      else
         trimx2 = trimx1 + 1.0 * (trimy2 - trimy1) * trimR;
   }

   if (trimx1 > trimx2-10) trimx1 = trimx2-10;                                   //  sanity limits
   if (trimy1 > trimy2-10) trimy1 = trimy2-10;

   if (trimx1 < 0) trimx1 = 0;                                                   //  keep within visible area
   if (trimx2 > E3ww) trimx2 = E3ww;
   if (trimy1 < 0) trimy1 = 0;
   if (trimy2 > E3hh) trimy2 = E3hh;

   trimww = trimx2 - trimx1;                                                     //  new rectangle dimensions
   trimhh = trimy2 - trimy1;

   drr = 1.0 * trimww / trimhh;                                                  //  new w/h ratio
   if (! rlock) trimR = drr;

   zdialog_stuff(zd,"width",trimww);                                             //  stuff width, height, ratio
   zdialog_stuff(zd,"height",trimhh);
   snprintf(text,20,"%.3f  ",trimR);
   zdialog_stuff(zd,"ratio",text);

   return;
}


//  Darken image pixels outside of current trim margins.
//  messy logic: update pixmaps only for changed pixels to increase speed

void trimrotate::trim_darkmargins()
{
   using namespace trimrotate;

   int      ox1, oy1, ox2, oy2;                                                  //  outer trim rectangle
   int      nx1, ny1, nx2, ny2;                                                  //  inner trim rectangle
   int      px, py;
   float    *pix1, *pix3;
   int      nc = E1pxm->nc, pcc = nc * sizeof(float);

   if (trimx1 < 0) trimx1 = 0;                                                   //  keep within visible area
   if (trimx2 > E3ww) trimx2 = E3ww;
   if (trimy1 < 0) trimy1 = 0;
   if (trimy2 > E3hh) trimy2 = E3hh;

   if (ptrimx1 < trimx1) ox1 = ptrimx1;                                          //  outer rectangle
   else ox1 = trimx1;
   if (ptrimx2 > trimx2) ox2 = ptrimx2;
   else ox2 = trimx2;
   if (ptrimy1 < trimy1) oy1 = ptrimy1;
   else oy1 = trimy1;
   if (ptrimy2 > trimy2) oy2 = ptrimy2;
   else oy2 = trimy2;

   if (ptrimx1 > trimx1) nx1 = ptrimx1;                                          //  inner rectangle
   else nx1 = trimx1;
   if (ptrimx2 < trimx2) nx2 = ptrimx2;
   else nx2 = trimx2;
   if (ptrimy1 > trimy1) ny1 = ptrimy1;
   else ny1 = trimy1;
   if (ptrimy2 < trimy2) ny2 = ptrimy2;
   else ny2 = trimy2;

   ox1 -= 2;                                                                     //  expand outer rectangle
   oy1 -= 2;
   ox2 += 2;
   oy2 += 2;
   nx1 += 2;                                                                     //  reduce inner rectangle
   ny1 += 2;
   nx2 -= 2;
   ny2 -= 2;

   if (ox1 < 0) ox1 = 0;
   if (oy1 < 0) oy1 = 0;
   if (ox2 > E3ww) ox2 = E3ww;
   if (oy2 > E3hh) oy2 = E3hh;

   if (nx1 < ox1) nx1 = ox1;
   if (ny1 < oy1) ny1 = oy1;
   if (nx2 > ox2) nx2 = ox2;
   if (ny2 > oy2) ny2 = oy2;

   if (nx2 < nx1) nx2 = nx1;
   if (ny2 < ny1) ny2 = ny1;

   if (! E9pxm) E9pxm = PXM_copy(E3pxm);                                         //  17.04

   for (py = oy1; py < ny1; py++)                                                //  top band of pixels
   for (px = ox1; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimx1 || px >= trimx2 ||
          py < trimy1 || py >= trimy2) {
         pix3[0] = pix1[0] / 2;                                                  //  outside trim margins
         pix3[1] = pix1[1] / 2;                                                  //  50% brightness
         pix3[2] = pix1[2] / 2;
      }
      else memcpy(pix3,pix1,pcc);                                                //  inside, full brightness
   }

   for (py = oy1; py < oy2; py++)                                                //  right band
   for (px = nx2; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimx1 || px >= trimx2 || 
          py < trimy1 || py >= trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else memcpy(pix3,pix1,pcc);
   }

   for (py = ny2; py < oy2; py++)                                                //  bottom band
   for (px = ox1; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimx1 || px >= trimx2 || 
          py < trimy1 || py >= trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else memcpy(pix3,pix1,pcc);
   }

   for (py = oy1; py < oy2; py++)                                                //  left band
   for (px = ox1; px < nx1; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimx1 || px >= trimx2 || 
          py < trimy1 || py >= trimy2) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else memcpy(pix3,pix1,pcc);
   }

   cairo_t *cr = draw_context_create(gdkwin,draw_context);
   
   Fpaint3(ox1,oy1,ox2-ox1,ny1-oy1,cr);                                          //  4 updated rectangles
   Fpaint3(nx2,oy1,ox2-nx2,oy2-oy1,cr);
   Fpaint3(ox1,ny2,ox2-ox1,oy2-ny2,cr);
   Fpaint3(ox1,oy1,nx1-ox1,oy2-oy1,cr);

   drawlines(cr);                                                                //  draw trim margin lines

   draw_context_destroy(draw_context); 
   
   ptrimx1 = trimx1;                                                             //  set prior trim rectangle
   ptrimx2 = trimx2;                                                             //    from current trim rectangle
   ptrimy1 = trimy1;
   ptrimy2 = trimy2;

   if (gridsettings[currgrid][GON]) Fpaint2();                                   //  refresh gridlines, delayed 

   return;
}


//  final trim - cut margins off
//  E3 is the input image, rotated and with black edges cropped if chosen
//  E3 is the output image from user trim rectangle

void trimrotate::trim_final()
{
   using namespace trimrotate;

   int      px1, py1, px2, py2;
   float    *pix3, *pix9;
   int      nc = E3pxm->nc, pcc = nc * sizeof(float);
   
   draw_toplines(2,0);                                                           //  erase trim rectangle

   if (trimx2 > E3ww) trimx2 = E3ww;                                             //  final check
   if (trimy2 > E3hh) trimy2 = E3hh;

   trimww = trimx2 - trimx1;                                                     //  new rectangle dimensions
   trimhh = trimy2 - trimy1;

   if (E9pxm) PXM_free(E9pxm);
   E9pxm = PXM_make(trimww,trimhh,nc);                                           //  new pixmap with requested size

   for (py1 = trimy1; py1 < trimy2; py1++)                                       //  copy E3 (rotated) to new size
   for (px1 = trimx1; px1 < trimx2; px1++)
   {
      px2 = px1 - trimx1;
      py2 = py1 - trimy1;
      pix3 = PXMpix(E3pxm,px1,py1);
      pix9 = PXMpix(E9pxm,px2,py2);
      memcpy(pix9,pix3,pcc);
   }

   PXM_free(E3pxm);
   E3pxm = E9pxm;
   E9pxm = 0;

   E3ww = E3pxm->ww;                                                             //  update E3 dimensions
   E3hh = E3pxm->hh;

   CEF->Fmods++;
   CEF->Fsaved = 0;

   Fpaint2();
   return;
}


//  rotate function
//  E3 = E1 with rotation

void trimrotate::rotate_func()
{
   using namespace trimrotate;

   zdialog     *zd = EFtrimrotate.zd;
   
   if (fabs(rotate_goal) < 0.01) {                                               //  no rotation
      rotate_goal = rotate_angle = 0.0;
      zdialog_stuff(zd,"degrees",0.0);
      PXM_free(E3pxm);                                                           //  E1 >> E3
      E3pxm = PXM_copy(E1pxm);
   }

   if (rotate_goal != rotate_angle) {                                            //  17.04
      paintlock(1);                                                              //  block window updates (rotate is thread)
      PXM_free(E3pxm);
      E3pxm = PXM_rotate(E1pxm,rotate_goal,1);                                   //  E3 is rotated E1            faster 17.04
      rotate_angle = rotate_goal;
      paintlock(0);                                                              //  unblock window updates
   }

   if (E9pxm) PXM_free(E9pxm);                                                   //  E9 invalid                         17.04
   E9pxm = 0;

   E3ww = E3pxm->ww;                                                             //  update E3 dimensions
   E3hh = E3pxm->hh;

   if (trimx2 > E3ww) trimx2 = E3ww;                                             //  contract trim rectangle if needed
   if (trimy2 > E3hh) trimy2 = E3hh;

   trimww = trimx2 - trimx1;                                                     //  new rectangle dimensions
   trimhh = trimy2 - trimy1;

   zdialog_stuff(zd,"width",trimww);                                             //  stuff width, height
   zdialog_stuff(zd,"height",trimhh);

   ptrimx1 = 0;                                                                  //  set prior trim rectangle
   ptrimy1 = 0;
   ptrimx2 = E3ww;                                                               //    = 100% of image
   ptrimy2 = E3hh;
   
   CEF->Fmods++;

   Fpaintnow();                                                                  //  bugfix                             17.08
   return;
}


//  draw lines on image: trim rectangle, guidelines, gridlines

void trimrotate::drawlines(cairo_t *cr)
{
   using namespace trimrotate;

   Ntoplines = 4;                                                                //  outline trim rectangle
   toplines[0].x1 = trimx1;
   toplines[0].y1 = trimy1;
   toplines[0].x2 = trimx2;
   toplines[0].y2 = trimy1;
   toplines[0].type = 1;
   toplines[1].x1 = trimx2;
   toplines[1].y1 = trimy1;
   toplines[1].x2 = trimx2;
   toplines[1].y2 = trimy2;
   toplines[1].type = 1;
   toplines[2].x1 = trimx2;
   toplines[2].y1 = trimy2;
   toplines[2].x2 = trimx1;
   toplines[2].y2 = trimy2;
   toplines[2].type = 1;
   toplines[3].x1 = trimx1;
   toplines[3].y1 = trimy2;
   toplines[3].x2 = trimx1;
   toplines[3].y2 = trimy1;
   toplines[3].type = 1;

   if (Fguidelines) {                                                            //  vert/horz rotate guidelines
      Ntoplines = 6;
      toplines[4].x1 = guidelineX;
      toplines[4].y1 = 0;
      toplines[4].x2 = guidelineX;
      toplines[4].y2 = E3hh-1;
      toplines[4].type = 2;
      toplines[5].x1 = 0;
      toplines[5].y1 = guidelineY;
      toplines[5].x2 = E3ww-1;
      toplines[5].y2 = guidelineY;
      toplines[5].type = 2;
   }

   draw_toplines(1,cr);                                                          //  draw trim rectangle and guidelines

   return;
}


//  auto-trim image - set trim rectangle to exclude transparent regions

void trimrotate::autotrim()
{
   using namespace trimrotate;

   PXM      *pxm = 0;
   int      px1, py1, px2, py2;
   int      qx1, qy1, qx2, qy2;
   int      qx, qy, step1, step2;
   int      area1, area2, Fgrow = 0;
   float    *ppix;

   if (! E0pxm) return;                                                          //  no edit image
   pxm = E0pxm;
   if (pxm->nc < 4) return;                                                      //  no alpha channel

   px1 = 0.4 * pxm->ww;                                                          //  select small rectangle in the middle
   py1 = 0.4 * pxm->hh;
   px2 = 0.6 * pxm->ww;
   py2 = 0.6 * pxm->hh;

   step1 = 0.02 * (pxm->ww + pxm->hh);                                           //  start with big search steps
   step2 = 0.2 * step1;
   if (step2 < 1) step2 = 1;
   
   while (true)
   {
      while (true)
      {
         Fgrow = 0;

         area1 = (px2 - px1) * (py2 - py1);                                      //  area of current selection rectangle
         area2 = area1;

         for (qx1 = px1-step1; qx1 <= px1+step1; qx1 += step2)                   //  loop, vary NW and SE corners +/-
         for (qy1 = py1-step1; qy1 <= py1+step1; qy1 += step2)
         for (qx2 = px2-step1; qx2 <= px2+step1; qx2 += step2)
         for (qy2 = py2-step1; qy2 <= py2+step1; qy2 += step2)
         {
            if (qx1 < 0) continue;                                               //  check image limits
            if (qy1 < 0) continue;
            if (qx1 > 0.5 * pxm->ww) continue;
            if (qy1 > 0.5 * pxm->hh) continue;
            if (qx2 > pxm->ww-1) continue;
            if (qy2 > pxm->hh-1) continue;
            if (qx2 < 0.5 * pxm->ww) continue;
            if (qy2 < 0.5 * pxm->hh) continue;

            ppix = PXMpix(pxm,qx1,qy1);                                          //  check 4 corners are not
            if (ppix[3] < 1) continue;                                           //    in transparent zones
            ppix = PXMpix(pxm,qx2,qy1);
            if (ppix[3] < 1) continue;
            ppix = PXMpix(pxm,qx2,qy2);
            if (ppix[3] < 1) continue;
            ppix = PXMpix(pxm,qx1,qy2);
            if (ppix[3] < 1) continue;

            area2 = (qx2 - qx1) * (qy2 - qy1);                                   //  look for larger enclosed area
            if (area2 <= area1) continue;

            for (qx = qx1; qx < qx2; qx++) {                                     //  check top/bottom sides not intersect
               ppix = PXMpix(pxm,qx,qy1);                                        //    transparent zones
               if (ppix[3] < 1) break;
               ppix = PXMpix(pxm,qx,qy2);
               if (ppix[3] < 1) break;
            }
            if (qx < qx2) continue;

            for (qy = qy1; qy < qy2; qy++) {                                     //  also left/right sides
               ppix = PXMpix(pxm,qx1,qy);
               if (ppix[3] < 1) break;
               ppix = PXMpix(pxm,qx2,qy);
               if (ppix[3] < 1) break;
            }
            if (qy < qy2) continue;

            Fgrow = 1;                                                           //  successfully grew the rectangle
            px1 = qx1;                                                           //  new bigger rectangle coordinates
            py1 = qy1;                                                           //    for the next search iteration
            px2 = qx2;
            py2 = qy2;
            goto breakout;                                                       //  leave all 4 loops
         }

      breakout:
         if (! Fgrow) break;
      }

      if (step2 == 1) break;                                                     //  done

      step1 = 0.6 * step1;                                                       //  reduce search step size
      if (step1 < 2) step1 = 2;
      step2 = 0.2 * step1;
      if (step2 < 1) step2 = 1;
   }

   trimx1 = px1;                                                                 //  set parameters for trim function
   trimx2 = px2;
   trimy1 = py1;
   trimy2 = py2;

   return;
}


//  dialog to get custom trim button names and corresponding ratios

void trimrotate::trim_customize()
{
   using namespace trimrotate;
   
   char        text[20], blab[8], rlab[8];
   float       r1, r2, ratio;
   cchar       *pp;
   int         ii, zstat;

/***
       ___________________________________________________________
      |          Trim Buttons                                     |
      |                                                           |
      |  label  [ 5:4 ] [ 4:3 ] [ 8:5 ] [ 16:9 ] [ 2:1 ] [ gold ] |
      |  ratio  [ 5:4 ] [ 4:3 ] [ 8:5 ] [ 16:9 ] [ 2:1 ] [1.62:1] |
      |                                                           |
      |                                           [Done] [Cancel] |
      |___________________________________________________________|
      
***/

   zdialog *zd = zdialog_new(ZTX("Trim Buttons"),Mwin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbb","dialog",0,"homog|space=3");
   zdialog_add_widget(zd,"hbox","hbr","dialog",0,"homog|space=3");
   zdialog_add_widget(zd,"label","label","hbb",ZTX("label"),"space=3");
   zdialog_add_widget(zd,"label","ratio","hbr",ZTX("ratio"),"space=3");

   strcpy(blab,"butt-0");
   strcpy(rlab,"ratio-0");

   for (ii = 0; ii < 6; ii++)                                                    //  add current trimbuttons to dialog
   {
      blab[5] = '0' + ii;
      rlab[6] = '0' + ii;
      zdialog_add_widget(zd,"zentry",blab,"hbb",trimbuttons[ii],"size=6");
      zdialog_add_widget(zd,"zentry",rlab,"hbr",trimratios[ii],"size=6");
   }

   zdialog_set_modal(zd);                                                        //  17.08
   zdialog_run(zd,0,"mouse");                                                    //  run dialog
   zstat = zdialog_wait(zd);                                                     //  wait for complete

   if (zstat == 1)                                                               //  done
   {
      for (ii = 0; ii < 6; ii++)                                                 //  get new button names
      {
         blab[5] = '0' + ii;
         zdialog_fetch(zd,blab,text,12);
         strTrim2(text);
         if (! *text) continue;
         zfree(trimbuttons[ii]);
         trimbuttons[ii] = zstrdup(text);
      }

      for (ii = 0; ii < 6; ii++)                                                 //  get new ratios
      {
         rlab[6] = '0' + ii;
         zdialog_fetch(zd,rlab,text,12);
         strTrim2(text);
         r1 = r2 = ratio = 0;
         pp = strField(text,':',1);
         if (! pp) continue;
         r1 = atof(pp);
         pp = strField(text,':',2);
         if (! pp) continue;
         r2 = atof(pp);
         if (r1 > 0 && r2 > 0) ratio = r1/r2;
         if (ratio < 0.1 || ratio > 10) continue;
         zfree(trimratios[ii]);
         trimratios[ii] = zstrdup(text);
      }
   }

   zdialog_free(zd);                                                             //  kill dialog
   save_params();                                                                //  save parameters

   return;
}


/********************************************************************************/

//  Upright a rotated image: -90, +90, 180 degrees.
//  This is not an edit transaction: file is replaced and re-opened.

void m_upright(GtkWidget *, cchar *menu)
{
   int upright_dialog_event(zdialog *zd, cchar *event);
   
   zdialog  *zd;

   F1_help_topic = "upright";

   if (checkpend("all")) return;

   if (clicked_file) {                                                           //  use clicked file if present
      if (! strmatch(clicked_file,curr_file))
         f_open(clicked_file,0,0,1,0);                                           //  avoid f_open() re-entry            18.01
      clicked_file = 0;
   }

   if (! curr_file) {
      if (zdupright) zdialog_free(zdupright);
      zdupright = 0;
      return;
   }

/***
       _____________________________________________________
      |      Upright Image                                  |
      |                                                     |
      | Image File: xxxxxxxxxxxxxxx.jpg                     |
      |   _______   _______   _______   _______   _______   |
      |  |       | |       | |       | |       | |       |  |
      |  |upright| |  -90  | |  +90  | |  180  | |   X   |  |
      |  |_______| |_______| |_______| |_______| |_______|  |
      |_____________________________________________________|

***/

   if (! zdupright) {
      zd = zdialog_new(ZTX("Upright Image"),Mwin,null);
      zdupright = zd;
      zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=5");
      zdialog_add_widget(zd,"label","labf","hbf",ZTX("Image File:"),"space=3");
      zdialog_add_widget(zd,"label","file","hbf","filename.jpg","space=5");
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"button","upright","hb1",ZTX("Upright"),"space=5");
      zdialog_add_widget(zd,"imagebutt","-90","hb1","rotate-left.png","size=32|space=5");
      zdialog_add_widget(zd,"imagebutt","+90","hb1","rotate-right.png","size=32|space=5");
      zdialog_add_widget(zd,"imagebutt","180","hb1","rotate-180.png","size=32|space=5");
      zdialog_add_widget(zd,"imagebutt","cancel","hb1","cancel.png","size=32|space=5");
      zdialog_run(zd,upright_dialog_event,"save");
   }
   
   zd = zdupright;

   char *pp = strrchr(curr_file,'/');                                            //  stuff file name in dialog
   if (pp) zdialog_stuff(zd,"file",pp+1);

   return;
}


//  dialog event and completion function

int upright_dialog_event(zdialog *zd, cchar *event)
{
   int      angle = 0;
   char     orientation = 0, *ppv[1];
   cchar    *exifkey[1] = { exif_orientation_key };
   cchar    *exifdata[1];
   
   if (! curr_file) return 1;

   if (zd->zstat) {                                                              //  [x] button
      zdialog_free(zd);
      zdupright = 0;
      return 1;
   }

   if (strmatch(event,"cancel")) {                                               //  cancel button
      zdialog_free(zd);
      zdupright = 0;
      return 1;
   }

   if (! strstr("upright -90 +90 180",event)) return 1;                          //  ignore other events
   
   if (strmatch(event,"upright"))                                                //  auto upright button
   {
      exif_get(curr_file,exifkey,ppv,1);                                         //  get EXIF: Orientation
      if (ppv[0]) {
         orientation = *ppv[0];                                                  //  single character
         zfree(ppv[0]);
      }
      if (orientation == '6') angle = +90;                                       //  rotate clockwise 90 deg.
      if (orientation == '8') angle = -90;                                       //  counterclockwise 90 deg.
      if (orientation == '3') angle = 180;                                       //  180 deg.
      if (! angle) {
         zmessageACK(Mwin,ZTX("rotation unknown"));
         return 1;
      }
   }

   if (strmatch(event,"+90")) angle += 90;                                       //  other buttons
   if (strmatch(event,"-90")) angle -= 90;
   if (strmatch(event,"180")) angle += 180;

   if (! angle) return 1;                                                        //  ignore other events
   
   if (checkpend("all")) return 1;

   paintlock(1);                                                                 //  block window updates
   Fblock = 1;
   E0pxm = PXM_load(curr_file,1);                                                //  load poss. 16-bit image
   if (E0pxm) {
      E3pxm = PXM_rotate(E0pxm,angle);                                           //  rotate (threaded)
      PXM_free(E0pxm);
      E0pxm = E3pxm;
      E3pxm = 0;
   }
   paintlock(0);                                                                 //  unblock window updates

   f_save(curr_file,curr_file_type,curr_file_bpc,1);

   PXM_free(E0pxm);

   exifdata[0] = (char *) "";                                                    //  remove EXIF orientation data
   exif_put(curr_file,exifkey,exifdata,1);                                       //  (f_save() omits if no edit made)

   update_image_index(curr_file);                                                //  update index data

   Fblock = 0;

   f_open(curr_file);
   gallery(curr_file,"paint",0);                                                 //  repaint gallery
   return 1;
}


/********************************************************************************/

//  Resize (rescale) image
//
//  Output pixels are composites of input pixels, e.g. 2/3 size means
//  that 3x3 input pixels are mapped into 2x2 output pixels, and an
//  image size of 1000 x 600 becomes 667 x 400.

int       resize_ww0, resize_hh0;                                                //  original size
int       resize_ww1, resize_hh1;                                                //  new size
float     resize_lockratio;                                                      //  lock w/h ratio
editfunc  EFresize;


void m_resize(GtkWidget *, cchar *menu)
{
   int    resize_dialog_event(zdialog *zd, cchar *event);
   void * resize_thread(void *);

   F1_help_topic = "resize_image";
   
   char     rtext[12];

   EFresize.menuname = menu;
   EFresize.menufunc = m_resize;
   EFresize.funcname = "resize";
   EFresize.Frestart = 1;                                                        //  allow restart
   EFresize.Fscript = 1;                                                         //  scripting supported
   EFresize.threadfunc = resize_thread;                                          //  thread function
   if (! edit_setup(EFresize)) return;                                           //  setup edit

/****
          _________________________________________________________________
         |                                                                 |
         |             pixels    percent                                   |
         |  width     [_____]    [_____]                                   |
         |  height    [_____]    [_____]                                   |
         |                                                                 |
         |  [x] 1/1  [_] 3/4  [_] 2/3  [_] 1/2  [_] 1/3  [_] 1/4  [_] Prev |
         |                                                                 |
         |  W/H Ratio: 1.333    Lock [_]                                   |
         |                                                [done] [cancel]  |
         |_________________________________________________________________|
       
****/

   zdialog *zd = zdialog_new(ZTX("Resize Image"),Mwin,Bdone,Bcancel,null);
   EFresize.zd = zd;
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","placeholder","vb11",0);
   zdialog_add_widget(zd,"label","labw","vb11",Bwidth);
   zdialog_add_widget(zd,"label","labh","vb11",Bheight);
   zdialog_add_widget(zd,"label","labpix","vb12","pixels");
   zdialog_add_widget(zd,"zspin","wpix","vb12","20|30000|1|20");                 //  fotoxx.h                           18.01
   zdialog_add_widget(zd,"zspin","hpix","vb12","20|30000|1|20");
   zdialog_add_widget(zd,"label","labpct","vb13",Bpercent);
   zdialog_add_widget(zd,"zspin","wpct","vb13","1|400|1|100");                   //  step size = 1.0 %                  17.04
   zdialog_add_widget(zd,"zspin","hpct","vb13","1|400|1|100");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","space","hb2",0);
   zdialog_add_widget(zd,"check","1/1","hb2","1/1");
   zdialog_add_widget(zd,"check","3/4","hb2","3/4");
   zdialog_add_widget(zd,"check","2/3","hb2","2/3");
   zdialog_add_widget(zd,"check","1/2","hb2","1/2");
   zdialog_add_widget(zd,"check","1/3","hb2","1/3");
   zdialog_add_widget(zd,"check","1/4","hb2","1/4");
   zdialog_add_widget(zd,"check","prev","hb2",Bprev);
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labratio","hb3",ZTX("W/H Ratio:"),"space=5");
   zdialog_add_widget(zd,"label","ratio","hb3","1.333");
   zdialog_add_widget(zd,"check","lockratio","hb3",ZTX("Lock"),"space=10");
   
   zdialog_add_ttip(zd,"prev",ZTX("use previous settings"));

   resize_ww0 = E1pxm->ww;                                                       //  original width, height
   resize_hh0 = E1pxm->hh;
   zdialog_stuff(zd,"wpix",resize_ww0);
   zdialog_stuff(zd,"hpix",resize_hh0);
   zdialog_stuff(zd,"lockratio",1);                                              //  default ratio locked
   
   zdialog_stuff(zd,"1/1",1);                                                    //  begin with full size
   zdialog_stuff(zd,"3/4",0);
   zdialog_stuff(zd,"2/3",0);
   zdialog_stuff(zd,"1/2",0);
   zdialog_stuff(zd,"1/3",0);
   zdialog_stuff(zd,"1/4",0);
   zdialog_stuff(zd,"prev",0);

   resize_lockratio = 1.0 * resize_ww0 / resize_hh0;
   snprintf(rtext,12,"%.3f",resize_lockratio);
   zdialog_stuff(zd,"ratio",rtext);

   zdialog_run(zd,resize_dialog_event,"save");                                   //  run dialog
   return;
}


//  dialog event and completion callback function

int resize_dialog_event(zdialog *zd, cchar *event)                               //  overhauled for scripting
{
   int         lock, nn;
   double      wwxhh;
   float       wpct1, hpct1, ratio;
   char        rtext[12];
   
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  cancel

   if (zd->zstat)                                                                //  dialog complete
   {
      if (zd->zstat == 1) {                                                      //  done
         editresize[0] = resize_ww1;                                             //  remember size used
         editresize[1] = resize_hh1;
         edit_done(0);
      }
      else edit_cancel(0);                                                       //  cancel or kill
      Fzoom = 0;
      return 1;
   }

   if (strmatch(event,"focus")) return 1;                                        //  ignore focus
   
   if (event[0] =='w' || event[0] == 'h') {                                      //  if width or height set directly,
      zdialog_stuff(zd,"1/1",0);                                                 //    set all ratio buttons off
      zdialog_stuff(zd,"3/4",0);
      zdialog_stuff(zd,"2/3",0);
      zdialog_stuff(zd,"1/2",0);
      zdialog_stuff(zd,"1/3",0);
      zdialog_stuff(zd,"1/4",0);
      zdialog_stuff(zd,"prev",0);
   }
   
   if (strstr("1/1 3/4 2/3 1/2 1/3 1/4 prev",event)) {                           //  a ratio button was selected
      zdialog_stuff(zd,"1/1",0);                                                 //  set all ratio buttons off
      zdialog_stuff(zd,"3/4",0);
      zdialog_stuff(zd,"2/3",0);
      zdialog_stuff(zd,"1/2",0);
      zdialog_stuff(zd,"1/3",0);
      zdialog_stuff(zd,"1/4",0);
      zdialog_stuff(zd,"prev",0);
      zdialog_stuff(zd,event,1);                                                 //  set selected button on
   }

   zdialog_fetch(zd,"wpix",resize_ww1);                                          //  get width and height values
   zdialog_fetch(zd,"hpix",resize_hh1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lockratio",lock);                                           //  lock ratio on/off
   
   if (strmatch(event,"lockratio") && lock)                                      //  lock ratio enabled
      resize_lockratio = 1.0 * resize_ww1 / resize_hh1;                          //  update lock ratio to hold
   
   if (strmatch(event,"wpct"))                                                   //  width % - set pixel width
      resize_ww1 = int(wpct1 / 100.0 * resize_ww0 + 0.5);

   if (strmatch(event,"hpct"))                                                   //  height % - set pixel height
      resize_hh1 = int(hpct1 / 100.0 * resize_hh0 + 0.5);

   if (lock && event[0] == 'w')                                                  //  preserve width/height ratio
      resize_hh1 = int(resize_ww1 * 1.0 / resize_lockratio + 0.5);

   if (lock && event[0] == 'h')
      resize_ww1 = int(resize_hh1 * resize_lockratio + 0.5);

   zdialog_fetch(zd,"1/1",nn);                                                   //  if a ratio was selected,
   if (nn) {                                                                     //    set image size accordingly
      resize_ww1 = resize_ww0;
      resize_hh1 = resize_hh0;
   }

   zdialog_fetch(zd,"3/4",nn);
   if (nn) {
      resize_ww1 = (3 * resize_ww0 + 3) / 4;
      resize_hh1 = (3 * resize_hh0 + 3) / 4;
   }

   zdialog_fetch(zd,"2/3",nn);
   if (nn) {
      resize_ww1 = (2 * resize_ww0 + 2) / 3;
      resize_hh1 = (2 * resize_hh0 + 2) / 3;
   }

   zdialog_fetch(zd,"1/2",nn);
   if (nn) {
      resize_ww1 = (resize_ww0 + 1) / 2;
      resize_hh1 = (resize_hh0 + 1) / 2;
   }

   zdialog_fetch(zd,"1/3",nn);
   if (nn) {
      resize_ww1 = (resize_ww0 + 2) / 3;
      resize_hh1 = (resize_hh0 + 2) / 3;
   }

   zdialog_fetch(zd,"1/4",nn);
   if (nn) {
      resize_ww1 = (resize_ww0 + 3) / 4;
      resize_hh1 = (resize_hh0 + 3) / 4;
   }

   zdialog_fetch(zd,"prev",nn);                                                  //  set previously used size
   if (nn) {
      resize_ww1 = editresize[0];
      resize_hh1 = editresize[1];
   }

   if (resize_ww1 < 20 || resize_hh1 < 20) return 1;                             //  refuse tiny size
   
   ratio = 1.0 * resize_ww1 / resize_hh1;                                        //  w/h ratio 

   if (resize_ww1 > wwhh_limit1) {                                               //  enforce limits                     18.01
      resize_ww1 = wwhh_limit1;
      resize_hh1 = resize_ww1 / ratio;
   }

   if (resize_hh1 > wwhh_limit1) {
      resize_hh1 = wwhh_limit1;
      resize_ww1 = resize_hh1 * ratio;
   }
   
   wwxhh = 1.0 * resize_ww1 * resize_hh1;                                        //  enforce image size limit           18.01
   if (wwxhh > wwhh_limit2) {
      resize_ww1 *= wwhh_limit2 / wwxhh;
      resize_hh1 *= wwhh_limit2 / wwxhh;
   }
   
   hpct1 = 100.0 * resize_hh1 / resize_hh0;                                      //  set percents to match pixels
   wpct1 = 100.0 * resize_ww1 / resize_ww0;
   
   zdialog_stuff(zd,"wpix",resize_ww1);                                          //  update all widget values
   zdialog_stuff(zd,"hpix",resize_hh1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   ratio = 1.0 * resize_ww1 / resize_hh1;                                        //  update w/h ratio 
   snprintf(rtext,12,"%.3f",ratio);
   zdialog_stuff(zd,"ratio",rtext);

   signal_thread();                                                              //  update image, no wait for idle
   return 1;
}


//  do the resize job

void * resize_thread(void *)
{
   PXM   *pxmtemp1, *pxmtemp2;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for signal
      pxmtemp1 = PXM_rescale(E1pxm,resize_ww1,resize_hh1);                       //  rescale the edit image
      paintlock(1);                                                              //  block window updates
      pxmtemp2 = E3pxm;
      E3pxm = pxmtemp1;
      PXM_free(pxmtemp2);
      paintlock(0);                                                              //  unblock window updates
      CEF->Fmods++;
      CEF->Fsaved = 0;
      Fpaint2();
   }

   return 0;
}


/********************************************************************************/

//  automatic image tuneup without user guidance

float       voodoo1_brdist[256];                                                 //  pixel count per brightness level
int         voodoo_01, voodoo_99;                                                //  1% and 99% brightness levels (0 - 255)
void        voodoo1_distribution();                                              //  compute brightness distribution
void *      voodoo1_thread(void *);
editfunc    EFvoodoo1;


void m_voodoo1(GtkWidget *, cchar *menu)
{
   F1_help_topic = "voodoo";

   EFvoodoo1.menuname = menu;
   EFvoodoo1.menufunc = m_voodoo1;
   EFvoodoo1.funcname = "voodoo1";
   EFvoodoo1.Farea = 1;                                                          //  select area ignored
   EFvoodoo1.Fscript = 1;                                                        //  scripting supported
   EFvoodoo1.threadfunc = voodoo1_thread;
   if (! edit_setup(EFvoodoo1)) return;                                          //  setup edit

   voodoo1_distribution();                                                       //  compute brightness distribution
   signal_thread();                                                              //  start update thread
   edit_done(0);                                                                 //  edit done
   return;
}
   

//  compute brightness distribution for image

void voodoo1_distribution()
{
   int         px, py, ii;
   int         ww = E1pxm->ww, hh = E1pxm->hh;
   float       bright1;
   float       *pix1;

   for (ii = 0; ii < 256; ii++)                                                  //  clear brightness distribution data
      voodoo1_brdist[ii] = 0;

   for (py = 0; py < hh; py++)                                                   //  compute brightness distribution
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright1 = pixbright(pix1);
      voodoo1_brdist[int(bright1)]++;
   }

   for (ii = 1; ii < 256; ii++)                                                  //  cumulative brightness distribution
      voodoo1_brdist[ii] += voodoo1_brdist[ii-1];                                //   0 ... (ww * hh)

   voodoo_01 = 0;
   voodoo_99 = 255;

   for (ii = 0; ii < 256; ii++)                                                  //  compute index values (0 - 255)
   {                                                                             //    for darkest 1% and brightest 1%
      if (voodoo1_brdist[ii] < 0.01 * ww * hh) voodoo_01 = ii;
      if (voodoo1_brdist[ii] < 0.99 * ww * hh) voodoo_99 = ii;
   }

   for (ii = 0; ii < 256; ii++)
      voodoo1_brdist[ii] = voodoo1_brdist[ii]                                    //  multiplier per brightness level
                    / (ww * hh) * 256.0 / (ii + 1);                              //  ( = 1 for a flat distribution)
   return;
}


//  thread function - use multiple working threads

void * voodoo1_thread(void *)
{
   void  * voodoo1_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      for (int ii = 0; ii < NWT; ii++)                                           //  start worker threads
         start_wthread(voodoo1_wthread,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      CEF->Fmods++;
      CEF->Fsaved = 0;
      Fpaint2();                                                                 //  update window
   }

   return 0;                                                                     //  not executed, stop g++ warning
}

void * voodoo1_wthread(void *arg)                                                //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py;
   float       *pix1, *pix3;
   float       bright1, bright2, bright3, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   float       flat1 = 0.3;                                                      //  strength of distribution flatten
   float       flat2 = 1.0 - flat1;
   float       sat1 = 0.3, sat2;                                                 //  strength of saturation increase
   float       f1, f2;
   float       expand = 256.0 / (voodoo_99 - voodoo_01 + 1);                     //  brightness range expander

   for (py = index; py < E1pxm->hh; py += NWT)                                   //  voodoo brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright2 = 0.25 * red1 + 0.65 * green1 + 0.10 * blue1;                      //  input brightness, 0 - 256
      bright2 = (bright2 - voodoo_01) * expand;                                  //  expand to clip low / high 1%
      if (bright2 < 0) bright2 = 0;
      if (bright2 > 255) bright2 = 255;

      bright1 = voodoo1_brdist[int(bright2)];                                    //  factor for flat output brightness
      bright3 = flat1 * bright1 + flat2;                                         //  attenuate

      f1 = (256.0 - bright2) / 256.0;                                            //  bright2 = 0 - 256  >>  f1 = 1 - 0
      f2 = 1.0 - f1;                                                             //  prevent banding in bright areas
      bright3 = f1 * bright3 + f2;                                               //  tends to 1.0 for brighter pixels

      red3 = red1 * bright3;                                                     //  blend new and old brightness
      green3 = green1 * bright3;
      blue3 = blue1 * bright3;

      bright3 = 0.333 * (red3 + green3 + blue3);                                 //  mean color brightness
      sat2 = sat1 * (256.0 - bright3) / 256.0;                                   //  bright3 = 0 - 256  >>  sat2 = sat1 - 0
      red3 = red3 + sat2 * (red3 - bright3);                                     //  amplified color, max for dark pixels
      green3 = green3 + sat2 * (green3 - bright3);
      blue3 = blue3 + sat2 * (blue3 - bright3);

      if (red3 < 0) red3 = 0;                                                    //  stop possible underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      cmax = red3;                                                               //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************/

//  flatten brightness distribution based on the distribution of nearby zones

namespace voodoo2_names
{
   float    flatten = 30;                          //  flatten value, 30%
   float    deband = 70;                           //  deband value, 70%
   int      Iww, Ihh;                              //  image dimensions
   int      NZ;                                    //  no. of image zones
   int      Zsize, Zrows, Zcols;                   //  zone parameters
   float    *Br;                                   //  Br[py][px]  pixel brightness
   int      *Zxlo, *Zylo, *Zxhi, *Zyhi;            //  Zxlo[ii] etc.  zone ii pixel range
   int      *Zcen;                                 //  Zcen[ii][2]  zone ii center (py,px)
   uint8    *Zn;                                   //  Zn[py][px][9]  9 nearest zones for pixel: 0-200 (255 = none)
   uint8    *Zw;                                   //  Zw[py][px][9]  zone weights for pixel: 0-100 = 1.0
   float    *Zff;                                  //  Zff[ii][1000]  zone ii flatten factors

   editfunc EFvoodoo2;                             //  edit function
   void * wthread(void *);                         //  working thread
}


void m_voodoo2(GtkWidget *, cchar *menu)
{
   using namespace voodoo2_names;

   int      gNZ, pNZ;
   int      px, py, cx, cy;
   int      rx, ii, jj, kk;
   int      zww, zhh, row, col;
   float    *pix1, bright;
   float    weight[9], sumweight;
   float    D, Dthresh;
   
   F1_help_topic = "voodoo";

   EFvoodoo2.menuname = menu;
   EFvoodoo2.menufunc = m_voodoo2;
   EFvoodoo2.funcname = "voodoo2";
   EFvoodoo2.Farea = 1;                                                          //  select area ignored
   EFvoodoo2.Fscript = 1;                                                        //  scripting supported

   if (! edit_setup(EFvoodoo2)) return;                                          //  setup edit

   Iww = E1pxm->ww;                                                              //  image dimensions
   Ihh = E1pxm->hh;
   
   if (Iww * Ihh > 227 * MEGA) {
      zmessageACK(Mwin,"image too big");
      edit_cancel(0);
      return;
   }

   Ffuncbusy = 1;

   NZ = 40;                                                                      //  zone count
   gNZ = pNZ = NZ;                                                               //  zone count goal

   while (true)
   {
      Zsize = sqrtf(Iww * Ihh / gNZ);                                            //  approx. zone size
      Zrows = Ihh / Zsize + 0.5;                                                 //  get appropriate rows and cols
      Zcols = Iww / Zsize + 0.5;
      NZ = Zrows * Zcols;                                                        //  NZ matching rows x cols
      if (NZ >= pNZ) break;
      gNZ++;
   }

   Br = (float *) zmalloc(Iww * Ihh * sizeof(float));                            //  allocate memory
   Zn = (uint8 *) zmalloc(Iww * Ihh * 9 * sizeof(uint8));                        //  cannot exceed 2048 M
   Zw = (uint8 *) zmalloc(Iww * Ihh * 9 * sizeof(uint8));
   Zxlo = (int *) zmalloc(NZ * sizeof(int));
   Zylo = (int *) zmalloc(NZ * sizeof(int));
   Zxhi = (int *) zmalloc(NZ * sizeof(int));
   Zyhi = (int *) zmalloc(NZ * sizeof(int));
   Zcen = (int *) zmalloc(NZ * 2 * sizeof(int));
   Zff = (float *) zmalloc(NZ * 1000 * sizeof(float));

   for (py = 0; py < Ihh; py++)                                                  //  get initial pixel brightness levels
   for (px = 0; px < Iww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright = pixbright(pix1);
      ii = py * Iww + px;
      Br[ii] = bright;
      zmainloop(1000);                                                           //  keep GTK alive
   }

   zww = Iww / Zcols;                                                            //  actual zone size
   zhh = Ihh / Zrows;

   for (row = 0; row < Zrows; row++)
   for (col = 0; col < Zcols; col++)                                             //  set low/high bounds per zone
   {
      ii = row * Zcols + col;
      Zxlo[ii] = col * zww;
      Zylo[ii] = row * zhh;
      Zxhi[ii] = Zxlo[ii] + zww;
      Zyhi[ii] = Zylo[ii] + zhh;
      Zcen[2*ii] = Zylo[ii] + zhh/2;
      Zcen[2*ii+1] = Zxlo[ii] + zww/2;
   }

   for (ii = 0; ii < NZ; ii++)                                                   //  compute brightness distributiion
   {                                                                             //    for each zone
      for (jj = 0; jj < 1000; jj++)
         Zff[1000*ii+jj] = 0;

      for (py = Zylo[ii]; py < Zyhi[ii]; py++)                                   //  brightness distribution
      for (px = Zxlo[ii]; px < Zxhi[ii]; px++)
      {
         pix1 = PXMpix(E1pxm,px,py);
         bright = 1000.0 / 256.0 * pixbright(pix1);
         Zff[1000*ii+int(bright)]++;
      }

      for (jj = 1; jj < 1000; jj++)                                              //  cumulative brightness distribution
         Zff[1000*ii+jj] += Zff[1000*ii+jj-1];

      for (jj = 0; jj < 1000; jj++)                                              //  multiplier per brightness level
         Zff[1000*ii+jj] = Zff[1000*ii+jj] / (zww*zhh) * 1000 / (jj+1);          //    to make distribution flat

      zmainloop(1000);                                                           //  keep GTK alive
   }

   for (py = 0; py < Ihh; py++)                                                  //  set 9 nearest zones per pixel
   for (px = 0; px < Iww; px++)
   {
      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      row = py / zhh;                                                            //  zone containing pixel
      col = px / zww;

      ii = 0;
      for (jj = row-1; jj <= row+1; jj++)                                        //  loop 3x3 surrounding zones
      for (kk = col-1; kk <= col+1; kk++)
      {
         if (jj >= 0 && jj < Zrows && kk >= 0 && kk < Zcols)                     //  zone is not off the edge
            Zn[rx+ii] = jj * Zcols + kk;
         else Zn[rx+ii] = 255;                                                   //  edge row/col: missing neighbor
         ii++;
      }
   }

   if (zww < zhh)                                                                //  pixel to zone distance threshold
      Dthresh = 1.5 * zww;                                                       //  area influence = 0 beyond this distance
   else Dthresh = 1.5 * zhh;

   for (py = 0; py < Ihh; py++)                                                  //  set zone weights per pixel
   for (px = 0; px < Iww; px++)
   {
      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      for (ii = 0; ii < 9; ii++)                                                 //  distance to each zone center
      {
         jj = Zn[rx+ii];
         if (jj == 255) {
            weight[ii] = 0;                                                      //  missing zone weight = 0
            continue;
         }
         cy = Zcen[2*jj];
         cx = Zcen[2*jj+1];
         D = sqrtf((py-cy)*(py-cy) + (px-cx)*(px-cx));                           //  distance from pixel to zone center
         D = D / Dthresh;
         if (D > 1) D = 1;                                                       //  zone influence reaches zero at Dthresh
         weight[ii] = 1.0 - D;
      }

      sumweight = 0;
      for (ii = 0; ii < 9; ii++)                                                 //  zone weights based on distance
         sumweight += weight[ii];

      for (ii = 0; ii < 9; ii++)                                                 //  normalize weights, sum = 1.0
         Zw[rx+ii] = 100 * weight[ii] / sumweight;                               //  0-100 = 1.0

      zmainloop(1000);                                                           //  keep GTK alive
   }

   for (int ii = 0; ii < NWT; ii++)                                              //  start worker thread per processor core
      start_wthread(wthread,&Nval[ii]);
   wait_wthreads();                                                              //  wait for completion

   zfree(Br);                                                                    //  free memory
   zfree(Zn);
   zfree(Zw);
   zfree(Zxlo);
   zfree(Zylo);
   zfree(Zxhi);
   zfree(Zyhi);
   zfree(Zcen);
   zfree(Zff);

   Ffuncbusy = 0;
   CEF->Fmods++;
   CEF->Fsaved = 0;
   edit_done(0);                                                                 //  edit done
   return;
}


//  worker thread for each CPU processor core

void * voodoo2_names::wthread(void *arg)
{
   using namespace voodoo2_names;

   int         index = *((int *) (arg));
   int         px, py, rx, ii, jj;
   float       *pix1, *pix3;
   float       fold, fnew, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   float       FF, weight, bright, debandx;

   for (py = index; py < Ihh; py += NWT)                                         //  loop all image pixels
   for (px = 0; px < Iww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright = 0.25 * red1 + 0.65 * green1 + 0.10 * blue1;                       //  input pixel brightness 0-255.9
      bright *= 1000.0 / 256.0;                                                  //  0-999

      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      FF = 0;
      for (ii = 0; ii < 9; ii++) {                                               //  loop 9 nearest zones
         weight = Zw[rx+ii];
         if (weight > 0) {                                                       //  0-100 = 1.0
            jj = Zn[rx+ii];
            FF += 0.01 * weight * Zff[1000*jj+int(bright)];                      //  sum weight * flatten factor
         }
      }

      red3 = FF * red1;                                                          //  fully flattened brightness
      green3 = FF * green1;
      blue3 = FF * blue1;

      debandx = (1000.0 - bright) / 1000.0;                                      //  bright = 0 to 1000  >>  debandx = 1 to 0
      debandx += (1.0 - debandx) * (1.0 - 0.01 * deband);                        //  debandx = 1 to 1  >>  1 to 0

      fnew = 0.01 * flatten;                                                     //  how much to flatten, 0 to 1
      fnew = debandx * fnew;                                                     //  attenuate flatten of brighter pixels
      fold = 1.0 - fnew;                                                         //  how much to retain, 1 to 0

      red3 = fnew * red3 + fold * red1;                                          //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      cmax = red3;                                                               //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************/

//  Retouch Combo
//  Adjust all aspects of brightness, contrast, color.
//  Brightness curves are used, overall and per RGB color.
//  Color saturation and white balance adjustments are also available.

namespace combo_names
{
   editfunc    EFcombo;
   float       brightness;                                                       //  brightness input, -1 ... +1
   float       contrast;                                                         //  contrast input, -1 ... +1
   float       wbalR, wbalG, wbalB;                                              //  white balance, white basis standard
   float       blackR, blackG, blackB;                                           //  RGB black point if not 0/0/0
   int         whitebal, blacklev;                                               //  white balance or black level active
   int         Fblack;                                                           //  flag, black level was set
   int         wbtemp;                                                           //  illumination temp. input, 1K ... 10K
   float       tempR, tempG, tempB;                                              //  RGB of illumination temp.
   float       combR, combG, combB;                                              //  combined RGB factors
   float       satlevel;                                                         //  saturation input, -1 ... +1 = saturated
   float       areaemph;                                                         //  emphasis input, -1 ... +1 = bright areas
   float       emphcurve[256];                                                   //  emphasis per brightness level, 0 ... 1
   int         Fapply = 0;                                                       //  flag, apply dialog controls to image
   int         Fdist = 0;                                                        //  flag, show brightness distribution
   float       amplify = 1.0;                                                    //  curve amplifier, 0 ... 2
   GtkWidget   *drawwin_dist, *drawwin_scale;                                    //  brightness distribution graph widgets
   int         RGBW[4] = { 0, 0, 0, 0 };                                         //     "  colors: red/green/blue/white (all)
}

void  blackbodyRGB(int K, float &R, float &G, float &B);


//  menu function

void m_retouch_combo(GtkWidget *, cchar *menu)
{
   using namespace combo_names;

   int    combo_dialog_event(zdialog* zd, cchar *event);
   void   combo_mousefunc();
   void   combo_curvedit(int spc);
   void * combo_thread(void *);

   F1_help_topic = "retouch_combo";

   EFcombo.menuname = menu;
   EFcombo.menufunc = m_retouch_combo;
   EFcombo.funcname = "retouch_combo";
   EFcombo.FprevReq = 1;                                                         //  use preview
   EFcombo.Farea = 2;                                                            //  select area usable
   EFcombo.Frestart = 1;                                                         //  restart allowed
   EFcombo.FusePL = 1;                                                           //  use with paint/lever edits OK
   EFcombo.Fscript = 1;                                                          //  scripting supported
   EFcombo.threadfunc = combo_thread;
   EFcombo.mousefunc = combo_mousefunc;                                          //  18.01
   if (! edit_setup(EFcombo)) return;                                            //  setup edit

/***
       _____________________________________________________
      |                    Retouch Combo                    |
      |  _________________________________________________  |
      | |                                                 | |                    //  5 curves are maintained:
      | |                                                 | |                    //  curve 0: current display curve
      | |                                                 | |                    //        1: curve for all colors
      | |                                                 | |                    //  curve 0: current display curve
      | |                                                 | |                    //        1: curve for all colors
      | |               curve edit area                   | |                    //        2,3,4: red, green, blue
      | |                                                 | |
      | |                                                 | |
      | |                                                 | |                    //  curve 0: current display curve
      | |                                                 | |                    //        1: curve for all colors
      | |_________________________________________________| |
      | |_________________________________________________| |                    //  brightness scale: black to white stripe
      |   (o) all  (o) red  (o) green  (o) blue             |                    //  select curve to display
      |                                                     |
      |  Amplifier  ================[]=============  Max.   |                    //  curve amplifier
      |  Brightness ================[]=============  High   |                    //  brightness
      |   Contrast  ================[]=============  High   |                    //  contrast
      |   Low Color ====================[]=========  High   |                    //  color saturation
      |    Warmer   ====================[]========= Cooler  |                    //  color temperature
      |  Dark Areas ==========[]=================== Bright  |                    //  color emphasis
      |                                                     |
      |  [x] Show Brightness Distribution                   |
      |  [x] Click for white balance   [x] or black level   |                    //  click gray/white spot or dark spot
      |                                                     |
      |  Settings File [Load] [Save]                        |
      |                                                     |
      |                      [Reset] [Prev] [Done] [Cancel] |
      |_____________________________________________________|

***/

   zdialog *zd = zdialog_new(ZTX("Retouch Combo"),Mwin,Breset,Bprev,Bdone,Bcancel,null);
   EFcombo.zd = zd;

   zdialog_add_widget(zd,"frame","frameH","dialog",0,"expand");                  //  edit-curve and distribution graph
   zdialog_add_widget(zd,"frame","frameB","dialog");                             //  black to white brightness scale

   zdialog_add_widget(zd,"hbox","hbrgb","dialog");
   zdialog_add_widget(zd,"radio","all","hbrgb",Ball,"space=5");
   zdialog_add_widget(zd,"radio","red","hbrgb",Bred,"space=3");
   zdialog_add_widget(zd,"radio","green","hbrgb",Bgreen,"space=3");
   zdialog_add_widget(zd,"radio","blue","hbrgb",Bblue,"space=3");

   zdialog_add_widget(zd,"hbox","hbcolor","dialog");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=1");
   zdialog_add_widget(zd,"vbox","vbcolor1","hbcolor",0,"homog");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=1");
   zdialog_add_widget(zd,"vbox","vbcolor2","hbcolor",0,"homog|expand");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=1");
   zdialog_add_widget(zd,"vbox","vbcolor3","hbcolor",0,"homog");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=1");

   zdialog_add_widget(zd,"label","labamp","vbcolor1",ZTX("Amplifier"));
   zdialog_add_widget(zd,"label","labrite","vbcolor1",ZTX("Brightness"));
   zdialog_add_widget(zd,"label","labcont","vbcolor1",ZTX("Contrast"));
   zdialog_add_widget(zd,"label","labsat1","vbcolor1",ZTX("Low Color"));
   zdialog_add_widget(zd,"label","labtemp1","vbcolor1",ZTX("Warmer"));
   zdialog_add_widget(zd,"label","labarea1","vbcolor1",ZTX("Dark Areas"));

   zdialog_add_widget(zd,"hscale","amplify","vbcolor2","0.0|2.0|0.01|1.0");
   zdialog_add_widget(zd,"hscale","brightness","vbcolor2","-1.0|1.0|0.01|0");
   zdialog_add_widget(zd,"hscale","contrast","vbcolor2","-1.0|1.0|0.01|0");
   zdialog_add_widget(zd,"hscale","satlevel","vbcolor2","-1.0|1.0|0.01|0");
   zdialog_add_widget(zd,"hscale","wbtemp","vbcolor2","2000|8000|1|5000");
   zdialog_add_widget(zd,"hscale","areaemph","vbcolor2","0|1.0|0.01|0.5");

   zdialog_add_widget(zd,"label","labrite2","vbcolor3",ZTX("Max."));
   zdialog_add_widget(zd,"label","labrite2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labcont2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labsat2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labtemp2","vbcolor3",ZTX("Cooler"));
   zdialog_add_widget(zd,"label","labarea2","vbcolor3",ZTX("Bright"));

   zdialog_add_widget(zd,"hbox","hbdist","dialog");
   zdialog_add_widget(zd,"check","dist","hbdist",ZTX("Brightness Distribution"),"space=3");

   zdialog_add_widget(zd,"hbox","hbclick","dialog");
   zdialog_add_widget(zd,"check","whitebal","hbclick",ZTX("Click for white balance"),"space=3");
   zdialog_add_widget(zd,"check","blacklev","hbclick",ZTX("Click for black level"),"space=3");

   zdialog_add_widget(zd,"hbox","hbset","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labset","hbset",ZTX("Settings File"),"space=5");
   zdialog_add_widget(zd,"button","load","hbset",Bload,"space=3");
   zdialog_add_widget(zd,"button","save","hbset",Bsave,"space=3");

   zdialog_add_ttip(zd,Bprev,ZTX("recall previous settings used"));

   zdialog_rescale(zd,"brightness",-1,0,1);
   zdialog_rescale(zd,"contrast",-1,0,1);
   zdialog_rescale(zd,"satlevel",-1,0,1);
   zdialog_rescale(zd,"wbtemp",2000,5000,8000);

   GtkWidget *frameH = zdialog_widget(zd,"frameH");                              //  setup edit curves
   spldat *sd = splcurve_init(frameH,combo_curvedit);
   EFcombo.curves = sd;

   sd->Nscale = 1;                                                               //  horizontal line, neutral value     17.08
   sd->xscale[0][0] = 0.00;
   sd->yscale[0][0] = 0.50;
   sd->xscale[1][0] = 1.00;
   sd->yscale[1][0] = 0.50;

   for (int ii = 0; ii < 4; ii++)                                                //  loop curves 0-3
   {
      sd->nap[ii] = 3;                                                           //  initial curves are neutral
      sd->vert[ii] = 0;
      sd->fact[ii] = 0;
      sd->apx[ii][0] = 0.01;                                                     //  horizontal lines                   17.08
      sd->apy[ii][0] = 0.50;
      sd->apx[ii][1] = 0.50;
      sd->apy[ii][1] = 0.50;                                                     //  curve 0 = overall brightness
      sd->apx[ii][2] = 0.99;
      sd->apy[ii][2] = 0.50;                                                     //  curve 1/2/3 = R/G/B adjustment
      splcurve_generate(sd,ii);
   }

   sd->Nspc = 4;                                                                 //  4 curves
   sd->fact[0] = 1;                                                              //  curve 0 active 
   zdialog_stuff(zd,"all",1);                                                    //  stuff default selection, all
   
   drawwin_dist = sd->drawarea;                                                  //  setup brightness distr. drawing area
   G_SIGNAL(drawwin_dist,"draw",brdist_drawgraph,RGBW);

   GtkWidget *frameB = zdialog_widget(zd,"frameB");                              //  setup brightness scale drawing area
   drawwin_scale = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(frameB),drawwin_scale);
   gtk_widget_set_size_request(drawwin_scale,300,12);
   G_SIGNAL(drawwin_scale,"draw",brdist_drawscale,0);

   brightness = 0;                                                               //  neutral brightness
   contrast = 0;                                                                 //  neutral contrast
   amplify = 1.0;                                                                //  neutral amplifier                  17.04.1

   whitebal = blacklev = 0;
   wbalR = wbalG = wbalB = 1.0;                                                  //  neutral white balance
   wbtemp = 5000;                                                                //  neutral illumination temp.
   blackbodyRGB(wbtemp,tempR,tempG,tempB);                                       //  all are 1.0
   combR = combG = combB = 1.0;                                                  //  neutral RGB factors
   blackR = blackG = blackB = 0.0;                                               //  zero black point                   17.08
   Fblack = 0;
   satlevel = 0;                                                                 //  neutral saturation
   Fapply = Fdist = 0;                                                           //  17.04.2

   areaemph = 0.5;                                                               //  even dark/bright area emphasis
   for (int ii = 0; ii < 256; ii++)
      emphcurve[ii] = 1;

   zdialog_stuff(zd,"whitebal",0);                                               //  reset mouse click status           18.01
   zdialog_stuff(zd,"blacklev",0);

   zdialog_resize(zd,350,450);
   zdialog_run(zd,combo_dialog_event,"save");                                    //  run dialog - parallel

   return;
}


//  dialog event and completion callback function

int combo_dialog_event(zdialog *zd, cchar *event)
{
   using namespace combo_names;

   void  combo_mousefunc();

   spldat      *sd = EFcombo.curves;
   float       c1, c2, xval, yval;
   float       bright0, dbrite, cont0, dcont;
   float       dx, dy;
   int         ii;
   
   Fapply = 0;
   
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 4;
   if (strmatch(event,"apply")) Fapply = 1;                                      //  from script

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();                                                           //  get full size image
      signal_thread();
      return 1;
   }

   if (zd->zstat == 1)                                                           //  [reset]
   {
      zd->zstat = 0;                                                             //  keep dialog active

      zdialog_stuff(zd,"amplify",1);                                             //  neutral amplifier
      zdialog_stuff(zd,"brightness",0);                                          //  neutral brightness
      zdialog_stuff(zd,"contrast",0);                                            //  neutral contrast
      brightness = contrast = 0;
      amplify = 1.0;                                                             //  17.04.1

      wbalR = wbalG = wbalB = 1.0;                                               //  neutral white balance
      wbtemp = 5000;                                                             //  neutral illumination temp.
      zdialog_stuff(zd,"wbtemp",wbtemp);
      blackbodyRGB(wbtemp,tempR,tempG,tempB);                                    //  all are 1.0
      combR = combG = combB = 1.0;                                               //  neutral RGB factors
      blackR = blackG = blackB = 0.0;                                            //  zero black point                   17.08
      Fblack = 0;

      satlevel = 0;                                                              //  neutral saturation
      zdialog_stuff(zd,"satlevel",0);

      areaemph = 0.5;                                                            //  even area emphasis
      zdialog_stuff(zd,"areaemph",0.5);
      for (int ii = 0; ii < 256; ii++)                                           //  flat curve
         emphcurve[ii] = 1;

      for (int ii = 0; ii < 4; ii++)                                             //  loop brightness curves 0-3
      {
         sd->nap[ii] = 3;                                                        //  all curves are neutral
         sd->vert[ii] = 0;
         sd->apx[ii][0] = 0.01;                                                  //  17.08
         sd->apy[ii][0] = 0.50;
         sd->apx[ii][1] = 0.50;
         sd->apy[ii][1] = 0.50;
         sd->apx[ii][2] = 0.99;
         sd->apy[ii][2] = 0.50;
         splcurve_generate(sd,ii);
         sd->fact[ii] = 0;
      }

      sd->fact[0] = 1;
      zdialog_stuff(zd,"all",1);
      zdialog_stuff(zd,"red",0); 
      zdialog_stuff(zd,"green",0);
      zdialog_stuff(zd,"blue",0);
      zdialog_stuff(zd,"whitebal",0);                                            //  18.01
      zdialog_stuff(zd,"blacklev",0);
      edit_reset();                                                              //  restore initial image
      event = "dograph";                                                         //  trigger graph update
   }

   if (zd->zstat == 2)                                                           //  [prev] restore previous settings
   {
      zd->zstat = 0;                                                             //  keep dialog active
      edit_load_prev_widgets(&EFcombo);
      Fapply = 1;                                                                //  trigger apply event
   }

   if (zd->zstat)                                                                //  [done] or [cancel]
   {
      freeMouse();
      if (zd->zstat == 3 && CEF->Fmods) {                                        //  [done]
         edit_fullsize();                                                        //  get full size image
         signal_thread();
         edit_save_last_widgets(&EFcombo);
         edit_done(0);
      }
      else edit_cancel(0);                                                       //  [cancel] or [x]
      return 1;
   }

   if (strmatch(event,"load"))                                                   //  load all settings from a file
      edit_load_widgets(&EFcombo);

   if (strmatch(event,"save"))                                                   //  save all settings to a file
      edit_save_widgets(&EFcombo);
      
   if (strmatch(event,"whitebal")) {                                             //  white balance checkbox             18.01
      zdialog_fetch(zd,"whitebal",whitebal);
      if (whitebal) {
         zdialog_stuff(zd,"blacklev",0);
         blacklev = 0;
      }
   }

   if (strmatch(event,"blacklev")) {                                             //  black level checkbox               18.01
      zdialog_fetch(zd,"blacklev",blacklev);
      if (blacklev) {
         zdialog_stuff(zd,"whitebal",0);
         whitebal = 0;
      }
   }

   if (strstr("whitebal blacklev",event)) {                                      //  white balance or black level       18.01
      if (whitebal || blacklev)                                                  //  one is on:
         takeMouse(combo_mousefunc,dragcursor);                                  //    connect mouse function
      else freeMouse();                                                          //  both off: free mouse
      return 1;
   }

   if (strmatch(event,"dist")) {                                                 //  distribution checkbox
      zdialog_fetch(zd,"dist",Fdist);                                            //  distribution graph on/off
      event = "dograph";                                                         //  trigger graph update
   }

   if (strstr("all red green blue",event))                                       //  new choice of curve
   {
      zdialog_fetch(zd,event,ii);
      if (! ii) return 0;                                                        //  button OFF event, wait for ON event

      for (ii = 0; ii < 4; ii++)
         sd->fact[ii] = 0;
      ii = strmatchV(event,"all","red","green","blue",null);
      ii = ii-1;                                                                 //  new active curve: 0, 1, 2, 3
      sd->fact[ii] = 1;

      splcurve_generate(sd,ii);                                                  //  regenerate curve
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve
   }

   if (strmatch(event,"brightness"))                                             //  brightness slider, 0 ... 1
   {
      bright0 = brightness;
      zdialog_fetch(zd,"brightness",brightness);
      dbrite = brightness - bright0;                                             //  change in brightness, + -

      zdialog_stuff(zd,"all",1);                                                 //  active curve is "all"
      sd->fact[0] = 1;
      for (ii = 1; ii < 4; ii++)
         sd->fact[ii] = 0;

      for (ii = 0; ii < sd->nap[0]; ii++)                                        //  update curve 0 nodes from slider
      {
         dy = sd->apy[0][ii] + 0.5 * dbrite;                                     //  increment brightness
         if (dy < 0) dy = 0;
         if (dy > 1) dy = 1;
         sd->apy[0][ii] = dy;
      }

      splcurve_generate(sd,0);                                                   //  regenerate curve 0
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve 0
   }

   if (strmatch(event,"contrast"))                                               //  contrast slider, 0 ... 1
   {
      cont0 = contrast;
      zdialog_fetch(zd,"contrast",contrast);
      dcont = contrast - cont0;                                                  //  change in contrast, + -

      zdialog_stuff(zd,"all",1);                                                 //  active curve is "all"
      sd->fact[0] = 1;
      for (ii = 1; ii < 4; ii++)
         sd->fact[ii] = 0;
      
      for (ii = 0; ii < sd->nap[0]; ii++)                                        //  update curve 0 nodes from slider
      {
         dx = sd->apx[0][ii];                                                    //  0 ... 0.5 ... 1
         if (dx < 0.0 || dx >= 1.0) continue;
         dy = dcont * (dx - 0.5);                                                //  -0.5 ... 0 ... +0.5 * dcont
         dy += sd->apy[0][ii];
         if (dy < 0) dy = 0;
         if (dy > 1) dy = 1;
         sd->apy[0][ii] = dy;
      }

      splcurve_generate(sd,0);                                                   //  regenerate curve 0
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve 0
   }

   if (Fdist) {                                                                  //  distribution enabled
      zdialog_fetch(zd,"red",RGBW[0]);                                           //  get graph color choice
      zdialog_fetch(zd,"green",RGBW[1]);
      zdialog_fetch(zd,"blue",RGBW[2]);
      zdialog_fetch(zd,"all",RGBW[3]);
      if (RGBW[3]) RGBW[0] = RGBW[1] = RGBW[2] = 1;
      RGBW[3] = 0;
   }
   else RGBW[0] = RGBW[1] = RGBW[2] = RGBW[3] = 0;

   if (strmatch(event,"dograph"))                                                //  thread done or new color choice
      gtk_widget_queue_draw(drawwin_dist);                                       //  update distribution graph

   if (strmatch(event,"blendwidth")) Fapply = 1;                                 //  trigger apply event
   if (strstr("amplify brightness contrast",event)) Fapply = 1;
   if (strstr("satlevel areaemph wbtemp",event)) Fapply = 1;
   if (strmatch(event,"load")) Fapply = 1;

   if (! Fapply) return 1;                                                       //  wait for change

   if (Fdist) gtk_widget_queue_draw(drawwin_dist);                               //  update distribution graph          17.04.2

   zdialog_fetch(zd,"amplify",amplify);                                          //  get curve amplifier setting
   zdialog_fetch(zd,"brightness",brightness);                                    //  get brightness setting
   zdialog_fetch(zd,"contrast",contrast);                                        //  get contrast setting
   zdialog_fetch(zd,"satlevel",satlevel);                                        //  get saturation setting

   zdialog_fetch(zd,"wbtemp",wbtemp);                                            //  get illumination temp. setting
   blackbodyRGB(wbtemp,tempR,tempG,tempB);                                       //  generate new temp. adjustments

   zdialog_fetch(zd,"areaemph",areaemph);                                        //  get dark/bright area emphasis

   if (areaemph < 0.5) {                                                         //  areaemph = 0 ... 0.5
      c1 = 1.2;                                                                  //  c1 = 1.2
      c2 = 2 * areaemph;                                                         //  c2 = 0 ... 1
   }
   else {                                                                        //  areaemph = 0.5 ... 1
      c1 = 2 * (1 - areaemph);                                                   //  c1 = 1 ... 0
      c2 = 1.2;                                                                  //  c2 = 1.2
   }

   for (ii = 0; ii < 256; ii++) {
      xval = ii / 256.0;                                                         //  xval = 0 ... 1
      yval = c1 + xval * (c2 - c1);                                              //  yval = c1 ... c2
      if (yval > 1.0) yval = 1.0;                                                //  limit to 1.0
      emphcurve[ii] = yval;
   }

   signal_thread();                                                              //  update the image
   return 1;
}


//  this function is called when a curve is edited

void combo_curvedit(int spc)
{
   using namespace combo_names;
   signal_thread();
   return;
}


//  get nominal white color or black point from mouse click position

void combo_mousefunc()                                                           //  mouse function
{
   using namespace combo_names;

   int         px, py, dx, dy;
   float       red, green, blue, rgbmean;
   float       *ppix;
   char        mousetext[60];

   if (! LMclick) return;
   LMclick = 0;

   px = Mxclick;                                                                 //  mouse click position
   py = Myclick;

   if (px < 2) px = 2;                                                           //  pull back from edge
   if (px > E3pxm->ww-3) px = E3pxm->ww-3;
   if (py < 2) py = 2;
   if (py > E3pxm->hh-3) py = E3pxm->hh-3;

   red = green = blue = 0;

   for (dy = -1; dy <= 1; dy++)                                                  //  3x3 block around mouse position
   for (dx = -1; dx <= 1; dx++)
   {
      ppix = PXMpix(E1pxm,px+dx,py+dy);                                          //  input image
      red += ppix[0];
      green += ppix[1];
      blue += ppix[2];
   }

   red = red / 9.0;                                                              //  mean RGB levels
   green = green / 9.0;
   blue = blue / 9.0;

   snprintf(mousetext,60,"3x3 pixels RGB: %.0f %.0f %.0f \n",red,green,blue);
   poptext_mouse(Mwin,mousetext,10,10,0,3);

   if (whitebal) {                                                               //  click pixel is new gray/white      18.01
      rgbmean = (red + green + blue) / 3.0;
      wbalR = rgbmean / red;                                                     //  = 1.0 if all RGB are equal
      wbalG = rgbmean / green;                                                   //  <1/>1 if RGB should be less/more
      wbalB = rgbmean / blue;
      signal_thread();                                                           //  trigger image update
   }

   if (blacklev) {                                                               //  click pixel is new black point     18.01
      blackR = red;
      blackG = green;
      blackB = blue;
      Fblack = 1;
      signal_thread();                                                           //  update image
   }
   
   return;
}


//  combo thread function

void * combo_thread(void *arg)
{
   using namespace combo_names;

   void * combo_wthread(void *);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      combR = wbalR * tempR / 256.0;                                             //  gray standard based on clicked pixel
      combG = wbalG * tempG / 256.0;                                             //    colors and illumination temp.
      combB = wbalB * tempB / 256.0;                                             //      <1/>1 if RGB should be less/more

      for (int ii = 0; ii < NWT; ii++)                                           //  start worker threads
         start_wthread(combo_wthread,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      CEF->Fmods++;                                                              //  image3 modified
      CEF->Fsaved = 0;

      Fpaint2();                                                                 //  update window

      if (CEF->thread_status > 1) continue;                                      //  continue working   Mint Lockup

      if (CEF->zd) {
         zd_thread = CEF->zd;                                                    //  signal dialog to update graph
         zd_thread_event = "dograph";
      }
      else zd_thread = 0;
   }

   return 0;                                                                     //  not executed, stop g++ warning
}


void * combo_wthread(void *arg)                                                  //  worker thread function
{
   using namespace combo_names;

   int         index = *((int *) arg);
   int         ii, dist = 0, px, py;
   float       *pix1, *pix3;
   float       red1, green1, blue1, maxrgb;
   float       red3, green3, blue3;
   float       Rbp, Gbp, Bbp;
   float       pixbrite, F1, F2;
   float       coeff = 1000.0 / 256.0;
   float       dold, dnew, ff;
   spldat      *sd = EFcombo.curves;

   Rbp = Gbp = Bbp = 1.0;   
   if (Fblack) {                                                                 //  if RGB black points defined,       17.08
      Rbp = 256.0 / (256.0 - blackR);                                            //    rescale for full brightness
      Gbp = 256.0 / (256.0 - blackG);
      Bbp = 256.0 / (256.0 - blackB);
   }
   
   for (py = index; py < E3pxm->hh; py += NWT)                                   //  loop all pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      red1 = pix1[0];                                                            //  input RGB values
      green1 = pix1[1];
      blue1 = pix1[2];
      
      //  apply RGB black points to input RGB values                             //  17.08
      
      if (Fblack) {
         red1 = red1 - blackR;                                                   //  clip black value at low end 
         red1 = red1 * Rbp;                                                      //  rescale so that high end = 256
         if (red1 < 0) red1 = 0;
         green1 = green1 - blackG;
         green1 = green1 * Gbp;
         if (green1 < 0) green1 = 0;
         blue1 = blue1 - blackB;
         blue1 = blue1 * Bbp;
         if (blue1 < 0) blue1 = 0;
      }

      //  apply white balance and temperature color shift

      red3 = combR * red1;                                                       //  <1/>1 if RGB should be less/more
      green3 = combG * green1;
      blue3 = combB * blue1;

      //  apply saturation color shift

      if (satlevel != 0) {
         pixbrite = 0.333 * (red3 + green3 + blue3);                             //  pixel brightness, 0 to 255.9
         red3 = red3 + satlevel * (red3 - pixbrite);                             //  satlevel is -1 ... +1
         green3 = green3 + satlevel * (green3 - pixbrite);
         blue3 = blue3 + satlevel * (blue3 - pixbrite);
      }

      if (red3 < 0) red3 = 0;                                                    //  stop underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      maxrgb = red3;                                                             //  stop overflow
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      if (maxrgb > 255.9) {
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
      }

      //  apply dark/bright area emphasis curve for color changes

      if (areaemph != 0.5) {
         maxrgb = red1;                                                          //  use original colors
         if (green1 > maxrgb) maxrgb = green1;
         if (blue1 > maxrgb) maxrgb = blue1;
         F1 = emphcurve[int(maxrgb)];                                            //  0 ... 1   neutral is flat curve = 1
         F2 = 1.0 - F1;
         red3 = F1 * red3 + F2 * red1;
         green3 = F1 * green3 + F2 * green1;
         blue3 = F1 * blue3 + F2 * blue1;
      }

      //  apply brightness/contrast curves

      pixbrite = red3;                                                           //  use max. RGB value
      if (green3 > pixbrite) pixbrite = green3;
      if (blue3 > pixbrite) pixbrite = blue3;
      
      ii = coeff * pixbrite;                                                     //  "all" curve index 0-999
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      ff = 1.0 + amplify * (sd->yval[0][ii] - 0.5);
      red3 = ff * red3;
      green3 = ff * green3;
      blue3 = ff * blue3;

      ii = coeff * red3;                                                         //  additional RGB curve adjustments
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      red3 = red3 * (1.0 + sd->yval[1][ii] - 0.5);                               //  17.08

      ii = coeff * green3;
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      green3 = green3 * (1.0 + sd->yval[2][ii] - 0.5); 

      ii = coeff * blue3;
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      blue3 = blue3 * (1.0 + sd->yval[3][ii] - 0.5); 

      if (red3 < 0) red3 = 0;                                                    //  stop underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      maxrgb = red3;                                                             //  stop overflow
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      if (maxrgb > 255.9) {
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
      }

      //  select area edge blending

      if (sa_stat == 3 && dist < sa_blend) {
         dnew = sa_blendfunc(dist);
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************/

//  Return relative RGB illumination values for a light source
//     having a given input temperature of 1000-10000 deg. K
//  5000 K is neutral: all returned factors relative to 1.0

void blackbodyRGB(int K, float &R, float &G, float &B)
{
   float    kk[19] = { 1.0, 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0, 5.5, 6.0, 6.5, 7.0, 7.5, 8.0, 8.5, 9.0, 9.5, 10.0 };
   float    r1[19] = { 255, 255, 255, 255, 255, 255, 255, 255, 254, 250, 242, 231, 220, 211, 204, 197, 192, 188, 184 };
   float    g1[19] = { 060, 148, 193, 216, 232, 242, 249, 252, 254, 254, 251, 245, 239, 233, 228, 224, 220, 217, 215 };
   float    b1[19] = { 000, 010, 056, 112, 158, 192, 219, 241, 253, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255 };

   static int     ftf = 1;
   static float   r2[10000], g2[10000], b2[10000];

   if (ftf) {                                                                    //  initialize
      spline1(19,kk,r1);
      for (int T = 1000; T < 10000; T++)
         r2[T] = spline2(0.001 * T);

      spline1(19,kk,g1);
      for (int T = 1000; T < 10000; T++)
         g2[T] = spline2(0.001 * T);

      spline1(19,kk,b1);
      for (int T = 1000; T < 10000; T++)
         b2[T] = spline2(0.001 * T);

      ftf = 0;
   }

   if (K < 1000 || K > 9999) zappcrash("blackbody bad K: %dK",K);

   R = r2[K];
   G = g2[K];
   B = b2[K];

   return;
}


/********************************************************************************/

//  adjust brightness distribution by flattening and/or expanding range

namespace britedist_names
{
   int      ww, hh;
   float    LC, HC;                                                              //  low, high cutoff levels
   float    LF, MF, HF;                                                          //  low, mid, high flatten parms
   float    LS, MS, HS;                                                          //  low, mid, high stretch parms
   float    BB[1000];                                                            //  adjusted B for input B 0-999

   int    dialog_event(zdialog* zd, cchar *event);
   void   compute_BB();
   void * thread(void *);
   void * wthread(void *);

   editfunc    EFbrightdist;
}


//  menu function

void m_edit_britedist(GtkWidget *, cchar *menu)
{
   using namespace britedist_names;

   cchar  *title = ZTX("Adjust Brightness Distribution");

   F1_help_topic = "edit_brightness";

   EFbrightdist.menuname = menu;
   EFbrightdist.menufunc = m_edit_britedist;
   EFbrightdist.funcname = "brightdist";
   EFbrightdist.FprevReq = 1;                                                    //  preview
   EFbrightdist.Farea = 2;                                                       //  select area usable
   EFbrightdist.Frestart = 1;                                                    //  restart allowed
   EFbrightdist.Fscript = 1;                                                     //  scripting supported 
   EFbrightdist.threadfunc = thread;
   if (! edit_setup(EFbrightdist)) return;                                       //  setup edit

/***
          __________________________________________
         |      Adjust Brightness Distribution      |
         |                                          |
         | Low Cutoff   ==========[]==============  |
         | High Cutoff  ==============[]==========  |
         | Low Flatten  =========[]===============  |
         | Mid Flatten  ============[]============  |
         | High Flatten ==============[]==========  |
         | Low Stretch  ================[]========  |
         | Mid Stretch  =============[]===========  |
         | High Stretch =========[]===============  |
         |                                          |
         |                  [Reset] [Done] [Cancel] |
         |__________________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,Breset,Bdone,Bcancel,null);
   EFbrightdist.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");

   zdialog_add_widget(zd,"label","labLC","vb1",ZTX("Low Cutoff"));
   zdialog_add_widget(zd,"label","labHC","vb1",ZTX("High Cutoff"));
   zdialog_add_widget(zd,"label","labLF","vb1",ZTX("Low Flatten"));
   zdialog_add_widget(zd,"label","labMF","vb1",ZTX("Mid Flatten"));
   zdialog_add_widget(zd,"label","labHF","vb1",ZTX("High Flatten"));
   zdialog_add_widget(zd,"label","labLS","vb1",ZTX("Low Stretch"));
   zdialog_add_widget(zd,"label","labMS","vb1",ZTX("Mid Stretch"));
   zdialog_add_widget(zd,"label","labHS","vb1",ZTX("High Stretch"));

   zdialog_add_widget(zd,"hscale","LC","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","HC","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","LF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","MF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","HF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","LS","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","MS","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale","HS","vb2","0|1.0|0.002|0","expand");

   zdialog_resize(zd,300,0);
   zdialog_run(zd,dialog_event,"save");                                          //  run dialog - parallel

   m_show_brdist(0,0);                                                           //  popup brightness histogram
   
   LC = HC = LF = MF = HF = LS = MS = HS = 0.0;
   compute_BB();
   
   return;
}


//  dialog event and completion function

int britedist_names::dialog_event(zdialog *zd, cchar *event)
{
   using namespace britedist_names;
   
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();                                                           //  get full size image
      compute_BB();
      signal_thread();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         LC = HC = LF = MF = HF = LS = MS = HS = 0.0;
         zdialog_stuff(zd,"LC",LC);
         zdialog_stuff(zd,"HC",HC);
         zdialog_stuff(zd,"LF",LF);
         zdialog_stuff(zd,"MF",MF);
         zdialog_stuff(zd,"HF",HF);
         zdialog_stuff(zd,"LS",LS);
         zdialog_stuff(zd,"MS",MS);
         zdialog_stuff(zd,"HS",HS);
         compute_BB();
         signal_thread();
      }
      else if (zd->zstat == 2 && CEF->Fmods) {                                   //  done
         edit_fullsize();                                                        //  get full size image
         compute_BB();
         signal_thread();
         m_show_brdist(0,"kill");                                                //  kill distribution graph
         edit_done(0);                                                           //  commit edit
         return 1;
      }
      else {
         edit_cancel(0);                                                         //  cancel - discard edit
         m_show_brdist(0,"kill");                                                //  kill distribution graph
         return 1;
      }
   }

   if (strmatch(event,"blendwidth"))                                             //  select area blendwidth change
      signal_thread();
   
   if (strstr("LC HC LF MF HF LS MS HS apply",event))
   {
      zdialog_fetch(zd,"LC",LC);
      zdialog_fetch(zd,"HC",HC);
      zdialog_fetch(zd,"LF",LF);
      zdialog_fetch(zd,"MF",MF);
      zdialog_fetch(zd,"HF",HF);
      zdialog_fetch(zd,"LS",LS);
      zdialog_fetch(zd,"MS",MS);
      zdialog_fetch(zd,"HS",HS);
      compute_BB();
      signal_thread();
   }

   wait_thread_idle();                                                           //  no overlap window update and threads
   Fpaintnow();
   return 1;
}


//  compute flattened brightness levels for preview size or full size image
//  FB[B] = flattened brightness level for brightness B, scaled 0-1000

void britedist_names::compute_BB()
{
   using namespace britedist_names;

   int      ii, npix, py, px, iB;
   float    *pix1;
   float    B, LC2, HC2;
   float    FB[1000];
   float    LF2, MF2, HF2, LS2, MS2, HS2;
   float    LFB, MFB, HFB, LSB, MSB, HSB;
   float    LFW, MFW, HFW, LSW, MSW, HSW, TWB;

   ww = E1pxm->ww;
   hh = E1pxm->hh;

   for (ii = 0; ii < 1000; ii++)                                                 //  clear brightness distribution data
      FB[ii] = 0;

   if (sa_stat == 3)                                                             //  process selected area
   {
      for (ii = npix = 0; ii < ww * hh; ii++)
      {
         if (! sa_pixmap[ii]) continue;
         py = ii / ww;
         px = ii - py * ww;
         pix1 = PXMpix(E1pxm,px,py);
         B = 1000.0 * (pix1[0] + pix1[1] + pix1[2]) / 768.0;
         FB[int(B)]++;
         npix++;
      }

      for (ii = 1; ii < 1000; ii++)                                              //  cumulative brightness distribution
         FB[ii] += FB[ii-1];                                                     //   0 ... npix

      for (ii = 0; ii < 1000; ii++)
         FB[ii] = FB[ii] / npix * 999.0;                                         //  flattened brightness level
   }

   else                                                                          //  process whole image
   {
      for (py = 0; py < hh; py++)                                                //  compute brightness distribution
      for (px = 0; px < ww; px++)
      {
         pix1 = PXMpix(E1pxm,px,py);
         B = 1000.0 * (pix1[0] + pix1[1] + pix1[2]) / 768.0;
         FB[int(B)]++;
      }

      for (ii = 1; ii < 1000; ii++)                                              //  cumulative brightness distribution
         FB[ii] += FB[ii-1];                                                     //   0 ... (ww * hh)
   
      for (ii = 0; ii < 1000; ii++)
         FB[ii] = FB[ii] / (ww * hh) * 999.0;                                    //  flattened brightness level
   }

   LC2 = 500 * LC;                                                               //  low cutoff, 0 ... 500
   HC2 = 1000 - 500 * HC;                                                        //  high cutoff, 1000 ... 500

   for (iB = 0; iB < 1000; iB++)                                                 //  loop brightness 0 - 1000   
   {
      B = iB;

      if (LC2 > 0 || HC2 < 1000) {                                               //  stretch to cutoff limits
         if (B < LC2) B = 0;
         else if (B > HC2) B = 999;
         else B = 1000.0 * (B - LC2) / (HC2 - LC2);
      }
      
      LF2 = LF * (1000 - B) / 1000;                                              //  low flatten  LF ... 0
      LF2 = LF2 * LF2;
      LFB = LF2 * FB[iB] + (1.0 - LF2) * B;
      
      LS2 = LS * (1000 - B) / 1000;                                              //  low stretch  LS ... 0
      LS2 = LS2 * LS2;
      LSB = B * (1 - LS2);
      
      MF2 = MF * (500 - fabsf(B - 500)) / 500;                                   //  mid flatten  0 ... MF ... 0
      MF2 = sqrtf(MF2);
      MFB = MF2 * FB[iB] + (1.0 - MF2) * B;

      MS2 = MS * (B - 500) / 500;                                                //  mid stretch  -MS ... 0 ... MS
      MSB = B * (1 + 0.5 * MS2);
      
      HF2 = HF * B / 1000;                                                       //  high flatten  0 ... HF
      HF2 = HF2 * HF2;
      HFB = HF2 * FB[iB] + (1.0 - HF2) * B;

      HS2 = HS * B / 1000;                                                       //  high stretch  0 ... HS
      HS2 = HS2 * HS2;
      HSB = B * (1 + HS2);
      
      LFW = fabsf(B - LFB) / (B + 1);                                            //  weight of each component
      LSW = fabsf(B - LSB) / (B + 1);
      MFW = fabsf(B - MFB) / (B + 1);
      MSW = fabsf(B - MSB) / (B + 1);
      HFW = fabsf(B - HFB) / (B + 1);
      HSW = fabsf(B - HSB) / (B + 1);
      
      TWB = LFW + LSW + MFW + MSW + HFW + HSW;                                   //  add weighted components
      if (TWB == 0) BB[iB] = B;
      else BB[iB] = (LFW * LFB + LSW * LSB
                   + MFW * MFB + MSW * MSB 
                   + HFW * HFB + HSW * HSB) / TWB;
   }
   
   return;
}


//  adjust brightness distribution thread function

void * britedist_names::thread(void *)
{
   using namespace britedist_names;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      for (int ii = 0; ii < NWT; ii++)                                           //  start worker thread per processor core
         start_wthread(wthread,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      CEF->Fmods++;                                                              //  image modified, not saved
      CEF->Fsaved = 0;
   }

   return 0;                                                                     //  not executed, stop g++ warning
}


//  worker thread for each CPU processor core

void * britedist_names::wthread(void *arg)
{
   using namespace britedist_names;

   int         index = *((int *) (arg));
   int         iB, px, py, ii, dist = 0;
   float       B, *pix1, *pix3;
   float       dold, dnew, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   
   for (py = index; py < E1pxm->hh; py += NWT)                                   //  flatten brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      
      B = (pix1[0] + pix1[1] + pix1[2]) / 768.0 * 1000.0;                        //  pixel brightness scaled 0-1000
      iB = int(B);
      if (B > 0) B = BB[iB] / B;
      
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = B * red1;
      green3 = B * green1;
      blue3 = B * blue1;

      if (sa_stat == 3 && dist < sa_blend) {                                     //  select area is active,
         dnew = sa_blendfunc(dist);                                              //    blend edge
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                               //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************

   Magnify Gradients function
   enhance detail by magnifying brightness gradients

   methodology:
   get brightness gradients for each pixel in 4 directions: SE SW NE NW
   amplify gradients using the edit curve: new gradient = F(old gradient)
   integrate 4 new brightness surfaces from the amplified gradients:
     - pixel brightness = prior pixel brightness + amplified gradient
     - the Amplify control varies amplification from zero to maximum
   new pixel brightness = average from 4 calculated brightness surfaces

*********************************************************************************/

namespace gradients_names
{
   float       *brmap1, *brmap2;
   float       *brmap4[4];
   int         contrast99;
   float       amplify;
   int         ww, hh;
   editfunc    EFgradients;

   void   gradients_initz(zdialog *zd);
   int    gradients_dialog_event(zdialog *zd, cchar *event);
   void   gradients_curvedit(int);
   void * gradients_thread(void *);
   void * gradients_wthread1(void *arg);
   void * gradients_wthread2(void *arg);
   void * gradients_wthread3(void *arg);

}

//  menu function

void m_gradients(GtkWidget *, cchar *menu)
{
   using namespace gradients_names;

   F1_help_topic = "gradients";

   cchar    *title = ZTX("Magnify Gradients");
   
   EFgradients.menuname = menu;
   EFgradients.menufunc = m_gradients;
   EFgradients.funcname = "gradients";
   EFgradients.Farea = 2;                                                        //  select area usable
   EFgradients.Frestart = 1;                                                     //  restart allowed
   EFgradients.FusePL = 1;                                                       //  use with paint/lever edits OK
   EFgradients.Fscript = 1;                                                      //  scripting supported
   EFgradients.threadfunc = gradients_thread;
   if (! edit_setup(EFgradients)) return;                                        //  setup: no preview, select area OK

/***
            _____________________________
           |                             |
           |                             |
           |    curve drawing area       |
           |                             |
           |_____________________________|
            low       contrast       high

            Amplify ========[]=========

            Curve File: [ Open ] [ Save ]

***/

   zdialog *zd = zdialog_new(title,Mwin,Bdone,Bcancel,null);
   EFgradients.zd = zd;

   zdialog_add_widget(zd,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcL","hb1",ZTX("low"),"space=4");
   zdialog_add_widget(zd,"label","labcM","hb1",Bcontrast,"expand");
   zdialog_add_widget(zd,"label","labcH","hb1",ZTX("high"),"space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcon","hb2",ZTX("Amplify"),"space=5");
   zdialog_add_widget(zd,"hscale","amplify","hb2","0|1|0.005|0","expand");
   zdialog_add_widget(zd,"label","ampval","hb2",0,"space=5");

   zdialog_add_widget(zd,"hbox","hbcf","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labcf","hbcf",Bcurvefile,"space=5");
   zdialog_add_widget(zd,"button","loadcurve","hbcf",Bopen,"space=5");
   zdialog_add_widget(zd,"button","savecurve","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zd,"frame");                                //  set up curve edit
   spldat *sd = splcurve_init(frame,gradients_curvedit);
   EFgradients.curves = sd;

   sd->Nspc = 1;
   sd->fact[0] = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 3;                                                               //  initial curve anchor points
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.10;
   sd->apx[0][1] = 0.40;
   sd->apy[0][1] = 0.30;
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.01;

   splcurve_generate(sd,0);                                                      //  generate curve data

   gradients_initz(zd);                                                           //  initialize
   return;
}


//  initialization for new image file

void gradients_names::gradients_initz(zdialog *zd)
{
   using namespace gradients_names;

   int         ii, cc, px, py;
   float       b1, *pix1;
   int         jj, sum, limit, condist[100];

   ww = E1pxm->ww;                                                               //  image dimensions
   hh = E1pxm->hh;

   cc = ww * hh * sizeof(float);                                                 //  allocate brightness map memory
   brmap1 = (float *) zmalloc(cc);
   brmap2 = (float *) zmalloc(cc);
   for (ii = 0; ii < 4; ii++)
      brmap4[ii] = (float *) zmalloc(cc);

   for (py = 0; py < hh; py++)                                                   //  map initial image brightness
   for (px = 0; px < ww; px++)
   {
      ii = py * ww + px;
      pix1 = PXMpix(E1pxm,px,py);
      b1 = 0.333 * (pix1[0] + pix1[1] + pix1[2]);                                //  pixel brightness 0-255.9
      if (b1 < 1) b1 = 1;
      brmap1[ii] = b1;
   }

   for (ii = 0; ii < 100; ii++)
      condist[ii] = 0;

   for (py = 1; py < hh; py++)                                                   //  map contrast distribution
   for (px = 1; px < ww; px++)
   {
      ii = py * ww + px;
      jj = 0.388 * fabsf(brmap1[ii] - brmap1[ii-1]);                             //  horiz. contrast, ranged 0 - 99
      condist[jj]++;
      jj = 0.388 * fabsf(brmap1[ii] - brmap1[ii-ww]);                            //  vertical
      condist[jj]++;
   }

   sum = 0;
   limit = 0.99 * 2 * (ww-1) * (hh-1);                                           //  find 99th percentile contrast

   for (ii = 0; ii < 100; ii++) {
      sum += condist[ii];
      if (sum > limit) break;
   }

   contrast99 = 255.0 * ii / 100.0;                                              //  0 to 255
   if (contrast99 < 4) contrast99 = 4;                                           //  rescale low-contrast image

   zdialog_resize(zd,300,300);
   zdialog_run(zd,gradients_dialog_event,"save");                                //  run dialog - parallel

   amplify = 0;
   return;
}


//  dialog event and completion callback function

int gradients_names::gradients_dialog_event(zdialog *zd, cchar *event)
{
   using namespace gradients_names;

   spldat   *sd = EFgradients.curves;
   char     text[8];

   if (strmatch(event,"done")) zd->zstat = 1;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  cancel

   if (zd->zstat) {                                                              //  dialog complete
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      zfree(brmap1);                                                             //  free memory
      zfree(brmap2);
      brmap1 = brmap2 = 0;
      for (int ii = 0; ii < 4; ii++) {
         zfree(brmap4[ii]);
         brmap4[ii] = 0;
      }
      return 1;
   }
   
   if (strmatch(event,"apply"))                                                  //  from script
      gradients_dialog_event(zd,"amplify");

   if (strmatch(event,"blendwidth")) signal_thread();

   if (strmatch(event,"amplify")) {                                              //  slider value
      zdialog_fetch(zd,"amplify",amplify);
      snprintf(text,8,"%.2f",amplify);                                           //  numeric feedback
      zdialog_stuff(zd,"ampval",text);
      signal_thread();                                                           //  trigger update thread
   }

   if (strmatch(event,"loadcurve")) {                                            //  load saved curve
      splcurve_load(sd);
      signal_thread();
      return 0;
   }

   if (strmatch(event,"savecurve")) {                                            //  save curve to file
      splcurve_save(sd);
      return 0;
   }

   return 0;
}


//  this function is called when the curve is edited

void gradients_names::gradients_curvedit(int)
{
   signal_thread();
   return;
}


//  thread function

void * gradients_names::gradients_thread(void *)
{
   using namespace gradients_names;

   int      ii;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      for (ii = 0; ii < 4; ii++)                                                 //  start working threads 1 (4)
         start_wthread(gradients_wthread1,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      for (ii = 0; ii < 4; ii++)                                                 //  start working threads 2 (4)
         start_wthread(gradients_wthread2,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      for (ii = 0; ii < NWT; ii++)                                               //  start working threads 3 (NWT)
         start_wthread(gradients_wthread3,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      CEF->Fmods++;
      CEF->Fsaved = 0;
      if (amplify == 0) CEF->Fmods = 0;

      Fpaint2();
   }

   return 0;                                                                     //  not executed, stop g++ warning
}


//  working threads

void * gradients_names::gradients_wthread1(void *arg)
{
   using namespace gradients_names;

   int         ii, kk, bii, pii, dist = 0;
   int         px, px1, px2, pxinc;
   int         py, py1, py2, pyinc;
   float       b1, b4, xval, yval, grad;
   float       amp, con99;
   spldat      *sd = EFgradients.curves;

   con99 = contrast99;                                                           //  99th percentile contrast
   con99 = 1.0 / con99;                                                          //  inverted

   amp = pow(amplify,0.5);                                                       //  get amplification, 0 to 1

   bii = *((int *) arg);                                                         //  thread 0...3

   if (bii == 0) {                                                               //  direction SE
      px1 = 1; px2 = ww; pxinc = 1;
      py1 = 1; py2 = hh; pyinc = 1;
      pii = - 1 - ww;
   }

   else if (bii == 1) {                                                          //  direction SW
      px1 = ww-2; px2 = 0; pxinc = -1;
      py1 = 1; py2 = hh; pyinc = 1;
      pii = + 1 - ww;
   }

   else if (bii == 2) {                                                          //  direction NE
      px1 = 1; px2 = ww; pxinc = 1;
      py1 = hh-2; py2 = 0; pyinc = -1;
      pii = - 1 + ww;
   }

   else {   /* bii == 3 */                                                       //  direction NW
      px1 = ww-2; px2 = 0; pxinc = -1;
      py1 = hh-2; py2 = 0; pyinc = -1;
      pii = + 1 + ww;
   }

   for (ii = 0; ii < ww * hh; ii++)                                              //  initial brightness map
      brmap4[bii][ii] = brmap1[ii];

   for (py = py1; py != py2; py += pyinc)                                        //  loop all image pixels
   for (px = px1; px != px2; px += pxinc)
   {
      ii = py * ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      b1 = brmap1[ii];                                                           //  this pixel brightness
      grad = b1 - brmap1[ii+pii];                                                //   - prior pixel --> gradient

      xval = fabsf(grad) * con99;                                                //  gradient scaled 0 to 1+
      kk = 1000.0 * xval;
      if (kk > 999) kk = 999;
      yval = 1.0 + 5.0 * sd->yval[0][kk];                                        //  amplifier = 1...6
      grad = grad * yval;                                                        //  magnified gradient

      b4 = brmap4[bii][ii+pii] + grad;                                           //  pixel brightness = prior + gradient
      b4 = (1.0 - amp) * b1 + amp * b4;                                          //  apply amplifier: b4 range = b1 --> b4

      brmap4[bii][ii] = b4;                                                      //  new pixel brightness
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


void * gradients_names::gradients_wthread2(void *arg)
{
   using namespace gradients_names;

   int      index, ii, px, py, dist = 0;
   float    b4;

   index = *((int *) arg);                                                       //  thread 0...3

   for (py = index; py < hh; py += 4)                                            //  loop all image pixels
   for (px = 0; px < ww; px++)
   {
      ii = py * ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel is outside area
      }

      b4 = brmap4[0][ii] + brmap4[1][ii]                                         //  new brightness = average of four
         + brmap4[2][ii] + brmap4[3][ii];                                        //    calculated brightness surfaces
      b4 = 0.25 * b4;

      if (b4 < 1) b4 = 1;

      brmap2[ii] = b4;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


void * gradients_names::gradients_wthread3(void *arg)
{
   using namespace gradients_names;

   float       *pix1, *pix3;
   int         index, ii, px, py, dist = 0;
   float       b1, b2, bf, f1, f2;
   float       red1, green1, blue1, red3, green3, blue3;

   index = *((int *) arg);

   for (py = index; py < hh; py += NWT)                                          //  loop all image pixels
   for (px = 0; px < ww; px++)
   {
      ii = py * ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      b1 = brmap1[ii];                                                           //  initial pixel brightness
      b2 = brmap2[ii];                                                           //  calculated new brightness

      bf = b2 / b1;                                                              //  brightness ratio

      red1 = pix1[0];                                                            //  input RGB
      green1 = pix1[1];
      blue1 = pix1[2];

      red3 = bf * red1;                                                          //  output RGB
      if (red3 > 255) red3 = 255;
      green3 = bf * green1;
      if (green3 > 255) green3 = 255;
      blue3 = bf * blue1;
      if (blue3 > 255) blue3 = 255;

      if (sa_stat == 3 && dist < sa_blend) {                                     //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, stop g++ warning
}


/********************************************************************************/

//  flatten brightness distribution based on the distribution of nearby zones

namespace flatten_names
{
   int      Zinit;                                 //  zone initialization needed
   float    flatten;                               //  flatten amount, 0 - 100
   float    deband1, deband2;                      //  deband dark, bright, 0 - 100
   int      Iww, Ihh;                              //  image dimensions
   int      NZ;                                    //  no. of image zones
   int      pNZ;                                   //  prior image zones
   int      Zsize, Zrows, Zcols;                   //  zone parameters
   float    *Br;                                   //  Br[py][px]  pixel brightness
   int      *Zxlo, *Zylo, *Zxhi, *Zyhi;            //  Zxlo[ii] etc.  zone ii pixel range
   int      *Zcen;                                 //  Zcen[ii][2]  zone ii center (py,px)
   int16    *Zn;                                   //  Zn[py][px][9]  9 nearest zones for pixel: 0-200 (-1 = none)
   uint8    *Zw;                                   //  Zw[py][px][9]  zone weights for pixel: 0-100 = 1.0
   float    *Zff;                                  //  Zff[ii][1000]  zone ii flatten factors

   editfunc    EFflatten;

   int    dialog_event(zdialog* zd, cchar *event);
   void   doflatten();
   void   calczones();
   void   initzones();
   void * thread(void *);
   void * wthread(void *);
}


//  menu function

void m_flatten(GtkWidget *, cchar *menu)
{
   using namespace flatten_names;

   cchar  *title = ZTX("Flatten Brightness");

   F1_help_topic = "flatten";

   Iww = Fpxb->ww;
   Ihh = Fpxb->hh;

   if (Iww * Ihh > 227 * MEGA) {
      zmessageACK(Mwin,"image too big");
      edit_cancel(0);
      return;
   }

   EFflatten.menuname = menu;
   EFflatten.menufunc = m_flatten;
   EFflatten.funcname = "flatten";
   EFflatten.FprevReq = 1;                                                       //  use preview image
   EFflatten.Farea = 2;                                                          //  select area usable
   EFflatten.Frestart = 1;                                                       //  restartable
   EFflatten.FusePL = 1;                                                         //  use with paint/lever edits OK
   EFflatten.Fscript = 1;                                                        //  scripting supported
   EFflatten.threadfunc = thread;
   if (! edit_setup(EFflatten)) return;                                          //  setup edit

   Iww = E0pxm->ww;
   Ihh = E0pxm->hh;

   if (Iww * Ihh > 227 * MEGA) {
      zmessageACK(Mwin,"image too big");
      edit_cancel(0);
      return;
   }

/***
          ______________________________________
         |        Flatten Brightness            |
         |                                      |
         | Zones  [ 123 ]   [Apply]          |
         | Flatten  =========[]============ NN  |
         | Deband Dark =========[]========= NN  |
         | Deband Bright ==========[]====== NN  |
         |                                      |
         |                     [Done] [Cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,Bdone,Bcancel,null);
   EFflatten.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labreg","hb1",ZTX("Zones"),"space=5");
   zdialog_add_widget(zd,"zspin","zones","hb1","1|999|1|100");                    //  zone range 1-999                   18.01
   zdialog_add_widget(zd,"button","apply","hb1",Bapply,"space=10");
   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","labflatten","hb2",Bflatten,"space=5");
   zdialog_add_widget(zd,"hscale","flatten","hb2","0|100|1|0","expand");
   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"label","labdeband1","hb3",ZTX("Deband Dark"),"space=5");
   zdialog_add_widget(zd,"hscale","deband1","hb3","0|100|1|0","expand");
   zdialog_add_widget(zd,"hbox","hb4","dialog");
   zdialog_add_widget(zd,"label","labdeband2","hb4",ZTX("Deband Bright"),"space=5");
   zdialog_add_widget(zd,"hscale","deband2","hb4","0|100|1|0","expand");

   zdialog_resize(zd,300,0);
   zdialog_run(zd,dialog_event,"save");                                          //  run dialog - parallel

   NZ = pNZ = 40;                                                                //  default zone count
   calczones();                                                                  //  adjust to fit image
   flatten = deband1 = deband2 = 0;                                              //  dialog controls = neutral
   Zinit = 1;                                                                    //  zone initialization needed
   Br = 0;                                                                       //  no memory allocated
   return;
}


//  dialog event and completion function

int flatten_names::dialog_event(zdialog *zd, cchar *event)
{
   using namespace flatten_names;
   
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  cancel

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      Zinit = 1;
      doflatten();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  done
         edit_fullsize();                                                        //  get full size image
         Zinit = 1;                                                              //  recalculate zones
         doflatten();                                                            //  flatten full image
         edit_done(0);                                                           //  commit edit
      }
      else edit_cancel(0);                                                       //  discard edit

      if (Br) {
         zfree(Br);                                                              //  free memory
         zfree(Zn);
         zfree(Zw);
         zfree(Zxlo);
         zfree(Zylo);
         zfree(Zxhi);
         zfree(Zyhi);
         zfree(Zcen);
         zfree(Zff);
         Br = 0;
      }

      return 1;
   }

   if (strmatch(event,"apply")) {                                                //  [apply]  (also from script)
      dialog_event(zd,"zones");
      dialog_event(zd,"flatten");
      dialog_event(zd,"deband1");
      dialog_event(zd,"deband2");
      doflatten();
   }

   if (strmatch(event,"blendwidth"))                                             //  select area blendwidth change
      doflatten();

   if (strmatch(event,"zones")) {                                                //  get zone count input
      zdialog_fetch(zd,"zones",NZ);
      if (NZ == pNZ) return 1;
      calczones();                                                               //  adjust to fit image
      zdialog_stuff(zd,"zones",NZ);                                              //  update dialog
      Zinit = 1;                                                                 //  zone initialization needed
   }

   if (strmatch(event,"flatten")) {
      zdialog_fetch(zd,"flatten",flatten);                                       //  get slider values
      doflatten();
   }

   if (strmatch(event,"deband1")) {
      zdialog_fetch(zd,"deband1",deband1);
      doflatten();
   }

   if (strmatch(event,"deband2")) {
      zdialog_fetch(zd,"deband2",deband2);
      doflatten();
   }

   return 1;
}


//  perform the flatten calculations and update the image

void flatten_names::doflatten()
{
   if (flatten > 0) {
      signal_thread();
      wait_thread_idle();                                                        //  no overlap window update and threads
      Fpaintnow();
   }
   else edit_reset();
   return;
}


//  recalculate zone count based on what fits the image dimensions
//  done only when the user zone count input changes
//  outputs: NZ, Zrows, Zcols

void flatten_names::calczones()
{
   int      gNZ, dNZ;

   Iww = E1pxm->ww;                                                              //  image dimensions
   Ihh = E1pxm->hh;

   gNZ = NZ;                                                                     //  new zone count goal
   dNZ = NZ - pNZ;                                                               //  direction of change

   while (true)
   {
      Zsize = sqrtf(Iww * Ihh / gNZ);                                            //  approx. zone size
      Zrows = Ihh / Zsize + 0.5;                                                 //  get appropriate rows and cols
      Zcols = Iww / Zsize + 0.5;
      if (Zrows < 1) Zrows = 1;                                                  //  bugfix                             18.01
      if (Zcols < 1) Zcols = 1;
      NZ = Zrows * Zcols;                                                        //  NZ matching rows x cols

      if (dNZ > 0 && NZ <= pNZ) {                                                //  no increase, try again
         if (NZ >= 999) break;
         gNZ++;
         continue;
      }

      if (dNZ < 0 && NZ >= pNZ) {                                                //  no decrease, try again
         if (NZ <= 20) break;
         gNZ--;
         continue;
      }

      if (dNZ == 0) break;
      if (dNZ > 0 && NZ > pNZ) break;
      if (dNZ < 0 && NZ < pNZ) break;
   }

   pNZ = NZ;                                                                     //  final zone count
   dNZ = 0;
   return;
}


//  build up the zones data when NZ or the image size changes
//  (preview or full size image)

void flatten_names::initzones()
{
   int      px, py, cx, cy;
   int      rx, ii, jj, kk;
   int      zww, zhh, row, col;
   float    *pix1, bright;
   float    weight[9], sumweight;
   float    D, Dthresh;

   if (Br) {
      zfree(Br);                                                                 //  free memory
      zfree(Zn);
      zfree(Zw);
      zfree(Zxlo);
      zfree(Zylo);
      zfree(Zxhi);
      zfree(Zyhi);
      zfree(Zcen);
      zfree(Zff);
      Br = 0;
   }

   Iww = E1pxm->ww;                                                              //  image dimensions
   Ihh = E1pxm->hh;

   Br = (float *) zmalloc(Iww * Ihh * sizeof(float));                            //  allocate memory
   Zn = (int16 *) zmalloc(Iww * Ihh * 9 * sizeof(int16));
   Zw = (uint8 *) zmalloc(Iww * Ihh * 9 * sizeof(uint8));
   Zxlo = (int *) zmalloc(NZ * sizeof(int));
   Zylo = (int *) zmalloc(NZ * sizeof(int));
   Zxhi = (int *) zmalloc(NZ * sizeof(int));
   Zyhi = (int *) zmalloc(NZ * sizeof(int));
   Zcen = (int *) zmalloc(NZ * 2 * sizeof(int));
   Zff = (float *) zmalloc(NZ * 1000 * sizeof(float));

   for (py = 0; py < Ihh; py++)                                                  //  get initial pixel brightness levels
   for (px = 0; px < Iww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright = pixbright(pix1);
      ii = py * Iww + px;
      Br[ii] = bright;
   }

   zww = Iww / Zcols;                                                            //  actual zone size
   zhh = Ihh / Zrows;

   for (row = 0; row < Zrows; row++)
   for (col = 0; col < Zcols; col++)                                             //  set low/high bounds per zone
   {
      ii = row * Zcols + col;
      Zxlo[ii] = col * zww;
      Zylo[ii] = row * zhh;
      Zxhi[ii] = Zxlo[ii] + zww;
      Zyhi[ii] = Zylo[ii] + zhh;
      Zcen[2*ii] = Zylo[ii] + zhh/2;
      Zcen[2*ii+1] = Zxlo[ii] + zww/2;
   }

   for (ii = 0; ii < NZ; ii++)                                                   //  compute brightness distributiion
   {                                                                             //    for each zone
      for (jj = 0; jj < 1000; jj++)
         Zff[1000*ii+jj] = 0;

      for (py = Zylo[ii]; py < Zyhi[ii]; py++)                                   //  brightness distribution
      for (px = Zxlo[ii]; px < Zxhi[ii]; px++)
      {
         pix1 = PXMpix(E1pxm,px,py);
         kk = pixbright(pix1);
         if (kk > 255) kk = 255;
         bright = 3.906 * kk;                                                    //  1000 / 256 * kk
         Zff[1000*ii+int(bright)]++;
      }

      for (jj = 1; jj < 1000; jj++)                                              //  cumulative brightness distribution
         Zff[1000*ii+jj] += Zff[1000*ii+jj-1];

      for (jj = 0; jj < 1000; jj++)                                              //  multiplier per brightness level
         Zff[1000*ii+jj] = Zff[1000*ii+jj] / (zww*zhh) * 1000 / (jj+1);          //    to make distribution flat
   }

   for (py = 0; py < Ihh; py++)                                                  //  set 9 nearest zones per pixel
   for (px = 0; px < Iww; px++)
   {
      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      row = py / zhh;                                                            //  zone containing pixel
      col = px / zww;

      ii = 0;
      for (jj = row-1; jj <= row+1; jj++)                                        //  loop 3x3 surrounding zones
      for (kk = col-1; kk <= col+1; kk++)
      {
         if (jj >= 0 && jj < Zrows && kk >= 0 && kk < Zcols)                     //  zone is not off the edge
            Zn[rx+ii] = jj * Zcols + kk;
         else Zn[rx+ii] = -1;                                                    //  edge row/col: missing neighbor
         ii++;
      }
   }

   if (zww < zhh)                                                                //  pixel to zone distance threshold
      Dthresh = 1.5 * zww;                                                       //  area influence = 0 beyond this distance
   else Dthresh = 1.5 * zhh;

   for (py = 0; py < Ihh; py++)                                                  //  set zone weights per pixel
   for (px = 0; px < Iww; px++)
   {
      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      for (ii = 0; ii < 9; ii++)                                                 //  distance to each zone center
      {
         jj = Zn[rx+ii];
         if (jj == -1) {                                                         //  missign zone
            weight[ii] = 0;                                                      //  weight = 0
            continue;
         }
         cy = Zcen[2*jj];
         cx = Zcen[2*jj+1];
         D = sqrtf((py-cy)*(py-cy) + (px-cx)*(px-cx));                           //  distance from pixel to zone center
         D = D / Dthresh;
         if (D > 1) D = 1;                                                       //  zone influence reaches zero at Dthresh
         weight[ii] = 1.0 - D;
      }

      sumweight = 0;
      for (ii = 0; ii < 9; ii++)                                                 //  zone weights based on distance
         sumweight += weight[ii];

      for (ii = 0; ii < 9; ii++)                                                 //  normalize weights, sum = 1.0
         Zw[rx+ii] = 100 * weight[ii] / sumweight;                               //  0-100 = 1.0
   }

   return;
}


//  adjust brightness distribution thread function

void * flatten_names::thread(void *)
{
   using namespace flatten_names;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      if (Zinit) initzones();                                                    //  reinitialize zones
      Zinit = 0;

      for (int ii = 0; ii < NWT; ii++)                                           //  start worker thread per processor core
         start_wthread(wthread,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      if (! flatten) CEF->Fmods = 0;                                             //  no modification
      else {
         CEF->Fmods++;                                                           //  image modified, not saved
         CEF->Fsaved = 0;
      }
   }

   return 0;                                                                     //  not executed, stop g++ warning
}


//  worker thread for each CPU processor core

void * flatten_names::wthread(void *arg)
{
   using namespace flatten_names;

   int         index = *((int *) (arg));
   int         px, py, rx, ii, jj, dist = 0;
   float       *pix1, *pix3;
   float       fnew1, fnew2, fnew, fold;
   float       dold, dnew, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   float       FF, weight, bright;

   for (py = index; py < Ihh; py += NWT)                                         //  loop all image pixels
   for (px = 0; px < Iww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright = 0.976 * red1 + 2.539 * green1 + 0.39 * blue1;                     //  brightness scaled 0-999.9          18.01
      
      rx = (py * Iww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      FF = 0;
      for (ii = 0; ii < 9; ii++) {                                               //  loop 9 nearest zones
         weight = Zw[rx+ii];
         if (weight > 0) {                                                       //  0-100 = 1.0
            jj = Zn[rx+ii];
            FF += 0.01 * weight * Zff[1000*jj+int(bright)];                      //  sum weight * flatten factor
         }
      }

      red3 = FF * red1;                                                          //  fully flattened brightness
      green3 = FF * green1;
      blue3 = FF * blue1;
      
      fnew1 = 1 - 0.01 * deband1 * 0.001 * (1000 - bright);                      //  attenuate dark pixels
      fnew2 = 1 - 0.01 * deband2 * 0.001 * bright;                               //  attenuate bright pixels
      fnew = fnew1 * fnew2;

      fnew = fnew * 0.01 * flatten;                                              //  how much to flatten, 0 to 1
      fold = 1.0 - fnew;                                                         //  how much to retain, 1 to 0

      red3 = fnew * red3 + fold * red1;                                          //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      if (sa_stat == 3 && dist < sa_blend) {                                     //  select area is active,
         dnew = sa_blendfunc(dist);                                              //  blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                               //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }
      
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************/

//  Retinex function
//  Global: rescale RGB values based on entire image - eliminate color caste and reduce fog/haze.
//  Zonal: rescale RGB values based on surrounding zones: increase local brightness range.
//  Transfer color (RGB ratios) from global to zonal result.

namespace retinex_names 
{
   editfunc    EFretinex;                                                        //  edit function data
   int         e3ww, e3hh;
   cchar       *thread_command;                                                  //  "global" or "zonal"

   float       Rdark, Gdark, Bdark;                                              //  global parameters
   float       Rbrite, Gbrite, Bbrite;
   float       Rmpy, Gmpy, Bmpy;
   float       pRdark, pGdark, pBdark;                                           //  prior values 
   float       pRbrite, pGbrite, pBbrite;
   float       pRmpy, pGmpy, pBmpy;
   int         Fbrightpoint, Fdarkpoint;

   int         zonesize, pzonesize;                                              //  zonal parameters
   float       blend, reducebright;
   float       *LminR, *LminG, *LminB, *LmaxR, *LmaxG, *LmaxB;                   //  RGB range
   float       *Lmin2R, *Lmin2G, *Lmin2B, *Lmax2R, *Lmax2G, *Lmax2B;
}


//  menu function

void m_retinex(GtkWidget *, cchar *menu)                                         //  new 18.01
{
   using namespace retinex_names;

   int    retinex_dialog_event(zdialog *zd, cchar *event);
   void   retinex_mousefunc();
   void * retinex_thread(void *);

   F1_help_topic = "retinex";

   EFretinex.menuname = menu;
   EFretinex.menufunc = m_retinex;
   EFretinex.funcname = "retinex";                                               //  function name
   EFretinex.FprevReq = 1;                                                       //  use preview
   EFretinex.Farea = 2;                                                          //  select area usable
   EFretinex.Frestart = 1;                                                       //  allow restart
   EFretinex.Fscript = 1;                                                        //  scripting supported
   EFretinex.threadfunc = retinex_thread;                                        //  thread function
   EFretinex.mousefunc = retinex_mousefunc;                                      //  mouse function

   if (! edit_setup(EFretinex)) return;                                          //  setup edit

   e3ww = E3pxm->ww;                                                             //  image size
   e3hh = E3pxm->hh;
   E8pxm = E9pxm = 0;
   
   int cc = e3ww * e3hh;

   LminR = (float *) zmalloc(cc * sizeof(float));                                //  allocate zone RGB range
   LminG = (float *) zmalloc(cc * sizeof(float));
   LminB = (float *) zmalloc(cc * sizeof(float));

   LmaxR = (float *) zmalloc(cc * sizeof(float));
   LmaxG = (float *) zmalloc(cc * sizeof(float));
   LmaxB = (float *) zmalloc(cc * sizeof(float));

/***
          ______________________________________
         |              Retinex                 |
         |                                      |
         |               Red  Green  Blue   +/- |
         |  Dark Point:  [__]  [__]  [__]  [__] |        0 ... 255      neutral point is 0
         | Bright Point: [__]  [__]  [__]  [__] |        0 ... 255      neutral point is 255
         |  Multiplyer:  [__]  [__]  [__]  [__] |        0 ... 5        neutral point is 1
         |                                      |
         | [auto] brightness rescale            |
         | [x] bright point  [x] dark point     |        mouse click spots
         | - - - - - - - - - - - - - - - - - -  |
         | zone size: [___]  [apply]            |
         | - - - - - - - - - - - - - - - - - -  |
         | blend: =======[]===================  |
         | reduce bright: ==========[]========  |
         |                                      |
         |             [reset] [done] [cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new("Retinex",Mwin,Breset,Bdone,Bcancel,null); 
   EFretinex.zd = zd;
   
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","space","hb1","","space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=5");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=5");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=5");
   zdialog_add_widget(zd,"vbox","vb5","hb1",0,"homog");
   
   zdialog_add_widget(zd,"label","labglobal","vb1","Global Retinex");
   zdialog_add_widget(zd,"label","labdark","vb1",ZTX("Dark Point"));
   zdialog_add_widget(zd,"label","labbrite","vb1",ZTX("Bright Point"));
   zdialog_add_widget(zd,"label","labmpy","vb1",ZTX("Multiplyer"));

   zdialog_add_widget(zd,"label","labred","vb2",Bred);
   zdialog_add_widget(zd,"zspin","Rdark","vb2","0|255|1|0");
   zdialog_add_widget(zd,"zspin","Rbrite","vb2","0|255|1|255");
   zdialog_add_widget(zd,"zspin","Rmpy","vb2","0.1|5|0.01|1");

   zdialog_add_widget(zd,"label","labgreen","vb3",Bgreen);
   zdialog_add_widget(zd,"zspin","Gdark","vb3","0|255|1|0");
   zdialog_add_widget(zd,"zspin","Gbrite","vb3","0|255|1|255");
   zdialog_add_widget(zd,"zspin","Gmpy","vb3","0.1|5|0.01|1");

   zdialog_add_widget(zd,"label","labred","vb4",Bblue);
   zdialog_add_widget(zd,"zspin","Bdark","vb4","0|255|1|0");
   zdialog_add_widget(zd,"zspin","Bbrite","vb4","0|255|1|255");
   zdialog_add_widget(zd,"zspin","Bmpy","vb4","0.1|5|0.01|1");

   zdialog_add_widget(zd,"label","laball","vb5",Ball);
   zdialog_add_widget(zd,"zspin","dark+-","vb5","-1|+1|1|0");
   zdialog_add_widget(zd,"zspin","brite+-","vb5","-1|+1|1|0");
   zdialog_add_widget(zd,"zspin","mpy+-","vb5","-1|+1|1|0");

   zdialog_add_widget(zd,"hbox","hbauto","dialog");
   zdialog_add_widget(zd,"button","autoscale","hbauto",Bauto,"space=3");
   zdialog_add_widget(zd,"label","labauto","hbauto",ZTX("brightness rescale"),"space=5");

   zdialog_add_widget(zd,"hbox","hbclicks","dialog");
   zdialog_add_widget(zd,"check","brightpoint","hbclicks",ZTX("click bright point"),"space=3");
   zdialog_add_widget(zd,"check","darkpoint","hbclicks",ZTX("click dark point"),"space=5");
   
   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbz","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labzr","hbz","Zonal Retinex","space=5");
   zdialog_add_widget(zd,"label","labzs","hbz","zone size:","space=5");
   zdialog_add_widget(zd,"zspin","zone size","hbz","10|500|1|200");
   zdialog_add_widget(zd,"button","apply","hbz",Bapply,"space=3");

   zdialog_add_widget(zd,"hsep","sep2","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbb","dialog");
   zdialog_add_widget(zd,"label","labb","hbb",ZTX("blend"),"space=5");
   zdialog_add_widget(zd,"hscale","blend","hbb","0|1.0|0.01|1.0","expand");
   zdialog_add_widget(zd,"hbox","hbrd","dialog");
   zdialog_add_widget(zd,"label","labrd","hbrd",ZTX("reduce bright"),"space=5");
   zdialog_add_widget(zd,"hscale","reduce bright","hbrd","0|1.0|0.01|0.0","expand");

   zdialog_run(zd,retinex_dialog_event,"save");                                  //  run dialog - parallel
   zdialog_send_event(zd,"reset");
   return;
}


//  dialog event and completion function

int retinex_dialog_event(zdialog *zd, cchar *event)
{
   using namespace retinex_names;
   
   void  retinex_mousefunc();
   void  retinex_fullsize();

   int      adddark, addbrite, addmpy;
   int      Fchange = 0;
   
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   if (strmatch(event,"reset")) zd->zstat = 1;                                   //  initz. 
   
   if (strmatch(event,"fullsize")) {                                             //  from select area
      retinex_fullsize();                                                        //  apply to full size image
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         Rdark = Gdark = Bdark = 0;                                              //  set neutral values
         Rbrite = Gbrite = Bbrite = 255;
         pRdark = pGdark = pBdark = 0;                                           //  prior values = same
         pRbrite = pGbrite = pBbrite = 255;                                      //  (no change)
         Rmpy = Gmpy = Bmpy = 1.0;
         Fbrightpoint = Fdarkpoint = 0;
         zonesize = pzonesize = 0;
         blend = 1.0;
         reducebright = 0.0;

         zdialog_stuff(zd,"Rdark",0);                                            //  stuff neutral values into dialog
         zdialog_stuff(zd,"Gdark",0);
         zdialog_stuff(zd,"Bdark",0);
         zdialog_stuff(zd,"Rbrite",255);
         zdialog_stuff(zd,"Gbrite",255);
         zdialog_stuff(zd,"Bbrite",255);
         zdialog_stuff(zd,"Rmpy",1.0);
         zdialog_stuff(zd,"Gmpy",1.0);
         zdialog_stuff(zd,"Bmpy",1.0);
         zdialog_stuff(zd,"brightpoint",0);
         zdialog_stuff(zd,"darkpoint",0);

         edit_reset();
         if (E8pxm) PXM_free(E8pxm);                                             //  free memory
         if (E9pxm) PXM_free(E9pxm);
         E8pxm = E9pxm = 0;
         return 1;
      }

      else if (zd->zstat == 2) {                                                 //  done
         retinex_fullsize();                                                     //  apply to full size image
         edit_done(0);                                                           //  commit edit
         if (E8pxm) PXM_free(E8pxm);                                             //  free memory
         if (E9pxm) PXM_free(E9pxm);
         E8pxm = E9pxm = 0;
         zfree(LminR);
         zfree(LminG);
         zfree(LminB);
         zfree(LmaxR);
         zfree(LmaxG);
         zfree(LmaxB);
         return 1;
      }

      else {
         edit_cancel(0);                                                         //  discard edit
         if (E8pxm) PXM_free(E8pxm);                                             //  free memory
         if (E9pxm) PXM_free(E9pxm);
         E8pxm = E9pxm = 0;
         zfree(LminR);
         zfree(LminG);
         zfree(LminB);
         zfree(LmaxR);
         zfree(LmaxG);
         zfree(LmaxB);
         return 1;
      }
   }

   Fbrightpoint = Fdarkpoint = 0;                                                //  disable mouse

   if (strmatch(event,"autoscale"))                                              //  auto set dark and bright points
   {
      edit_reset();
      
      thread_command = "autoscale";                                              //  get dark/bright limits from
      signal_thread();                                                           //    darkest/brightest pixels
      wait_thread_idle();

      zdialog_stuff(zd,"Rdark",Rdark);                                           //  update dialog widgets
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
      zdialog_stuff(zd,"Rbrite",Rbrite);
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
      zdialog_stuff(zd,"Rmpy",1.0);
      zdialog_stuff(zd,"Gmpy",1.0);
      zdialog_stuff(zd,"Bmpy",1.0);
      zdialog_stuff(zd,"brightpoint",0);
      zdialog_stuff(zd,"darkpoint",0);

      pRdark = Rdark;                                                            //  prior values = same
      pGdark = Gdark;
      pBdark = Bdark;
      pRbrite = Rbrite; 
      pGbrite = Gbrite; 
      pBbrite = Bbrite;
      pRmpy = Rmpy;
      pGmpy = Gmpy;
      pBmpy = Bmpy;
      return 1;
   }
   
   if (strmatch(event,"brightpoint")) {
      zdialog_fetch(zd,"brightpoint",Fbrightpoint);
      if (Fbrightpoint) {
         zdialog_stuff(zd,"darkpoint",0);
         Fdarkpoint = 0;
      }
   }

   if (strmatch(event,"darkpoint")) {
      zdialog_fetch(zd,"darkpoint",Fdarkpoint);
      if (Fdarkpoint) {
         zdialog_stuff(zd,"brightpoint",0);
         Fbrightpoint = 0;
      }
   }
   
   if (strstr("brightpoint darkpoint",event)) {                                  //  brightpoint or darkpoint
      takeMouse(retinex_mousefunc,dragcursor);                                   //     connect mouse function
      return 1;
   }
   else {
      zdialog_stuff(zd,"brightpoint",0);                                         //  reset zdialog checkboxes
      zdialog_stuff(zd,"darkpoint",0);
   }
   
   adddark = addbrite = addmpy = 0;
   if (strmatch(event,"dark+-")) zdialog_fetch(zd,"dark+-",adddark);             //  [+/-] button
   if (strmatch(event,"brite+-")) zdialog_fetch(zd,"brite+-",addbrite);
   if (strmatch(event,"mpy+-")) zdialog_fetch(zd,"mpy+-",addmpy);
   
   if (adddark) {
      zdialog_fetch(zd,"Rdark",Rdark);                                           //  increment dark levels
      zdialog_fetch(zd,"Gdark",Gdark);
      zdialog_fetch(zd,"Bdark",Bdark);
      Rdark += adddark;
      Gdark += adddark;
      Bdark += adddark;
      if (Rdark < 0) Rdark = 0;
      if (Gdark < 0) Gdark = 0;
      if (Bdark < 0) Bdark = 0;
      if (Rdark > 255) Rdark = 255;
      if (Gdark > 255) Gdark = 255;
      if (Bdark > 255) Bdark = 255;
      zdialog_stuff(zd,"Rdark",Rdark);
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
   }

   if (addbrite) {                                                               //  increment bright levels
      zdialog_fetch(zd,"Rbrite",Rbrite);
      zdialog_fetch(zd,"Gbrite",Gbrite);
      zdialog_fetch(zd,"Bbrite",Bbrite);
      Rbrite += addbrite;
      Gbrite += addbrite;
      Bbrite += addbrite;
      if (Rbrite < 0) Rbrite = 0;
      if (Gbrite < 0) Gbrite = 0;
      if (Bbrite < 0) Bbrite = 0;
      if (Rbrite > 255) Rbrite = 255;
      if (Gbrite > 255) Gbrite = 255;
      if (Bbrite > 255) Bbrite = 255;
      zdialog_stuff(zd,"Rbrite",Rbrite);
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
   }

   if (addmpy) {                                                                 //  increment mpy factors
      zdialog_fetch(zd,"Rmpy",Rmpy);
      zdialog_fetch(zd,"Gmpy",Gmpy);
      zdialog_fetch(zd,"Bmpy",Bmpy);
      Rmpy += 0.01 * addmpy;
      Gmpy += 0.01 * addmpy;
      Bmpy += 0.01 * addmpy;
      if (Rmpy < 0.1) Rmpy = 0.1;
      if (Gmpy < 0.1) Gmpy = 0.1;
      if (Bmpy < 0.1) Bmpy = 0.1;
      if (Rmpy > 5) Rmpy = 5;
      if (Gmpy > 5) Gmpy = 5;
      if (Bmpy > 5) Bmpy = 5;
      zdialog_stuff(zd,"Rmpy",Rmpy);
      zdialog_stuff(zd,"Gmpy",Gmpy);
      zdialog_stuff(zd,"Bmpy",Bmpy);
   }
   
   zdialog_fetch(zd,"Rdark",Rdark);                                              //  get all params
   zdialog_fetch(zd,"Gdark",Gdark);
   zdialog_fetch(zd,"Bdark",Bdark);
   zdialog_fetch(zd,"Rbrite",Rbrite);
   zdialog_fetch(zd,"Gbrite",Gbrite);
   zdialog_fetch(zd,"Bbrite",Bbrite);
   zdialog_fetch(zd,"Rmpy",Rmpy);
   zdialog_fetch(zd,"Gmpy",Gmpy);
   zdialog_fetch(zd,"Bmpy",Bmpy);
   
   Fchange = 0;
   if (Rdark != pRdark) Fchange = 1;                                             //  detect changed params
   if (Gdark != pGdark) Fchange = 1;
   if (Bdark != pBdark) Fchange = 1;
   if (Rbrite != pRbrite) Fchange = 1;
   if (Gbrite != pGbrite) Fchange = 1;
   if (Bbrite != pBbrite) Fchange = 1;
   if (Rmpy != pRmpy) Fchange = 1;
   if (Gmpy != pGmpy) Fchange = 1;
   if (Bmpy != pBmpy) Fchange = 1;
   
   pRdark = Rdark;                                                               //  remember values for change detection
   pGdark = Gdark;
   pBdark = Bdark;
   pRbrite = Rbrite; 
   pGbrite = Gbrite; 
   pBbrite = Bbrite; 
   pRmpy = Rmpy; 
   pGmpy = Gmpy; 
   pBmpy = Bmpy; 
   
   if (Fchange) {                                                                //  global params changed
      thread_command = "global";
      signal_thread();                                                           //  update image
      wait_thread_idle();
      Fchange = 0;
   }
   
   if (strmatch(event,"apply")) {                                                //  new zone size
      zdialog_fetch(zd,"zone size",zonesize);
      thread_command = "zonal";
      signal_thread();                                                           //  update image
      wait_thread_idle();
   }

   if (strstr("blend, reduce bright",event)) {                                   //  blend params changed
      zdialog_fetch(zd,"blend",blend);
      zdialog_fetch(zd,"reduce bright",reducebright);
      thread_command = "blend";
      signal_thread();                                                           //  update image
      wait_thread_idle();
   }

   return 1;
}


//  get dark point or bright point from mouse click position

void retinex_mousefunc()
{
   using namespace retinex_names;
   
   int         px, py, dx, dy;
   float       red, green, blue;
   float       *ppix;
   char        mousetext[60];
   zdialog     *zd = EFretinex.zd;
   
   if (! zd) {
      freeMouse();
      return;
   }

   if (! Fbrightpoint && ! Fdarkpoint) {
      freeMouse();
      return;
   }

   if (! LMclick) return;
   LMclick = 0;

   px = Mxclick;                                                                 //  mouse click position
   py = Myclick;

   if (px < 2) px = 2;                                                           //  pull back from edge
   if (px > E3pxm->ww-3) px = E3pxm->ww-3;
   if (py < 2) py = 2;
   if (py > E3pxm->hh-3) py = E3pxm->hh-3;

   red = green = blue = 0;

   for (dy = -1; dy <= 1; dy++)                                                  //  3x3 block around mouse position
   for (dx = -1; dx <= 1; dx++)
   {
      ppix = PXMpix(E1pxm,px+dx,py+dy);                                          //  input image
      red += ppix[0];
      green += ppix[1];
      blue += ppix[2];
   }

   red = red / 9.0;                                                              //  mean RGB levels
   green = green / 9.0;
   blue = blue / 9.0;

   snprintf(mousetext,60,"3x3 pixels RGB: %.0f %.0f %.0f \n",red,green,blue);
   poptext_mouse(Mwin,mousetext,10,10,0,3);
   
   if (Fbrightpoint) {                                                           //  click pixel is new bright point
      Rbrite= red;
      Gbrite = green;
      Bbrite = blue;
      zdialog_stuff(zd,"Rbrite",Rbrite);                                         //  stuff values into dialog
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
   }

   if (Fdarkpoint) {                                                             //  click pixel is new dark point
      Rdark = red;
      Gdark = green;
      Bdark = blue;
      zdialog_stuff(zd,"Rdark",Rdark);                                           //  stuff values into dialog
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
   }
   
   if (Fbrightpoint || Fdarkpoint) {
      thread_command = "global";
      signal_thread();                                                           //  trigger image update
      wait_thread_idle();
   }
   
   return;
}


//  edit fullsize - scale zones from preview to full size image

void retinex_fullsize()
{
   using namespace retinex_names;

   int      pww, phh, cc, ii, kk;
   int      px, py, qx, qy;
   int      zonesize2;

   if (! CEF->Fpreview) return;                                                  //  already full-size
   
   pww = e3ww;                                                                   //  save preview image size
   phh = e3hh;
   
   zonesize2 = zonesize;                                                         //  preserve zone size

   edit_fullsize();                                                              //  get full size image

   e3ww = E3pxm->ww;                                                             //  new image size
   e3hh = E3pxm->hh;

   thread_command = "global";
   signal_thread();                                                              //  update image
   wait_thread_idle();
   
   Lmin2R = LminR;                                                               //  preview zone limits
   Lmin2G = LminG;
   Lmin2B = LminB;

   Lmax2R = LmaxR;
   Lmax2G = LmaxG;
   Lmax2B = LmaxB;

   cc = e3ww * e3hh;

   LminR = (float *) zmalloc(cc * sizeof(float));                                //  allocate full-size zone limits
   LminG = (float *) zmalloc(cc * sizeof(float));
   LminB = (float *) zmalloc(cc * sizeof(float));

   LmaxR = (float *) zmalloc(cc * sizeof(float));
   LmaxG = (float *) zmalloc(cc * sizeof(float));
   LmaxB = (float *) zmalloc(cc * sizeof(float));

   for (py = 0; py < e3hh; py++)                                                 //  loop full size image pixels
   for (px = 0; px < e3ww; px++)
   {
      qy = py * 1.0 * phh / e3hh;                                                //  corresp. preview image pixels
      qx = px * 1.0 * pww / e3ww;

      ii = py * e3ww + px;                                                       //  copy the preview zone limits to
      kk = qy * pww + qx;                                                        //    corresp. full-size zone limits

      LminR[ii] = Lmin2R[kk];
      LminG[ii] = Lmin2G[kk];
      LminB[ii] = Lmin2B[kk];

      LmaxR[ii] = Lmax2R[kk]; 
      LmaxG[ii] = Lmax2G[kk]; 
      LmaxB[ii] = Lmax2B[kk]; 
   }
   
   zfree(Lmin2R);                                                                //  free preview zone limits
   zfree(Lmin2G);
   zfree(Lmin2B);

   zfree(Lmax2R);
   zfree(Lmax2G);
   zfree(Lmax2B);
   
   zonesize = pzonesize = zonesize2;                                             //  restore zone size

   thread_command = "zonal";
   signal_thread();                                                              //  update image
   wait_thread_idle();

   return;
}


//  thread function - multiple working threads to update image

void * retinex_thread(void *)
{
   using namespace retinex_names;

   void  * Aretinex_wthread(void *arg);                                          //  worker threads
   void  * Gretinex_wthread(void *arg);
   void  * Zretinex_wthread(void *arg);
   void  * Bretinex_wthread(void *arg);
   
   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request
      
      if (strmatch(thread_command,"autoscale")) {
         Aretinex_wthread(0);                                                    //  worker thread for autoscale retinex
         thread_command = "global";
      }
      
      if (strmatch(thread_command,"global"))
      {
         for (int ii = 0; ii < NWT; ii++)                                        //  worker thread for global retinex
            start_wthread(Gretinex_wthread,&Nval[ii]);
         wait_wthreads();                                                        //  wait for completion

         if (E8pxm) PXM_free(E8pxm);                                             //  E8 image = global retinex output
         E8pxm = PXM_copy(E3pxm);

         zonesize = pzonesize = 0;                                               //  invalidate zonal retinex
         if (E9pxm) PXM_free(E9pxm);
         E9pxm = 0;
         
         thread_command = "blend";                                               //  auto blend after global retinex
      }
      
      if (strmatch(thread_command,"zonal"))                                      //  worker thread for zonal retinex
      {
         Zretinex_wthread(0);
         
         if (E9pxm) PXM_free(E9pxm);                                             //  E9 image = zonal retinex output
         E9pxm = PXM_copy(E3pxm);
         
         thread_command = "blend";                                               //  auto blend after zonal retinex
      }
      
      if (strmatch(thread_command,"blend"))
      {
         for (int ii = 0; ii < NWT; ii++)                                        //  worker thread for blend retinex
            start_wthread(Bretinex_wthread,&Nval[ii]);
         wait_wthreads();                                                        //  wait for completion
      }

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;
      Fpaint2();                                                                 //  update window
   }

   return 0;                                                                     //  not executed, stop warning
}


//  worker thread function - autoscale retinex

void * Aretinex_wthread(void *)
{
   using namespace retinex_names;
   
   int      ii, dist = 0;
   int      px, py, dx, dy;
   int      red, green, blue;
   float    *pix1;

   Rdark = Gdark = Bdark = 255;
   Rbrite = Gbrite = Bbrite = 0;
   Rmpy = Gmpy = Bmpy = 1.0;
   Fbrightpoint = Fdarkpoint = 0;

   for (py = 1; py < E1pxm->hh-1; py++)
   for (px = 1; px < E1pxm->ww-1; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      red = green = blue = 0;

      for (dy = -1; dy <= 1; dy++)                                               //  3x3 block around mouse position
      for (dx = -1; dx <= 1; dx++)
      {
         pix1 = PXMpix(E1pxm,px+dx,py+dy);                                       //  input image
         red += pix1[0];
         green += pix1[1];
         blue += pix1[2];
      }

      red = red / 9.0;                                                           //  mean RGB levels
      green = green / 9.0;
      blue = blue / 9.0;
      
      if (red < Rdark) Rdark = red;                                              //  update limits
      if (green < Gdark) Gdark = green;
      if (blue < Bdark) Bdark = blue;
      if (red > Rbrite) Rbrite = red;
      if (green > Gbrite) Gbrite = green;
      if (blue > Bbrite) Bbrite = blue;
   }

   return 0;                                                                     //  not executed, avoid gcc warning
}


//  worker thread function - global retinex

void * Gretinex_wthread(void *arg)
{
   using namespace retinex_names;
   
   int      ii, index, dist = 0;
   int      px, py;
   float    R1, G1, B1, R3, G3, B3;
   float    f1, f2, F, cmax;
   float    *pix1, *pix3;

   index = *((int *) arg);
   
   for (py = index; py < E3pxm->hh; py += NWT)                                   //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];                                                              //  input RGB values
      G1 = pix1[1];
      B1 = pix1[2];
      
      R1 = 255 * (R1 - Rdark) / (Rbrite - Rdark);                                //  rescale for full 0-255 range
      G1 = 255 * (G1 - Gdark) / (Gbrite - Gdark);
      B1 = 255 * (B1 - Bdark) / (Bbrite - Bdark);

      if (R1 < 0) R1 = 0;
      if (G1 < 0) G1 = 0;
      if (B1 < 0) B1 = 0;

      R3 = R1 * Rmpy;
      G3 = G1 * Gmpy;
      B3 = B1 * Bmpy;

      if (R3 > 255 || G3 > 255 || B3 > 255) {                                    //  stop overflow
         cmax = R3;
         if (G3 > cmax) cmax = G3;
         if (B3 > cmax) cmax = B3;
         F = 255 / cmax;
         R3 *= F;
         G3 *= F;
         B3 *= F;
      }

      if (sa_stat == 3 && dist < sa_blend) {                                     //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blend
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }

      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


//  worker thread function - zonal retinex

void * Zretinex_wthread(void *)
{
   using namespace retinex_names;

   void  * Zretinex_wthread1(void *arg);                                         //  worker threads
   void  * Zretinex_wthread2(void *arg);
   void  * Zretinex_wthread3(void *arg);
   void  * Zretinex_wthread4(void *arg);

   int      ii, cc;
   
   if (zonesize == 0) return 0;                                                  //  no zone size set

   if (zonesize == pzonesize) goto rescale;                                      //  no change, can skip
   pzonesize = zonesize;

//  compute luminance range for zone surrounding each pixel

   Fbusy_goal = e3ww * e3hh;                                                     //  track progress
   Fbusy_done = 0;

   for (ii = 0; ii < NWT; ii++)                                                  //  start worker threads
      start_wthread(Zretinex_wthread1,&Nval[ii]);
   wait_wthreads();                                                              //  wait for completion

   Fbusy_goal = 0;

//  "anneal" the zone luminance values with neighboring zones

   cc = e3ww * e3hh;                                                             

   Lmin2R = (float *) zmalloc(cc * sizeof(float));
   Lmin2G = (float *) zmalloc(cc * sizeof(float));
   Lmin2B = (float *) zmalloc(cc * sizeof(float));

   Lmax2R = (float *) zmalloc(cc * sizeof(float));
   Lmax2G = (float *) zmalloc(cc * sizeof(float));
   Lmax2B = (float *) zmalloc(cc * sizeof(float));

   Fbusy_goal = e3ww * e3hh;                                                     //  track progress
   Fbusy_done = 0;

   for (ii = 0; ii < NWT; ii++)                                                  //  start worker threads
      start_wthread(Zretinex_wthread2,&Nval[ii]);
   wait_wthreads();                                                              //  wait for completion

   Fbusy_goal = 0;

   zfree(LminR);                                                                 //  replace zone luminance values
   zfree(LminG);
   zfree(LminB);

   zfree(LmaxR);
   zfree(LmaxG);
   zfree(LmaxB);

   LminR = Lmin2R;
   LminG = Lmin2G;
   LminB = Lmin2B;

   LmaxR = Lmax2R;
   LmaxG = Lmax2G;
   LmaxB = Lmax2B;

rescale:

//  rescale the luminance of each pixel so that the full 0-255 range is used

   for (ii = 0; ii < NWT; ii++)                                                  //  start worker threads
      start_wthread(Zretinex_wthread3,&Nval[ii]);
   wait_wthreads();                                                              //  wait for completion
   
   return 0;
}


//  zonal worker thread 1 - compute RGB range for each zone

void * Zretinex_wthread1(void *arg)
{
   using namespace retinex_names;

   int      index = *((int *) arg);
   int      px, py, qx, qy;
   int      xlo, xhi, ylo, yhi;
   int      ii, rad, dist = 0;
   float    *pix1;
   float    lminR, lminG, lminB, lmaxR, lmaxG, lmaxB;

   rad = zonesize / 2;

   for (py = index; py < e3hh; py += NWT)                                        //  loop each image pixel
   for (px = 0; px < e3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * e3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }
   
      lminR = lminG = lminB = 255;
      lmaxR = lmaxG = lmaxB = 0;
      
      ylo = py - rad;
      if (ylo < 0) ylo = 0;
      yhi = py + rad;
      if (yhi > e3hh) yhi = e3hh;

      xlo = px - rad;
      if (xlo < 0) xlo = 0;
      xhi = px + rad;
      if (xhi > e3ww) xhi = e3ww;

      for (qy = ylo; qy < yhi; qy += 2)                                          //  find min and max RGB
      for (qx = xlo; qx < xhi; qx += 2)                                          //    for zone surrounding pixel
      {
         if (sa_stat == 3) {                                                     //  select area active
            ii = qy * e3ww + qx;
            dist = sa_pixmap[ii];                                                //  distance from edge
            if (! dist) continue;                                                //  outside area
         }

         if (E8pxm) pix1 = PXMpix(E8pxm,qx,qy);                                  //  input image from global retinex
         else pix1 = PXMpix(E1pxm,qx,qy);                                        //  else original input image
         if (pix1[0] < lminR) lminR = pix1[0];
         if (pix1[0] > lmaxR) lmaxR = pix1[0];
         if (pix1[1] < lminG) lminG = pix1[1];
         if (pix1[1] > lmaxG) lmaxG = pix1[1];
         if (pix1[2] < lminB) lminB = pix1[2];
         if (pix1[2] > lmaxB) lmaxB = pix1[2];
      }

      ii = py * e3ww + px;
      
      LminR[ii] = lminR;
      LmaxR[ii] = lmaxR;
      LminG[ii] = lminG;
      LmaxG[ii] = lmaxG;
      LminB[ii] = lminB;
      LmaxB[ii] = lmaxB;

      Fbusy_done++;
   }
   
   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


//  zonal worker thread 2 - anneal zone RGB values

void * Zretinex_wthread2(void *arg)
{
   using namespace retinex_names;

   int      index = *((int *) arg);
   int      px, py, qx, qy;
   int      ylo, yhi;
   int      rad, ii, lsum;
   double   lminR, lminG, lminB, lmaxR, lmaxG, lmaxB;
   
   rad = zonesize;

   for (py = index; py < e3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < e3ww; px++)
   {
      if (px == 0)                                                               //  start new row
      {
         lminR = lminG = lminB = 0;
         lmaxR = lmaxG = lmaxB = 0;
         lsum = 0;
         
         ylo = py - rad;
         if (ylo < 0) ylo = 0;
         yhi = py + rad;
         if (yhi > e3hh) yhi = e3hh;

         for (qy = ylo; qy < yhi; qy++)                                          //  comute min/max RGB sums
         for (qx = 0; qx < px+rad; qx++)                                         //  (sum all columns for row)
         {
            ii = qy * e3ww + qx;
            lminR += LminR[ii];
            lminG += LminG[ii];
            lminB += LminB[ii];

            lmaxR += LmaxR[ii];
            lmaxG += LmaxG[ii];
            lmaxB += LmaxB[ii];

            lsum++;
         }
      }
      else                                                                       //  continue row
      {
         qx = px-rad-1;                                                          //  subtract column lost
         if (qx >= 0) {
            for (qy = ylo; qy < yhi; qy++)
            {
               ii = qy * e3ww + qx;
               lminR -= LminR[ii];
               lminG -= LminG[ii];
               lminB -= LminB[ii];

               lmaxR -= LmaxR[ii];
               lmaxG -= LmaxG[ii];
               lmaxB -= LmaxB[ii];

               lsum--;
            }
         }
         qx = px+rad-1;                                                          //  add column gained
         if (qx < e3ww) {
            for (qy = ylo; qy < yhi; qy++)
            {
               ii = qy * e3ww + qx;
               lminR += LminR[ii];
               lminG += LminG[ii];
               lminB += LminB[ii];

               lmaxR += LmaxR[ii];
               lmaxG += LmaxG[ii];
               lmaxB += LmaxB[ii];

               lsum++;
            }
         }
      }

      ii = py * e3ww + px;

      Lmin2R[ii] = lminR / lsum;                                                 //  mean RGB
      Lmin2G[ii] = lminG / lsum;
      Lmin2B[ii] = lminB / lsum;

      Lmax2R[ii] = lmaxR / lsum;
      Lmax2G[ii] = lmaxG / lsum;
      Lmax2B[ii] = lmaxB / lsum;

      Fbusy_done++;
   }
   
   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


//  zonal worker thread 3
//  rescale each pixel RGB value so that the full 0-255 range is used

void * Zretinex_wthread3(void *arg)
{
   using namespace retinex_names;

   int      index = *((int *) arg);

   int      px, py;
   int      ii, dist = 0;
   float    *pix1, *pix3;
   float    L1R, L1G, L1B, L2R, L2G, L2B;
   float    lminR, lminG, lminB, lmaxR, lmaxG, lmaxB;
   float    R1, G1, B1, R3, G3, B3;
   float    range, F1;

   for (py = index; py < e3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < e3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * e3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      if (E8pxm) pix1 = PXMpix(E8pxm,px,py);                                     //  input image from global retinex
      else pix1 = PXMpix(E1pxm,px,py);                                           //  else original input image
      pix3 = PXMpix(E3pxm,px,py);                                                //  output image pixel

      ii = py * e3ww + px;
      
      lminR = LminR[ii];                                                         //  RGB range of pixel zone
      lminG = LminG[ii];
      lminB = LminB[ii];

      lmaxR = LmaxR[ii];
      lmaxG = LmaxG[ii];
      lmaxB = LmaxB[ii];

      L1R = pix1[0];                                                             //  rescale RGB for full range
      if (L1R < lminR) L1R = lminR;
      range = (lmaxR - lminR);
      if (range > 1) L2R = 255.0 * (L1R - lminR) / range;
      else L2R = L1R;

      L1G = pix1[1];
      if (L1G < lminG) L1G = lminG;
      range = (lmaxG - lminG);
      if (range > 1) L2G = 255.0 * (L1G - lminG) / range;
      else L2G = L1G;

      L1B = pix1[2];
      if (L1B < lminB) L1B = lminB;
      range = (lmaxB - lminB);
      if (range > 1) L2B = 255.0 * (L1B - lminB) / range;
      else L2B = L1B;
      
      if (L1R < 1) L1R = 1;
      if (L1G < 1) L1G = 1;
      if (L1B < 1) L1B = 1;
     
      R1 = pix1[0];                                                              //  input RGB values
      G1 = pix1[1];
      B1 = pix1[2];

      R3 = L2R/L1R * R1;                                                         //  output RGB values
      G3 = L2G/L1G * G1;
      B3 = L2B/L1B * B1;
      
      F1 = R3;                                                                   //  prevent overflow
      if (G3 > F1) F1 = G3;
      if (B3 > F1) F1 = B3;
      if (F1 > 255) {
         F1 = 255.0 / F1;
         R3 = F1 * R3;
         G3 = F1 * G3;
         B3 = F1 * B3;
      }

      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


//  retinex blend thread
//  Transfer global retinex color to zonal retinex image.
//  Blend input and output images, attenuate bright pixels.

void * Bretinex_wthread(void *arg)
{
   using namespace retinex_names;

   int      index = *((int *) arg);
   int      px, py;
   int      ii, dist = 0;
   float    *pix1, *pix2, *pix3;
   float    R1, G1, B1, R2, G2, B2, R3, G3, B3;
   float    F1, F2, Lmax;
   
   for (py = index; py < e3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < e3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * e3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input image pixel

      if (E9pxm) pix2 = PXMpix(E9pxm,px,py);                                     //  use zonal retinex image if available
      else if (E8pxm) pix2 = PXMpix(E8pxm,px,py);                                //  else use global retinex image
      else pix2 = PXMpix(E3pxm,px,py);                                           //  else use output image pixel

      pix3 = PXMpix(E3pxm,px,py);                                                //  output image pixel

      R1 = pix1[0];                                                              //  input color - original image
      G1 = pix1[1];
      B1 = pix1[2];

      R2 = pix2[0];                                                              //  input color - retinex image
      G2 = pix2[1];
      B2 = pix2[2];

      if (sa_stat == 3 && dist < sa_blend) {                                     //  select area is active,
         F1 = sa_blendfunc(dist);                                                //    blend changes over sa_blend
         F2 = 1.0 - F1;
         R2 = F1 * R2 + F2 * R1;
         G2 = F1 * G2 + F2 * G1;
         B2 = F1 * B2 + F2 * B1;
      }

      Lmax = R1;                                                                 //  max. RGB input
      if (G1 > Lmax) Lmax = G1;
      if (B1 > Lmax) Lmax = B1;
      
      F1 = blend;                                                                //  0 ... 1  >>  E1 ... E3
      F2 = reducebright;                                                         //  0 ... 1
      F1 = F1 * (1.0 - F2 * Lmax / 255.0);                                       //  reduce F1 for high F2 * Lmax

      R3 = F1 * R2 + (1.0 - F1) * R1;                                            //  output RGB = blended inputs
      G3 = F1 * G2 + (1.0 - F1) * G1;
      B3 = F1 * B2 + (1.0 - F1) * B1;
      
      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   exit_wthread();
   return 0;                                                                     //  not executed, avoid gcc warning
}


/********************************************************************************/

//  mirror an image horizontally or vertically

editfunc    EFmirror;

void m_mirror(GtkWidget *, cchar *menu)
{
   int mirror_dialog_event(zdialog *zd, cchar *event);

   F1_help_topic = "mirror_image";

   EFmirror.menuname = menu;
   EFmirror.menufunc = m_mirror;
   EFmirror.funcname = "mirror";
   EFmirror.Fscript = 1;                                                         //  allow scripting                    17.04
   EFmirror.Frestart = 1;                                                        //  allow restart
   if (! edit_setup(EFmirror)) return;                                           //  setup edit

   zdialog *zd = zdialog_new(ZTX("Mirror Image"),Mwin,Bdone,Bcancel,null);
   EFmirror.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"check","horz","hb1",ZTX("horizontal"),"space=5");
   zdialog_add_widget(zd,"check","vert","hb1",ZTX("vertical"),"space=5");

   zdialog_run(zd,mirror_dialog_event,"save");                                   //  run dialog, parallel

   return;
}


//  dialog event and completion callback function

int mirror_dialog_event(zdialog *zd, cchar *event)
{
   int mirror_horz();
   int mirror_vert();
   
   int      nn;

   if (strmatch(event,"focus")) return 1;
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  cancel

   if (zd->zstat)                                                                //  dialog complete
   {
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      return 1;
   }
   
   if (strmatch(event,"apply")) {                                                //  from script                        17.04
      zdialog_fetch(zd,"horz",nn);
      if (nn) mirror_horz();
      zdialog_fetch(zd,"vert",nn);
      if (nn) mirror_vert();
   }

   if (strmatch(event,"horz")) mirror_horz();
   if (strmatch(event,"vert")) mirror_vert();

   return 1;
}


int mirror_horz()
{
   int      px, py;
   float    *pix3, *pix9;
   int      nc = E3pxm->nc, pcc = nc * sizeof(float);

   E9pxm = PXM_copy(E3pxm);

   for (py = 0; py < E3pxm->hh; py++)
   for (px = 0; px < E3pxm->ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);                                                //  image9 = mirrored image3
      pix9 = PXMpix(E9pxm,E3pxm->ww-1-px,py);
      memcpy(pix9,pix3,pcc);
   }

   PXM_free(E3pxm);                                                              //  image9 >> image3
   E3pxm = E9pxm;
   E9pxm = 0;

   CEF->Fmods++;
   CEF->Fsaved = 0;
   Fpaint2();
   return 0;
}


int mirror_vert()
{
   int      px, py;
   float    *pix3, *pix9;
   int      nc = E3pxm->nc, pcc = nc * sizeof(float);

   E9pxm = PXM_copy(E3pxm);

   for (py = 0; py < E3pxm->hh; py++)
   for (px = 0; px < E3pxm->ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);                                                //  image9 = mirrored image3
      pix9 = PXMpix(E9pxm,px,E3pxm->hh-1-py);
      memcpy(pix9,pix3,pcc);
   }

   PXM_free(E3pxm);                                                              //  image9 >> image3
   E3pxm = E9pxm;
   E9pxm = 0;

   CEF->Fmods++;
   CEF->Fsaved = 0;
   Fpaint2();
   return 0;
}


/********************************************************************************/

//  Pixel paint function - paint individual pixels with the mouse.
//  The mouse circle paints a selected color.

namespace paint_names
{
   int   paint_dialog_event(zdialog* zd, cchar *event);
   void  paint_mousefunc();
   void  paint_dopixels(int px, int py);                                         //  update pixel block
   void  paint_savepixB(int px, int py);                                         //  save pixel block for poss. undo
   void  paint_undolastB();                                                      //  undo last pixel block, free memory
   void  paint_freefirstB();                                                     //  free memory for first pixel block
   void  paint_freeallB();                                                       //  free memory for all pixel blocks

   uint8    RGB[3] = { 255, 0, 0 };                                              //  color to paint
   int      mode;                                                                //  1/2 = paint / erase
   int      Mradius;                                                             //  mouse radius
   int      Fptran = 0;                                                          //  flag, paint over transparent areas
   int      Fdrag = 0;                                                           //  flag, mouse drags image
   int      nc, ac;                                                              //  no. channels, alpha channel
   float    kernel[202][202];                                                    //  Mradius <= 100

   int64    maxmem = 2000 * MEGA;                                                //  max. pixel block memory (2 GB)
   int64    totmem;                                                              //  pixB memory allocated
   int      maxpixB = 10000;                                                     //  max. pixel blocks
   int      totpixB = 0;                                                         //  total pixel blocks
   int      pixBseq = 0;                                                         //  last pixel block sequence no.

   typedef struct {                                                              //  pixel block before edit
      int         seq;                                                           //  block sequence no.
      uint16      px, py;                                                        //  center pixel (radius org.)
      uint16      radius;                                                        //  radius of pixel block
      float       pixel[][4];                                                    //  array of pixel[npix][4]
   }  pixBmem_t;
   
   pixBmem_t   **pixBmem = 0;                                                    //  *pixBmem_t[]

   int   pixBmem_cc = 12;                                                        //  all except pixel array + pad
   int   pcc4 = 4 * sizeof(float);                                               //  pixel cc: RGBA = 4 channels

   editfunc    EFpaint;
}


//  menu function

void m_paint_image(GtkWidget *, cchar *)                                         //  separate paint and clone           17.04
{
   using namespace paint_names;

   cchar    *mess1 = ZTX("shift + left click: pick color from image \n"
                         "left drag: paint color on image \n"                    //  remove click actions               17.04
                         "right drag: restore original image");
   cchar    *dash = "  \xcc\xb6 ";

   F1_help_topic = "paint_image";

   EFpaint.menufunc = m_paint_image;
   EFpaint.funcname = "paint";
   EFpaint.Farea = 2;                                                            //  select area OK
   EFpaint.mousefunc = paint_mousefunc;                                          //  mouse function
   if (! edit_setup(EFpaint)) return;                                            //  setup edit

   /****
             __________________________________________________
            |                 Paint on Image                   |
            |                                                  |
            |  shift + left click: pick color from image       |
            |  left drag: paint color on image                 |
            |  right drag: restore original image              |
            |                                                  |
            |  paint color [######]     [palette]   [HSL]      |
            |                                                  |
            |  brush size      NN ============[]=============  |
            |  opacity center  NN ==================[]=======  |
            |  opacity edge    NN ====[]=====================  |
            |                                                  |
            |  [x] paint transparent areas                     |
            |  (o) paint  (o) erase    [undo last] [undo all]  |
            |  [x] drag image     zoom image [+] [-]           |
            |                                                  |
            |                                 [done] [cancel]  |
            |__________________________________________________|

   ****/

   zdialog *zd = zdialog_new(ZTX("Paint on Image"),Mwin,Bdone,Bcancel,null);
   EFpaint.zd = zd;

   zdialog_add_widget(zd,"label","labm","dialog",mess1,"space=5");

   zdialog_add_widget(zd,"hbox","hbp","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labp","hbp",ZTX("paint color"),"space=5");
   zdialog_add_widget(zd,"colorbutt","colorbutt","hbp","255|0|0");
   zdialog_add_widget(zd,"label","space","hbp",0,"space=10");
   zdialog_add_widget(zd,"button","palette","hbp","palette","space=10");
   zdialog_add_widget(zd,"button","HSL","hbp","HSL");

   zdialog_add_widget(zd,"hbox","hbbru","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vbbru1","hbbru",0,"homog|space=1");
   zdialog_add_widget(zd,"vbox","vbbru2","hbbru",0,"homog|space=1");
   zdialog_add_widget(zd,"vbox","vbbru3","hbbru",0,"homog|expand|space=1");

   zdialog_add_widget(zd,"label","labbr","vbbru1",ZTX("brush size"));
   zdialog_add_widget(zd,"label","laboc","vbbru1",ZTX("opacity center"));
   zdialog_add_widget(zd,"label","laboe","vbbru1",ZTX("opacity edge"));

   zdialog_add_widget(zd,"label","labbrNN","vbbru2","NN");
   zdialog_add_widget(zd,"label","labocNN","vbbru2","NNN");
   zdialog_add_widget(zd,"label","laboeNN","vbbru2","NNN");

   zdialog_add_widget(zd,"hscale","Mradius","vbbru3","1|100|1|20","expand");
   zdialog_add_widget(zd,"hscale","opccent","vbbru3","1|100|1|50","expand");
   zdialog_add_widget(zd,"hscale","opcedge","vbbru3","0|100|1|100","expand");

   zdialog_add_widget(zd,"hbox","hb5","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","paint","hb5",ZTX("paint"),"space=5");
   zdialog_add_widget(zd,"radio","erase","hb5",ZTX("erase"));
   zdialog_add_widget(zd,"check","Fptran","hb5",ZTX("include transparent areas"),"space=8");
   
   zdialog_add_widget(zd,"hbox","hbdo","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","undlast","hbdo",Bundolast,"space=5");
   zdialog_add_widget(zd,"button","undall","hbdo",Bundoall,"space=5");

   zdialog_add_widget(zd,"hbox","hb6","dialog");
   zdialog_add_widget(zd,"check","Fdrag","hb6",ZTX("drag image"),"space=5");
   zdialog_add_widget(zd,"label","space","hb6",0,"space=10");
   zdialog_add_widget(zd,"label","labzoom","hb6",ZTX("zoom image"),"space=3");
   zdialog_add_widget(zd,"button","zoom+","hb6"," + ","space=3");
   zdialog_add_widget(zd,"button","zoom-","hb6",dash,"space=3");

   zdialog_rescale(zd,"Mradius",1,2,100);                                        //  stretch scales at sensitive end    17.04
   zdialog_rescale(zd,"opccent",1,2,100);
   zdialog_rescale(zd,"opcedge",0,1,100);
   
   zdialog_restore_inputs(zd);                                                   //  preload prior user inputs
   
   zdialog_stuff(zd,"Fptran",0);                                                 //  initialize
   zdialog_stuff(zd,"paint",1);
   zdialog_stuff(zd,"erase",0);
   zdialog_stuff(zd,"Fdrag",0);
   mode = 1;
   Fdrag = 0;

   zdialog_run(zd,paint_dialog_event,"save");                                    //  run dialog, parallel
   
   zdialog_send_event(zd,"colorbutt");                                           //  initialize paint color
   zdialog_send_event(zd,"Mradius");                                             //  get kernel initialized
   zdialog_fetch(zd,"Fptran",Fptran);                                            //  paint over transparent areas

   totmem = 0;                                                                   //  memory used
   pixBmem = 0;                                                                  //  pixel block memory
   totpixB = 0;                                                                  //  pixel blocks
   pixBseq = 0;
   
   ac = 0;
   nc = E1pxm->nc;                                                               //  channels, RGBA
   if (nc > 3) ac = 1;                                                           //  alpha channel present

   takeMouse(paint_mousefunc,drawcursor);                                        //  connect mouse function
   return;
}


//  dialog event and completion callback function

int paint_names::paint_dialog_event(zdialog *zd, cchar *event)
{
   using namespace paint_names;

   cchar       *pp;
   char        color[20], text[20];
   float       opccent, opcedge;                                                 //  center and edge opacities
   int         paint, radius, dx, dy, err;
   float       rad, kern;

   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done(0);                                          //  commit edit
      else edit_cancel(0);                                                       //  discard edit
      paint_freeallB();                                                          //  free pixel block memory
      return 1;
   }

   draw_mousecircle(0,0,0,1,0);                                                  //  erase mouse circle

   if (strmatch(event,"focus"))                                                  //  toggle mouse capture
      takeMouse(paint_mousefunc,drawcursor);

   if (strmatch(event,"colorbutt"))
   {
      zdialog_fetch(zd,"colorbutt",color,19);                                    //  get paint color from color wheel
      pp = strField(color,"|",1);
      if (pp) RGB[0] = atoi(pp);
      pp = strField(color,"|",2);
      if (pp) RGB[1] = atoi(pp);
      pp = strField(color,"|",3);
      if (pp) RGB[2] = atoi(pp);
   }
   
   if (strmatch(event,"palette")) {
      err = color_palette(RGB);                                                  //  select color from palette          17.04
      if (err) return 1;
      snprintf(color,20,"%d|%d|%d",RGB[0],RGB[1],RGB[2]);
      zdialog_stuff(zd,"colorbutt",color);
   }

   if (strmatch(event,"HSL")) {
      err = HSL_chooser(RGB);                                                    //  select color from palette          17.08
      if (err) return 1;
      snprintf(color,20,"%d|%d|%d",RGB[0],RGB[1],RGB[2]);
      zdialog_stuff(zd,"colorbutt",color);
   }

   if (strstr("Mradius opccent opcedge",event))
   {
      zdialog_fetch(zd,"Mradius",Mradius);                                       //  get new brush attributes
      zdialog_fetch(zd,"opccent",opccent);
      zdialog_fetch(zd,"opcedge",opcedge);

      sprintf(text,"%3d",Mradius);                                               //  stuff corresp. number values
      zdialog_stuff(zd,"labbrNN",text);
      sprintf(text,"%3.0f",opccent);
      zdialog_stuff(zd,"labocNN",text);
      sprintf(text,"%3.0f",opcedge);
      zdialog_stuff(zd,"laboeNN",text);

      opccent = 0.01 * opccent;                                                  //  opacity  0 ... 1
      opcedge = 0.01 * opcedge;
      opccent = pow(opccent,2.0);                                                //  change response curve
      opcedge = opccent * opcedge;                                               //  edge relative to center

      radius = Mradius;

      for (dy = -radius; dy <= radius; dy++)                                     //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                         //  center ... edge  >>  1 ... 0
         kern = kern * (opccent - opcedge) + opcedge;                            //  opacity  center ... edge
         if (rad > radius) kern = 0;                                             //  beyond radius, within square
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         kernel[dx+radius][dy+radius] = kern;
      }
   }

   if (strmatch(event,"undlast"))                                                //  undo last edit (click or drag)
      paint_undolastB();

   if (strmatch(event,"undall")) {                                               //  undo all edits
      edit_reset();
      paint_freeallB();
   }

   if (strmatch(event,"Fptran"))                                                 //  flag, paint over transparency
      zdialog_fetch(zd,"Fptran",Fptran);
   
   if (strstr("paint erase",event)) {                                            //  set paint or erase mode
      zdialog_fetch(zd,"paint",paint);
      if (paint) mode = 1;
      else mode = 2;
   }
   
   if (strmatch(event,"Fdrag"))                                                  //  mouse drags image
      zdialog_fetch(zd,"Fdrag",Fdrag);
   
   if (strmatch(event,"zoom+")) m_zoom(0,"in");                                  //  zoom image in or out
   if (strmatch(event,"zoom-")) m_zoom(0,"out");

   return 1;
}


//  mouse function

void paint_names::paint_mousefunc()                                              //  no action on clicks                17.04
{
   using namespace paint_names;

   static int  pmxdown = 0, pmydown = 0;
   static int  pmxdrag = 0, pmydrag = 0;
   char        color[20];
   float       *pix3;
   zdialog     *zd = EFpaint.zd;
   
   if (Fdrag) return;                                                            //  pass through to main()

   if (LMclick && KBshiftkey)                                                    //  shift + left mouse click
   {
      LMclick = 0;
      pix3 = PXMpix(E3pxm,Mxclick,Myclick);                                      //  pick new color from image
      RGB[0] = pix3[0];
      RGB[1] = pix3[1];
      RGB[2] = pix3[2];
      snprintf(color,19,"%d|%d|%d",RGB[0],RGB[1],RGB[2]);
      if (zd) zdialog_stuff(zd,"colorbutt",color);
      return;
   }
   
   if (Mxdrag || Mydrag)                                                         //  drag in progress
   {
      if (Mxdown != pmxdown || Mydown != pmydown) {                              //  new drag
         pixBseq++;                                                              //  new undo seq. no.
         pmxdown = Mxdown;
         pmydown = Mydown;
      }

      if (Mxdrag == pmxdrag && Mydrag == pmydrag) {                              //  no movement
         Mxdrag = Mydrag = 0;
         return;
      }

      pmxdrag = Mxdrag;
      pmydrag = Mydrag;

      paint_dopixels(Mxdrag,Mydrag);                                             //  do 1 block of pixels
   }

   draw_mousecircle(Mxposn,Myposn,Mradius,0,0);                                  //  draw mouse circle

   Mxdrag = Mydrag = LMclick = 0;
   return;
}


//  paint or erase 1 block of pixels within mouse radius of px, py

void paint_names::paint_dopixels(int px, int py)
{
   using namespace paint_names;

   float       *pix1, *pix3;
   int         radius, dx, dy, qx, qy;
   int         ii, ww, hh, dist = 0;
   int         pot = ac * Fptran;                                                //  paint over transparent areas
   float       red, green, blue;
   float       kern;

   cairo_t *cr = draw_context_create(gdkwin,draw_context);                       //  17.04

   draw_mousecircle(0,0,0,1,cr);                                                 //  erase mouse circle

   ww = E3pxm->ww;
   hh = E3pxm->hh;

   paint_savepixB(px,py);                                                        //  save pixels for poss. undo

   red = RGB[0];                                                                 //  paint color
   green = RGB[1];
   blue = RGB[2];

   radius = Mradius;

   if (mode == 1) {                                                              //  paint
      CEF->Fmods++;
      CEF->Fsaved = 0;
   }

   for (dy = -radius; dy <= radius; dy++)                                        //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      qx = px + dx;
      qy = py + dy;

      if (qx < 0 || qx > ww-1) continue;
      if (qy < 0 || qy > hh-1) continue;

      if (sa_stat == 3) {                                                        //  select area active
         ii = qy * ww + qx;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel is outside area
      }

      kern = kernel[dx+radius][dy+radius];                                       //  mouse opacities
      if (kern == 0) continue;                                                   //  outside mouse radius

      if (sa_stat == 3 && dist < sa_blend)                                       //  select area edge blend,
         kern = kern * sa_blendfunc(dist);                                       //    reduce opacity

      pix1 = PXMpix(E1pxm,qx,qy);                                                //  source image pixel
      pix3 = PXMpix(E3pxm,qx,qy);                                                //  edited image pixel
      
      if (mode == 1 && Mbutton < 2)                                              //  paint
      {
         kern = kern * (1.0 + 50.0 * wacom_pressure);                            //  wacom                              17.04
         if (kern > 1.0) kern = 1.0;
         pix3[0] = kern * red   + (1.0 - kern) * pix3[0];                        //  overpaints accumulate
         pix3[1] = kern * green + (1.0 - kern) * pix3[1];
         pix3[2] = kern * blue  + (1.0 - kern) * pix3[2];
         if (pot) pix3[3] = kern * 255.0  + (1.0 - kern) * pix3[3];              //  overpaint transparent area
      }

      if (mode == 2 || Mbutton >= 2)                                             //  erase
      {
         kern = kern * (2.0 + 50.0 * wacom_pressure);                            //  wacom                              17.04
         if (kern > 1.0) kern = 1.0;
         if (kern > 1) kern = 1;
         pix3[0] = kern * pix1[0] + (1.0 - kern) * pix3[0];                      //  gradual erase
         pix3[1] = kern * pix1[1] + (1.0 - kern) * pix3[1];
         pix3[2] = kern * pix1[2] + (1.0 - kern) * pix3[2];
         if (pot) pix3[3] = kern * pix1[3] + (1.0 - kern) * pix3[3];
      }
   }

   px = px - radius - 1;                                                         //  repaint modified area
   py = py - radius - 1;
   ww = 2 * radius + 3;
   Fpaint3(px,py,ww,ww,cr);

   draw_context_destroy(draw_context);                                           //  17.04
   return;
}


//  save 1 block of pixels for possible undo

void paint_names::paint_savepixB(int px, int py)
{
   using namespace paint_names;

   int            cc, npix, radius, dx, dy;
   float          *pix3;
   pixBmem_t      *paintsave1;
   
   if (! pixBmem) {                                                              //  first time
      pixBmem = (pixBmem_t **) zmalloc(maxpixB * sizeof(void *));
      totpixB = 0;
      totmem = 0;
   }

   if (totmem > maxmem || totpixB == maxpixB)                                    //  free memory for oldest updates
      while (totmem > 0.7 * maxmem || totpixB > 0.7 * maxpixB)
         paint_freefirstB();                                     

   radius = Mradius;
   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                        //  count pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
      if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
      npix++;
   }

   cc = npix * pcc4 + pixBmem_cc;
   paintsave1 = (pixBmem_t *) zmalloc(cc);                                       //  allocate memory for block
   pixBmem[totpixB] = paintsave1;
   totpixB += 1;
   totmem += cc;

   paintsave1->seq = pixBseq;                                                    //  save pixel block poop
   paintsave1->px = px;
   paintsave1->py = py;
   paintsave1->radius = radius;

   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                        //  save pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
      if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
      pix3 = PXMpix(E3pxm,(px+dx),(py+dy));                                      //  edited image pixel
      paintsave1->pixel[npix][0] = pix3[0];
      paintsave1->pixel[npix][1] = pix3[1];
      paintsave1->pixel[npix][2] = pix3[2];
      if (ac) paintsave1->pixel[npix][3] = pix3[3];
      npix++;
   }

   return;
}


//  undo last pixel block (newest edit) and free memory

void paint_names::paint_undolastB()
{
   using namespace paint_names;

   int            ii, cc, npix, radius;
   int            ww, px, py, dx, dy;
   float          *pix3;
   pixBmem_t      *paintsave1;

   for (ii = totpixB-1; ii >= 0; ii--)
   {
      paintsave1 = pixBmem[ii];
      if (paintsave1->seq != pixBseq) break;
      px = paintsave1->px;
      py = paintsave1->py;
      radius = paintsave1->radius;

      npix = 0;
      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
         if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
         pix3 = PXMpix(E3pxm,(px+dx),(py+dy));
         pix3[0] = paintsave1->pixel[npix][0];
         pix3[1] = paintsave1->pixel[npix][1];
         pix3[2] = paintsave1->pixel[npix][2];
         if (ac) pix3[3] = paintsave1->pixel[npix][3];
         npix++;
      }

      px = px - radius - 1;
      py = py - radius - 1;
      ww = 2 * radius + 3;
      Fpaint3(px,py,ww,ww,0);

      zfree(paintsave1);
      pixBmem[ii] = 0;
      cc = npix * pcc4 + pixBmem_cc;
      totmem -= cc;
      totpixB--;
   }

   if (pixBseq > 0) --pixBseq;
   return;
}


//  free memory for first pixel block (oldest edit)

void paint_names::paint_freefirstB()
{
   using namespace paint_names;

   int            firstseq;
   int            ii, jj, cc, npix, radius;
   int            px, py, dx, dy;
   pixBmem_t      *paintsave1;
   
   if (! totpixB) return;
   firstseq = pixBmem[0]->seq;
   
   for (ii = 0; ii < totpixB; ii++)
   {
      paintsave1 = pixBmem[ii];
      if (paintsave1->seq != firstseq) break;
      px = paintsave1->px;
      py = paintsave1->py;
      radius = paintsave1->radius;
      npix = 0;
      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
         if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
         npix++;
      }

      zfree(paintsave1);
      pixBmem[ii] = 0;
      cc = npix * pcc4 + pixBmem_cc;
      totmem -= cc;
   }
   
   for (jj = 0; ii < totpixB; jj++, ii++)
      pixBmem[jj] = pixBmem[ii];   
   
   totpixB = jj;
   return;
}


//  free all pixel block memory

void paint_names::paint_freeallB()
{
   using namespace paint_names;

   int            ii;
   pixBmem_t      *paintsave1;

   for (ii = totpixB-1; ii >= 0; ii--)
   {
      paintsave1 = pixBmem[ii];
      zfree(paintsave1);
   }

   if (pixBmem) zfree(pixBmem);
   pixBmem = 0;

   pixBseq = 0;
   totpixB = 0;
   totmem = 0;

   return;
}


/********************************************************************************/

//  Select a color from a color palette image file.
//  Returns status = 0 if OK, = 1 if error or user cancel.
//  Returns selected RGB color in rgb[3] argument.

namespace color_palette_names
{
// char        *color_palette_file;                                              //  defined in fotoxx.h
   uint8       RGB[3] = { 255, 255, 255 };                                       //  last color selection
   char        color[20];
   zdialog     *zd;
   PIXBUF      *pixbuf = 0;
   GtkWidget   *frame, *drawarea;
}


int color_palette(uint8 rgb[3])                                                  //  17.04
{
   using namespace color_palette_names;

   int   color_palette_dialog_event(zdialog *zd, cchar *event);                  //  dialog events function
   int   color_palette_draw(GtkWidget *window, cairo_t *cr);                     //  window draw function
   int   color_palette_mouse(GtkWidget *window, GdkEventButton *);               //  window mouse button event

   int         zstat;

   /***
          ____________________________________________
         |             Color Palette                  |
         |                                            |
         |  #####                                     |
         |  #####  click on image to choose color     |
         |  #####                                     |
         |  ________________________________________  |
         | |                                        | |
         | |                                        | |
         | |                                        | |
         | |                  image                 | |
         | |                                        | |
         | |                                        | |
         | |                                        | |
         | |                                        | |
         | |________________________________________| |
         |                                            |
         | [_______________________________] [browse] |
         |                                            |
         |                            [done] [cancel] |
         |____________________________________________|

   ***/

   zd = zdialog_new(ZTX("Color Palette"),Mwin,Bdone,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbcolor","dialog",0,"space=3");
   zdialog_add_widget(zd,"colorbutt","color","hbcolor","255|255|255","space=5");
   zdialog_add_widget(zd,"label","labcolor","hbcolor",ZTX("click on image to choose color"));

   zdialog_add_widget(zd,"frame","frame","dialog");
   frame = zdialog_widget(zd,"frame");
   drawarea = gtk_drawing_area_new();
   gtk_widget_set_size_request(drawarea,-1,200);
   gtk_container_add(GTK_CONTAINER(frame),drawarea);

   zdialog_add_widget(zd,"hbox","hbfile","dialog",0,"space=3");
   zdialog_add_widget(zd,"zentry","file","hbfile",0,"space=3|expand");
   zdialog_add_widget(zd,"button","browse","hbfile",Bbrowse,"space=3");
   zdialog_stuff(zd,"file",color_palette_file);
   
   G_SIGNAL(drawarea,"draw",color_palette_draw,0);
   G_SIGNAL(drawarea,"button-press-event",color_palette_mouse,0);
   gtk_widget_add_events(drawarea,GDK_BUTTON_PRESS_MASK);
   
   snprintf(color,20,"%d|%d|%d",RGB[0],RGB[1],RGB[2]);
   zdialog_stuff(zd,"color",color);

   zdialog_resize(zd,300,0);
   zdialog_run(zd,color_palette_dialog_event,"save");
   zstat = zdialog_wait(zd);

   if (zstat != 1) {                                                             //  [cancel] or [x]
      zdialog_free(zd);
      return 1;
   }
   
   rgb[0] = RGB[0];                                                              //  [done]
   rgb[1] = RGB[1];
   rgb[2] = RGB[2];
   
   zdialog_free(zd);
   return 0;
}


//  dialog event and completion function

int color_palette_dialog_event(zdialog *zd, cchar *event)
{
   using namespace color_palette_names;

   char     *pp;
   
   if (strmatch(event,"browse")) 
   {
      pp = gallery_select1(color_palette_file);                                  //  17.08
      if (! pp) return 1;
      zdialog_stuff(zd,"file",pp);
      if (color_palette_file) zfree(color_palette_file);
      color_palette_file = pp;
      gtk_widget_queue_draw(drawarea);
   }
   
   return 1;
}


//  color palette window draw function

int color_palette_draw(GtkWidget *widget, cairo_t *cr)
{
   using namespace color_palette_names;
   
   PIXBUF      *pixbuf1;
   GError      *gerror = 0;
   GdkWindow   *gdkwin;
   int         ww1, hh1, ww2, hh2;

   if (*color_palette_file != '/') return 1;                                     //  load last color palette file

   pixbuf1 = gdk_pixbuf_new_from_file_at_size(color_palette_file,500,500,&gerror);
   if (! pixbuf1) {
      printz("pixbuf error: %s \n",gerror->message);                             //  popup message >> draw event loop   17.04.2
      return 1;                                                                  //     GTK 3.22.11
   }

   ww1 = gdk_pixbuf_get_width(pixbuf1);                                          //  image dimensions
   hh1 = gdk_pixbuf_get_height(pixbuf1);

   gdkwin = gtk_widget_get_window(widget);                                       //  set drawing area to match
   ww2 = gdk_window_get_width(gdkwin);                                           //    aspect ratio
   hh2 = ww2 * hh1 / ww1;
   gtk_widget_set_size_request(widget,-1,hh2);

   if (pixbuf) g_object_unref(pixbuf);
   pixbuf = gdk_pixbuf_scale_simple(pixbuf1,ww2,hh2,BILINEAR);
   g_object_unref(pixbuf1);
   if (! pixbuf) return 1;

   gdk_cairo_set_source_pixbuf(cr,pixbuf,0,0);                                   //  draw image
   cairo_paint(cr);

   return 1;
}


//  color palette mouse click function

int color_palette_mouse(GtkWidget *widget, GdkEventButton *event)
{
   using namespace color_palette_names;

   int      mx, my, rs, nc;
   uint8    *pixels, *pix1;
   
   if (! pixbuf) return 1;

   rs = gdk_pixbuf_get_rowstride(pixbuf);
   nc = gdk_pixbuf_get_n_channels(pixbuf);
   pixels = gdk_pixbuf_get_pixels(pixbuf);

   mx = event->x;
   my = event->y;
   pix1 = pixels + my * rs + mx * nc;

   RGB[0] = pix1[0];
   RGB[1] = pix1[1];
   RGB[2] = pix1[2];
   
   snprintf(color,20,"%d|%d|%d",RGB[0],RGB[1],RGB[2]);
   zdialog_stuff(zd,"color",color);
   
   return 1;
}


/********************************************************************************/

//  HSL color chooser function
//  Returns status = 0 if OK, = 1 if error or user cancel.
//  Returns selected RGB color in rgb[3] argument.

namespace HSL_chooser_names                                                      //  17.08
{
   zdialog     *HSLzdialog;
   GtkWidget   *RGBframe, *RGBcolor;
   GtkWidget   *Hframe, *Hscale;
   float       H, S, L;                                                          //  chosen HSL color
   float       R, G, B;                                                          //  corresp. RGB color
}

int HSL_chooser(uint8 rgb[3])
{
   using namespace HSL_chooser_names;

   void   HSL_chooser_RGB(GtkWidget *drawarea, cairo_t *cr, int *);
   void   HSL_chooser_Hscale(GtkWidget *drawarea, cairo_t *cr, int *);
   int    HSL_chooser_dialog_event(zdialog *zd, cchar *event);
   void   HSL_chooser_mousefunc();
   
   zdialog     *zd;
   int         zstat;
   
/***
       ________________________________________________
      |                                                |
      |  [#######]   [##############################]  |
      |  Color Hue   ================[]==============  |
      |  Saturation  =====================[]=========  |
      |  Lightness   ===========[]===================  |
      |                                                |
      |                             [select] [cancel]  |
      |________________________________________________|

***/

   zd = zdialog_new("Adjust HSL",Mwin,Bselect,Bcancel,null);
   HSLzdialog = zd;

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb2",0,"homog|space=0");
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"homog|expand|space=0");

   zdialog_add_widget(zd,"frame","RGBframe","vb1",0,"space=1");                  //  drawing area for RGB color
   RGBframe = zdialog_widget(zd,"RGBframe");
   RGBcolor = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(RGBframe),RGBcolor);
   gtk_widget_set_size_request(RGBcolor,0,16);
   G_SIGNAL(RGBcolor,"draw",HSL_chooser_RGB,0);

   zdialog_add_widget(zd,"frame","Hframe","vb2",0,"space=1");                    //  drawing area for hue scale
   Hframe = zdialog_widget(zd,"Hframe");
   Hscale = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(Hframe),Hscale);
   gtk_widget_set_size_request(Hscale,200,16);
   G_SIGNAL(Hscale,"draw",HSL_chooser_Hscale,0);
   
   zdialog_add_widget(zd,"label","labhue","vb1",ZTX("Color Hue"));
   zdialog_add_widget(zd,"label","labsat","vb1",ZTX("Saturation"));
   zdialog_add_widget(zd,"label","lablgt","vb1",ZTX("Lightness"));

   zdialog_add_widget(zd,"hscale","H","vb2","0|359.9|0.1|180","expand");
   zdialog_add_widget(zd,"hscale","S","vb2","0|1|0.001|0.5","expand");
   zdialog_add_widget(zd,"hscale","L","vb2","0|1|0.001|0.5","expand");

   H = 180;                                                                      //  chosen HSL color = not set
   S = 0.5;
   L = 0.5;
   
   zdialog_run(zd,HSL_chooser_dialog_event,"save");                              //  run dialog - parallel
   takeMouse(HSL_chooser_mousefunc,arrowcursor);                                 //  connect mouse function
   zstat = zdialog_wait(zd);                                                     //  wait for dialog complete

   if (zstat == 1) return 1;                                                     //  cancel

   rgb[0] = 255 * R;                                                             //  return chosen RGB color
   rgb[1] = 255 * G;
   rgb[2] = 255 * B;
   return 0;
}


//  Paint RGBcolor drawing area with RGB color from chosen HSL color

void HSL_chooser_RGB(GtkWidget *drawarea, cairo_t *cr, int *)
{
   using namespace HSL_chooser_names;

   int      ww, hh;

   ww = gtk_widget_get_allocated_width(drawarea);                                //  drawing area size
   hh = gtk_widget_get_allocated_height(drawarea);
   
   HSLtoRGB(H,S,L,R,G,B);                                                        //  RGB color from chosen HSL

   cairo_set_source_rgb(cr,R,G,B);
   cairo_rectangle(cr,0,0,ww-1,hh-1);
   cairo_fill(cr);

   return;
}


//  Paint Hscale drawing area with all hue values in a horizontal scale

void HSL_chooser_Hscale(GtkWidget *drawarea, cairo_t *cr, int *)
{
   using namespace HSL_chooser_names;

   int      px, ww, hh;
   float    H, S, L, R, G, B;

   ww = gtk_widget_get_allocated_width(drawarea);                                //  drawing area size
   hh = gtk_widget_get_allocated_height(drawarea);
   
   S = L = 0.5;
   
   for (px = 0; px < ww; px++)                                                   //  paint hue color scale
   {
      H = 360 * px / ww;
      HSLtoRGB(H,S,L,R,G,B);
      cairo_set_source_rgb(cr,R,G,B);
      cairo_move_to(cr,px,0);
      cairo_line_to(cr,px,hh-1);
      cairo_stroke(cr);
   }

   return;
}


//  HSL dialog event and completion function

int HSL_chooser_dialog_event(zdialog *zd, cchar *event)                          //  HSL dialog event function
{
   using namespace HSL_chooser_names;

   void   HSL_chooser_mousefunc();

   if (zd->zstat) {                                                              //  zdialog complete
      zdialog_free(zd);
      freeMouse();                                                               //  18.01
      HSLzdialog = 0;
      return 1;
   }

   if (strstr("H S L",event)) {                                                  //  HSL inputs changed
      zdialog_fetch(zd,"H",H);
      zdialog_fetch(zd,"S",S);
      zdialog_fetch(zd,"L",L);
      gtk_widget_queue_draw(RGBcolor);                                           //  draw corresp. RGB color
   }

   if (strmatch("focus",event)) 
      takeMouse(HSL_chooser_mousefunc,arrowcursor);

   return 1;
}


//  mouse function
//  click on image to set the color

void HSL_chooser_mousefunc()
{
   using namespace HSL_chooser_names;

   float       *pix1;
   float       f256 = 1.0 / 256.0;
   zdialog     *zd = HSLzdialog;
   
   if (! zd) return;
   
   if (LMclick && KBshiftkey)                                                    //  shift + left mouse click
   {
      LMclick = 0;
      pix1 = PXMpix(E1pxm,Mxclick,Myclick);                                      //  pick output color from image
      R = pix1[0] * f256;
      G = pix1[1] * f256;
      B = pix1[2] * f256;
      RGBtoHSL(R,G,B,H,S,L);                                                     //  set corresp. HSL
      zdialog_stuff(zd,"H",H);
      zdialog_stuff(zd,"S",S);
      zdialog_stuff(zd,"L",L);
      gtk_widget_queue_draw(RGBcolor);                                           //  draw current RGB color
   }

   return;
}


/********************************************************************************/

//  Clone Image function.
//  Copy from one image area to another with variable opacity.

namespace clone_names
{
   int   clone_dialog_event(zdialog* zd, cchar *event);
   void  clone_mousefunc();
   void  clone_dopixels(int px, int py);                                         //  update pixel block
   void  clone_savepixB(int px, int py);                                         //  save pixel block for poss. undo
   void  clone_undolastB();                                                      //  undo last pixel block, free memory
   void  clone_freefirstB();                                                     //  free memory for first pixel block
   void  clone_freeallB();                                                       //  free memory for all pixel blocks

   int      mode;                                                                //  1/2 = paint / erase
   int      Mradius;                                                             //  mouse radius
   int      imagex, imagey;                                                      //  source image location
   float    kernel[402][402];                                                    //  radius <= 200
   int      Fptran = 0;                                                          //  flag, paint over transparent areas
   int      nc, ac;                                                              //  no. channels, alpha channel

   int64    maxmem = 2000 * MEGA;                                                //  max. pixel block memory
   int64    totmem;                                                              //  pixB memory allocated
   int      maxpixB = 10000;                                                     //  max. pixel blocks
   int      totpixB = 0;                                                         //  total pixel blocks
   int      pixBseq = 0;                                                         //  last pixel block sequence no.

   typedef struct {                                                              //  pixel block before edit
      int         seq;                                                           //  block sequence no.
      uint16      px, py;                                                        //  center pixel (radius org.)
      uint16      radius;                                                        //  radius of pixel block
      float       pixel[][4];                                                    //  array of pixel[npix][4]
   }  pixBmem_t;
   
   pixBmem_t   **pixBmem = 0;                                                    //  *pixBmem_t[]

   int   pixBmem_cc = 12;                                                        //  all except pixel array + pad
   int   pcc4 = 4 * sizeof(float);                                               //  pixel cc: RGBA = 4 channels

   editfunc    EFclone;
}


//  menu function

void m_clone_image(GtkWidget *, cchar *)                                         //  separate paint and clone           17.04
{
   using namespace clone_names;

   cchar    *mess1 = ZTX("shift + left click: pick image position to copy \n"
                         "left drag: copy image to mouse position \n"
                         "right drag: restore image");

   F1_help_topic = "clone_image";

   EFclone.menufunc = m_clone_image;
   EFclone.funcname = "clone";
   EFclone.Farea = 2;                                                            //  select area OK
   EFclone.mousefunc = clone_mousefunc;                                          //  mouse function
   if (! edit_setup(EFclone)) return;                                            //  setup edit

   /********
             ____________________________________________________
            |                  Clone Image                       |
            |                                                    |
            |  shift + left click: pick image position to copy   |
            |  left click or drag: copy image to mouse position  |
            |  right click or drag: restore image                |
            |                                                    |
            |  brush size      [____]     [undo last]            |
            |  opacity center  [____]     [undo all]             |
            |  opacity edge    [____]                            |
            |                                                    |
            |  [x] paint transparent areas                       |
            |                                                    |
            |                                   [done] [cancel]  |
            |____________________________________________________|

   ********/

   zdialog *zd = zdialog_new(ZTX("Clone Image"),Mwin,Bdone,Bcancel,null);
   EFclone.zd = zd;

   zdialog_add_widget(zd,"hbox","hbr","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labm","dialog",mess1,"space=5");
   zdialog_add_widget(zd,"hbox","hbbri","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vbbr1","hbbri",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbbr2","hbbri",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","space","hbbri",0,"space=10");
   zdialog_add_widget(zd,"vbox","vbbr3","hbbri",0,"space=10");
   zdialog_add_widget(zd,"label","labbr","vbbr1",ZTX("brush size"));
   zdialog_add_widget(zd,"label","labtc","vbbr1",ZTX("opacity center"));
   zdialog_add_widget(zd,"label","labte","vbbr1",ZTX("opacity edge"));
   zdialog_add_widget(zd,"zspin","Mradius","vbbr2","1|200|1|30");
   zdialog_add_widget(zd,"zspin","opccent","vbbr2","1|100|1|10");
   zdialog_add_widget(zd,"zspin","opcedge","vbbr2","0|100|1|0");
   zdialog_add_widget(zd,"button","undlast","vbbr3",Bundolast);
   zdialog_add_widget(zd,"button","undall","vbbr3",Bundoall);
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=3");
   zdialog_add_widget(zd,"check","Fptran","hb4",ZTX("paint over transparent areas"),"space=5");

   zdialog_restore_inputs(zd);                                                   //  preload prior user inputs
   zdialog_run(zd,clone_dialog_event,"save");                                    //  run dialog, parallel

   zdialog_fetch(zd,"Fptran",Fptran);                                            //  paint over transparent areas

   zdialog_send_event(zd,"Mradius");                                             //  get kernel initialized

   totmem = 0;                                                                   //  memory used
   pixBmem = 0;                                                                  //  pixel block memory
   totpixB = 0;                                                                  //  pixel blocks
   pixBseq = 0;
   imagex = imagey = 0;                                                          //  no clone source pixels
   
   ac = 0;
   nc = E1pxm->nc;                                                               //  channels, RGBA
   if (nc > 3) ac = 1;                                                           //  alpha channel present

   takeMouse(clone_mousefunc,drawcursor);                                        //  connect mouse function
   return;
}


//  dialog event and completion callback function

int clone_names::clone_dialog_event(zdialog *zd, cchar *event)
{
   using namespace clone_names;

   int         radius, dx, dy;
   float       rad, kern, opccent, opcedge;

   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done(0);                                          //  commit edit
      else edit_cancel(0);                                                       //  discard edit
      clone_freeallB();                                                          //  free pixel block memory
      return 1;
   }

   draw_mousecircle(0,0,0,1,0);                                                  //  erase mouse circle
   draw_mousecircle2(0,0,0,1,0);                                                 //  erase source tracking circle

   if (strmatch(event,"focus"))                                                  //  toggle mouse capture
      takeMouse(clone_mousefunc,drawcursor);

   if (strstr("Mradius opccent opcedge",event))                                  //  get new brush attributes
   {
      zdialog_fetch(zd,"Mradius",Mradius);
      zdialog_fetch(zd,"opccent",opccent);
      zdialog_fetch(zd,"opcedge",opcedge);

      opccent = 0.01 * opccent;                                                  //  scale 0 ... 1
      opcedge = 0.01 * opcedge;
      opccent = pow(opccent,2);                                                  //  change response curve
      opcedge = opccent * opcedge;                                               //  edge relative to center

      radius = Mradius;

      for (dy = -radius; dy <= radius; dy++)                                     //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                         //  center ... edge  >>  1 ... 0
         kern = kern * (opccent - opcedge) + opcedge;                            //  opacity  center ... edge
         if (rad > radius) kern = 0;                                             //  beyond radius, within square
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         kernel[dx+radius][dy+radius] = kern;
      }
   }

   if (strmatch(event,"undlast"))                                                //  undo last edit (click or drag)
      clone_undolastB();

   if (strmatch(event,"undall")) {                                               //  undo all edits
      edit_reset();
      clone_freeallB();
   }

   if (strmatch(event,"Fptran"))                                                 //  flag, paint over transparency
      zdialog_fetch(zd,"Fptran",Fptran);

   return 1;
}


//  pixel paint mouse function

void clone_names::clone_mousefunc()
{
   using namespace clone_names;

   static int  pmxdown = 0, pmydown = 0;
   int         px, py;

   if (LMclick && KBshiftkey)                                                    //  shift + left mouse click
   {
      imagex = Mxclick;                                                          //  new source image location
      imagey = Myclick;
   }

   else if (LMclick || RMclick)
   {
      if (LMclick) mode = 1;                                                     //  left click, paint
      if (RMclick) mode = 2;                                                     //  right click, erase

      px = Mxdown = Mxclick;                                                     //  bugfix                             17.04
      py = Mydown = Myclick;
 
      pixBseq++;                                                                 //  new undo seq. no.

      clone_dopixels(px,py);                                                     //  do 1 block of pixels
   }

   else if (Mxdrag || Mydrag)                                                    //  drag in progress
   {
      if (Mbutton == 1) mode = 1;                                                //  left drag, paint
      if (Mbutton == 3) mode = 2;                                                //  right drag, erase

      px = Mxdrag;
      py = Mydrag;

      if (Mxdown != pmxdown || Mydown != pmydown) {                              //  new drag
         pixBseq++;                                                              //  new undo seq. no.
         pmxdown = Mxdown;
         pmydown = Mydown;
      }

      clone_dopixels(px,py);                                                     //  do 1 block of pixels
   }

   draw_mousecircle(Mxposn,Myposn,Mradius,0,0);                                  //  draw mouse circle

   if (mode == 1 && (Mxdown || Mydown)) {                                        //  2nd circle tracks source pixels
      px = imagex + Mxposn - Mxdown;
      py = imagey + Myposn - Mydown;
      if (px > 0 && px < E3pxm->ww-1 && py > 0 && py < E3pxm->hh-1)
         draw_mousecircle2(px,py,Mradius,0,0);
   }
   else draw_mousecircle2(0,0,0,1,0);                                            //  no 2nd circle

   LMclick = RMclick = Mxdrag = Mydrag = 0;
   return;
}


//  paint or erase 1 block of pixels within mouse radius of px, py

void clone_names::clone_dopixels(int px, int py)
{
   using namespace clone_names;

   float       *pix1, *pix3, *pix9;
   int         radius, dx, dy, qx, qy, sx, sy;
   int         ii, ww, hh, dist = 0;
   int         pot = ac * Fptran;                                                //  paint over transparent areas
   float       kern;

   if (! imagex && ! imagey) return;                                             //  no source area defined

   cairo_t *cr = draw_context_create(gdkwin,draw_context);                       //  17.04

   draw_mousecircle(0,0,0,1,cr);                                                 //  erase mouse circle
   draw_mousecircle2(0,0,0,1,cr);                                                //  erase source tracking circle

   ww = E3pxm->ww;
   hh = E3pxm->hh;

   clone_savepixB(px,py);                                                        //  save pixels for poss. undo

   radius = Mradius;

   if (mode == 1) {
      CEF->Fmods++;
      CEF->Fsaved = 0;
   }

   for (dy = -radius; dy <= radius; dy++)                                        //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      qx = px + dx;
      qy = py + dy;

      if (qx < 0 || qx > ww-1) continue;
      if (qy < 0 || qy > hh-1) continue;

      if (sa_stat == 3) {                                                        //  select area active
         ii = qy * ww + qx;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel is outside area
      }

      kern = kernel[dx+radius][dy+radius];                                       //  mouse opacities
      if (kern == 0) continue;                                                   //  outside mouse radius

      if (sa_stat == 3 && dist < sa_blend)                                       //  select area edge blend,
         kern = kern * sa_blendfunc(dist);                                       //    reduce opacity

      pix1 = PXMpix(E1pxm,qx,qy);                                                //  source image pixel
      pix3 = PXMpix(E3pxm,qx,qy);                                                //  edited image pixel

      if (mode == 1)                                                             //  paint (clone)
      {
         sx = imagex + qx - Mxdown;                                              //  image location + mouse drag shift
         sy = imagey + qy - Mydown;
         if (sx < 0) sx = 0;
         if (sx > ww-1) sx = ww-1;
         if (sy < 0) sy = 0;
         if (sy > hh-1) sy = hh-1;
         pix9 = PXMpix(E1pxm,sx,sy);                                             //  source image pixel at location

         kern = 0.3 * kern;
         pix3[0] = kern * pix9[0] + (1.0 - kern) * pix3[0];                      //  overpaints accumulate
         pix3[1] = kern * pix9[1] + (1.0 - kern) * pix3[1];
         pix3[2] = kern * pix9[2] + (1.0 - kern) * pix3[2];
         if (pot) pix3[3] = kern * pix9[3] + (1.0 - kern) * pix3[3];             //  overpaint transparent area
      }

      if (mode == 2)                                                             //  erase
      {
         pix3[0] = kern * pix1[0] + (1.0 - kern) * pix3[0];                      //  gradual erase
         pix3[1] = kern * pix1[1] + (1.0 - kern) * pix3[1];
         pix3[2] = kern * pix1[2] + (1.0 - kern) * pix3[2];
         if (pot) pix3[3] = kern * pix1[3] + (1.0 - kern) * pix3[3];
      }
   }

   px = px - radius - 1;                                                         //  repaint modified area
   py = py - radius - 1;
   ww = 2 * radius + 3;
   Fpaint3(px,py,ww,ww,cr);

   draw_context_destroy(draw_context);                                           //  17.04
   return;
}


//  save 1 block of pixels for possible undo

void clone_names::clone_savepixB(int px, int py)
{
   using namespace clone_names;

   int            cc, npix, radius, dx, dy;
   float          *pix3;
   pixBmem_t      *clonesave1;
   
   if (! pixBmem) {                                                              //  first time
      pixBmem = (pixBmem_t **) zmalloc(maxpixB * sizeof(void *));
      totpixB = 0;
      totmem = 0;
   }

   if (totmem > maxmem || totpixB == maxpixB)                                    //  free memory for oldest updates
      while (totmem > 0.7 * maxmem || totpixB > 0.7 * maxpixB)
         clone_freefirstB();                                     

   radius = Mradius;
   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                        //  count pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
      if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
      npix++;
   }

   cc = npix * pcc4 + pixBmem_cc;
   clonesave1 = (pixBmem_t *) zmalloc(cc);                                       //  allocate memory for block
   pixBmem[totpixB] = clonesave1;
   totpixB += 1;
   totmem += cc;

   clonesave1->seq = pixBseq;                                                    //  save pixel block poop
   clonesave1->px = px;
   clonesave1->py = py;
   clonesave1->radius = radius;

   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                        //  save pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
      if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
      pix3 = PXMpix(E3pxm,(px+dx),(py+dy));                                      //  edited image pixel
      clonesave1->pixel[npix][0] = pix3[0];
      clonesave1->pixel[npix][1] = pix3[1];
      clonesave1->pixel[npix][2] = pix3[2];
      if (ac) clonesave1->pixel[npix][3] = pix3[3];
      npix++;
   }

   return;
}


//  undo last pixel block (newest edit) and free memory

void clone_names::clone_undolastB()
{
   using namespace clone_names;

   int            ii, cc, npix, radius;
   int            ww, px, py, dx, dy;
   float          *pix3;
   pixBmem_t      *clonesave1;

   for (ii = totpixB-1; ii >= 0; ii--)
   {
      clonesave1 = pixBmem[ii];
      if (clonesave1->seq != pixBseq) break;
      px = clonesave1->px;
      py = clonesave1->py;
      radius = clonesave1->radius;

      npix = 0;
      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
         if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
         pix3 = PXMpix(E3pxm,(px+dx),(py+dy));
         pix3[0] = clonesave1->pixel[npix][0];
         pix3[1] = clonesave1->pixel[npix][1];
         pix3[2] = clonesave1->pixel[npix][2];
         if (ac) pix3[3] = clonesave1->pixel[npix][3];
         npix++;
      }

      px = px - radius - 1;
      py = py - radius - 1;
      ww = 2 * radius + 3;
      Fpaint3(px,py,ww,ww,0);

      zfree(clonesave1);
      pixBmem[ii] = 0;
      cc = npix * pcc4 + pixBmem_cc;
      totmem -= cc;
      totpixB--;
   }

   if (pixBseq > 0) --pixBseq;
   return;
}


//  free memory for first pixel (oldest edit)

void clone_names::clone_freefirstB() 
{
   using namespace clone_names;

   int            firstseq;
   int            ii, jj, cc, npix, radius;
   int            px, py, dx, dy;
   pixBmem_t      *clonesave1;
   
   if (! totpixB) return;
   firstseq = pixBmem[0]->seq;
   
   for (ii = 0; ii < totpixB; ii++)
   {
      clonesave1 = pixBmem[ii];
      if (clonesave1->seq != firstseq) break;
      px = clonesave1->px;
      py = clonesave1->py;
      radius = clonesave1->radius;
      npix = 0;
      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3pxm->ww-1) continue;
         if (py + dy < 0 || py + dy > E3pxm->hh-1) continue;
         npix++;
      }

      zfree(clonesave1);
      pixBmem[ii] = 0;
      cc = npix * pcc4 + pixBmem_cc;
      totmem -= cc;
   }
   
   for (jj = 0; ii < totpixB; jj++, ii++)
      pixBmem[jj] = pixBmem[ii];   
   
   totpixB = jj;
   return;
}


//  free all pixel block memory

void clone_names::clone_freeallB()
{
   using namespace clone_names;

   int            ii;
   pixBmem_t      *clonesave1;

   for (ii = totpixB-1; ii >= 0; ii--)
   {
      clonesave1 = pixBmem[ii];
      zfree(clonesave1);
   }

   if (pixBmem) zfree(pixBmem);
   pixBmem = 0;

   pixBseq = 0;
   totpixB = 0;
   totmem = 0;

   return;
}


/********************************************************************************/

//  Blend Image function - blend image in areas painted with the mouse

namespace bi_names
{
   int    bi_dialog_event(zdialog* zd, cchar *event);
   void   bi_mousefunc();
   void * bi_thread(void *);
   void * bi_wthread(void *);

   int      mode = 1;                                                            //  1/2 = blend/restore
   int      Mradius = 20;                                                        //  mouse radius
   float    kernel[400][400];                                                    //  radius <= 199
   int      mousex, mousey;                                                      //  mouse click/drag position

   editfunc    EFblendimage;
}


//  menu function

void m_blend_image(GtkWidget *, cchar *)                                         //  17.01
{
   using namespace bi_names;

   cchar    *mess1 = ZTX("left drag: blend image \n"
                         "right drag: restore image");

   F1_help_topic = "blend_image";

   EFblendimage.menufunc = m_blend_image;
   EFblendimage.funcname = "blend_image";
   EFblendimage.Farea = 2;                                                       //  select area OK
   EFblendimage.mousefunc = bi_mousefunc;                                        //  mouse function
   EFblendimage.threadfunc = bi_thread;                                          //  thread function
   if (! edit_setup(EFblendimage)) return;                                       //  setup edit

   /***
             ____________________________________
            |          Blend Image               |
            |                                    |
            |  left drag: blend image            |
            |  right drag: restore image         |
            |                                    |
            |  paint radius       [____]         |
            |  strength center    [____]         |
            |  strength edge      [____]         |
            |                                    |
            |                   [done] [cancel]  |
            |____________________________________|

   ***/

   zdialog *zd = zdialog_new(ZTX("Blend Image"),Mwin,Bdone,Bcancel,null);
   EFblendimage.zd = zd;

   zdialog_add_widget(zd,"label","labm","dialog",mess1,"space=5");
   zdialog_add_widget(zd,"hbox","hbbr","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vbbr1","hbbr",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbbr2","hbbr",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labbr","vbbr1",Bpaintradius);
   zdialog_add_widget(zd,"label","labsc","vbbr1",ZTX("strength center"));
   zdialog_add_widget(zd,"label","labse","vbbr1",ZTX("strength edge"));
   zdialog_add_widget(zd,"zspin","radius","vbbr2","2|199|1|20");
   zdialog_add_widget(zd,"zspin","stcent","vbbr2","0|100|1|50");
   zdialog_add_widget(zd,"zspin","stedge","vbbr2","0|100|1|10");
   
   zdialog_restore_inputs(zd);                                                   //  preload prior user inputs

   zdialog_run(zd,bi_dialog_event,"save");                                       //  run dialog, parallel
   zdialog_send_event(zd,"radius");                                              //  get kernel initialized
   mode = 1;                                                                     //  start with paint mode
   takeMouse(bi_mousefunc,drawcursor);                                           //  connect mouse function
   return;
}


//  dialog event and completion callback function

int bi_names::bi_dialog_event(zdialog *zd, cchar *event)
{
   using namespace bi_names;

   int         radius, dx, dy;
   float       rad, kern, stcent, stedge;

   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done(0);                                          //  commit edit
      else edit_cancel(0);                                                       //  discard edit
      return 1;
   }

   if (strmatch(event,"focus"))                                                  //  toggle mouse capture
      takeMouse(bi_mousefunc,drawcursor);

   if (strstr("radius stcent stedge",event))                                     //  get new brush attributes
   {
      zdialog_fetch(zd,"radius",Mradius);                                        //  mouse radius
      zdialog_fetch(zd,"stcent",stcent);                                         //  center transparency
      zdialog_fetch(zd,"stedge",stedge);                                         //  edge transparency

      stcent = 0.01 * stcent;                                                    //  scale 0 ... 1
      stedge = 0.01 * stedge;

      radius = Mradius;

      for (dy = -radius; dy <= radius; dy++)                                     //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                         //  center ... edge  >>  1 ... 0
         kern = kern * (stcent - stedge) + stedge;                               //  strength  center ... edge
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         if (rad > radius) kern = 2;                                             //  beyond radius, within square
         kernel[dx+radius][dy+radius] = kern;
      }
   }

   return 1;
}


//  blend image mouse function

void bi_names::bi_mousefunc()
{
   using namespace bi_names;
   
   int      px, py, ww;

   if (LMclick || RMclick)
   {
      if (LMclick) mode = 1;                                                     //  left click, paint
      if (RMclick) mode = 2;                                                     //  right click, erase
      mousex = Mxclick;
      mousey = Myclick;
      signal_thread();
   }

   else if (Mxdrag || Mydrag)                                                    //  drag in progress
   {
      if (Mbutton == 1) mode = 1;                                                //  left drag, paint
      if (Mbutton == 3) mode = 2;                                                //  right drag, erase
      mousex = Mxdrag;
      mousey = Mydrag;
      signal_thread();
   }

   cairo_t *cr = draw_context_create(gdkwin,draw_context);                       //  17.04

   px = mousex - Mradius - 1;                                                    //  repaint modified area
   py = mousey - Mradius - 1;
   ww = 2 * Mradius + 3;
   Fpaint3_thread(px,py,ww,ww);                                                  //  17.01

   draw_mousecircle(Mxposn,Myposn,Mradius,0,cr);                                 //  draw mouse circle

   draw_context_destroy(draw_context);                                           //  17.04

   LMclick = RMclick = Mxdrag = Mydrag = 0;
   return;
}


//  blend image thread fuction

void * bi_names::bi_thread(void *)
{
   using namespace bi_names;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      for (int ii = 0; ii < NWT; ii++)                                           //  start worker threads
         start_wthread(bi_wthread,&Nval[ii]);
      wait_wthreads();                                                           //  wait for completion

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;                                                           //  not saved
   }

   return 0;                                                                     //  not executed, stop warning
}


void * bi_names::bi_wthread(void *arg)                                           //  worker thread function
{
   using namespace bi_names;
   
   int         index = *((int *) arg);
   float       *pix1, *pix3, *pixm;
   int         radius, radius2, npix;
   int         px, py, dx, dy, qx, qy, rx, ry, sx, sy;
   int         ii, ww, hh, dist = 0;
   float       kern, kern2, meanR, meanG, meanB;

   ww = E3pxm->ww;
   hh = E3pxm->hh;
   px = mousex;
   py = mousey;
   radius = Mradius;

   for (dy = -radius+index; dy <= radius; dy += NWT)                             //  loop within mouse radius
   for (dx = -radius; dx <= radius; dx++)
   {
      qx = px + dx;
      qy = py + dy;

      if (qx < 0 || qx > ww-1) continue;                                         //  off image
      if (qy < 0 || qy > hh-1) continue;

      if (sa_stat == 3) {                                                        //  select area active
         ii = qy * ww + qx;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel is outside area
      }

      kern = kernel[dx+radius][dy+radius];                                       //  mouse transparencies
      if (kern > 1) continue;                                                    //  outside mouse radius

      if (sa_stat == 3 && dist < sa_blend)                                       //  within blend distance
         kern = kern * sa_blendfunc(dist);
         
      pix1 = PXMpix(E1pxm,qx,qy);                                                //  original pixel
      pix3 = PXMpix(E3pxm,qx,qy);                                                //  edited pixel
      
      meanR = meanG = meanB = npix = 0;
      radius2 = sqrtf(radius);                                                   //  radius = 2..99  >>  radius2 = 1..9
      
      for (ry = -radius2; ry <= radius2; ry++)
      for (rx = -radius2; rx <= radius2; rx++)
      {
         sx = qx + rx;
         sy = qy + ry;
         
         if (px - sx < -radius || px - sx > radius) continue;                    //  outside mouse radius
         if (py - sy < -radius || py - sy > radius) continue;

         if (sx < 0 || sx > ww-1) continue;                                      //  off image
         if (sy < 0 || sy > hh-1) continue;
         
         pixm = PXMpix(E3pxm,sx,sy);
         meanR += pixm[0];         
         meanG += pixm[1];         
         meanB += pixm[2];         
         npix++;
      }
      
      if (npix == 0) continue;
      
      meanR = meanR / npix;
      meanG = meanG / npix;
      meanB = meanB / npix;
      
      if (mode == 1) {                                                           //  blend
         kern2 = 0.5 * kern;
         pix3[0] = kern2 * meanR + (1.0 - kern2) * pix3[0];                      //  pix3 tends to regional mean
         pix3[1] = kern2 * meanG + (1.0 - kern2) * pix3[1];
         pix3[2] = kern2 * meanB + (1.0 - kern2) * pix3[2];
      }

      if (mode == 2) {                                                           //  restore
         kern2 = 0.1 * kern;
         pix3[0] = kern2 * pix1[0] + (1.0 - kern2) * pix3[0];                    //  pix3 tends to pix1
         pix3[1] = kern2 * pix1[1] + (1.0 - kern2) * pix3[1];
         pix3[2] = kern2 * pix1[2] + (1.0 - kern2) * pix3[2];
      }
   }

   exit_wthread();
   return 0;
}


/********************************************************************************/

//  add text on top of the image

namespace addtext
{
   #define Bversion ZTX("+Version")

   textattr_t  attr;                                                             //  text attributes and image

   char     file[1000] = "";                                                     //  file for write_text data
   char     metakey[60] = "";
   int      px, py;                                                              //  text position on image
   int      textpresent;                                                         //  flag, text present on image

   int   dialog_event(zdialog *zd, cchar *event);                                //  dialog event function
   void  mousefunc();                                                            //  mouse event function
   void  write(int mode);                                                        //  write text on image

   editfunc    EFaddtext;
}


//  menu function

void m_add_text(GtkWidget *, cchar *menu)
{
   using namespace addtext;

   cchar    *title = ZTX("Add Text to Image");
   cchar    *tip = ZTX("Enter text, click/drag on image, right click to remove");

   F1_help_topic = "add_text";                                                   //  user guide topic

   EFaddtext.menufunc = m_add_text;
   EFaddtext.funcname = "add_text";
   EFaddtext.Farea = 1;                                                          //  select area ignored
   EFaddtext.Frestart = 1;                                                       //  allow restart
   EFaddtext.mousefunc = mousefunc;                                              //  18.01
   if (! edit_setup(EFaddtext)) return;                                          //  setup edit

/***
       ____________________________________________________________________
      |                     Add Text to Image                              |
      |                                                                    |
      |  Enter text, click/drag on image, right click to remove.           |
      |                                                                    |
      |  Use settings file  [Open] [Save]                                  |     Bopen Bsave
      |  Text [____________________________________________________]       |     text
      |  Use metadata key [________________________________] [Fetch]       |     metakey Bfetch
      |  [Font] [FreeSans_________]  Size [ 44|v]                          |     Bfont fontname fontsize
      |                                                                    |
      |            color   transp.   width     angle                       |
      |  text     [#####] [_______]           [______]                     |     fgcolor fgtransp fgangle
      |  backing  [#####] [_______]                                        |     bgcolor bgtransp
      |  outline  [#####] [_______] [_______]                              |     tocolor totransp towidth
      |  shadow   [#####] [_______] [_______] [______]                     |     shcolor shtransp shwidth shangle
      |                                                                    |
      |        [Clear] [Replace] [+Version] [Next] [Apply] [Done] [Cancel] |
      |____________________________________________________________________|

      [Clear]     clear text and metadata key
      [Replace]   edit_done(), replace current file, restart dialog
      [+Version]  edit_done(), create new file version, restart dialog
      [Next]      edit_done(), replace current file, move to next file, restart dialog, write same text
      [Apply]     edit_done, restart dialog
      [Done]      edit_done()
      [Cancel]    edit_cancel()

***/

   zdialog *zd = zdialog_new(title,Mwin,Bclear,Breplace,Bversion,Bnext,Bapply,Bdone,Bcancel,null);
   EFaddtext.zd = zd;
   EFaddtext.mousefunc = mousefunc;
   EFaddtext.menufunc = m_add_text;                                              //  allow restart

   zdialog_add_widget(zd,"label","tip","dialog",tip,"space=5");

   zdialog_add_widget(zd,"hbox","hbfile","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labfile","hbfile",ZTX("Use settings file"),"space=3");
   zdialog_add_widget(zd,"button",Bopen,"hbfile",Bopen);
   zdialog_add_widget(zd,"button",Bsave,"hbfile",Bsave);

   zdialog_add_widget(zd,"hbox","hbtext","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labtext","hbtext",ZTX("Text"),"space=5");
   zdialog_add_widget(zd,"frame","frtext","hbtext",0,"expand");
   zdialog_add_widget(zd,"edit","text","frtext","text","expand|wrap");
   
   zdialog_add_widget(zd,"hbox","hbmeta","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labmeta","hbmeta",ZTX("Use metadata key"),"space=5");
   zdialog_add_widget(zd,"zentry","metakey","hbmeta",0,"space=2|expand");
   zdialog_add_widget(zd,"button",Bfetch,"hbmeta",Bfetch);

   zdialog_add_widget(zd,"hbox","hbfont","dialog",0,"space=2");
   zdialog_add_widget(zd,"button",Bfont,"hbfont",Bfont);
   zdialog_add_widget(zd,"zentry","fontname","hbfont","FreeSans","space=2|size=20");
   zdialog_add_widget(zd,"label","space","hbfont",0,"space=10");
   zdialog_add_widget(zd,"label","labfsize","hbfont",Bsize);
   zdialog_add_widget(zd,"zspin","fontsize","hbfont","8|500|1|40","space=3");

   zdialog_add_widget(zd,"hbox","hbattr","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbattr1","hbattr",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbattr2","hbattr",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbattr3","hbattr",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbattr4","hbattr",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbattr5","hbattr",0,"homog|space=2");

   zdialog_add_widget(zd,"label","space","vbattr1");
   zdialog_add_widget(zd,"label","labtext","vbattr1",ZTX("text"));
   zdialog_add_widget(zd,"label","labback","vbattr1",ZTX("backing"));
   zdialog_add_widget(zd,"label","laboutln","vbattr1",ZTX("outline"));
   zdialog_add_widget(zd,"label","labshadow","vbattr1",ZTX("shadow"));

   zdialog_add_widget(zd,"label","labcol","vbattr2",Bcolor);
   zdialog_add_widget(zd,"colorbutt","fgcolor","vbattr2","0|0|0");
   zdialog_add_widget(zd,"colorbutt","bgcolor","vbattr2","255|255|255");
   zdialog_add_widget(zd,"colorbutt","tocolor","vbattr2","255|0|0");
   zdialog_add_widget(zd,"colorbutt","shcolor","vbattr2","255|0|0");

   zdialog_add_widget(zd,"label","labtran","vbattr3","transp.");
   zdialog_add_widget(zd,"zspin","fgtransp","vbattr3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","bgtransp","vbattr3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","totransp","vbattr3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","shtransp","vbattr3","0|100|1|0");

   zdialog_add_widget(zd,"label","labw","vbattr4",Bwidth);
   zdialog_add_widget(zd,"label","space","vbattr4");
   zdialog_add_widget(zd,"label","space","vbattr4");
   zdialog_add_widget(zd,"zspin","towidth","vbattr4","0|30|1|0");
   zdialog_add_widget(zd,"zspin","shwidth","vbattr4","0|50|1|0");

   zdialog_add_widget(zd,"label","labw","vbattr5",Bangle);
   zdialog_add_widget(zd,"zspin","fgangle","vbattr5","-180|180|0.5|0");
   zdialog_add_widget(zd,"label","space","vbattr5");
   zdialog_add_widget(zd,"label","space","vbattr5");
   zdialog_add_widget(zd,"zspin","shangle","vbattr5","-180|180|1|0");

   zdialog_add_ttip(zd,Breplace,ZTX("save to current file"));
   zdialog_add_ttip(zd,Bversion,ZTX("save as new file version"));
   zdialog_add_ttip(zd,Bnext,ZTX("save to current file \n"
                                 "open next file with same text"));

   zdialog_restore_inputs(zd);                                                   //  restore prior inputs

   memset(&attr,0,sizeof(attr));

   zdialog_fetch(zd,"text",attr.text,1000);                                      //  get defaults or prior inputs
   zdialog_fetch(zd,"fontname",attr.font,80);
   zdialog_fetch(zd,"fontsize",attr.size);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"fgangle",attr.angle);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);
   zdialog_fetch(zd,"metakey",metakey,60);

   gentext(&attr);                                                               //  initial text

   takeMouse(mousefunc,dragcursor);                                              //  connect mouse function
   textpresent = 0;                                                              //  no text on image yet
   zdialog_run(zd,dialog_event,"save");                                          //  run dialog, parallel
   if (*metakey) zdialog_send_event(zd,Bfetch);                                  //  metadata key active, get text
   return;
}


//  dialog event and completion callback function

int addtext::dialog_event(zdialog *zd, cchar *event)
{
   using namespace addtext;

   GtkWidget   *font_dialog;
   char        font[100];                                                        //  font name and size
   int         size, err;
   char        *newfilename, *pp;
   cchar       *keyname[1];
   char        *keyvals[1];

   if (strmatch(event,"done")) zd->zstat = 6;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 7;                                  //  cancel

   if (zd->zstat)
   {
      if (zd->zstat < 0 || zd->zstat > 7) zd->zstat = 7;                         //  cancel

      if (zd->zstat == 1) {                                                      //  clear all inputs
         *attr.text = 0;
         *metakey = 0;
         zdialog_stuff(zd,"text","");
         zdialog_stuff(zd,"metakey","");
         zd->zstat = 0;                                                          //  keep dialog active
      }
   
      if (zd->zstat == 2) {                                                      //  replace current file
         if (textpresent) edit_done(0);                                          //  finish
         else edit_cancel(0);
         f_save(curr_file,curr_file_type,curr_file_bpc);                         //  replace curr. file
         curr_file_size = f_save_size;
         m_add_text(0,0);                                                        //  start again
         return 1;
      }

      if (zd->zstat == 3) {                                                      //  make a new file version
         if (textpresent) edit_done(0);                                          //  finish
         else edit_cancel(0);
         newfilename = file_new_version(curr_file);                              //  get next avail. file version name
         if (! newfilename) return 1;
         err = f_save(newfilename,curr_file_type,curr_file_bpc);                 //  save file
         if (! err) f_open_saved();                                              //  open saved file with edit hist
         zfree(newfilename);
         m_add_text(0,0);                                                        //  start again
         return 1;
      }

      if (zd->zstat == 4) {                                                      //  finish and go to next image file
         if (textpresent) edit_done(0);                                          //  save mods
         else edit_cancel(0);
         f_save(curr_file,curr_file_type,curr_file_bpc);                         //  replace curr. file
         curr_file_size = f_save_size;
         m_next(0,0);                                                            //  open next file
         m_add_text(0,0);                                                        //  start again
         write(1);                                                               //  put same text etc. onto image
         return 1;
      }
      
      if (zd->zstat == 5) {                                                      //  apply
         if (! textpresent) return 1;
         edit_done(0);                                                           //  save mods
         m_add_text(0,0);                                                        //  restart
         return 1;
      }

      if (zd->zstat == 6) {                                                      //  done
         if (textpresent) edit_done(0);                                          //  save mods
         else edit_cancel(0);
         return 1;
      }

      if (zd->zstat == 7) {
         edit_cancel(0);                                                         //  cancel or [x]
         return 1;
      }
   }

   if (strmatch(event,"focus")) {                                                //  toggle mouse capture
      takeMouse(mousefunc,dragcursor);                                           //  connect mouse function
      return 1;
   }

   if (strmatch(event,Bopen))                                                    //  load zdialog fields from a file
   {
      load_text(zd);
      zdialog_fetch(zd,"text",attr.text,1000);                                   //  get all zdialog fields
      zdialog_fetch(zd,"fontname",attr.font,80);
      zdialog_fetch(zd,"fontsize",attr.size);
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);
      zdialog_fetch(zd,"fgangle",attr.angle);
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);
      zdialog_fetch(zd,"tocolor",attr.color[2],20);
      zdialog_fetch(zd,"totransp",attr.transp[2]);
      zdialog_fetch(zd,"towidth",attr.towidth);
      zdialog_fetch(zd,"shcolor",attr.color[3],20);
      zdialog_fetch(zd,"shtransp",attr.transp[3]);
      zdialog_fetch(zd,"shwidth",attr.shwidth);
      zdialog_fetch(zd,"shangle",attr.shangle);
   }

   if (strmatch(event,Bsave)) {                                                  //  save zdialog fields to file
      save_text(zd);
      return 1;
   }
   
   if (strmatch(event,Bfetch)) {                                                 //  load text from metadata keyname
      zdialog_fetch(zd,"metakey",metakey,60);
      if (*metakey < ' ') return 1;
      keyname[0] = metakey;
      exif_get(curr_file,keyname,keyvals,1);
      if (! keyvals[0]) return 1;
      if (strlen(keyvals[0]) > 999) keyvals[0][999] = 0;
      repl_1str(keyvals[0],attr.text,"\\n","\n");                                //  replace "\n" with newlines
      zfree(keyvals[0]);
      zdialog_stuff(zd,"text",attr.text);                                        //  stuff dialog with metadata
   }

   if (strmatch(event,"text"))                                                   //  get text from dialog
      zdialog_fetch(zd,"text",attr.text,1000);

   if (strmatch(event,Bfont)) {                                                  //  select new font
      snprintf(font,100,"%s %d",attr.font,attr.size);
      font_dialog = gtk_font_chooser_dialog_new(ZTX("select font"),MWIN);
      gtk_font_chooser_set_font(GTK_FONT_CHOOSER(font_dialog),font);
      gtk_dialog_run(GTK_DIALOG(font_dialog));
      pp = gtk_font_chooser_get_font(GTK_FONT_CHOOSER(font_dialog));
      gtk_widget_destroy(font_dialog);

      if (pp) {                                                                  //  should have "fontname nn"
         strncpy0(font,pp,100);
         g_free(pp);
         pp = font + strlen(font);
         while (*pp != ' ') pp--;
         if (pp > font) {
            size = atoi(pp);
            if (size < 8) size = 8;
            zdialog_stuff(zd,"fontsize",size);
            attr.size = size;
            *pp = 0;
            strncpy0(attr.font,font,80);                                         //  get fontname = new font name
            zdialog_stuff(zd,"fontname",font);
         }
      }
   }

   if (strmatch(event,"fontsize"))                                               //  new font size
      zdialog_fetch(zd,"fontsize",attr.size);

   if (strmatch(event,"fgangle"))
      zdialog_fetch(zd,"fgangle",attr.angle);

   if (strmatch(event,"fgcolor"))                                                //  foreground (text) color
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);

   if (strmatch(event,"bgcolor"))                                                //  background color
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);

   if (strmatch(event,"tocolor"))                                                //  text outline color
      zdialog_fetch(zd,"tocolor",attr.color[2],20);

   if (strmatch(event,"shcolor"))                                                //  text shadow color
      zdialog_fetch(zd,"shcolor",attr.color[3],20);

   if (strmatch(event,"fgtransp"))                                               //  foreground transparency
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);

   if (strmatch(event,"bgtransp"))                                               //  background transparency
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);

   if (strmatch(event,"totransp"))                                               //  text outline transparency
      zdialog_fetch(zd,"totransp",attr.transp[2]);

   if (strmatch(event,"shtransp"))                                               //  text shadow transparency
      zdialog_fetch(zd,"shtransp",attr.transp[3]);

   if (strmatch(event,"towidth"))                                                //  text outline width
      zdialog_fetch(zd,"towidth",attr.towidth);

   if (strmatch(event,"shwidth"))                                                //  text shadow width
      zdialog_fetch(zd,"shwidth",attr.shwidth);

   if (strmatch(event,"shangle"))                                                //  text shadow angle
      zdialog_fetch(zd,"shangle",attr.shangle);

   gentext(&attr);                                                               //  build text image from text and attributes

   if (textpresent) write(1);                                                    //  update text on image
   return 1;
}


//  mouse function, set position for text on image

void addtext::mousefunc()
{
   using namespace addtext;

   if (LMclick) {                                                                //  left mouse click
      px = Mxclick;                                                              //  new text position on image
      py = Myclick;
      write(1);                                                                  //  erase old, write new text on image
   }

   if (RMclick) {                                                                //  right mouse click
      write(2);                                                                  //  erase old text, if any
      px = py = -1;
   }

   if (Mxdrag || Mydrag)                                                         //  mouse dragged
   {
      px = Mxdrag;                                                               //  new text position on image
      py = Mydrag;
      write(1);                                                                  //  erase old, write new text on image
   }

   LMclick = RMclick = Mxdrag = Mydrag = 0;
   return;
}


//  write text on image at designated location
//  mode: 1  erase old and write to new position
//        2  erase old and write nothing

void addtext::write(int mode)
{
   using namespace addtext;

   float       *pix1, *pix3;
   uint8       *pixT;
   int         px1, py1, px3, py3, done;
   float       e3part, Ot, Om, Ob;
   static int  orgx1, orgy1, ww1, hh1;                                           //  old text image overlap rectangle
   int         orgx2, orgy2, ww2, hh2;                                           //  new overlap rectangle
   int         nc = E1pxm->nc, pcc = nc * sizeof(float);

   cairo_t *cr = draw_context_create(gdkwin,draw_context);
   
   if (textpresent)
   {
      for (py3 = orgy1; py3 < orgy1 + hh1; py3++)                                //  erase prior text image
      for (px3 = orgx1; px3 < orgx1 + ww1; px3++)                                //  replace E3 pixels with E1 pixels
      {                                                                          //    in prior overlap rectangle
         if (px3 < 0 || px3 >= E3pxm->ww) continue;
         if (py3 < 0 || py3 >= E3pxm->hh) continue;
         pix1 = PXMpix(E1pxm,px3,py3);
         pix3 = PXMpix(E3pxm,px3,py3);
         memcpy(pix3,pix1,pcc);
      }
   }

   done = 0;
   if (mode == 2) done = 1;                                                      //  erase only
   if (! *attr.text) done = 2;                                                   //  no text defined
   if (px < 0 && py < 0) done = 3;                                               //  no position defined

   if (done) {
      if (textpresent) {
         Fpaint3(orgx1,orgy1,ww1,hh1,cr);                                        //  update window to erase old text
         textpresent = 0;                                                        //  mark no text present
         CEF->Fmods--;
      }

      draw_context_destroy(draw_context); 
      return;
   }
   
   ww2 = attr.pxb_text->ww;                                                      //  text image size
   hh2 = attr.pxb_text->hh;

   if (px > E3pxm->ww) px = E3pxm->ww;                                           //  if off screen, pull back in sight
   if (py > E3pxm->hh) py = E3pxm->hh;

   orgx2 = px - ww2/2;                                                           //  copy-to image3 location
   orgy2 = py - hh2/2;

   for (py1 = 0; py1 < hh2; py1++)                                               //  loop all pixels in text image
   for (px1 = 0; px1 < ww2; px1++)
   {
      px3 = orgx2 + px1;                                                         //  copy-to image3 pixel
      py3 = orgy2 + py1;

      if (px3 < 0 || px3 >= E3pxm->ww) continue;                                 //  omit parts beyond edges
      if (py3 < 0 || py3 >= E3pxm->hh) continue;

      pixT = PXBpix(attr.pxb_text,px1,py1);                                      //  copy-from text pixel
      pix3 = PXMpix(E3pxm,px3,py3);                                              //  copy-to image pixel

      e3part = pixT[3] / 256.0;                                                  //  text image transparency
      
      pix3[0] = pixT[0] + e3part * pix3[0];                                      //  combine text part + image part
      pix3[1] = pixT[1] + e3part * pix3[1];
      pix3[2] = pixT[2] + e3part * pix3[2];

      if (nc > 3) {
         Ot = (1.0 - e3part);                                                    //  text opacity
         Om = pix3[3] / 256.0;                                                   //  image opacity
         Ob = 1.0 - (1.0 - Ot) * (1.0 - Om);                                     //  combined opacity
         pix3[3] = 255.0 * Ob;
      }
   }

   if (textpresent) {
      Fpaint3(orgx1,orgy1,ww1,hh1,cr);                                            //  update window to erase old text
      textpresent = 0;
      CEF->Fmods--;
   }
                                                                                 //  (updates together to reduce flicker)
   Fpaint3(orgx2,orgy2,ww2,hh2,cr);                                              //  update window for new text

   draw_context_destroy(draw_context); 

   textpresent = 1;                                                              //  mark text is present
   CEF->Fmods++;                                                                 //  increase mod count
   CEF->Fsaved = 0;

   orgx1 = orgx2;                                                                //  remember overlap rectangle
   orgy1 = orgy2;                                                                //    for next call
   ww1 = ww2;
   hh1 = hh2;
   return;
}


//  load text and attributes from a file

void load_text(zdialog *zd)
{
   FILE        *fid;
   int         err, nn;
   char        *pp, *pp2, *file, buff[1200];
   cchar       *dialogtitle = "load text data from a file";
   textattr_t  attr;

   file = zgetfile(dialogtitle,MWIN,"file",addtext_dirk);                        //  get input file from user
   if (! file) return;

   fid = fopen(file,"r");                                                        //  open for read
   if (! fid) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   pp = fgets_trim(buff,1200,fid);                                               //  read text string
   if (! pp) goto badfile;
   if (! strmatchN(pp,"text string: ",13)) goto badfile;
   pp += 13;
   if (strlen(pp) < 2) goto badfile;
   repl_Nstrs(pp,attr.text,"\\n","\n",null);                                     //  replace "\n" with newline char.

   pp = fgets_trim(buff,1200,fid);                                               //  read font and size
   if (! pp) goto badfile;
   if (! strmatchN(pp,"font: ",6)) goto badfile;
   pp += 6;
   strTrim(pp);
   pp2 = strstr(pp,"size: ");
   if (! pp2) goto badfile;
   *pp2 = 0;
   pp2 += 6;
   strncpy0(attr.font,pp,80);
   attr.size = atoi(pp2);
   if (attr.size < 8) goto badfile;

   pp = fgets_trim(buff,1200,fid);                                               //  read text attributes
   if (! pp) goto badfile;
   nn = sscanf(pp,"attributes: %f  %s %s %s %s  %d %d %d %d  %d %d %d",
            &attr.angle, attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            &attr.transp[0], &attr.transp[1], &attr.transp[2], &attr.transp[3],
            &attr.towidth, &attr.shwidth, &attr.shangle);
   if (nn != 12) goto badfile;

   err = fclose(fid);
   if (err) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   zdialog_stuff(zd,"text",attr.text);                                           //  stuff zdialog fields
   zdialog_stuff(zd,"fontname",attr.font);
   zdialog_stuff(zd,"fontsize",attr.size);
   zdialog_stuff(zd,"fgangle",attr.angle);
   zdialog_stuff(zd,"fgcolor",attr.color[0]);
   zdialog_stuff(zd,"bgcolor",attr.color[1]);
   zdialog_stuff(zd,"tocolor",attr.color[2]);
   zdialog_stuff(zd,"shcolor",attr.color[3]);
   zdialog_stuff(zd,"fgtransp",attr.transp[0]);
   zdialog_stuff(zd,"bgtransp",attr.transp[1]);
   zdialog_stuff(zd,"totransp",attr.transp[2]);
   zdialog_stuff(zd,"shtransp",attr.transp[3]);
   zdialog_stuff(zd,"towidth",attr.towidth);
   zdialog_stuff(zd,"shwidth",attr.shwidth);
   zdialog_stuff(zd,"shangle",attr.shangle);
   return;

badfile:                                                                         //  project file had a problem
   fclose(fid);
   zmessageACK(Mwin,ZTX("text file is defective"));
   printz("buff: %s\n",buff);
}


//  save text to a file

void save_text(zdialog *zd)
{
   cchar       *dialogtitle = "save text data to a file";
   FILE        *fid;
   char        *file, text2[1200];
   textattr_t  attr;

   file = zgetfile(dialogtitle,MWIN,"save",addtext_dirk);                        //  get output file from user
   if (! file) return;

   fid = fopen(file,"w");                                                        //  open for write
   if (! fid) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   zdialog_fetch(zd,"text",attr.text,1000);                                      //  get text and attributes from zdialog
   zdialog_fetch(zd,"fontname",attr.font,80);
   zdialog_fetch(zd,"fontsize",attr.size);
   zdialog_fetch(zd,"fgangle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   repl_Nstrs(attr.text,text2,"\n","\\n",null);                                  //  replace newlines with "\n"

   fprintf(fid,"text string: %s\n",text2);

   fprintf(fid,"font: %s  size: %d \n",attr.font, attr.size);

   fprintf(fid,"attributes: %.4f  %s %s %s %s  %d %d %d %d  %d %d %d \n",
            attr.angle, attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            attr.transp[0], attr.transp[1], attr.transp[2], attr.transp[3],
            attr.towidth, attr.shwidth, attr.shangle);

   fclose(fid);
   return;
}


/********************************************************************************/

/***

   Create a graphic image with text, any color, any font, any angle.
   Add outline and shadow colors. Apply transparencies.

   struct textattr_t                                     //  attributes for gentext() function
      char     text[1000];                               //  text to generate image from
      char     font[80];                                 //  font name
      int      size;                                     //  font size
      int      tww, thh;                                 //  generated image size, unrotated
      float    angle;                                    //  text angle, degrees
      float    sinT, cosT;                               //  trig funcs for text angle
      char     color[4][20];                             //  text, backing, outline, shadow "R|G|B"
      int      transp[4];                                //  corresponding transparencies 0-255
      int      towidth;                                  //  outline width, pixels
      int      shwidth;                                  //  shadow width
      int      shangle;                                  //  shadow angle -180...+180
      PXB      *pxb_text;                                //  image with text/outline/shadow

***/

int gentext(textattr_t *attr)
{
   PXB * gentext_outline(textattr_t *, PXB *);
   PXB * gentext_shadow(textattr_t *, PXB *);

   PangoFontDescription    *pfont;
   PangoLayout             *playout;
   cairo_surface_t         *surface;
   cairo_t                 *cr;

   float          angle;
   int            fgred, fggreen, fgblue;
   int            bgred, bggreen, bgblue;
   int            tored, togreen, toblue;
   int            shred, shgreen, shblue;
   float          fgtransp, bgtransp, totransp, shtransp;

   PXB            *pxb_temp1, *pxb_temp2;
   uint8          *cairo_data, *cpix;
   uint8          *pix1, *pix2;
   int            px, py, ww, hh;
   char           font[100];                                                     //  font name and size
   int            red, green, blue;
   float          fgpart, topart, shpart, bgpart;

   if (! *attr->text) strcpy(attr->text,"null");

   if (attr->pxb_text) PXB_free(attr->pxb_text);

   snprintf(font,100,"%s %d",attr->font,attr->size);                             //  font name and size together
   angle = attr->angle;                                                          //  text angle, degrees
   attr->sinT = sin(angle/57.296);                                               //  trig funcs for text angle
   attr->cosT = cos(angle/57.296);
   fgred = atoi(strField(attr->color[0],'|',1));                                 //  get text foreground color
   fggreen = atoi(strField(attr->color[0],'|',2));
   fgblue = atoi(strField(attr->color[0],'|',3));
   bgred = atoi(strField(attr->color[1],'|',1));                                 //  get text background color
   bggreen = atoi(strField(attr->color[1],'|',2));
   bgblue = atoi(strField(attr->color[1],'|',3));
   tored = atoi(strField(attr->color[2],'|',1));                                 //  get text outline color
   togreen = atoi(strField(attr->color[2],'|',2));
   toblue = atoi(strField(attr->color[2],'|',3));
   shred = atoi(strField(attr->color[3],'|',1));                                 //  get text shadow color
   shgreen = atoi(strField(attr->color[3],'|',2));
   shblue = atoi(strField(attr->color[3],'|',3));
   fgtransp = 0.01 * attr->transp[0];                                            //  get transparencies
   bgtransp = 0.01 * attr->transp[1];                                            //  text, background, outline, shadow
   totransp = 0.01 * attr->transp[2];
   shtransp = 0.01 * attr->transp[3];

   pfont = pango_font_description_from_string(font);                             //  make layout with text
   playout = gtk_widget_create_pango_layout(Fdrawin,null);                       //  Fdrawin instead of Cdrawin         17.08
   if (! playout) zappcrash("gentext(): cannot create pango layout");
   pango_layout_set_font_description(playout,pfont);
   pango_layout_set_text(playout,attr->text,-1);

   pango_layout_get_pixel_size(playout,&ww,&hh);
   ww += 2 + 0.2 * attr->size;                                                   //  compensate bad font metrics
   hh += 2 + 0.1 * attr->size;
   attr->tww = ww;                                                               //  save text image size before rotate
   attr->thh = hh;

   surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,ww,hh);               //  cairo output image
   cr = cairo_create(surface);
   pango_cairo_show_layout(cr,playout);                                          //  write text layout to image

   cairo_data = cairo_image_surface_get_data(surface);                           //  get text image pixels

   pxb_temp1 = PXB_make(ww+5,hh,0);                                              //  create PXB                         18.01

   for (py = 0; py < hh; py++)                                                   //  copy text image to PXB
   for (px = 0; px < ww; px++)
   {
      cpix = cairo_data + 4 * (ww * py + px);                                    //  pango output is monocolor
      pix2 = PXBpix(pxb_temp1,px+4,py);                                          //  small pad before text              18.01
      pix2[0] = cpix[3];                                                         //  use red [0] for text intensity
      pix2[1] = pix2[2] = 0;
   }

   pango_font_description_free(pfont);                                           //  free resources
   g_object_unref(playout);
   cairo_destroy(cr);
   cairo_surface_destroy(surface);

   pxb_temp2 = gentext_outline(attr,pxb_temp1);                                  //  add text outline if any
   if (pxb_temp2) {                                                              //    using green [1] for outline intensity
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   pxb_temp2 = gentext_shadow(attr,pxb_temp1);                                   //  add text shadow color if any
   if (pxb_temp2) {                                                              //    using blue [2] for shadow intensity
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   if (fabsf(angle) > 0.1) {                                                     //  rotate text if wanted
      pxb_temp2 = PXB_rotate(pxb_temp1,angle);
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   ww = pxb_temp1->ww;                                                           //  text image input PXB
   hh = pxb_temp1->hh;
   pxb_temp2 = PXB_make(ww,hh,1);                                                //  text image output PXB

   for (py = 0; py < hh; py++)                                                   //  loop all pixels in text image
   for (px = 0; px < ww; px++)
   {
      pix1 = PXBpix(pxb_temp1,px,py);                                            //  copy-from pixel (text + outline + shadow)
      pix2 = PXBpix(pxb_temp2,px,py);                                            //  copy-to pixel

      fgpart = pix1[0] / 256.0;                                                  //  text opacity 0-1
      topart = pix1[1] / 256.0;                                                  //  outline
      shpart = pix1[2] / 256.0;                                                  //  shadow
      bgpart = (1.0 - fgpart - topart - shpart);                                 //  background 
      fgpart = fgpart * (1.0 - fgtransp);                                        //  reduce for transparencies
      topart = topart * (1.0 - totransp);
      shpart = shpart * (1.0 - shtransp);
      bgpart = bgpart * (1.0 - bgtransp);

      red = fgpart * fgred + topart * tored + shpart * shred + bgpart * bgred;
      green = fgpart * fggreen + topart * togreen + shpart * shgreen + bgpart * bggreen;
      blue = fgpart * fgblue + topart * toblue + shpart * shblue + bgpart * bgblue;

      pix2[0] = red;                                                             //  output total red, green, blue
      pix2[1] = green;
      pix2[2] = blue;

      pix2[3] = 255 * (1.0 - fgpart - topart - shpart - bgpart);                 //  image part visible through text
   }

   PXB_free(pxb_temp1);
   attr->pxb_text = pxb_temp2;
   return 0;
}


//  add an outline color to the text character edges
//  red color [0] is original monocolor text
//  use green color [1] for added outline

PXB * gentext_outline(textattr_t *attr, PXB *pxb1)
{
   PXB         *pxb2;
   int         toww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   uint8       *pix1, *pix2;
   float       theta;

   toww = attr->towidth;                                                         //  text outline color width
   if (toww == 0) return 0;                                                      //  zero

   ww1 = pxb1->ww;                                                               //  input PXB dimensions
   hh1 = pxb1->hh;
   ww2 = ww1 + toww * 2;                                                         //  add margins for outline width
   hh2 = hh1 + toww * 2;
   pxb2 = PXB_make(ww2,hh2,0);                                                   //  output PXB

   for (py = 0; py < hh1; py++)                                                  //  copy text pixels to outline pixels
   for (px = 0; px < ww1; px++)                                                  //    displaced by outline width
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+toww,py+toww);
      pix2[0] = pix1[0];
   }

   theta = 0.7 / toww;
   for (theta = 0; theta < 6.3; theta += 0.7/toww)                               //  displace outline pixels in all directions
   {
      dx = roundf(toww * sinf(theta));
      dy = roundf(toww * cosf(theta));

      for (py = 0; py < hh1; py++)
      for (px = 0; px < ww1; px++)
      {
         pix1 = PXBpix(pxb1,px,py);
         pix2 = PXBpix(pxb2,px+toww+dx,py+toww+dy);
         if (pix2[1] < pix1[0] - pix2[0])                                        //  compare text to outline brightness
            pix2[1] = pix1[0] - pix2[0];                                         //  brighter part is outline pixel
      }
   }

   return pxb2;                                                                  //  pix2[0] / pix2[1] = text / outline
}


//  add a shadow to the text character edges
//  red color [0] is original monocolor text intensity
//  green color [1] is added outline if any
//  use blue color [2] for added shadow

PXB * gentext_shadow(textattr_t *attr, PXB *pxb1)
{
   PXB         *pxb2;
   int         shww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   uint8       *pix1, *pix2;
   float       theta;

   shww = attr->shwidth;                                                         //  text shadow width
   if (shww == 0) return 0;                                                      //  zero

   ww1 = pxb1->ww;                                                               //  input PXB dimensions
   hh1 = pxb1->hh;
   ww2 = ww1 + shww * 2;                                                         //  add margins for shadow width
   hh2 = hh1 + shww * 2;
   pxb2 = PXB_make(ww2,hh2,0);                                                   //  output PXB

   for (py = 0; py < hh1; py++)                                                  //  copy text pixels to shadow pixels
   for (px = 0; px < ww1; px++)                                                  //    displaced by shadow width
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+shww,py+shww);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
   }

   theta = (90 - attr->shangle) / 57.3;                                          //  degrees to radians, 0 = to the right
   dx = roundf(shww * sinf(theta));
   dy = roundf(shww * cosf(theta));

   for (py = 0; py < hh1; py++)                                                  //  displace text by shadow width
   for (px = 0; px < ww1; px++)
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+shww+dx,py+shww+dy);
      if (pix2[2] < pix1[0] + pix1[1] - pix2[0] - pix2[1])                       //  compare text+outline to shadow pixels
         pix2[2] = pix1[0] + pix1[1] - pix2[0] - pix2[1];                        //  brighter part is shadow pixel
   }

   return pxb2;                                                                  //  pix2[0] / pix2[1] / pix2[2]
}                                                                                //    = text / outline / shadow brightness


/********************************************************************************/

//  write a line or arrow on top of the image

namespace addline
{
   lineattr_t  attr;                                                             //  line/arrow attributes and image

   int      mpx, mpy;                                                            //  mouse position on image
   int      linepresent;                                                         //  flag, line present on image
   int      orgx1, orgy1, ww1, hh1;                                              //  old line image overlap rectangle
   int      orgx2, orgy2, ww2, hh2;                                              //  new overlap rectangle
   zdialog  *zd;

   int   dialog_event(zdialog *zd, cchar *event);                                //  dialog event function
   void  mousefunc();                                                            //  mouse event function
   void  write(int mode);                                                        //  write line on image

   editfunc    EFaddline;
}


void m_add_lines(GtkWidget *, cchar *menu)
{
   using namespace addline;

   cchar    *intro = ZTX("Enter line or arrow properties in dialog, \n"
                         "click/drag on image, right click to remove");

   F1_help_topic = "add_lines";                                                  //  user guide topic

   EFaddline.menufunc = m_add_lines;
   EFaddline.funcname = "add_lines";
   EFaddline.Farea = 1;                                                          //  select area ignored
   EFaddline.mousefunc = mousefunc;                                              //  18.01
   if (! edit_setup(EFaddline)) return;                                          //  setup edit

/***
       ___________________________________________________
      |         Add lines or arrows to an image           |
      |                                                   |
      |  Enter line or arrow properties in dialog,        |
      |  click/drag on image, right click to remove.      |
      |                                                   |
      |  Use settings file  [Open] [Save]                 |
      |                                                   |
      |  Line length [____]  width [____]                 |
      |  Arrow head  [x] left   [x] right                 |
      |                                                   |
      |            color   transp.   width     angle      |
      |  line     [#####] [_______]           [______]    |                      fgcolor fgtransp fgangle
      |  backing  [#####] [_______]                       |                      bgcolor bgtransp
      |  outline  [#####] [_______] [_______]             |                      tocolor totransp towidth
      |  shadow   [#####] [_______] [_______] [______]    |                      shcolor shtransp shwidth shangle
      |                                                   |
      |                           [Apply] [Done] [Cancel] |
      |___________________________________________________|

***/

   zd = zdialog_new(ZTX("Add lines or arrows to an image"),Mwin,Bapply,Bdone,Bcancel,null);
   EFaddline.zd = zd;
   EFaddline.mousefunc = mousefunc;
   EFaddline.menufunc = m_add_lines;                                             //  allow restart

   zdialog_add_widget(zd,"label","intro","dialog",intro,"space=3");

   zdialog_add_widget(zd,"hbox","hbfile","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labfile","hbfile",ZTX("Use settings file"),"space=3");
   zdialog_add_widget(zd,"button",Bopen,"hbfile",Bopen);
   zdialog_add_widget(zd,"button",Bsave,"hbfile",Bsave);

   zdialog_add_widget(zd,"hbox","hbline","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lablength","hbline",ZTX("Line length"),"space=5");
   zdialog_add_widget(zd,"zspin","length","hbline","2|9999|1|20");
   zdialog_add_widget(zd,"label","space","hbline",0,"space=10");
   zdialog_add_widget(zd,"label","labwidth","hbline",Bwidth,"space=5");
   zdialog_add_widget(zd,"zspin","width","hbline","1|99|1|2");

   zdialog_add_widget(zd,"hbox","hbarrow","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labarrow","hbarrow",ZTX("Arrow head"),"space=5");
   zdialog_add_widget(zd,"check","larrow","hbarrow",Bleft);
   zdialog_add_widget(zd,"label","space","hbarrow",0,"space=10");
   zdialog_add_widget(zd,"check","rarrow","hbarrow",Bright);

   zdialog_add_widget(zd,"hbox","hbcol","dialog");
   zdialog_add_widget(zd,"vbox","vbcol1","hbcol",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbcol2","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol3","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol4","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol5","hbcol",0,"homog|space=2");

   zdialog_add_widget(zd,"label","space","vbcol1");
   zdialog_add_widget(zd,"label","labline","vbcol1",ZTX("line"));
   zdialog_add_widget(zd,"label","labback","vbcol1",ZTX("backing"));
   zdialog_add_widget(zd,"label","laboutln","vbcol1",ZTX("outline"));
   zdialog_add_widget(zd,"label","labshadow","vbcol1",ZTX("shadow"));

   zdialog_add_widget(zd,"label","labcol","vbcol2",Bcolor);
   zdialog_add_widget(zd,"colorbutt","fgcolor","vbcol2","0|0|0");
   zdialog_add_widget(zd,"colorbutt","bgcolor","vbcol2","255|255|255");
   zdialog_add_widget(zd,"colorbutt","tocolor","vbcol2","255|0|0");
   zdialog_add_widget(zd,"colorbutt","shcolor","vbcol2","255|0|0");

   zdialog_add_widget(zd,"label","labcol","vbcol3","Transp.");
   zdialog_add_widget(zd,"zspin","fgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","bgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","totransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"zspin","shtransp","vbcol3","0|100|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol4",Bwidth);
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"zspin","towidth","vbcol4","0|30|1|0");
   zdialog_add_widget(zd,"zspin","shwidth","vbcol4","0|50|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol5",Bangle);
   zdialog_add_widget(zd,"zspin","fgangle","vbcol5","-180|180|0.1|0");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"zspin","shangle","vbcol5","-180|180|1|0");

   zdialog_add_ttip(zd,Bapply,ZTX("fix line/arrow in layout \n start new line/arrow"));

   zdialog_restore_inputs(zd);                                                   //  restore prior inputs

   memset(&attr,0,sizeof(attr));

   zdialog_fetch(zd,"length",attr.length);                                       //  get defaults or prior inputs
   zdialog_fetch(zd,"width",attr.width);
   zdialog_fetch(zd,"larrow",attr.larrow);
   zdialog_fetch(zd,"rarrow",attr.rarrow);
   zdialog_fetch(zd,"fgangle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   genline(&attr);                                                               //  generate initial line

   takeMouse(mousefunc,dragcursor);                                              //  connect mouse function
   linepresent = 0;                                                              //  no line on image yet
   mpx = mpy = -1;                                                               //  no position defined yet

   zdialog_run(zd,dialog_event,"save");                                          //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int addline::dialog_event(zdialog *zd, cchar *event)
{
   using namespace addline;

   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  Apply, commit present line to image
         write(1);
         linepresent = 0;                                                        //  (no old line to erase)
         mpx = mpy = -1;
         zd->zstat = 0;
         edit_done(0);                                                           //  finish
         m_add_lines(0,0);                                                       //  start again
         return 1;
      }

      if (zd->zstat == 2 && CEF->Fmods) edit_done(0);                            //  Done, complete pending edit
      else edit_cancel(0);                                                       //  Cancel or kill
      return 1;
   }

   if (strmatch(event,"focus")) {                                                //  toggle mouse capture
      takeMouse(mousefunc,dragcursor);                                           //  connect mouse function
      return 1;
   }
   
   if (strmatch(event,Bopen))                                                    //  load zdialog fields from a file
   {
      load_line(zd);
      zdialog_fetch(zd,"length",attr.length);
      zdialog_fetch(zd,"width",attr.width);
      zdialog_fetch(zd,"larrow",attr.larrow);
      zdialog_fetch(zd,"rarrow",attr.rarrow);
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);
      zdialog_fetch(zd,"fgangle",attr.angle);
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);
      zdialog_fetch(zd,"tocolor",attr.color[2],20);
      zdialog_fetch(zd,"totransp",attr.transp[2]);
      zdialog_fetch(zd,"towidth",attr.towidth);
      zdialog_fetch(zd,"shcolor",attr.color[3],20);
      zdialog_fetch(zd,"shtransp",attr.transp[3]);
      zdialog_fetch(zd,"shwidth",attr.shwidth);
      zdialog_fetch(zd,"shangle",attr.shangle);
   }

   if (strmatch(event,Bsave)) {                                                  //  save zdialog fields to file
      save_line(zd);
      return 1;
   }

   if (strmatch(event,"length"))                                                 //  line length
      zdialog_fetch(zd,"length",attr.length);

   if (strmatch(event,"width"))                                                  //  line width
      zdialog_fetch(zd,"width",attr.width);

   if (strmatch(event,"larrow"))                                                 //  left arrow head
      zdialog_fetch(zd,"larrow",attr.larrow);

   if (strmatch(event,"rarrow"))                                                 //  right arrow head
      zdialog_fetch(zd,"rarrow",attr.rarrow);

   if (strmatch(event,"fgangle"))                                                //  line angle
      zdialog_fetch(zd,"fgangle",attr.angle);

   if (strmatch(event,"fgcolor"))                                                //  foreground (line) color
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);

   if (strmatch(event,"bgcolor"))                                                //  background color
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);

   if (strmatch(event,"tocolor"))                                                //  line outline color
      zdialog_fetch(zd,"tocolor",attr.color[2],20);

   if (strmatch(event,"shcolor"))                                                //  line shadow color
      zdialog_fetch(zd,"shcolor",attr.color[3],20);

   if (strmatch(event,"fgtransp"))                                               //  foreground transparency
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);

   if (strmatch(event,"bgtransp"))                                               //  background transparency
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);

   if (strmatch(event,"totransp"))                                               //  line outline transparency
      zdialog_fetch(zd,"totransp",attr.transp[2]);

   if (strmatch(event,"shtransp"))                                               //  line shadow transparency
      zdialog_fetch(zd,"shtransp",attr.transp[3]);

   if (strmatch(event,"towidth"))                                                //  line outline width
      zdialog_fetch(zd,"towidth",attr.towidth);

   if (strmatch(event,"shwidth"))                                                //  line shadow width
      zdialog_fetch(zd,"shwidth",attr.shwidth);

   if (strmatch(event,"shangle"))                                                //  line shadow angle
      zdialog_fetch(zd,"shangle",attr.shangle);

   genline(&attr);                                                               //  build line image from attributes
   write(1);                                                                     //  write on image
   return 1;
}


//  mouse function, set new position for line on image

void addline::mousefunc()
{
   using namespace addline;

   float    ax1, ay1, ax2, ay2;                                                  //  line/arrow end points
   float    angle, rad, l2;
   float    amx, amy;
   float    d1, d2, sinv;

   if (RMclick) {                                                                //  right mouse click
      write(2);                                                                  //  erase old line, if any
      mpx = mpy = -1;
      LMclick = RMclick = Mxdrag = Mydrag = 0;
      return;
   }

   if (LMclick + Mxdrag + Mydrag == 0) return;

   if (LMclick) {
      mpx = Mxclick;                                                             //  new line position on image
      mpy = Myclick;
   }
   else {
      mpx = Mxdrag;
      mpy = Mydrag;
   }

   LMclick = RMclick = Mxdrag = Mydrag = 0;

   if (! linepresent) {
      orgx2 = mpx;
      orgy2 = mpy;
      write(1);
      return;
   }

   //  move the closest line endpoint to the mouse position and leave the other endpoint fixed

   angle = attr.angle;
   rad = -angle / 57.296;
   l2 = attr.length / 2.0;

   ww2 = attr.pxb_line->ww;                                                      //  line image buffer
   hh2 = attr.pxb_line->hh;

   amx = ww2 / 2.0;                                                              //  line midpoint within line image
   amy = hh2 / 2.0;
   
   ax1 = amx - l2 * cosf(rad) + 0.5;                                             //  line end points
   ay1 = amy + l2 * sinf(rad) + 0.5;
   ax2 = amx + l2 * cosf(rad) + 0.5;
   ay2 = amy - l2 * sinf(rad) + 0.5;
   
   d1 = (mpx-ax1-orgx1) * (mpx-ax1-orgx1) + (mpy-ay1-orgy1) * (mpy-ay1-orgy1);
   d2 = (mpx-ax2-orgx1) * (mpx-ax2-orgx1) + (mpy-ay2-orgy1) * (mpy-ay2-orgy1);

   d1 = sqrtf(d1);                                                               //  mouse - end point distance
   d2 = sqrtf(d2);

   if (d1 < d2) {                                                                //  move ax1/ay1 end to mouse
      ax2 += orgx1;
      ay2 += orgy1;
      ax1 = mpx;
      ay1 = mpy;
      attr.length = d2 + 0.5;
      sinv = (ay1-ay2) / d2;
      if (sinv > 1.0) sinv = 1.0;
      if (sinv < -1.0) sinv = -1.0;
      rad = asinf(sinv);
      angle = -57.296 * rad;
      if (mpx > ax2) angle = -180 - angle;
   }

   else {                                                                        //  move ax2/ay2 end to mouse
      ax1 += orgx1;
      ay1 += orgy1;
      ax2 = mpx;
      ay2 = mpy;
      attr.length = d1 + 0.5;
      sinv = (ay1-ay2) / d1;
      if (sinv > 1.0) sinv = 1.0;
      if (sinv < -1.0) sinv = -1.0;
      rad = asinf(sinv);
      angle = -57.296 * rad;
      if (mpx < ax1) angle = -180 - angle;
   }

   if (angle < -180) angle += 360;
   if (angle > 180) angle -= 360;
   attr.angle = angle;
   genline(&attr);
   ww2 = attr.pxb_line->ww;
   hh2 = attr.pxb_line->hh;
   amx = (ax1 + ax2) / 2.0;
   amy = (ay1 + ay2) / 2.0;
   orgx2 = amx - ww2 / 2.0;
   orgy2 = amy - hh2 / 2.0;
   write(1);

   zdialog_stuff(zd,"fgangle",attr.angle);
   zdialog_stuff(zd,"length",attr.length);
   return;
}


//  write line on image at designated location
//  mode: 1  erase old and write to new position
//        2  erase old and write nothing

void addline::write(int mode)
{
   using namespace addline;

   float       *pix1, *pix3;
   uint8       *pixL;
   int         px1, py1, px3, py3, done;
   float       e3part, Ot, Om, Ob;
   int         nc = E1pxm->nc, pcc = nc * sizeof(float);

   cairo_t *cr = draw_context_create(gdkwin,draw_context);                       //  17.04

   if (linepresent)
   {
      for (py3 = orgy1; py3 < orgy1 + hh1; py3++)                                //  erase prior line image
      for (px3 = orgx1; px3 < orgx1 + ww1; px3++)                                //  replace E3 pixels with E1 pixels
      {                                                                          //    in prior overlap rectangle
         if (px3 < 0 || px3 >= E3pxm->ww) continue;
         if (py3 < 0 || py3 >= E3pxm->hh) continue;
         pix1 = PXMpix(E1pxm,px3,py3);
         pix3 = PXMpix(E3pxm,px3,py3);
         memcpy(pix3,pix1,pcc);
      }
   }

   done = 0;
   if (mode == 2) done = 1;                                                      //  erase only
   if (mpx < 0 && mpy < 0) done = 1;                                             //  no position defined

   if (done) {
      if (linepresent) {
         Fpaint3(orgx1,orgy1,ww1,hh1,cr);                                        //  update window to erase old line
         linepresent = 0;                                                        //  mark no line present
      }

      draw_context_destroy(draw_context);                                        //  17.04
      return;
   }

   ww2 = attr.pxb_line->ww;                                                      //  line image buffer
   hh2 = attr.pxb_line->hh;

   for (py1 = 0; py1 < hh2; py1++)                                               //  loop all pixels in line image
   for (px1 = 0; px1 < ww2; px1++)
   {
      px3 = orgx2 + px1;                                                         //  copy-to image3 pixel
      py3 = orgy2 + py1;

      if (px3 < 0 || px3 >= E3pxm->ww) continue;                                 //  omit parts beyond edges
      if (py3 < 0 || py3 >= E3pxm->hh) continue;

      pixL = PXBpix(attr.pxb_line,px1,py1);                                      //  copy-from line pixel
      pix3 = PXMpix(E3pxm,px3,py3);                                              //  copy-to image pixel

      e3part = pixL[3] / 256.0;                                                  //  line image transparency

      pix3[0] = pixL[0] + e3part * pix3[0];                                      //  combine line part + image part
      pix3[1] = pixL[1] + e3part * pix3[1];
      pix3[2] = pixL[2] + e3part * pix3[2];

      if (nc > 3) {
         Ot = (1.0 - e3part);                                                    //  line opacity
         Om = pix3[3] / 256.0;                                                   //  image opacity
         Ob = 1.0 - (1.0 - Ot) * (1.0 - Om);                                     //  combined opacity
         pix3[3] = 255.0 * Ob;
      }
   }

   if (linepresent) {
      Fpaint3(orgx1,orgy1,ww1,hh1,cr);                                           //  update window to erase old line
      linepresent = 0;
   }
                                                                                 //  (updates together to reduce flicker)
   Fpaint3(orgx2,orgy2,ww2,hh2,cr);                                              //  update window for new line

   draw_context_destroy(draw_context);                                           //  17.04

   CEF->Fmods++;
   CEF->Fsaved = 0;
   linepresent = 1;                                                              //  mark line is present

   orgx1 = orgx2;                                                                //  remember overlap rectangle
   orgy1 = orgy2;                                                                //    for next call
   ww1 = ww2;
   hh1 = hh2;

   return;
}


//  load line attributes from a file

void load_line(zdialog *zd)
{
   FILE        *fid;
   int         err, nn;
   char        *pp, *file, buff[200];
   cchar       *dialogtitle = "load text data from a file";
   lineattr_t  attr;

   file = zgetfile(dialogtitle,MWIN,"file",addline_dirk);                        //  get input file from user
   if (! file) return;

   fid = fopen(file,"r");                                                        //  open for read
   if (! fid) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   pp = fgets_trim(buff,200,fid);                                                //  read line attributes
   if (! pp) goto badfile;
   nn = sscanf(pp,"line attributes: %d %d %d %d %f  %s %s %s %s  %d %d %d %d  %d %d %d",
            &attr.length, &attr.width, &attr.larrow, &attr.rarrow, &attr.angle, 
            attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            &attr.transp[0], &attr.transp[1], &attr.transp[2], &attr.transp[3],
            &attr.towidth, &attr.shwidth, &attr.shangle);
   if (nn != 16) goto badfile;

   err = fclose(fid);
   if (err) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   zdialog_stuff(zd,"length",attr.length);                                       //  stuff line attributes into zdialog
   zdialog_stuff(zd,"width",attr.width);
   zdialog_stuff(zd,"larrow",attr.larrow);
   zdialog_stuff(zd,"rarrow",attr.rarrow);
   zdialog_stuff(zd,"fgangle",attr.angle);
   zdialog_stuff(zd,"fgcolor",attr.color[0]);
   zdialog_stuff(zd,"bgcolor",attr.color[1]);
   zdialog_stuff(zd,"tocolor",attr.color[2]);
   zdialog_stuff(zd,"shcolor",attr.color[3]);
   zdialog_stuff(zd,"fgtransp",attr.transp[0]);
   zdialog_stuff(zd,"bgtransp",attr.transp[1]);
   zdialog_stuff(zd,"totransp",attr.transp[2]);
   zdialog_stuff(zd,"shtransp",attr.transp[3]);
   zdialog_stuff(zd,"towidth",attr.towidth);
   zdialog_stuff(zd,"shwidth",attr.shwidth);
   zdialog_stuff(zd,"shangle",attr.shangle);
   return;

badfile:
   fclose(fid);
   zmessageACK(Mwin,ZTX("text file is defective"));
   printz("buff: %s\n",buff);
}


//  save line attributes to a file

void save_line(zdialog *zd)
{
   cchar       *dialogtitle = "save text data to a file";
   FILE        *fid;
   char        *file;
   lineattr_t  attr;

   file = zgetfile(dialogtitle,MWIN,"save",addline_dirk);                        //  get output file from user
   if (! file) return;

   fid = fopen(file,"w");                                                        //  open for write
   if (! fid) {
      zmessageACK(Mwin,strerror(errno));
      zfree(file);
      return;
   }

   zdialog_fetch(zd,"length",attr.length);                                       //  get line attributes from zdialog
   zdialog_fetch(zd,"width",attr.width);
   zdialog_fetch(zd,"larrow",attr.larrow);
   zdialog_fetch(zd,"rarrow",attr.rarrow);
   zdialog_fetch(zd,"fgangle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   fprintf(fid,"line attributes: %d %d %d %d %.4f  %s %s %s %s  %d %d %d %d  %d %d %d \n",
            attr.length, attr.width, attr.larrow, attr.rarrow, attr.angle, 
            attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            attr.transp[0], attr.transp[1], attr.transp[2], attr.transp[3],
            attr.towidth, attr.shwidth, attr.shangle);

   fclose(fid);
   return;
}


/********************************************************************************/

//  Create a graphic image of a line or arrow, any color,
//  any size, any angle. Add outline and shadow colors.

int genline(lineattr_t *attr)
{
   PXB * genline_outline(lineattr_t *, PXB *);
   PXB * genline_shadow(lineattr_t *, PXB *);

   cairo_surface_t         *surface;
   cairo_t                 *cr;

   float          angle;
   int            fgred, fggreen, fgblue;
   int            bgred, bggreen, bgblue;
   int            tored, togreen, toblue;
   int            shred, shgreen, shblue;
   float          fgtransp, bgtransp, totransp, shtransp;

   PXB            *pxb_temp1, *pxb_temp2;
   uint8          *cairo_data, *cpix;
   uint8          *pix1, *pix2;
   float          length, width;
   int            px, py, ww, hh;
   int            red, green, blue;
   float          fgpart, topart, shpart, bgpart;

   if (attr->pxb_line) PXB_free(attr->pxb_line);

   angle = attr->angle;                                                          //  line angle, degrees
   attr->sinT = sin(angle/57.296);                                               //  trig funcs for line angle
   attr->cosT = cos(angle/57.296);
   fgred = atoi(strField(attr->color[0],'|',1));                                 //  get line foreground color
   fggreen = atoi(strField(attr->color[0],'|',2));
   fgblue = atoi(strField(attr->color[0],'|',3));
   bgred = atoi(strField(attr->color[1],'|',1));                                 //  get line background color
   bggreen = atoi(strField(attr->color[1],'|',2));
   bgblue = atoi(strField(attr->color[1],'|',3));
   tored = atoi(strField(attr->color[2],'|',1));                                 //  get line outline color
   togreen = atoi(strField(attr->color[2],'|',2));
   toblue = atoi(strField(attr->color[2],'|',3));
   shred = atoi(strField(attr->color[3],'|',1));                                 //  get line shadow color
   shgreen = atoi(strField(attr->color[3],'|',2));
   shblue = atoi(strField(attr->color[3],'|',3));
   fgtransp = 0.01 * attr->transp[0];                                            //  get transparencies
   bgtransp = 0.01 * attr->transp[1];                                            //  line, background, outline, shadow
   totransp = 0.01 * attr->transp[2];
   shtransp = 0.01 * attr->transp[3];

   length = attr->length;                                                        //  line dimensions
   width = attr->width;

   ww = length + 20;                                                             //  create cairo surface
   hh = width + 20;                                                              //    with margins all around

   if (attr->larrow || attr->rarrow)                                             //  wider if arrow head used
      hh += width * 4;

   attr->lww = ww;
   attr->lhh = hh;

   surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,ww,hh);
   cr = cairo_create(surface);

   cairo_set_antialias(cr,CAIRO_ANTIALIAS_BEST);
   cairo_set_line_width(cr,width);
   cairo_set_line_cap(cr,CAIRO_LINE_CAP_ROUND);

   cairo_move_to(cr, 10, hh/2.0);                                                //  draw line in middle of surface
   cairo_line_to(cr, length+10, hh/2.0);

   if (attr->larrow) {                                                           //  add arrow heads if req.
      cairo_move_to(cr, 10, hh/2.0);
      cairo_line_to(cr, 10 + 2 * width, hh/2.0 - 2 * width);
      cairo_move_to(cr, 10, hh/2.0);
      cairo_line_to(cr, 10 + 2 * width, hh/2.0 + 2 * width);
   }

   if (attr->rarrow) {
      cairo_move_to(cr, length+10, hh/2.0);
      cairo_line_to(cr, length+10 - 2 * width, hh/2.0 - 2 * width);
      cairo_move_to(cr, length+10, hh/2.0);
      cairo_line_to(cr, length+10 - 2 * width, hh/2.0 + 2 * width);
   }

   cairo_stroke(cr);

   cairo_data = cairo_image_surface_get_data(surface);                           //  cairo image pixels

   pxb_temp1 = PXB_make(ww,hh,0);                                                //  create PXB

   for (py = 0; py < hh; py++)                                                   //  copy image to PXB
   for (px = 0; px < ww; px++)
   {
      cpix = cairo_data + 4 * (ww * py + px);                                    //  pango output is monocolor
      pix2 = PXBpix(pxb_temp1,px,py);
      pix2[0] = cpix[3];                                                         //  use red [0] for line intensity
      pix2[1] = pix2[2] = 0;
   }

   cairo_destroy(cr);                                                            //  free resources
   cairo_surface_destroy(surface);

   pxb_temp2 = genline_outline(attr,pxb_temp1);                                  //  add line outline if any
   if (pxb_temp2) {                                                              //    using green [1] for outline intensity
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   pxb_temp2 = genline_shadow(attr,pxb_temp1);                                   //  add line shadow color if any
   if (pxb_temp2) {                                                              //    using blue [2] for shadow intensity
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   if (fabsf(angle) > 0.1) {                                                     //  rotate line if wanted
      pxb_temp2 = PXB_rotate(pxb_temp1,angle);
      PXB_free(pxb_temp1);
      pxb_temp1 = pxb_temp2;
   }

   ww = pxb_temp1->ww;                                                           //  line image input PXB
   hh = pxb_temp1->hh;
   pxb_temp2 = PXB_make(ww,hh,1);                                                //  line image output PXB

   for (py = 0; py < hh; py++)                                                   //  loop all pixels in line image
   for (px = 0; px < ww; px++)
   {
      pix1 = PXBpix(pxb_temp1,px,py);                                            //  copy-from pixel (line + outline + shadow)
      pix2 = PXBpix(pxb_temp2,px,py);                                            //  copy-to pixel

      fgpart = pix1[0] / 256.0;
      topart = pix1[1] / 256.0;
      shpart = pix1[2] / 256.0;
      bgpart = (1.0 - fgpart - topart - shpart);
      fgpart = fgpart * (1.0 - fgtransp);
      topart = topart * (1.0 - totransp);
      shpart = shpart * (1.0 - shtransp);
      bgpart = bgpart * (1.0 - bgtransp);

      red = fgpart * fgred + topart * tored + shpart * shred + bgpart * bgred;
      green = fgpart * fggreen + topart * togreen + shpart * shgreen + bgpart * bggreen;
      blue = fgpart * fgblue + topart * toblue + shpart * shblue + bgpart * bgblue;

      pix2[0] = red;                                                             //  output total red, green blue
      pix2[1] = green;
      pix2[2] = blue;

      pix2[3] = 255 * (1.0 - fgpart - topart - shpart - bgpart);                 //  image part visible through line
   }

   PXB_free(pxb_temp1);
   attr->pxb_line = pxb_temp2;
   return 0;
}


//  add an outline color to the line edges
//  red color [0] is original monocolor line
//  use green color [1] for added outline

PXB * genline_outline(lineattr_t *attr, PXB *pxb1)
{
   PXB         *pxb2;
   int         toww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   uint8       *pix1, *pix2;
   float       theta;

   toww = attr->towidth;                                                         //  line outline color width
   if (toww == 0) return 0;                                                      //  zero

   ww1 = pxb1->ww;                                                               //  input PXB dimensions
   hh1 = pxb1->hh;
   ww2 = ww1 + toww * 2;                                                         //  add margins for outline width
   hh2 = hh1 + toww * 2;
   pxb2 = PXB_make(ww2,hh2,0);                                                   //  output PXB

   for (py = 0; py < hh1; py++)                                                  //  copy line pixels to outline pixels
   for (px = 0; px < ww1; px++)                                                  //    displaced by outline width
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+toww,py+toww);
      pix2[0] = pix1[0];
   }

   theta = 0.7 / toww;
   for (theta = 0; theta < 6.3; theta += 0.7/toww)                               //  displace outline pixels in all directions
   {
      dx = roundf(toww * sinf(theta));
      dy = roundf(toww * cosf(theta));

      for (py = 0; py < hh1; py++)
      for (px = 0; px < ww1; px++)
      {
         pix1 = PXBpix(pxb1,px,py);
         pix2 = PXBpix(pxb2,px+toww+dx,py+toww+dy);
         if (pix2[1] < pix1[0] - pix2[0])                                        //  compare line to outline brightness
            pix2[1] = pix1[0] - pix2[0];                                         //  brighter part is outline pixel
      }
   }

   return pxb2;                                                                  //  pix2[0] / pix2[1] = line / outline
}


//  add a shadow to the line edges
//  red color [0] is original monocolor line intensity
//  green color [1] is added outline if any
//  use blue color [2] for added shadow

PXB * genline_shadow(lineattr_t *attr, PXB *pxb1)
{
   PXB         *pxb2;
   int         shww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   uint8       *pix1, *pix2;
   float       theta;

   shww = attr->shwidth;                                                         //  line shadow width
   if (shww == 0) return 0;                                                      //  zero

   ww1 = pxb1->ww;                                                               //  input PXB dimensions
   hh1 = pxb1->hh;
   ww2 = ww1 + shww * 2;                                                         //  add margins for shadow width
   hh2 = hh1 + shww * 2;
   pxb2 = PXB_make(ww2,hh2,0);                                                   //  output PXB

   for (py = 0; py < hh1; py++)                                                  //  copy line pixels to shadow pixels
   for (px = 0; px < ww1; px++)                                                  //    displaced by shadow width
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+shww,py+shww);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
   }

   theta = (90 - attr->shangle) / 57.3;                                          //  degrees to radians, 0 = to the right
   dx = roundf(shww * sinf(theta));
   dy = roundf(shww * cosf(theta));

   for (py = 0; py < hh1; py++)                                                  //  displace line by shadow width
   for (px = 0; px < ww1; px++)
   {
      pix1 = PXBpix(pxb1,px,py);
      pix2 = PXBpix(pxb2,px+shww+dx,py+shww+dy);
      if (pix2[2] < pix1[0] + pix1[1] - pix2[0] - pix2[1])                       //  compare line+outline to shadow pixels
         pix2[2] = pix1[0] + pix1[1] - pix2[0] - pix2[1];                        //  brighter part is shadow pixel
   }

   return pxb2;                                                                  //  pix2[0] / pix2[1] / pix2[2]
}                                                                                //    = line / outline / shadow brightness


/********************************************************************************/

//  Select area and edit in parallel
//  Current edit function is applied to areas painted with the mouse.
//  Mouse can be weak or strong, and edits are applied incrementally.
//  
//  method:
//  entire image is a select area with all pixel edge distance = 0 (outside area)
//  blendwidth = 10000 (infinite)
//  pixels painted with mouse have increasing edge distance to amplify edits

int   paint_edits_radius;
int   paint_edits_cpower;
int   paint_edits_epower;

void m_paint_edits(GtkWidget *, cchar *)                                         //  menu function
{
   int   paint_edits_dialog_event(zdialog *, cchar *event);                      //  dialog event function
   void  paint_edits_mousefunc();                                                //  mouse function

   cchar    *title = ZTX("Paint Edits");
   cchar    *helptext = ZTX("Press F1 for help");
   int      yn, cc;

   F1_help_topic = "paint_edits";

   if (sa_stat || zdsela) {                                                      //  warn select area will be lost
      yn = zmessageYN(Mwin,ZTX("Select area cannot be kept.\n"
                               "Continue?"));
      if (! yn) return;
      sa_unselect();                                                             //  unselect area
      if (zdsela) zdialog_free(zdsela);
   }

   if (! CEF) {                                                                  //  edit func must be active
      zmessageACK(Mwin,ZTX("Edit function must be active"));
      return;
   }
   
   if (! CEF->FusePL) {
      zmessageACK(Mwin,ZTX("Cannot use Paint Edits"));
      return;
   }
   
   if (CEF->Fpreview) 
      zdialog_send_event(CEF->zd,"fullsize");                                    //  use full-size image

/***
    ______________________________________
   |         Press F1 for help            |
   |     Edit Function must be active     |
   |                                      |
   | mouse radius [____]                  |
   | power:  center [____]  edge [____]   |
   | [reset area]                         |
   |                              [done]  |
   |______________________________________|

***/

   zdsela = zdialog_new(title,Mwin,Bdone,null);
   zdialog_add_widget(zdsela,"label","labhelp1","dialog",helptext,"space=5");
   zdialog_add_widget(zdsela,"label","labspace","dialog");
   zdialog_add_widget(zdsela,"hbox","hbr","dialog",0,"space=3");
   zdialog_add_widget(zdsela,"label","labr","hbr",Bmouseradius,"space=5");
   zdialog_add_widget(zdsela,"zspin","radius","hbr","2|500|1|50");
   zdialog_add_widget(zdsela,"hbox","hbt","dialog",0,"space=3");
   zdialog_add_widget(zdsela,"label","labtc","hbt",ZTX("power:  center"),"space=5");
   zdialog_add_widget(zdsela,"zspin","center","hbt","0|100|1|50");
   zdialog_add_widget(zdsela,"label","labte","hbt",Bedge,"space=5");
   zdialog_add_widget(zdsela,"zspin","edge","hbt","0|100|1|0");
   zdialog_add_widget(zdsela,"hbox","hbra","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"button","reset","hbra",ZTX("reset area"),"space=5");

   paint_edits_radius = 50;
   paint_edits_cpower = 50;
   paint_edits_epower = 0;

   cc = Fpxb->ww * Fpxb->hh * sizeof(uint16);                                    //  allocate sa_pixmap[] for area
   sa_pixmap = (uint16 *) zmalloc(cc);
   memset(sa_pixmap,0,cc);                                                       //  edge distance = 0 for all pixels

   sa_minx = 0;                                                                  //  enclosing rectangle
   sa_maxx = Fpxb->ww;
   sa_miny = 0;
   sa_maxy = Fpxb->hh;

   sa_Npixel = Fpxb->ww * Fpxb->hh;
   sa_stat = 3;                                                                  //  area status = complete
   sa_mode = mode_image;                                                         //  area mode = whole image
   sa_calced = 1;                                                                //  edge calculation complete
   sa_blend = 10000;                                                             //  "blend width"
   sa_fww = Fpxb->ww;                                                            //  valid image dimensions
   sa_fhh = Fpxb->hh;
   areanumber++;                                                                 //  next sequential number

   zdialog_run(zdsela,paint_edits_dialog_event,"save");                          //  run dialog - parallel
   return;
}


//  Adjust whole image area to increase edit power for pixels within the mouse radius
//  sa_pixmap[*]  = 0 = never touched by mouse
//                = 1 = minimum edit power (barely painted)
//                = sa_blend = maximum edit power (edit fully applied)

int paint_edits_dialog_event(zdialog *zd, cchar *event)
{
   void  paint_edits_mousefunc();                                                //  mouse function
   int      cc;

   if (zd->zstat)                                                                //  done or cancel
   {
      freeMouse();                                                               //  disconnect mouse function
      if (CEF) zdialog_send_event(CEF->zd,"done");                               //  complete edit
      zdialog_free(zdsela);                                                      //  kill dialog
      sa_unselect();                                                             //  unselect area
      return 0;
   }

   if (sa_stat != 3) return 1;                                                   //  area gone
   if (! sa_validate()) return 1;                                                //  area invalid for curr. image file

   if (strmatch(event,"focus")) {                                                //  toggle mouse capture
      if (CEF) takeMouse(paint_edits_mousefunc,0);
      else freeMouse();                                                          //  disconnect mouse
   }

   if (strmatch(event,"radius"))
      zdialog_fetch(zd,"radius",paint_edits_radius);                             //  set mouse radius

   if (strmatch(event,"center"))
      zdialog_fetch(zd,"center",paint_edits_cpower);                             //  set mouse center power

   if (strmatch(event,"edge"))
      zdialog_fetch(zd,"edge",paint_edits_epower);                               //  set mouse edge power

   if (strmatch(event,"reset")) {
      sa_unselect();                                                             //  unselect current area if any
      cc = Fpxb->ww * Fpxb->hh * sizeof(uint16);                                 //  allocate sa_pixmap[] for new area
      sa_pixmap = (uint16 *) zmalloc(cc);
      memset(sa_pixmap,0,cc);                                                    //  edge distance = 0 for all pixels

      sa_minx = 0;                                                               //  enclosing rectangle
      sa_maxx = Fpxb->ww;
      sa_miny = 0;
      sa_maxy = Fpxb->hh;

      sa_Npixel = Fpxb->ww * Fpxb->hh;
      sa_stat = 3;                                                               //  area status = complete
      sa_mode = mode_image;                                                      //  area mode = whole image
      sa_calced = 1;                                                             //  edge calculation complete
      sa_blend = 10000;                                                          //  "blend width"
      sa_fww = Fpxb->ww;                                                         //  valid image dimensions
      sa_fhh = Fpxb->hh;
      areanumber++;                                                              //  next sequential number
   }

   return 1;
}


//  mouse function - adjust edit strength for areas within mouse radius
//  "edge distance" is increased for more strength, decreased for less

void paint_edits_mousefunc()
{
   int      ii, px, py, rx, ry;
   int      radius, radius2, cpower, epower;
   float    rad, rad2, power;

   if (! CEF) return;                                                            //  no active edit
   if (sa_stat != 3) return;                                                     //  area gone?

   radius = paint_edits_radius;                                                  //  pixel selection radius
   radius2 = radius * radius;
   cpower = paint_edits_cpower;
   epower = paint_edits_epower;

   draw_mousecircle(Mxposn,Myposn,radius,0,0);                                   //  show mouse selection circle

   if (LMclick || RMclick)                                                       //  mouse click, process normally
      return;

   if (Mbutton != 1 && Mbutton != 3)                                             //  button released
      return;

   Mxdrag = Mydrag = 0;                                                          //  neutralize drag

   for (rx = -radius; rx <= radius; rx++)                                        //  loop every pixel in radius
   for (ry = -radius; ry <= radius; ry++)
   {
      rad2 = rx * rx + ry * ry;
      if (rad2 > radius2) continue;                                              //  outside radius
      px = Mxposn + rx;
      py = Myposn + ry;
      if (px < 0 || px > Fpxb->ww-1) continue;                                   //  off the image edge
      if (py < 0 || py > Fpxb->hh-1) continue;

      ii = Fpxb->ww * py + px;
      rad = sqrt(rad2);
      power = cpower + rad / radius * (epower - cpower);                         //  power at pixel radius

      if (Mbutton == 1) {                                                        //  left mouse button
         sa_pixmap[ii] += 5.0 * power;                                           //  increase edit power
         if (sa_pixmap[ii] > sa_blend) sa_pixmap[ii] = sa_blend;
      }

      if (Mbutton == 3) {                                                        //  right mouse button
         if (sa_pixmap[ii] <= 5.0 * power) sa_pixmap[ii] = 0;                    //  weaken edit power
         else sa_pixmap[ii] -= 5.0 * power;
      }
   }

   zdialog_send_event(CEF->zd,"blendwidth");                                     //  notify edit dialog

   draw_mousecircle(Mxposn,Myposn,radius,0,0);                                   //  show mouse selection circle
   return;
}


/********************************************************************************/

//  Use the image brightness or color values to leverage subsequent edits.
//  Method:
//  Select the whole image as an area.
//  Set "edge distance" 1 to 999 from pixel brightness or RGB color.
//  Set "blend width" to 999.
//  Edit function coefficient = edge distance / blend width.

spldat   *leveds_curve;
int      leveds_type, leveds_color;
int      leveds_ptype, leveds_pcolor;
float    *leveds_lever = 0;


//  menu function

void m_lever_edits(GtkWidget *, cchar *)
{
   int    leveds_event(zdialog *, cchar *event);                                 //  dialog event and completion func
   void   leveds_curve_update(int spc);                                          //  curve update callback function

   cchar    *title = ZTX("Leverage Edits");
   cchar    *legend = ZTX("Edit Function Amplifier");
   int      yn;

   F1_help_topic = "leverage_edits";

   if (sa_stat || zdsela) {                                                      //  warn select area will be lost
      yn = zmessageYN(Mwin,ZTX("Select area cannot be kept.\n"
                               "Continue?"));
      if (! yn) return;
      sa_unselect();                                                             //  unselect area
      if (zdsela) zdialog_free(zdsela);
   }

   if (! CEF) {                                                                  //  edit func must be active
      zmessageACK(Mwin,ZTX("Edit function must be active"));
      return;
   }

   if (! CEF->FusePL) {
      zmessageACK(Mwin,ZTX("Cannot use Leverage Edits"));
      return;
   }

   if (CEF->Fpreview) 
      zdialog_send_event(CEF->zd,"fullsize");                                    //  use full-size image

/***
                  Edit Function Amplifier
             ------------------------------------------
            |                                          |
            |                                          |
            |           curve drawing area             |
            |                                          |
            |                                          |
             ------------------------------------------
             minimum                            maximum

             (o) Brightness  (o) Contrast
             (o) All  (o) Red  (o) Green  (o) Blue
             Curve File: [ Open ] [ Save ]
                                              [ Done ]
***/

   zdsela = zdialog_new(title,Mwin,Bdone,null);

   zdialog_add_widget(zdsela,"label","labt","dialog",legend);
   zdialog_add_widget(zdsela,"frame","fr1","dialog",0,"expand");
   zdialog_add_widget(zdsela,"hbox","hba","dialog");
   zdialog_add_widget(zdsela,"label","labda","hba",ZTX("minimum"),"space=5");
   zdialog_add_widget(zdsela,"label","space","hba",0,"expand");
   zdialog_add_widget(zdsela,"label","labba","hba",ZTX("maximum"),"space=5");

   zdialog_add_widget(zdsela,"hbox","hbbr1","dialog");
   zdialog_add_widget(zdsela,"radio","bright","hbbr1",Bbrightness,"space=5");
   zdialog_add_widget(zdsela,"radio","contrast","hbbr1",Bcontrast,"space=5");
   zdialog_add_widget(zdsela,"hbox","hbbr2","dialog");
   zdialog_add_widget(zdsela,"radio","all","hbbr2",Ball,"space=5");
   zdialog_add_widget(zdsela,"radio","red","hbbr2",Bred,"space=5");
   zdialog_add_widget(zdsela,"radio","green","hbbr2",Bgreen,"space=5");
   zdialog_add_widget(zdsela,"radio","blue","hbbr2",Bblue,"space=5");

   zdialog_add_widget(zdsela,"hbox","hbcf","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"label","labcf","hbcf",Bcurvefile,"space=5");
   zdialog_add_widget(zdsela,"button","loadcurve","hbcf",Bopen,"space=5");
   zdialog_add_widget(zdsela,"button","savecurve","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zdsela,"fr1");                              //  setup for curve editing
   spldat *sd = splcurve_init(frame,leveds_curve_update);
   leveds_curve = sd;

   sd->Nspc = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 3;                                                               //  initial curve anchor points
   sd->fact[0] = 1;
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.5;
   sd->apx[0][1] = 0.50;
   sd->apy[0][1] = 0.5;
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.5;
   splcurve_generate(sd,0);                                                      //  generate curve data

   zdialog_stuff(zdsela,"bright",1);                                             //  type leverage = brightness
   zdialog_stuff(zdsela,"contrast",0);
   zdialog_stuff(zdsela,"all",1);                                                //  color used = all
   zdialog_stuff(zdsela,"red",0);
   zdialog_stuff(zdsela,"green",0);
   zdialog_stuff(zdsela,"blue",0);
   leveds_type = 1;
   leveds_color = 1;
   leveds_ptype = 0;
   leveds_pcolor = 0;

   int cc = Fpxb->ww * Fpxb->hh * sizeof(uint16);                                //  allocate sa_pixmap[] for area
   sa_pixmap = (uint16 *) zmalloc(cc);
   memset(sa_pixmap,0,cc);                                                       //  edge distance = 0 for all pixels

   sa_minx = 0;                                                                  //  enclosing rectangle
   sa_maxx = Fpxb->ww;
   sa_miny = 0;
   sa_maxy = Fpxb->hh;

   sa_Npixel = Fpxb->ww * Fpxb->hh;
   sa_stat = 3;                                                                  //  area status = complete
   sa_mode = mode_image;                                                         //  area mode = whole image
   sa_calced = 1;                                                                //  edge calculation complete
   sa_blend = 999;                                                               //  "blend width" = 999
   sa_fww = Fpxb->ww;                                                            //  valid image dimensions
   sa_fhh = Fpxb->hh;
   areanumber++;                                                                 //  next sequential number

   if (leveds_lever) zfree(leveds_lever);
   cc = E1pxm->ww * E1pxm->hh * sizeof(float);                                   //  allocate memory for lever
   leveds_lever = (float *) zmalloc(cc);

   zdialog_resize(zdsela,0,360);
   zdialog_run(zdsela,leveds_event,"save");                                      //  run dialog - parallel
   leveds_event(zdsela,"init");                                                  //  initialize default params
   return;
}


//  dialog event and completion function

int leveds_event(zdialog *zd, cchar *event)
{
   int         ii, kk, pixdist;
   float       xval, yval, lever;
   float       *pixel0, *pixel1, *pixel2, *pixel3, *pixel4;
   spldat      *sd = leveds_curve;
   
   if (zd->zstat) {                                                              //  done or cancel
      if (CEF && CEF->zd) zdialog_send_event(CEF->zd,"done");                    //  notify edit dialog
      sa_unselect();                                                             //  delete area
      zdialog_free(zdsela);
      zfree(sd);                                                                 //  free curve edit memory
      zfree(leveds_lever);
      leveds_lever = 0;
      return 1;
   }

   if (! sa_validate() || sa_stat != 3) {                                        //  select area gone
      zdialog_free(zdsela);
      zfree(sd);                                                                 //  free curve edit memory
      zfree(leveds_lever);
      leveds_lever = 0;
      return 1;
   }

   if (strmatch(event,"loadcurve")) {                                            //  load saved curve
      splcurve_load(sd);
      if (CEF && CEF->zd) zdialog_send_event(CEF->zd,"blendwidth");              //  notify edit dialog
      return 1;
   }

   if (strmatch(event,"savecurve")) {                                            //  save curve to file
      splcurve_save(sd);
      return 1;
   }

   ii = strmatchV(event,"bright","contrast",null);                               //  new lever type
   if (ii >= 1 && ii <= 2) leveds_type = ii;

   ii = strmatchV(event,"all","red","green","blue",null);                        //  new lever color
   if (ii >= 1 && ii <= 4) leveds_color = ii;

   if (leveds_type != leveds_ptype || leveds_color != leveds_pcolor)             //  test for change
   {
      if (! CEF) return 1;                                                       //  edit canceled

      leveds_ptype = leveds_type;
      leveds_pcolor = leveds_color;

      for (int ipy = 1; ipy < E1pxm->hh-1; ipy++)
      for (int ipx = 1; ipx < E1pxm->ww-1; ipx++)
      {
         pixel0 = PXMpix(E1pxm,ipx,ipy);                                         //  target pixel to measure
         lever = 0;

         if (leveds_type == 1)                                                   //  lever type = brightness
         {
            if (leveds_color == 1)
               lever = 0.333 * (pixel0[0] + pixel0[1] + pixel0[2]);              //  use all colors
            else {
               ii = leveds_color - 2;                                            //  use single color
               lever = pixel0[ii];
            }
         }

         else if (leveds_type == 2)                                              //  lever type = contrast
         {
            pixel1 = PXMpix(E1pxm,ipx-1,ipy);
            pixel2 = PXMpix(E1pxm,ipx+1,ipy);
            pixel3 = PXMpix(E1pxm,ipx,ipy-1);
            pixel4 = PXMpix(E1pxm,ipx,ipy+1);
            
            if (leveds_color == 1)                                               //  use all colors
            {
               lever  = fabsf(pixel0[0] - pixel1[0]) + fabsf(pixel0[0] - pixel2[0])
                      + fabsf(pixel0[0] - pixel3[0]) + fabsf(pixel0[0] - pixel4[0]);
               lever += fabsf(pixel0[1] - pixel1[1]) + fabsf(pixel0[1] - pixel2[1])
                      + fabsf(pixel0[1] - pixel3[1]) + fabsf(pixel0[1] - pixel4[1]);
               lever += fabsf(pixel0[2] - pixel1[2]) + fabsf(pixel0[2] - pixel2[2])
                      + fabsf(pixel0[2] - pixel3[2]) + fabsf(pixel0[2] - pixel4[2]);
               lever = lever / 12.0;
            }
            else                                                                 //  use single color
            {
               ii = leveds_color - 2;
               lever  = fabsf(pixel0[ii] - pixel1[ii]) + fabsf(pixel0[ii] - pixel2[ii])
                      + fabsf(pixel0[ii] - pixel3[ii]) + fabsf(pixel0[ii] - pixel4[ii]);
               lever = lever / 4.0;
            }
         }

         lever = lever / 1000.0;                                                 //  scale 0.0 to 0.999
         lever = pow(lever,0.3);                                                 //  log scale: 0.1 >> 0.5, 0.8 >> 0.94

         ii = ipy * E1pxm->ww + ipx;                                             //  save lever for each pixel
         leveds_lever[ii] = lever;
      }
   }

   if (strmatch(event,"edit")) {
      splcurve_generate(sd,0);                                                   //  regenerate the curve
      gtk_widget_queue_draw(sd->drawarea);
   }

   if (! CEF) return 1;                                                          //  edit canceled

   for (int ipy = 1; ipy < E1pxm->hh-1; ipy++)
   for (int ipx = 1; ipx < E1pxm->ww-1; ipx++)
   {
      pixel0 = PXMpix(E1pxm,ipx,ipy);                                            //  target pixel to measure
      ii = ipy * E1pxm->ww + ipx;
      lever = leveds_lever[ii];                                                  //  leverage to apply, 0.0 to 0.999
      xval = lever;                                                              //  curve x-value, 0.0 to 0.999
      kk = 1000 * xval;
      if (kk > 999) kk = 999;
      yval = sd->yval[0][kk];                                                    //  y-value, 0 to 0.999
      pixdist = 1 + 998 * yval;                                                  //  pixel edge distance, 1 to 999
      sa_pixmap[ii] = pixdist;
   }

   if (CEF->zd) zdialog_send_event(CEF->zd,"blendwidth");                        //  notify edit dialog
   return 1;
}


//  this function is called when curve is edited using mouse

void  leveds_curve_update(int)
{
   if (! zdsela) return;
   leveds_event(zdsela,"edit");
   return;
}


/********************************************************************************/

//  Plugin menu functions

namespace plugins_names
{
   #define maxplugins 100
   int         Nplugins;                                                         //  plugin menu items
   char        *plugins[100];
   GtkWidget   *popup_plugmenu = 0;
   editfunc    EFplugin;
}


//  edit plugins menu or choose and run a plugin function

void m_plugins(GtkWidget *, cchar *)
{
   using namespace plugins_names;

   void  m_edit_plugins(GtkWidget *, cchar *);
   void  m_run_plugin(GtkWidget *, cchar *);

   char        plugfile[200], buff[200], *pp;
   FILE        *fid;
   STATB       stbuff;
   int         ii, err;

   if (popup_plugmenu) {
      popup_menu(Mwin,popup_plugmenu);                                           //  popup the plugins menu
      return;
   }

   snprintf(plugfile,200,"%s/plugins",get_zhomedir());                           //  plugins file

   err = stat(plugfile,&stbuff);                                                 //  exists?
   if (err)
   {
      fid = fopen(plugfile,"w");                                                 //  no, create default
      fprintf(fid,"Gimp = gimp %%s \n");
      fprintf(fid,"auto-gamma = mogrify -auto-gamma %%s \n");
      fprintf(fid,"whiteboard cleanup = mogrify "                                //  ImageMagick white board cleanup
                  "-morphology Convolve DoG:15,100,0 "
                  "-negate -normalize -blur 0x1 -channel RBG "
                  "-level 60%%,91%%,0.1 %%s \n");
      fclose(fid);
   }

   fid = fopen(plugfile,"r");                                                    //  open plugins file
   if (! fid) {
      zmessageACK(Mwin,"plugins file: %s",strerror(errno));
      return;
   }

   for (ii = 0; ii < 99; ii++)                                                   //  read list of plugins
   {
      pp = fgets_trim(buff,200,fid,1);
      if (! pp) break;
      plugins[ii] = zstrdup(buff);
   }

   fclose(fid);
   Nplugins = ii;

   popup_plugmenu = create_popmenu();                                            //  create popup menu for plugins

   add_popmenu_item(popup_plugmenu, ZTX("Edit Plugins"),                         //  1st entry is Edit Plugins
                     m_edit_plugins, 0, ZTX("Edit plugins menu"));

   for (ii = 0; ii < Nplugins; ii++)                                             //  add the plugin menu functions
   {
      char *pp = strstr(plugins[ii]," = ");
      if (! pp) continue;
      *pp = 0;
      add_popmenu_item(popup_plugmenu, plugins[ii], m_run_plugin, 0,
                                 ZTX("Run as Fotoxx edit function"));
      *pp = ' ';
   }

   popup_menu(Mwin,popup_plugmenu);                                              //  popup the menu
   return;
}


//  edit plugins menu

void  m_edit_plugins(GtkWidget *, cchar *)                                       //  overhauled 
{
   using namespace plugins_names;

   int   edit_plugins_event(zdialog *zd, cchar *event);

   int         ii;
   char        *pp;
   zdialog     *zd;

   F1_help_topic = "plugins";

/***
       ___________________________________
      |         Edit Plugins              |
      |                                   |
      |  menu name [_____________|v]      |     e.g. edit with gimp
      |  command   [___________________]  |     e.g. gimp %s
      |                                   |
      |             [Add] [Remove] [Done] |
      |___________________________________|

***/

   zd = zdialog_new(ZTX("Edit Plugins"),Mwin,Badd,Bremove,Bdone,null);
   zdialog_add_widget(zd,"hbox","hbm","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labm","hbm",ZTX("menu name"),"space=5");
   zdialog_add_widget(zd,"comboE","menuname","hbm",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labc","hbc",ZTX("command"),"space=5");
   zdialog_add_widget(zd,"zentry","command","hbc",0,"space=5|expand");

   for (ii = 0; ii < Nplugins; ii++)                                             //  stuff combo box with available menus
   {
      pp = strstr(plugins[ii]," = ");                                            //  menu name = command line
      if (! pp) continue;
      *pp = 0;
      zdialog_cb_app(zd,"menuname",plugins[ii]);
      *pp = ' ';
   }

   zdialog_set_modal(zd);                                                        //  17.08
   zdialog_run(zd,edit_plugins_event,"mouse");
   return;
}


//  dialog event function

int edit_plugins_event(zdialog *zd, cchar *event)
{
   using namespace plugins_names;

   int      ii, jj, cc, zstat;
   char     *pp, menuname[100], command[200];
   char     buff[200];
   FILE     *fid;

   if (strmatch(event,"menuname"))
   {
      zdialog_fetch(zd,"menuname",menuname,100);

      for (ii = 0; ii < Nplugins; ii++)                                          //  find selected menu name
      {
         pp = strstr(plugins[ii]," = ");
         if (! pp) continue;
         *pp = 0;
         jj = strmatch(menuname,plugins[ii]);
         *pp = ' ';
         if (jj) {
            zdialog_stuff(zd,"command",pp+3);                                    //  stuff corresp. command in dialog
            break;
         }
      }

      return 1;
   }

   zstat = zd->zstat;                                                            //  wait for dialog completion
   if (! zstat) return 1;

   if (zstat < 1 || zstat > 3) {                                                 //  cancel
      zdialog_free(zd);
      return 1;
   }

   if (zstat == 1)                                                               //  add plugin or replace same menu name
   {
      zd->zstat = 0;                                                             //  keep dialog active

      if (Nplugins == maxplugins) {
         zmessageACK(Mwin,"too many plugins");
         return 1;
      }

      zdialog_fetch(zd,"menuname",menuname,100);
      zdialog_fetch(zd,"command",command,200);

      pp = strstr(command," %s");
      if (! pp) zmessageACK(Mwin,"Warning: command without \"%%s\" ");

      for (ii = 0; ii < Nplugins; ii++)                                          //  find existing plugin record
      {
         pp = strstr(plugins[ii]," = ");
         if (! pp) continue;
         *pp = 0;
         jj = strmatch(menuname,plugins[ii]);
         *pp = ' ';
         if (jj) break;
      }

      if (ii == Nplugins) {                                                      //  new plugin record
         plugins[ii] = 0;
         Nplugins++;
         zdialog_cb_app(zd,"menuname",menuname);
         pp = zdialog_cb_get(zd,"menuname",ii);
      }

      if (plugins[ii]) zfree(plugins[ii]);
      cc = strlen(menuname) + strlen(command) + 4;
      plugins[ii] = (char *) zmalloc(cc);
      *plugins[ii] = 0;
      strncatv(plugins[ii],cc,menuname," = ",command,0);
   }

   if (zstat == 2)                                                               //  remove current plugin
   {
      zd->zstat = 0;                                                             //  keep dialog active

      zdialog_fetch(zd,"menuname",menuname,100);

      for (ii = 0; ii < Nplugins; ii++)                                          //  find existing plugin record
      {
         pp = strstr(plugins[ii]," = ");
         if (! pp) continue;
         *pp = 0;
         jj = strmatch(menuname,plugins[ii]);
         *pp = ' ';
         if (jj) break;
      }

      if (ii == Nplugins) return 1;                                              //  not found

      zfree(plugins[ii]);                                                        //  remove plugin record
      Nplugins--;
      for (jj = ii; jj < Nplugins; jj++)
         plugins[jj] = plugins[jj+1];
      zdialog_cb_delete(zd,"menuname",menuname);                                 //  delete entry from combo box
      zdialog_stuff(zd,"menuname","");
      zdialog_stuff(zd,"command","");
   }

   if (zstat == 3)                                                               //  done
   {
      snprintf(buff,199,"%s/plugins",get_zhomedir());                            //  open file for plugins
      fid = fopen(buff,"w");
      if (! fid) {
         zmessageACK(Mwin,strerror(errno));
         return 1;
      }

      for (int ii = 0; ii < Nplugins; ii++)                                      //  save plugins
         if (plugins[ii])
            fprintf(fid,"%s \n",plugins[ii]);

      fclose(fid);

      zdialog_free(zd);
      popup_plugmenu = 0;                                                        //  rebuild popup menu
   }

   return 1;
}


//  process plugin menu selection
//  execute correspinding command using current image file

void m_run_plugin(GtkWidget *, cchar *menu)
{
   using namespace plugins_names;

   int         ii, jj, err;
   char        *pp = 0, plugincommand[200], pluginfile[100];
   PXM         *pxmtemp;
   zdialog     *zd = 0;

   F1_help_topic = "plugins";

   for (ii = 0; ii < Nplugins; ii++)                                             //  search plugins for menu name
   {
      pp = strchr(plugins[ii],'=');                                              //  match menu name to plugin command
      if (! pp) continue;                                                        //  menu name = ...
      *pp = 0;
      jj = strmatch(plugins[ii],menu);
      *pp = '=';
      if (jj) break;
   }

   if (ii == Nplugins) {
      zmessageACK(Mwin,"plugin menu not found %s",menu);
      return;
   }

   strncpy0(plugincommand,pp+1,200);                                             //  corresp. command
   strTrim2(plugincommand);

   pp = strstr(plugincommand,"%s");                                              //  no file placeholder in command
   if (! pp) {
      err = shell_ack(plugincommand);                                            //  execute non-edit plugin command
      goto RETURN;
   }
   
   EFplugin.menufunc = m_run_plugin;
   EFplugin.funcname = menu;
   if (! edit_setup(EFplugin)) return;                                           //  start edit function

   snprintf(pluginfile,100,"%s/plugfile.tif",tempdir);                           //  /tmp/fotoxx-nnnnnn/plugfile.tif

   err = PXM_TIFF_save(E1pxm,pluginfile,8);                                      //  E1 >> plugin_file
   if (err) {
      if (*file_errmess)                                                         //  pass error to user
         zmessageACK(Mwin,file_errmess);
      goto FAIL;
   }

   zd = zmessage_post(Mwin,0,ZTX("Plugin working ..."));

   repl_1str(plugincommand,command,"%s",pluginfile);                             //  command filename

   err = shell_ack(command);                                                     //  execute plugin command
   if (err) goto FAIL;
   
   pxmtemp = TIFF_PXM_load(pluginfile);                                          //  read command output file
   if (! pxmtemp) {
      zmessageACK(Mwin,ZTX("plugin failed"));
      goto FAIL;
   }

   PXM_free(E3pxm);                                                              //  plugin_file >> E3
   E3pxm = pxmtemp;

   CEF->Fmods = 1;                                                               //  assume image was modified
   CEF->Fsaved = 0;
   edit_done(0);
   goto RETURN;

FAIL:
   edit_cancel(0);

RETURN:
   if (zd) zdialog_free(zd);
   return;
}



