//**************************************************************************************************
//                                          HelpTasks.cpp                                          *
//                                         ---------------                                         *
//  Started     : 2005-06-03                                                                       *
//  Last Update : 2015-01-28                                                                       *
//  Copyright   : (C) 2005 by MSWaters                                                             *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "HelpTasks.hpp"
#include "FrmMain.hpp"

// The application icons
#include "icons/gspiceui-32x32.xpm"
#include "icons/html-forward.xpm"
#include "icons/html-back.xpm"
#include "icons/html-close.xpm"

//**************************************************************************************************
// Implement an event table in which the events are routed to their respective handler functions in
// the class. If -1 is given as the ID, the given handler will be invoked for any event of the
// specified type.

wxBEGIN_EVENT_TABLE( HelpTasks, wxFrame )

  EVT_TOOL( HelpTasks::ID_TBR_FORWARD, HelpTasks::OnForward )
  EVT_TOOL( HelpTasks::ID_TBR_BACK   , HelpTasks::OnBack    )
  EVT_TOOL( HelpTasks::ID_TBR_CLOSE  , HelpTasks::OnClose   )

  EVT_CLOSE(                           HelpTasks::OnClose   )

wxEND_EVENT_TABLE( )

//**************************************************************************************************
// Constructor.
//
// Argument List :
//   poFrmMain - A pointer to the parent frame

HelpTasks::HelpTasks( FrmMain * poFrmMain ) :
                  wxFrame( poFrmMain, -1, wxT(""), wxDefaultPosition,
                           wxDefaultSize, wxDEFAULT_FRAME_STYLE ),
                  m_oHtmlWin( this )
{
  // Set the pointer to the parent frame
  m_poFrmMain = poFrmMain;

  // Get the global configuration object
  m_poCfg = (wxConfig *) wxConfig::Get( );

  // Initialize the help viewer
  Initialize( );
}

//**************************************************************************************************
// Destructor.

HelpTasks::~HelpTasks( )
{
}

//**************************************************************************************************
// Initialize the main frame.

void  HelpTasks::Initialize( void )
{
  // Set the frame icon
  SetIcon( wxICON( gspiceui32x32 ) );

  // Set the frame title
  SetTitle( );

  // Call all the initialization functions
  InitToolBar( );
  InitHtmlWin( );
  InitPosnSize( );
}

//**************************************************************************************************
// Initialize the tool bar.

void  HelpTasks::InitToolBar( void )
{
  wxBitmap  * poPixMap[ 3 ];
  wxToolBar * poToolBar;

  // Create the tool bar
  poToolBar = CreateToolBar( wxHORIZONTAL | wxTB_FLAT );

  // Create the bitmaps for the tools
  poPixMap[ 0 ] = new wxBitmap( html_back_xpm    );
  poPixMap[ 1 ] = new wxBitmap( html_forward_xpm );
  poPixMap[ 2 ] = new wxBitmap( html_close_xpm   );

  // Add the tools to the toolbar
  poToolBar->AddTool( ID_TBR_BACK   , wxT(""), *(poPixMap[ 0 ]),
                      wxT("Back") );
  poToolBar->AddTool( ID_TBR_FORWARD, wxT(""), *(poPixMap[ 1 ]),
                      wxT("Forward") );
  poToolBar->AddTool( ID_TBR_CLOSE  , wxT(""), *(poPixMap[ 2 ]),
                      wxT("Close") );

  // Realize the toolbar
  poToolBar->Realize( );

  // Can delete the bitmaps since they're reference counted
  for( int i1=0; i1<3; i1++ ) delete poPixMap[ i1 ];
}

//**************************************************************************************************
// Initialize the frames position amd size.

void  HelpTasks::InitHtmlWin( void )
{
  // Enable display of image types
  wxInitAllImageHandlers( );

  // Set suitable wxHtmlWindow font sizes
  int  ia[ 7 ]={ 9,10,11,14,16,18,20 };
  m_oHtmlWin.SetFonts( wxT(""), wxT(""), ia );
}

//**************************************************************************************************
// Initialize the frames position amd size.

void  HelpTasks::InitPosnSize( void )
{
  int  ix, iy, iw, ih;

  // Set the position and size from the configuration object
  m_poCfg->SetPath( wxT("/Help") );
  ix = m_poCfg->Read( wxT("PosnX"), 500 );
  iy = m_poCfg->Read( wxT("PosnY"),   1 );
  iw = m_poCfg->Read( wxT("SizeW"), 500 );
  ih = m_poCfg->Read( wxT("SizeH"), 700 );

  if( ix>0 && iy>0 ) Move( ix, iy );
  if( iw>0 && ih>0 ) SetClientSize( iw, ih );
}

//**************************************************************************************************
// Determine the path to the applications base installation point.
//
// Return Values :
//   Success - The path to the installation point
//   Failure - An empty string

wxString & HelpTasks::rosGetInstallPath( void )
{
  static  wxString  osPath;
  wxPathList  opl1;
  wxFileName  ofn1;
  wxString    os1;

  osPath.Empty( );

  // Get the command used to envoke this process
  ofn1 = wxTheApp->argv[ 0 ];

  if( ! ofn1.GetPath( ).IsEmpty( ) )
  { // Extract the desired part of the path
    ofn1.Normalize( );  // Expand abbreviations eg. "..", "~', etc.
  }
  else
  { // Search env. var. PATH for the first occurrence of the app. name
    opl1.AddEnvList( wxT("PATH") );
    os1 = opl1.FindAbsoluteValidPath( ofn1.GetFullName( ) );
    ofn1 = os1;
  }

  osPath = ofn1.GetPath( );
  if( osPath.AfterLast( wxT('/') ) == wxT("bin") )
    osPath = osPath.BeforeLast( wxT('/') );

  return( osPath );
}

//**************************************************************************************************
// Set the help frame title.

void  HelpTasks::SetTitle( void )
{
  wxString  os1;

  os1 = wxT("GNU Spice GUI - User Manual");

  if( ! m_oHtmlWin.GetOpenedPage( ).IsEmpty( ) )
    os1 << wxT("  -  ") << m_oHtmlWin.GetOpenedPage( );

  wxFrame::SetTitle( os1 );
}

//**************************************************************************************************
// Display the gSpiceUI User Manual.

void  HelpTasks::ManualUser( void )
{
  wxFileName  ofn1, ofn2;
  wxString    os1;

  // Check if the manual has been loaded into memory
  if( m_oHtmlWin.GetOpenedPage( ).IsEmpty( ) )
  {
    // Specify the two locations to search
    os1 = rosGetInstallPath( );
    if( os1.IsEmpty( ) ) return;
    ofn1 = os1 + wxT("/share/gspiceui/html/User-Manual.html");
    ofn2 = os1 + wxT("/html/User-Manual.html");

    // Search for the manual
    if(      ofn1.FileExists( ) ) os1 = ofn1.GetFullPath( );
    else if( ofn2.FileExists( ) ) os1 = ofn2.GetFullPath( );
    else
    {
      // Display an error message
      os1.Empty( );
      os1 << wxT("\nThe base manual page \"") << ofn1.GetFullName( ) << wxT('\"')
          << wxT(" couldn't be found in either of the following locations :\n\n")
          << wxT("     ") << ofn1.GetPath( ) << wxT("/\n")
          << wxT("     ") << ofn2.GetPath( ) << wxT("/\n");
      wxMessageBox( os1, wxT("User Manual"), wxOK | wxCENTRE, m_poFrmMain );
      return;
    }

    // Load the manual
    m_oHtmlWin.LoadPage( os1 );
    SetTitle( );
  }

  // Show the manual
  Show( true );
}

//**************************************************************************************************
// Display the about message dialog.

void  HelpTasks::About( void )
{
  wxString  os1;
  long      lStyle;

  os1 << wxT("\n                                     " ) << APP_NAME
      << wxT("\n                     Version ")          << APP_VERSION
                                            << wxT(" (") << APP_DATE << wxT(")")
      << wxT("\n                    ")                   << APP_COPYRIGHT
      << wxT("\n\n")

      << wxT("\nThis application is intended to provide a GUI for various")
      << wxT("\nfreely available electronic circuit simulation engines :\n")
      << wxT("\n                 - NG-Spice")
      << wxT("\n                 - GNU-Cap\n")

      << wxT("\nSchematic files are imported using gnetlist and waveform     ")
      << wxT("\ndata can be viewed using either ") << CLP_GWAVE << wxT(" or ")
      << CLP_GAW << wxT(".\n")

      << wxT("\nThis application is written in C++ and uses the wxWidgets")
      << wxT("\nlibrary which is a free and open source widget toolkit and")
      << wxT("\ntools library. wxWidgets version ")
      << wxMAJOR_VERSION << '.' << wxMINOR_VERSION << '.' << wxRELEASE_NUMBER
      << wxT(" was used in this")
      << wxT("\nbuild of ") << APP_NAME << wxT(".\n")

      << wxT("\n") << APP_NAME << wxT(" is free software; you can redistribute it and/or")
      << wxT("\nmodify it under the terms of the GNU Library General")
      << wxT("\nPublic Licence as published by the Free Software")
      << wxT("\nFoundation; either version 3 of the Licence, or (at your")
      << wxT("\noption) any later version.\n")

      << wxT("\n") << APP_NAME << wxT(" is distributed in the hope that it will be useful,")
      << wxT("\nbut WITHOUT ANY WARRANTY; without even the implied")
      << wxT("\nwarranty of MERCHANTABILITY or FITNESS FOR A")
      << wxT("\nPARTICULAR PURPOSE.");

  lStyle = wxOK | wxICON_INFORMATION;

  wxMessageBox( os1, wxT("About gSpiceUI"), lStyle, m_poFrmMain );
}

//**************************************************************************************************
// Go to the next page in the history store.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  HelpTasks::OnForward( wxCommandEvent & WXUNUSED( roEvtCmd ) )
{
  if( m_oHtmlWin.HistoryCanForward( ) )
    m_oHtmlWin.HistoryForward( );
}

//**************************************************************************************************
// Go to the previous page in the history store.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  HelpTasks::OnBack( wxCommandEvent & WXUNUSED( roEvtCmd ) )
{
  if( m_oHtmlWin.HistoryCanBack( ) )
    m_oHtmlWin.HistoryBack( );
}

//**************************************************************************************************
// Close the help viewer frame.
//
// Argument List :
//   roEvtCmd - The event to be processed

void  HelpTasks::OnClose( wxCommandEvent & WXUNUSED( roEvtCmd ) )
{
  int  ix, iy, iw, ih;

  // Save the frame size and position
  GetPosition( &ix, &iy );
  GetClientSize( &iw, &ih );
  m_poCfg->SetPath( wxT("/Help") );
  m_poCfg->Write( wxT("PosnX"), (long) ix );
  m_poCfg->Write( wxT("PosnY"), (long) iy );
  m_poCfg->Write( wxT("SizeW"), (long) iw );
  m_poCfg->Write( wxT("SizeH"), (long) ih );
  m_poCfg->Flush( );

  Show( false );  // Hide the frame

  // Reset the HTML Window settings
  while( m_oHtmlWin.HistoryCanBack( ) )
    m_oHtmlWin.HistoryBack( );
  m_oHtmlWin.HistoryClear( );
  m_oHtmlWin.Scroll( 0, 0 );
}

//**************************************************************************************************
// Close the help viewer frame.
//
// Argument List :
//   roEvtClose - The event to be processed

void  HelpTasks::OnClose( wxCloseEvent & WXUNUSED( roEvtClose ) )
{
  wxCommandEvent  oEvtCmd;

  OnClose( oEvtCmd );
}

//**************************************************************************************************
