/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icamerapath.h"


#include "iactor.h"
#include "icontrolmodule.h"
#include "ierror.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkCellPicker.h>
#include <vtkConeSource.h>
#include <vtkMath.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkSplineWidget.h>

//
//  Helper class
//
class iSplineWidget : public vtkSplineWidget
{

public:

	iSplineWidget()
	{
		this->HandlePicker->SetTolerance(0.01);
		this->LinePicker->SetTolerance(0.01);
	}

	virtual int GetCurrentHandleIndex(){ return CurrentHandleIndex; }
};


//
//  Main class
//
iCameraPath* iCameraPath::New(iViewModule *vm)
{ 
	IERROR_ASSERT(vm);
	return new iCameraPath(vm);
}


iCameraPath::iCameraPath(iViewModule *vm) : iEventObserver(vm->GetControlModule()->GetShell()), mViewModule(vm), mNumIntervalsPerSegment(5)
{
	mIsFocalPath = mIsFocalPathToPoint = false;

	mCamPathWidget = new iSplineWidget; IERROR_ASSERT(mCamPathWidget);
	mCamPathWidget->SetNumberOfHandles(3);
	mCamPathWidget->SetResolution(mNumIntervalsPerSegment*2);
	mCamPathWidget->SetInteractor(this->GetViewModule()->GetInteractor());
	mCamPathWidget->GetLineProperty()->SetLineWidth(4.0);
	mCamPathWidget->GetLineProperty()->SetColor(0.0,0.0,0.0);
	mCamPathWidget->SetHandlePosition(2,-1.0,0.0,1.0);
	mCamPathWidget->SetHandlePosition(1,0.0,0.0,1.0);
	mCamPathWidget->SetHandlePosition(0,1.0,0.0,1.0);

	mFocPathWidget = new iSplineWidget; IERROR_ASSERT(mFocPathWidget);
	mFocPathWidget->SetNumberOfHandles(3);
	mFocPathWidget->SetResolution(mNumIntervalsPerSegment*2);
	mFocPathWidget->SetInteractor(this->GetViewModule()->GetInteractor());
	mFocPathWidget->GetLineProperty()->SetLineWidth(4.0);
	mFocPathWidget->GetLineProperty()->SetColor(0.4,0.5,0.6);
	mFocPathWidget->SetHandlePosition(2,-0.3,0.0,0.0);
	mFocPathWidget->SetHandlePosition(1,0.0,0.0,0.0);
	mFocPathWidget->SetHandlePosition(0,0.3,0.0,0.0);

	mCamPathWidget->AddObserver(vtkCommand::InteractionEvent,this);
	mCamPathWidget->AddObserver(vtkCommand::EnableEvent,this);
	mCamPathWidget->AddObserver(vtkCommand::DisableEvent,this);
	mFocPathWidget->AddObserver(vtkCommand::InteractionEvent,this);
	mFocPathWidget->AddObserver(vtkCommand::EnableEvent,this);
	mFocPathWidget->AddObserver(vtkCommand::DisableEvent,this);

	mCamPathData = vtkPolyData::New(); IERROR_ASSERT(mCamPathData);
	mFocPathData = vtkPolyData::New(); IERROR_ASSERT(mFocPathData);
		
	mCamPathHeadActor = iActor::New(true); IERROR_ASSERT(mCamPathHeadActor);
	mCamPathHeadSource = vtkConeSource::New(); IERROR_ASSERT(mCamPathHeadSource);
	mCamPathHeadSource->SetResolution(12);
	mCamPathHeadSource->SetRadius(0.1);
	mCamPathHeadSource->SetHeight(0.15);
	mCamPathHeadActor->SetInput(mCamPathHeadSource->GetOutput());
	mCamPathHeadActor->GetProperty()->SetColor(0.0,0.68,1.0);
	mCamPathHeadActor->PickableOff();

	mFocPathBandActor = iActor::New(); IERROR_ASSERT(mFocPathBandActor);
	mFocPathBandSource = vtkPolyData::New(); IERROR_ASSERT(mFocPathBandSource);
	mFocPathBandActor->SetInput(mFocPathBandSource);
	mFocPathBandActor->GetProperty()->SetColor(0.4,0.5,0.6);
	mFocPathBandActor->PickableOff();

	mDxPrev[0] = 1.0; mDxPrev[1] = mDxPrev[2] = 0.0;
}


iCameraPath::~iCameraPath()
{
	this->SetEnabled(false);

	mCamPathData->Delete();
	mFocPathData->Delete();
	mCamPathHeadActor->Delete();
	mCamPathHeadSource->Delete();
	mFocPathBandActor->Delete();
	mFocPathBandSource->Delete();
	mCamPathWidget->Delete();
	mFocPathWidget->Delete();
}


void iCameraPath::SetEnabled(bool s)
{
	if(s && !this->GetEnabled())
	{
		mCamPathWidget->SetEnabled(1);
		mFocPathWidget->SetEnabled(mIsFocalPath?1:0);
		this->GetViewModule()->AddObject(mCamPathHeadActor);
		this->GetViewModule()->AddObject(mFocPathBandActor);
		mFocPathBandActor->SetVisibility(mIsFocalPath?1:0);
		mCamPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
		if(mIsFocalPath) mFocPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
	}
	if(!s && this->GetEnabled())
	{
		mCamPathWidget->SetEnabled(0);
		mFocPathWidget->SetEnabled(0);
		this->GetViewModule()->RemoveObject(mCamPathHeadActor);
		this->GetViewModule()->RemoveObject(mFocPathBandActor);
	}
}


bool iCameraPath::GetEnabled() const
{
	return mCamPathWidget->GetEnabled() != 0;
}


void iCameraPath::SetNumberOfHandles(int v)
{
	if(v > 1)
	{
		mCamPathWidget->SetNumberOfHandles(v);
		mCamPathWidget->SetResolution(mNumIntervalsPerSegment*(v-1));
		if(mIsFocalPath)
		{
			mFocPathWidget->SetNumberOfHandles(v);
			mFocPathWidget->SetResolution(mNumIntervalsPerSegment*(v-1));
			mFocPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
		}
	}
}

	
int iCameraPath::GetNumberOfHandles() const
{
	return mCamPathWidget->GetNumberOfHandles();
}
	

//int iCameraPath::GetNumberOfIntervals() const
//{
//	return mCamPathWidget->GetResolution();
//}


void iCameraPath::SetClosed(bool v)
{
	if(v) 
	{
		mCamPathWidget->ClosedOn(); 
		mFocPathWidget->ClosedOn(); 
	}
	else 
	{
		mCamPathWidget->ClosedOff();
		mFocPathWidget->ClosedOff();
	}
	if(mIsFocalPath) mFocPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
}


bool iCameraPath::GetClosed() const
{
	return mCamPathWidget->GetClosed() != 0;
}


void iCameraPath::SetFocalPathEnabled(bool s)
{
	mIsFocalPath = s;
	if(!this->GetEnabled()) return;

	if(s)
	{
		if(mCamPathWidget->GetResolution() != mFocPathWidget->GetResolution()) mFocPathWidget->SetResolution(mCamPathWidget->GetResolution());
		if(mCamPathWidget->GetNumberOfHandles() != mFocPathWidget->GetNumberOfHandles()) mFocPathWidget->SetNumberOfHandles(mCamPathWidget->GetNumberOfHandles());
		mFocPathWidget->SetEnabled(1);
		mFocPathBandActor->VisibilityOn();
		mFocPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
	}
	else
	{
		mFocPathWidget->SetEnabled(0);
		mFocPathBandActor->VisibilityOff();
	}
}


void iCameraPath::SetFocalPathToPoint(bool s)
{ 
	mIsFocalPathToPoint = s;
	if(mIsFocalPath)
	{
		if(!s)
		{
			int i;
			double x[3], sf;
			mFocPathWidget->GetHandlePosition(0,x);
			sf = 0.6/(mFocPathWidget->GetNumberOfHandles()-1);
			for(i=1; i<mFocPathWidget->GetNumberOfHandles(); i++) 
			{
				x[0] -= sf;
				mFocPathWidget->SetHandlePosition(i,x);
			}
		}
		mFocPathWidget->InvokeEvent(vtkCommand::InteractionEvent);
	}
}


void iCameraPath::SetLineColor(iColor &c)
{
	mCamPathWidget->GetLineProperty()->SetColor(c.ToVTK());
}


void iCameraPath::SetHandleOpacity(float v)
{
	mCamPathWidget->GetHandleProperty()->SetOpacity(v);
}


void iCameraPath::ExecuteBody(vtkObject *caller, unsigned long eventId, void *)
{
	int i;

	switch (eventId)
	{
	case vtkCommand::InteractionEvent:
		{
			vtkPoints *cp = this->GetCameraPathPoints();

			//
			//  Camera path head
			//
			if(caller == mCamPathWidget)
			{
				double q1, x1[3], x2[3], dx[3], ax[3];
				int i2;
								
				cp->GetPoint(0,x1);
				i2 = 1;
				while(i2 < mCamPathData->GetNumberOfPoints())
				{
					cp->GetPoint(i2,x2);
					for(i=0; i<3; i++) dx[i] = x2[i] - x1[i];
					if(vtkMath::Norm(dx) > 0) break;
					i2++;
				}
				vtkMath::Normalize(dx);

				//
				//  Rotate
				//
				mCamPathHeadActor->SetOrientation(0.0,0.0,0.0);
				ax[0] = 0.5*(1+dx[0]); ax[1] = 0.5*dx[1]; ax[2] = 0.5*dx[2];
				if(vtkMath::Norm(ax) > 0.0)
				{
					mCamPathHeadActor->RotateWXYZ(180.0,ax[0],ax[1],ax[2]);
				}
				else
				{
					if(vtkMath::Dot(dx,mDxPrev) < 0.0)
					{
						//
						//  Flip
						//
						mCamPathHeadActor->RotateWXYZ(180.0,0.0,1.0,0.0);
					}
				}
				
				//
				//  Translate
				//
				q1 = mCamPathHeadSource->GetHeight();
				for(i=0; i<3; i++) dx[i] = x1[i] - q1*dx[i];
				mCamPathHeadActor->SetPosition(dx);
			}

			//
			//  Collapse focal path to point
			//
			if(caller==mFocPathWidget && mIsFocalPathToPoint)
			{
				double x[3];
				i = mFocPathWidget->GetCurrentHandleIndex();
				if(i<0 || i>=mFocPathWidget->GetNumberOfHandles()) i = 0;
				mFocPathWidget->GetHandlePosition(i,x);
				for(i=0; i<mFocPathWidget->GetNumberOfHandles(); i++) mFocPathWidget->SetHandlePosition(i,x);
			}

			//
			//  Focal path band
			//
			if(mIsFocalPath)
			{
				vtkPoints *fp = this->GetFocalPathPoints();
				int n = cp->GetNumberOfPoints();
				int nstep = 1;
				if(n > 32) 
				{
					n = 32;
					nstep = cp->GetNumberOfPoints()/n;
				}

				vtkPoints *newPoints = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(newPoints);
				vtkCellArray *newLines = vtkCellArray::New(); IERROR_ASSERT(newLines);
				newPoints->SetNumberOfPoints(2*n);
				newLines->Allocate(3*n);

				vtkIdType cell[2];
				for(i=0; i<n; i++)
				{
					newPoints->SetPoint(2*i+0,cp->GetPoint(i*nstep));
					newPoints->SetPoint(2*i+1,fp->GetPoint(i*nstep));
					cell[0] = 2*i; cell[1] = 2*i + 1;
					newLines->InsertNextCell(2,cell);
				}

				mFocPathBandSource->SetPoints(newPoints);
				newPoints->Delete();

				mFocPathBandSource->SetLines(newLines);
				newLines->Delete();
			}
			break;
		}
	case vtkCommand::EnableEvent:
		{
			if(caller == mCamPathWidget) mCamPathHeadActor->VisibilityOn();
			if(caller == mFocPathWidget) mFocPathBandActor->VisibilityOn();
			break;
		}
	case vtkCommand::DisableEvent:
		{
			if(caller == mCamPathWidget) mCamPathHeadActor->VisibilityOff();
			if(caller == mFocPathWidget) mFocPathBandActor->VisibilityOff();
			break;
		}
	}
}


void iCameraPath::SetCameraPathHandlePosition(int n, double x[3])
{
	if(n>=0 && n<mCamPathWidget->GetNumberOfHandles())
	{
		mCamPathWidget->SetHandlePosition(n,x);
	}
}


void iCameraPath::GetCameraPathHandlePosition(int n, double x[3]) const
{
	if(n>=0 && n<mCamPathWidget->GetNumberOfHandles())
	{
		mCamPathWidget->GetHandlePosition(n,x);
	}
}


void iCameraPath::SetFocalPathHandlePosition(int n, double x[3])
{
	if(n>=0 && n<mFocPathWidget->GetNumberOfHandles())
	{
		mFocPathWidget->SetHandlePosition(n,x);
	}
}


void iCameraPath::GetFocalPathHandlePosition(int n, double x[3]) const
{
	if(n>=0 && n<mFocPathWidget->GetNumberOfHandles())
	{
		mFocPathWidget->GetHandlePosition(n,x);
	}
}

	
//
//  Special forms for Animator packing/unpacking
//
void iCameraPath::SetCameraPathHandlePositions(int dir, const float *x)
{
	if(dir>=0 && dir<3)
	{
		double pos[3];
		int i, n = mCamPathWidget->GetNumberOfHandles();
		for(i=0; i<n; i++)
		{
			mCamPathWidget->GetHandlePosition(i,pos);
			pos[dir] = x[i];
			mCamPathWidget->SetHandlePosition(i,pos);
		}
	}
}


void iCameraPath::GetCameraPathHandlePositions(float *x1, float *x2, float *x3) const
{
	double pos[3];
	int i, n = mCamPathWidget->GetNumberOfHandles();
	for(i=0; i<n; i++)
	{
		mCamPathWidget->GetHandlePosition(i,pos);
		x1[i] = pos[0];
		x2[i] = pos[1];
		x3[i] = pos[2];
	}
}


void iCameraPath::SetFocalPathHandlePositions(int dir, const float *x)
{
	if(dir>=0 && dir<3)
	{
		double pos[3];
		int i, n = mFocPathWidget->GetNumberOfHandles();
		for(i=0; i<n; i++)
		{
			mFocPathWidget->GetHandlePosition(i,pos);
			pos[dir] = x[i];
			mFocPathWidget->SetHandlePosition(i,pos);
		}
	}
}


void iCameraPath::GetFocalPathHandlePositions(float *x1, float *x2, float *x3) const
{
	double pos[3];
	int i, n = mFocPathWidget->GetNumberOfHandles();
	for(i=0; i<n; i++)
	{
		mFocPathWidget->GetHandlePosition(i,pos);
		x1[i] = pos[0];
		x2[i] = pos[1];
		x3[i] = pos[2];
	}
}


vtkPoints* iCameraPath::GetCameraPathPoints()
{ 
	mCamPathWidget->GetPolyData(mCamPathData);
	return mCamPathData->GetPoints(); 
}


vtkPoints* iCameraPath::GetFocalPathPoints()
{ 
	if(mIsFocalPath)
	{
		mFocPathWidget->GetPolyData(mFocPathData);
		return mFocPathData->GetPoints();
	}
	else return 0; 
}
