/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
 *  \file pdl_policy.c
 *
 *  \brief Implementation of the pdl policies.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 17818 $
 *  \date    $Date: 2014-07-03 13:42:13 +0200 (Thu, 03 Jul 2014) $
 *
 */


/* Needed for NULL */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lcmaps_log.h"
#include "pdl.h"
#include "pdl_policy.h"
#include "pdl_rule.h"

static BOOL policies_reduced = FALSE;   /* Tell if reduce_policy() has been called. */

static policy_t *top_policy=0, *last_policy=0;

static int num_of_policies_to_evaluate = 0;
static char** policies_to_evaluate = NULL;

BOOL _lcmaps_add_policy(record_t* name, rule_t* rules);


/*!
 *  Allow or disallow the additions of rules depending on the
 *  argument. When for example a policy is defined for the second
 *  time, an error should be generated, but the parsing should still
 *  continue. However, no rules can be added to the policy as there
 *  is currently no policy defined.
 *
 *  \param allow TRUE if addition of new rules is allowd, FALSE
 *               otherwise.
 */
void lcmaps_allow_rules(BOOL allow)
{
  lcmaps_allow_new_rules(allow);
}


/*!
 *  Wrapper around the _add_policy(name) function.
 *
 *  When the _add_policy() call fails, this function cleans up the
 *  data structure allocated for holding information about the policy
 *  that was found. See _add_policy() for information about the kind
 *  of reasons it can fail.
 *
 *  \param policy  Name of the policy.
 *  \param rules List of associated rules for the policy.
 *
 */
void lcmaps_add_policy(record_t* policy, rule_t* rules)
{
  /*
   *  Do not switch the order of the and operator, because the
   *  _add_policy needs to be done always.
   */
  if (!_lcmaps_add_policy(policy, rules)) {
    lcmaps_free_rules(rules);     rules          = NULL;
    free(policy->string);  policy->string = NULL;
    free(policy);          policy         = NULL;

    lcmaps_set_yylval(0);
  }

  free(policy);
  policy = NULL;

  lcmaps_start_new_rules();
}


/*!
 *  Add a policy with its rules to the list of policies.
 *
 *  Before the policy name is actually added to list of policies, a
 *  check is made to see weather or not a policy by the same name exists.
 *  if it does, the policy name will not be added and an error message
 *  is displayed, letting the user know that the configuration file
 *  contains multiple policy rules with the same name.
 *
 *  \param name Name of the new policy.
 *  \param rules List of associated rules for the policy.
 *  \return TRUE, If the policy has been added successfully; FALSE otherwise.
 *
 */
BOOL _lcmaps_add_policy(record_t* name, rule_t* rules)
{
  policy_t *policy;

  if ((policy = lcmaps_find_policy(name->string))) {
    lcmaps_pdl_warning(PDL_ERROR, "policy '%s' already defined at line %d.\n",
	    name->string, policy->lineno);
    lcmaps_allow_rules(FALSE);
    return FALSE;
  }

  if ((policy = (policy_t *)malloc(sizeof(policy_t)))) {
    policy->name   = name->string;
    policy->rule   = rules;
    policy->lineno = name->lineno;
    policy->next   = 0;
    policy->prev   = last_policy;

    if (top_policy)
      last_policy->next = policy;
    else
      top_policy = policy;

    last_policy = policy;
  } else {
    lcmaps_pdl_warning(PDL_ERROR, "Out of memory; cannot add policy '%s'.\n",
	    name->string);
    return FALSE;
  }

  return TRUE;
}


/*!
 *  Remove a policy from the list of policies and free all associated
 *  resources of the policy.
 *
 *  \param policy Policy to be removed.
 *
 */
void lcmaps_remove_policy(record_t* policy)
{
  free(policy->string);  policy->string = NULL;
  free(policy);          policy         = NULL;
}


/*!
 *  Find a policy by its name
 *
 *  \param name Name of the policy to be found.
 *  \return The policy if a polict with name 'name' exists, 0 otherwise.
 *
 */ 
policy_t* lcmaps_find_policy(const char* name)
{
  policy_t* policy=top_policy;

  while (policy && strcmp(name, policy->name)!=0) {
    policy = policy->next;
  }

  return policy;  
}


/*!
 *  Check for recursions in the policy rules.
 *
 *  \return TRUE if at least one recustion has been found, FALSE
 *  otherwise.
 *
 */
BOOL lcmaps_check_policies_for_recursion(void)
{
  BOOL recursion = FALSE;
  policy_t* policy = lcmaps_get_policies();

  while (policy) {
    lcmaps_log_debug(3, "Checking policy '%s' for recursions.\n", policy->name);

    if (lcmaps_check_rule_for_recursion(policy->rule)) {
      recursion = TRUE;
      lcmaps_log_debug(3, "Recursions were found.\n");
    }
    else
      lcmaps_log_debug(3, "No recursions were found.\n");

    policy = policy->next;
  }

  return recursion;
}


/*!
 *  lcmaps_reduce_policies to its elemantry form, i.e. each policy has a list
 *  of rules which need to be reduced.
 *
 */
void lcmaps_reduce_policies(void)
{
  policy_t* policy = lcmaps_get_policies();

  while (policy) {
    rule_t* rule = policy->rule;
    lcmaps_set_top_rule(rule);

    while (rule) {
      lcmaps_reduce_rule(rule);
      rule = rule->next;
    }

    policy = policy->next;
  }

  policies_reduced = TRUE;
}



/*!
 *  Get the list of policies.
 *
 *  \return First policy in the list.
 *
 */
policy_t* lcmaps_get_policies(void)
{
  return top_policy;
}


/*!
 *  Display the policies and the rules associated with the policy.
 *
 */
void lcmaps_show_policies(void)
{
  policy_t* policy = top_policy;

  while (policy) {
    lcmaps_log_debug(3, "policy: %s\n", policy->name);
    lcmaps_show_rules(policy->rule);

    policy = policy->next;
  }
}


/*!
 *  Cleanup the policies.
 *
 *  The tree that had been built up might contain policy rules which do
 *  not contain any rules. This function removes all policy rules with
 *  an empty rule set.
 *
 */
void lcmaps_cleanup_policies(void)
{
  policy_t *tmp, *policy = top_policy;

  while (policy) {
    if (!policy->rule) {
      if (policy->prev) {
        policy->prev->next = policy->next;
      }
      else
      {
        /* move the top_policy to the new head of the list */
        top_policy = policy->next;
      }
      if (policy->next) {
        policy->next->prev = policy->prev;
      }
      tmp = policy;
      policy = policy->next;
      free (tmp);
    } else {
      policy = policy->next;
    }
  }
}


/*!
 *  Free all policies and their allocated resources.
 *
 */
void lcmaps_free_policies(void)
{
  policy_t* next_pol, *policy = top_policy;

  while (policy) {
    next_pol = policy->next;
    free(policy->name);         policy->name = NULL;
    lcmaps_free_rules(policy->rule);   policy->rule = NULL;
    free(policy);
    policy = next_pol;
  }

  top_policy = 0;
  /* When there is no policy defined, it does not make sense to have a top rule either. */
  lcmaps_set_top_rule(0);
}


/*!
 *  Determine if the current policy rule should be included for 
 *  evaluating policy rules. The policy rules are still parsed though.
 *  Any error in any policy rule will be reported, i.e. your config
 *  file should be valid regardless of the policy rules to be evaluated.
 *
 */
BOOL lcmaps_allowed_policy_rule(const char* label)
{
  BOOL allowed = TRUE;
  int i=0;

  if (num_of_policies_to_evaluate > 0)
  {
    allowed = FALSE;
    while (i<num_of_policies_to_evaluate) {
      if (strcmp(label, policies_to_evaluate[i]) == 0) {
        allowed = TRUE;
        break;
      }
      i++;
    }
  }

  return allowed;
}


/*!
 *
 */
void lcmaps_SetSetOfRules(int argc, char* argv[])
{
  num_of_policies_to_evaluate = argc;
  policies_to_evaluate = argv;
}

/*!
 *  Tell if the reduce_policy() call has been called.
 *
 *  \return TRUE if reduce_policy() has been called; FALSE otherwise.
 *
 */
BOOL lcmaps_policies_have_been_reduced(void)
{
  return policies_reduced;
}
