/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_runvars.c
    \brief  Extract variables that will be used by the plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This module takes the data that are presented to LCMAPS (the global credential and Job request) and extracts the
    variables that will be used by the plugins from it and stores them into a list.
    The interface to the LCMAPS module is composed of:
    -# lcmaps_extractRunVars(): takes the global credential and Job request and extracts run variables from them
    -# lcmaps_setRunVars(): adds run variables to a list
    -# lcmaps_getRunVars(): gets run variables from list
    \internal
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_config.h"

/* Needed for NULL */
#include <stdio.h>

/* LCMAPS includes */
#include "lcmaps_log.h"
#include "lcmaps_arguments.h"
#include "lcmaps_types.h"
#include "_lcmaps_runvars.h"
#include "_lcmaps_credential.h"

/******************************************************************************
                             Define constants
******************************************************************************/

/* 
 * Extended for GUMS and Job Repository to include a whole array of VOMS Data Structures
 *
    #ifdef LCMAPS_GSI_MODE
    #   define NUMBER_OF_RUNVARS ((int)16)
    #else
    #   define NUMBER_OF_RUNVARS ((int)12)
    #endif
 *
 */

/* Make sure these numbers match number of entries in lcmaps_argument_t */
#ifdef LCMAPS_GSI_MODE
#   define NUMBER_OF_RUNVARS ((int)20)
#else
#   define NUMBER_OF_RUNVARS ((int)15)
#endif


/******************************************************************************
                               Type definitions
******************************************************************************/

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static lcmaps_argument_t runvars_list[] = {
    { "user_dn"     ,           "char *"              ,  0,   NULL},
    { "job_request" ,           "lcmaps_request_t"    ,  0,   NULL},
    { "job_request" ,           "char *"              ,  0,   NULL},
    { "fqan_list"   ,           "char **"             ,  0,   NULL},
    { "nfqan"       ,           "int"                 ,  0,   NULL},
    { "mapcounter"  ,           "int"                 ,  0,   NULL},
    { "requested_uid",          "uid_t"               ,  0,   NULL},
    { "requested_pgid_list",    "gid_t *"             ,  0,   NULL},
    { "requested_npgid",        "int"                 ,  0,   NULL},
    { "requested_sgid_list",    "gid_t *"             ,  0,   NULL},
    { "requested_nsgid",        "int"                 ,  0,   NULL},
    { "requested_poolindex",    "char *"              ,  0,   NULL},
    /* Newly added to allow requesting target account in gsissh */
    { "requested_username",     "char *"              ,  0,   NULL},
    /* Newly added vo_data for GUMS and Job Repository */
    { "voms_data_list",         "lcmaps_vomsdata_t *" ,  0,   NULL},
    { "nvoms_data"    ,         "int"                 ,  0,   NULL},
#ifdef LCMAPS_GSI_MODE
    { "pem_string"    ,         "char *"              ,  0,   NULL},
    { "px509_cred"  ,           "X509 *"              ,  0,   NULL},
    { "px509_chain" ,           "STACK_OF(X509) *"    ,  0,   NULL},
    { "user_cred"   ,           "gss_cred_id_t"       ,  0,   NULL},
    { "gss_context" ,           "gss_ctx_id_t"        ,  0,   NULL},
#endif /* LCMAPS_GSI_MODE */
    { NULL          , NULL                            , -1,   NULL}
}; /*!< \internal */


/******************************************************************************
Function:       lcmaps_extractRunVars()
Description:    extract the variables from user credential that can be used by the plugins
Parameters:
                request:     the job request (RSL)
                lcmaps_cred: the credential presented by the user
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcmaps_extractRunVars(
        lcmaps_request_t request,
        lcmaps_cred_id_t lcmaps_cred
        )
    \brief extract the variables from user credential that can be used by the plugins
    
    This function takes the user credential and job request (in RSL) and extracts the
    information which is published in the runvars_list. These variables can be accessed
    by the plugins.

    \param request     the job request (RSL)
    \param lcmaps_cred the credential presented by the user

    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
int lcmaps_extractRunVars(
        lcmaps_request_t request,
        lcmaps_cred_id_t lcmaps_cred,
	char *req_username
)
{
    static lcmaps_request_t job_request;
    static lcmaps_cred_id_t lcmaps_credential;
    static char *requested_username;

    int number_of_runvars=0;
    const char * logstr = "lcmaps.mod-lcmaps_extractRunVars()";

    /* Get the number of variables
     */
    number_of_runvars = lcmaps_cntArgs(runvars_list);
    if (NUMBER_OF_RUNVARS != number_of_runvars)
    {
        lcmaps_log(LOG_ERR,"%s: conflict in number of run variables:\n", logstr);
        lcmaps_log(LOG_ERR,"%s: estimated = %d, defined = %d\n",
                   logstr, number_of_runvars, NUMBER_OF_RUNVARS);
        return 1;
    }
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Number of runvars: %d\n", logstr, NUMBER_OF_RUNVARS);
    lcmaps_log_debug(2,"%s: Address of runvars_list (first element): 0x%x\n", logstr, runvars_list);
#endif

    /* Save request en credential in static variables
     */
    job_request=request;
    lcmaps_credential=lcmaps_cred;
    requested_username=req_username;

    /* decompose request and credential
     * for each run variable do lcmaps_setRunVars(name, type, value)
     */
    /* Set user_dn */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"user_dn\": %s, address: 0x%x\n", logstr, lcmaps_credential.dn, &(lcmaps_credential.dn));
#endif
    if (lcmaps_setRunVars("user_dn" , "char *", (void *) &(lcmaps_credential.dn)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"user_dn\" variable\n", logstr);
        return 1;
    }

    /* Set the list of FQANs */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"fqan_list\", address: 0x%x\n",
                     logstr, &(lcmaps_credential.fqan));
#endif
    if (lcmaps_setRunVars("fqan_list" , "char **", (void *) &(lcmaps_credential.fqan)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"fqan_list\" variable\n", logstr);
        return 1;
    }
    /* Set the number of FQANs */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"nfqan\": %d, address: 0x%x\n",
                     logstr, lcmaps_credential.nfqan, &(lcmaps_credential.nfqan));
#endif
    if (lcmaps_setRunVars("nfqan" , "int", (void *) &(lcmaps_credential.nfqan)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"nfqan\" variable\n", logstr);
        return 1;
    }

#ifdef LCMAPS_GSI_MODE
    /* Set user_cred */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"user_cred\"\n", logstr);
#endif
    if (lcmaps_setRunVars("user_cred" , "gss_cred_id_t", (void *) &(lcmaps_credential.cred)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"user_cred\" variable\n", logstr);
        return 1;
    }

    /* Set security context */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"gss_context\"\n", logstr);
#endif
    if (lcmaps_setRunVars("gss_context" , "gss_ctx_id_t", (void *) &(lcmaps_credential.context)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"gss_context\" variable\n", logstr);
        return 1;
    }

    /* Set pointer to X509 proxy certificate */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"px509_cred\"\n", logstr);
#endif
    if (lcmaps_setRunVars("px509_cred" , "X509 *", (void *) &(lcmaps_credential.px509_cred)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"px509_cred\" variable\n", logstr);
        return 1;
    }

    /* Set pointer to X509 proxy certificate chain */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"px509_chain\"\n", logstr);
#endif
    if (lcmaps_setRunVars("px509_chain" , "STACK_OF(X509) *", (void *) &(lcmaps_credential.px509_chain)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"px509_chain\" variable\n", logstr);
        return 1;
    }

    /* Set pointer to PEM string of proxy certificate */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"pem_string\"\n", logstr);
#endif
    if (lcmaps_setRunVars("pem_string" , "char *", (void *) &(lcmaps_credential.pem_string)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"pem_string\" variable\n", logstr);
        return 1;
    }



#endif /* LCMAPS_GSI_MODE */

    /* Set job_request */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"job_request\" of type \"lcmaps_request_t\"\n", logstr);
#endif
    if (lcmaps_setRunVars("job_request" , "lcmaps_request_t", (void *) &job_request) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"job_request\" variable of type \"lcmaps_request_t\"\n", logstr);
        return 1;
    }
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"job_request\" of type \"char *\"\n", logstr);
#endif
    if (lcmaps_setRunVars("job_request" , "char *", (void *) &job_request) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"job_request\" variable of type \"char *\"\n", logstr);
        return 1;
    }

    /* Set the mapcounter */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"mapcounter\": %d, address: 0x%x\n",
                     logstr, lcmaps_credential.mapcounter, &(lcmaps_credential.mapcounter));
#endif
    if (lcmaps_setRunVars("mapcounter" , "int", (void *) &(lcmaps_credential.mapcounter)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"mapcounter\" variable\n", logstr);
        return 1;
    }

    /* Set the requested uid */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_uid\": %d, address: 0x%x\n",
                     logstr, (int) lcmaps_credential.requested_account.uid,
                     &(lcmaps_credential.requested_account.uid));
#endif
    if (lcmaps_setRunVars("requested_uid" , "uid_t",
        (void *) &(lcmaps_credential.requested_account.uid)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_uid\" variable\n", logstr);
        return 1;
    }

    /* Set the list of requested primary gids */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_pgid_list\", address: 0x%x\n",
                     logstr, &(lcmaps_credential.requested_account.pgid_list));
#endif
    if (lcmaps_setRunVars("requested_pgid_list" , "gid_t *",
        (void *) &(lcmaps_credential.requested_account.pgid_list)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_pgid_list\" variable\n", logstr);
        return 1;
    }
    /* Set the number of requested primary gids */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_npgid\": %d, address: 0x%x\n",
                     logstr, lcmaps_credential.requested_account.npgid,
                     &(lcmaps_credential.requested_account.npgid));
#endif
    if (lcmaps_setRunVars("requested_npgid" , "int",
        (void *) &(lcmaps_credential.requested_account.npgid)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_npgid\" variable\n", logstr);
        return 1;
    }

    /* Set the list of requested secondary gids */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_sgid_list\", address: 0x%x\n",
                     logstr, &(lcmaps_credential.requested_account.sgid_list));
#endif
    if (lcmaps_setRunVars("requested_sgid_list" , "gid_t *",
        (void *) &(lcmaps_credential.requested_account.sgid_list)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_sgid_list\" variable\n", logstr);
        return 1;
    }
    /* Set the number of requested secondary gids */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_nsgid\": %d, address: 0x%x\n",
                     logstr, lcmaps_credential.requested_account.nsgid,
                     &(lcmaps_credential.requested_account.nsgid));
#endif
    if (lcmaps_setRunVars("requested_nsgid" , "int",
        (void *) &(lcmaps_credential.requested_account.nsgid)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_nsgid\" variable\n", logstr);
        return 1;
    }

    /* Set the requested poolindex */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_poolindex\", address: 0x%x\n",
                     logstr, &(lcmaps_credential.requested_account.poolindex));
    lcmaps_log_debug(2,"lcmaps.mod-lcmaps_extractRunVars(): requested_poolindex value: %s\n",
        (lcmaps_credential.requested_account).poolindex ? (lcmaps_credential.requested_account).poolindex : "(null)" );
#endif
    if (lcmaps_setRunVars("requested_poolindex" , "char *",
        (void *) &(lcmaps_credential.requested_account.poolindex)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_poolindex\" variable\n", logstr);
        return 1;
    }

    /* Set the requested username */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"requested_username\", address: 0x%x\n",
                     logstr, &(requested_username));
    lcmaps_log_debug(2,"lcmaps.mod-lcmaps_extractRunVars(): requested_username value: %s\n",
        requested_username ? requested_username : "(null)" );
#endif
    if (lcmaps_setRunVars("requested_username" , "char *",
        (void *) &(requested_username)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"requested_username\" variable\n", logstr);
        return 1;
    }

    /* Newly added vo_data for GUMS and Job Repository */

    /* Set pointer to multiple VOMS Data structures */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"voms_data_list\"\n", logstr);
#endif
    if (lcmaps_setRunVars("voms_data_list" , "lcmaps_vomsdata_t *", (void *) &(lcmaps_credential.voms_data_list)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"voms_data_list\" variable\n", logstr);
        return 1;
    }

    /* Set number of VOMS Data structures */
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(2,"%s: Setting \"nvoms_data\"\n", logstr);
#endif
    if (lcmaps_setRunVars("nvoms_data" , "int", (void *) &(lcmaps_credential.nvoms_data)) != 0)
    {
        lcmaps_log(LOG_ERR,"%s: error while setting \"nvoms_data\" variable\n", logstr);
        return 1;
    }

    return 0;
}

/******************************************************************************
Function:       lcmaps_getRunVars()
Description:    returns a void pointer to the requested value
Parameters:
                argName: name of the variable
                argType: type of the variable

Returns:        void pointer to the value or NULL
******************************************************************************/
/*!
    \fn lcmaps_getRunVars(
        const char *argName,
        const char *argType
        )
    \brief returns a void pointer to the requested value
    
    This function returns a void pointer to the requested variable with name
    argName and type argType in the runvars_list.
    Internally it uses lcmaps_getArgValue().

    \param argName  name of the variable
    \param argType  type of the variable

    \returns void pointer to the value or NULL
    \internal
*/
void * lcmaps_getRunVars(
        const char *argName,
        const char *argType
)
{
    /* do lcmaps_getArgValue with runvars_list and NUMBER_OF_RUNVARS
     */
    return lcmaps_getArgValue(argName, argType, NUMBER_OF_RUNVARS, runvars_list);
}

/******************************************************************************
Function:       lcmaps_setRunVars()
Description:    fill the runvars_list with a value for argName and argType
Parameters:
                argName: name of the runvars variable
                argType: type of the runvars variable
                values:  void pointer to the value

Returns:        0 succes
                -1 failure
******************************************************************************/
/*!
    \fn lcmaps_setRunVars(
        const char *argName,
        const char *argType,
        void *value
        )
    \brief fill the runvars_list with a value for argName and argType
    
    This function fills the (internal) runvars_list with the value for the variable with
    name argName and type argType.
    Internally lcmaps_setArgValue() is used.

    \param argName  name of the runvars variable
    \param argType  type of the runvars variable
    \param value    void pointer to the value

    \retval 0 succes.
    \retval -1 failure.
    \internal
*/
int lcmaps_setRunVars(
        const char *argName,
        const char *argType,
        void *value
)
{
    lcmaps_argument_t *pargument=NULL;
#ifdef LCMAPS_DEBUG
    const char *  logstr = "lcmaps.mod-lcmaps_setRunVars()";
#endif

    /* store address of 1st element of runvars_list in pargument,
     * the address of which can be passed on to lcmaps_setArgValue
     * (the address of runvars_list does not exist, since its only a symbol, not a value)
     */
    pargument=runvars_list;

#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(3,"%s: Address of first element of runvars_list: 0x%x\n", logstr, runvars_list);
    lcmaps_log_debug(3,"%s: Address of address of first element of runvars_list: 0x%x\n", logstr, &pargument);
#endif

    return lcmaps_setArgValue(argName, argType, value, NUMBER_OF_RUNVARS, &pargument);
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/lcmaps_runvars.c,v $
    $Date: 2014-03-13 11:56:06 +0100 (Thu, 13 Mar 2014) $
    $Revision: 17647 $
    $Author: msalle $
******************************************************************************/
