//
//  Lynkeos
//  $Id: MyImageListWindowPrivate.m,v 1.8 2005/02/01 23:01:25 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Fri Oct 1 2004.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include "MyDocument.h"

#include "MyImageListWindowPrivate.h"

@implementation MyImageListWindow(Private)

- (NSString*) windowStateName
{
   switch( _windowState )
   {
      case ListMode: return( @"ListMode" );
      case AlignMode: return( @"AlignMode" );
      case Aligning: return( @"Aligning" );
      case AnalyzeMode: return( @"AnalyzeMode" );
      case Analyzing: return( @"Analyzing" );
      case StackMode: return( @"StackMode" );
      case Stacking: return( @"Stacking" );
      case ProcessMode: return( @"ProcessMode" );
      default: return( [NSString stringWithFormat:@"Unknown state : %d", 
                                 _windowState] );
   }
}

- (NSString*) windowModeName
{
   switch( _windowMode )
   {
      case DarkFrameMode: return( @"DarkFrameMode" );
      case FlatFieldMode: return( @"FlatFieldMode" );
      case ImageMode: return( @"ImageMode" );
      default: return( [NSString stringWithFormat:@"Unknown mode : %d", 
                                 _windowMode] );
   }
}

- (void) highlightOther :(BOOL)direct
{
   MyImageListEnumerator *list;
   MyImageListItem *item;

   if ( _windowState == ListMode || _windowState == AlignMode 
        || _windowState == AnalyzeMode )
   {
      list = [_currentList imageEnumeratorStartAt:_highlightedItem 
                                      directSense:direct];

      while ( (item = [list nextObject]) != nil 
              && [item getSelectionState] != NSOnState )
	 ;

      if ( item == nil )
	 return;

      if ( [item isMemberOfClass:[MyMovieImage class]] )
	 [_textView expandItem:[(MyMovieImage*)item getParent]];

      [self highlightItem:item];
   }
}

// Fill the popup with power of 2 up to the shorter side of the images
- (void) fillSidePopup
{
   NSEnumerator* list;
   MyImageListItem* item;
   long limit = -1.0;
   int side;
   
   NSAssert( _windowMode == ImageMode, @"fillSidePopup in non image mode" );
   
   // Check the minimum size from the list
   list = [[(MyObjectImageList*)_currentList imageArray] objectEnumerator];
   while ( (item = [list nextObject]) != nil )
   {
      NSSize size = [item imageSize];
      if ( size.width < limit || limit < 0 )
	 limit = size.width;
      if ( size.height < limit )
	 limit = size.height;
   }
   
   // Optimization : reconstruct only on size change
   if ( limit <= _sideMenuLimit/2 || limit >= _sideMenuLimit*2 )
   {
      NSString *searchSelection = nil, *analyzeSelection = nil;
      
      _sideMenuLimit = limit;
      [_searchSideMenu removeAllItems];
      [_analyzeSideMenu removeAllItems];
      for ( side = 16; side <= limit; side *= 2 )
      {
	 NSString* label = [NSString stringWithFormat:@"%d",side];
	 [_searchSideMenu addItemWithTitle:label];
	 [_analyzeSideMenu addItemWithTitle:label];
	 if ( side == (int)[(MyObjectImageList*)_currentList searchSquareSide] )
	    searchSelection = label;
	 if ( side == (int)[(MyObjectImageList*)_currentList analyzeSquareSide] )
	    analyzeSelection = label;
      }
      
      if ( searchSelection == nil )
      {
	 [_searchSideMenu selectItem:nil];
	 [_alignButton setEnabled:NO];
      }
      else
      {
	 [_searchSideMenu selectItemWithTitle:searchSelection];
	 [_alignButton setEnabled:YES];
      }
      if ( analyzeSelection == nil )
      {
	 [_analyzeSideMenu selectItem:nil];
	 [_analyzeButton setEnabled:NO];
      }
      else
      {
	 [_analyzeSideMenu selectItemWithTitle:analyzeSelection];
	 [_analyzeButton setEnabled:YES];
      }
   }

   _fillSideArmed = NO;
}

- (void) updateListControls
{
   BOOL controlsEnabled = _highlightedItem != nil;
   BOOL nextEnabled = [[_currentList imageArray] count] != 0;

   [_minusButton setEnabled:controlsEnabled];
   [_prevButton setEnabled:nextEnabled];
   [_nextButton setEnabled:nextEnabled];
   [_toggleButton setEnabled:controlsEnabled];
}

- (void) updateAnalyzeControls
{
   MyIntegerPoint o;
   MyObjectImageList *list = (MyObjectImageList*)_currentList;

   NSAssert1( _windowMode == ImageMode, 
              @"Analyze controls updating in mode %@", [self windowModeName] );

   if ( _windowState == AnalyzeMode )
   {
      BOOL menuEnabled = ( _highlightedItem != nil
                           && ![_highlightedItem isMemberOfClass:
                                                            [MyMovie class]] );
      BOOL controlsEnabled = ( menuEnabled
                               && [_analyzeSideMenu indexOfSelectedItem] >= 0);

      o = [list analyzeSquareOrigin];

      [_analyzeFieldX setFloatValue: o.x];
      [_analyzeFieldX setEnabled: controlsEnabled];
      [_analyzeFieldY setFloatValue: o.y];
      [_analyzeFieldY setEnabled: controlsEnabled];
      [_analyzeSideMenu setEnabled: menuEnabled];
      [_analyzeMethodMenu selectItemAtIndex: 
                                 [(MyDocument*)[self document] analysisMethod]];
      [_analyzeMethodMenu setEnabled: YES];

      [_analyzeButton setEnabled:controlsEnabled];
   }
   else
   {
      [_analyzeFieldX setEnabled: NO];
      [_analyzeFieldY setEnabled: NO];
      [_analyzeSideMenu setEnabled: NO];
      [_analyzeMethodMenu setEnabled: NO];

      [_analyzeButton setEnabled:_windowState == Analyzing];
   }
   

}

- (void) updateSelectThresholdSlide
{
   MyObjectImageList *list = (MyObjectImageList*)_currentList;
   double qmin, qmax;

   NSAssert1( _windowMode == ImageMode, @"Select threshold change in mode : %@", 
              [self windowModeName] );

   qmin = [list minQuality],
   qmax = [list maxQuality];

   if ( qmin < 0.0 || qmax < 0.0 )
   {
      [_minQualityText setStringValue:@""];
      [_maxQualityText setStringValue:@""];
      [_selectThresholdSlide setEnabled:NO];
   }
   else
   {
      [_selectThresholdSlide setEnabled:YES];
      [_minQualityText setDoubleValue:qmin];
      [_selectThresholdSlide setMinValue:qmin];
      [_maxQualityText setDoubleValue:qmax];
      [_selectThresholdSlide setMaxValue:qmax];
      [_selectThresholdSlide setDoubleValue:[list qualityThreshold]];
   }
}

- (void) updateStackControls
{
   // In stack mode, the hilight is set to the first selected line
   // therefore, if there is no hilight, there is no image at all
   BOOL enabled = (_highlightedItem != nil);
   MyIntegerSize s;

   if ( _windowMode == ImageMode )
   {
      if ( _windowState == StackMode )
      {
         MyObjectImageList *list = (MyObjectImageList*)_currentList;
         MyIntegerRect r = [list cropRectangle];

         s = r.size;

         [_cropX setFloatValue: r.origin.x];
         [_cropX setEnabled: enabled];
         [_cropY setFloatValue: r.origin.y];
         [_cropY setEnabled: enabled];
         [_cropW setFloatValue: r.size.width];
         [_cropW setEnabled: enabled];
         [_cropH setFloatValue: r.size.height];
         [_cropH setEnabled: enabled];
         [_doubleButton setState:
                           ([list sizeFactor] == 2 ? NSOnState : NSOffState)];
         [_doubleButton setEnabled: enabled];
      }
      else
      {
         [_cropX setEnabled: NO];
         [_cropY setEnabled: NO];
         [_cropW setEnabled: NO];
         [_cropH setEnabled: NO];
         [_doubleButton setEnabled: NO];
      }
   }
   else
   {
      s = [(MyDocument*)[self document] lockedSize];

      [_cropX setFloatValue: 0];
      [_cropX setEnabled: NO];
      [_cropY setFloatValue: 0];
      [_cropY setEnabled: NO];
      [_cropW setFloatValue: s.width];
      [_cropW setEnabled: NO];
      [_cropH setFloatValue: s.height];
      [_cropH setEnabled: NO];
      [_doubleButton setEnabled: NO];
   }

   if ( _windowMode == FlatFieldMode )
      [_monoStackButton setState: 
         ([(MyDocument*)[self document] monochromeFlat] ? 
                                                     NSOnState : NSOffState)];
   else
      [_monoStackButton setState: NSOffState ];
   [_monoStackButton setEnabled: enabled && _windowMode == FlatFieldMode ];

   [_stackButton setEnabled: _windowState == Stacking ||
                             (enabled && s.width != 0 && s.height != 0) ];
}

- (void) updateProcessControls
{
   REAL w = [_currentList whiteLevel];
   REAL b = [_currentList blackLevel];

   // Update the levels (common to all modes)
   if ( w > b )
   {
      REAL minL = [(MyDocument*)[self document] minLevel];
      REAL maxL = [(MyDocument*)[self document] maxLevel];

      if ( _windowMode == FlatFieldMode )
      {
         // Trick to display the levels 0..100
         minL *= 100;
         maxL *= 100;
      }
      
      if ( b < minL )
	 b = minL;
      if ( b > maxL )
	 b = maxL - 1.0/HUGE;
      [_blackLevel setEnabled:YES];
      [_blackTextLevel setEnabled:YES];
      [_blackLevel setMinValue:minL];
      [_blackLevel setMaxValue:maxL];
      [_blackLevel setDoubleValue:b];
      [_blackTextLevel setDoubleValue:b];

      if ( w < minL )
	 w = minL + 1.0/HUGE;
      if ( w > maxL )
	 w = maxL;
      [_whiteLevel setEnabled:YES];
      [_whiteTextLevel setEnabled:YES];
      [_whiteLevel setMinValue:minL];
      [_whiteLevel setMaxValue:maxL];
      [_whiteLevel setDoubleValue:w];
      [_whiteTextLevel setDoubleValue:w];
   }
   else
   {
      [_blackLevel setEnabled:NO];
      [_blackTextLevel setEnabled:NO];
      [_blackTextLevel setStringValue:@""];
      [_whiteLevel setEnabled:NO];
      [_whiteTextLevel setEnabled:NO];
      [_whiteTextLevel setStringValue:@""];
   }

   // Update the process parameters
   if ( _windowMode == ImageMode )
   {
      // Controls are valid in image mode, and values are in the image list
      MyObjectImageList *list = (MyObjectImageList*)_currentList;
      double dR = [list dRadius];
      double dT = [list dThreshold];
      double uR = [list uRadius];
      double uG = [list uGain];

      [_deconvRadius setDoubleValue:dR];
      [_deconvRadius setEnabled:YES];
      [_deconvTextRadius setDoubleValue:dR];
      [_deconvTextRadius setEnabled:YES];
      [_deconvThreshold setDoubleValue:dT];
      [_deconvThreshold setEnabled:YES];
      [_deconvTextThreshold setDoubleValue:dT];
      [_deconvTextThreshold setEnabled:YES];
      [_unsharpRadius setDoubleValue:uR];
      [_unsharpRadius setEnabled:YES];
      [_unsharpTextRadius setDoubleValue:uR];
      [_unsharpTextRadius setEnabled:YES];
      [_unsharpGain setDoubleValue:uG];
      [_unsharpGain setEnabled:YES];
      [_unsharpTextGain setDoubleValue:uG];
      [_unsharpTextGain setEnabled:YES];
   }
   else
   {
      // Controls are disabled and reset to default values in other modes
      [_deconvRadius setDoubleValue:1.0];
      [_deconvRadius setEnabled:NO];
      [_deconvTextRadius setDoubleValue:1.0];
      [_deconvTextRadius setEnabled:NO];
      [_deconvThreshold setDoubleValue:1.0];
      [_deconvThreshold setEnabled:NO];
      [_deconvTextThreshold setDoubleValue:1.0];
      [_deconvTextThreshold setEnabled:NO];
      [_unsharpRadius setDoubleValue:1.0];
      [_unsharpRadius setEnabled:NO];
      [_unsharpTextRadius setDoubleValue:1.0];
      [_unsharpTextRadius setEnabled:NO];
      [_unsharpGain setDoubleValue:0.0];
      [_unsharpGain setEnabled:NO];
      [_unsharpTextGain setDoubleValue:0.0];
      [_unsharpTextGain setEnabled:NO];
   }
}

@end
