/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: James Henstridge <james.henstridge@canonical.com>
 */

#include "mediascanner/logging.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/mediaartcache.h"
#include "mediascanner/mediaartdownloader.h"

#define ALBUM_ART_PLUGIN "grl-lastfm-albumart"
#define ALBUM_ART_SOURCE "grl-lastfm-albumart"

namespace mediascanner {

static const logging::Domain kInfo("info/mediaartdownloader", logging::info());
static const logging::Domain kWarning("warning/mediaartdownloader", logging::warning());

class MediaArtDownloader::Private {
public:
    struct ResolveData {
        ResolveData(Private *d,
                    const std::string &artist,
                    const std::string &album)
            : d(d)
            , artist(artist)
            , album(album) {
        }

        Private *const d;
        const std::string artist;
        const std::string album;

        guint operation_id; // Grilo operation
        Wrapper<SoupMessage> message;
    };
    typedef std::shared_ptr<ResolveData> ResolveDataPtr;

    Private() {
        g_mutex_init(&mutex);
        g_cond_init(&cond);
        setup_resolver();
        setup_soup_session();
    }

    ~Private() {
        cancel();
    }

    bool is_idle() const {
        return jobs.empty();
    }

    GMutex mutex;
    GCond cond;
    MediaArtCache cache;
    Wrapper<GrlSource> art_resolver;
    Wrapper<SoupSession> session;

    std::vector<ResolveDataPtr> jobs;

    void cancel();
    void setup_resolver();
    void setup_soup_session();

    void add_job(ResolveDataPtr data);
    void complete(ResolveData *data);

    static void OnResolveThumbnail(GrlSource *source, unsigned operation_id,
                                   GrlMedia *media, void *user_data,
                                   const GError *error);
    static void OnResponseReceived(SoupSession *session, SoupMessage *message,
                                   void *user_data);
};

MediaArtDownloader::MediaArtDownloader()
    : d(new Private) {
}

MediaArtDownloader::~MediaArtDownloader() {
    delete d;
}

void MediaArtDownloader::LoadGriloPlugin() {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    Wrapper<GError> error;

    if (not grl_registry_load_plugin_by_id(
            registry.get(), ALBUM_ART_PLUGIN, error.out_param())) {
        const std::string error_message = to_string(error);
        kWarning("Could not load album art plugin: {1}") % error_message;
        return;
    }
}

void MediaArtDownloader::Private::setup_resolver() {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());

    art_resolver = wrap(grl_registry_lookup_source(
                            registry.get(), ALBUM_ART_SOURCE));
    if (not art_resolver) {
        kWarning("Could not load album art source");
        return;
    }
}

void MediaArtDownloader::Private::setup_soup_session() {
    session = wrap(soup_session_async_new_with_options(
                       SOUP_SESSION_SSL_USE_SYSTEM_CA_FILE, TRUE,
                       SOUP_SESSION_SSL_STRICT, TRUE,
                       SOUP_SESSION_USER_AGENT, "Ubuntu-Media-Scanner/1.0",
                       NULL));
    soup_session_add_feature_by_type(
        session.get(), SOUP_TYPE_PROXY_RESOLVER_DEFAULT);
}

void MediaArtDownloader::Private::add_job(ResolveDataPtr data)
{
    jobs.push_back(data);
}

void MediaArtDownloader::Private::complete(ResolveData *data)
{
    for (auto it = jobs.begin(); it != jobs.end(); it++) {
        if (it->get() == data) {
            jobs.erase(it);
            break;
        }
    }
    if (is_idle())
        g_cond_signal(&cond);
}

void MediaArtDownloader::resolve(const std::string &artist,
                                 const std::string &album) {
    // If we couldn't configure either of these objects, then we won't
    // have any luck resolving the album art.
    if (not d->art_resolver || not d->session)
        return;

    kInfo("Resolving art for {1} - {2}") % artist % album;

    if (d->cache.has_art(artist, album)) {
        return;
    }

    g_mutex_lock(&d->mutex);

    /* First check whether there is any queued requests for this art */
    for (auto it = d->jobs.begin(); it != d->jobs.end(); it++) {
        if ((*it)->artist == artist && (*it)->album == album) {
            kInfo("Art for {1} - {2} already being resolved") % artist % album;
            g_mutex_unlock(&d->mutex);
            return;
        }
    }


    const Private::ResolveDataPtr data(
        new Private::ResolveData(d, artist, album));
    d->add_job(data);

    Wrapper<GrlMedia> media = take(grl_media_audio_new());
    grl_media_audio_set_artist(GRL_MEDIA_AUDIO(media.get()), artist.c_str());
    grl_media_audio_set_album(GRL_MEDIA_AUDIO(media.get()), album.c_str());
    Wrapper<GList> resolve_keys = take(grl_metadata_key_list_new(
        GRL_METADATA_KEY_THUMBNAIL,
        GRL_METADATA_KEY_INVALID));
    Wrapper<GrlOperationOptions> options =
        take(grl_operation_options_new(nullptr));

    data->operation_id = grl_source_resolve(
        d->art_resolver.get(), media.get(), resolve_keys,
        options.get(), &Private::OnResolveThumbnail, data.get());
    g_mutex_unlock(&d->mutex);
}

void MediaArtDownloader::Private::OnResolveThumbnail(
    GrlSource *source, unsigned operation_id, GrlMedia *media,
    void *user_data, const GError *error) {
    ResolveData *data = static_cast<ResolveData*>(user_data);
    Private *const d = data->d;

    g_mutex_lock(&d->mutex);
    data->operation_id = 0;
    if (error != NULL) {
        kWarning("Error determining album art for {1} - {2}: {3}")
            % data->artist % data->album % error->message;
        d->complete(data);
        g_mutex_unlock(&d->mutex);
        return;
    }

    const char *thumbnail = grl_media_get_thumbnail(media);
    if (thumbnail == NULL) {
        kInfo("No thumbnail found for {1} - {2}") % data->artist % data->album;
        d->complete(data);
        g_mutex_unlock(&d->mutex);
        return;
    }

    kInfo("Thumbnail for {1} - {2}: {3}") % data->artist % data->album % thumbnail;

    data->message = take(soup_message_new("GET", thumbnail));
    // queue_message() takes ownership of the message passed to it.
    soup_session_queue_message(d->session.get(), data->message.dup(),
                               &OnResponseReceived, data);
    g_mutex_unlock(&d->mutex);
}

void MediaArtDownloader::Private::OnResponseReceived(
    SoupSession *session, SoupMessage *message, void *user_data) {
    // If we have been cancelled, then our data pointer has been
    // cleaned up.
    if (message->status_code == SOUP_STATUS_CANCELLED)
        return;

    ResolveData *data = static_cast<ResolveData*>(user_data);
    Private *const d = data->d;

    g_mutex_lock(&d->mutex);
    data->message = NULL;

    if (message->status_code != SOUP_STATUS_OK) {
        kWarning("Failed to retrieve album art for {1} - {2}")
            % data->artist % data->album;
        d->complete(data);
        g_mutex_unlock(&d->mutex);
        return;
    }

    std::string content_type = soup_message_headers_get_content_type(
        message->response_headers, NULL);
    if (content_type != "image/jpeg") {
        kWarning("Expected album art for {1} - {2} to be image/jpeg, but got {3}")
            % data->artist % data->album % content_type;
        d->complete(data);
        g_mutex_unlock(&d->mutex);
        return;
    }

    kInfo("Storing album art for {1} - {2}") % data->artist % data->album;
    d->cache.add_art(
        data->artist, data->album,
        message->response_body->data, message->response_body->length);
    d->complete(data);
    g_mutex_unlock(&d->mutex);
}

bool MediaArtDownloader::is_idle() const {
    return d->is_idle();
}

void MediaArtDownloader::WaitForFinished() {
    g_mutex_lock(&d->mutex);
    kInfo("Waiting for pending album art downloads.");
    while (not is_idle()) {
        g_cond_wait(&d->cond, &d->mutex);
    }
    kInfo("Album art downloads complete.");
    g_mutex_unlock(&d->mutex);
}

void MediaArtDownloader::Private::cancel() {
    g_mutex_lock(&mutex);

    for (const auto &data: jobs) {
        if (data->operation_id != 0) {
            grl_operation_cancel(data->operation_id);
        }
        data->operation_id = 0;
        if (data->message) {
            soup_session_cancel_message(
                session.get(), data->message.get(), SOUP_STATUS_CANCELLED);
        }
        data->message = NULL;
    }

    g_mutex_unlock(&mutex);
}

}
