(* calc.sml *)

(* This file provides glue code for building the calculator using the
 * parser and lexer specified in calc.lex and calc.grm.
*)

structure Calc : sig
                   val parse : unit -> unit
                 end =
struct

(*
 * We apply the functors generated from calc.lex and calc.grm to produce
 * the CalcParser structure.
 *)

  structure CalcLrVals =
    CalcLrValsFun(structure Token = LrParser.Token)

  structure CalcLex =
    CalcLexFun(structure Tokens = CalcLrVals.Tokens)

  structure CalcParser =
    Join(structure LrParser = LrParser
         structure ParserData = CalcLrVals.ParserData
         structure Lex = CalcLex)

(*
 * We need a function which given a lexer invokes the parser. The
 * function invoke does this.
 *)

  fun invoke lexstream =
      let fun print_error (s,i:int,_) =
              TextIO.output(TextIO.stdOut,
                            "Error, line " ^ (Int.toString i) ^ ", " ^ s ^ "\n")
       in CalcParser.parse(0,lexstream,print_error,())
      end

(*
 * Finally, we need a driver function that reads one or more expressions
 * from the standard input. The function parse, shown below, does
 * this. It runs the calculator on the standard input and terminates when
 * an end-of-file is encountered.
 *)

  fun parse () =
      let val lexer = CalcParser.makeLexer (fn _ => TextIO.inputLine TextIO.stdIn)
          val dummyEOF = CalcLrVals.Tokens.EOF(0,0)
          val dummySEMI = CalcLrVals.Tokens.SEMI(0,0)
          fun loop lexer =
              let val (result,lexer) = invoke lexer
                  val (nextToken,lexer) = CalcParser.Stream.get lexer
                  val _ = case result
                            of SOME r =>
                                TextIO.output(TextIO.stdOut,
                                       "result = " ^ (Int.toString r) ^ "\n")
                             | NONE => ()
               in if CalcParser.sameToken(nextToken,dummyEOF) then ()
                  else loop lexer
              end
       in loop lexer
      end

end (* structure Calc *)
