from collections import defaultdict, Iterable
from functools import wraps
import re
import os
import unittest
import sys

# Add the directory containing 'grendel' to the sys.path

sys.path.append(os.path.join(os.path.dirname(__file__), os.pardir, os.pardir, os.pardir, 'lib', 'python'))

import grendel
from grendel.util.strings import classname
from grendel.util.decorators import with_attributes
from grendel.util.iteration import flattened
from grendel.util.sentinal_values import All

#--------------------------------------------------------------------------------#
# Module setup
def setUpModule():
    if not grendel.get_env_bool("GRENDEL_TEST_NO_CHECK", False):
        grendel.type_checking_enabled = True
        grendel.sanity_checking_enabled = True


#--------------------------------------------------------------------------------#
# profiling decorator
profile = with_attributes(profile=True)
def profile(*func_or_subsuites):
    if callable(func_or_subsuites[0]):
        # No argument given
        func_or_subsuites[0].__dict__['__profile__'] = True
        return func_or_subsuites[0]
    else:
        def mark_profile_subsuite(f):
            f.__dict__['__profile__'] = func_or_subsuites
            return f
        return mark_profile_subsuite
#--------------------------------------------------------------------------------#
# Decorator for generators in unittest.TestCase subclasses
generator_regex = r'(^|[\b_\.])([Gg]enerator)'
test_regex = r'(?:^|[\b_\.-])[Tt]est'
def allow_generators(cls):
    """Add tests to `cls` generated by functions matching `(?:^|[\\b_\\.-])[Gg]enerator`.
    """
    generator_names = [name for name in dir(cls) if re.search(generator_regex, name)]
    generators = []
    for gen_name in generator_names:
        try:
            gen = getattr(cls, gen_name)
            if isinstance(cls.__dict__[gen_name], classmethod):
                generators.append(gen)
        except TypeError:
            pass # The generator is probably skipped.  just move on
    for generator, gen_name in zip(generators, generator_names):
        for args in generator():
            def _generated_test(self, args=args[1:], f=args[0]):
                f(self, *args)
            _generated_test.__name__ = args[0].__name__
            _generated_test.__generated__ = True
            if not _generated_test.__name__ == '_pretty_skipped_function':
                if hasattr(cls, _generated_test.__name__):
                    raise NameError("Class {clsname} already has a test"
                                    " named {testname}.  Generator will"
                                    " not overwrite this.".format(
                        clsname=cls.__name__,
                        testname=_generated_test.__name__
                    ))
                if not re.search(test_regex, _generated_test.__name__):
                    raise NameError("Generated function name {testname} does"
                                    " not match the test name pattern and"
                                    " thus will not be detected as a test.  (Test"
                                    " function names must contain a match of"
                                    " {testre})".format(
                        testname=_generated_test.__name__,
                        testre=test_regex
                    ))
                setattr(cls, _generated_test.__name__, _generated_test)
        delattr(cls, gen_name)
    return cls
# a simple shorthand that may be a bit more readable...
generator=classmethod
#--------------------------------------------------------------------------------#
# Get some environment variables
do_long_tests = grendel.get_env_bool('GRENDEL_LONG_TESTS', False)
hide_skipped_tests = grendel.get_env_bool('GRENDEL_HIDE_SKIPPED', False)
#--------------------------------------------------------------------------------#
# Skip more gracefully...
def pretty_skip(msg_or_f):
    if callable(msg_or_f):
        if hide_skipped_tests:
            ret_val = unittest.skip(msg_or_f)
            ret_val.__name__ = '_pretty_skipped_function'
            return ret_val
        else:
            return unittest.skip(msg_or_f)
    else:
        def decorate_it(f):
            if hide_skipped_tests:
                ret_val = unittest.skip(msg_or_f)
                ret_val.__name__ = '_pretty_skipped_function'
                return ret_val
            else:
                return unittest.skip(msg_or_f)(f)
        return decorate_it
def pretty_skip_if(condition, reason):
    def decorate_it(f):
        if condition:
            return pretty_skip(reason)(f)
        else:
            return f
    return decorate_it
def pretty_skip_unless(condition, reason):
    return pretty_skip_if(not condition, reason)
# Aliases for uniformity
skip=pretty_skip
skipUnless=pretty_skip_unless
skipIf=pretty_skip_if
#--------------------------------------------------------------------------------#
# Decorators for long tests and long generators
# TODO @testing degrees of 'longness'; for instance the expected time that the test will take to execute, and suites to sort these tests
def long_test(f):
    long_test = pretty_skip_unless(
                do_long_tests,
                "Not running long tests.  Set the envirnoment variable 'GRENDEL_LONG_TESTS' to 1 to run long tests."
    )(f)
    long_test.__is_long_test__ = True
    return long_test
# prevent automatic test discovery from
long_test.__name__ = '_dont_pick_me'
if do_long_tests:
    long_generator = classmethod
else:
    def long_generator(f):
        def _skipped_gen(self):
            pass
        _skipped_gen.__name__ = re.sub(generator_regex, r'\1', f.__name__)
        return pretty_skip("Not generating tests for generators marked 'long_generator'."
                             "  Set the envirnoment variable 'GRENDEL_LONG_TESTS' to 1 to"
                             " generate these tests.")(_skipped_gen)
#--------------------------------------------------------------------------------#
# replace unittest.expectedFailure, since for some reason that doesn't work
def expected_failure(*args, **kwargs):
    if len(args) == 1 and callable(args[0]) and len(kwargs) == 0:
        f = args[0]
        @wraps(f)
        def _wrapped(self, func=f):
            try:
                func(self)
            except AssertionError:
                pass
            else:
                self.fail("test didn't fail")
        return _wrapped
    else:
        allow_exceptions = kwargs.pop('allow_exceptions', False)
        if allow_exceptions is True:
            allow_exceptions = All
        elif allow_exceptions is False:
            allow_exceptions = [AssertionError]
        elif isinstance(allow_exceptions, Iterable):
            allow_exceptions = list(allow_exceptions) + [AssertionError]
        else:
            allow_exceptions = [allow_exceptions] + [AssertionError]
        def _decorate_it(f):
            @wraps(f)
            def _wrapped(self, func=f, allow_exceptions=allow_exceptions):
                try:
                    func(self)
                #----------------------------------------#
                except Exception as e:
                    # see if the raised exception is allowed
                    if allow_exceptions is not All and type(e) not in allow_exceptions:
                        self.fail("test raised unexpected exception {}: {}".format(
                            type(e).__name__, str(e)))
                    # Otherwise, we're okay, so do nothing...
                #----------------------------------------#
                else:
                    # Test didn't fail, so fail
                    self.fail("test didn't fail")
            return _wrapped
        return _decorate_it
#--------------------------------------------------------------------------------#
# Subsuite decorator for automatic subsuiting
def in_subsuite(*args):
    def decorate_it(f, args=args):
        if not hasattr(f, '__subsuites__'):
            f.__subsuites__ = []
        for arg in args:
            f.__subsuites__.append(arg)
        return f
    return decorate_it
#--------------------------------------------------------------------------------#
# construct suites
# get the list of tests
grendel_suite_all = unittest.defaultTestLoader.discover(os.path.dirname(__file__))
profile_suite = unittest.TestSuite()
profile_subsuites = defaultdict(lambda: unittest.TestSuite())
standard_suite = unittest.TestSuite()
long_only_suite = unittest.TestSuite()
subsuites = defaultdict(lambda: unittest.TestSuite())
# Now iterate over all of the tests
for test in flattened(grendel_suite_all):
    test_func = getattr(test, test.id().split('.')[-1], None)
    if test_func is not None:
        #----------------------------------------#
        profile_subsuite = getattr(test_func, '__profile__', None)
        if profile_subsuite is not None:
            profile_suite.addTest(test)
            if isinstance(profile_subsuite, tuple):
                for subsuite in profile_subsuite:
                    profile_subsuites[subsuite].addTest(test)
        #----------------------------------------#
        long_test_attr = getattr(test_func, '__is_long_test__', None)
        if long_test_attr:
            long_only_suite.addTest(test)
            if hide_skipped_tests:
                # Change the __name__ attribute so that it doesn't get discovered
                test_func.__func__.__name__ = '_dont_pick_me'
        else:
            standard_suite.addTest(test)
        #----------------------------------------#
        subsuite_names = getattr(test_func, '__subsuites__', None)
        if subsuite_names is not None:
            for name in subsuite_names:
                subsuites[name].addTest(test)
        #----------------------------------------#
        if hide_skipped_tests:
            skiptest = getattr(test_func, '__unittest_skip__', False)
            if skiptest:
                # Change the __name__ attribute so that it doesn't get discovered
                test_func.__func__.__name__ = '_dont_pick_me'
#--------------------------------------------------------------------------------#







