;;; pyim-pymap.el --- Pinyin map used by pyim  -*- lexical-binding: t; -*-

;; * Header
;; Copyright (C) 2015-2020 Free Software Foundation, Inc.

;; Author: Feng Shu <tumashu@163.com>
;; Maintainer: Feng Shu <tumashu@163.com>
;; URL: https://github.com/tumashu/pyim
;; Keywords: convenience, Chinese, pinyin, input-method

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; * 说明文档                                                              :doc:

;; 这个文件是 pyim 内部使用的 "拼音-汉字" 对照表，这个对照表用来实现拼音查询功能，
;; 即，查询某个汉字对应的拼音代码。

;; `pyim-pymap' 是使用 `pyim-pymap-build-pymap' 命令从 libpinyin-2.6.0.tar.gz 包
;; 含的 data 文件转换得到的。
;; 1. [[https://github.com/libpinyin/libpinyin/releases/download/2.6.0/libpinyin-2.6.0.tar.gz][libpinyin-2.6.0.tar.gz]]

;; 注意： 这个文件 *不用于* 输入法自定义词库！！！

;;; Code:
;; * 代码                                                                 :code:
(require 'pyim-common)

(defvar pyim-pymap
  '(("a" "阿啊呵腌|嗄吖锕||錒")
    ("ai" "爱埃艾碍癌呆哀挨矮隘蔼唉皑哎|霭捱暧嫒嗳瑷嗌锿砹诶乂欸叆毐|佁烠堨|㤅嘊伌礙䨠嬡娭鴱䀳䬵㘷䠹昹䔽娾䑂靄硋㕌呝嵦塧譪鱫䅬㱯噯敱㝶㑸䝽鎄㢊誒皚䶣馤璦皧銰躷瞹㿄㗒凒懓曖懝㗨濭藹賹阸㝵閡絠鯦譺㾢䔇㾨焥諰溰醷謁騃愛啀鑀靉敳薆厓僾餲愒壒溾")
    ("an" "厂广安案按岸暗鞍氨俺胺铵谙庵黯|鹌桉埯犴揞唵|匼垵垾盦|錌匎儑䬓堓䜙㜝洝阥㸩堷䅁葊荌䅖㱘㽢腤婩鵪罯晻厈貋隌䎏侒鶕䮗誝鮟䎨銨玵㟁峖䯥盫黬蓭諳痷雸韽㡋咹頇屽䤃遃㭺裺䤶萻豻啽広鞌媕闇菴馣")
    ("ang" "昂盎|肮卬||䬓䀚㼜昻䩕䍩䭹䭺㭿岇枊醠䒢骯㦹軮䇦")
    ("ao" "奥澳傲熬凹嚣鳌敖遨鏖袄坳翱嗷|拗懊岙螯骜獒鏊艹媪廒聱璈|奡嶅隩|鴁㠂㤇䴈爊䜒㠗㜜蔜簢㜩㘬㘭鰲謸䐿驁㑃䵅詏獓襖䥝慠㕭䁱扷镺㥿垇䦋梎澚䞝抝隞䚫厫䮯芺媼囂鷔擙䯠䫨柪軪嫯㿰滶翶嗸翺㟼礉䆟熝鏕㩠墽郩岰䉛蝹鼇謷奧摮嶴墺磝眑鴢泑")
    ("ba" "把八巴拔吧坝爸霸罢芭跋扒叭靶疤|笆耙杷鲅粑岜灞钹钯捌菝魃茇朳|妭胈蚆羓鲃鲌|鈀欛㔜鼥紦笩鈸䰾夿䩗詙䱝魞䥯罷坺䩻鮁䆉鮊㞎玐㶚馛㖠墢炦䎬犮颰䎱垻䳁㧊䳊叐䇑釟秡䟦壩哵䃻矲䮂柭䈈㭭釛皅䶕䯲㭛癹抪騃猈弝茷豝覇抜仈軷峇跁")
    ("bai" "百白败摆伯拜柏佰掰|呗擘捭稗|鞁|䥯贁唄㠔㼟栢㼣䴽䙓絔敗兡庍䒔䢙薭䳆㗑㗗㿟矲擺蛽韛挀㧳䠋䪹㔥竡猈襬粨粺拝")
    ("ban" "办版半班般板颁伴搬斑扮拌扳瓣坂阪绊|钣瘢舨癍柈|昄攽湴靽|朌怑鈑㸞褩䬳蝂絆鉡虨㩯䕰䉽粄岅螌㚘㺜肦辦辬㪵闆秚瓪賁㻞䛀埿䰉搫䈲坢䃑跘螁頒斒姅籓魬鳻")
    ("bang" "帮邦旁榜棒膀镑绑傍磅蚌谤梆|浜蒡氆甏|玤蚄棓塝搒|縍稖謗㔙蜯䰷挷幇㭋幚捠幫綁㮄鎊䂜鞤㾦䎧䖫邫垹㯁髈䧛䟺埲硥蛖騯嫎牓艕")
    ("bao" "报保包宝暴胞薄爆炮饱抱堡剥鲍曝葆瀑豹刨褒雹孢苞|煲褓趵鸨龅勹|枹|鴇怉䨌䈏䴐䨔䤖堢笣報襃㙅䭋鉋靌蕔齙㵡䥤㙸䎂窇袌㲏鮑㲒嚗犦䪨蚫宲骲忁闁駂㻄铇䳈嫑曓寚瓝珤緥㫧䳰賲鳵寶㿺勽飽鸔䭸䯽靤髱儤虣鑤媬菢寳扢嘐")
    ("bei" "北被备倍背杯贝辈悲碑臂卑悖惫蓓裨|陂钡狈呗焙俾碚褙埤鞴孛鹎萆邶鐾|哱琲椑棓鞁糒|㱯䱝䥯垻唄㗗䟺昁鄁愂錃㰆㼎䔒伓䠙㸢㔨㸬㤳紴鐴㸽䰽䩀䁅牬杮鵯鉳䡶偹禆憊㶔犕梖誖備貝㾱綼㣁僃鋇㓈㻗韛㛝藣郥軰䋳㷶揹俻珼狽䟛箄骳㔗㭭萯䣙苝㫲諀輩偝桮奰盃柸")
    ("ben" "本奔苯笨夯|贲锛畚坌栟坋|倴犇|渀㤓錛㨧逩䬱夲輽楍獖奙桳㱵㡷捹撪㮺賁燌泍軬蟦鐼炃㮥喯翉")
    ("beng" "泵崩蚌蹦迸绷甭|嘣甏堋琫镚|玤祊洴弸搒|㔙蜯挷㛝逬䨻䙀繃㑟奟䩬嵭䭰塴㱶誁閍鞛熢綳䳞㷯槰跰鏰埲䋽㧍漨琣錋㼞傰埄蠯伻絣菶偪痭")
    ("bi" "比必币笔毕秘避闭辟壁弊彼逼碧鼻臂蔽泌璧庇痹毙弼匕鄙陛裨|贲敝蓖吡篦纰俾铋毖筚荸薜婢哔跸埤濞秕荜愎睥妣芘蚍箅庳髀畀滗狴萆嬖襞擗鐾舭|坒佖沘邲诐苾咇珌祕椑皕赑鞁馝觱鲾皦|䐿㕭㿰㗉㮄紴鐴䁅綼㷶賁繴鮩䨆朇鼊䠋蜌鰏䬛堛䌟㘠䘡䀣鄨㘩縪鄪稫㠲䄶夶弻朼㡀畁㹃彃㙄䵄筆饆佊豍鉍煏畐襒魓蹕鵖䕗㡙啚䩛聛畢幣襣幤㵥㵨䭮魮潷腷㱸䁹罼悂斃躃箄鮅箆閇枈閉粊㪏貏䪐袐螕䦘檘熚䮡㪤䊧䖩疪㚰㢰㮰貱骳䎵㢶㢸鞸妼㮿柀䫁䟆㯇鏎䋔㓖䧗䯗㧙鷝韠䏢闢䟤䣥廦㻫㿫柫滭嫳篳毴㻶䏶嗶䇷㳼蓽䃾䫾䏟㵒胇踾䵗鶝䌘䉾鴓娝鸊笓紕崥怶睤肶㯅旇翍奰蘗愊贔怭屄楅驆払詖獘獙湢偪皀粃躄梐疕鎞閟駜鷩柲飶痺閈俛髲")
    ("bian" "变便边编遍辩鞭辨贬匾扁卞汴|辫砭苄蝙鳊弁窆笾煸褊碥髟忭缏抃|昪萹|䐔鶣豍甂鴘㴜稨猵楄牑㭓䡢籩獱䁵㵷㝸䉸鍽邉覍㦚炞閞辡㲢玣辧䒪辮辯貶㺹䪻糄柉藊㳎㣐㳒䛒揙㻞編緶惼鯾鯿䟍疺匥㺕臱頨鞕変艑汳變邊徧箯辺釆")
    ("biao" "表标彪镖裱|飚膘飙镳婊骠嘌飑杓髟鳔灬瘭俵摽|骉脿蔈幖熛儦藨瀌|䙳爂贆㠒標謤䔸褾鰾驃鑣墂䞄羆檦颩颮徱颷飆飇飈飍諘鏢磦㯱㯹㧼㟽淲䅺膔篻猋錶驫穮麃僄臕滮")
    ("bie" "别鳖憋|瘪蹩|咇|㜩柭縪襒㱸㓖㧙䟤䇷㲢鼈䠥別彆鱉虌癟䭱䉲莂龞徶㢼蛂㿜蟞䏟䋢穪㭭䌘柲")
    ("bin" "宾滨斌彬濒殡缤鬓槟|摈膑玢镔豳髌傧邠||䳁虨鬂儐䐔椕瀕鬢霦汃鑌驞穦豩顮瑸繽鶣馪殯檳梹矉臏賓賔髕髩擯濱濵濹㻞砏訜蠙璸贇")
    ("bing" "并病兵冰屏饼炳秉丙摒柄槟|禀廪邴冫栟|昺蛃|㓈誁檳梹㨀䈂倂餅琕昞稟餠栤眪䴵鈵陃偋靐癝鵧幷窉傡抦鮩庰棅仌䋑䓑䗒竝鞸綆鞆癛磝絣並怲氷併鉼鞞掤寎苪痭")
    ("bo" "波博播勃拨薄伯玻搏柏泊舶剥渤魄卜驳脖膊簸菠礴箔铂|亳钵帛鲅擘饽跛蘖钹趵薜檗啵鹁泺孛擗踣僰|妭砵哱浡袯桲鲌嶓镈馞欂|㔜笩鈸詙鮁鮊馛墢䳁秡䟦㼟栢㼣䰷䨌㙸嚗犦瓝㼎紴㑟挀踄謈䰊㬍餑蔔鸔簙礡㬧䬪挬猼愽缽㴾㹀敀彂噃䑈艊驋鱍䙏鉑鵓牔剝㱟鉢䭦㩧䍨䥬㩭鑮䭯蹳繴癶䍸孹㝿煿䮀䮂䒄碆䪇䶈疈䢌㪍䂍馎嶏㖕岥撥箥䢪䪬䞳犻侼蚾㶿䊿駁胉髉䫊駊䯋㟑㗘䗚䟛郣苩懪糪柭㧳䌟䄶䵄㵨檘䮡䟆䃗肑仢襎僠殕䵗䟔䟮豰袹瓟蔢㬥狛穛茷蘗餺襏譒葧襮袚鎛髆盋鋍駮帗皪壆彴")
    ("bu" "不部步布补捕堡埔卜埠簿哺怖璞|钚卟瓿逋晡醭钸|埗蔀|踄鈈䬏㨐䴝餢䀯㘵䴺鈽䍌鵏㙛獛歨䑰䝵荹䪁䊇䒈悑䪔咘㾟庯㚴㻉㳍勏補䳝䋠鳪峬郶䏽㱛捬䒀秿吥婄轐陠箁抪柨僄餔佈捗不歩篰尃輹鯆誧")
    ("ca" "拆擦|嚓搽礤||䵽傪䟃囃䌨遪攃")
    ("cai" "才采财材菜彩裁蔡猜踩睬||偲|䴺䰂䐆䌨縩䴭婇㥒睵財溨㒲綵䣋棌犲啋倸䌽纔採寀埰跴扐")
    ("can" "参残餐灿惨蚕掺璨惭|粲孱骖黪|䅟|歺傪䟃䘉㘔㜗嬠䬫嬱謲蠶蠺鰺㨻䙁驂蝅噆㥇湌慘慙㱚慚㽩䝳䑶摻䍼㺑薒殘㦧澯喰參叄叅㿊㛑㣓䗝䗞䣟蓡燦㻮䫮黲鯵䛹䳻䏼㜞鏒嘇篸囋朁憯飡嵾縿穇摲")
    ("cang" "藏仓苍沧舱臧|伧|鸧|倉匨艙嵢䅮㵴鑶㶓傖䢢蒼滄賶仺獊篬蔵螥鶬凔")
    ("cao" "草操曹槽糙嘈漕|艹螬艚屮||㽩㜖䄚䐬騲鄵愺褿襙䒃嶆䒑撡䏆㯥鏪蓸曺慒鼜鐰㿷艸肏喿懆慅傮")
    ("ce" "策测册侧厕|恻||萗䈟笧測㨲䔴䜺敇㩍荝筞遪側㥽䊂墄冊厠憡箣粣廁蓛矠嫧䇲拺惻幘簎茦畟筴莋萴")
    ("cen" "参岑|涔骖|梣|參叄叅䤁笒䨙橬硶䅾䲋㞥梫埁䯔䃡㻸汵㞤嵾")
    ("ceng" "层曾|蹭噌缯|鄫嶒|㬝䉕䁬碀㣒竲曽繒驓層橧")
    ("cha" "查察差茶插叉刹茬楂岔诧|碴嚓喳姹杈汊衩搽槎镲檫馇锸猹嵖|沄侘垞荖梌蹅|䰈䠡䑘䂳紁訍㤞䤩餷剎䕓鑔偛䁟鍤詧䡨詫靫奼疀鎈㢉㢎㢒䆛肞銟䲦䶪䊬㪯䒲㫅嫅秅嗏䟕仛䓭㛳揷㣾芆㳗釵䅊岎㜘摖㮑褨㛼捈㲼㡸䛽挿艖扠査臿扱")
    ("chai" "差拆柴|钗豺侪虿瘥|茝|䜺䐤訍䡨蠆䘍儕祡㑪袃芆喍犲㾹囆㳗䓱釵㦅㼮齜偨")
    ("chan" "产单阐缠掺禅颤铲蝉搀潺|蟾馋忏婵孱觇廛谄谗澶骣羼镡躔蒇冁挦巉镵辿刬|旵浐梴啴瀍襜儳韂|䴺㨻摻䫮䳻㬄簅㔆嬋嬓鐔䤘攙䜛脠䀡㸥䠨䤫䐮䴼嘽鄽䥀酁譂蕆㵌㹌摌䑎䵐獑煘灛幝饞硟繟䡪顫䡲㙴繵䩶剷湹潹㙻㹽䱿䂁㦃㢆讇纏纒讒誗覘䪜㢟㶣㺥醦袩禪單䊲閳諂囅㯆惉鋋棎䣑鋓鏟闡壥蟬䧯懴懺滻緾讖䩥㺗㚲螹䦲緂裧斺榐辴欃產産丳瀺剗単艬鑱嵼劖嚵燀毚磛僝幨僤儃摲蟺襢鉆")
    ("chang" "场长厂常偿昌唱畅倡尝肠敞倘猖娼裳徜昶怅|嫦菖鲳阊伥苌氅惝鬯玚|鲿|厈䠀倀䠆萇嘗甞償锠琩場焻㙊䕋兏䩨腸镸瑺畼晿㦂嚐膓暢誯悵閶䗅淐僘鏛廠㫤仧鯧裮䯴瓺鋿椙闛瑒鼚錩鱨塲長厰粻韔鋹")
    ("chao" "超朝潮炒钞抄巢吵剿绰嘲|晁焯耖怊|弨槱|鼂鼌樔鈔鄛䬤眧欩䰫䄻謿轈罺煼禉䎐㶤麨㷅巐觘䏚濤仦仯趠綽漅劋焣䫸窲粆牊䳂䲃䮓訬摷巣勦趫搯")
    ("che" "车彻撤尺挚扯辙澈掣|坼砗屮|㬚|䤁䨁㨋唓頙䜠伡爡䰩㔭㵃聅㥉㱌轍㵔偖䁤硨硩䑲奲䒆莗㾝䚢䞣瞮徹車㯙迠俥䧪㿭䋲烲勶䛸蛼揊喢㒤㤴謵㳧䊞烢撦摰詀呫")
    ("chen" "称陈沈沉晨琛臣尘辰衬趁忱橙郴|娠宸谌碜嗔抻榇伧谶龀肜胂瞋|飏疢棽煁|謲䗞傖硶醦贂鈂霃樄謓伔堔愖䜟䤟夦訦㴴踸敐屒齓齔鍖穪襯陳㕴塵捵敶㽸㲀莀趂䢅䢈䞋墋麎莐讖䚘䒞䆣嚫螴䢻趻薼諃䣅鷐䫖裖軙曟䟢磣迧櫬㧱㫳諶藽揨跈綝䀼儬瘎㥲䏹䫈縝侲茞儭蔯稱搷賝瘨桭")
    ("cheng" "成程城承称盛乘诚呈惩撑澄秤橙骋逞瞠|丞晟铛埕塍噌伧蛏柽铖酲裎樘浈枨琤|宬珹珵偁庱牚赪澂憕|碀䁤穪䚘脀騁㨃洆䄇椉娍瀓䀕堘崝㼩騬脭爯挰䔲頳阷鐺㐼睈䁎饓塖靗摚牜湞晠塣筬畻浾絾荿掁㲂憆檉徎撐䆑䞓窚誠䮪侱䆵䆸㞼竀䫆䗊䧕棖泟埥棦揨鋮懲蟶䇸峸緽鏿僜檙矃溗橖浧氶撜埩鏳稱橕乗赬悜郕虰踜朾")
    ("chi" "持吃池迟赤驰尺斥齿翅匙痴耻炽侈沱弛叱啻|坻哆郗眙嗤墀哧茌豉敕笞饬踟踅蚩媸魑篪沲褫彳鸱螭瘛痄眵傺搋|佁胣栻瓻痓摛漦䗖|誀䐤䰩㔭㥉䜄㔑䈕栘謘耛㘜䔟鴟䠠䰡伬䤲䜵訵䜻㱀恀筂噄鵄荎齒鉓慗杘䙙㽚䑛㙜恜齝腟遟鵣恥汦彨歭遲鉹齹灻㡿赿㢁傂㒆蚇䮈箈㢋趍粎憏銐垑䶔侙㮛妛瞝徥䪧趩㢮徲袳馳㞴㶴䶵麶誺䮻岻䊼䞾熾貾㞿㟂䛂翄㳏黐勑淔懘鷘翤裭飭忯䳵䟷拸痸叺㓼䇼姼㓾烾俿䀸慸䧝軧捇卙䏧狋迡樆䄜邌㷰雴汖紕㩽湁卶翨菭蝭脪㥡趐謻㑜呹沶胵饎戠欼遅呎扡癡遫歯摰彲絺誃箎鶒蚔喫蚳抶勅跮竾鍉彽瘈柅呬乿袲騺迣鳷")
    ("chong" "重种充冲崇虫宠忡憧|舂茺铳艟|珫翀埫摏漴|嘃崈褈䌬祌㹐蹖衝蝩浺隀銃䖝徸痋㧤䳯蟲寵㓽㓼䡴爞㮔䆹偅蹱喠緟湧沖傭揰憃罿")
    ("chou" "筹抽绸酬愁丑臭仇畴稠瞅踌|溴惆俦瘳雠帱杻|尨侴婤椆犨|䔏儔㐜吜栦䌧㨶䌷㤽籌䥒絒魗㵞酧幬杽疇躊嚋讎讐醔䲖皗醜㦞殠綢莥犫䊭䪮憱薵醻矁䓓雔菗㿧䛬臰㛶䇺㘜遚㿒㾄鈕㨨跾䱔牰鮋搊丒嬦紬詶鯈篘懤裯燽眣偢怞")
    ("chu" "出处础初除储畜触楚厨雏矗橱锄褚滁躇|怵绌搐刍蜍黜杵蹰恹亍樗憷楮㤘俶|柷琡摴濋斶|㜗㗰豠䎌㘜踀䠂礎㤕㔘椘耝㼥䠧欪儲䜴嘼絀㙇敊㕏蕏㕑䙕處䙘遚㡡齭幮荲䅳㹼齼岀㶆䦌趎䖏犓躕傗貙檚䎝䮞䎤䊰墸芻拀滀䧁臅櫉竌泏竐廚雛䟞䟣鋤櫥懨藸觸鯺䇍珿摢䶥㗙櫖鸀埱䅷㾥媰䐍䐢㾻諔炪儊鄐耡蠩鉏豖詘歜齣蒢閦蒭鶵処跦篨慉菆蓫迍")
    ("chuai" "揣|踹啜嘬搋膪||䴝㱀㪓㪜䦟䦤䦷腄欼膗")
    ("chuan" "传川船穿串喘|椽舛钏遄氚蝽巛舡|堾圌|鐉舩輲踳㼷歂汌瑏㱛䁣傳暷篅㯌賗僢釧諯鶨椯剶丳荈玔猭伝")
    ("chuang" "创床窗闯幢疮|怆艟疒|漴噇橦|䆫囪刅瘡戧愴搶漺㵂獊牎剏摐牕㡖剙䭚摤䡴創䚒䎫窻闖䃥䇬䄝傸磢㼽凔刱牀憃窓朣")
    ("chui" "吹垂锤炊椎陲槌|捶棰|倕圌魋|䞼㷃䕓錘䄲娷顀腄䍋桘㥨湷㝽㩾㓃諈菙䳠䅜䜅甀搥鎚箠龡埀")
    ("chun" "春纯醇淳唇椿蠢|鹑朐莼蝽䏝|堾焞瑃|䮞踳萅䐇䐏䔚䄝㸪萶㝄偆㝇䥎陙㵮浱睶湻鶉媋憌䞐蒓醕暙鶞膥䦮㖺箺䫃櫄䓐䏛滣㿤䣨䣩旾䡅輇賰鰆純漘錞脣輴杶鯙蓴惷犉芚橁")
    ("chuo" "辍缀绰戳|啜龊踔辶|逴惙婼|拺鏃娖輟䄪腏繛鑡齪酫齱䍳㲋䂐䮕㚟趠䆯涰辵嚽綽諁䇍䓎䃗䋘磭珿㪕娕跿箹孎䄌㪬吷歠婥畷醊擉")
    ("ci" "次此差词辞刺瓷磁兹慈茨赐祠伺雌疵|蚝鹚糍呲螅茈粢跐|佽泚玼堲莿|厠廁㘂甆䰍䨏䈘礠嬨栨㤵㘹刾㠿㹂㩞詞䭣㡹赼㢀趀䆅螆皉薋垐㞖庛辝䂣辤䖪辭枱䦻䲿鷀珁䳄柌䛐䳐賜䓧㓨濨䧳䯸䗹飺縒髊齹㾊紪㐸栜鈶蠀鴜姕茦餈朿佌絘骴蛓偨鮆澬鶿")
    ("cong" "从丛匆聪葱囱|琮淙枞骢苁璁熜|悰|樅欉漗騘爜錝蔥謥焧樬樷㼻婃鍃驄灇慒䉘聡聦孮鍯聰繱䕺徖瞛從悤䆫暰誴藂棇蟌総鏓叢忩賩囪燪緫蓯㗰篵䳷楤䡯䰌憁碂潨漎怱潀瑽従鏦賨鍐菆")
    ("cou" "凑|辏腠楱||輳㫶㔌湊")
    ("cu" "促粗簇醋卒蹴猝|蹙酢蔟殂徂㤘||鼀瘄䠓㰗䠞䬨脨縬䥄噈䥘豠䙯梀麁䎌䢐趗殧趨鏃䃚䓚䟟㗤䛤蔖憱踀娕蔍趥誎䣯踓蹵麆踧瘯麄麤觕怚顣踤")
    ("cuan" "窜攒篡|蹿氽撺爨汆镩|僔|攅㸑簒䰖攛㠝攢㵀㭫鑹劗䆘躥殩熶䞼竄巑櫕濽穳襸䂎灒鋑篹欑菆")
    ("cui" "衰催崔脆翠萃粹摧璀瘁|悴淬橇啐隹毳榱|崒缞漼|䢪䔴脃琗䄟紣脺襊㵏獕㱖慛㥞㝮䆊墔皠疩䊫膬䂱綷䃀㳃㷃翆臎㯔凗㧘鏙㯜濢㷪槯忰嗺䧽㓹繀熣䘹嶉倅伜焠顇啛粋趡膵竁磪縗踤椊")
    ("cun" "村存寸蹲|忖皴邨|瑳|刌䍎籿澊袸竴侟䞭拵吋洊踆墫")
    ("cuo" "措错磋挫搓撮|蹉锉厝嵯痤矬瘥脞鹾|玼莝棤瑳酂|䥘㭫襊䰈縒蔖䠡䐣䐤逪錯䴾睉剒䑘虘䱜㽨硰嵳遳歵鹺莡蒫䂳侳銼㟇髊䣜䟶齹㿷諎躦夎酇剉齰醝蓌斮")
    ("da" "大达打答搭沓瘩|嗒挞哒耷鞑疸靼褡笪怛妲跶褟|垯荙炟溚阘|䐊匒㜓眔鐽畗詚噠䩢䵣畣㙮剳䑽呾㾑墖撘箚鎝墶韃㯚䃮㿯䳴㟷蟽汏䪚毼䪏躂羍龖㯓迖㗳矺笚㩉觰謷逹荅達繨咑薘闒僤搨鎉龘")
    ("dai" "大代带待贷戴袋歹呆逮岱傣怠殆黛甙|埭绐玳呔迨骀嘚|轪媞叇|䲦䠠䈆帒簤㐲霴帶瀻紿獃瑇汏蹛酨䚞䚟㶡䒫㞭貸㯂㿃曃柋軑忕㻖廗軚懛跢軩黱㫹緿蝳箉駘鴏詒帯靆艜襶遰")
    ("dan" "但单石担丹胆旦弹蛋淡诞氮郸耽檐赡殚|惮儋眈疸澹掸亻膻啖箪钽聃萏瘅赕亶|疍惔憺|譂䡲繵覘單伔愖呾蹛㔊帎刐舕頕簞䨢匰鄲䨵䄷耼嘾㵅腅䱋㽎饏灗癚䭛䉞噡䩥㕪潬衴䉷聸㡺㱽啿媅妉沊誕㺗憚撣嚪殫玬䮰㲷澸暺膽馾䳉擔㗖狚觛㫜柦䃫瓭駳黵泹㯯㐤贍醈䦔䊤䏙娊唌䗺䪴鴤単燀砃蜑甔紞鴠霮丼弾彈噉癉襌酖啗幨窞撢禫躭僤髧勯劻儃蟺贉餤黮皽抌黕")
    ("dang" "当党档荡挡宕|砀铛裆凼菪谠垱珰玚|筜|鐺礑欓蘯圵氹逿䑗灙襠艡譡嵣當瞊䦒檔璗讜澢趤碭䣊壋擋䣣闣黨雼瓽偒潒崵婸儅愓簜攩簹蕩噹璫盪蟷")
    ("dao" "到道导岛倒刀盗稻蹈悼捣叨祷|焘氘啁纛刂帱忉捯|舠鱽梼|濤㨶幬䊭薵㔑䮻䣣㠀搗䌦朷幍衜衟噵䆃嶋嶌隝檤隯禱嶹䲽䧂軇㿒釖瓙盜島燾菿虭嘄㧅翢絩箌陦導稲魛禂壔擣翿")
    ("de" "的地得德底|锝嘚扽||㝶㤫㥀鍀㥁恴㝵䙷䙸悳淂棏㯖脦扥徳惪農")
    ("dei" "得|嘚||䮻哋")
    ("deng" "等登邓灯澄凳瞪蹬|噔磴嶝镫簦戥|澂憕璒|䔲嬁鐙鄧䠬霯邆隥䒭墱䮴覴櫈燈僜竳朩豋")
    ("di" "的地第提低底抵弟迪递帝敌堤蒂缔滴涤翟娣笛棣荻谛狄邸嘀|砥坻诋嫡镝碲骶犒氐莜柢籴笫羝睇觌玓杕|轪茋浟菂媂髢禘䗖踶鞮蹢|䲦赿䞾䨢䨀樀㰅焍蔐鸐頔唙䴞怟眡䨤㼵䀸阺䀿䱃奃詆啇㹍坔䍕摕䩘坘偙䩚聜遞䵠腣潪鉪䑭䑯䱱㡳虳楴豴祶慸㭽嵽嚁隄㦅㪆墆趆掋䶍䢑墑梑肑袛䂡䮤㢩䊮馰嶳䞶覿僀埊䣌滌䏑鏑苖㣙䧝埞拞締䟡藡仢諦軧旳糴䯼仾磾扚魡儥埅鬄碮苐俤䪘眱䐭䏄蚳哋甋蔋逓蔕弤渧鍉呧靮啲牴敵彽梊螮藋觝菧釱篴珶廸適惸諟揥墬遰厎蝃")
    ("dia" "|嗲||")
    ("dian" "电点店典奠甸碘淀殿垫颠滇癫巅惦掂|癜玷佃踮靛钿簟坫丶阽|扂琔磹|䀡頕䩚㸃蜔錪㼭鈿橂䍄㥆蕇奌顚顛婝敟㵤㝪嵮婰癲齻墊㶘㞟厧澱㚲嚸壂䧃槇䟍巓巔槙點㓠䓦橝痶蒧琠攧瘨敁蹎傎痁電跕腍驔詀")
    ("diao" "调掉雕鸟吊钓刁貂凋|碉鲷叼啁莜踔铫铞汈|窎椆嬥|䳝䄪䄷䲽伄訋弔䔙鈟鼦弴瘹䠼鑃䉆扚奝魡虭鵰䵲㹿鮉㪕瞗銚㒛殦䂪銱䂽調鯛釣竨鳭矵鋽㢯錭絩䳂䯾軺雿鵃淍㣿藋琱屌彫誂窵蛁蓧鉥匽劚")
    ("die" "跌叠蝶迭碟爹谍牒|耋喋堞鳎瓞鲽垤揲蹀|昳咥绖䏲楪|䳻蹛嵽㦅鰈耊褋氎䴑挕戜䠟㬪䘭眰褺牃詄㥈恎聑艓㑙㭯㩸㩹䲀嶀疂趃䞇疉疊䞕䮢䪥㲲㲳㦶䳀胅峌諜㻡曡惵苵臷㷸㫼啑䩞崼鰨怢㤴鞢軼殜挃螲䫕㗧跮哋攧眣蜨絰畳镻跕槢柣")
    ("ding" "定订顶丁鼎盯钉锭叮仃铤|町酊啶碇腚疔玎耵|萣|訂頂帄鼑㼗娗錠鐤甼㴿顁㝎饤奵嵿檙掟碠㫀忊鋌濎釘矴磸㝪㞟䵺艼椗靪虰薡飣")
    ("diu" "丢|铥||颩丟乣銩")
    ("dong" "动东董冬洞懂冻栋侗咚|峒氡恫胴硐垌鸫岽胨夂||㨂挏笗霘戙嬞崠崬氭㼯㜱倲昸娻㑈䅍䵔腖東䍶鶇䞒䂢㢥墥㖦炵箽菄㓊諌働凍動峝棟鯟㗢駧埬絧詷䆚㣚姛狪勭迵鴤㣫揰鼕蝀涷苳蕫湩騆")
    ("dou" "都斗读豆抖兜陡逗窦|蚪痘蔸钭篼||䇺䄈唗唞鬥䬦鬦阧㨮吺䕆兠浢敨䕱橷饾讀㢄閗枡斣梪㞳㪷鋀㷆竇㛒郖䛠毭瞗㤱㓸氀侸鈄酘餖鬪鬬鬭脰荳枓読闘")
    ("du" "度都独督读顿毒渡杜堵赌睹肚镀渎笃|嘟犊妒牍蠹橐椟纛碡黩芏髑厾阇||䰩㓃讀㹍瀆䄍錖䐗䈞儥蠧騳匵簵嬻琽䀾帾㸿豄䅊鍍牘鑟獨㱩靯荰䙱蝳㾄䪅暏醏螙涜犢䲧覩妬殬䢱䮷皾瓄韇槖櫝凟篤韥賭䓯䫳黷秺䟻裻剫䐁樚晵䦠㒔㞘鍺㧻㡯噣罜剢読頓贕斁讟喥殰闍睪襡韣陼")
    ("duan" "断段短端锻缎|煅椴簖|塅瑖|篅褍耑䠪鍛籪剬㱭葮偳鍴媏躖斷㫁緞㟨腶碫毈")
    ("dui" "对队敦兑堆|碓镦怼憝|祋|㨃桘䬈對鐓㠚鐜㬣頧瀩䔪䨴䨺䬽㙂兊兌轛嵟塠陮㵽隊綐垖憞薱㟋䇏磓㳔䯟懟䇤濧濻痽䏨膭謉㢈瀢㢂鈗鴭搥錞譈対鋭杸啍")
    ("dun" "不吨顿盾敦蹲墩囤沌钝炖|盹遁趸砘镦礅惇楯蹾坉扽||鐓鐜憞伅逇鈍䤜踲蜳崸㬿驐腞潡噸撉躉犜鶨墪撴燉橔頓遯庉")
    ("duo" "多度夺朵躲铎咄堕沱舵驮垛惰|锗哆踱跺掇剁柁缍沲裰哚埵杕|饳剟亸|㤞䁟奲袳㪜腏跢悳䄍挅挆鬌㔍夛崜䠤茤䤪鈬尮椯刴鐸䤻䐾䙃陊陏㙐敓䩔桗襗敚䅜䙟敠䩣䙤䑨㥩奪剫䍴鵽貀嚉袉莌趓垜憜綞炧炨墮墯躱馱䒳鮵㖼䫂軃㛆㛊痑㻔凙㣞痥跥㻧䯬毲㧷飿隓跿䲊㨊䝐鍺㡯誃喥朶敪嶞嚲駄柮媠沰柂硾")
    ("e" "额阿俄恶鹅遏鄂厄饿峨扼娥鳄哦蛾噩愕|讹锷垩婀垭鹗萼桠谔莪腭锇颚呃阏屙苊轭蚵硪丨钶|姶涐堨崿|娾呝匎䜙洝痷騀蘁㼂䄉堊㠋鈋椏砐鰐餓圔唖訛頟㼢琧砨㔩㼰阸戹搹䔾娿蝁湂硆䝈㡋睋譌遌額㕎顎䕏偔鍔蕚鵝啞鵞豟魤䑥魥㱦鑩䑪䱮䙳㩵屵鱷䩹歺卾蚅岋㦍讍誐䆓貖㮙鶚覨䞩䖸妸咹閼㖾妿磀㗁㷈㗉峉䓊㧖䛖䳗迗䳘軛䣞埡諤㟧鋨峩僫䳬櫮珴㓵軶䫷廅鰪齃佮頋歞閜擜砈䚰礘㩽䛩㧴㣂㫊㛕齾囐隲頞搤阨餩吪堮齶詻皒枙咢悪嶭惡囮痾輵噁")
    ("ei" "|诶欸||誒")
    ("en" "恩|嗯唔摁蒽||䬶奀䭓䭡䅰㕶煾䊐糦峎饎")
    ("er" "而二尔儿耳迩饵洱蛾|贰铒珥佴鸸鲕|陑耏咡峏谞臑|餌㜨栮鴯眲䌺荋聏兒轜鉺誀㒃㖇邇㮕侕䮘鮞䎟䎠隭貮䎶㚷袻㢽檽㛅䋙毦䋩䏪㧫䣵髵粫杒髶腝輀弍洏弐尒尓栭唲樲刵爾衈児貳薾駬胹陾卲")
    ("fa" "发法罚乏伐阀筏|砝垡珐|酦擿|笩彂撥貶㳒栰㘺琺橃灋蕟罰佱罸㕹瞂醗傠䒥閥醱䂲疺姂䇅藅髮㛲䣹茷発發髪沷")
    ("fan" "反范犯繁饭泛翻凡返番贩烦帆樊藩矾梵蕃|钒幡畈犭蘩蹯燔璠氾|墦|䕰㾱噃䪻柉㳎㴀䀀㤆訉㸋嬎嬏䌓㼝䀟舤匥舧䐪瀪礬㠶瀿㝃䉊䡊杋橎襎䉒鱕㕨畨煩䭵㽹奿㺕㶗䪛䒠薠憣䪤䒦䊩販䮳羳䛀緐軓盕旙忛飜僠凢䋣凣棥䋦釩勫軬飯飰䣲䫶滼㮥楓䟪婏犿㛯払鐇嬔蠜鄤笲笵汎籓轓繙魬籵膰颿範旛鷭渢")
    ("fang" "方放房防访纺芳仿坊妨肪邡|舫彷枋鲂匚钫昉|牥蚄鳑|眪䉊鈁㤃眆鴋昘鰟紡訪㑂㕫魴汸䦈䢍鶭䲱趽埅旊㧍㯐瓬䰃錺倣淓髣")
    ("fei" "费非飞肥废菲肺啡沸匪斐蜚妃诽扉翡霏|吠绯腓痱芾淝悱狒榧砩鲱怫篚镄|朏剕棐|墢㸬杮㾱㓈賁䍨䞳犻䯋䟛㹃䒈橃蕟䀟䰁渄猆䈈萉䠊㔗騛鼣餥蜰昲䤵䨽䨾䕁靅㩌㵒婓䑔蕜靟䕠䉬㭭㥱䆏厞䚨誹費胇曊緋䛍飝櫠鯡廢蟦裶濷橨紼陫婔騑鐨癈奜屝馡疿廃飛俷蕡笰胏")
    ("fen" "分份纷奋粉氛芬愤粪坟汾焚皂酚|吩忿棼玢鼢瀵偾鲼棻坋|翂豮|朌肦獖㱵賁燌㻞䀟䢍䆏䴅弅朆帉砏昐鈖紛訜㸮餴鐼饙鱝橨奮衯㥹䩿炃岎羒馚蚠妢憤梤㮥馩隫墳蒶㖹膹黂㷊㿎秎燓糞䯨僨黺䫞葐竕鼖坆轒兝蕡幩魵豶枌蚡羵濆雰鳻錀")
    ("feng" "风丰封峰奉凤锋冯逢缝蜂枫疯焚讽烽|俸沣酆砜葑唪沨|浲崶赗|䩬熢䵄䒠㕫瘋鴌猦漨焨縫蠭甮蘴鄷堸捀艂灃煈靊偑摓楓湗䙜㡝㵯桻䩼㦀犎莑檒妦馮麷碸篈䏎鋒鏠凨䟪凬凮峯鳯鳳寷諷仹僼盽舽炐鵬埄㛔夆渢豊豐覂綘風飌賵炰鉥")
    ("fo" "佛|||㤇坲髴仸仏")
    ("fou" "否|罘缶芣|垺|䬏鴀缹缻剻殕䳕裦雬哹妚紑炰")
    ("fu" "府服副负富复福夫妇幅付扶父符附腐赴佛浮覆辅傅伏抚赋辐腹弗肤阜袱缚甫氟斧孚敷俯拂俘咐腑莆孵芙涪釜脯|茯馥绂讣呋罘麸蝠匐芾蜉跗凫滏蝮驸绋蚨砩桴赙菔呒趺苻拊阝鲋怫稃郛莩幞祓艴黻黼鳆茀芣|㕮㳇玞韨枹砆洑垺琈䓛鄜榑簠|蓲㞎颰㚘䈏䥤㾱䬪㝿䒄㗘柭畐䭮柫䍌䪔䳝畗㷆萉㔗昲㡝髴䳕哹䘀訃䘄㠅錇娐㜑鰒㤔輔鴔伕䌗䨗縛䘠怤焤紨稪萯䔰㤱䨱椱紱鬴䠵甶嘸輻㬼紼踾䌿栿乀艀癁詂䩉畉䡍䕎㙏婏鍑䍖㵗䵗鉘虙陚蝜鉜筟鍢絥婦䑧蕧豧㽬捬荴乶䭸䝾䵾坿彿䒀㪄鮄䎅㚆䒇㾈暊妋媍枎綒鮒澓㚕垘䪙膚䮛䞜袝鶝䞞邞負䦣䂤炥蚥冨復麩岪撫颫麬䞯粰麱璷䞸玸冹䗄姇翇旉㟊峊俌䫍竎糐䟔胕蛗㫙盙駙䫝䧞㓡釡賦鳧䋨諨鳬䟮緮懯烰䯱棴軵䋹鳺賻䃽䯽秿荂郍邚捊圑纀箁䞤汱撨酻袚払嬔仏尃蜅複鈇褔帗刜弣頫笰輹葍祔罦偩鵩衭咈綍箙沷蚹鯆韍柎郙泭巿俛襆痡沕湋")
    ("ga" "夹轧咖嘎尬噶旮伽|尕呷钆尜||嘠錷玍軋釓魀")
    ("gai" "改该盖概溉钙丐芥|赅垓陔戤胲|荄晐|䬵䶣匃匄漑鈣絠摡畡㕢乢葢豥絯屲該祴隑㮣䪱瓂郂賅㧉忋賌峐䏗姟槩槪杚䐩阣侅蓋")
    ("gan" "干感赶敢甘肝杆赣乾柑尴竿秆橄矸淦|苷擀酐绀泔坩旰疳澉玕扞|虷浛嵅鳡|玵豃佄䔈稈贑䤗贛倝蜬尲簳尶尷紺攼詌䵟鱤衦汵乹幹桿亁㺂芉檊粓趕皯䲺涻迀䯎凎忓䇞仠飦䃭盰鳱凲笴㔶鰔魐漧錎䊻㓧灨筸咁榦骭諴")
    ("gang" "港钢刚岗纲冈杠缸扛亢|肛罡颃戆筻舡冮|矼堽|頏戅戇崗䴚焵罁塂罓㽘剛牨犅掆碙鎠岡綱㟠棡㟵鋼堈笐疘槓釭犺阬")
    ("gao" "高告搞稿膏糕镐皋|羔锆杲郜睾诰藁篙缟槁槔|筶|䨢㵆獋勂稁縞攪夰餻吿祮橰煰祰㚏皐㚖羙禞誥鎬㾸檺菒滜韟䗣臯鋯鷱藳槹槀叝䓘鷎鼛稾睪暠髙櫜")
    ("ge" "个合各革格歌哥盖隔割阁戈葛鸽搁菏胳舸疙铬骼蛤|咯圪阖镉砝颌仡硌嗝髂鬲膈纥袼搿塥哿虼牁|郃滆|鉿頜㭘詥秴葢䘁嘅愅紇個匌謌䈓戓騔䐙鴚䨣戨鰪㨰笴㠷鴿齃呄晄敆敋䩐㝓獦㵧牫佮魺䕻鉻䪂鲄䢔鎘䆟閣閤麧鮯㦴鎶鞷䪺㪾䧄槅㗆䛋韐滒蛒裓䗘韚䫦嗰擱㷴臵諽䛿杚牱鴐鉀䩡鞈吤䅥鵅鮥䬣㓣鎑䣬猲蓋挌茖渮輵轕亇箇肐擖觡仮扢犵")
    ("gei" "给|||給")
    ("gen" "根跟亘|艮哏茛||刯㮓亙䫀㫔揯")
    ("geng" "更耕颈庚耿梗埂羹哽|赓绠鲠|浭暅鹒|䨣刯亙䌄搄縆頚堩挭頸恆畊䱍䱎絙絚䱭䱴莄綆㾘羮䋁鯁賡菮揯峺䢚䎴鶊骾郠緪")
    ("gong" "工公共供功红贡攻宫巩龚恭拱躬弓汞|蚣珙觥肱廾|唝|䱋渱䍔玜䐵䔈贛匑㼦礦㤨愩㔶摃幊䡗杛㭟塨穬鞏龏鞐龔䢚貢㺬䂬宮躳厷碽羾銾㓋㫒鋛䇨㧬蛬㯯拲觵髸糼輁㕬魟嗊疘釭栱篢")
    ("gou" "构购够句沟狗钩勾苟垢枸|篝佝媾诟岣彀缑笱鞲觏遘|泃耇姤雊馌|䪷豰耈耉㜌㨌鈎夠褠唦茩簼訽㝅㝤鉤詬䝭坸煹豿䞀撀㺃瞉傋溝覯芶蚼玽構䃓㗕緱㳶購㽛袧痀䵶䐟搆茍冓韝軥呴")
    ("gu" "股古顾故固鼓骨估谷贾姑孤雇辜菇沽咕呱锢钴鹄箍汩|梏痼崮哌轱鸪牯蛊诂毂鹘菰罟嘏臌觚瞽蛄酤牿鲴榖|羖堌䓛蓇馉瀔|怘䉉啒磆吿唂唃夃䀇㼋頋縎稒崓鼔䀜笟鴣䀦䐨錮䀰蠱愲鈲尳脵餶鈷焸䜼詁轂䍍杚䍛鵠顧䡩橭䵻㽽䅽㚉鮕䶜薣䮩㒴㾶皷䊺鶻皼淈賈㯏糓嗗狜棝鯝䓢柧軲嫴㧽苽櫎䡰䇢穀罛扢祻箛榾凅蛌泒盬僱軱濲惸呴")
    ("gua" "括挂瓜刮寡卦呱|褂蜗聒剐胍诖鸹栝脶|坬蹐|焻㓡諣䏦䀨捖䯏䈑瘑頢鴰踻葀歄腡啩剮煱蝸詿劀冎趏掛颳㒷膼銽懖叧緺袿㱙咶騧筈絓罣罫髺銛")
    ("guai" "怪拐乖|掴夬||䃶㽇癐箉䂯䊽㷇叏㧔廥摑罫恠枴旝柺")
    ("guan" "关管观馆官贯冠惯灌罐莞纶棺|矜倌鹳鳏盥掼脘涫琯瓘毌|婠蒄祼筦爟鳤|閞鵍䂯樌舘鸛鰥㴦錧輨館䌯礶䘾潅罆癏䙛摜鱞慣遦䩪䙮鱹䝺䪀覌䦎䲘䎚㮡窤貫綸悹悺觀䗆鏆䏓闗䗰泴淉瘝丱鑵関観矔雚關痯")
    ("guang" "广光逛犷|胱咣桄|垙洸珖|撗趪䍍輄茪灮獷侊炗炚炛黆矌櫎僙㫛烡廣臦臩姯㤮硄俇挄広飡銧謭迋")
    ("gui" "规贵归轨桂柜圭鬼硅瑰跪龟匮闺诡|癸炔鳜桧皈鲑刽晷傀妫炅沩庋簋刿宄匦氿|邽姽珪硊跂筀廆僇鬶|㱦鳺䃽櫰䃶禬癐嬀䌆昋䰎椝猤䤥䠩匭攰攱㔳㨳䐴㸵鬹䙆陒鱖襘潙䁛鱥祪摫詭敮䍯䍷㙺㩻䝿楿亀瞆㪈䞈溈劊劌規窐檜龜䞨閨暩庪厬膭䖯媯㲹鞼袿䣀櫃䇈瓌軌郌䳏蓕巜蟡嫢䯣槣觤㧪蛫胿簂䐩䤆謉䈐椢騩蘬匱湀撌槶䟸瞡赽朹㰪䮹螝吷鴂帰恑歸佹溎垝璝嶡鮭貴槻槼鐀茥瞶趹巂")
    ("gun" "滚棍辊|衮磙鲧绲丨||睔㨰鰥䙛㔳䜇輥謴㙥䵪睴鮌掍袞璭䎾䃂緄㫎蓘裩㯻滾錕惃袬裷緷蔉鯀")
    ("guo" "国过果郭锅裹帼涡|椁囗蝈虢聒埚掴猓崞蜾呙馘馃粿|腘|楇瘑矌櫎錁簂䤋漍䬎鈛餜堝渦䐸䴹鐹腂鍋彍過摑慖幗䙨啯㕵㶁㞅㚍墎䆐㖪綶䂸㳀濄惈蟈淉槨囻埻彉蔮㗥咶圀國嘓輠聝膕咼菓囯囶")
    ("ha" "哈虾蛤|铪||蝦鉿奤")
    ("hai" "海还害孩亥咳骸嘿骇|氦嗨颏胲醢嗐|咍浬|䠹㱚䝳拸䍖䂤㨟㤥頦䠽㜾饚䱺酼㺔㦟䇋䯐㧡駭烸㕢絯䪱郂㧉䜕妎餀還侅皁")
    ("han" "汉韩含旱寒汗涵函喊憾罕焊翰邯撼瀚憨捍酣悍|阚鼾颌邗颔蚶晗菡顸犴焓撖扞|虷垾浛琀崡嵅蔊暵㘎|厈貋䎏笒䦈䏎㸁猂䈄䐄圅頇攌䌍㨔㘕㘚頜甝䨡娢㜦㼨蜭䤴䘶頷譀䥁豃㵄佄顄㙈㽉㵎䍐䍑㙔䁔晘筨浫㙳㽳㑵屽魽䕿㮀嚂䶃涆㒈㪋澏梒䖔皔㺖莟馠㖤㲦䮧㢨䎯馯㶰銲炶傼鶾䓍鋎㟏㟔䗙䛞闞鋡嫨闬䧲䫲䏷哻駻凾䓿蛿蜬攼汵仠鳱䗣鬫㰹㺝蘫㦑肣鈐欦顩䜗軒豻唅漢谽睅晥閈螒熯淊韓雗釬迒榦椷歛")
    ("hang" "行航杭巷夯蒿吭珩|酐桁沆绗颃鸻||䴂頏䘕㤚䀪絎筕魧蚢貥䦳䲳斻苀䣈䟘忼雽䂫㬻䢚㰠邟妔迒笐")
    ("hao" "好号毫豪耗浩郝皓昊镐蒿壕|灏嚎濠蚝貉颢嗥薅嚆鸮|诐淏鄗皞|椃䬉㬔蠔㠙鰝瀥昦㘪儫㬶嘷㝀㵆獆籇獋恏噑獔聕㩝灝䝞號虠䝥顥晧㙱㕺悎傐皡暤皥㚪暭鶴䒵㞻䚽䪽侾勂滈曍䧚哠狢䧫䯫峼鎬竓藃譹薃澔皜秏諕暠薧呺茠")
    ("he" "和合河何核贺喝赫荷盒鹤吓呵酶禾菏壑褐涸|阂阖劾诃颌嗬貉曷翮纥蚵盍饸盉鞨|郃垎隺龁滆鹖熇翯龢|䙓㚘噈㾑䮤㻧䯨頜㙳㪋㲦鶴狢爀萂焃礉鸖蠚㬞㔠䴳訶訸楁䅂煂㹇癋靎靏㵑㭘啝㕡䕣詥㕰㭱䵱㥺䚂䶅熆嚇貈㪉咊䎋碋袔澕䢗䪚蒚螛㮝閡㦦䞦䒩㮫皬鶮抲姀賀廅峆柇盇寉㷎䃒惒闔篕䫘䳚釛㿣㷤哬㓭秴䓼毼䳽㿥喛紇齃敆魺鲄䢔閤麧鵠䐧㰤㪃厒餄輅㰰㧁䶎鑉䓇欱轄㗿藃籺謞搤猲餲鉌齕佫覈暍鶡趷嗃挌渮犵")
    ("hei" "黑嘿|嗨||潶㱄黒")
    ("hen" "很狠恨痕|||詪鞎㯊䓳拫佷")
    ("heng" "行衡横恒亨哼珩|桁蘅绗黉姮鸻|堼|絎䄓䬖䬝㔰鴴鑅啈橫楻㶇澋撔悙誙䒛䯒揘諻恆烆胻脝佷狟")
    ("hong" "红宏洪轰虹鸿弘哄烘泓汞|訇蕻闳讧荭黉薨吽竑玒|纮翃唝浲硔谼|㗢㷎䬝澋撔娂紅訌䨎鈜霟焢䀧鬨䜫紭渱㬴輷渹鴻舼耾谾潂彋䡌呍晎䡏䩑葒葓䍔鍙轟硡鍧顭汯鉷䉺妅綋閎䞑㖓䆖宖玜粠䪦䲨䂫㢬垬㶹哅䧆揈篊黌䃔仜翝軣竤瓨苰峵䫹䫺叿嚝愩厷銾屸霐闂㧦巆紘谹魟浤灴鞃澒閧嗊鋐吰謍")
    ("hou" "后候厚侯猴喉吼|逅篌糇骺後鲎瘊堠齁|郈垕鲘|㸸腄䫺洉㬋㤧帿缿㕈䙈葔豞鱟㖃㺅䂉銗鮜㮢䞧䪷犼㗋䗔㫗䫛矦䳧翭詬䞀䡩㰯翵㰻鄇餱鍭睺鯸呴")
    ("hu" "和护许户核湖互乎呼胡戏忽虎沪糊壶葫狐蝴弧瑚浒鹄琥|扈唬滹惚祜囫斛鹘笏醐猢怙唿戽槲觳煳鹕冱瓠虍岵鹱烀轷縠|昒昈峘隺淴欻鄠嫭濩鳠|俿雽鶮㷤鍙焀萀簄鸌帍鬍嘑䠒弖鰗怘蔛嘝匢謢䨥㸦匫瀫㨭䔯蔰許戲戶頶䈸䨼䭅㕆䉉䭌䭍歑啒䍓乕恗絗歘楜虝婟摢䁫豰㹱魱䩴䕶護鍸汻䉿䊀垀䎁抇㺉枑綔喖鶘䚛嚛螜䪝媩熩㾰䞱冴䲵㪶箶䮸㦿䗂㗅䛎雐惖䇘泘㫚㯛寣軤㿥滬嫮淲瓳槴曶壷滸苸鳸壺䧼烼鱯嗀韄䩐縎鵠䵻䊺鶻淈㧽㽇㺀礐㴶㱿㦌㳷芐䤕鴩瓡嗃頀餬戯吰搰戸謼虖衚楛幠沍鶦膴臛芔擭鋘觷")
    ("hua" "化华划话花画滑哗豁骅桦|猾铧浍狯砉粿|婳觟鹢|魤䊐䴳䔯嬅䠉㠏舙㠢䔢嘩錵樺譁㩇㭉驊蕐㕦畫譮話㕲畵㕷杹䱻䅿劃澅㦊㚌㦎螖䶤澮㮯芲㟆磆䛡諣䏦鷨釫槬華㓰鏵竵䀨㼫䯏䇈腂䋀姡黊搳崋繣摦嫿咶檴罫鮭輠莋鋘釪")
    ("huai" "划怀坏淮徊槐|踝孬||㠢劃䴜瀤䈭耲㜳蘹蘾諙櫰竵䃶懷歠褢褱佪咶壊懐壞狥")
    ("huan" "还环换欢患缓唤焕幻痪桓寰涣宦垸|苋洹浣豢奂郇圜獾鲩脘鬟萑逭漶锾缳擐鹮|荁峘㬊貆澴嬛镮|䊐螌豩㢰䁵攌䁔䒛䠉㠢鰀鴅㬇萈愌輐䀓瘓渙阛䴟䈠䀨㼫酄嵈瑍鵍奐㕕㹖捖䝠歡煥䥧繯鍰㡲豲䭴㵹䍺㶎肒梙喚喛貛䮝䆠䦡瞣㦥羦環㪱嚾犿懁鯇雈㓉烉寏䯘藧觨緩㣪㿪糫槵䴷睔䄆絙鸛㔳巜孉䑏鉮綄䴉䴋欥蒝還鐶睆歓轘驩讙梡澣換瓛狟闤懽雚瞏蠸")
    ("huang" "黄荒煌皇凰慌晃潢谎惶簧璜恍幌湟蝗磺隍徨|遑肓篁鳇蟥癀|洸喤媓瑝滉锽艎|䄓楻揘諻篊汻䅿鐄鰉謊䌙朚騜䐠縨㨪餭愰崲䐵㤺㬻塃鱑穔奛䁜䑟葟鍠䅣兤詤䍿䪄䊗撗嚝㾠䊣皩趪㾮䮲墴㞷䞹宺曂軦䳨鷬晄櫎爌䵃䀮炾堭怳衁獚偟熀皝鎤榥熿黃巟韹")
    ("hui" "会回挥汇惠辉恢徽绘毁慧灰贿卉悔徊秽溃荟晖彗讳诲珲堕诙蕙|晦麾烩茴喙桧钺蛔洄浍虺恚蟪咴隳缋哕翚㧑|袆烠翙廆橞|䁤䢈䫖墮㺔䠉譮澮洃㜇䌇嬇瀈䜋阓蜖褘圚輝㨤㰥逥䤧蔧㬩㤬鐬頮匯餯鰴儶㨹琿鼿䕇䩈䙌婎䅏湏幑㩓潓譓噕饖彚䙡穢繢㥣噦噧靧㩨獩繪顪譭詯㑰潰㱱灳睳詴㑹詼譿媈禈薈暉㒑颒憓檓隓䂕嚖㞧殨誨禬鮰暳瞺叀毀㷄賄㻅毇泋㷐蛕囘櫘䛛僡韢䃣烣䧥䏨囬燬翬䫭寭揮藱諱燴䇻廻拻䛼廽䵻㽇㷇䎚䰎襘䍷檜鞼㫎䶐蘬瘣孈㣛蘳㾯嬒椲螝銊戯佪會褌嘒噅煇繐豗彙恛恵絵芔蚘撝璯濊痐滙闠槥篲迴鏸翽楎輠铓鉥薉徻鑴")
    ("hun" "婚混魂浑昏珲荤|馄诨溷阍|惛婫|䊐湷䐊㮯䮝觨琿堚餛焝䰟㨡倱鼲䴷渾繉轋顐睔䅙睧㑮敯䅱㥵祵葷䚠涽閽俒棔棞䫟諢䛰䧰忶鯶惽㨰輥䵪掍緄㯻䴹㕵緡㖧㗃眃梡濊圂餫昬慁楎睯殙梱焄")
    ("huo" "和活或货获火伙惑霍祸豁|嚯藿锪蠖钬耠镬鹱夥灬劐攉砉嚄|佸漷濩彟㸌鳠|鍃㶡䬉焃㕡咊䨥豰㯛㩇㚌㦎㓉䄀䄆礊謋䄑眓瀖㘞騞䰥鈥㨯耯萿靃捇楇鑊䉟彠艧䁨剨癨穫奯鱯獲䂄閄禍沎讗䦚㦜掝咟蒦貨邩㦯嚿嗀瓁韄矆㯉䯏矐臒旤曤䋭俰㗲秳䣶㨰趏䐸㖪姡秮䱛吙諕臛濊嗀湱檴雘擭矱咷篧")
    ("ji" "济机其技基记计系革期际及集级几给积极己纪即继击既激绩急奇吉季疾迹鸡剂辑籍寄挤骑圾冀亟寂暨脊跻荆肌稽忌饥祭缉棘矶汲畸姬秸藉瘠骥羁妓讥稷蓟悸嫉岌|叽伎鲫诘楫荠戟箕霁嵇觊麂畿玑笈犄芨唧帙屐髻戢佶偈笄跽蒺萁乩咭赍嵴虮掎脔齑殛鲚剞洎丌墼蕺彐芰哜罽伋|枅垍姞堲䓫徛鱾惎耤漈踦觭鹡穄蹐瀱|㱯魥㱦岋䆅䓧脨覘䈕姼茤䦈㑵䰥䩐䛋給瞉㨳䍯暩槣䐀紀褀㤂漃㸄鸄㨈計㴉刉簊㘍㠍樍㠎蘎騎䠏䰏刏焏礏尐䤒㔕㴕䐕萕㠖䨖鐖稘記䐚欚級䜞朞㰟䤠稩耭蘮輯嘰㠱愱霵鴶蘻霽績鬾鰿蕀襀㹄嵆㡇鑇塉穊鵋㥍積齍譏齏彑饑䁒湒筓鍓魕坖楖穖癘卙鑙轚㥛魝㱞魢䍤譤驥㑧㙨葪㙫繫㡭鱭㡮偮䩯㭰衱㭲極㡶彶橶䝸㽺聻繼卽兾妀䒁㞃撃㲅㞆䦇羈銈覉㾊斊薊覊鞊䢋誋躋嚌膌墍皍螏䚐钑㾒䶓檕䞘㞛際檝禝喞梞㮟撠㚡瞡銡㖢璣㞦㮨䶩㒫覬䲯嶯犱䢳㾵檵㦸躸㲺䮺薺亼璾箿㧀䗁曁䯂毄櫅諅郆槉㗊䟌㻑竒嗘觙裚鯚峜叝緝雞䋟僟濟擠跡䣢飢忣旣蟣揤狤雦雧賫䳭櫭擮臮鳮磯鯯㗱韲㳵鏶㻷賷懻蓻䓽鯽䁶憿㸅訐紒艥趌谻屰㞋鷑㓹䀈踑帺㞓䛴淁趞䀘鈘乁蠀吇㧗蒩簎䱥汥䇧皀畟茍欿蔇済瘈鈒踖霙丮鰶虀鱀塈繋襋齌蝍齎潗機蹟癠剤穧癪幾鮆羇斉莋鶏劑継禨隮庴鶺鞿鷄濈擊旡勣蛣泲痵苙錤鄿敧懠姫")
    ("jia" "家加价假佳架甲嘉贾驾嫁夹稼钾茄挟迦伽|颊浃枷戛荚痂镓笳珈岬胛嘏袈郏铪葭袷瘕铗跏蛱恝饸揳|泇浉叚斝跱槚|䇲鉿㔠㸦攪䢔賈鴶餃餄唊踋鴐戞㼪䀫頬頰䀹夾挾圿鉀捁㕅硈鵊䁍幏䕛䑝絞䩡乫鉫繳扴腵鉸浹婽鞂鞈宊犌貑㮖斚撟傢榢莢徦㪴鎵撹玾拁埉迌鋏㿓駕䛟郟毠嗧矯價蛺頡䘥忦抸筴仮耞猳椵舺豭榎麚梜檟価裌跲帢")
    ("jian" "建间件见坚检健监减简艰践兼鉴键渐柬剑尖肩舰荐箭浅剪俭碱茧奸歼斩拣捡煎贱溅槛涧笺|谏饯锏缄睑謇蹇腱菅翦戬毽笕犍硷鞯牮枧湔鲣囝裥踺搛缣鹣蒹谫僭戋趼楗谮鞬|钘靬暕篯鳒|㱯黬劗䞼袸䟢㽉㵎㹇㺂䭙聻礀㰄礆儉㔋鰎猏瀐踐縑蠒㔓䤔戔鰔礛騝簡鐧戩蔪樫䄯餰㨴稴㨵挸漸瀸倹鰹䬻瀽錽弿偂虃詃䩆襇䭈襉葌䉍䅐瑐魐鑒䭕湕䵖譖䵛籛䥜繝䭠䵡䵤艦筧鑬繭剱艱鍳鑳鵳鍵睷絸襺鹻譼鹼譾麉碊箋見劍劎劒銒間劔㶕㦗榗澗梘閚㺝趝熞侟䶠檢馢徤薦玪鎫殱殲覵覸螹檻瞼鶼䮿冿撿揀韀鋄䟅藆韉寋糋賎囏諓旔䧖緘䯛䇟䯡監㣤惤賤姦姧㳨鏩諫糮䟰釰䛳磵擶臶㓺㯺淺濺鋻櫼釼鳽跈䌠騫鐱㡨塹㪠黚䫡攕帴涀䶢瞯軒醶椾橏籈瑊朁鋑鈃堅洊鬋鐗減検鰜餞栫謭瀳椷堿牋蕑鑑豜剣豣鑯蕳幵鹸醎閒瞷熸揃廌珔俴菺葥傔揵僣轞")
    ("jiang" "将强江奖讲降疆蒋姜虹浆匠酱僵桨绛|缰犟豇礓洚茳糨耩丬膙|矼弶蔃鳉|䥒䜫瓨塂傋夅將講弜匞蔣強謽漿葁鱂䉃獎畕繦摪繮䁰絳䙹嵹畺摾螀䒂疅㢡醤䞪殭袶壃䋌㯍翞糡勥滰槳彊顜奨奬橿薑醬螿韁")
    ("jiao" "教交较校角觉叫脚缴胶轿郊焦窑骄浇椒礁佼蕉娇矫搅绞酵剿嚼饺窖跤蛟侥狡姣荞|皎茭峤菽铰醮鲛湫徼鹪僬噍艽挢敫藠|峧恔䴔滘斠漖皛燋璬皦皭|㿟㱶㩭䢪嬓樔虠悎稁攪餃踋捁絞繳鉸撟撹矯較嘄漅訆䀊嬌鐎㠐䠛䴛嘦䘨㬭茮鴵㰾㭂孂晈呌敎轎潐譑驕灚穚䥞譥湬㩰㽱㽲腳䁶敽芁䂃斆澆劋覐䪒嶕䆗暞嶠膠䚩鮫喬徺覺憿蟂烄㳅賋鷍曒䣤僥鷦勪臫鷮䡈䍊嵺䰘隦㹾嶣骹釥嫶䙼獥䕧憢敥䉰趭灂摷勦稾嘂儌挍鵁筊煍鱎噭蹻窌憍覚膲釂珓蟜蟭轇摎簥敿趫嘐詨")
    ("jie" "家结解价界接节届介阶街借杰洁截姐揭捷劫戒皆竭桔诫楷秸睫藉栉拮偕芥|阂诘碣嗟颉蚧锴孑婕疖桀讦疥偈羯袷喈卩鲒骱玠劼|岊㛃悈絜湝蜐僇耤鹖褯檞|栨嫅䦈㑵䞦䫘祴䘁紇騔㨰裓鍇㔚㸄㠍㠎尐㡇魝衱鞊䟌揤狤櫭㼪䀹䂃瀄㸅昅砎訐鐑紒䌖蠘蠞頡吤刦崨㨩搩稭丯洯㘶䀷㠹䰺堺脻刼㔾琾䔿䱄屆腉㝌畍㝏楐結啑嵑䁓潔蝔㑘䕙䥛鉣癤䅥嵥艥卪魪煯詰繲楶鍻偼鎅趌媎庎階㾏傑䂒岕犗媘䲙鮚鶛䂝㮞誡㦢徣躤㮮㞯誱䂶䲸喼巀節㿍痎滐擑䇒㓗䣠㓤觧菨䯰擳緳蓵擷飷䛺䗻迼袓疌蛶㫸衸妎斺雃諎艐毑謯㞏仮扢蝍蛣価椄踕㨗倢堦刧唶蠽魀楬幯榤莭掲犵袺緁脥姉")
    ("jin" "进金今近仅紧尽津斤禁锦劲晋谨筋巾浸襟靳瑾|烬缙钅矜觐堇馑荩噤廑妗槿赆衿卺浕|珒琎祲溍瑧瑨墐慬殣|笒䃡㻸䢻䉷䌍㴆䀆縉漌㬐䤐贐伒㬜䌝錦嬧唫䐶㨷㰹謹瘽䥆㱈晉饉㹏歏荕䭙䑤䝲鹶㝻齽嚍䖐䶖嶜劤㶦厪䆮覲㶳璶䒺勁僅濅黅緊惍藎埐凚鋟盡嫤䋮䗯㯲菳蓳䫴㯸僸巹燼㦗榗竻暜斳婜兓馸嬐䃸䤺釿枃堻臸儘紟搢進璡侭肵觔寖濜菫賮")
    ("jing" "经京精境竞景警竟井惊径静劲敬净镜睛晶颈荆兢靖泾憬鲸茎腈菁胫烃|阱旌晟粳靓痉陉箐儆迳婧肼刭弪獍璟|汫倞猄竫䴖璥麖鼱|晠頚頸粇䑤劤勁踁稉㬌䔔逕䜘脛瀞㘫弳剄荊㕋葏坕䡖坙婙聙靚婛鵛葝䵞汬桱㹵㵾鶁鶄涇妌宑徑經莖麠㢣誩亰梷傹憼旍㣏擏曔秔痙燛巠鏡竧淨鯨濪仱烴競竸蟼幜靘䝼坓鋞㸒頴丼浄経驚靜穽暻凈凊俓殑")
    ("jiong" "坷窘炯迥|垧炅冂扃颎|冏坰泂䌹䐃皛|臦臩昋逈蘏蘔㤯顈㑋煛幜澃冋綗䢛㢠㖥侰㓏僒㷗埛㷡囧駫烱䅃扄銄浻㯋褧絅煚熲駉")
    ("jiu" "就究九酒久救旧纠舅灸炙疚揪咎蹴韭|玖臼柩赳鸠鹫厩啾阄湫桕僦齑鬏氿|㠇|䠓䬨殧乣㼋㽱㝌倃舊舏匓匛萛紤丩鬮匶朻䡂㩆乆牞䅢慦齨㡱镹奺㲃䊆䆒麔䊘媨㺩㶭揂廄䳎廐觓䳔㧕䓘鯦鳩䛮韮鷲糺勼柾糾繆㙀剹㤹㥢杦稵樛捄轇㺵廏揫摎穋鉥")
    ("ju" "车局据具举且居剧巨聚距句拒俱柜菊拘炬桔惧矩鞠驹锯踞咀瞿枸掬沮|莒橘飓疽钜趄踽遽琚龃椐苣裾榘狙倨榉苴讵雎锔窭菹鞫犋屦醵焗蒟|岠坥弆岨泃珇帡秬砠竘崌娵筥腒鲏䴗澽|㡹㪯車俥㹼䎤㕢㯯櫃怐蓻勪䳔輂欅怇椇舉輋㘌䤎鄓䄔眗㜘踙昛蘜刟耟鴡焣㠪㬬䰬䈮䜯洰㘲愳鼳簴㨿㩀婅葅鉅絇㥌艍詎䅓䵕䡞䱟齟䱡㽤驧屨湨歫蹫䕮婮陱㩴鵴蝺㮂抅躆劇鮈檋鶋鮍㞐犑袓鮔䪕貗䶙趜冣箤䶥斪鶪㞫邭梮窶颶侷粷蚷䢹躹涺䆽痀䗇巈壉䃊狊諊駏駒跔姖淗據揟㳥凥泦鋦擧毩勮䛯䋰䣰䏱毱駶櫸鋸懼埾䀠爠螶忂㖩閰萭聥㵵蒩趉趡鉏処蹻椈匊踘挙怚蜛倶挶鐻楀鵙罝虡佢粔跙寠拠埧郹跼臄僪豦懅軥")
    ("juan" "卷捐圈眷娟倦绢隽镌涓鹃|鄄蠲狷睃锩桊|焆婘棬蜎|䐪鵍䄅錈餋鐫鬳䌸腃奆鵑䡓剶絹慻羂䚈㢧䖭劵鎸㪻㢾淃雋勌韏裐㯞埢姢菤䳪勬飬㷷瓹蔨呟絭闂梋埍鋗萒襈鋑儁圏朘帣脧睊睠罥獧捲悁臇巻屩弮巂鞙縳")
    ("jue" "决绝角觉脚掘崛诀獗抉爵嚼倔厥蕨|攫珏矍蹶谲镢嗟鳜噱桷柽噘撅堀橛孓蠼觖劂鞒爝玦|叕砄焆傕㵐潏燋玃|髉䍳虳㻡䈑䊽㔳鱖䳏㲄㭰踋腳覐覺䣤勪䤎㔃䠇鐍㰐爑氒刔欔㸕䀗挗䐘㤜鐝弡㔢訣崫逫欮焳爴谻䘿㭈䡈䍊鱊䝌譎彏孒捔絕橜㹟䙠屫㩱穱啳瑴䁷蹷決赽㭾亅䦆殌斍䆕蚗憠䆢龣嶥熦疦玨䞵芵䞷䖼䦼鞽䋉䏐㻕矡鷢䏣蟩㟲䇶㓸㷾䟾袦繑匷鶌誳闋闕泬妜趉䶂吷鴂嶡駃騤蹻覚鴃戄鈌瘚蕝屩絶钁貜撧躩憰趹臄蟨僪觼璚壆繘")
    ("jun" "军均俊君峻菌竣钧骏龟浚隽郡筠|蕈皲麇捃|囷莙珺晙焌葰畯䐃焞鲪|棞亀龜㤯㢠㖥雋䝌攈鈞攟蜠蔨㴫䜭頵呁䝍䕑陖鵘㕙㽙㝦汮㑺桾袀銁馂碅麏蚐㒞銞箟覠皸皹燇埈懏㻒寯姰㓴䇹駿鍕㼱鵔儁餕麕箘鮶軍濬")
    ("ka" "卡喀咖|咔咯佧胩||衉㮟擖鉲呿")
    ("kai" "开凯慨楷恺|揩锴铠忾垲剀锎蒈闿|炌|嵦濭䠽嘅㗆䫦㳀輆鐦䐩愷愾㡁鍇豈塏奒䁗剴䡷㲉炏䒓暟鎧勓闓烗凱㪡欯鎎幆侅愒欬開颽壒")
    ("kan" "看刊勘堪坎砍侃嵌槛瞰|阚龛戡凵莰丬|衎崁嵁墈磡|㵎㙳䖔闞䫲䳚䀍䘓㸝栞輡鬫轁偘歞塪龕冚䶫惂竷䁍䬻檻㪁輱欿歁顑轗矙埳穅扻")
    ("kang" "康抗扛慷炕亢糠|铛伉钪闶||忼摃䦎䲘砊㰠鈧漮鱇䡉嵻粇閌邟躿嫝鏮羫匟穅犺囥槺")
    ("kao" "考靠烤拷铐|栲尻犒|洘熇|㼥䎋稁䘓丂焅䐧鲓銬鮳槀鯌髛䯪燺嵪稾攷薧")
    ("ke" "可科克客刻课颗渴壳柯棵坷恪苛咳磕珂|稞瞌溘轲窠嗑疴蝌岢铪颏髁蚵缂氪骒哿钶锞剋牁|沄匼垎炣悈|錒濭㕎㼎㒆鉿頦䕣袔䫘毼磆礊䘁㝓䆟䗘錁䡷㲉堁礍騍娔樖愙礚㰤㤩萪鈳顆㕉䙐㵣兣敤牱㪃犐厒䶗醘㪙閜㪡碦㾧嶱課㞹㪼殼窼勀勊緙揢胢軻㥛㲺䁍嵑䋉㒞翗愘䯊硞䂺䓇艐趷歁尅渇搕砢簻嵙薖殻榼峇")
    ("ken" "肯垦恳啃|龈颀裉|珢硍龂|褃㸧錹豤齦肎肻墾懇貇頎齗掯")
    ("keng" "坑铿胫吭|铒|硁硍|鉺奟誙䡩娙挳坈鍞䡰摼妔劥銵鏗踁䀴揁阬硜硻牼殸")
    ("kong" "空控孔恐|倥崆箜|埪硿|錓㸜㤟鵼涳㑋鞚悾椌")
    ("kou" "口扣寇叩抠|佝蔻芤眍筘|竘|䁱㸸䧆㜌瞉䍍怐㔚㰯蔲敂㽛摳剾㲄瞘窛冦袧劶宼㓂鏂䳟釦滱䳹眗飡簆彄鷇茠")
    ("ku" "苦库哭酷裤枯窟|骷槁堀绔刳喾|矻绹圐|䊿泏䔯橭狜焅瘔褲㠸桍扝絝顝㒂庫鮬嚳趶郀俈跍胐秙䇢捁崫㗄硞楛袴窋")
    ("kua" "跨夸垮挎|胯侉髁锞|姱|袔恗楇䦚錁㡁顝㐄荂晇誇銙咵骻䋀䦱㛻")
    ("kuai" "会快块筷脍|蒯刽侩浍郐蒉狯呙哙㧟|鲙|澮㨤㻅廥檜巜㫎儈䈛㔞㬮欳鄶塊㙕㙗䭝獪㱮噲䶐墤膾䓒擓䯤糩㟴凷㕟蕢狤㭈㹟會璯旝咼鬠鱠喎駃")
    ("kuan" "款宽|髋||㮯䤭䕀歀䥗䲌髖臗㯘寬㱁梡欵窾寛")
    ("kuang" "况矿框狂旷眶匡筐|邝圹哐贶夼诳诓纩|洭|㾠軦䂄礦穬矌㫛㔞爌眖儣匩㤮鄺昿砿䵃硄彉絋㑌卝鵟筺躀誆纊誑䊯岲貺邼況俇黋軖壙曠懬懭軭忹丱恇絖鑛鉱抂劻迋")
    ("kui" "亏奎愧魁馈溃匮葵窥盔逵|睽馗聩喟夔篑岿喹揆隗傀暌跬蒉愦悝蝰|壸硊骙煃戣櫆|㾠嬇㤬䙌䙡潰㒑䇻㧉䏗㷇䰎䙆䁛㙺瞆膭䖯䯣胿㚍顝欳㱮㟴簀䤆謉䈐䠑㨒椢簣騩蘬蔮尯匱嘳蘷樻湀虁籄晆饋鑎楏楑㙓䕚蹞㕟蕢聧虧鍨䍪䕫聭聵鍷撌憒㚝躨犪䦱窺藈巋䯓巙䫥䳫䧶槶䟸㛻䠏㡭腃㰐䞚䫔闋䠿䃬噅鮭鐀鄈頍騤茥頯刲餽瞶鞹媿闚峞頄磈")
    ("kun" "困昆坤捆|琨锟鲲醌髡悃阃焜|壸堃裈婫鹍|㕎䐊餛祵鰥裩豤齦䠅稇崐崑猑錕蜫睏齫硱㩲瑻鵾涃貇綑閫閸惃髠鯤髨㫻罤鶤褌騉稛晜梱裍菎壼")
    ("kuo" "适扩括阔廓|栝蛞|漷|䩹䙃㣪䄆䄑㨯萿秳頢葀懖㪙㾧挄籗桰鞟䦢擃闊姡㗥䟯擴濶䯺噋筈髺鬠鞹霩適拡")
    ("la" "拉落垃腊啦辣蜡喇剌旯|砬邋瘌鞡镴|蝲|䀳㱞爉㸊攋蠟㕇楋蝋䝓䱫溂䪉䶛辢䂰嚹䏀柆臈菈㻋翋瓎䟑磖臘㻝藞䓥揦揧䃳䗶鯻癩儠㯿䏠搚㩉䖃䱨撦鬎鑞擸")
    ("lai" "来莱赖睐徕|籁涞赉濑癞崃铼|俫梾|䀳鵣誺勑㸊攋倈萊崍猍娕琜㠣䄤瀨逨䠭錸唻顂㥎䅘筙睞籟婡癩襰䚅來鶆㚓䲚徠㾢庲麳䂾䧒賚䋱郲櫴賴䓶棶淶孻騋瀬頼箂鯠藾")
    ("lan" "兰览蓝篮栏岚烂滥缆揽澜拦懒榄斓婪阑|褴廪罱谰镧漤|襕|坔㘕嚂㔋諫䈒䄤䲚䧒爁欄䰐㘓攔㰖儖欖欗爛爤爦㨫䌫爫蘫攬蘭㜮礷瀾䍀籃孄灆譋䑌孏嵐㩜灡㑣襤浨㱫鑭幱顲葻襽繿醂钄䪍斕讕䊖纜㦨䦨㞩厱璼覽䆾韊闌藍壏瓓㳕燗懢燣擥㛦懶燷糷䃹䳿漣僋蘭嬾灠籣襴躝覧壈囒濫惏湅")
    ("lang" "浪朗郎廊狼琅|榔螂阆锒莨啷蒗稂|俍埌崀悢桹㫰烺蓢筤塱㮾|䃹樃朖朤欴䀶䁁艆㝗䡙䍚䕞㙟硠㱢瑯䱶㢃斏誏㾗䆡閬躴㾿鋃勆蓈㟍峎嫏郒䯖㓪脼哴駺鎯樠蜋郞")
    ("lao" "老劳落络牢捞涝烙姥佬崂唠|酪潦栎痨醪獠铑铹栳耢|荖嫪橑|䜎鐒㨓簩嘮耮䝁恅癆䵏絡䝤䕩橯浶澇撈䲏嶗㞠銠憥憦軂㟉䳓䃕僗㟙髝勞㗦蟧䇭㧯磱㟹䜮顟㺐老轑窂労咾狫哰嘐")
    ("le" "了乐勒肋|叻鳓嘞仂泐饹|簕|㨋㿭牞樂餎頱鰳㔹氻㖀玏㦡忇哷韷竻阞砳扐艻楽")
    ("lei" "类累雷勒泪蕾垒磊擂漯镭肋|羸耒儡嫘埒缧酹嘞诔檑|藟礌癗罍|㵽腂㲺㗊㰐㻋郲攂㴃㼍䴎錑䨓頛蠝洡㔣瘣礨頪䐯蘱縲蘲鐳鼺蘽靁䉂灅蕌㑍㹎䉓鑘㡞類轠㵢䍣䍥㭩䉪鑸㙼誄讄銇㒍䮑㲕纝㶟颣㒦䢮禷䒹厽䣂瓃囄櫐櫑㿔壘淚䣦壨䛶㠥祱䃬樏鸓欙礧儽塁虆絫畾纇纍涙傫磥蔂")
    ("leng" "冷梭愣棱楞|塄崚|堎|䬋倰䉄䚏䮚輘㱥碐蔆稜踜薐睖")
    ("li" "理力利立里李历例离励礼丽黎璃厉厘粒莉梨隶栗荔沥犁漓哩狸藜罹篱鲤砺吏澧俐骊溧砾|莅锂笠蠡蛎痢雳镉俪嶙傈叻醴栎郦俚枥喱逦娌鹂戾砬唳坜疠蜊黧猁鬲粝蓠呖跞疬酾刂缡鲡鳢嫠詈悝苈篥轹棁朸|叕峛珕浬浰凓漦醨釐㰀|銐麶黐荲扚蒚䕻鎘㷴兣欚癘䗶㠣㥎㴃錑䍥囄䚏䔁氂䴄爄䬅錅䔆䬆樆蠇䘈䔉攊琍㔏㬏爏㘑刕㼖茘䤙㰚㸚䤚蘚䰛栛䄜䰜㴝瀝㠟㤡䴡攡䔣蠣㤦攦䔧蜧謧礪礫蠫攭縭儮搮礰娳儷脷騹蘺䴻㽁㡂靂䡃艃䅄慄㹈酈孋靋䵓剓鑗䱘塛㽝睝鉝䍠轢筣轣㑦䍦癧鱧䵩驪䉫籬瑮婯䙰鱱穲赲鱳䥶孷蝷㕸㱹鵹鱺䁻䅻䍽犂纅讈䮋邌䊍粍㾐邐㦒躒䚕㾖涖麗䶘㮚粚麜䲞蒞嚟綟㺡悡犡䖥䮥嚦岦㒧悧䊪檪禮厯厲禲粴沵悷梸蚸隸䖽㒿䖿巁棃釃瓅鷅曆囇瓈䧉鯉矋䗍蟍㻎糎䟏䇐䟐瓑鷑䣓櫔棙釙㗚㿛秝䓞曞菞櫟盠蛠壢峢離㯤䋥瓥㿨鳨櫪䣫鋫鏫鯬㓯䃯㧰㷰鋰峲廲㟳雴勵藶蟸㻺觻濿磿颯叓矖豊儗扐鬁栃砅唎欐鴗鸝栵褵蔾灕屴歴歷剺斄貍劙犛厤暦梩皪沴隷裏苙裡盭糲擽戻纚杝柂")
    ("lia" "俩|||倆")
    ("lian" "联连练廉炼脸莲恋链帘怜涟敛琏|镰濂楝鲢殓潋裢裣臁膦奁莶蠊娈蔹梿|浰瑓磏|䐄稴輦戀㜃㰈鰊㼑㼓㜕嬚堜蘝䌞蘞漣連䨬鐮萰謰鰱匲瀲匳褳㰸縺㜻煉孌䭑噒㥕㡘襝䁠籢䥥蹥譧㱨籨聨奩慩聫聮奱㝺䙺㦁䆂斂劆璉螊㶌憐㦑㶑熑薕㢘㪘㪝覝纞薟媡㺦殮亷羷㾾㟀僆䏈鏈㓎濓䃛䇜櫣槤燫㯬蓮練翴嫾槏醶憰鬑錬鄻簾湅鍊摙歛聯鎌溓練澰臉嗹鱄")
    ("liang" "量两粮良辆亮梁凉俩谅粱晾|靓踉阆莨椋魉墚|俍倞悢辌|靚䀶䁁㾗閬倆䠃輌樑簗輛㔝唡脼蜽㹁鍄魎䭪䩫䝶湸掚綡㒳緉諒䓣糧哴駺䣼両輬啢兩喨涼裲刓媟")
    ("liao" "了料疗辽廖聊寥僚燎缭撂撩|嘹潦镣寮蓼獠钌尥鹩瞭蹽|漻憭嫽簝髎|䣈繆䕞㨓䝤㞠蟧㜃䨅䜍鐐爒鄝尞䄦尦䜮䝀療豂㝋䩍蹘繚顟䑠䍡㙩㵳嵺㡻䉼遼镽䎆㺒炓璙嶚嶛䢧㶫熮窲窷蟉飉釕鷚蟟廫鷯賿窌轑摎屪敹憀膋膫暸飂鏐")
    ("lie" "列烈劣裂猎冽|咧趔洌鬣埒捩躐|䴕脟|䋑獦爉㸊䝓䪉䶛䃳哷㭩爄䬅㤡䮋綟巁棙㻺挒鬛㤠儠㬯㼲䜲㸹䅀浖聗㭞㽟䉭煭鱲獵䁽犣鮤㲱颲劽埓㧜䓟巤䟩姴蛶䟹迾㯿燤忚奊栵猟茢鴷睙蛚擸劦")
    ("lin" "林临邻赁琳磷淋麟霖鳞凛拎遴蔺吝粼|嶙躏廪檩啉辚膦瞵懔璘|箖潾翷|癝㖼玪顲䚏㔂㨆崊甐焛㐭鄰瀶橉驎轔繗鱗癛㝝恡轥罧䉮䕲蹸晽㖁檁疄碄悋躙撛綝澟粦躪䚬䢯斴閵厸䮼暽賃懍䫐凜矝㷠壣臨廩䫰䗲痳藺魿㡘㝺䢧䟹獜亃麐隣惏燐僯菻鏻溓")
    ("ling" "领令另零灵龄陵岭凌玲铃菱棱伶拎羚苓聆翎泠|瓴囹绫呤棂蛉酃鲮柃祾|坽姈昤鸰堎舲㥄澪|㭩倰䮚靇錂䴇䈊琌朎䴒䔖霗輘領霛欞㬡䌢刢䄥蘦爧䨩䴫䰱砱䠲㸳鈴紷阾䉁孁䍅婈靈衑䉖䕘齡㱥䙥㡵蕶鹷䉹䡼彾䡿魿䖅㲆㾉皊袊碐掕䚖龗閝麢炩㖫㦭㪮岺嶺綾跉䯍㻏狑駖䧙竛秢㯪鯪裬燯櫺蔆稜〇霊夌鴒霝笭詅睖齢呬醽軨淩")
    ("liu" "流刘六留陆柳瘤硫泵溜碌浏榴琉馏|铆遛鎏骝绺硐镏旒熘蒌鹨锍镠|珋飗罶鹠瑬疁|䒥㳅㧕㚹䉹䝀㝋㵳熮蟉鷚栁䄂鐂瀏䰘䬟鰡㨨㐬騮鬸嬼餾㙀畂畄癅橊㽌驑䱖鉚㽞䱞瑠䉧嵧橮塯畱䭷桺羀䶉劉麍澑嚠沠璢蒥鎦䚧㶯媹綹磂蓅飅廇旈鷎裗䗜駠雡藰駵鋶䋷飹窌摎飂霤蹓坴陸鶹飀翏鏐懰磟")
    ("lo" "|咯||")
    ("long" "龙隆弄垄笼拢聋陇胧珑窿|茏咙砻垅泷栊癃挦眬|昽哢祲漋|谾簼㳥挊㰍攏爖鸗贚蘢朧瀧蠪蠬礱㴳霳挵䡁豅靇㙙㑝籠驡䥢㡣硦鑨䙪㝫湰襱屸聾㚅㢅梇䪊䆍龍嶐龒龓㦕嚨隴徿巄竉䏊瓏矓㟖㛞槞壟壠䃧鏧曨篭櫳䰱䮾蕯篢儱礲衖躘巃竜滝捰")
    ("lou" "楼露漏陋娄搂篓|喽镂偻瘘髅耧蝼嵝蒌窭䁖|剅溇|㡞甊嘍簍樓䄛蔞耬謱㔷瘻婁䝏屚艛摟㥪遱䅹䱾嶁㲎㺏瞜熡䮫㪹螻鞻軁僂髏廔䣚鷜鏤䫫㟺慺寠漊瘺塿")
    ("lu" "路六陆录绿露鲁卢炉鹿禄赂芦庐碌麓颅泸卤潞鹭辘虏璐|漉噜闾戮鲈掳橹轳逯渌蓼撸鸬栌氇胪镥簏舻辂垆箓甪|垏菉琭稑僇勠澛蕗|賁璷繆㪹鱳觻䍡䐂椂瀂錄輅蘆氌蔍攎樐爐䌒䰕鸕瀘㜙樚踛㠠圥蠦鈩㔪䴪鐪舮錴䘵簶騼娽㼾虂轆硉䡎㭔蹗籙䱚䡜虜摝䕡轤鑥鵦睩艪艫䩮顱鵱魲硵鹵塶塷鱸剹獹㱺祿䚄㦇䮉㪐䲐䎑侓膔嚕㪖㢚熝趢枦嚧㖨㪭螰㢳粶䎼鏀賂擄峍勎㓐瓐矑櫓淕鏕㿖髗廘䃙壚臚㯝㯟㟤曥淥盧櫨㛬廬㯭觮鏴滷黸鷺擼㫽蓾䟿䥨櫚坴陸翏騄簬録盧穋罏彔籚艣鑪魯醁玈纑盝磟磠鯥緑")
    ("luan" "乱卵滦峦鸾栾|銮挛孪脔娈|脟|䚕㼑㰸孌奱䏈䜌欒圝鸞攣㝈鵉虊㱍灓灤㡩癴癵鑾孿䖂亂羉薍巒臠釠曫覶臡圞乿")
    ("lun" "论轮伦仑纶沦|抡囵||睔㕢綸䈁圇稐崘崙踚耣蜦輪倫腀婨陯䑳掄侖碖溣㖮惀棆㷍菕論埨鯩淪錀")
    ("luo" "落罗络洛逻螺锣骆萝裸漯烙摞铬骡|咯箩珞捋荦硌雒椤橐镙跞瘰泺猓脶袼猡倮蠃啰|蓏|槖䎅㮝䉿鉻㪾蛒腡䴹䙨㕵㞅㵣㾧絡䃕頱㖀䉓䮑硦㔏攭鱳躒㾖䲞䗍㸹攎䈁㰁鸁㼈欏㴖洜䀩䌱䌴䈷鴼騾笿蘿鵅剆㽋驘饠㩡㑩籮癳㱻鑼玀邏犖鮥䊨㒩㦬鎯覶躶覼䯁峈鏍䇔㿚嗠㓢曪駱濼渃䎊覙詻砢皪蔂攞儸羅纙囉臝捰")
    ("lv" "率律旅绿虑履吕铝屡氯缕滤侣驴榈|闾偻褛瘘捋嵝膂稆|垏梠菉葎溇|哷㡞謱瘻婁䝏䮫軁僂䣚鷜䕡䮉㢳氀爈挔䔞儢㠥㔧縷褸呂㭚捛穞屢驢祣䥨慮慺絽膐㾔䢖綠膢閭㲶侶箻馿鋁櫖郘櫚㻲勴濾寽繂寠漊録魯嵂卛鑢穭膟緑藘")
    ("lve" "略缕掠|锊|䂮圙|㗉攊䤚㔀䌎䤣稤㨼畧㑼䛚鋝鋢寽詻擽")
    ("ma" "么马吗摩麻码妈玛嘛骂抹蚂|唛蟆犸杩嬷麽孖|祃|䠋䀣䠨䧞㕰帓尛嘜鰢嬤㜫蔴㐷㨸獁瑪睰罵㑻閁㦄犘螞禡榪鎷㾺碼媽蟇鷌嗎䣕䣖䯦䗫擵䳸䀛鬕驀礣亇傌溤馬麼駡痲俛")
    ("mai" "买卖麦迈脉埋|霾荬劢|骍|㦟䳸脈䘑䜕霡㜥䨪嘪䨫㼮䈿眿佅蝐蕒䁲邁䚑麥䮮買賣勱鷶衇薶売霢貍")
    ("man" "满慢曼漫埋蔓瞒蛮鳗馒|幔谩螨熳缦镘颟墁鞔嫚|鬘|䒥䨫蔄㬅䰋䜱縵蠻鰻䐽謾饅獌襔䕕㵘䝡㙢㡢䝢顢蹣䡬䑱摱慲䅼悗瞞䊡澷㒼䟂㗄㗈僈鏋蟎矕鏝㛧䛲䯶滿姏睌㿸鄤鬗屘満璊澫槾樠絻")
    ("mang" "忙盲茫芒萌氓莽|蟒邙硭漭牤|杧尨牻|奀㵃䓼朚笀吂鼆蘉䈍蠎娏㬒䀮茻㙁㝑䅒汒㡛浝硥䵨䁳恾䒎䖟莾㻊㟌㟐蛖痝䟥釯盳狵壾㟿䏵䙪㜃㤶㴇杗庬铓哤鋩駹鸏")
    ("mao" "贸毛矛冒貌茂茅帽猫髦锚懋袤牦|卯铆耄峁瑁蟊茆蝥旄泖昴瞀芼眊|鄚楙疐|貇蝐㬒㴘䀤堥錨渵笷萺㡌艒酕䡚罞㝟䅦冃枆媌冐瞐覒暓貓㮘鶜㪞㲠㒵㚹㺺㒻貿䋃㧇䫉㧌柕㿞軞䓮㫯嫹霿愗緢毣氂㹈䖥鉚䭷嵍蓩秏夘鄮戼兞皃冇媢髳毷鉾犛")
    ("me" "么|麽||濹尛麼嚜")
    ("mei" "美没每煤梅媒枚妹眉魅霉谜昧媚玫酶镁湄寐莓袂楣|糜嵋瑁镅浼猸鹛郿|渼媄瑂溦|䞼䆊羙䀜㜫㙁㡌䤂栂脄嬍謎攗䀛䰨䰪挴䜸鬽睂嵄呅湈䉋㭑䍙祙蝞䵢楳睸塺葿䆀鎂鎇䊈䊊沒鶥㶬㺳䒽徾凂篃跊槑燘黣䓺珻抺䱕蘪櫗娒氼矀坆膴眛脢堳腜煝禖沬媺鋂毎韎痗黴苺穈")
    ("men" "们门闷|扪焖懑鞔钔|亹|㙢悗㻊們㨺㥃鍆㡈虋㵍䝧㱪捫門閅㦖䊟暪悶䫒菛燜懣怋䪸玧殙璊樠穈")
    ("meng" "蒙盟梦猛孟萌氓朦苎锰檬|勐懵蟒蜢虻黾蠓艨甍艋瞢礞獴|尨㠓幪鹲|䁅顭䁫朚㽇䝢鼆蘉䈍䖟㻊䟥䀄氋䴌䰒䤓儚萠䠢夢夣霥鄳錳㜴鄸䴿霿靀䥂䑃蕄䑅䵆橗㩚䉚饛䙦䙩䥰㝱蝱㙹䒐溕䲛㚞冡䇇矇䗈鯍矒懜䓝懞鯭䏵雺瓾嫇鱦庬鋂鸏甿莔曚濛")
    ("mi" "米密秘迷弥蜜谜觅靡泌眯麋猕谧咪|糜宓汨醚嘧弭脒纟冖幂祢縻蘼芈糸敉|狝洣祕醾|䋳㘠䀣㵥䣥䉲䊫㦄䈿謎㨺覛熐戂㜆䤉䤍鸍㸏䌏䌐謐樒㸓䌕䌘簚蔝漞㨠攠蠠䴢爢蔤㠧䌩蘪眫瀰渳㴵㜷㰽䁇䱊孊㵋籋彌幎灖䍘罙㥝葞㩢㝥䭧䭩䕳䕷䥸詸獼䉾镾瞇麊羋侎䖑榓覓覔冪䮭䊳檷宻䪾蒾醿櫁釄㣆䛉藌䛑瓕濗㫘峚䋛㟜擟䛧淧㳴滵㳽䣾淿䈼鑖沵摵瞴䖹鼏塓幦羃麛冞銤禰濔沕")
    ("mian" "面棉免绵缅勉眠冕娩|腼渑湎沔黾宀眄丏|勔偭愐|牑㝃㐷䯶㨺㡈緡䌏蠠㥝䛉㫘㤁㰃䤄㬆䀎㴐䰓嬵婂汅蝒靣㝰芇㮌媔喕㒙澠麪粫麫莬檰麵麺綿䏃糆䃇矈矊櫋矏緬㛯㻰臱䫵黽厸䩄葂靦絻鮸俛緜")
    ("miao" "苗秒妙描庙瞄缪渺淼|藐缈邈鹋杪眇喵||㷅䏚仯媌㠺繆鱙㑤䁧䅺鶓㦝䖢庿篎竗廟緢緲彯訬玅")
    ("mie" "灭蔑|篾乜咩蠛||眜䌩㩢羋吀䘊瀎鴓搣礣䈼䩏鑖孭鱴䁾薎覕㒝滅烕櫗懱哶衊幭")
    ("min" "民敏闽闵皿泯|岷悯渑珉抿黾缗玟愍苠鳘旻|忞盷湣碈|簢䶔㞴湏敯惽䞀䝧砇㨉錉怋崏琘笢鈱鰵䡅瑉䡑䁕慜潣捪鍲㥸䡻䲄閔閖䂥閩憫㢯㞶䪸冺勄姄䃉䋋緍忟緡䟨㟩㟭賯僶盿蠠㬆㮌澠黽厸渂鴖㳷刡敃罠暋閺痻旼呡")
    ("ming" "明名命鸣铭冥茗|溟酩瞑螟暝|洺蓂|䳟眀䄙朙眳猽慏㝠佲詺䊅䒌熐銘䆨䆩覭嫇凕䫤㫥㟰姳鳴䤉䏃鄍榠")
    ("miu" "缪谬|||繆唒謬嘐")
    ("mo" "万无没么模末冒莫摩墨默磨摸漠脉膜魔沫陌抹寞蘑摹蓦馍茉嘿谟|幺秣貉嫫镆殁耱嬷麽瘼貊貘靺|鄚藦|絔狢貈㠢㵹䳟帓尛嘜㾺擵脈䘑㜥眿艒冐瞐沒㶬䘃爅䜆鬕眜帞砞無謨昩謩萬䴲礳䬴縸圽驀饃㱄䱅橅衇絈䩋獏湐䉑慔饝魩㹮㱳塻䁼䁿歿貃庅銆鞆莈皌粖瞙覛嚤嚩䒬䮬暯劰嚰袹抺枺鏌髍蟔䏞懡䯢蛨㷬䳮㷵䃺嗼嫼戂䤉䌕攠䭩䊳瀎譕麼霢眽歾纆劘嚜妺麿黙糢灑")
    ("mou" "某谋牟缪眸|哞蝥鍪蛑侔呣||㖼䦈㼋䜼堥䍙謀踇愗洠鴾恈㭌䥐䍒䱕桙劺䗋䏬蟱繆鞪䋷敄瞴冇鉾麰")
    ("mu" "目模木亩幕母牧穆姆墓慕牟牡募睦沐暮拇姥鹜|钼苜仫毪坶||㟂朷䮸㫛茻㙁萺艒縸橅獏慔䁼踇繆氁㜈娒樢砪䀲畆䥈畒畝畞鉧鉬畮䱯䑵㒇㾇炑蚞鞪莯墲䊾㧅狇㣎䧔峔胟毣雮霂楘幙牳鶩拏")
    ("na" "南那纳拿哪娜钠呐捺|衲镎肭|乸|秅㧱㭯䞕貀㙁鈉納䀑笝訤㨥訥䈫㴸䅞魶䱹靹豽䎎䪏抐䖓妠袦䖧蒳誽鎿䛔㗙䟜䇣䏧雫䇱䫱嗱㮏詉蒘蹃郍䋈㪎㵊笚䋾挐吶拏軜淰")
    ("nai" "乃奶耐奈|鼐萘氖柰佴艿|迺耏|㜨㚷腉㜷䅞䘅倷渿摨䍲㮈熋㾍㮏疓㲡釢䯮廼䱞掜搱氝褦嬭錼孻螚妳")
    ("nan" "南难男楠喃|囡罱赧腩囝蝻|萳婻|嫨㬮妠䈒娚䔜䔳㽖畘䁪湳奻䕼枏枬莮䶲侽䛁揇㓓難㫱䊖䣸䩅戁暔柟諵")
    ("nang" "囊|馕囔曩攮齉||涳擃欜儾饢乪㒄䂇㶞嚢憹鬞䁸譨㚂搑灢蠰")
    ("nao" "脑闹恼挠瑙|淖孬垴铙桡呶硇猱蛲|峱臑|㧘碙巙䙹憹䜀䴃鐃夒怓堖匘䜧鬧䄩橈詉䑋㑎腦獶繷獿㺀㺁撓閙嶩㞪碯巎嫐髐䛝䃩蟯惱㛴䫸㺒蝚腝䐉婥脳譊悩摎")
    ("ne" "呢哪呐讷|疒||眲䭚訥䅞抐䭆䎪㕯吶")
    ("nei" "内那哪馁|||㐻䜆䇣㨅㼏餒㘨脮腇㕯䡾䲎㖏鮾䳖鯘娞錗氝內婑浽")
    ("nen" "嫩|恁|媆臑|㒄㜛㶧黁㯎腝嫰")
    ("neng" "能|||䪏䘅㴰㲌薴濘䏻螚儜竜")
    ("ng" "|嗯唔唵||㕶㐻")
    ("ni" "你尼呢泥拟逆倪妮腻匿霓溺|旎昵坭铌鲵伲怩睨祢疒猊慝|鹝鹢薿麑|懝兒䮘㧱聻㮞檷䀑抐誽䍲㮏㲡䘌縌㠜儞䘦鈮䰯伱愵嬺氼䘽䵑䵒屔婗䝚䁥䕥孨㵫屰䭲孴㹸譺㥾籾㦐㪒馜隬蚭抳䦵㲻㞾痆㣇䧇狋䛏胒鯓狔秜跜嫟迡鯢淣苨擬觬埿鑈棿腝蛪㘈眤㩘晲掜禰妳堄儗輗蜺鉨齯鶂貎膩暱惄柅鷊臡郳衵")
    ("nian" "年念粘辗碾廿|捻撵拈蔫鲶埝鲇辇黏||䟢㮟䧔痆攆簐㘝輦䄭䬯鼰䄹艌䩞蹨㞋躎鮎䚓撚㲽跈秊秥姩鯰㜤䴴輾蹍榐唸卄齞涊淰溓")
    ("niang" "娘酿|||孃䖆釀嬢醸")
    ("niao" "鸟尿溺袅|脲氽茑嬲鸮||㼭茮樢䐁㠡蔦褭㜵䙚㭤䦊䮍㞙㒟裊鳥㳮䃵嬝嫋枿")
    ("nie" "摄聂捏涅镍|孽坭蘖啮蹑嗫臬镊颞乜陧|菍嵲糵|㟧㖕䞕峊㜦䜆籋䇣㘨䡾㖏䳖痆㘝踂帇㸎䄒䜓踗䌜錜鈢蠥㴪㜸圼㘿鑈噛㙞鉩㡪顳㩶聶鑷孼钀㮆隉疌㚔㖖嚙躡鎳䂼䯀囁䯅揑巕惗擜篞櫱䯵棿䭃䌰諗褹掜槷囐鋷讘銸嶭蘗摰鉨摂敜齧湼枿闑囓糱臲苶喦")
    ("nin" "您|恁||㤛䚾䛘囜拰䋻")
    ("ning" "宁凝拧泞柠|咛坭狞佞聍甯|苧|䆨薴濘鸋鬡嬣䔭鑏㝕䭢橣獰聹嚀侫㲰檸矃寍寕寗寜㿦寧擰㣷䗿㩶鬤儜")
    ("niu" "牛纽扭钮|拗妞蚴忸狃杻||䮗抝牜莥䤔㽱䂇怓紐䀔鈕靵汼炄䒜㺲䏔䋴衂沑㖻")
    ("nong" "农浓弄脓侬|哝秾||㺜莀䂇憹繷欁儂鬞癑䵜噥䁸蕽檂䢉禯辳㶶膿濃挊挵齈襛穠農醲")
    ("nou" "|耨|嬬|檽䫖㕢䔈㜌㳶鐞䘫䰭䨲啂譨譳䅶㝹羺槈嗕搙獳鎒")
    ("nu" "努怒汝奴弩|驽帑呶孥胬|砮笯傉媆|䢪詉黁搙伮蒘䢞㚢駑㣽")
    ("nuan" "暖||臑|奻㬉渜愞䙇煗䎡湪餪煖")
    ("nuo" "诺挪娜糯|懦傩喏搦锘砹||䎟䎠㖠袳毭搙愞砈㰙㐡稬㔮儺搻挼逽蹃㡅㑚橠穤鍩榒梛䚥㛂郍糑糥懧䇔捼掿諾媠袲")
    ("nv" "女|钕忸衄恧||聏㮟䏔䘐朒衂㵖䶊䖡䚼釹沑籹")
    ("nve" "虐疟|谑||婩䨋謔瘧硸䖈䖋")
    ("o" "哦|噢喔嚄||")
    ("ou" "区欧偶殴呕藕讴鸥瓯|沤耦怄吽||䌂甌䌔嘔吘漚謳㼴㸸蕅歐湡腢慪塸䚆膒㒖毆㛏鷗櫙䯚蓲鏂㭝藲㰶紆醧鴎區熰齵")
    ("pa" "怕帕爬扒趴琶啪葩|耙杷钯筢|蚆舥潖|鈀㞎䎬䎱苩汃㕷帊皅妑䶕袙䯲歠掱跁")
    ("pai" "牌排派拍迫徘湃|俳哌蒎|椑簰|䮘䱝䃻矲箄㵒猅輫簲汖㭛㵺鎃䖰俖篺犤棑廹")
    ("pan" "判盘番潘攀盼拚畔胖叛扳蹒磐|爿蟠泮袢襻丬柈|槃磻|褩㩯螌賁䙃䩔畨緐䋣㵗蹣眫䙪䰉瀊䰔砙搫䈲㐴牉坢奤鑻炍沜鎜蒰冸溿䃑跘盤㳪鋬䃲䏒乑洀籓頄眅縏頖踫詊幋鞶媻")
    ("pang" "旁庞乓膀胖磅螃|彷滂逄蒡耪雱|尨厖鳑|鎊髈䠙鰟㑂汸趽㝑䂇㜊䨦騯㤶霶舽㥬䅭䒍龎厐炐龐膖沗肨覫䮾㫄嫎篣胮㕩夆庬徬嗙")
    ("pao" "跑炮泡抛刨袍|咆疱庖狍匏龅脬摽|脟|袌㲏鮑嚗蚫䠙謈颮㯱㘐礟爮礮奅䩝靤麅垉䶌麭皰㚿拋䛌瓟㯡髱軳犥麃窌砲鞄炰")
    ("pei" "配培赔佩陪沛裴胚|碚霈旆帔呸醅辔锫|荖垺衃|笩䂜㾦㯁䟺㸬嶏犻䫊柭䕗䩛㚰㨐䏽䡊錇翇䯱攈㣆俖伂㤄娝昢琣阫轡陫浿䊃妚肧岯馷䪹䲹斾㳈䣙㟝苝䫠賠㧩毰㫲姵裵駍婄怌抷茷珮柸")
    ("pen" "喷盆|湓||㾦衯㖹葐噴喯翉瓫翸濆歕呠")
    ("peng" "鹏朋彭膨蓬碰棚捧烹篷澎抨硼|怦砰嘭蟛堋椪芃|泙荓淜弸搒|㔙挷㮄䧛逬㑟㱶熢槰䋽䵄漨摓莑剻㮟騯篣㘐䣙䰃錋鬔椖倗䄘稝㼞輣樥䴶㥊硑塜鑝䡫鵬恲塳纄梈憉皏閛亯傰埄駍淎㛔蟚磞軯韸竼韼髼輧胓絣踫鬅匉漰掽痭")
    ("pi" "批否皮辟啤匹披疲僻毗坯脾譬劈媲屁琵邳裨痞癖|陂丕枇噼霹吡纰砒铍淠郫埤濞秕睥苤芘蚍圮鼙罴蜱疋貔仳庀擗甓陴玭苉|伾沘狉椑鲏澼䴙嚭|㼰罷䇑䴽伓䡶藣㱟疈嶏蚾鵧朇稫㡙㵨魮悂枈䪐螕䖩㮰闢羆鈈䤵㽬䗄䯱鈲鮍㔃䦼篺㳪妚岯䪹䲹䫠鸊怌錍䤏渒笓礔礕紕䠘耚鈚㔥崥䰦䤨蠯阰脴怶焷鈹㨽䑀䑄腗䡟鉟睤豼噽豾魾嚊炋膍銔犤隦䚰憵肶抷螷䚹蚽壀諀㯅旇磇揊䫌翍嫓䏘㿙秛毞髬釽鷿奊銢猈鞞鎞壊壞鴄潎狓駓毘秠髲")
    ("pian" "便片篇偏骗匾翩扁堑|骈胼蹁谝犏缏|萹㛹楩|跰鶣猵楄獱緶騈㼐騗騙㸤頨腁䮁覑媥貵骿賆䏒諚諞駢囨㓲璸")
    ("piao" "票朴漂飘嫖瓢|剽缥殍瞟骠嘌髟莩螵摽|蔈薸|㲏謤驃徱㯱淲鷅㬓䴩縹㼼魒慓㩠顠䕯彯㵱㹾㺓犥醥竂飃䏇旚闝勡翲篻驫麃僄潎皫飄")
    ("pie" "瞥撇|苤氕丿||嫳覕鐅䥕撆暼潎")
    ("pin" "品贫聘频拼拚泵|颦姘嫔榀牝玭|涄|㲏瀕驞穦馪矉琕獱㻞砏汖㰋蘋礗娦朩嬪䀻頻顰貧薲蠙嚬")
    ("ping" "平评凭瓶冯屏萍苹乒坪|枰娉俜鲆堋|泙玶荓帡洴涄蚲淜|㲏䈂䓑檘馮㵗倗硑蘋䀻甁簈輧焩頩砯帲甹䍈幈評屛聠呯艵慿鮃䶄憑覮箳㺸炾㻂胓蛢竮蓱郱凴絣鉼缾軿")
    ("po" "破繁坡迫颇朴泊婆泼魄粕鄱珀|陂叵笸泺皤钋钷|桲酦皛馞|䮘㧊㩯謈猼敀䍨岥駊㤕䦌醗醱緐䋣㜑䘠䎅䪙屰釙搫昢尀㨇頗嘙蔢洦䨰㰴砶䄸潑鉕癹炇䎊溌䪖䞟蒪岶櫇㛘䯙烞䣪䣮鏺翍尃巿廹")
    ("pou" "剖|掊瓿锫裒抔|垺棓|堷㼜䎧䳝錇䯽娝㟝㰴吥婄捊㕻犃咅抙廍哣㧵㩠箁")
    ("pu" "普暴铺浦朴堡葡谱埔扑仆蒲曝瀑溥莆圃璞濮菩脯|蹼匍噗氆攵镨攴镤|蒱酺墣潽穙|㙸䪬獛荹豧秿釙䴆圑䔕舖鐠圤㬥䈬䈻轐䑑㹒譜陠襥纀箁檏㒒䲕暜瞨㺪抪㲫撲擈僕䧤柨諩鋪䗱㯷鏷駇捗蜅鯆舗樸襆誧菐痡烳")
    ("qi" "企其起期气七器汽奇齐启旗棋妻弃歧欺骑契迄亟漆戚岂稽岐琦栖缉琪泣乞砌祁崎绮祺蹊祈凄淇杞脐麒圻憩芪|俟畦荠箕耆葺沏萋骐鳍綦讫蕲屺忾颀亓碛柒汔趿綮萁嘁蛴赍槭欹芑桤丌蜞謦祇锜玘扺|呇洓郪埼䓫跂徛婍棨愭墘碶踦磜鲯谿|䯥䔾㱦畁喰刾㰗䠞䐤㫅軙㔑袳㞿忯䟷竐㩾㝄䇍矵舙䛛㩻䣀豈㘍㠍㠎騎焏萕鐖稘朞㰟㠱鬾㹄㙨䩯斊䶓㖢薺䗁竒緝㦢㓗㼮䥢罊蘄䔇䰇䀈甈猉霋䌌㜎䄎頎騏蠐踑訖唘礘䀙倛鼜蜝䐡夡䄢䬣㼤紪䄫欫唭鰭簱䰴帺鬿䙄湆湇䁈䱈䁉齊䡋魌籏噐㥓䩓䡔啔摖剘呚嵜䉝啟䅤㩩䭫䭬呮䅲䑴晵䭶鵸䉻䭼慼桼㩽慽捿䢀鶀炁玂璂纃㒅憇鶈䚉䚍㮑掑蚑䶒㞓岓邔碕䒗㞚䞚蚚亝䶞芞麡䎢暣禥綥㾨綨䲬檱疷粸綺䒻榿櫀盀緀䟄棄䏅䓅迉䏌臍㟓㫓䫔忔翗䧘闙㟚䟚僛濝㓞䏠㟢䳢竢㯦磧磩㣬諬釮䋯棲䛴䧵䏿裿諿缼趞螇漬傶䅩蚔扢愒肵斉蛣犵躩鸂鬐栔気錡氣錤攲娸鄿啓顣敧艩衹碁悊咠鮨悽旂藄諆棊磎淒鯕鏚軝懠盵蟿渓扱疧滊")
    ("qia" "卡恰洽掐|髂袷咭葜拤||㵄㮫抲䈓㦴鞐㰤䶗硈䁍䂒鮚㓤㓞㤉䠍䨐䜑愘圶㡊酠殎䶝䯊跒㓣䛩冾磍㧎帢鉥")
    ("qian" "前千钱签潜迁欠纤牵浅遣谦乾铅歉黔谴嵌倩钳茜虔堑|钎骞阡掮羟钤扦芊犍荨仟芡悭缱佥愆锓褰凵肷撖岍搴箝慊椠鹐圲|杄汧靬葴蒨墘磏孅|㝿䢪䑶橬䅾㹂濽㥞䤘馯㦿乹齦摼䐶鋟䋮䔔仱唊騝偂䅐䵖䵛䭠麉玪鶼䯡㓺淺鳽艌厱䭑䙺䆂熑䃛䇜㪁㡊刋攐鈐攑攓儙謙騚㨜鬜鬝㜞錢䈤欦漧㸫騫鰬圱鐱蔳䈴縴㐸蜸谸簽䨿䥅顅䍉灊籖鉗汘兛潛鉛婜譣䭤籤㡨顩歬䕭㩮䁮鵮孯嵰扲慳譴奷遷塹牽繾撁膁䪈䖍岒悓皘蚙箞㪠羥榩㦮䦲䊴檶㧄韆棈僉壍姏槏櫏篏諐黚㯠䫡軡臤槧瓩拪濳忴鳹㟻鑓㥶竏煔錎羬藖雃㢛撍朁灒歁鰜鑯幵麕輤鬵蕁葥粁蚈媊傔掔綪銭燂拑嗛篟揵釺廞俔脥燖鈆鉆")
    ("qiang" "强抢墙枪腔锵呛羌|蔷爿襁羟跄箐樯戕嫱戗炝镪锖蜣丬|玱矼鸧蔃|戧搶剏將強繦摪勥羥錆猐漒謒嬙蘠唴䵁牆蹌鑓㩖䅚艢嶈墏薔羗檣羫溬傸墻羻嗆槍鏘磢㛨篬鏹㱿啌刱彊椌牄繈蹡瑲熗鎗斨廧")
    ("qiao" "桥乔侨巧悄敲俏壳雀瞧翘窍峭锹撬荞|跷樵憔鞘橇峤诮谯愀敫鞒硗劁缲睄|硚燋|㱶䏆䦒䵲銚䂪䯨礉丂殼潐譑㩰嶠䚩喬勪鞽㺒䫞㨽鐈䀉㤍頝鄡㴥鄥帩鐰嘺䱁癄橋䩌蕎㡑繑陗譙顦幧嵪鍫鍬㝯繰鵲蹺㚁躈㢗䎗誚㪣嶣墧庨趬骹䆻㚽墽撽䲾燆䇌僑鏒韒槗髚髜䃝釥藮嫶翹僺郻磽毃犞幓踃踍顤菬摮磝喿殻蹻蟜礄簥塙敿墝趫竅偢荍橾箾")
    ("qie" "切且窃茄锲怯伽惬|妾趄挈郄箧慊唼癿||魥㙻䞣鰈㰤䶗鐑洯䌌㼤䫔䠍䬊愜笡踥匧㰰㤲㰼䤿穕籡㹤鍥㥦聺㾀悏㚗㾜䦧淁竊篋㛍藒㛙䟙鯜蛪㗫㓶㫸䈉帹稧㛗輵椄緁跙朅")
    ("qin" "亲勤侵秦钦琴禽芹沁寝擒覃|噙矜嗪揿溱芩衾廑锓吣檎螓骎|嵚慬|橬䅾梫埁䃡鈂䢈藽㱽頜瘽䥆䶖嶜厪埐鋟菳㕋䵖䇒㓎蠄搇㤈瀙鈙䈜吢㘦儬䠴䔷耹欽坅顉㩒捦鵭㝲靲赺赾㪁媇抋䖌㮗㢙㾛誛肣䦦親㞬撳斳螼鮼澿嫀懄曋寑菦寴䥅顩扲蚙軡臤忴鳹㛙䰼㕂笉䫬昑唚琹綅庈嶔雂懃珡寢菣駸鬵鈊")
    ("qing" "情青清请亲轻庆倾顷卿晴氢擎氰|罄磬蜻箐鲭綮苘黥圊檠謦鸮|汫勍庼碃|埥渹䡖葝䵞㵾鶄㢣擏殌㔀親漀頃樈輕䔛夝䌠甠氫罊鑋靑靘剠啨慶䝼掅䞍暒䲔傾檾狅寈請鯖磘䋜䯧㷫郬㯳淸㩩䋯棾䂩凊莔殑殸廎軽")
    ("qiong" "穷琼穹|邛芎茕筇跫蛩銎|䓖嬛|㵌憌舼䧆䃔㤨蛬㑋輁㼇匔儝焪焭䠻䅃桏睘煢橩卭赹䊄宆蒆㒌熍㮪窮䆳㷀竆瓊藑䛪㧭藭棾䁚瓗㝁笻瞏璚惸琁")
    ("qiu" "求球秋丘邱仇酋裘龟囚|遒鳅虬蚯泅楸钆氽湫犰逑巯艽俅蝤赇鼽糗鞧|訄萩鹙璆|㛏蓲䠓醔釓牫亀龜㲹䔔䊆㺩觓鳩趜唒㧄㐀訅紌鰌鰍逎㼒䠗脙䤛搝渞丠㐤蠤䜪崷㤹朹鱃扏穐汓浗恘㭝㥢㕤煪湭虯蝵梂鮂䆋玌肍莍殏㞗媝龝趥鞦㺫䞭皳䊵䎿䣇㟈僋秌篍㷕賕蟗釚緧㧨䟬苬叴䟵蛷頄鰽偢坵絿鶖銶厹鯄盚觩毬巰橚櫹")
    ("qu" "区去取曲趋渠趣驱屈躯衢娶祛瞿戌岖|龋觑朐蛐癯蛆苣阒诎劬蕖蘧氍黢蠼凵璩鞫麴鸲磲彡|坥岨胠砠竘袪䓛焌麹鼩灈|㲒䪨䠓趨䓚㪯趍耝欪䢗蚼㽛㜘絇䅓䕮㩴蝺趜斪䢹䆽䗇䏣䌌㚁㤲㾀鼁㠊欋䠐鸜鴝刞䀠爠匤㰦戵紶匷蠷鰸㜹伹浀筁驅葋鱋詓䝣䁦齲䵶鑺镼魼䂂㖆嶇憈鶌璖㲘䶚躣䞤䒧粬麯覰誳岴閴螶覷厺覻抾軀㧁忂菃闃㣄駈䟊翑蟝臞㫢䋧㯫竬淭翵髷區詘怚佢跙阹佉豦籧敺呿麮懅駆胊軥脥")
    ("quan" "全权券泉圈拳劝犬铨痊诠荃|醛蜷颧绻犭筌鬈悛辁畎|佺琄婘棬瑔鳈|鐉酄鸛䌯㩲腃韏埢啳狋䅚鰁輇權䄐縓騡洤権䀬䠰搼硂虇孉䑏呟絟齤絭恮詮汱葲顴湶牶犈䊎銓綣㒰烇巏峑䟒觠㟨駩㟫勸姾槫謜矔圏踡弮蠸甽灥牷惓勧跧讬")
    ("que" "确却缺雀鹊阙|瘸榷炔阕悫攉汋|囷埆隺碏|䱜䍳寉㹱䲵䀇䐨㲉殼蚗䢧䟩鵲墧㹤崅㰌礐愨礭㴶琷缼㕁㩁㱋灍硞慤㱿蒛趞㾡䦬皵確毃闋䇎闕燩䧿棊塙搉卻碻")
    ("qun" "群裙|逡麇|囷|歏䆽麏䊎夋帬䭽㪊宭㿏裠峮輑麕箘踆羣")
    ("ran" "然染燃冉|苒髯蚺|䎃|䇋嫨㹱䔳䕼䶲㒄䰯蹨䳿㸐䤡㜣嘫繎䑙呥橪衻䖄冄㾆袇蚒袡䒣媣蚦㚩㲯肰珃䫇姌㯗髥㿵䣸㦓蛅")
    ("rang" "让壤攘嚷|瓤穰禳|儴蘘瀼|䑋爙鬤穣䉴獽㚂讓躟懹欀蠰譲壌勷纕忀")
    ("rao" "绕扰饶|娆桡荛蛲||犪橈蟯繚嬈饒襓蕘㹛繞㑱隢䫞擾穘遶")
    ("re" "热惹|喏|爇|㻰蹃渃熱㳧捼")
    ("ren" "人任认仁赁忍韧刃纫饪|妊荏稔壬仞轫亻衽恁葚纴鸮讱|菍|釰㤛䚾䛘㲽䄒䀔賃餁刄紉鈓紝栠栣䴦朲㠴鴹䀼㸾䌾䭃杒魜扨靭靱屻銋認梕芢㶵袵秂忈韌忎軔䋕䏕姙軠飪仭䇮䏰秹㣼訒鵀絍腍牣荵肕棯")
    ("reng" "仍扔|||䄧㭁䚮㺱辸礽陾芿")
    ("ri" "日||驲|釰氜鈤䒤囸衵馹")
    ("rong" "容荣融绒溶蓉熔戎榕茸冗嵘|肜狨蝾|瑢镕|軵坈䋴䄧爃㘇䤊搑蠑縙茙䠜瀜褣鴧鰫䘬䈶㼸穁䡆㝐㭜嵤䡥絨䩸宂䢇傇㺎螎㲓鎔㲝㲨榮榵媶嶸鎹䇀㣑鷛駥曧䇯髶烿䄾縟巆搈嬫傛嫆栄氄羢毧")
    ("rou" "肉柔揉|糅鞣蹂||莥䥆㮟䏔䋴䢇䰆鰇䐓渘騥䄾瑈鍒鍕蝚煣㽥腬媃粈宍沑鶔㖻瓇䧷輮葇楺禸韖厹鑐")
    ("ru" "入如乳儒辱汝茹褥孺濡|蠕嚅缛溽铷洳薷襦颥蓐|傉嬬臑|檽㱶吺䞕㨌㳶㼋㾒䤉䫱䘫䰭嗕䋴蒘䐓䄾㨎鴑縟帤渪䰰偄桇㹘顬鱬肗邚侞銣媷㦺䋈曘燸挐獳鄏鴽筎蕠醹袽擩蝡鑐繻")
    ("ruan" "软阮|朊|堧媆瓀|䆓㮕檽㧫䙃䪀㓴㜛䙇䎡㨎偄燸礝輭㼱瑌腝㽭䞂碝䪭緛軟䓴耎蝡撋壖")
    ("rui" "瑞锐睿芮蕊|蕤蚋枘棁|汭|䞩㧫䂱䦌䄲䳠䇤䍴㻔䜭㕙鈉抐㨅㹘蘂蘃甤蜹䌼蕋䅑桵㮃㲊㪫銳壡惢㛱䓲㓹綏苼䬐㢻笍鏸撋婑繠橤緌叡鋭")
    ("run" "润闰|||㝄䏕䏰㠈橍閏䦞閠瞤膶撋潤犉")
    ("ruo" "若弱偌|箬|鄀婼蒻僇爇|㬉挼䚥渃鰙䐞焫鰯楉叒捼撋鶸篛")
    ("sa" "萨洒撒|飒卅仨挲趿脎靸|潵|䑥钑鞈䓲䬃䘮訯㽂䙣蕯㒎䊛隡㪪颯馺㳐櫒挱㚫泧䛽鈒攃灑薩鏾摋纚躠")
    ("sai" "赛塞|腮噻鳃||㘔鰓䈢愢簺顋恖毢嗮㗷賽毸䰄揌僿儛")
    ("san" "三散伞|氵叁糁馓霰毵彡|潵|傪謲叄悷鏒嘇䀐攕㤾䉈饊䊉閐㪔傘犙㪚糂䫅仐俕糝糣糤䫩㧲毿帴弎鬖繖佡鏾蔘")
    ("sang" "桑丧嗓|搡颡磉||䘮褬桒顙䡦鎟䫙喪纕")
    ("sao" "扫骚嫂缫|搔瘙臊埽缲鳋|溞|鰺㺑鯵鄵鐰繰鰠騷䐹䕅繅鱢掃䖣掻矂㿋㛮㲧髞㺐喿氉騒慅颾縿橾")
    ("se" "色塞瑟涩|啬穑铯槭|洓㴔溹璱|鉍䊂粣拺琗㽇㾊㺩㻎瀒䨛䔼㱇穡䉢繬歮歰㥶澀澁趇㒊犞㮦銫嗇飋擌懎濏翜㻭雭栜瘷鏼漬廧渋譅轖濇鎍闟")
    ("sen" "森|||傪㜗摻罧襂幓穼曑槮篸蔘椮")
    ("seng" "僧|||鬙")
    ("sha" "沙厦杀纱砂啥莎刹杉傻煞赊|鲨霎嗄痧裟挲铩唼歃||硰剎㵤䯫唦啑菨㠺䀉䬊㰼歰翜挱猀䈉儍紗㰱蔱帴帹倽䝊䵘桬乷粆䶎䮜榝喢鎩㚫閯殺廈鯊鯋㛼髿㬠閷賒濈萐樧摋魦箑翣")
    ("shai" "色筛晒|骰酾||攦釃簛䵘㬠㩄籭繺閷篩曬諰㴓簁")
    ("shan" "山单善陕闪衫擅汕扇杉掺珊禅删膳缮赡鄯栅煽姗跚鳝|嬗潸讪舢苫疝掸膻钐剡蟮芟埏髟彡骟|飏烻掞晱墡嶦鳣|㱯㨻㱚摻㺑䗞笧脠䡪㹽禪單㯆䛸灗潬撣㚲䩔閄蔪㬭㴸㾆嘇䀐襂幓贍訕笘椙㨛䄠鐥刪椫䴮騸挻䠾䥇䱇䱉饍鱓鱔繕陝杦譱赸剼敾陿䦂閃䆄䦅檆熌㪎銏㚒㶒傓邖澘覢㪨䚲羴鯅㣌姍僐軕㣣釤狦磰煔襳顃柵䁴䪌僤痁儃謆搧縿穇睒陜摲墠羶蟺猭樿鱣")
    ("shang" "上商尚伤赏汤裳|墒晌垧觞殇熵绱|埫|爙丄蔏䬕尙漡鬺扄鑜慯湯銄殤螪傷緔賞姠仩滳觴恦禓塲愓蠰謪踼鞝")
    ("shao" "少绍召烧稍邵哨韶捎勺梢|鞘芍苕劭艄筲杓蛸潲睄|玿柖|韒娋輎㸛䔠䬰圴紹牊佋蕱䙼㲈袑䒚㪢綤颵鮹竰䏴㷹䈾萷綃弰焼卲莦燒旓髾")
    ("she" "设社摄涉射折舍蛇舌奢慑辙赦赊佘麝|歙畲厍猞铊揲滠阇|奓畬|䤁㵃䬦挕䠟䕣㓭䰥涻䀹輋䜆䜓摵䏴䀅㴇欇騇㰒䄕攝設䤮䌰䠶䬷弽灄譇鉈葉䁋捑㭙䵥鍦捨䁯䞌厙䂠㒤㢵賒賖蛥懾㤴磼闍摂蠂渉舎蔎虵檨韘肬慴")
    ("shei" "谁|||誰")
    ("shen" "深参身神什审申甚沈伸慎渗肾绅莘呻|婶娠砷蜃抻哂椹葚吲糁渖诜谂矧胂瘆屾|侁珅甡棽跱鲹燊瞫|㱯鰺參叄叅蓡鯵堔䫖䯂㚞罙䚓罧嫀曋䧵㛙嘇糂糝幓穼曑槮㰂㶒鯅䄕谉瀋瘎眒眘昚弞䰠氠訠阠頣㔤㜤甧㰮瘮紳訷嬸愼㵊腎敒兓㵕扟鵢魫鉮㥲詵籶涁薓㾕邥䆦宷侺妽覾裑諗柛鋠矤審滲峷䰼㜪頥㕥䅸蜄祳寀搷淰棯蔘椮脤葠籸讅駪黮兟姺抌")
    ("sheng" "生声升胜盛乘圣剩牲绳笙|冼甥嵊晟渑眚省|昇陞|㹌䋲椉娍㼩晠㞼䇸澠琞阩㼳焺蕂䱆鉎聖偗譝䁞湦鱦繩聲陹鵿殅䚇䞉溗斘㾪墭䎴憴箵貹榺䪿珄狌竍竏竓竔竕竡曻苼渻㮐乗熯褱殸縄鼪呏剰㗂勝泩賸")
    ("shi" "是市时实事十使世施式势视识师史示石食始士失适试什室似诗饰殖释驶氏逝湿蚀狮誓拾尸匙仕柿矢峙侍噬嗜拭嘘屎恃|唑轼虱耆豉舐莳铈谥炻豕铊鲥饣螫酾筮埘弑礻蓍鲺贳湜奭鳀|邿鸤䴓䏡浉祏栻媞鼫襫|宲㫅䈕䜵䜻恀䑛㢁徥㶴䶵㳏飭忯姼䜴忕䄷䲽眡啇楴鈟痑䤭釃佀螄䂖䦙飠籭篩鉈鍦䂠竍眂䰄褆眎弒舓嬕丗餙謚餝鰣簭鼭昰鈰䤱笶㸷褷㔺崼㱁湁鉂䩃鉃䭄獅鉇遈卋衋屍詍鉐塒㵓噓蝕睗兘識兙齛㕜㹝呞瑡湤試乨呩詩㹬乭卶㹷襹䁺鉽遾亊蒒䊓蒔視冟䦠䶡溡䒨䖨宩碩榯貰鶳銴㮶㖷箷䦹㒾枾䛈釋㫑烒叓揓峕濕䟗駛勢實瓧翨鯴釶柹軾飾鳾惿醳沶魳檡䌤馶觢秲徧戠鍉咶適鰤師戺絁時蝨葹実澨溮溼釈寔諟諡跩鳲旹識狧剸狶埶遰")
    ("shou" "收手受首售授守熟寿瘦兽瘠狩|绶艏扌||濤嘼敊夀㥅㝊䭭獸膄㖟垨綬涭鏉䛵壽掱荍獣収痩")
    ("shu" "术书数属树输束述署熟殊蔬舒疏鼠淑叔暑枢墅俞曙抒竖蜀薯梳沐戍恕孰沭赎庶戌漱塾|嗽倏澍纾姝菽黍腧秫毹殳疋摅丨|祋陎悆鄃隃婌稌翛摴疐潏|㵂䇬㫹䠼䙱暏䢞氀捒㟬䩳䭭鸀焂攄尌㜐倐紓圕贖尗樜樞䘤䴰錰䠱輸䜹䉀䝂絉豎術䑕荗鵨襩䝪兪虪屬㽰䩱潻㾁蒁䎉鶐掓㒔㶖䞖䆝䢤薥侸璹庻凁㷂㳆俆濖軗䃞鏣竪糬㯮埱藲㛸書㣽跾㻿鮛咰㡏㓱捈瑹怷蒣忬鷸蠾孎䟉鐲謶籔踈鼡儵樹潄癙襡鉥數杸橾綀钃疎裋韣藷刓朮")
    ("shua" "刷耍|唰||鮛誜")
    ("shuai" "率衰帅摔甩|蟀|缞|釃綏帥繂孈䢦咰卛縗")
    ("shuan" "栓拴涮|闩|腨|閂䧠槫")
    ("shuang" "双爽霜|孀泷|骦礵鹴|漺䔪䨥瀧傱欆樉縔鸘騻㼽灀孇慡䡯㦼雙䫪䗮滝驦艭塽鷞")
    ("shui" "说水税谁睡绥|氵|帨|㔑娷㥨㝽䳠䲧䬽閖誰稅挩氺脽捝䭨祱㽷瞓涗涚說裞蛻説")
    ("shun" "顺瞬舜|吮巛楯||䐏䞐䴄㯗順鬊䀢䀵橓䑞㥧賰蕣瞚揗眴")
    ("shuo" "说数硕烁朔|铄妁槊嗍蒴搠汋||䁻洬銏㸛䔠碩㮶帥說䌃爍䀥欶䈾鑠獡鎙矟揱哾萷數説箾")
    ("si" "司四思斯食私死似丝饲寺肆撕泗伺嗣祀厮驷嘶|锶俟饴巳蛳咝耜笥饣纟糸鸶缌澌姒汜厶兕菥|佁虒涘偲楒飔凘|厠廁枱賜傂䃽㜦竢愢恖鐁瀃㐌洍謕簛蜤騦娰鼶㸻鈻佀㹑㭒噝孠杫蕬牭鉰絲鍶蕼㕽肂螄貄㾅䂖亖禗䦙䎣媤禩㚶榹泀䇃磃㣈燍蟖鋖駟飠䏤泤飤鷥緦俬釲蟴糹飼籭䒨㮐逘鈶螔飴銉価儩罳覗禠颸廝柶餧")
    ("song" "送松宋讼颂耸诵嵩淞|锶怂悚崧凇忪竦菘㧐|娀|㬝樬棇蘴鎹鬆頌蜙訟㨦䜬倯愯餸捒楤䉥慫㕬㩳聳嵷硹枀䢠梥誦檧枩傱㮸庺憽濍柗䛦駷摗鍶吅憁揔漎")
    ("sou" "艘搜漱|擞嗽嗖叟馊薮飕嗾溲锼螋瞍|蒐廋|䈭䐹捒騪瘶䈹餿摉摗䉤㵻傁醙䮟㖩鎪颼廀䏂蓃㛐叜櫢藪㟬擻䩳膄鏉欶棷颾鄋謏獀籔捜潚")
    ("su" "速素苏诉缩塑肃俗宿粟溯酥夙|愫簌稣僳谡涑蔌嗉觫窣|骕甦傃溹鹔蹜|梀䇤棴㥛圱㔄㴋䔎㴑蘓謖㬘㜚䌚栜㨞稡縤洬縮訴蜶䘻㴼䅇㑉㩋塐䥔㕖㑛㝛遡潥遬䩳䑿肅膆玊誎殐䎘趚璛憟㪩溸碿鯂㯈鋉嫊䏋藗㓘珟䃤櫯䛾埣樎摵僁莤䘘㲞遫蘇樕餗愬穌驌鱐橚潚粛榡囌泝鷫")
    ("suan" "算酸蒜|狻||䔉㱹笇㔯匴祘筭痠")
    ("sui" "随岁虽碎尿隧遂髓穗绥隋邃|睢祟濉燧谇眭攵荽|葰嶲璲穟襚旞|㥞嗺䲀陏韢靃䤭䜆䅑䆳鐆夊䜔䠔嬘㴚娞䔹繀遀䍁䭉䉌䅗嵗䥙譢㵦湪煫歲䡵荾檅綏芕鞖亗熣䢫禭誶㒸㞸膸澻瓍滖雖䯝㻟埣賥㻪哸毸㻽倠䯿粋繐鏸撋砕瀡鐩穂睟歳繸浽檖隨巂髄雟縗")
    ("sun" "损孙笋荪|榫隼狲飧||喰䐣摌㰂損愻㔼㡄筍䁚蕵㦏薞鎨箰蓀槂飱㨚鶽搎簨猻潠孫扻")
    ("suo" "所索缩锁莎梭琐|嗦唆唢娑蓑羧挲桫嗍睃|葰溹蹜|縒鎈嫅䌇靃惢㮦縮蜶樎琑簔䈗䐝逤褨挱瘷䵀摍䅴摵䞆莏溑趖鎖䖛暛䂹鎻鮻㪽䞽鏁䣔䗢嗩鏼髿䲃㛗簑瑣璅鎍傞魦麞")
    ("ta" "他它她拓塔踏塌榻沓漯獭|嗒挞蹋趿遢铊鳎溻闼靸跶褟|溚阘鞳|䑽呾墖鎝㯚䳴䪚䠅鞈鉈䌈崉䈋䜚鰨䈳䍇鑉䑜䍝䵬譶蹹㹺獺㭼橽䶀䶁躂羍㒓䎓龖榙鮙㺚鞜傝躢亣嚺撻誻涾澾䂿濌㯓迖㣛㳠䓠㛥闥㳫㗳㣵㿹㧺鎑咜闒錔搨祂遝牠嚃鎉龘禢狧毾闟")
    ("tai" "台太态泰抬胎汰钛苔薹|肽跆邰鲐呔酞骀炱||枱㙜箈忕奤㸀㬃㘆䈚舦鈦嬯夳籉態䑓孡㥭坮㙵㑷粏鮐㒗溙冭檯䢰颱炲㣍㷘駘擡燤䣭菭秮忲旲臺囼䔶鈶詒珆斄儓")
    ("tan" "谈探坦摊弹炭坛滩贪叹谭潭碳毯瘫檀痰袒坍覃|忐昙郯澹钽荨锬镡赕|倓菼啴惔榃璮磹亹|鐔䀡嘽繵㺥㵃䢻䗊㛶舕䨢嘾㵅㽎潬㫜痑悷䑙僋㒎㪔㶒㜤傝嘆舑䜖錟攤騨䐺䕊罎㽑婒灘譚橝湠䉡鉭癱醈䦔貚憛㲜䞡䊤抩貪㲭墰䆱憳墵憻緂壇曇談㷋䏙壜鷤裧賧䃪藫埮擹顃譠餔弾彈禫燂儃贉餤怹罈歎驔襢醓醰黮嗿")
    ("tang" "堂唐糖汤塘躺趟倘棠烫淌膛搪|镗傥螳溏帑羰饧樘醣惝螗耥铴瑭嘡蹚|鄌䣘镋|䠀䕋㦂閶欓逿嵣闣䞶㼒漡湯戃䌅鐋㜍伖鼞漟爣餳餹㼺儻饄䉎偒橖坣摥蝪䅯赯㙶啺㭻㑽钂膅㒉傏劏鎕薚隚禟㲥鎲榶鶶鞺糃磄蓎䟖篖矘燙糛闛䧜鏜㿩曭愓簜踼煻")
    ("tao" "套讨陶涛逃桃萄淘掏滔韬叨|焘洮啕绦饕鼗丶|弢梼绹慆|夲䄻濤䛬幍燾轁䛌䱇㲈涭騊討䬞謟縧蜪錭饀㹗䵚絛詜瑫祹醄鞉㚐䚯鞱䚵槄嫍韜翢駣裪迯飸鋾㣠夵䑬抭窾匋縚搯鞀檮綯咷")
    ("te" "特|忑忒铽慝|螣|貸鴏脦㥂貣蚮蟘鋱㧹犆")
    ("teng" "腾疼藤滕|誊熥|螣縢䲢|痋霯邆䮴僜謄漛鼟鰧䠮儯騰幐籘驣䕨䒅䲍膯駦籐")
    ("ti" "体提题替梯踢惕剔蹄啼屉剃涕|锑倜悌逖嚏荑醍扌绨鹈缇裼俶鳀|钖虒䏲遆媞媂瑅髢禔䗖䴘擿趯|䬫䣟䫮䈕徥徲姼焍奃詆䱱楴䶍䞶諦惖啑䣠嵜謕磃褆㖷䢰鷤鬀鬄褅䨑鐟䌡錫挮㬱䔶崹鴺漽䬾題偍蹏歒㡗䙗鍗蕛屜鵜屟䅠罤蝭䝰䪆䶏䶑㖒嚔厗鶙䚣掦趧䎮碮躰禵骵鮷銻嗁䧅軆鷈鷉迏苐體䯜䣡㣢㗣俤㯩髰䛱鯷緹䣽惿䩟㥴鮧瓋䄺渧珶稊騠逷戻穉籊睼綈悐鶗薙殢揥洟桋")
    ("tian" "天田添填甜恬|腆佃舔钿阗忝殄畋锘掭湉|沺盷淟晪颋瑱黇|㧱娗奵錪鈿顚敟䧃䏦䣶㤁䬯䄹䩞跈鍩䑙沗䄕㜤㑷緂䵺㐁䠄䐌倎舚甛唺䄼䄽䩄㙉㥏畑䡒睓煔靔䥖婖䡘䑚靝酟塡兲㮇覥㖭璳㶺悿㧂鷆胋磌觍闐賟䟧䣯痶搷琠靦鼪睼餂屇窴菾銛鷏")
    ("tiao" "条调挑跳迢眺|苕窕笤佻啁踔粜髫铫祧龆蜩鲦|㕮朓嬥|趠朷䧂銚䂪調㩖脁萔㸠樤䠷鰷晀聎祒䱔齠䩦絩㑿芀䎄䒒條鮡庣鎥宨窱岹䖺䳂㟘鋚旫䟭䯾蓚蓨艞䑬鯈誂頫恌趒鞗覜斢岧糶")
    ("tie" "铁贴帖|锇餮怙萜揲||鋨聑惵䩞䑜䂿䴴鐵䥫䵿貼蛈飻鴩惓怗鐡鉄驖僣銕蓺詀鉆呫")
    ("ting" "听庭停厅廷挺亭艇婷汀铤烃霆|町蜓葶梃莛圢|侹珽烶珵桯颋渟䗴|庍䫖䁎侱䇸䅍㼗娗甼忊鋌濎烴鯅耓鼮嵉䱓聤㹶䵺聼艼聽涏䦐邒鞓誔厛閮榳諪廰廳䋼脡頲朾蝏楟筳聴庁綎")
    ("tong" "同通统童痛洞铜桶桐筒彤侗甬佟潼捅酮|砼瞳恸峒仝恫硐嗵僮垌艟茼湲熥|峂哃烔㛚鲖橦曈穜皦|炵峝蚒䴀氃㠉樋㤏㸗爞㼧鼨眮蘳㠽㼿晍鉖慟絧統浵鉵詷綂憅銅䂈㪌㮔䆚庝犝粡膧䶱䮵䆹䳋燑㣚姛㣠狪勭秱迵蜼赨硧偅鉥餇朣衕獞筩蕫罿鮦痌蓪囲湩")
    ("tou" "投头透偷|钭骰亠||諳㼥敨䕱鋀䡩㤟㰯㕻㳆埱䚵鈄紏蘣頭坄䵉㡏䱏酘鍮偸婾㢏㪗㖣䞬䟝㓱綉諭飳牏媮妵黈斢緰皁")
    ("tu" "土突图途徒涂吐屠兔秃凸荼|钍菟堍酴|梌葖稌腯㻬|啚鋀摕嶀趃腞䩣宊迌䴛䭭圕㒔䤅䠈唋圖圗堗圡怢鈯鼵捈兎鍎鵚筡湥潳䅷㭸捸瑹涋䖘馟鶟蒤庩莵凃䣄鷋㻌䳜廜痜䣝㻠揬㻯鋵鷵釷駼跿䔑墿鵌悇檡斁瘏塗嵞鵵禿図峹")
    ("tuan" "团湍|鹑疃抟彖|猯煓|墥褍剬䳪湪褖團䜝㩛䵯摶貒䊜檲鏄黗槫糰鷻鷒嫥篿漙慱剸団畽鱄塼")
    ("tui" "推退腿褪颓蜕|忒煺|焞魋|㥆䩣墤謉騩饋㰐脮蛻怢䀃蘈弚娧尵頹頺頽蹆橔㱣䅪蹪䍾螁㞂㢈㦌㢑㾯㾼㾽骽俀㿉䫋僓㿗㷟藬侻㟎聉㳷讉㢂穨隤蓷駾啍")
    ("tun" "吞屯囤褪豚|臀饨暾氽挦鲀坉|忳焞|錪憞蜳㬿㧷䀫膯黗螁㼊霕朜噋呑㩔豘㹠㖔㞘臋飩拵庉啍魨畽涒芚軘旽讬")
    ("tuo" "托脱拓拖妥魄驼陀沱鸵驮唾|椭坨佗砣跎庹柁橐乇铊沲酡鼍箨柝棁|侂萚|䲦仛㵃槖鬌䅜袉莌憜馱軃㯐踻㰐鋖綏鉈挩捝騨鼉鴕鰖託㼠砤鼧㸰㸱䴱堶䰿鵎䡐汑杔楕籜驝魠橢扥饦䍫彵䭾㾃咃䲊䪑碢袥岮馲涶侻㟎狏䓕拕駝駞毤迱嫷矺毻㨊袘扡駄蟺説蘀阤脫紽陁詑驒鮀媠讬沰撱跅飥")
    ("wa" "瓦挖娃洼袜蛙哇|佤娲鲑腽|畖窊|屲嗗窐䖯䍪帓㒝鼃㼘䠚漥攨㰪搲聉啘膃徍咓媧窪䎳䚴邷劸韈㧚嗢韤黳䯉唲鮭襪穵溛")
    ("wai" "外歪|崴呙||䴜瀤竵㨤䶐䠿㖞顡喎")
    ("wan" "万完晚湾玩碗顽挽弯蔓丸莞皖宛婉腕蜿惋烷娩|琬畹豌剜纨绾鲩脘菀芄鞔箢塆|妧涴婠椀|㺜㼝㝃㽹䏦輐捖錽鎫鋄䯛䈮䑱萬莬䨲薍孯䵥贃倇紈唍贎頑琓脕萖㸘踠帵㜶䘼鑁葂䩊䅋睌彎卐䥑睕㽜灣潫㝴䝹晼綄岏抏梚䖤貦綩綰䂺䳃蟃䯈盌鋔䗕埦忨仴䛷㿸䛃莧㹉魭蚖鋺鄤澣澫夘絻掔刓輓卍汍鍐捥晩翫夗惌")
    ("wang" "王望往网忘亡旺汪枉妄|惘罔辋魍尢|尪忳|朚臦㔞㑌忹盳䒽琞徍瀇㴏䤑輞䰣尣尩尫蝄罒兦彺暀徃蚟莣網誷亾㲿㓁䛃䋄䋞棢蛧迬菵㳹仼抂朢迋")
    ("wei" "为位委未维卫围违威伟危味微遗唯谓伪慰尾魏韦胃畏帷喂巍萎蔚纬潍尉渭惟薇苇炜圩娓诿玮|崴桅偎逶猥囗葳隗痿猬涠嵬韪煨艉帏闱洧沩隈鲔軎霨㧑|沇峗洈硙硊琟廆煟溦碨薳鳂螱皦鳚亹|㥨濻䙟䍴㺔㧡㙔䩈䅏饖詴䇻癐䲘䗆䤥㸵潙䍷溈䣀㧪㙗㟴䠑㨒䦱䫥䥆䵢䊊䓺㻰䩞䱞㮃㛱䛪膸㾯䫋鰖䲊䓕贀謂鰃愄鰄䜅愇欈㨊䬐䬑嬒踓㠕錗㬙䘙䜜瀢崣䈧儰椲蜲餵蘶䔺霺蜼䬿蔿幃偉䑊轊䵋瑋㭏䝐㕒煒癓婔嵔違浘蝛衛㥜䉠鍡葦葨䥩㱬荱楲腲䭳䵳䡺遺偽䙿潿醀䲁媁厃㞇隇鮇䪋蒍璏讏㖐㞑䞔斖䪘媙犚芛躛喡㦣覣媦犩骩骪鮪徫熭維喴梶䮹覹颹為㢻懀矀闈㷉諉揋菋韋鏏韑韙叞䧦㟪䃬緭緯藯濰燰㣲蓶䗽苿唩瓗猚茟趡濊恑瞶峞捼穌鼪囲撱圍頠蘤餧渨鄬爲椳褽湋鍏衞蝟罻讆薉暐躗鮠骫徻溾磈痏磑烓壝僞韡寪")
    ("wen" "文问闻稳温蕴纹吻蚊雯紊瘟汶|刎璺玟阌芠|炆缊榅辒煴蕰鳁馧|㼂䦟䦤䦷䐇㵮顐呅㡈㨉忟㟩莬呚脕渂䘇紋鴍瘒鴖脗䰚鰛琝鼤鰮昷輼問彣瑥㝧穩塭魰豱桽瞃蚉妏熓㒚殟螡㖧溫肳䎹馼䎽閿蟁㗃闅駇闦珳㳷緼鳼揾褞藴歾閺絻搵轀穏聞呡抆皁")
    ("weng" "翁嗡|瓮蓊蕹滃|鹟|䨴㙂霐㘢䐥䤰㜲嵡奣聬浻瞈鎓暡勜㹙甕齆罋塕螉鶲")
    ("wo" "我握窝沃卧挝涡斡渥幄|蜗喔倭莴龌肟嗌硪踒|偓涴馧|㠗䮸艧蒦瓁矆臒龏蝸渦濄䙠䁷䇶䀑腛媉踠䩊䂺仴瞃䰀䠎㠛焥唩嬳萵楃䁊㱧齷捾窩㦱涹撾濣臥㧴枂雘擭薶捼婑猧婐捰")
    ("wu" "务物无五武午吴舞伍污乌误亡恶屋晤悟吾雾芜梧勿巫侮坞毋诬呜钨邬捂鹜兀婺|妩於戊鹉垭仡浯蜈唔骛仵焐芴鋈庑鼯牾怃圬忤痦迕杌寤阢靰铻屼|旿郚胠洿峿珸珷鹀煟|㤇㘬蘁堊䓊埡䨁陚釫鼿㱱扝趶䚈䁷堥㝟䀛霿雺無䉑蟱㒇墲䅶䜑渞㐅伆鄔洖騖弙霚逜鰞茣霧娪娬鴮㐳㬳吳倵摀齀䑁㑄乄敄嵍歍鵐陓㡔譕腛鵡䍢塢奦瑦䡧嵨蕪剭㵲㹳㽾莁熃粅誈䒉媉岉躌䦍䮏窏螐璑禑隖䦜玝悞鎢誣誤溩憮瞴䎸箼鯃䳇俉㻍䫓䃖務嗚廡鷡忢䛩蓩䳱嫵矹㷻䟼兦亾熓䵦扜齬盓悪惡膴鶩兀甒儛噁橆杇呉啎潕汙汚扤祦遻卼麌碔沕悮窹烏軏鋘鋙杅皁")
    ("xi" "系西席息希习吸喜细析戏洗悉锡溪惜稀袭夕晰昔牺腊烯熙媳栖膝隙犀蹊硒兮熄曦禧茜嬉玺奚汐徙羲铣淅嘻|歙熹矽蟋畦褶郗唏皙隰樨浠蜥檄郄翕阋鳃舾屣葸匚螅咭酾粞觋欷僖醯鼷裼穸饩舄禊菥蓰郤恓晞豨潟窸䜣|屃钖肸咥俙饻枲绤棤睎傒㴔嶍嶲憙暿熻谿釐鳛巇酅爔|觿娭䑂誒䚫糦遟㒆㓾欪䵱䚂㦦戲㰥㱱㗲摡㳀嵆繫䒁墍旣䴛鼳瓕鈢鉩臈臘蝷嚊㜎䬣捿䚉䫔棲䦧䧿䘮㪪鰓䈢䨛㱇趇雭蜤杫燍䁯餙謚衋錫䯜䜁焁礂舃㠄嬆漇焈琋㸍㔒䀘縘漝椞焟稧鬩䈪怬焬欯戱騱䨳㤴謵怷㤸匸椺䐼霼餼唽騽橀豀齂恄葈噏屓煕屖歖㩗鵗鱚嵠㥡㭡䩤卥獥繥赥驨赩㹫蕮㽯豯襲晳㑶扸桸敼䙽㙾㵿䂀厀覀熂薂莃䢄徆螇熈䮎鎎媐銑㞒犔悕貕憘趘澙㚛螝隟䊠蒠覡覤瞦犧躧䲪嚱鎴蒵隵䚷肹喺熺㦻榽璽僁惁飁係蓆㿇䓇黊郋䧍壐矖翖黖瓗忚㣟䫣䛥忥㳧㗩鏭䏮諰磶觹觽㨙慀㩉奊䙎㩦㾙䚿䐅㘊餏䐖㰻羛緆戯扢蔇鈒繋莔鉨呬鸂磎巂雟瞚簁褉鄎謑渓瘜氥霫細縰蠵怸鰼譆卌潝蹝虩屭扱鑴綌纚犠疧徯滊囍習凞闟槢蟢燨釳狶釸盻諠")
    ("xia" "下夏峡厦辖霞虾狭吓侠暇遐瞎匣瑕|唬浃呷黠硖罅狎柙岈|叚翈鹡|䞩㵤䯟䒠蝦㙈㔠㭱嚇螛䒩㮫㿣敮閜挾浹徦埉䠍䨐䶝㰰㥦㛍廈陝陿捾縀丅夓鰕縖笚㘡䘥欱搳㰺谺轄鍜㽠㙤硤筪赮睱煵魻傄鎋䖎芐閕䖖䦖䪗疜炠疨颬鶷冾䛅㗇磍䫗懗烚俠珨鏬狹㗿諕仮舺憍陜謑舝騢祫蕸碬峽煆")
    ("xian" "现先县见线限显险献鲜洗宪纤陷闲贤仙衔掀咸嫌羡弦腺痫娴舷馅酰铣癣|冼涎暹籼锨苋蚬跹岘藓燹钐鹇氙锬莶霰跣痃猃彡祆筅挦伣|伭忺狝玹盷崄硍葴缐禒㬎孅|䦘䍎䐄譀㽉闞㦥㸝塪㰹礆㔓鰔䄯稴䵖䵛䵤筧見梘䶠覸糮䙹㔾䇒顈䡅㜷㲽壏懢蘚㱹䭑㦑薟㾾㯗欦譣㩮膁槏㯠臤㥦㛍攕姍釤揱錟饀銑㘅䘆蜆攇甉娊㘋唌錎䀏嬐贒嘕㬗瀗䨘搟尠餡䜢縣礥娨㜪訮輱㔵䨷鼸娹䤼褼鍁䁂䵇癇㩈㡉橌鍌䕔衘㭠啣䱤絤灦䝨㵪䥪獫癬蹮䉯顯婱䉳屳襳齴㭹鹹獻赻鱻㡾奾涀䞁㪇㺌㶍纎撏㦓澖纖䲗䚚躚銜咞䶟䶢誢䦥馦莧羨險羬㮭枮鮮瞯粯憲箲麲垷䒸㢺䢾㯀㳄㿅燅㧋䧋烍軐蓒臔藖珗胘蛝䧟賢㧥铦鋧毨㫫㳭姭䧮苮哯廯韯䃱僲僴峴櫶䃸䏹嫺嫻鷼臽㛾䗾搚䩙妶誸軒㰊麙䢭醶咁醎閒溓綅佡鏾縿県尟絃繊睍癎豏顕晛轞陥獮幰杴険玁撊閑銛憪綫嶮憸蚿韅秈僊俔仚線僩韱鷳諴鷴姺現鶱綖")
    ("xiang" "向项相想乡象响香降像享箱祥湘详橡巷翔襄厢镶飨|饷缃骧芗庠鲞舡葙蟓|珦瓖|䄈䛠䴂䦳䣈瓨缿銗塂䇨楿䤔夅䙹䒂鴹亯䬕銄姠欀項鄉餉鄊鐌鄕䔗栙䐟萫㐮䜶啌鱌襐晑潒饗鱜驤恦佭鑲詳絴鱶蚃薌羏䊑嶑麘鮝䖮䢽闀廂闂㟄緗鯗㟟勨跭㗽響皀澒閧鴂衖愬啍膷蠁饟纕嚮忀曏郷")
    ("xiao" "小效销消校晓笑肖削孝萧潇硝宵啸嚣霄淆哮筱逍|箫骁枭哓绡蛸崤枵魈鸮睄|洨恔虓涍猇翛敩滧熇皛薢蟏|囂㩭㒆㕺侾䒁絞䠛䴛鴵斆蟂訤髐㺒㪣庨骹燆騷䐹叜䇃㩋颵䌃揱䳋歒䎄䳂㵿薂踃㔅笅焇㤊嘋踍簘瀟嘨簫嘯㬵嘵萷彇虈啋婋驍歗穘硣顤䕧灱㹲灲䥵㑾筿綃斅䒕窙傚䒝梟誟憢皢㚣䊥撨鞩膮宯咲誵嚻熽藃曉毊㗛痟郩蟰俲謞澩獟䉰㮁嗃謼鵁筊勷謏橚潚縿莦箾嘐鴞蠨歊獢詨蕭呺暁銷効髇烋痚篠櫹殽鷕")
    ("xie" "些解协写血叶谢械鞋胁斜携懈契卸谐泄蟹邪歇泻屑挟燮榭蝎撷偕|歙亵楔颉缬邂鲑瀣勰榍薤绁渫廨獬骱躞揳澥|偰絜觟薢檞|㗨烲䞕䪥諜炧䍖䇋駭㜦㽳㷎䓳譮儶噧讗獦鬹㨒挾頡吤繲觧緳擷䬅㚗魼碿榝䁋䁯㢵齛孈屜屟䀘齂屓㩗䩤䙽㞒僁䏮䦖脅㸉瀉脋蠍㔎娎焎䔑㰔爕㨙搚攜㨝謝㰡㴬㴮紲伳愶缷褻㴽慀祄䕈㩉㽊䙊奊恊㝍䙎䭎瑎䉏㕐協齘㙝䙝㥟䡡蝢䉣齥㩦䵦䩧㭨卨㩪絬塮㙰䥱䕵衸籺衺䥾熁纈綊妎膎䦏㖑䦑䲒㞕䊝㒠䢡徢鞢垥冩㦪暬䚳鞵䚸斺禼㖿嗋糏翓㓔擕藛緤㳦諧旪寫㣯㣰韰燲拹㳿蝑䋶梋㡜㐖偞峫猲鮭褉滊脇蠏脥洩絏屧襭灺媟躠龤劦嶰疶㶸駴鍱")
    ("xin" "新心信寻辛欣薪馨鑫芯锌忻莘昕衅|歆囟忄镡炘伈䜣||庍㖕㻸鐔愖㽎㜦䅽姰㛙䎣阠㔤䪿㹷橝尋䜗嬜訢伩脪訫㐰攳樳䰼卂㭄襑顖孞煡桪噷杺䒖妡邤㚯䚱馸鋅㛛惞䛨盺俽馫㭡䛥礥軐㩪枔㾙鬵鈊舋鄩焮兟膷釁廞埶")
    ("xing" "行性形型兴星刑幸姓醒邢杏腥猩|悻荥陉惺饧擤硎荇省|钘骍铏婞瑆煋|䋲㒷娙㴆㬐銒䮍㚔箵狌餳餹興洐㨘㼛䰢㐩㼬䤯嬹渻䁄硏坓陘鍟㝭睲鮏㮐䂔皨馫垶篂䣆烆緈郉滎曐㓑臖䳙濚㣜㓝鋞觪䛭觲濴䓷鯹胻鈃讬騂倖鉶侀莕涬蛵嫈")
    ("xiong" "雄兄熊胸凶匈汹|芎讻|诇夐|哅焸賯䠗䎿訩㐫洶兇赨詾忷胷䧺詗恟敻")
    ("xiu" "修秀休宿袖绣臭朽锈羞嗅|岫溴庥馐咻髹鸺貅琇滫|珛脩|飍殠臰㩰㽲嚊脙鱃苬䜬㵻玊樇褏鵂潃齅饈繍㱙繡鎀綇綉㾋螑璓銝銹峀鏅烌蓚㗜俢髤鏥蓨臹鏽讬烋櫹褎茠糔")
    ("xu" "需许续须序徐蓄畜虚吁绪叙旭恤墟栩絮圩婿戌胥嘘浒煦旮酗|诩朐盱蓿溆洫顼勖馘肷糈砉醑媭魆|旴昫㳚珝垿冔谞欻湑嬬|蓲䨆嘼滀䧁暊㽳許歘滸烼掝蚼㒷䬎㖪䂸晇眗揟諿噓䎉俆㡏㦌朂伃㜅須頊䘏䔓䬔鬚倠䜡㤢鄦㘧㐨欰稰㰲怴伵漵稸㜿幁譃楈驉潊敍蝑䙒㑔魖虗敘㕛䱛虛聟㥠䅡詡魣偦蕦㑯㵰䍱籲卹汿沀瞁㖅䂆㞊銊續䢕䦗㚜疞窢蒣㾥㞰瞲㺷㦽䦽烅賉槒緒䛙盢㷦盨珬䣱䳳䣴㗵䋶壻妶蛡䬄謣惐鉥嬃縃訏欨訹慉鑐歔鱮呴繻殈侐続喣芧姁藇緖勗藚諝獝淢緰")
    ("xuan" "县选宣券旋悬轩喧玄绚渲璇炫萱癣漩眩|暄煊铉埙巽馔楦泫谖痃碹揎镟儇烜瑄鹮|伭玹咺昡晅琄衒愃暅暶禤嬛翾|㜗䛹䁔䥧喛㦥懁顈䡓姰㛐㔯㒸㻽㦏䚯䀏縣㔵癬㯀㿅㧋蓒䧟蜁吅䴉蠉䴋愋昍蘐贙㘣䠣䘩怰弲萲縼塇鉉繏蝖䍗䩙㹡䁢絢塤䝮䩰選䍻睻蕿䲂讂䮄檈梋㾌媗袨箮颴妶誸䲻鏇埍䧎矎軒鋗㳙嫙䗠㧦翧㳬懸藼諼㝁駨饌鐶狟煖佡敻琁蔙眴譞楥鍹鞙鶱璿諠駽玆")
    ("xue" "学血雪削薛穴靴|谑噱鳕踅嚯泶茓|峃敩|䠼㻡瀥㕰㷤䛎㨹䩈㦜矆䀜勪㔃䀗㔢疦䖼䋉謔㔧蒆㰒䆝㙾䒸斅䦑䜡怴瞲䬂䤕吙謞䨮鸴鱈䱑桖㡜䭥乴坹䎀㶅斈趐袕岤膤辥澩㖸㞽鞾㧒燢雤泬㿱䫻䫼䆷泧吷蹻疶學嶨壆觷鷽狘")
    ("xun" "讯训迅寻询循旬巡汛勋逊熏徇浚殉驯鲟荤薰|荀浔洵峋埙巽郇醺恂荨窨蕈曛獯挦珣|栒噀|㟧䃀㵌葷姰䡅㯠愻㡄咰尋攳樳卂桪㚯䀏撏潃䘩塤㜄㰊訊爋蘍蔒訓訙㨚伨㰬䭀㝁鑂坃畃䙉杊鱏鵔鵕鱘遜奞詢㽦顨潯偱䞊枔璕侚㢲䖲馴殾矄槆賐韗䛜毥壦駨勲勳廵䋸巺燻䵫稄鶽濬蕁灥潠揗鄩眴璿紃焄橁噚纁薫壎臐燖勛狥蟳迿篔")
    ("ya" "亚压雅牙押鸭呀轧涯崖芽哑讶鸦娅衙|丫蚜垭伢氩桠琊揠吖睚痖疋迓辂岈砑玡|姶堐猰|鶕椏圔唖啞埡䨙䆘䫖呾䍓軋䒁䀹拁㾏䂒㮞㴫㝦輅䖌㼤䫔㤉䧅疨㗇磍㹞䀴瘂漄笌稏猚鐚挜訝圠鴨氬䄰䰲㰳鼼䅉聐齖䵝㝞䝟蕥婭孲鵶齾厊庌庘劜䢝䊦䦪窫䪵犽雃䯉㧎壓铔俹㿿椻瑘釾厓輵烏潝鴉錏崕圧煆襾枒掗亜亞顔")
    ("yan" "研严验演言眼烟沿延盐炎燕岩宴艳颜殷彦掩淹阎衍铅雁咽厌焰堰砚唁焉晏檐蜒奄俨腌妍谚兖筵焱偃闫|嫣鄢湮赝胭琰滟靥阉魇酽郾恹崦芫阏剡鼹菸餍埏谳讠厣罨阽|觃沇䶮弇姶烻惔棪渰扊墕漹蝘戭虤黡嬿甗|䅬㜝洝䅖㱘婩晻䮗黬屵閼䪜㱌㒆懨䉷㺗墱䘶㺂鳱炏䶫䛳鬳㰽橪䌠鉛顩膁䦲䫡䦧挻挩捝錟䍾厃硏唌嬐㡉㵪齴䞁羬麲㥼黫訁琂䨄蔅嬊䀋䄋贋䴏騐猒萒焔㘖昖褗贗㘙愝䜩䌪㬫娫娮嬮騴鼴夵䤷渷簷礹椻儼椼䀽酀嵃遃顃楌䑍孍灎顏偐豓酓灔葕兗驗䭘魘䁙煙饜驠㕣偣彥敥剦䅧灧灩牪硯䱲乵艶艷㭺湺硽詽鹽䕾抁隁㢂醃㶄閆覎䲓㦔嶖䖗䊙麙㢛讞妟鶠麣䢥暥䎦㚧䂩䢭厭閰䂴厴嚴醶閹䊻閻躽檿㳂䓂㫃燄釅巌䣍㷔㿕懕曕巖巗巚珚䳛壛㫟䗡曣黤㓧壧櫩㛪鳫黭曮鷰姲㷳㗴篶姸䗺䳺裺諺㷼㿼䇾狿㖶阭囐淊溎齗穽豣撧齞縯鈆鴈鰋爓験鴳匽嵒嵓豔顔噞塩啱讌龑隒綖莚碞馣嚥喦喭厳醼傿鷃揅郔巘臙揜黶觾殗")
    ("yang" "样养央阳洋扬杨羊氧仰秧痒漾疡泱殃恙鸯徉|佯怏炀烊鞅蛘旸卬玚|飏咉钖垟羕|䍩㦹軮崸飬鴹氜勜歍䞁潒羏䭥楧攁鸉鐊養瘍眏鰑㬕䬗樣㔦阦鴦䬬輰氱崵霷䬺眻紻㨾慃䑆蝆詇䭐䁑佒瑒鉠煬敭䵮婸陽㺊䒋禓傟岟羪抰炴劷䖹㟅柍姎揚駚珜䇦懩㿮雵諹愓瀁昜楊鍚癢坱暘颺様胦")
    ("yao" "要么约药邀陶摇耀腰遥姚窑瑶咬尧钥谣肴夭侥吆疟妖|幺杳舀洮窈曜鹞爻徼繇崤徭荛轺铫鳐崾珧峣靿|垚窅猇隃猺媱|㑸鴁驁詏䳬䚆䯚䛂鑃銚䠛㬭䆗烄僥瘧纅㺒㓞嫍䒒㤊顤㑾䭥㹓鼼䔄倄約餆䌊搖䬙䌛訞䴠謠蘨鰩㴭尭堯㨱㔽䍃筄䙅楆䁏㝔䁘遙艞獟偠瑤齩䑬䉰鑰㵸婹穾㞁㮁岆宎熎袎讑㢓溔枖䆙䂚暚榚䆞覞䢣榣嶤䶧抭趭窯窰䖴䚺䚻䋂嗂鷂闄矅㫏㫐㿑烑蓔狕㿢䋤藥苭㟱仸軺柼怮㫍箹鎐恌眑騕餚謡愮祅葯葽殀喓窔傜嶢薬颻殽鷕飖揺燿鴢邎")
    ("ye" "业也夜叶野液冶页爷耶邪咽椰烨掖拽曳晔|谒腋噎揶琊靥邺铘玡|倻焆堨楪馌|䅬䅖㽢䎨痷㡋糦㓈䴾㱌㷸殕嘩㜇㧉礏鄓㴸䡾䳖䖣䮜葉釶啘䔑䭎㭨㩪䢡㖿䁆䭇歋䱒抴䇩䋵䝟䊦窫謁頁鸈䈎吔㐖爗嘢㸣餣䤳鄴䤶鐷爺㱉㩎㙒捓瑘驜偞䥟䭟䥡㝣靨㙪業潱㥷䥺鵺鎁嚈㪑鎑䲜殜㖡皣嶪嶫亱瞱澲㖶墷瞸枼枽燁壄曄曅䓉曗擛鋣忦䧨擨擪峫曵僷㗼釾饎暍熀狥漜蠮饁煠鍱殗埜擫皁")
    ("yi" "一以已意议义益亿易医艺依移衣异伊仪宜遗疑毅谊亦艾疫役忆抑乙译翼蛇溢椅沂逸蚁夷邑怡绎彝裔姨熠贻矣屹颐倚诣胰奕翌弈轶驿壹曳猗|臆弋铱旖漪迤佚翊诒怿痍懿饴峄揖眙镒仡黟肄咿翳挹缢呓刈咦嶷羿钇殪荑薏蜴镱噫癔苡悒嗌瘗笫阝衤欹佾埸硪圯绁舣搋酏劓乂匜祎浥锜螠杙|佁枍㑊叕昳胣洢宧扅扆勚㴔椸裛廙潩嫕鹝踦鹢燚繄簃|㱯㢊凒賹㼢䛖䳬糦㛳㔭䈕栘耛䔟䑛㮛拸䇼䧝䂽㲲䳀襗㕈䐙㵧䤭㔕㠖㱞䇣䄩䡾䰯䕥㵫䭲孴譺㞾㣇䧇狋钀䑄䖌䔇䱈䉝㾨綺䏌忔裿㐌㹑㚶鉈謚鉇詍䁺箷釶䴰䃞坄鴺䧅苐䣡彵狏迱贀遺焈焬歖㹫敼媐䚷㔎紲䣱辥鏔儀㘁㴁䄁䬁夁洂鸃億嬄䐅伇㘈怈頉㘊䴊縊㜋餏嬑㜒㴒礒萓䐖笖㰘逘鈘䰙蘙吚㰝䘝瘞嬟帠鈠輢阣眤頤䬥頥稦栧䄬䔬䴬弬霬欭䬮㠯瘱眱焲訲㔴鈶瀷䘸㐹褹崺攺㰻䌻謻䄿伿鐿乁䭂浂㱅䁆幆䭇㽈獈虉顊歋䱌䕍䱒詒㹓穓㝖㩘䝘捙湙豙彛豛驛㑜彜㵝䝝歝䭞彞恞䩟㙠敡顡詣艤㑥㕥䉨㵩譩衪㡫㹭呭陭㙯䝯譯鉯異議㱲晲浳㥴繶齸呹晹繹㡼靾䭿兿䖁㖂䢃鶃䞅芅暆䎈炈㦉膉讉䖊䮊肊撎榏宐䲑㞔殔肔螔瞖悘袘羛讛掜冝㢞㶠羠庡袣㚤㦤悥檥銥鞥㚦嶧鮧義熪醫嶬劮骮溰鎰醳䦴抴玴憶沶妷醷檹殹澺㲼垼熼誼貽㦾㺿䆿墿隿㣂䓃苅䯆珆緆迆秇跇䓈囈毉㫊俋懌曎㳑䗑旑釔㛕鷖䋚嫛藝䗟跠蛡苢鯣蛦䣧䧧鷧䇩竩痬棭寱燱寲黳釴飴䇵䋵瓵㓷䗷槷巸槸䓹觺㣻蟻軼郼䵝㸣㝣㙪㥷欥耴栺饎扡誃斁睪愒蜺鶂鷊衵錡鮨虵戺阤詑唈弌訑圛帟洟訳桋饐艗顗轙杝鹥齮偯豷襼瑿亄侇檍禕銕螘熤貤袲枻曀鷁柂駅泆峓黓藙蛜燡鳦勩壱埶蓺迻鷾喦皁樴")
    ("yin" "因引银印音饮阴隐姻殷淫尹荫吟瘾寅茵圻垠|鄞湮蚓氤鳏胤龈窨喑铟洇狺夤廴吲霪茚夂堙訚愔䜣|纼骃珢硍崟龂歅跱䲟溵禋慭蟫嚚|阥堷韽䗞䨙梀䡨㶣䒞㸧齦伒唫䖐玪䴛䴦䠴赺䌠㥲㒚㗃䰼㴽圁䤃朄㐆㴈㼉猌鈏㸒訔洕紖鈝蘟霠訡䌥蔩蔭崯栶䨸䤺㕂㱃䕃筃癊噖㝙乚湚䡛㹜㹞婣㡥㙬癮㥯陰酳荶陻㥼噾摿銀隂檃鞇侌璌粌蒑碒讔㖗憗㾙趛䖜䒡鮣㞤殥㪦斦銦㦩䪩檭馻檼垽嶾犾䚿苂䓄㧈烎凐鏔䇙囙懚滛鷣磤濥濦㣧黫绬飮諲飲秵緸櫽峾淾韾泿釿霒欭㝖酓䅧䕾㖶輑䏖㐺乑齗窾瘖縯慇絪婬靷垔憖冘隠隱螾誾裀闉廕駰傿")
    ("ying" "应营影英迎映硬盈赢颖婴鹰荧莹樱瑛蝇萦莺颍膺缨瀛楹|罂荥萤鹦滢蓥郢茔嘤璎嬴瘿媵撄潆|媖溁锳|䩕䒢㦹䚆㕲夃䀰焸㬐䔔㹵㚞唡嵤覮啨䵥鱦旲䋼䤰㜲滎濚濴绬㴄瀅蠅縈䨍攍鸎贏霒㨕攖鸚鐛䤝瀠蘡鴬褮瀯礯嬰蠳䀴瀴頴渶萾偀孆鍈䑉䭊塋罌譍鑍穎䁐䭗㹙㹚䁝籝㑞噟䕦楧浧蝧瑩㵬䙬摬癭譻孾蝿璄㶈㢍暎熒纓鞕颕㲟膡螢鎣碤鶧梬䪯鶯嚶藀盁巆應巊㯋僌緓瓔㿘濙營矨鷪廮珱䃷鷹櫻櫿䑍䭘眏䬬柍䇦耺䵴吋剉呎韹俓霙啢縄甇謍甖朠潁罃煐桜籯営嫈蛍賏応韺禜")
    ("yo" "哟|唷||嚛喲")
    ("yong" "用永拥勇涌泳庸俑踊佣咏雍甬镛臃|邕蛹恿慵壅痈鳙墉饔喁蕹颙|栐㶲埇㛚鄘鲬澭|㑙㦶揘飬鰫鷛䮵㣧㴄搈愑嬫㐯踴愹慂牅鱅塎癕㝘詠硧㽫彮噰癰嵱㙲悀傛醟銢㞲㦷䞻銿擁嫆勈㷏鯒苚嫞鏞柡䗤惥槦郺滽湧傭撧灉顒禜嗈雝廱")
    ("you" "有由又优游油友右邮尤忧幼犹诱悠幽佑釉柚铀鱿囿坳酉攸黝|莠猷蝣疣呦蚴莸莜铕宥繇卣牖鼬蝤尢蚰侑羑|祐朓浟鲉麀槱戭耰|㑸梎禉妋㶭揂獶汼羗汓䒒尣銹㕛䌊㔽䍃䂚䚻䬀唀瀀䀁蜏輏㤑脜丣怣㘥優怮逰㰶猶鈾䱂聈䅎㽕蕕㕗牗偤㹨酭牰㕱湵魷䑻貁憂䚃梄䢊㮋沋纋鮋羐嚘誘䆜㾞䢟㺠㒡莤䞥銪㚭庮岰䒴亴䖻㻀駀苃哊㫍䳑狛㓜㛜秞峟峳郵迶姷㳺滺䛻櫾糿栯㳛甴麕搧褎冘逌鴢訧輶鄾遊楢邎肬斿櫌泑狖懮怞")
    ("yu" "于与育余预域予遇语誉玉鱼雨渔裕谷愈娱欲吁舆宇羽逾豫郁寓狱喻御浴愉禹俞蔚榆愚渝尉淤虞屿峪粥驭瑜禺毓钰隅芋熨瘀迂煜昱|於臾畲盂聿竽萸妪腴圄谕觎揄龉谀俣馀庾妤瘐鬻欤腧鹬阏阈嵛雩鹆毹菀菸圉蜮伛纡窬窳饫蓣狳肀丿舁蝓燠玙彧堉嵎铻滪邘圫|㳚峿敔鄅悆鄃淯隃琟棫畬腘㺄矞缐瑀艅褕隩薁潏遹燏髃|澚䞝閼䓊湡䬛㳼䨁䧕㹼㫹䞕儥㵄㽳雽䛎䰥䋭糓䤋丂㧕蝺貗㤜鱊㭌桙㻰䱾氀䢖頨䘻䵥兪䩱䎉㪌㢏娪陓䬑㷉叞衘伃䬔䜡㤢㰲魣籲㚜㦽㧒䫻捓㮋㺠㻀䳑䴁漁㬂䬄䈅鰅紆鸆礇與㼌蘌輍欎䄏預䨒礖㠘䘘餘琙娛蘛欝䨞茟蜟騟䔡砡稢堣謣㤤欥錥㠨䄨嬩爩鴪戫匬堬萭鐭萮栯㬰茰㔱䘱霱鬱䐳逳焴㼶稶鈺䰻䜽輿䍂獄籅偊噊䁌鵌䥏㙑酑䩒癒鵒㥔㥚魚䉛扜鍝䍞籞歟楡㝢硢㽣祤㥥穥聥衧㑨䁩灪䵫齬畭睮㡰硲扵歶虶䱷蕷穻譽慾㪀媀㒁喅䖇䮇悇銉䂊芌嶎亐喐貐䢓斔䮙㦛㶛䂛㒜㺞澞語䲣螤㚥龥禦覦醧䢩袬骬馭㺮蒮傴䆷螸嶼庽鮽㲾閾䣁俁㣃迃軉䃋惐㷒盓雓䛕䋖嫗秗懙㳛棛諛鷠黦䗨飫忬諭櫲䏸鷸蓹毺俼篽鳿灹楀僪幙貍牏媮麌鋙捥藇顒瘉鸒礜鴥娯踰鬰崳唹杅饇歈驈魊蕍繘獝轝罭楰噳齵箊玗斞澦喩羭璵鋊緎寙棜旟淢釪緰惌")
    ("yuan" "元员院原源远愿园援圆缘袁怨渊苑冤媛猿垣沅塬垸鸳|辕鸢瑗圜爰芫鼋橼螈眢箢掾湲|妧焆涴蜎嫄薳羱|㼂䳁䟦輐䈠䝠䭴喛䣰㟲䝹䖤㱧㘣弲萲䲻䱲㛪渁渆䬇圎褑園圓渕鼘鴛謜鼝贠褤䬧猨㤪蜵逺䬼嬽灁獂轅㭇䅈剈㹉衏㥐遠䩩杬魭噮㥳葾傆邍禐溒蚖薗蒝榞貟䖠厡鶢榬蒬䲮鶰鎱妴媴厵䛄䛇駌䏍䳒櫞䳣緣囦棩裫㟶盶鳶裷鋺黿䡝隕䦾悁縁笎夗猭騵鹓願蝝蝯鵷円肙邧寃惌員淵")
    ("yue" "月说约越乐跃阅岳粤悦曰钥|栎曜钺樾瀹龠哕刖玥汋|骍焆彟趯爚籥|扚噦彠矆臒㯞㭾䆕䆢䟑樂攊檪櫟說哾䎳焥嬳䒸䬂䎀㧒約鑰㵸䋤藥黦鈅鸙㰛蘥㬦䤦㜧䠯㜰䥃籆䡇㹊鉞籰恱捳䢁枂䖃岄悅躍蚎蚏妜閱䢲䶳粵箹嶽跀䋐篗䟠泧曱矱髺楽擽鋭説軏戉鸑礿抈閲禴狘")
    ("yun" "员运云允孕蕴韵酝耘晕匀芸陨纭熨郧筠|恽韫郓氲殒愠昀狁鋆赟|沄妘涢缊榅筼煴蕰馧|煾頵㚞㷉叞輼緼㩈韗馻㒁贠蜵貟眃鄆蘊㜏輑鄖鈗紜縜䤞褞愪阭氳䨶耺夽饂運慍橒䉙䡝腪齳䵴䩵蝹荺蒀㚃枃暈傊䚋㞌抎蒕隕醖喗殞醞鶤玧䞫䆬䲰榲䪳䢵蒷㚺䦾秐䇖韞㟦囩賱惲雲藴緷勻韻餫涒員贇縕伝霣畇蕓奫薀熅熉澐枟溳磒篔")
    ("za" "杂扎砸咋匝|咂拶臜||囃噈䆘濽䦈㳨啑䣠䨿帀紮襍鉔魳䕹偺䞙䪞沞抸鮺嶻迊囐雜雥韴磼囋紥灒雑臢喒")
    ("zai" "在再灾载栽仔宰哉|崽甾||睵酨載儎㴓渽扗䵧㱰災侢傤㞨䮨䏁烖䣬賳䔂洅縡菑")
    ("zan" "赞暂咱攒|昝簪糌瓒錾趱拶臜|寁酂|䟃㨻噆䍼䗞䥄攅攢濽㔆㙻䬤䐶䭕䟅㳨鏩篸兓䍝㳫鐟䙉偺鐕儧簮㤰礸㜺鄼兂桚饡穳襸讃撍讚暫沯趲禶㣅瓉囋瓚㟛鏨㭮灒酇臢贊欑儹喒賛揝蹔")
    ("zang" "藏脏葬赃臧|奘戕驵牂臜||匨牫驡弉贓贜㘸塟鍺羘銺賍髒駔賘臟蔵臢臓")
    ("zao" "造早遭枣噪灶燥糟凿躁藻皂澡缫|蚤缲唣璪|惛慥簉|謲䒃㯥繰繅䖣䗢䜊唕艁䥣蹧鑿䲃梍㲧醩薻竃竈棗髞㿷喿譟皁傮趮")
    ("ze" "责则泽择侧咋啧|仄箦赜笮舴昃迮帻|䃎鲗|䔾荝側㥽厠廁矠嫧䯔㤞㢎伬襗㮣㚖簀樍禝䂝㟙㺓䟄䅚䔼㻭捑䶡鰂稄蠈蠌嘖昗夨耫謮蔶䰹汄則䕉幘齚䕪庂瞔皟澤䶦責溭㖽㳁擇諎䇥㣱㳻賾葃礋飵鸅謫灂泎齰睪唶崱萴択沢")
    ("zei" "贼||鲗|鯽鰂蠈㖽戝鱡賊")
    ("zen" "怎|谮||㻸譖撍譛䫈")
    ("zeng" "增曾赠憎|锃甑罾缯|鄫矰鬷|曽䇸䵴贈䰝譄繒驓䙢㽪璔綜熷鋥磳縡橧増")
    ("zha" "查扎炸诈轧闸渣蜡咋乍榨楂札栅眨咤|柞喳喋铡蚱吒砟揸痄哳齄膪拃|挓奓溠鲊䃎劄鲝霅齇|囃㯥䈟蔖䆛蕏藸㓃剳箚鰈㩹䮢䄍䢱䟻㔍㑵軋䀹搩齟䶥䱹蠟蝋䙄㞚㳐䮜㚫譇潳鞢䄰偞紮鮺耫諎㽪䞢䋏簎㴙樝紥謯甴踷蠿牐詐譗鍘䵙㱜䕢摣偧䵵䥷㡸灹㒀厏閘咜抯宱䖳皶皻㷢苲柵䛽䋾觰挿査喥箑煠搾鮓醡蚻柤")
    ("zhai" "债择宅寨侧摘窄斋祭翟|砦瘵膪|擿|㢎䓱䑲啇㟙齊亝䍉駘擇㒀厏礋齋鉙捚㩟㡯債榸飵厇檡抧骴択斎岝")
    ("zhan" "展战占站崭粘湛沾瞻颤詹斩盏辗绽毡栈蘸|觇旃谵骣搌拃|飐栴偡嵁嶦鹯鳣|䗞䳻䏼䐤顫㙴覘惉䧯䬤䵣㔊噡黵飦餰䩆碊閚鳽䁪譧㜊䈴㟻椫䆄䦅暫氈氊霑輚㠭戰䘺椾輾䡀䩅蹍橏驏佔魙䱠譠虦譫䁴䱼嶃嶄趈䪌榐䎒㺘嶘薝讝㞡斬覱綻䗃䟋䋎菚旜㟞盞棧㻵㣶嫸欃蛅儃襢怗鸇琖戦樿詀鉆轏蹔饘驙桟鱣虥呫邅醆颭皽")
    ("zhang" "长张章障涨掌帐胀彰丈仗漳樟账杖璋|嶂仉瘴蟑獐幛鄣嫜鸮|暲|㙊兏䩨镸弡㢓蔁礃餦瘬漲帳張脹騿扙慞㙣遧㕩㽴粀墇涱傽賬痮䛫長粻鱆瞕鞝麞")
    ("zhao" "着照招找召朝赵兆昭肇罩钊沼嘲爪|诏啁棹笊|柖鸼旐曌|㫤鼌䰫䄻䑲䧂㹿鳭窼爫佋㷹㑿鮡䖺䈃㨄䈇鉊詔䝖㕚䍜鍣䍮瑵㡽肁羄肈箌㺐䮓趙枛妱瞾盄䃍㷖釗狣菬燳巶雿啅皽鵫垗炤櫂駋")
    ("zhe" "这者着浙折哲蔗遮辙辄柘|锗褶蜇蛰鹧谪赭摺乇磔螫夂|晢啫喆詟䗖䗪|㸞㢎䆛袩轍䁤㯙䧪䋲㞼䮰䙷棏㰅潪㡳䊮聑㭯㫼㡇䂝䜆粍䀅䁋㭙懾㵊樜杔驝馲矺扸㦻軼鍺鸅䐑輒砓䠦謫䐲耴謺䝃䝕啠晣䵭虴襵籷歽䩾讁厇禇㞏讘䂞䊞嚞檡䎲銸㪿蟄蟅䓆瓋埑鷓烢嫬㯰䏳嗻㝂㙷慹淛䇽謶窾蹻屩悊輙這慴讋鮿")
    ("zhen" "镇真针圳振震珍阵诊侦臻贞枕桢赈祯帧甄斟缜箴疹|娠鸠砧榛揆鸩轸稹溱蓁胗椹朕畛浈|纼昣袗揕葴瑧瑱禛鬒|㭫䴾㨋鈂謓屒鍖敶㲀裖䀕湞䫃㱽鐤槇䡩竧䡅䪾沵䀼䠴䧵䚯鷆㛛枮鮝㼉紖榐蜄鴆笉挋栕栚縝縥㴨帪弫䨯㐱錱搸診堻樼幀鉁轃聄籈瑊䑐靕酙陣塦潧楨䝩祳偵晸絼禎蒖覙㪛媜殝貞䂦䂧蒧誫抮鎮侲㮳䪴䲴辴澵䊶㣀揁嫃㓄寊賑䏖針㯢軫䫬䳲䟴酖沴甽黮鉆眕眞眹紾獉遉桭鱵鍼抌碪鎭薽珎鷏黕駗姫黰")
    ("zheng" "政正证争整征郑丁症挣蒸睁铮筝拯桢帧峥|怔诤狰徵钲鲭|烝憕|脀崝㐼塣憆徎䆸䫆埥掟鯖鮏浧幀晸禎媜貞揁鬇鴊猙錚崢䈣踭鄭氶愸㽀佂聇證䥌㱏䡕睜㡠鉦㡧䥭抍箏掙䦛撜炡徰䦶䂻姃諍埩䛫鏳糽朾襼眐爭証癥徴篜")
    ("zhi" "制之治质职只志至指织支值知识直致执置止植纸智殖秩旨址滞氏枝芝脂帜汁肢挚稚酯掷峙炙栉侄芷窒咫吱趾|痔蜘郅桎雉祉郦陟痣蛭帙枳踯徵胝栀贽祗豸鸷摭轵卮轾彘觯絷跖埴夂黹忮骘膣踬沚祇扺|泜咥庤珪晊虒梽畤铚媞蔀跱锧稙滍禔榰疐擿蹢|䜠鵄遟汦歭傂淔䩢蹛㿃㥀㥁㜱䀸䱃摕潪䑭㡳墆䂡䏑䧝䟡㬪䘭眰䲀䞇㲳臷㫼剬鶨䃽謢䤠楖㡶鯯䪒瀄擳鳩摨䰴呮疷杫㾅䚇識䶡䖨銴㫑菭䝰嚔䣽璏䚳斦妷秇䘣㧗㧻䠦㯰䦛㨁倁褁嬂挃贄鼅䐈㘉輊帋砋漐洔㨖紙鴙㴛娡䌤椥儨訨礩鴩䐭騭搱鴲縶洷䬹䄺栺稺崻㜼聀䭁坁㝂䵂㕄䉅䑇驇䕌潌䥍祑豑豒織䉜幟䱥汥陦坧襧䱨䅩筫祬摯㙷䝷職䡹慹衼垁䞃禃憄䆈墌誌徏躑躓梔劕綕芖㲛徝覟袠庢䚦劧抧熫貭䦯膱螲隲馶㮹䎺疻隻馽䏄軄䳅瓆䟈㗌䓌櫍翐秓䫕秖䛗觗旘蟙鷙䇛櫛淛釞瓡藢觢駤㣥㗧姪質䏯滯擲秲巵胵觶執臸軹緻䇽淽製狾㛿䇧厓樲捗茞戠蚔瘈廌乿楺軽衹穉殢鉄阤讬碔疧埶扻搘猘値紩阯樴蘵騺恉鑕蹠偫遰犆銍厎厔袟胑鋕寘柣迣懥秪懫秷鳷")
    ("zhong" "中重种众终钟忠仲衷肿踵|冢盅忪锺舯螽夂|茽柊穜|褈祌蝩徸塜鼨㼿䳋銿螤瘇鐘蔠鈡刣鴤種堹㐺眾終煄偅乑幒筗塚籦潨腫䱰歱蹱衳衶汷鍾炂妐媑妕蚛喠㲴泈緟諥㣫鈆伀尰衆湩彸狆")
    ("zhou" "州周洲舟骤轴昼宙粥皱肘柚咒|帚胄绉啄纣妯啁诌繇碡籀驺酎荮㤘|㑇辀鸼婤椆赒鲖|㫶㹐㨶嚋薵㿒㥁㣙胕㝌㗙䔭䏔䓟翢䐌䭥俼謅驟諏㨄㲴洀紂㔌霌䐍縐輖㼙䈙伷鵃詋籒籕䩜晝噣葤呪晭睭㥮㑳銂䶇冑箒粙疛䖞徟㾭皺䎻䛆淍駎烐䋓珘賙鯞䇠㛩矪郮菷軸䐢紬詶鞚斨碻埶掫甃騆輈怞週譸喌侜咮盩诪僽")
    ("zhu" "主术住注助属逐著筑驻朱珠祝猪诸柱竹铸株瞩嘱贮煮烛苎褚蛛拄铢洙竺蛀渚伫杼|侏诛茱箸炷躅翥潴邾槠舳橥丶瘃麈疰|纻苧柷砫蠋|敱䫖蕏㶆䎝墸泏藸鯺䇬䳠䠼䐗䘄䪒逫磩鸀尌䠱術兪屬䆝薥迬嗻噣䇠軸笁鼄㔉霔㤖欘䘚笜眝䰞䬡䐢䘢爥樦瀦䌵紸鴸註蠾茿嵀鱁硃鑄䍆佇㝉孎㑏煑絑䝒鉒䭖罜灟㹥祩蝫䝬豬㵭䥮筯䕽坾劅誅麆莇趉宔銖㺛炢鮢劯貯䮱殶䎷㾻袾㫂䟉築濐駐囑跓諔矚竚䇡櫡㧣䇧櫧櫫篫燭曯駯觰飳壴軴諸秼㿾斀蠩跦篴跙藷芧珎朮紵詝陼窋劚羜馵斸蓫鋳")
    ("zhua" "抓挝爪||髽|膼撾檛簻")
    ("zhuai" "拽|||睉尵顡跩転")
    ("zhuan" "专转传赚砖撰篆沌|馔啭颛抟䏝|瑑僎|㙛㼷傳諯耑腞叀䚈䤄㛯孨䧘恮㑷䏙賺專甎堟鄟襈饌籑顓譔䡱䉵灷瑼蒃膞囀鷒磚蟤嫥竱篿篹剸転専縳鱄轉塼")
    ("zhuang" "装状庄壮撞妆幢汝桩|奘僮戆|漴|䚒湷贛戅戇弉樁焋娤梉莊妝狀糚裝壵獞泆荘粧壯")
    ("zhui" "追坠缀椎锥赘|惴隹骓缒|倕|䧳娷腄桘㝽㩾諈腏䋘䨺轛磓㚝㾽錗醀䢡㿢䃍贅騅礈縋䄌笍錐鴭娺錺鑆譵墜沝窡綴錣甀畷醊餟鵻硾膇")
    ("zhun" "准屯谆|肫窀圫|䐃衠|埻飩稕訰宒綧鶽諄凖啍旽準迍")
    ("zhuo" "着著缴桌卓捉赃琢灼浊酌拙茁涿|镯绌啄濯焯倬擢斫踔棹诼浞禚棁魆汋|叕椓晫燋|㧳䴵矠䙯䔲㲋䂐䮕趠䆯諁䃗䵠㹿㠚㒂籗繳捔穱䞵䏐㓸䟾狵䐁圴鉵聉蓔㧻蠿䵵䮓噣坧墌劅櫡娺窡錣鐯鐲丵謶琸㑁灂啅穛剢㭬罬籱䕴䅵硺斀䶂㺟炪㪬撯妰斱犳斵濁擆泎諑槕鋜烵㣿藋鵫蠗蝃彴斮斲梲鷟篧棳")
    ("zi" "资自子字咨滋仔姿紫兹孜淄籽梓鲻渍姊吱|秭恣甾孳訾滓锱鹚辎趑龇赀眦缁呲笫谘嵫髭茈粢觜耔嗞孖|虸洓鄑渟镃鼒|㠿㢀䆅䖪䦻㧘䴾㑵㬐齍䁒璾蓻㾅㞨蠀䔂吇䐉茊紎椔錙輜鴜茡㰣䘣栥鰦漬鈭崰茲倳稵㰷輺㜽頾訿頿䅆杍葘啙齜呰㱴孶湽鍿鶅禌芓榟趦㺭貲秄緇鯔姕㧗釨諮秶矷沝穧磒菑眥剚牸扻玆鎡澬鶿資姉胏胔胾")
    ("zong" "总综宗纵踪棕|粽鬃偬枞腙疭熜|倧鬷|樅錝鍯総鏓緫䙕䎫䁓䡯鑁䖲綜縂搃鬉倊䰌騌猔愡朡猣縦踨堫昮稯縱椶總㹅䝋嵏艐嵕䍟蝬捴嵸葼憁碂㢔骔熧緃糉㷓揔蓗惣磫糭鯮緵㯶鯼惾潨撦鍐騣瘲摠蹤豵傯翪")
    ("zou" "走邹奏|揍诌诹驺陬楱鄹鲰|飏|䅳齱䲀㞫謅鄒䠫騶㔿驟㵵齺箃媰黀諏鯫棷㔌赱掫緅菆郰棸")
    ("zu" "组族足祖租阻卒|俎诅镞嗾槭驵菹捽|岨珇砠崒疐|䬨鏃䃚㵀紣䙘㲋㩆葅箤伹稡鉃鉐趲駔䔃唨崪㰵爼踿組詛䱣蹵䅸靻鎐䖕䚝㲞蒩傶㞺㧻䯿錊謯伜顇籓怚踤卆哫柤邅")
    ("zuan" "钻赚纂攥|缵躜|敩|䰖㷪䡅饡䌣䤸鑚繤鑽纉䂎纘躦賺籑䉵篹儹揝籫")
    ("zui" "最罪嘴醉咀|羧蕞觜|衒槜|濢槯嗺䓱㵃酨嫢㭰稡欈䯿洅錊㰎㠑栬樶䘹絊㝡酻祽噿嶉嶊檌䮔纗嶵璻蟕鋷朘脧蕝睟酔晬檇厜辠墬")
    ("zun" "尊遵|鳟樽撙|僔僎噂嶟|袸䔿拵罇譐鱒繜銌鷷栫瀳跧鐏捘墫壿")
    ("zuo" "作做座左坐昨佐馈琢凿撮祚|柞唑嘬酢怍笮砟阼胙捽|岞|侳䟶䬤㶡䦈㩇䟄䩦䟭㤰㘸䜊鑿諎䇥㣱唨䯿㘀夎稓䔘㸲㘴鈼葃㑅繓䝫㭮㝾袏咗䞢䎰䞰䋏毑㛗秨糳飵莋椊葄筰岝"))
  "拼音汉字对照表.

第一个元素为拼音，第二个元素为拼音对应的汉字组成的字符串，字符串
中汉字按照使用频率排序，字符串用 “|” 隔开，分成一级常用汉字，二级
常用汉字，三级常用汉字和不常用汉字。分类方式参考了国务院公布的
《通用规范汉字表》:

    国务院关于公布《通用规范汉字表》的通知(国发〔2013〕23号)
    http://www.gov.cn/zwgk/2013-08/19/content_2469793.htm

但不是完全一致。")

(defvar pyim-pymap-duoyinzi-chars
  '(("a" "阿")
    ("ai" "艾")
    ("ao" "坳")
    ("ba" "扒")
    ("bai" "柏")
    ("bang" "磅")
    ("bao" "薄" "曝" "暴" "堡" "剥" "刨")
    ("beng" "蚌" "泵")
    ("bi" "裨" "臂")
    ("bian" "扁" "匾" "便")
    ("bo" "伯" "泊")
    ("bu" "卜" "不")
    ("can" "参")
    ("cang" "藏")
    ("ce" "侧")
    ("ceng" "曾")
    ("cha" "查" "差")
    ("chai" "拆")
    ("chan" "禅" "掺")
    ("chang" "厂" "长")
    ("chao" "嘲" "朝")
    ("che" "辙" "车")
    ("cheng" "乘" "称" "橙" "澄")
    ("chi" "尺")
    ("chou" "臭" "仇")
    ("chuan" "传")
    ("chuo" "绰")
    ("ci" "伺")
    ("da" "大")
    ("dai" "呆")
    ("dan" "单" "弹")
    ("dao" "叨")
    ("de" "的" "得")
    ("di" "底" "地")
    ("diao" "调")
    ("ding" "丁")
    ("dong" "洞")
    ("dou" "都")
    ("du" "读" "度")
    ("dun" "顿" "蹲" "沌" "敦")
    ("duo" "堕")
    ("e" "恶")
    ("fan" "繁" "番")
    ("feng" "冯")
    ("fo" "佛")
    ("fou" "否")
    ("ga" "旮")
    ("gai" "盖")
    ("gang" "扛")
    ("gao" "镐")
    ("ge" "革" "铬" "蛤")
    ("gei" "给")
    ("gong" "汞")
    ("gou" "枸")
    ("gu" "谷")
    ("gua" "括" "呱")
    ("guan" "莞")
    ("guang" "广")
    ("gui" "龟" "柜")
    ("hang" "夯")
    ("hao" "蒿")
    ("he" "菏" "核" "和" "呵" "合")
    ("hei" "嘿")
    ("hong" "虹" "红")
    ("hu" "鹄" "浒")
    ("hua" "划")
    ("huai" "徊")
    ("huan" "还")
    ("hui" "会")
    ("hun" "荤")
    ("huo" "豁")
    ("ji" "藉" "缉" "稽" "祭" "瘠" "亟")
    ("jia" "贾" "家" "夹" "伽" "价")
    ("jian" "见")
    ("jiao" "脚" "缴" "角")
    ("jie" "芥" "秸" "解")
    ("jin" "劲")
    ("jing" "颈" "荆")
    ("ju" "桔" "咀" "句")
    ("jue" "jiao" "觉")
    ("jun" "浚")
    ("ka" "咖" "卡")
    ("kai" "楷")
    ("kan" "槛")
    ("kang" "亢")
    ("ke" "坷" "咳")
    ("keng" "吭")
    ("kui" "馈" "溃" "匮")
    ("la" "蜡" "腊")
    ("lao" "烙" "姥")
    ("le" "勒" "了" "乐")
    ("lei" "肋")
    ("lin" "赁")
    ("ling" "棱")
    ("liu" "六")
    ("lu" "陆" "碌")
    ("luo" "络" "落")
    ("lv" "缕" "绿" "率")
    ("mai" "脉" "埋")
    ("mang" "氓")
    ("mao" "冒")
    ("me" "么")
    ("mei" "酶" "没")
    ("meng" "萌")
    ("mi" "谜" "秘" "泌")
    ("mian" "娩")
    ("miu" "缪")
    ("mo" "模" "摩" "抹")
    ("mou" "牟")
    ("mu" "沐")
    ("na" "那" "哪" "呐")
    ("nan" "南")
    ("ne" "呢")
    ("ni" "溺")
    ("niao" "鸟" "尿")
    ("nong" "弄")
    ("nuo" "娜")
    ("nve" "疟")
    ("pan" "ban" "扳")
    ("pang" "膀" "胖" "旁")
    ("pao" "炮")
    ("pi" "辟")
    ("pin" "拚")
    ("ping" "屏")
    ("po" "魄" "迫")
    ("pu" "璞" "莆" "脯" "瀑" "朴" "埔")
    ("qi" "骑" "蹊" "栖" "期" "契" "其" "奇")
    ("qian" "乾" "铅" "茜" "浅" "嵌" "堑")
    ("qiang" "强")
    ("qiao" "荞")
    ("qie" "且" "茄")
    ("qin" "亲")
    ("qu" "区")
    ("quan" "券" "圈")
    ("que" "雀")
    ("sai" "塞")
    ("sao" "缫")
    ("se" "色")
    ("sha" "莎" "厦")
    ("shan" "杉")
    ("shang" "裳")
    ("she" "赊" "蛇" "摄")
    ("shen" "莘" "沈" "参" "什")
    ("sheng" "盛" "省")
    ("shi" "食" "适" "识" "石" "氏")
    ("shu" "熟" "漱" "术" "数" "戌" "属")
    ("shuai" "衰" "率")
    ("shui" "谁")
    ("shuo" "说")
    ("si" "似")
    ("su" "宿")
    ("suo" "缩" "梭")
    ("tang" "汤" "倘")
    ("tao" "陶")
    ("ti" "提")
    ("ting" "铤" "烃")
    ("tui" "褪")
    ("tun" "屯" "囤")
    ("tuo" "拓" "沱")
    ("wan" "万")
    ("wang" "亡")
    ("wei" "蔚" "圩")
    ("wo" "涡" "挝")
    ("wu" "鹜" "无")
    ("xi" "铣" "洗" "戏" "系")
    ("xia" "虾" "吓")
    ("xian" "癣" "县")
    ("xiang" "降" "巷")
    ("xiao" "嚣" "削" "校")
    ("xie" "邪" "挟" "偕")
    ("xing" "行")
    ("xu" "嘘" "许")
    ("xue" "血")
    ("xun" "寻")
    ("yan" "腌" "咽")
    ("yao" "钥" "窑" "侥")
    ("ye" "曳" "叶")
    ("yi" "遗")
    ("yin" "殷")
    ("yong" "甬")
    ("you" "柚")
    ("yu" "尉" "吁")
    ("yuan" "员")
    ("yue" "约")
    ("yun" "蕴" "熨")
    ("za" "咋")
    ("zai" "仔")
    ("zang" "赃" "臧")
    ("zao" "皂")
    ("ze" "择")
    ("zha" "楂" "栅" "扎")
    ("zhan" "辗" "斩")
    ("zhao" "召")
    ("zhe" "着" "折")
    ("zhi" "炙" "殖" "挚" "峙" "吱")
    ("zhong" "种" "重")
    ("zhou" "粥")
    ("zhu" "著")
    ("zhua" "爪")
    ("zhuai" "拽")
    ("zhuan" "赚")
    ("zhui" "缀" "椎")
    ("zhuo" "琢")
    ("zi" "兹")
    ("zu" "卒"))
  "多音字最常用的读音。")

(defvar pyim-pymap-duoyinzi-words
  '(("a" "吖啶" "腌臜")
    ("ang" "张卬")
    ("ao" "拗口" "拗陷")
    ("ba" "峇厘")
    ("bai" "伯特" "伯琮" "伯颜" "叔伯")
    ("ban" "扳动" "扳回" "扳子" "扳平" "扳手" "扳机")
    ("bang" "彩蚌" "翅膀" "肩膀" "膀子" "膀臂" "臂膀")
    ("bao" "龅牙")
    ("bei" "孛星" "摇臂" "断臂" "月孛" "胳臂")
    ("bi" "俾使" "俾倪" "刘辟" "吡啶" "复辟" "泌阳" "睥睨" "秕子" "秕糠" "秘鲁" "豆皀" "贲临" "辟邪")
    ("biao" "嘌呤")
    ("bin" "刘玢" "槟城" "槟子" "香槟")
    ("bing" "主屏" "屏息" "屏气" "屏除" "槟州" "槟榔")
    ("bo" "丝柏" "刻薄" "剥削" "剥啄" "剥夺" "剥掉" "剥皮" "剥离" "剥落" "剥蚀" "单薄" "厌薄" "巨擘" "微薄" "摊薄" "擘开" "擘画" "柏林" "浅薄" "淡薄" "稀薄" "绵薄" "萝卜" "薄产" "薄利" "薄厚" "薄命" "薄地" "薄幸" "薄弱" "薄待" "薄情" "薄技" "薄明" "薄晓" "薄暗" "薄暮" "薄海" "薄田" "薄礼" "薄膜" "薄荷" "薄酬" "薄雾" "薄面" "超薄" "轻薄" "鄙薄")
    ("bu" "吴堡" "堡寨" "堡的" "大埔" "安瓿" "座堡" "米堡")
    ("ca" "喀嚓" "拆毁")
    ("cen" "参差")
    ("cha" "刹那" "咔嚓" "喀嚓" "梵刹" "罗刹")
    ("chai" "交差" "信差" "出差" "差事" "差旅" "差派" "差遣" "支差" "邮差")
    ("chan" "单于" "惊颤" "震颤" "颤动" "颤抖" "颤栗")
    ("chang" "裳凤" "週長" "霓裳")
    ("chao" "征剿")
    ("chen" "不称" "匀称" "可称" "后称" "对称" "抻面" "沈积" "沈重" "沈默" "相称" "称多" "称德" "称心" "称是" "称职" "称臣")
    ("cheng" "丁盛" "之盛" "刘晟" "李晟" "盛的" "祐樘" "称稱" "饼铛")
    ("chi" "一匙" "汤匙" "茶匙" "豆豉")
    ("chong" "三重" "二重" "双重" "多重" "李重" "艨艟" "辎重" "重传" "重估" "重修" "重做" "重写" "重印" "重发" "重叠" "重合" "重名" "重启" "重回" "重围" "重塑" "重复" "重审" "重庆" "重建" "重开" "重拍" "重拾" "重振" "重播" "重整" "重新" "重来" "重构" "重温" "重演" "重点" "重现" "重生" "重用" "重申" "重的" "重组" "重罚" "重置" "重获" "重装" "重覆" "重评" "重试" "重读" "重赛" "重蹈" "重载" "重返" "重造" "重逢" "重重" "重门" "重阳")
    ("chu" "人畜" "仔畜" "公畜" "家畜" "李柷" "牧畜" "牲畜" "畜力" "畜栏" "畜牲" "畜生" "畜禽" "畜类" "畜肥" "禽畜" "耕畜" "褚遂")
    ("chuai" "啜饮" "搋子" "搋面")
    ("chun" "鹌鹑")
    ("ci" "兹的" "参差" "鸬鹚")
    ("cong" "单枞" "厚熜" "枞树")
    ("cu" "卒于" "吏卒" "士卒" "戍卒")
    ("cuan" "拼攒")
    ("cui" "衰变")
    ("cuo" "公撮" "妄撮" "撮合")
    ("dai" "大夫" "大黄")
    ("dan" "王赡" "脏彈" "黄疸")
    ("dang" "铃铛")
    ("de" "代地" "似地" "其地" "北地" "地与" "地为" "地做" "地向" "地听" "地在" "地委" "地学" "地对" "地将" "地底" "地把" "地村" "地用" "地的" "地矿" "地置" "地说" "地走" "多地" "少地" "市地" "汉地" "猛地" "的地" "目地" "般地")
    ("dei" "之得" "也得" "习得" "你得" "又得" "就得" "得之" "得先" "得其" "得去" "得多" "得太" "得好" "得宠" "得很" "得最" "得有" "必得" "总得" "我得" "真得" "而得" "还得" "都得" "非得")
    ("deng" "澄清" "真澄" "高澄")
    ("di" "中的" "宝坻" "寻的" "打的" "提溜" "提防" "标的" "的士" "的确" "目的" "美的" "面的")
    ("dian" "佃农")
    ("die" "喋喋")
    ("ding" "樱町" "畹町" "虰蛵")
    ("dong" "侗族" "恫吓" "杨侗" "硐室")
    ("dou" "句读")
    ("du" "上都" "下都" "东都" "中都" "丰都" "为都" "之都" "乐都" "于都" "京都" "他都" "们都" "军都" "副都" "南都" "古都" "后都" "商都" "国都" "大都" "天都" "奠都" "宁都" "宇都" "安都" "宗都" "官都" "定都" "帝都" "建都" "德都" "性都" "成都" "故都" "文都" "新都" "昌都" "望都" "武都" "江都" "留都" "的都" "皇都" "等都" "臣都" "花都" "计都" "迁都" "都会" "都兰" "都匀" "都城" "都山" "都市" "都护" "都昌" "都是" "都有" "都灵" "都的" "都督" "都等" "都统" "都要" "首都")
    ("duo" "哆啦" "哆嗦" "审度" "揣度" "驮子")
    ("e" "东阿" "吟哦" "哦哦" "噁嗪" "噁心" "蛾类" "阿堵" "阿胶" "阿谀" "阿附" "阿难" "飞蛾")
    ("ei" "诶笑" "诶诒")
    ("en" "嗯啊" "嗯嗯")
    ("fa" "砝码" "酦酵")
    ("fei" "米芾")
    ("fen" "刘玢" "坋粒" "焚书" "焚毁" "焚烧" "自焚")
    ("feng" "焚书")
    ("fu" "仿佛" "果脯")
    ("ga" "伽马" "呷呷" "咖喱")
    ("gai" "芥兰" "芥蓝" "蓋兹")
    ("gan" "乾癣" "乾道" "孙乾" "承乾" "烘乾")
    ("ge" "咯咯" "回纥" "袼褙")
    ("gu" "回鹘" "鸿鹄")
    ("gua" "蜗牛")
    ("gui" "王珪" "鲑鱼" "鳜鱼")
    ("guo" "涡阳" "絮聒" "聒噪" "聒耳")
    ("ha" "虾夷" "蛤蟆" "蛤蟹")
    ("hai" "上还" "中还" "也还" "人还" "他还" "但还" "你还" "倒还" "却还" "她还" "它还" "宗还" "帝还" "您还" "我还" "放还" "谁还" "还不" "还为" "还以" "还会" "还像" "还剩" "还去" "还叫" "还可" "还和" "还在" "还多" "还大" "还好" "还嫌" "还对" "还将" "还小" "还少" "还差" "还带" "还应" "还很" "还得" "还快" "还怕" "还想" "还找" "还把" "还挺" "还敢" "还早" "还是" "还曾" "还有" "还望" "还未" "还来" "还没" "还玩" "还用" "还看" "还真" "还算" "还给" "还能" "还行" "还被" "还要" "还让" "还说" "还请" "还贵" "还远" "还问" "还需" "还须" "还高" "这还" "那还" "里还")
    ("hang" "一行" "世行" "两行" "中行" "交行" "人行" "内行" "农行" "几行" "分行" "同行" "商行" "在行" "外行" "太行" "央行" "巡行" "工行" "建行" "总行" "懂行" "我行" "投行" "招行" "换行" "排行" "支行" "改行" "本行" "桁杨" "每行" "港行" "生行" "行业" "行中" "行会" "行列" "行在" "行家" "行市" "行幸" "行当" "行情" "行数" "行的" "行省" "行署" "行营" "行规" "行货" "行长" "该行" "跨行" "车行" "转行" "这行" "银行" "闵行")
    ("hao" "蚝油" "貉子" "貉绒" "镐京")
    ("he" "上颌" "下颌" "吓声" "回纥" "威吓" "恐吓" "恫吓" "无颌" "有颌" "阖家" "阖庐" "阖闾" "隔阂")
    ("heng" "桁架" "桁梁" "道行")
    ("hu" "丝核" "吓唬" "和牌" "回鹘" "核菌" "菌核" "诈唬")
    ("huan" "朱鹮" "苋菜")
    ("hui" "秦桧" "讌會")
    ("hun" "珲春")
    ("huo" "上和" "为和" "义和" "书和" "会和" "值和" "元和" "克和" "党和" "兽和" "军和" "力和" "又和" "和一" "和三" "和不" "和与" "和东" "和两" "和中" "和之" "和二" "和亲" "和以" "和伊" "和元" "和党" "和六" "和其" "和分" "和刘" "和副" "和北" "和反" "和可" "和合" "和后" "和周" "和唐" "和在" "和外" "和多" "和大" "和嫔" "和对" "和小" "和州" "和帕" "和帝" "和张" "和文" "和新" "和晋" "和暖" "和更" "和最" "和有" "和本" "和李" "和村" "和杰" "和梁" "和王" "和的" "和省" "和秦" "和等" "和红" "和罗" "和萧" "和被" "和西" "和议" "和赵" "和辽" "和队" "和陈" "和非" "和马" "和高" "团和" "国和" "在和" "城和" "夫和" "奖和" "子和" "字和" "孙和" "孝和" "宗和" "官和" "室和" "家和" "寨和" "将和" "局和" "岩和" "州和" "年和" "庄和" "廷和" "性和" "感和" "成和" "战和" "戚和" "所和" "拌和" "搀和" "搅和" "斯和" "是和" "暖和" "机和" "权和" "条和" "林和" "梗和" "氏和" "求和" "汉和" "法和" "清和" "灯和" "特和" "王和" "田和" "的和" "福和" "税和" "组和" "罗和" "罪和" "者和" "营和" "部和" "险和")
    ("ji" "供给" "光奇" "奇偶" "奇尼" "奇数" "奇羡" "奇袭" "扶箕" "畚箕" "箕子" "簸箕" "给与" "给予" "给付" "给水" "自给" "补给")
    ("jia" "沦浃" "茄克" "雪茄")
    ("jian" "槛车" "轩槛")
    ("jiang" "下降" "不降" "倔强" "再降" "升降" "天降" "强嘴" "归降" "招降" "沉降" "空降" "被降" "起降" "迫降" "降下" "降世" "降临" "降为" "降价" "降低" "降到" "降压" "降哥" "降噪" "降将" "降幅" "降息" "降水" "降清" "降温" "降生" "降福" "降级" "降职" "降至" "降落" "降解" "降雨" "降雪" "霜降")
    ("jiao" "一觉" "侥幸" "倒嚼" "剿匪" "剿灭" "包剿" "午觉" "咬嚼" "围剿" "好觉" "审校" "征剿" "懒觉" "校准" "校勘" "校对" "校正" "校的" "校订" "校阅" "校验" "检校" "清剿" "温峤" "睡觉" "觉觉" "调校" "追剿")
    ("jie" "仓颉" "慰藉" "桔梗" "盘诘" "蕴藉" "藉口" "藉着")
    ("jin" "矜持" "骄矜")
    ("jing" "刘晟" "刚劲" "劲卒" "劲吹" "劲射" "劲峭" "劲急" "劲拔" "劲挺" "劲敌" "劲旅" "劲烈" "劲爆" "劲直" "劲草" "劲风" "强劲" "李晟" "箐村")
    ("jiong" "何炅")
    ("ju" "莴苣" "趑趄")
    ("juan" "垫圈" "猪圈")
    ("jue" "主角" "名角" "咀嚼" "张角" "角力" "角抵" "角斗" "角色" "角逐" "角闪" "角阙" "配角")
    ("jun" "朱隽" "溥儁" "象龟")
    ("kang" "扛着")
    ("kao" "犒劳")
    ("ke" "借壳" "壳口" "壳子" "壳幔" "壳牌" "壳虫" "壳阶" "外壳" "弹壳" "机壳" "空壳" "脑壳" "蛋壳" "贝壳" "龟壳")
    ("kuai" "会计" "财会")
    ("kui" "傀儡" "隗嚣")
    ("kuo" "包括" "囊括" "总括" "括号" "括弧" "括毒" "括线" "概括" "综括" "赵括")
    ("la" "世㻋" "落下")
    ("lang" "莨菪" "阆中")
    ("lao" "落汗")
    ("le" "中肋")
    ("lei" "世㻋" "勒姆" "勒紧" "勒莫" "富勒" "帕勒" "福勒")
    ("leng" "棱台" "棱柱" "棱角" "棱锥" "棱镜" "模棱" "穆棱" "绥棱")
    ("lia" "他俩" "俩人" "咱俩" "我俩" "这俩")
    ("liang" "伎俩" "咱俩" "技俩" "雅靓" "靓丽" "靓仔" "靓女" "靓妹" "靓照" "靓颖")
    ("liao" "不了" "了了" "了事" "了却" "了得" "了断" "了然" "了结" "了解" "多了" "得了" "明了" "未了" "末了" "潦倒" "潦草" "知了" "罢了" "蓼蓝")
    ("lin" "拎着")
    ("liu" "常陆" "水陆" "陆凯" "陆励" "陆贽")
    ("long" "泷泽" "泷船")
    ("lou" "泄露" "露出" "露脸" "露面" "露馅")
    ("lu" "不露" "六合" "六安" "出露" "初露" "吐露" "外露" "寒露" "展露" "带露" "披露" "揭露" "显露" "暴露" "曝露" "梦露" "泄露" "泻露" "流露" "甘露" "白露" "蔗露" "蜜露" "表露" "裸露" "败露" "透露" "陈露" "雨露" "霜露" "露体" "露出" "露台" "露天" "露宿" "露水" "露点" "露珠" "露茜" "露营" "露西" "露露" "露面" "露韩" "露骨" "面露")
    ("luan" "娈童")
    ("lun" "丙纶" "涤纶" "绦纶" "腈纶")
    ("luo" "漯河")
    ("lv" "棕闾" "阖闾")
    ("ma" "干麼" "干麽" "抹布")
    ("man" "埋入" "埋怨" "滋蔓" "苏蔓" "蔓延" "蔓草" "蔓菁" "藤蔓")
    ("mang" "巨蟒" "蟒蛇")
    ("mao" "玳瑁" "铆工" "铆接" "铆钉")
    ("me" "什麼" "什麽" "这麼" "这麽" "那麽")
    ("mei" "谜儿")
    ("mi" "糜烂")
    ("mian" "渑池")
    ("mo" "出没" "吞没" "埋没" "嬷嬷" "沉没" "没收" "没药" "没落" "沦没" "浸没" "淹没" "潜没" "覆没" "隐没")
    ("mu" "姥爷" "模具" "模板" "模样" "牟平" "牟羽" "超模")
    ("na" "丽娜" "厉娜" "多娜" "娜娜" "娜的" "安娜" "李娜" "狄娜" "王娜" "琳娜" "谢娜")
    ("nao" "孬种")
    ("ne" "哪吒" "声呐")
    ("nei" "那个" "那位" "那台" "那天" "那家" "那张" "那次" "那种" "那边")
    ("ni" "去呢" "呢喃")
    ("nian" "粘土" "粘帖" "粘度" "粘性" "粘液" "粘痰" "粘稠" "粘粘" "粘胶" "粘膜" "粘贴" "胶粘")
    ("nie" "蘖枝")
    ("niu" "执拗")
    ("nu" "内帑")
    ("o" "你哦" "哦哟" "哦哦" "好哦")
    ("pa" "枇杷" "钉耙")
    ("pan" "拚命" "拚弃" "番禺" "番莲")
    ("pang" "彷徨" "牛蒡" "磅礴")
    ("pao" "龅牙")
    ("pei" "碳碚")
    ("pi" "蚍蜉")
    ("pian" "便与" "便便" "便宜" "便由" "扁壶")
    ("piao" "朴茨" "王朴")
    ("pie" "苤蓝")
    ("ping" "冯保" "冯剑" "冯异")
    ("po" "朴子" "朴茨" "泊头" "湖泊" "王朴")
    ("pu" "堡寨" "堡的" "座堡" "曝光" "曝露" "米堡")
    ("qi" "会稽" "侦缉" "槭木" "焉耆" "荸荠" "菜畦" "蒲圻" "鮨科")
    ("qia" "博卡" "卡洛" "卡的" "卡迪" "卡队" "和卡" "哨卡" "尔卡" "洛卡" "的卡" "菲卡" "髂骨")
    ("qian" "光纤" "犍为" "纤夫" "纤纤" "蕁麻")
    ("qiang" "戕害" "箐村" "羟基" "自戕")
    ("qiao" "冰橇" "刀鞘" "叶鞘" "地壳" "壳口" "壳牌" "壳虫" "壳质" "壳阶" "温峤" "甲壳" "躯壳" "雀子" "雪橇" "鞘脂" "髓鞘")
    ("qing" "亲家" "箐村" "血亲")
    ("qiu" "仇和" "仇士" "象龟")
    ("qu" "區旗" "蠼螋")
    ("que" "乙炔")
    ("rao" "刍荛" "桡骨" "阻桡")
    ("ru" "汝南" "汝城" "汝州" "汝阳")
    ("sa" "灑脱")
    ("sai" "鳃裂")
    ("se" "充塞" "出塞" "堵塞" "塞罗" "填塞" "拥塞" "搪塞" "杜塞" "栓塞" "梗塞" "淤塞" "血塞" "闭塞" "阻塞" "鼻塞")
    ("sha" "上杉" "刹住" "刹把" "刹车" "手刹")
    ("shai" "套色" "本色" "色子" "褪色")
    ("shan" "人单" "光栅" "单县" "单发" "单桨" "单生" "单福" "单超" "受禅" "封禅" "掸邦" "栏栅" "格栅" "王赡" "的单" "禅位" "禅定" "禅寺" "禅让" "赡养")
    ("shang" "羹汤")
    ("shao" "召入" "召到" "召来" "叶鞘" "杓子")
    ("she" "和折" "折可" "折本" "折秤" "折耗" "折钱" "歙县" "畲族" "瞽阇" "阇梨")
    ("shei" "是谁" "有谁" "谁也" "谁人" "谁会" "谁又" "谁家" "谁就" "谁是" "谁有" "谁来" "谁的" "谁知" "谁能" "谁要" "谁让" "谁说" "谁都")
    ("shen" "妊娠" "桑椹")
    ("sheng" "刘晟" "李晟")
    ("shi" "乌什" "什叶" "什邡" "似地" "似的" "喀什" "奥什" "好似" "布什" "殖质" "波什" "繁峙" "纳什" "腐殖" "豆豉" "通什" "钥匙")
    ("shou" "常熟" "成熟" "煮熟" "熟语" "腐熟")
    ("shu" "數周" "腧穴")
    ("shuang" "泷水")
    ("shui" "游说" "说客" "说服")
    ("si" "伺服" "伺机" "伺隙" "同食" "服食" "食材" "食邑")
    ("sou" "咳嗽")
    ("su" "李愬")
    ("sui" "尿样" "扶绥" "晋绥" "浽溦" "绥中" "绥化" "绥宁" "绥德" "绥棱" "绥江" "绥滨" "绥蒙" "绥阳" "绥靖")
    ("suo" "羧基" "羧酸")
    ("ta" "拓印" "拓片" "拖沓" "杂沓" "蛋挞" "鞭挞")
    ("tan" "乱弹" "动弹" "反弹" "弹出" "弹力" "弹劾" "弹匣" "弹压" "弹唱" "弹奏" "弹射" "弹性" "弹拨" "弹指" "弹斥" "弹牙" "弹琴" "弹簧" "弹纠" "弹花" "弹词" "弹跳" "手弹" "枪弹" "难弹")
    ("tang" "祐樘")
    ("tao" "临洮" "洮南")
    ("tiao" "上调" "下调" "不调" "区调" "协调" "可调" "回调" "失调" "宫调" "征调" "微调" "汉调" "烹调" "特调" "空调" "联调" "调价" "调休" "调低" "调侃" "调停" "调养" "调到" "调制" "调剂" "调匀" "调味" "调和" "调回" "调处" "调好" "调性" "调情" "调戏" "调成" "调控" "调教" "调整" "调料" "调来" "调校" "调温" "调理" "调的" "调皮" "调羹" "调至" "调节" "调解" "调试" "调速" "调配" "调门" "调队" "调音" "调频" "调驯" "调高")
    ("tie" "呫哔" "呫嗫" "呫毕" "嗫呫")
    ("ting" "室町" "樱町")
    ("tong" "杨侗" "洪洞")
    ("tou" "骰子")
    ("tun" "褪绿" "褪色")
    ("tuo" "驮兽")
    ("wan" "苏蔓" "莞尔")
    ("wei" "上尉" "中尉" "卫尉" "太尉" "尉氏" "少尉" "廷尉" "校尉" "浽溦" "都尉" "隗嚣")
    ("wen" "李玟")
    ("weng" "蕹菜")
    ("wo" "喔喔" "耳蜗" "蜗居" "蜗庐" "蜗旋" "蜗牛" "蜗蜒" "蜗行")
    ("wu" "厌恶" "可恶" "唔好" "唔知" "唔系" "嫌恶" "恨恶" "憎恶")
    ("xi" "勐腊" "栖霞" "水螅" "的腊" "腊味" "露茜" "鸂鶒")
    ("xia" "厦华" "厦大" "厦门" "陜西")
    ("xian" "光纤" "化纤" "安閒" "秋狝" "纤体" "纤小" "纤巧" "纤度" "纤弱" "纤毛" "纤细" "纤维" "纤芯" "苋菜" "閒暇" "閒混" "閒话" "閒谈")
    ("xiang" "李愬")
    ("xie" "仓颉" "便血" "出血" "吐血" "吸血" "的血")
    ("xin" "奕䜣" "莘庄")
    ("xing" "井陉" "反省" "省亲" "省察" "省悟" "省视" "自省" "荥阳")
    ("xiu" "一宿" "参宿" "宿命" "宿存" "星宿")
    ("xu" "丙戌" "壬戌" "庚戌" "戊戌" "戌狗" "牧畜" "甲戌" "畜牧")
    ("xuan" "乾癣")
    ("xue" "削减" "削去" "削夺" "削平" "削弱" "削职" "削藩" "剥削" "噱头" "戏谑" "瘦削" "磨削" "谐谑" "谑剧" "谑戏" "谑语")
    ("xun" "张浚" "毒蕈" "荨麻" "香蕈")
    ("ya" "倾轧" "枝桠" "琅琊" "轧场" "轧染" "轧棉")
    ("yan" "刘縯" "屋檐" "抽菸" "湮灭" "芫荽" "菸硷" "菸碱" "菸蒂")
    ("yang" "张卬")
    ("yao" "么点" "对幺" "幺二" "星曜" "约塔" "钟繇" "锺繇")
    ("ye" "呜咽" "哽咽" "琅玡" "琅琊" "琅邪")
    ("yin" "吲哚" "湮灭" "牙龈" "齿龈")
    ("ying" "荥经")
    ("you" "钟繇")
    ("yu" "俞夏" "蔚县")
    ("yuan" "胡芫" "芫花" "雇員")
    ("yue" "乐之" "乐器" "乐团" "乐坛" "乐工" "乐师" "乐府" "乐手" "乐曲" "乐清" "乐章" "乐经" "乐谱" "乐迷" "乐队" "乐音" "哀乐" "器乐" "国乐" "声乐" "奏乐" "密钥" "弦乐" "民乐" "爱乐" "礼乐" "胡乐" "配乐" "钟乐" "阎乐" "雅乐" "音乐")
    ("yun" "筠连")
    ("za" "包扎" "塔扎" "扎囊" "腌臜")
    ("zai" "甾酮")
    ("zan" "攒钱" "积攒")
    ("zang" "地藏" "宝藏" "川藏" "援藏" "搜藏" "滇藏" "玄奘" "藏人" "藏区" "藏文" "藏族" "藏民" "藏独" "藏獒" "藏羚" "藏语" "西藏" "进藏" "道藏" "青藏")
    ("zao" "凿井" "凿子" "凿枘" "凿空" "开凿" "榫凿" "确凿")
    ("zeng" "中曾" "他曾" "克曾" "曾哥" "曾孙" "曾山" "曾巩" "曾布" "曾祖" "曾随" "特曾" "还曾")
    ("zha" "冷轧" "柞水" "热轧" "轧制" "轧机" "轧钢")
    ("zhai" "祭天" "祭礼" "翟凌")
    ("zhan" "呫吨" "粘在" "粘贴" "粘连")
    ("zhang" "不长" "乡长" "了长" "二长" "会长" "信长" "兄长" "光长" "军长" "刘长" "助长" "区长" "厂长" "厅长" "县长" "台长" "司长" "团长" "园长" "在长" "增长" "处长" "外长" "大长" "学长" "家长" "尊长" "局长" "州长" "市长" "师长" "年长" "庆长" "店长" "庭长" "总长" "成长" "或长" "所长" "排长" "斜长" "旅长" "族长" "机长" "村长" "校长" "次长" "水长" "消长" "滋长" "状长" "班长" "生长" "的长" "省长" "社长" "科长" "站长" "等长" "组长" "至长" "舰长" "船长" "色长" "营长" "行长" "见长" "警长" "议长" "财长" "车长" "连长" "道长" "部长" "酋长" "镇长" "长了" "长兄" "长兼" "长冈" "长出" "长到" "长势" "长史" "长吏" "长圆" "长在" "长大" "长女" "长姊" "长子" "长孙" "长官" "长庆" "长得" "长成" "长毛" "长满" "长片" "长的" "长相" "长着" "长老" "长者" "长而" "长肉" "长胖" "长辈" "长进" "长高" "队长" "院长" "馆长" "首长")
    ("zhao" "一朝" "上朝" "不着" "了朝" "今朝" "入朝" "历朝" "嘲哳" "失着" "宗朝" "当朝" "得着" "找着" "早朝" "明朝" "朝之" "朝会" "朝夕" "朝政" "朝时" "朝晖" "朝有" "朝气" "朝的" "朝臣" "朝见" "朝贡" "朝阳" "朝霞" "李朝" "来朝" "爪哇" "爪牙" "的朝" "着凉" "着忙" "着急" "着想" "着慌" "着手" "着法" "着火" "着紧" "着落" "着边" "着迷" "着陆" "着魔" "睡着" "累朝" "视朝" "辽朝" "金朝" "阮朝" "陈朝")
    ("zhe" "故辙" "没辙" "苏辙" "褶皱" "覆辙")
    ("zheng" "魏徵")
    ("zhi" "月氏" "标识" "栉比" "王珪" "魏徵")
    ("zhou" "钟繇")
    ("zhu" "白术" "苎麻" "褚遂")
    ("zhuan" "传位" "传略" "传至" "传记" "传赞" "侠传" "假传" "列传" "前传" "史传" "外传" "小传" "左传" "盛馔" "自传" "记传" "重传")
    ("zhuang" "一幢")
    ("zhun" "家屯" "屯兵" "屯村" "屯田" "驻屯")
    ("zhuo" "免缴" "剥啄" "吸着" "扣缴" "执着" "沉着" "着力" "着墨" "着处" "着实" "着床" "着想" "着意" "着手" "着棋" "着眼" "着笔" "着色" "着落" "着装" "着重" "着陆" "穿着" "缴销" "衣着" "身着" "附着" "预缴" "黏着")
    ("zi" "产仔" "仔姜" "仔猪" "仔畜" "仔细" "华仔" "狗仔")
    ("zong" "厚熜" "枞阳")
    ("zu" "菹醢" "顇奴")
    ("zuan" "赚钱")
    ("zuo" "枘凿" "琢磨" "酬酢"))
  "多音字对应的词组。")

(defvar pyim-pymap--py2cchar-cache1 nil
  "拼音查汉字功能需要的变量.

类似: \"a\" -> (\"阿啊呵腌|嗄吖锕||錒\")")

(defvar pyim-pymap--py2cchar-cache2 nil
  "拼音查汉字功能需要的变量.

类似: \"a\" -> (\"阿\" \"啊\" \"呵\" \"腌\" \"|\" \"嗄\" \"吖\" \"锕\" \"|\" \"|\" \"錒\" \"\")")

(defvar pyim-pymap--py2cchar-cache3 nil
  "拼音查汉字功能需要的变量.

类似: \"a\" -> (\"阿啊呵腌|嗄吖锕||錒\" \"爱 ... 溾\" \"厂广安 ... 菴馣\" \"昂盎 ... 軮䇦\" \"奥澳 ... 鴢泑\")")

(defvar pyim-pymap--cchar2py-cache nil
  "汉字转拼音功能需要的变量.

类似: \"艾\" -> (\"yi\" \"ai\")")

(defvar pyim-pymap--py2duoyinzi-cache1 nil
  "汉字转拼音功能需要的变量")

(defvar pyim-pymap--py2duoyinzi-cache2 nil
  "汉字转拼音功能需要的变量")

;; ** "汉字 -> 拼音" 以及 "拼音 -> 汉字" 的转换函数
(defun pyim-pymap-cache-create (&optional force)
  "创建 pymap 相关的 cache."
  (pyim-pymap--cchar2py-cache-create force)
  (pyim-pymap--py2cchar-cache-create force)
  (pyim-pymap--py2duoyinzi-cache-create force))

(defun pyim-pymap--cchar2py-cache-create (&optional force)
  "Build pinyin cchar->pinyin hashtable from `pyim-pymap'.

If FORCE is non-nil, FORCE build."
  (when (or force (not pyim-pymap--cchar2py-cache))
    (setq pyim-pymap--cchar2py-cache
          (make-hash-table :size 50000 :test #'equal))
    (dolist (x pyim-pymap)
      (let ((py (car x))
            (cchar-list (string-to-list (car (cdr x)))))
        ;; NOTE: "|" 是做为分割符使用的，删除。
        (dolist (cchar (remove ?| cchar-list))
          (let* ((key (char-to-string cchar))
                 (cache (gethash key pyim-pymap--cchar2py-cache)))
            (if cache
                (puthash key (append (list py) cache) pyim-pymap--cchar2py-cache)
              (puthash key (list py) pyim-pymap--cchar2py-cache))))))))

(defun pyim-pymap--py2cchar-cache-create (&optional force)
  "构建 pinyin 到 chinese char 的缓存.

用于加快搜索速度，这个函数将缓存保存到 `pyim-pymap-py2cchar-cache' 变量中，
如果 FORCE 设置为 t, 强制更新索引。"
  (when (or force
            (not pyim-pymap--py2cchar-cache1)
            (not pyim-pymap--py2cchar-cache2)
            ;; FIXME: 我偶尔会遇到一个奇怪的问题，创建的缓存没有包含所有的汉字拼
            ;; 音，原因未知，所以这里测试一下，看排在最后面的一个汉字拼音是否包
            ;; 含在缓存中，如果不包含，就重新创建缓存。
            (and pyim-pymap--py2cchar-cache1
                 (not (gethash "zuo" pyim-pymap--py2cchar-cache1))))
    (setq pyim-pymap--py2cchar-cache1
          (make-hash-table :size 50000 :test #'equal))
    (setq pyim-pymap--py2cchar-cache2
          (make-hash-table :size 50000 :test #'equal))
    (setq pyim-pymap--py2cchar-cache3
          (make-hash-table :size 50000 :test #'equal))
    (dolist (x pyim-pymap)
      (let* ((py (car x))
             (cchars (cdr x))
             (n (min (length py) 7)))
        (puthash py cchars pyim-pymap--py2cchar-cache1)
        (puthash py (cdr (split-string (car cchars) ""))
                 pyim-pymap--py2cchar-cache2)
        (dotimes (i n)
          (let* ((key (substring py 0 (+ i 1)))
                 (orig-value (gethash key pyim-pymap--py2cchar-cache3)))
            (puthash key (delete-dups `(,@orig-value ,@cchars))
                     pyim-pymap--py2cchar-cache3)))))))

(defun pyim-pymap--py2duoyinzi-cache-create (&optional force)
  "构建 pinyin 到多音字的缓存，如果 FORCE 设置为 t, 强制更新索引。"
  (when (or force
            (not pyim-pymap--py2duoyinzi-cache1)
            (not pyim-pymap--py2duoyinzi-cache2))
    (setq pyim-pymap--py2duoyinzi-cache1
          (make-hash-table :size 50000 :test #'equal))
    (setq pyim-pymap--py2duoyinzi-cache2
          (make-hash-table :size 50000 :test #'equal))

    (dolist (x pyim-pymap-duoyinzi-chars)
      (let* ((py (car x))
             (chars (delete-dups
                     `(,@(cdr x) ,@(gethash py pyim-pymap--py2duoyinzi-cache1)))))
        (puthash py chars pyim-pymap--py2duoyinzi-cache1)))

    (dolist (x pyim-pymap-duoyinzi-words)
      (let* ((py (car x))
             (words (delete-dups
                     `(,@(cdr x) ,@(gethash py pyim-pymap--py2duoyinzi-cache2)))))
        (puthash py words pyim-pymap--py2duoyinzi-cache2)))))

(defun pyim-pymap-py2cchar-get (pinyin &optional equal-match return-list include-seperator)
  "获取拼音与 PINYIN 想匹配的所有汉字.

比如：

“man” -> (\"忙茫盲芒氓莽蟒邙漭硭\" \"满慢漫曼蛮馒瞒蔓颟谩墁幔螨鞔鳗缦熳镘\")

如果 EQUAL-MATCH 是 non-nil, 获取和 PINYIN 完全匹配的汉字。
如果 RETURN-LIST 是 non-nil, 返回一个由单个汉字字符串组成的列表。

(\"满\" \"慢\" \"漫\"  ...)

如果 INCLUDE-SEPERATOR 是 non-nil, 返回的列表包含一个 \"|\" 号，pyim 用这个分隔符
来区分 3500 个常用汉字和生僻字。"
  (pyim-pymap--py2cchar-cache-create)
  (when (and pinyin (stringp pinyin))
    (let ((output
           (if equal-match
               (if return-list
                   (gethash pinyin pyim-pymap--py2cchar-cache2)
                 (gethash pinyin pyim-pymap--py2cchar-cache1))
             (gethash pinyin pyim-pymap--py2cchar-cache3))))
      (setq output (remove "" output))
      (if include-seperator
          output
        (remove "|" output)))))

(defun pyim-pymap-str2py-get (string)
  "将 STRING 转换为拼音列表，转换过程中尽可能矫正多音字。

比如：
1. STRING:  你好
2. OUTPUTS: ((\"ni\" \"hao\"))

注意事项：
1. 这个函数遇到非汉字字符串时，原样输出。
2. 多音字矫正依赖 pymap 自带的多音字矫正信息的完善程度，可能会出
   现矫正不正确的情况，这个函数为了保证性能，只处理常用多音字。"
  (let* ((chars (pyim-pymap-split-string string t))
         (pinyins-list
          ;; ("Hello" "银" "行") -> (("Hello") ("yin") ("hang" "xing"))
          (mapcar (lambda (str)
                    (if (pyim-string-match-p "\\cc" str)
                        (pyim-pymap-cchar2py-get str)
                      (list str)))
                  chars)))
    ;; 通过排列组合的方式, 重排 pinyins-list。
    ;; 比如：(("Hello") ("yin") ("hang")) -> (("Hello" "yin" "hang"))
    (pyim-permutate-list
     (pyim-pymap--adjust-duoyinzi
      chars pinyins-list))))

(defun pyim-pymap-split-string (string &optional to-cchar)
  "将 STRING 按照中文处理的标准切开.

1. Hello你好 -> (\"Hello\" \"你\" \"好\"), when TO-CCHAR is non-nil.
2. Hello你好 -> (\"Hello\" \"你好\"), when TO-CCHAR is nil."
  (let* ((chars (split-string string ""))
         (sep 'pymap-separator)
         (chars-with-seps
          (pyim-pymap--add-seps-to-chars chars sep to-cchar))
         (chars-grouped
          (pyim-split-list chars-with-seps sep))
         (substrings-list
          (mapcar #'string-join chars-grouped)))
    (remove "" substrings-list)))

(defun pyim-pymap--add-seps-to-chars (chars separator to-cchar)
  "在 CCHRS 列表的中英文之间或者中文与中文之间插入 SEPARATOR 元素。

如果 SEPARATOR 设置为 mysep, 那么：
1. TO-CCHAR is nil:

   (\"a\" \"你\" \"好\" \"b\" \"c\")
=> (\"a\" mysep \"你\" \"好\" mysep \"b\" \"c\")

2. TO-CCHAR is non-nil:

   (\"a\" \"你\" \"好\" \"b\" \"c\")
=> (\"a\" mysep \"你\" mysep \"好\" mysep \"b\" \"c\")"
  (cl-mapcan (lambda (a b)
               (let ((a-cchar-p (pyim-pymap-cchar2py-get a))
                     (b-cchar-p (pyim-pymap-cchar2py-get b)))
                 (cond
                  ;; 如果 to-char 为 non-nil, 中文与中文之间插入一个
                  ;; sep, 否则不需要插入 sep.
                  ((and a-cchar-p b-cchar-p)
                   (if to-cchar
                       (list a separator)
                     (list a)))
                  ;; 非中文之间不需要插入一个 sep.
                  ((and (not a-cchar-p) (not b-cchar-p))
                   (list a))
                  ;; 中文和非中文之间需要插入一个 sep.
                  (t (list a separator)))))
             chars (cdr chars)))

(defun pyim-pymap-cchar2py-get (char-or-str)
  "获取字符或者字符串 CHAR-OR-STR 对应的拼音 code.

pyim 在特定的时候需要读取一个汉字的拼音，这个工作由此完成，函数
从 `pyim-pymap--cchar2py-cache' 查询得到一个汉字字符的拼音， 例如：

(pyim-pymap-cchar2py-get ?我)

结果为:

(\"wo\")"
  (pyim-pymap--cchar2py-cache-create)
  (let ((key (if (characterp char-or-str)
                 (char-to-string char-or-str)
               char-or-str)))
    (when (= (length key) 1)
      (gethash key pyim-pymap--cchar2py-cache))))

(defun pyim-pymap--adjust-duoyinzi (cchars-list pinyins-list)
  "根据 CCHARS-LIST 对 PINYINS-LIST 进行校正。

比如：

1. CCHARS-LIST:  (\"人\" \"民\" \"银\" \"行\")
2. PINYINS-LIST: ((\"ren\") (\"min\") (\"yin\") (\"hang\" \"xing\"))
3. 输出结果为：  ((\"ren\") (\"min\") (\"yin\") (\"hang\"))

这个函数依赖 `pyim-pymap-duoyinzi' 提供的多音字数据。"
  (let ((n (length pinyins-list))
        output)
    (dotimes (i n)
      (let ((pinyins (nth i pinyins-list))
            ;; 当前位置对应的汉字和位置前后汉字组成的两字词语。
            (words-list (list (when (>= (- i 1) 0)
                                (concat (nth (- i 1) cchars-list)
                                        (nth i cchars-list)))
                              (when (< (+ i 1) n)
                                (concat (nth i cchars-list)
                                        (nth (+ i 1) cchars-list)))))
            ;; 当前位置汉字
            (char-list (list (nth i cchars-list))))
        (if (= (length pinyins) 1)
            (push pinyins output)
          (let ((py-adjusted
                 (or
                  ;; NOTE: 多音字校正规则：
                  ;; 1. 首先通过 pyim 自带的多音字词语来校正，具体见：
                  ;; `pyim-pymap-duoyinzi-words'
                  (pyim-pymap--possible-cchar-pinyin pinyins words-list)
                  ;; 2. 然后通过 pyim 自带的多音字常用读音进行校正, 具体见：
                  ;; `pyim-pymap-duoyinzi-chars',
                  ;;
                  ;; NOTE: 如果用户想要使用某个汉字的偏僻读音，这样处理是有问题
                  ;; 的，但大多数情况我们还是使用汉字的常用读音，让偏僻的读音进
                  ;; 入用户个人词库似乎也没有什么好处。
                  (pyim-pymap--possible-cchar-pinyin pinyins char-list t))))
            ;; 3. 如果多音字校正没有结果，就使用未校正的信息。
            (push (if py-adjusted
                      (list py-adjusted)
                    pinyins)
                  output)))))
    (reverse output)))

(defun pyim-pymap--possible-cchar-pinyin (cchar-pinyins cchar-words &optional search-char)
  "寻找一个汉字当前最可能的读音。

以 (行) 作为例子：
1. PINYINS:     此汉字所有的读音组成的列表，比如: (xing hang)
2. WORDS:       此汉字本身或者和前后汉字组成的词语，比如: (银行 行业)
3. SEARCH-CHAR: 如果仅仅搜索汉字本身，就设置为 t, 此处设置为 nil.
4. 返回结果：   hang"
  (cl-find-if
   (lambda (pinyin)
     (when-let ((x (string-join
                    (pyim-pymap--py2duoyinzi-get
                     pinyin search-char)
                    "-")))
       (cl-some
        (lambda (word)
          (and word (string-match-p word x)))
        cchar-words)))
   cchar-pinyins))

(defun pyim-pymap--py2duoyinzi-get (pinyin &optional return-chars)
  "获取与 PINYIN 想匹配的多音字（词）。"
  (pyim-pymap--py2duoyinzi-cache-create)
  (when (and pinyin (stringp pinyin))
    (if return-chars
        (gethash pinyin pyim-pymap--py2duoyinzi-cache1)
      (gethash pinyin pyim-pymap--py2duoyinzi-cache2))))

(defun pyim-pymap-duoyinzi-include-p (cstring)
  "判断 CSTRING 中文字符串是否包含多音字。"
  (pyim-pymap--cchar2py-cache-create)
  (let ((chars (split-string cstring "")))
    (cl-some
     (lambda (char)
       (> (length (pyim-pymap-cchar2py-get char)) 1))
     chars)))

;; * Footer
(provide 'pyim-pymap)

;;; pyim-pymap.el ends here
