import numpy as np

# This file contains reference geometries, energies, and gradients
#  for comparing QC program results. It is placed in QCEngine but is also
#  used directly by QCDB and Psi4. New or filled-out systems+modelchems
#  welcome, but it's good to start a discussion on GitHub to check that
#  its downstream roles not impinged.

std_refs = ["rhf", "uhf", "rohf"]

std_molecules = {
    "hf": """
                H
                F 1 0.917
              """,
    # "bh3p": """
    #            1 2
    #            B     0.10369114     0.00000000     0.00000000
    #            H    -1.13269886     0.00000000     0.00000000
    #            H     3.00000000     0.37149000     0.00000000
    #            H     3.00000000    -0.37149000     0.00000000
    #          """,
    "bh3p": """
                1 2
                B     0.00000000     0.00000000     0.10369114
                H     0.00000000     0.00000000    -1.13269886
                H     0.00000000    -0.37149000     3.00000000
                H     0.00000000     0.37149000     3.00000000
              """,
    "h2o": """
                O
                H 1 R
                H 1 R 2 A

                R=0.958
                A=104.5
              """,
    "nh2": """
                0 2
                N
                H 1 R
                H 1 R 2 A

                R=1.008
                A=105.0
              """,
    "h2o-xyz": """
 # R=0.958 A=104.5
 O                  0.000000000000     0.000000000000    -0.124038860300
 H                  0.000000000000    -1.431430901356     0.984293362719
 H                  0.000000000000     1.431430901356     0.984293362719
 units au
              """,
    "nh2-xyz": """
 # R=1.008 #A=105.0
 0 2
 N   0.000000000000000   0.000000000000000  -0.145912918634892
 H   0.000000000000000  -1.511214298139000   1.013682596946108
 H   0.000000000000000   1.511214298139000   1.013682596946108
 units au
              """,
    "hf-xyz": """
        H    0.          0.         -1.64558411
        F    0.          0.          0.08729475
        units bohr
               """,
}
std_molecules["bh3p-xyz"] = std_molecules["bh3p"]

_std_generics = {
    "hf_cc-pvdz_ae": (19, 19, 5, 5),
    "hf_cc-pvdz_fc": (19, 19, 5, 5),
    "bh3p_cc-pvdz_ae": (29, 29, 4, 3),
    "bh3p_cc-pvdz_fc": (29, 29, 4, 3),
    "h2o_aug-cc-pvdz_ae": (41, 41, 5, 5),
    "h2o_aug-cc-pvdz_fc": (41, 41, 5, 5),
    "nh2_aug-cc-pvdz_ae": (41, 41, 5, 4),
    "nh2_aug-cc-pvdz_fc": (41, 41, 5, 4),
    "h2o_cfour-qz2p_ae": (48, 48, 5, 5),
    "h2o_cfour-qz2p_fc": (48, 48, 5, 5),
    "nh2_cfour-qz2p_ae": (48, 48, 5, 4),
    "nh2_cfour-qz2p_fc": (48, 48, 5, 4),
}
_std_generics = {
    k: dict(zip(["N BASIS FUNCTIONS", "N MOLECULAR ORBITALS", "N ALPHA ELECTRONS", "N BETA ELECTRONS"], v))
    for k, v in _std_generics.items()
}

_scf_hf_dz_pk_rhf = -100.01941126902270
_scf_bh3p_dz_pk_uhf = -25.94513842869638
_scf_bh3p_dz_pk_rohf = -25.943614318546

_scf_hf_dz_df_rhf = -100.019400605629
_scf_bh3p_dz_df_uhf = -25.945130559147
_scf_bh3p_dz_df_rohf = -25.943606522029

_scf_hf_dz_cd_rhf = -100.01939270219628
_scf_bh3p_dz_cd_uhf = -25.94511891510799
_scf_bh3p_dz_cd_rohf = -25.943595251664313


_scf_h2o_qz2p_pk_rhf = -76.0627484601
_scf_nh2_qz2p_pk_uhf = -55.5893469688
_scf_nh2_qz2p_pk_rohf = -55.5847372601

_scf_h2o_qz2p_df_rhf = -76.06274142753659
_scf_nh2_qz2p_df_uhf = -55.58934323208402
_scf_nh2_qz2p_df_rohf = -55.58473319013903

_scf_h2o_qz2p_cd_rhf = -76.06277445978574
_scf_nh2_qz2p_cd_uhf = -55.58934916135869
_scf_nh2_qz2p_cd_rohf = -55.58473942870229


_scf_h2o_adz_pk_rhf = -76.0413815332
_scf_nh2_adz_pk_uhf = -55.57513805247548
_scf_nh2_adz_pk_rohf = -55.570724348574

_scf_h2o_adz_df_rhf = -76.04136132628614
_scf_nh2_adz_df_uhf = -55.57512538464817
_scf_nh2_adz_df_rohf = -55.57071142443952

_scf_h2o_adz_cd_rhf = -76.04132169763341
_scf_nh2_adz_cd_uhf = -55.57506886675886
_scf_nh2_adz_cd_rohf = -55.57065536578708


# fmt: off
_grad_scf_hf_dz_pk_rhf = np.array([0.0, 0.0, -0.019789748774, -0.0, -0.0, 0.019789748774]).reshape((-1, 3))
_grad_scf_bh3p_dz_pk_uhf = np.array([ 0.0, 0.0, 0.018283596898, 0.0, 0.0, -0.016914487538, -0.0, 0.004869782481, -0.00068455468, 0.0, -0.004869782481, -0.00068455468,]).reshape((-1, 3))
_grad_scf_bh3p_dz_pk_rohf = np.array([ 0.0, 0.0, 0.022062218595, 0.0, 0.0, -0.02050835054, -0.0, 0.004706239974, -0.000776934027, 0.0, -0.004706239974, -0.000776934027,]).reshape((-1, 3))

_grad_scf_hf_dz_df_rhf = np.array([0.0, -0.0, -0.0197887676, -0.0, 0.0, 0.0197887676]).reshape((-1, 3))
_grad_scf_bh3p_dz_df_uhf = np.array([-0.0, 0.0, 0.018279955719, 0.0, 0.0, -0.016906923007, -0.0, 0.004865311963, -0.000686516356, 0.0, -0.004865311963, -0.000686516356,]).reshape((-1, 3))
_grad_scf_bh3p_dz_df_rohf = np.array([-0.0, 0.0, 0.022058469913, 0.0, 0.0, -0.020500364247, -0.0, 0.004701465609, -0.000779052833, 0.0, -0.004701465609, -0.000779052833,]).reshape((-1, 3))

_grad_scf_hf_dz_cd_rhf = np.array([0.0, 0.0, -0.019796742926, 0.0, 0.0, 0.019796742926]).reshape((-1, 3))
_grad_scf_bh3p_dz_cd_uhf = np.array([ 0.0, 0.0, 0.018280917408, 0.0, 0.0, -0.016907239597, 0.0, 0.004874331571, -0.000686838905, 0.0, -0.004874331571, -0.000686838905,]).reshape((-1, 3))
_grad_scf_bh3p_dz_cd_rohf = np.array([ 0.0, 0.0, 0.022058379283, 0.0, 0.0, -0.020499895346, 0.0, 0.004710759277, -0.000779241969, 0.0, -0.004710759277, -0.000779241969,]).reshape((-1, 3))


_grad_scf_h2o_qz2p_pk_rhf = np.array([-0.0, 0.0, -0.026426043127, 0.0, -0.013943334328, 0.013213021564, -0.0, 0.013943334328, 0.013213021564]).reshape((-1, 3))
_grad_scf_nh2_qz2p_pk_uhf = np.array([0.0, 0.0, 0.001550839727, -0.0, 0.000092404341, -0.000775419863, 0.0, -0.000092404341, -0.000775419863]).reshape((-1, 3))
_grad_scf_nh2_qz2p_pk_rohf = np.array([-0.0, 0.0, 0.000683766715, 0.0, -0.000526016129, -0.000341883358, -0.0, 0.000526016129, -0.000341883358]).reshape((-1, 3))

_grad_scf_h2o_qz2p_df_rhf = np.array([0.0, 0.0, -0.026454963718, 0.0, -0.013950706431, 0.013227481859, -0.0, 0.013950706431, 0.013227481859]).reshape((-1, 3))
_grad_scf_nh2_qz2p_df_uhf = np.array([-0.0, 0.0, 0.001532526876, -0.0, 0.000082065307, -0.000766263438, 0.0, -0.000082065307, -0.000766263438]).reshape((-1, 3))
_grad_scf_nh2_qz2p_df_rohf = np.array([0.0, 0.0, 0.000665139571, 0.0, -0.000536617267, -0.000332569786, -0.0, 0.000536617267, -0.000332569786]).reshape((-1, 3))

_grad_scf_h2o_qz2p_cd_rhf = np.array([0.0, 0.0, -0.026440883169, 0.0, -0.013949099753, 0.013220441584, 0.0, 0.013949099753, 0.013220441584]).reshape((-1, 3))
_grad_scf_nh2_qz2p_cd_uhf = np.array([0.0, 0.0, 0.001549105775, 0.0, 0.000091952809, -0.000774552887, 0.0, -0.000091952809, -0.000774552887]).reshape((-1, 3))
_grad_scf_nh2_qz2p_cd_rohf = np.array([0.0, 0.0, 0.000681888827, 0.0, -0.000526508263, -0.000340944414, 0.0, 0.000526508263, -0.000340944414]).reshape((-1, 3))


_grad_scf_h2o_adz_pk_rhf = np.array([-0.0, 0.0, -0.021391822417, 0.0, -0.011259127342, 0.010695911208, -0.0, 0.011259127342, 0.010695911208]).reshape((-1, 3))
_grad_scf_nh2_adz_pk_uhf = np.array([-0.0, 0.0, 0.008498123792, -0.0, 0.004518040488, -0.004249061896, 0.0, -0.004518040488, -0.004249061896]).reshape((-1, 3))
_grad_scf_nh2_adz_pk_rohf = np.array([-0.0, 0.0, 0.007678819724, -0.0, 0.003856927713, -0.003839409862, 0.0, -0.003856927713, -0.003839409862]).reshape((-1, 3))

_grad_scf_h2o_adz_df_rhf = np.array([0.0, 0.0, -0.021402714298, 0.0, -0.011255158091, 0.010701357149, -0.0, 0.011255158091, 0.010701357149]).reshape((-1, 3))
_grad_scf_nh2_adz_df_uhf = np.array([0.0, 0.0, 0.00848503512, -0.0, 0.004519001873, -0.00424251756, 0.0, -0.004519001873, -0.00424251756]).reshape((-1, 3))
_grad_scf_nh2_adz_df_rohf = np.array([0.0, 0.0, 0.007664957774, -0.0, 0.003857593701, -0.003832478887, 0.0, -0.003857593701, -0.003832478887]).reshape((-1, 3))

_grad_scf_h2o_adz_cd_rhf = np.array([0.0, 0.0, -0.021374103228, 0.0, -0.011265854947, 0.010687051614, 0.0, 0.011265854947, 0.010687051614]).reshape((-1, 3))
_grad_scf_nh2_adz_cd_uhf = np.array([0.0, 0.0, 0.00848026807, 0.0, 0.004537635752, -0.004240134035, 0.0, -0.004537635752, -0.004240134035]).reshape((-1, 3))
_grad_scf_nh2_adz_cd_rohf = np.array([0.0, 0.0, 0.007661397585, 0.0, 0.003876803846, -0.003830698792, 0.0, -0.003876803846, -0.003830698792]).reshape((-1, 3))
# fmt: on


# fmt: off
_hess_scf_hf_dz_pk_rhf = np.array([
    0.011420157, 0., 0., -0.011420157, 0., 0.,
    0., 0.011420157, 0., 0., -0.011420157, 0.,
    0., 0., 0.6371334086, 0., 0., -0.6371334086,
    -0.011420157, 0., 0., 0.011420157, 0., 0.,
    0., -0.011420157, 0., 0., 0.011420157, 0.,
    0., 0., -0.6371334086, 0., 0., 0.6371334086]).reshape((6, 6))
_hess_scf_bh3p_dz_pk_uhf = np.array([
    0.0051661616, 0., 0., -0.0059617366, 0., 0., 0.0003977875, 0., 0., 0.0003977875, 0., 0.,
    0., 0.005028448, 0., 0., -0.0057802433, 0., 0., 0.0003758976, 0.0004726863, 0., 0.0003758976, -0.0004726863,
    0., 0., 0.2162990046, 0., 0., -0.2130428571, 0., 0.0001869025, -0.0016280737, 0., -0.0001869025, -0.0016280737,
    -0.0059617366, 0., 0., 0.0063439881, 0., 0., -0.0001911257, 0., 0., -0.0001911257, 0., 0.,
    0., -0.0057802433, 0., 0., 0.0062399336, 0., 0., -0.0002298451, 0.0001287182, 0., -0.0002298451, -0.0001287182,
    0., 0., -0.2130428571, 0., 0., 0.213169144, 0., 0.000042202, -0.0000631434, 0., -0.000042202, -0.0000631434,
    0.0003977875, 0., 0., -0.0001911257, 0., 0., -0.0035717661, 0., 0., 0.0033651042, 0., 0.,
    0., 0.0003758976, 0.0001869025, 0., -0.0002298451, 0.000042202, 0., 0.4101678639, -0.0004152545, 0., -0.4103139164, 0.00018615,
    0., 0.0004726863, -0.0016280737, 0., 0.0001287182, -0.0000631434, 0., -0.0004152545, -0.0000642125, 0., -0.00018615, 0.0017554296,
    0.0003977875, 0., 0., -0.0001911257, 0., 0., 0.0033651042, 0., 0., -0.0035717661, 0., 0.,
    0., 0.0003758976, -0.0001869025, 0., -0.0002298451, -0.000042202, 0., -0.4103139164, -0.00018615, 0., 0.4101678639, 0.0004152545,
    0., -0.0004726863, -0.0016280737, 0., -0.0001287182, -0.0000631434, 0., 0.00018615, 0.0017554296, 0., 0.0004152545, -0.0000642125]).reshape((12, 12))
_hess_scf_bh3p_dz_pk_rohf = np.array([
    0.0065592076, 0., 0., -0.0074218605, 0., 0., 0.0004313264, 0., 0., 0.0004313264, 0., 0.,
    0., 0.006418851, 0., 0., -0.0072356386, 0., 0., 0.0004083938, 0.0004886847, 0., 0.0004083938, -0.0004886847,
    0., 0., 0.2263725717, 0., 0., -0.2227777355, 0., -0.0000454089, -0.0017974181, 0., 0.0000454089, -0.0017974181,
    -0.0074218605, 0., 0., 0.0078274642, 0., 0., -0.0002028018, 0., 0., -0.0002028018, 0., 0.,
    0., -0.0072356386, 0., 0., 0.007720208, 0., 0., -0.0002422847, 0.0001293422, 0., -0.0002422847, -0.0001293422,
    0., 0., -0.2227777355, 0., 0., 0.2229077847, 0., -0.0000149119, -0.0000650246, 0., 0.0000149119, -0.0000650246,
    0.0004313264, 0., 0., -0.0002028018, 0., 0., -0.003466217, 0., 0., 0.0032376924, 0., 0.,
    0., 0.0004083938, -0.0000454089, 0., -0.0002422847, -0.0000149119, 0., 0.4104056572, -0.0002788531, 0., -0.4105717664, 0.0003391738,
    0., 0.0004886847, -0.0017974181, 0., 0.0001293422, -0.0000650246, 0., -0.0002788531, 0.0002037173, 0., -0.0003391738, 0.0016587254,
    0.0004313264, 0., 0., -0.0002028018, 0., 0., 0.0032376924, 0., 0., -0.003466217, 0., 0.,
    0., 0.0004083938, 0.0000454089, 0., -0.0002422847, 0.0000149119, 0., -0.4105717664, -0.0003391738, 0., 0.4104056572, 0.0002788531,
    0., -0.0004886847, -0.0017974181, 0., -0.0001293422, -0.0000650246, 0., 0.0003391738, 0.0016587254, 0., 0.0002788531, 0.0002037173]).reshape((12, 12))

_hess_scf_hf_dz_df_rhf = np.zeros(36).reshape((6, 6))
_hess_scf_bh3p_dz_df_uhf = np.zeros(144).reshape((12, 12))
_hess_scf_bh3p_dz_df_rohf = np.zeros(144).reshape((12, 12))

_hess_scf_hf_dz_cd_rhf = np.zeros(36).reshape((6, 6))
_hess_scf_bh3p_dz_cd_uhf = np.zeros(144).reshape((12, 12))
_hess_scf_bh3p_dz_cd_rohf = np.zeros(144).reshape((12, 12))


_hess_scf_h2o_qz2p_pk_rhf = np.array([
    0.0238430708, 0., 0., -0.0119215354, 0., 0., -0.0119215354, 0., 0.,
    0., 0.7093110919, 0., 0., -0.3546555459, 0.2653730246, 0., -0.3546555459, -0.2653730246,
    0., 0., 0.4985784094, 0., 0.2027912096, -0.2492892047, 0., -0.2027912096, -0.2492892047,
    -0.0119215354, 0., 0., 0.0108311859, 0., 0., 0.0010903495, 0., 0.,
    0., -0.3546555459, 0.2027912096, 0., 0.3886078743, -0.2340821171, 0., -0.0339523283, 0.0312909075,
    0., 0.2653730246, -0.2492892047, 0., -0.2340821171, 0.2322518953, 0., -0.0312909075, 0.0170373094,
    -0.0119215354, 0., 0., 0.0010903495, 0., 0., 0.0108311859, 0., 0.,
    0., -0.3546555459, -0.2027912096, 0., -0.0339523283, -0.0312909075, 0., 0.3886078743, 0.2340821171,
    0., -0.2653730246, -0.2492892047, 0., 0.0312909075, 0.0170373094, 0., 0.2340821171, 0.2322518953]).reshape((9, 9))
_hess_scf_nh2_qz2p_pk_uhf=np.array([
    -0.0013373967, 0., 0., 0.0006686983, 0., 0., 0.0006686983, 0., 0.,
    0., 0.6034171677, 0., 0., -0.3017085838, 0.2320222491, 0., -0.3017085838, -0.2320222491,
    0., 0., 0.4139303729, 0., 0.1753720372, -0.2069651864, 0., -0.1753720372, -0.2069651864,
    0.0006686983, 0., 0., -0.000364922, 0., 0., -0.0003037763, 0., 0.,
    0., -0.3017085838, 0.1753720372, 0., 0.3327449782, -0.2036971432, 0., -0.0310363944, 0.0283251059,
    0., 0.2320222491, -0.2069651864, 0., -0.2036971432, 0.1924704871, 0., -0.0283251059, 0.0144946993,
    0.0006686983, 0., 0., -0.0003037763, 0., 0., -0.000364922, 0., 0.,
    0., -0.3017085838, -0.1753720372, 0., -0.0310363944, -0.0283251059, 0., 0.3327449782, 0.2036971432,
    0., -0.2320222491, -0.2069651864, 0., 0.0283251059, 0.0144946993, 0., 0.2036971432, 0.1924704871]).reshape((9, 9))
_hess_scf_nh2_qz2p_pk_rohf=np.array([
    -0.0005896607, 0., 0., 0.0002948303, 0., 0., 0.0002948303, 0., 0.,
    0., 0.6051456253, 0., 0., -0.3025728127, 0.2323985163, 0., -0.3025728127, -0.2323985163,
    0., 0., 0.4152409466, 0., 0.1756925779, -0.2076204733, 0., -0.1756925779, -0.2076204733,
    0.0002948303, 0., 0., 0.0000266223, 0., 0., -0.0003214527, 0., 0.,
    0., -0.3025728127, 0.1756925779, 0., 0.3336853184, -0.2040455471, 0., -0.0311125058, 0.0283529691,
    0., 0.2323985163, -0.2076204733, 0., -0.2040455471, 0.1931471008, 0., -0.0283529691, 0.0144733724,
    0.0002948303, 0., 0., -0.0003214527, 0., 0., 0.0000266223, 0., 0.,
    0., -0.3025728127, -0.1756925779, 0., -0.0311125058, -0.0283529691, 0., 0.3336853184, 0.2040455471,
    0., -0.2323985163, -0.2076204733, 0., 0.0283529691, 0.0144733724, 0., 0.2040455471, 0.1931471008]).reshape((9, 9))

_hess_scf_h2o_qz2p_df_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_df_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_df_rohf = np.zeros(81).reshape((9, 9))

_hess_scf_h2o_qz2p_cd_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_cd_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_qz2p_cd_rohf = np.zeros(81).reshape((9, 9))


_hess_scf_h2o_adz_pk_rhf = np.array([
    0.0193009116, 0., 0., -0.0096504557, 0., 0., -0.0096504557, 0., 0.,
    0., 0.7259570062, 0., 0., -0.362978503, 0.2735758043, 0., -0.362978503, -0.2735758043,
    0., 0., 0.507233142, 0., 0.2101340878, -0.253616571, 0., -0.2101340878, -0.253616571,
    -0.0096504557, 0., 0., 0.0087580503, 0., 0., 0.0008924054, 0., 0.,
    0., -0.362978503, 0.2101340878, 0., 0.3955011984, -0.2418549461, 0., -0.0325226952, 0.0317208583,
    0., 0.2735758043, -0.253616571, 0., -0.2418549461, 0.236653618, 0., -0.0317208583, 0.016962953,
    -0.0096504557, 0., 0., 0.0008924054, 0., 0., 0.0087580503, 0., 0.,
    0., -0.362978503, -0.2101340878, 0., -0.0325226952, -0.0317208583, 0., 0.3955011984, 0.2418549461,
    0., -0.2735758043, -0.253616571, 0., 0.0317208583, 0.016962953, 0., 0.2418549461, 0.236653618 ]).reshape((9, 9))
_hess_scf_nh2_adz_pk_uhf = np.array([
    -0.0073285241, 0., 0., 0.0036642621, 0., 0., 0.0036642621, 0., 0.,
    0., 0.6248676289, 0., 0., -0.3124338145, 0.2425505849, 0., -0.3124338145, -0.2425505849,
    0., 0., 0.4213971779, 0., 0.185122888, -0.210698589, 0., -0.185122888, -0.210698589,
    0.0036642621, 0., 0., -0.0033269688, 0., 0., -0.0003372933, 0., 0.,
    0., -0.3124338145, 0.185122888, 0., 0.3427670403, -0.2138367365, 0., -0.0303332259, 0.0287138485,
    0., 0.2425505849, -0.210698589, 0., -0.2138367365, 0.1969122616, 0., -0.0287138485, 0.0137863273,
    0.0036642621, 0., 0., -0.0003372933, 0., 0., -0.0033269688, 0., 0.,
    0., -0.3124338145, -0.185122888, 0., -0.0303332259, -0.0287138485, 0., 0.3427670403, 0.2138367365,
    0., -0.2425505849, -0.210698589, 0., 0.0287138485, 0.0137863273, 0., 0.2138367365, 0.1969122616]).reshape((9, 9))
_hess_scf_nh2_adz_pk_rohf = np.array([
    -0.006621982, 0., 0., 0.003310991, 0., 0., 0.003310991, 0., 0.,
    0., 0.6266642672, 0., 0., -0.3133321336, 0.2429688151, 0., -0.3133321336, -0.2429688151,
    0., 0., 0.4229135704, 0., 0.1853731231, -0.2114567853, 0., -0.1853731231, -0.2114567853,
    0.003310991, 0., 0., -0.0029315977, 0., 0., -0.0003793933, 0., 0.,
    0., -0.3133321336, 0.1853731231, 0., 0.3437839161, -0.2141709691, 0., -0.0304517825, 0.028797846,
    0., 0.2429688151, -0.2114567853, 0., -0.2141709691, 0.1976705549, 0., -0.028797846, 0.0137862303,
    0.003310991, 0., 0., -0.0003793933, 0., 0., -0.0029315977, 0., 0.,
    0., -0.3133321336, -0.1853731231, 0., -0.0304517825, -0.028797846, 0., 0.3437839161, 0.2141709691,
    0., -0.2429688151, -0.2114567853, 0., 0.028797846, 0.0137862303, 0., 0.2141709691, 0.1976705549]).reshape((9, 9))

_hess_scf_h2o_adz_df_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_df_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_df_rohf = np.zeros(81).reshape((9, 9))

_hess_scf_h2o_adz_cd_rhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_cd_uhf = np.zeros(81).reshape((9, 9))
_hess_scf_nh2_adz_cd_rohf = np.zeros(81).reshape((9, 9))
# fmt: on


_std_suite = [
    # <<<  CONV-AE-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.203781911950,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427697023782003,
            "MP2 TOTAL GRADIENT": np.array(
                [  # fnocc findif-5 ae pk+conv
                    0.0000000000,
                    0.0000000000,
                    0.0028193375,
                    0.0000000000,
                    0.0000000000,
                    -0.0028193375,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.00162697, 0.0, 0.0, 0.00162697, 0.0, 0.0],
                    [0.0, -0.00162697, 0.0, 0.0, 0.00162697, 0.0],
                    [0.0, 0.0, 0.64124876, 0.0, 0.0, -0.64124876],
                    [0.00162697, 0.0, 0.0, -0.00162697, 0.0, 0.0],
                    [0.0, 0.00162697, 0.0, 0.0, -0.00162697, 0.0],
                    [0.0, 0.0, -0.64124876, 0.0, 0.0, 0.64124876],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.20647996,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04989037,  # fnocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000917944631, 0.0, 0.0, 0.000917944631]).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.20864512,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.002048407683,  # vcc
            "CISD CORRELATION ENERGY": -0.20226573645347,  # vcc
            "QCISD CORRELATION ENERGY": -0.20892771089382,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00182210,  # vcc
            "FCI CORRELATION ENERGY": -0.21117389325,  # detci
            "LCCD CORRELATION ENERGY": -0.2099060277,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048339903547,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, -0.0, 0.002203664242, -0.0, 0.0, -0.002203664242]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107436391,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048460183760,  # fnocc
            "CCD CORRELATION ENERGY": -0.20815176752849,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04851867,  # vcc
            "CCD TOTAL GRADIENT": np.array([[0.0, 0.0, 0.0008368526], [0.0, 0.0, -0.0008368526]]),  # vcc
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-4.82892500e-04, 0.00000000e00, 0.00000000e00, 4.82892500e-04, 0.00000000e00, 0.00000000e00],
                    [0.00000000e00, -4.82892500e-04, 0.00000000e00, 0.00000000e00, 4.82892500e-04, 0.00000000e00],
                    [0.00000000e00, 0.00000000e00, 6.40299636e-01, 0.00000000e00, 0.00000000e00, -6.40299636e-01],
                    [4.82892500e-04, 0.00000000e00, 0.00000000e00, -4.82892500e-04, 0.00000000e00, 0.00000000e00],
                    [0.00000000e00, 4.82892500e-04, 0.00000000e00, 0.00000000e00, -4.82892500e-04, 0.00000000e00],
                    [0.00000000e00, 0.00000000e00, -6.40299636e-01, 0.00000000e00, 0.00000000e00, 6.40299636e-01],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.208743643,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857419039,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.001989217717,
                    0.0,
                    0.0,
                    -0.001989217717,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.00114793, 0.0, 0.0, 0.00114793, 0.0, 0.0],
                    [0.0, -0.00114793, 0.0, 0.0, 0.00114793, 0.0],
                    [0.0, 0.0, 0.6378403, 0.0, 0.0, -0.6378403],
                    [0.00114793, 0.0, 0.0, -0.00114793, 0.0, 0.0],
                    [0.0, 0.00114793, 0.0, 0.0, -0.00114793, 0.0],
                    [0.0, 0.0, -0.6378403, 0.0, 0.0, 0.6378403],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.002058455537,  # vcc
            "(T) CORRECTION ENERGY": -0.0019363896542312043,
            "CCSD(T) TOTAL GRADIENT": np.array([[0.0, 0.0, 0.00308437284379], [0.0, 0.0, -0.00308437284379]]),  # ncc fd
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [-0.00177990234, 0.0, 0.0, 0.00177990234, 0.0, 0.0],
                    [0.0, -0.00177990234, 0.0, 0.0, 0.00177990234, 0.0],
                    [0.0, 0.0, 0.636483940298, 0.0, 0.0, -0.636483940298],
                    [0.00177990234, 0.0, 0.0, -0.00177990234, 0.0, 0.0],
                    [0.0, 0.00177990234, 0.0, 0.0, -0.00177990234, 0.0],
                    [0.0, 0.0, -0.636483940298, 0.0, 0.0, 0.636483940298],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.001961066509136,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, 0.003071874425935],
                    [0.000000000000000, 0.000000000000000, -0.003071874425935],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.210768179766,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00316187041423], [0.0, 0.0, -0.00316187041423]]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001824798014, 0.0, 0.0, 0.001824798014, 0.0, 0.0],
                    [0.0, -0.001824795113, 0.0, 0.0, 0.001824795113, 0.0],
                    [0.0, 0.0, 0.63641316654, 0.0, 0.0, -0.63641316654],
                    [0.001824798014, 0.0, 0.0, -0.001824798014, 0.0, 0.0],
                    [0.0, 0.001824795113, 0.0, 0.0, -0.001824795113, 0.0],
                    [0.0, 0.0, -0.63641316654, 0.0, 0.0, 0.63641316654],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.210771725787,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00316311457309], [0.0, 0.0, -0.00316311457309]]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001825508871, 0.000000011606, 0.000000016683, 0.001825508871, -0.000000011606, -0.000000016683],
                    [0.000000011606, -0.001825503068, 0.000000007254, -0.000000011606, 0.001825503068, -0.000000007254],
                    [0.000000016683, 0.000000007254, 0.636412451331, -0.000000016683, -0.000000007254, -0.636412451331],
                    [0.001825508871, -0.000000011606, -0.000000016683, -0.001825508871, 0.000000011606, 0.000000016683],
                    [-0.000000011606, 0.001825503068, -0.000000007254, 0.000000011606, -0.001825503068, 0.000000007254],
                    [-0.000000016683, -0.000000007254, -0.636412451331, 0.000000016683, 0.000000007254, 0.636412451331],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.210638072036,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[-0.0, 0.0, 0.00310773649518], [0.0, 0.0, -0.00310773649518]]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.001793373798, -0.000000001451, -0.000000029015, 0.001793373798, 0.000000001451, 0.000000029015],
                    [-0.000000001451, -0.001793373798, -0.000000029015, 0.000000001451, 0.001793373798, 0.000000029015],
                    [-0.000000029015, -0.000000029015, 0.636438446923, 0.000000029015, 0.000000029015, -0.636438446923],
                    [0.001793373798, 0.000000001451, 0.000000029015, -0.001793373798, -0.000000001451, -0.000000029015],
                    [0.000000001451, 0.001793373798, 0.000000029015, -0.000000001451, -0.001793373798, -0.000000029015],
                    [0.000000029015, 0.000000029015, -0.636438446923, -0.000000029015, -0.000000029015, 0.636438446923],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.21063680423440,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00310517589721], [0.0, 0.0, -0.00310517589721]]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00179205, 0.0, 0.0, 0.00179205, 0.0, 0.0],
                    [0.0, -0.00179205, 0.0, 0.0, 0.00179205, 0.0],
                    [0.0, 0.0, 0.63647126, 0.0, 0.0, -0.63647126],
                    [0.00179205, 0.0, 0.0, -0.00179205, 0.0, 0.0],
                    [0.0, 0.00179205, 0.0, 0.0, -0.00179205, 0.0],
                    [0.0, 0.0, -0.63647126, 0.0, 0.0, 0.63647126],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.210768503167476,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.003174888578646,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003174888578646,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00183215, 0.0, 0.0, 0.00183215, 0.0, 0.0],
                    [0.0, -0.00183215, 0.0, 0.0, 0.00183215, 0.0],
                    [0.0, 0.0, 0.63633545, 0.0, 0.0, -0.63633545],
                    [0.00183215, 0.0, 0.0, -0.00183215, 0.0, 0.0],
                    [0.0, 0.00183215, 0.0, 0.0, -0.00183215, 0.0],
                    [0.0, 0.0, -0.63633545, 0.0, 0.0, 0.63633545],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000380822242155,  # ncc
            "(Q) CORRECTION ENERGY": -0.000414550980182,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [  # ncc
                    0.000000000000000,
                    0.000000000000000,
                    0.003539304813171,
                    0.000000000000000,
                    0.000000000000000,
                    -0.00353930481317,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.211162705496338,  # ncc
            "CCSDTQ TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.003507320893113,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003507320893113,
                ]
            ).reshape((-1, 3)),
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0005522939,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2104417743,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0484443079,  # occ
            "PBE TOTAL ENERGY": -100.33517315116806,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array([[0.0, 0.0, 0.020107103338], [0.0, 0.0, -0.020107128125]]),  # psi 99,590
            "B3LYP TOTAL ENERGY": -100.43544624005466,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [[-2.37578116e-16, -5.15211396e-17, 1.13731953e-02], [-5.23661679e-14, 5.41065792e-15, -1.13730866e-02]]
            ),
            "B3LYP5 TOTAL ENERGY": -100.39768534808519,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [[-6.94208991e-16, 1.87550833e-16, 1.16101107e-02], [3.49213371e-14, 3.53945060e-15, -1.16100020e-02]]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2218977246,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05669988343022163,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009624481085,
                    0.0,
                    0.005505796371,
                    -0.004812240542,
                    0.0,
                    -0.005505796371,
                    -0.004812240542,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -8.68375090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.18925148e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.59462574e-01,
                        2.81687501e-01,
                        0.00000000e00,
                        -3.59462574e-01,
                        -2.81687501e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.67076274e-01,
                        0.00000000e00,
                        2.16194016e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        -2.16194016e-01,
                        -2.33538137e-01,
                    ],
                    [
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.09411690e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.47758500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.59462574e-01,
                        2.16194016e-01,
                        0.00000000e00,
                        3.87745363e-01,
                        -2.48940759e-01,
                        0.00000000e00,
                        -2.82827890e-02,
                        3.27467430e-02,
                    ],
                    [
                        0.00000000e00,
                        2.81687501e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        -2.48940759e-01,
                        2.23898773e-01,
                        0.00000000e00,
                        -3.27467430e-02,
                        9.63936430e-03,
                    ],
                    [
                        4.34187540e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.47758500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.09411690e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.59462574e-01,
                        -2.16194016e-01,
                        0.00000000e00,
                        -2.82827890e-02,
                        -3.27467430e-02,
                        0.00000000e00,
                        3.87745363e-01,
                        2.48940759e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.81687501e-01,
                        -2.33538137e-01,
                        0.00000000e00,
                        3.27467430e-02,
                        9.63936430e-03,
                        0.00000000e00,
                        2.48940759e-01,
                        2.23898773e-01,
                    ],
                ]
            ),  # cfour
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.051851951707,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.226431141430,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.003997280108, 0.0, 0.00255475829, -0.001998640054, 0.0, -0.00255475829, -0.001998640054]
            ).reshape(-1, 3),
            "MP4(SDQ) CORRELATION ENERGY": -0.229617624405,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.005637388222,  # vcc
            "CISD CORRELATION ENERGY": -0.21978965712829,  # vcc
            "QCISD CORRELATION ENERGY": -0.22998871354660,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.0048836279,  # vcc
            "LCCD CORRELATION ENERGY": -0.2318870702,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.049937236558,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.008553750217, -0.0, 0.005238229687, -0.004276875109, 0.0, -0.005238229687, -0.004276875109]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2341051403,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.050442387759,  # fnocc
            "CCD CORRELATION ENERGY": -0.22802083552559,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0499178,  # vcc
            "CCD TOTAL GRADIENT": np.array(
                [[0.0, 0.0, 0.00617203], [0.0, 0.00392094, -0.00308601], [0.0, -0.00392094, -0.00308601]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.0055683485, 0.0, 0.0, 0.0027841742, 0.0, 0.0, 0.0027841742, 0.0, 0.0],
                    [0.0, 0.7149628559, 0.0, 0.0, -0.357481428, 0.2789479625, 0.0, -0.357481428, -0.2789479625],
                    [0.0, 0.0, 0.4751691519, 0.0, 0.215124084, -0.237584576, 0.0, -0.215124084, -0.237584576],
                    [0.0027841742, 0.0, 0.0, -0.0027615491, 0.0, 0.0, -0.0000226252, 0.0, 0.0],
                    [
                        0.0,
                        -0.357481428,
                        0.215124084,
                        0.0,
                        0.3876970703,
                        -0.2470360126,
                        0.0,
                        -0.0302156423,
                        0.0319119498,
                    ],
                    [
                        0.0,
                        0.2789479625,
                        -0.237584576,
                        0.0,
                        -0.2470360126,
                        0.2254150323,
                        0.0,
                        -0.0319119498,
                        0.0121695437,
                    ],
                    [0.0027841742, 0.0, 0.0, -0.0000226252, 0.0, 0.0, -0.0027615491, 0.0, 0.0],
                    [
                        0.0,
                        -0.357481428,
                        -0.215124084,
                        0.0,
                        -0.0302156423,
                        -0.0319119498,
                        0.0,
                        0.3876970703,
                        0.2470360126,
                    ],
                    [
                        0.0,
                        -0.2789479625,
                        -0.237584576,
                        0.0,
                        0.0319119498,
                        0.0121695437,
                        0.0,
                        0.2470360126,
                        0.2254150323,
                    ],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.2294105794,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.050177977945205,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.007512595487,
                    0.0,
                    0.004613769715,
                    -0.003756297743,
                    0.0,
                    -0.004613769715,
                    -0.003756297743,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        -6.77828820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.12577187e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.56288594e-01,
                        2.78492260e-01,
                        0.00000000e00,
                        -3.56288594e-01,
                        -2.78492260e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.72789375e-01,
                        0.00000000e00,
                        2.14574581e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        -2.14574581e-01,
                        -2.36394688e-01,
                    ],
                    [
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.30616560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -8.29785000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.56288594e-01,
                        2.14574581e-01,
                        0.00000000e00,
                        3.86544885e-01,
                        -2.46533421e-01,
                        0.00000000e00,
                        -3.02562911e-02,
                        3.19588394e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78492260e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        -2.46533421e-01,
                        2.24401624e-01,
                        0.00000000e00,
                        -3.19588394e-02,
                        1.19930640e-02,
                    ],
                    [
                        3.38914410e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -8.29785000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -3.30616560e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.56288594e-01,
                        -2.14574581e-01,
                        0.00000000e00,
                        -3.02562911e-02,
                        -3.19588394e-02,
                        0.00000000e00,
                        3.86544885e-01,
                        2.46533421e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78492260e-01,
                        -2.36394688e-01,
                        0.00000000e00,
                        3.19588394e-02,
                        1.19930640e-02,
                        0.00000000e00,
                        2.46533421e-01,
                        2.24401624e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005640585707,  # vcc
            "(T) CORRECTION ENERGY": -0.00523856,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01019219792989],
                    [0.00000000000000, 0.00604927147843, -0.00509609896494],
                    [0.00000000000000, -0.00604927147843, -0.00509609896494],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -9.19597780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.10457601e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55228801e-01,
                        2.78607515e-01,
                        0.00000000e00,
                        -3.55228801e-01,
                        -2.78607515e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68590076e-01,
                        0.00000000e00,
                        2.14495554e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        -2.14495554e-01,
                        -2.34295038e-01,
                    ],
                    [
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.41199179e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.85997115e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55228801e-01,
                        2.14495554e-01,
                        0.00000000e00,
                        3.85055647e-01,
                        -2.46551535e-01,
                        0.00000000e00,
                        -2.98268463e-02,
                        3.20559807e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78607515e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        -2.46551535e-01,
                        2.22895271e-01,
                        0.00000000e00,
                        -3.20559807e-02,
                        1.13997664e-02,
                    ],
                    [
                        4.59798890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.85997115e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.41199179e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55228801e-01,
                        -2.14495554e-01,
                        0.00000000e00,
                        -2.98268463e-02,
                        -3.20559807e-02,
                        0.00000000e00,
                        3.85055647e-01,
                        2.46551535e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78607515e-01,
                        -2.34295038e-01,
                        0.00000000e00,
                        3.20559807e-02,
                        1.13997664e-02,
                        0.00000000e00,
                        2.46551535e-01,
                        2.22895271e-01,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.005236187785,  # ecc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, 0.010120884870498],
                    [0.000000000000000, 0.006023996226962, -0.005060442435251],
                    [0.000000000000000, -0.006023996226962, -0.005060442435251],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.23500915078065532,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01040697979689],
                    [0.00000000000000, 0.00619487901010, -0.00520348989845],
                    [0.00000000000000, -0.00619487901010, -0.00520348989845],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00938924, 0.0, 0.0, 0.00469462, 0.0, 0.0, 0.00469462, 0.0, 0.0],
                    [0.0, 0.71035902, 0.0, 0.0, -0.35517951, 0.27864419, 0.0, -0.35517951, -0.27864419],
                    [0.0, 0.0, 0.46816235, 0.0, 0.21453598, -0.23408118, 0.0, -0.21453598, -0.23408118],
                    [0.00469462, 0.0, 0.0, -0.00451106, 0.0, 0.0, -0.00018357, 0.0, 0.0],
                    [0.0, -0.35517951, 0.21453598, 0.0, 0.38493736, -0.24659008, 0.0, -0.02975785, 0.0320541],
                    [0.0, 0.27864419, -0.23408118, 0.0, -0.24659008, 0.2227516, 0.0, -0.0320541, 0.01132957],
                    [0.00469462, 0.0, 0.0, -0.00018357, 0.0, 0.0, -0.00451106, 0.0, 0.0],
                    [0.0, -0.35517951, -0.21453598, 0.0, -0.02975785, -0.0320541, 0.0, 0.38493736, 0.24659008],
                    [0.0, -0.27864419, -0.23408118, 0.0, 0.0320541, 0.01132957, 0.0, 0.24659008, 0.2227516],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.23501318627899082,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01040244518005],
                    [0.00000000000000, 0.00619405837854, -0.00520122259002],
                    [0.00000000000000, -0.00619405837854, -0.00520122259002],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.009385784958, 0.0, 0.0, 0.004692892479, 0.0, 0.0, 0.004692892479, 0.0, 0.0],
                    [
                        0.0,
                        0.710344410436,
                        0.0,
                        0.0,
                        -0.355172205218,
                        0.278637075614,
                        0.0,
                        -0.355172205218,
                        -0.278637075614,
                    ],
                    [
                        0.0,
                        0.0,
                        0.468154078949,
                        0.0,
                        0.214541783785,
                        -0.234077039475,
                        0.0,
                        -0.214541783785,
                        -0.234077039475,
                    ],
                    [0.004692892479, 0.0, 0.0, -0.004510041295, 0.0, 0.0, -0.000182851184, 0.0, 0.0],
                    [
                        0.0,
                        -0.355172205218,
                        0.214541783785,
                        0.0,
                        0.384930286936,
                        -0.2465894297,
                        0.0,
                        -0.029758081719,
                        0.032047645914,
                    ],
                    [
                        0.0,
                        0.278637075614,
                        -0.234077039475,
                        0.0,
                        -0.2465894297,
                        0.222747174147,
                        0.0,
                        -0.032047645914,
                        0.011329865328,
                    ],
                    [0.004692892479, 0.0, 0.0, -0.000182851184, 0.0, 0.0, -0.004510041295, 0.0, 0.0],
                    [
                        0.0,
                        -0.355172205218,
                        -0.214541783785,
                        0.0,
                        -0.029758081719,
                        -0.032047645914,
                        0.0,
                        0.384930286936,
                        0.2465894297,
                    ],
                    [
                        0.0,
                        -0.278637075614,
                        -0.234077039475,
                        0.0,
                        0.032047645914,
                        0.011329865328,
                        0.0,
                        0.2465894297,
                        0.222747174147,
                    ],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.234514404732,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01007036070231],
                    [0.00000000000000, 0.00600051212420, -0.00503518035116],
                    [0.00000000000000, -0.00600051212420, -0.00503518035116],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [-0.009086018755, 0.0, 0.0, 0.004543009378, 0.0, 0.0, 0.004543009378, 0.0, 0.0],
                    [
                        0.0,
                        0.710438905643,
                        0.0,
                        0.0,
                        -0.355219452821,
                        0.278557716256,
                        0.0,
                        -0.355219452821,
                        -0.278557716256,
                    ],
                    [
                        0.0,
                        0.0,
                        0.468711648157,
                        0.0,
                        0.214551521677,
                        -0.234355824078,
                        0.0,
                        -0.214551521677,
                        -0.234355824078,
                    ],
                    [0.004543009378, 0.0, 0.0, -0.004367495061, 0.0, 0.0, -0.000175514317, 0.0, 0.0],
                    [
                        0.0,
                        -0.355219452821,
                        0.214551521677,
                        0.0,
                        0.385072138348,
                        -0.246554618967,
                        0.0,
                        -0.029852685527,
                        0.032003097289,
                    ],
                    [
                        0.0,
                        0.278557716256,
                        -0.234355824078,
                        0.0,
                        -0.246554618967,
                        0.222923456244,
                        0.0,
                        -0.032003097289,
                        0.011432367835,
                    ],
                    [0.004543009378, 0.0, 0.0, -0.000175514317, 0.0, 0.0, -0.004367495061, 0.0, 0.0],
                    [
                        0.0,
                        -0.355219452821,
                        -0.214551521677,
                        0.0,
                        -0.029852685527,
                        -0.032003097289,
                        0.0,
                        0.385072138348,
                        0.246554618967,
                    ],
                    [
                        0.0,
                        -0.278557716256,
                        -0.234355824078,
                        0.0,
                        0.032003097289,
                        0.011432367835,
                        0.0,
                        0.246554618967,
                        0.222923456244,
                    ],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.2345361459852029,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01009389495078],
                    [0.00000000000000, 0.00601318997564, -0.00504694747539],
                    [0.00000000000000, -0.00601318997564, -0.00504694747539],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00910672, 0.0, 0.0, 0.00455336, 0.0, 0.0, 0.00455336, 0.0, 0.0],
                    [0.0, 0.71040265, 0.0, 0.0, -0.35520132, 0.27855168, 0.0, -0.35520132, -0.27855168],
                    [0.0, 0.0, 0.46867889, 0.0, 0.21454552, -0.23433945, 0.0, -0.21454552, -0.23433945],
                    [0.00455336, 0.0, 0.0, -0.00437695, 0.0, 0.0, -0.00017641, 0.0, 0.0],
                    [0.0, -0.35520132, 0.21454552, 0.0, 0.38506038, -0.2465486, 0.0, -0.02985906, 0.03200308],
                    [0.0, 0.27855168, -0.23433945, 0.0, -0.2465486, 0.22290841, 0.0, -0.03200308, 0.01143103],
                    [0.00455336, 0.0, 0.0, -0.00017641, 0.0, 0.0, -0.00437695, 0.0, 0.0],
                    [0.0, -0.35520132, -0.21454552, 0.0, -0.02985906, -0.03200308, 0.0, 0.38506038, 0.2465486],
                    [0.0, -0.27855168, -0.23433945, 0.0, 0.03200308, 0.01143103, 0.0, 0.2465486, 0.22290841],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.234880733453953,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.010281901133138,
                    0.000000000000000,
                    0.006116030816197,
                    -0.005140950566570,
                    0.000000000000000,
                    -0.006116030816197,
                    -0.005140950566570,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -9.27691040e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.10153874e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55076937e-01,
                        2.78521415e-01,
                        0.00000000e00,
                        -3.55076937e-01,
                        -2.78521415e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68513289e-01,
                        0.00000000e00,
                        2.14488040e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        -2.14488040e-01,
                        -2.34256644e-01,
                    ],
                    [
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45556240e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.82892800e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55076937e-01,
                        2.14488040e-01,
                        0.00000000e00,
                        3.84940871e-01,
                        -2.46504728e-01,
                        0.00000000e00,
                        -2.98639346e-02,
                        3.20166873e-02,
                    ],
                    [
                        0.00000000e00,
                        2.78521415e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        -2.46504728e-01,
                        2.22819148e-01,
                        0.00000000e00,
                        -3.20166873e-02,
                        1.14374966e-02,
                    ],
                    [
                        4.63845520e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.82892800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45556240e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55076937e-01,
                        -2.14488040e-01,
                        0.00000000e00,
                        -2.98639346e-02,
                        -3.20166873e-02,
                        0.00000000e00,
                        3.84940871e-01,
                        2.46504728e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.78521415e-01,
                        -2.34256644e-01,
                        0.00000000e00,
                        3.20166873e-02,
                        1.14374966e-02,
                        0.00000000e00,
                        2.46504728e-01,
                        2.22819148e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000291847368718,  # ncc
            "(Q) CORRECTION ENERGY": -0.000507547985814,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [  # ncc
                    0.000000000000000,
                    0.000000000000000,
                    0.010822433278893,
                    0.000000000000000,
                    0.006408928209897,
                    -0.005411216639448,
                    0.000000000000000,
                    -0.006408928209897,
                    -0.005411216639448,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.235338854850175,  # ncc
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011895155,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2330452995,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0503175223,  # occ
            "PBE TOTAL ENERGY": -76.35896461348528,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.0, 0.0, 0.018069712075],
                    [-0.0, 0.011568971307, -0.009035203776],
                    [0.0, -0.011568971307, -0.009035203776],
                ]
            ),
            "B3LYP TOTAL ENERGY": -76.4445487826922090,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-1.77493704e-30, 0.00000000e00, 7.45592796e-03],
                    [-1.87490767e-19, 6.12391320e-03, -3.72819302e-03],
                    [1.87490767e-19, -6.12391320e-03, -3.72819302e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -76.40749597292017,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, -7.88860905e-31, 7.74045385e-03],
                    [-1.93603325e-19, 6.32356448e-03, -3.87045600e-03],
                    [1.93603325e-19, -6.32356448e-03, -3.87045600e-03],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2701916672,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530131,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.000531535533,
                    0.0,
                    -0.000960201925,
                    0.000265767766,
                    0.0,
                    0.000960201925,
                    0.000265767766,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        4.79582300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.99657262e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.49828631e-01,
                        2.70680600e-01,
                        0.00000000e00,
                        -3.49828631e-01,
                        -2.70680600e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.58837997e-01,
                        0.00000000e00,
                        2.05924494e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        -2.05924494e-01,
                        -2.29418999e-01,
                    ],
                    [
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        4.55295000e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15503800e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.49828631e-01,
                        2.05924494e-01,
                        0.00000000e00,
                        3.80569607e-01,
                        -2.38302547e-01,
                        0.00000000e00,
                        -3.07409760e-02,
                        3.23780531e-02,
                    ],
                    [
                        0.00000000e00,
                        2.70680600e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        -2.38302547e-01,
                        2.19836556e-01,
                        0.00000000e00,
                        -3.23780531e-02,
                        9.58244310e-03,
                    ],
                    [
                        -2.39791100e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15503800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        4.55295000e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.49828631e-01,
                        -2.05924494e-01,
                        0.00000000e00,
                        -3.07409760e-02,
                        -3.23780531e-02,
                        0.00000000e00,
                        3.80569607e-01,
                        2.38302547e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.70680600e-01,
                        -2.29418999e-01,
                        0.00000000e00,
                        3.23780531e-02,
                        9.58244310e-03,
                        0.00000000e00,
                        2.38302547e-01,
                        2.19836556e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.27294451,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.05952936,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.007003320116, 0.0, -0.004485227443, 0.003501660058, 0.0, 0.004485227443, 0.003501660058]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.27601976,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.007944754373,  # vcc
            "CISD CORRELATION ENERGY": -0.26425382513286,  # vcc
            "QCISD CORRELATION ENERGY": -0.27614913924585,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00704158,  # vcc
            "LCCD CORRELATION ENERGY": -0.2786913134,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.057792990490,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.002309010869, 0.0, -0.0017745445, 0.001154505435, -0.0, 0.0017745445, 0.001154505435]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2808517417,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.058297242512,  # fnocc
            "CCD CORRELATION ENERGY": -0.27417224324851,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.05767874,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, -0.0047556235], [0.0, -0.0030400734, 0.0023778117], [0.0, 0.0030400734, 0.0023778117]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        4.29117820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.94523343e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.47261672e-01,
                        2.67217473e-01,
                        0.00000000e00,
                        -3.47261672e-01,
                        -2.67217473e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.66482117e-01,
                        0.00000000e00,
                        2.04034191e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        -2.04034191e-01,
                        -2.33241059e-01,
                    ],
                    [
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.13479200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.07971000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.47261672e-01,
                        2.04034191e-01,
                        0.00000000e00,
                        3.79908779e-01,
                        -2.35625839e-01,
                        0.00000000e00,
                        -3.26471076e-02,
                        3.15916340e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67217473e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        -2.35625839e-01,
                        2.21133541e-01,
                        0.00000000e00,
                        -3.15916340e-02,
                        1.21075177e-02,
                    ],
                    [
                        -2.14558910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.07971000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        2.13479200e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.47261672e-01,
                        -2.04034191e-01,
                        0.00000000e00,
                        -3.26471076e-02,
                        -3.15916340e-02,
                        0.00000000e00,
                        3.79908779e-01,
                        2.35625839e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67217473e-01,
                        -2.33241059e-01,
                        0.00000000e00,
                        3.15916340e-02,
                        1.21075177e-02,
                        0.00000000e00,
                        2.35625839e-01,
                        2.21133541e-01,
                    ],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.275705491773,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.058006927914493,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.003374258422,
                    0.0,
                    -0.002334452569,
                    0.001687129211,
                    0.0,
                    0.002334452569,
                    0.001687129211,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        3.04444750e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.92271308e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.46135654e-01,
                        2.66828227e-01,
                        0.00000000e00,
                        -3.46135654e-01,
                        -2.66828227e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.64211859e-01,
                        0.00000000e00,
                        2.03582577e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        -2.03582577e-01,
                        -2.32105930e-01,
                    ],
                    [
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.57653820e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -5.43145000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46135654e-01,
                        2.03582577e-01,
                        0.00000000e00,
                        3.78818316e-01,
                        -2.35205402e-01,
                        0.00000000e00,
                        -3.26826626e-02,
                        3.16228251e-02,
                    ],
                    [
                        0.00000000e00,
                        2.66828227e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        -2.35205402e-01,
                        2.20168635e-01,
                        0.00000000e00,
                        -3.16228251e-02,
                        1.19372948e-02,
                    ],
                    [
                        -1.52222380e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -5.43145000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        1.57653820e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46135654e-01,
                        -2.03582577e-01,
                        0.00000000e00,
                        -3.26826626e-02,
                        -3.16228251e-02,
                        0.00000000e00,
                        3.78818316e-01,
                        2.35205402e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.66828227e-01,
                        -2.32105930e-01,
                        0.00000000e00,
                        3.16228251e-02,
                        1.19372948e-02,
                        0.00000000e00,
                        2.35205402e-01,
                        2.20168635e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.007532613647,  # vcc
            "(T) CORRECTION ENERGY": -0.007263596331,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, -0.00018184005110],
                    [0.00000000000000, -0.00073407755645, 0.00009092002555],
                    [0.00000000000000, 0.00073407755645, 0.00009092002555],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [0.000164043137, 0.0, 0.0, -0.000082021569, 0.0, 0.0, -0.000082021569, 0.0, 0.0],
                    [
                        0.0,
                        0.690399868053,
                        0.0,
                        0.0,
                        -0.345199934026,
                        0.26721874523,
                        0.0,
                        -0.345199934026,
                        -0.26721874523,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459903012519,
                        0.0,
                        0.203684212528,
                        -0.229951506259,
                        0.0,
                        -0.203684212528,
                        -0.229951506259,
                    ],
                    [-0.000082021569, 0.0, 0.0, 0.000297412235, 0.0, 0.0, -0.000215390667, 0.0, 0.0],
                    [
                        0.0,
                        -0.345199934026,
                        0.203684212528,
                        0.0,
                        0.377546525502,
                        -0.235451478879,
                        0.0,
                        -0.032346591476,
                        0.031767266351,
                    ],
                    [
                        0.0,
                        0.26721874523,
                        -0.229951506259,
                        0.0,
                        -0.235451478879,
                        0.218683624079,
                        0.0,
                        -0.031767266351,
                        0.011267882181,
                    ],
                    [-0.000082021569, 0.0, 0.0, -0.000215390667, 0.0, 0.0, 0.000297412235, 0.0, 0.0],
                    [
                        0.0,
                        -0.345199934026,
                        -0.203684212528,
                        0.0,
                        -0.032346591476,
                        -0.031767266351,
                        0.0,
                        0.377546525502,
                        0.235451478879,
                    ],
                    [
                        0.0,
                        -0.26721874523,
                        -0.229951506259,
                        0.0,
                        0.031767266351,
                        0.011267882181,
                        0.0,
                        0.235451478879,
                        0.218683624079,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.007181434345419,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ncc
                [
                    [0.000000000000000, 0.000000000000000, -0.000263235380777],
                    [0.000000000000000, -0.000769830346953, 0.000131617690376],
                    [0.000000000000000, 0.000769830346953, 0.000131617690376],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.283301014194,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00003469032178],
                    [0.00000000000000, -0.00061071465370, -0.00001734516089],
                    [0.00000000000000, 0.00061071465370, -0.00001734516089],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000031470364, 0.0, 0.0, 0.000015735182, 0.0, 0.0, 0.000015735182, 0.0, 0.0],
                    [
                        0.0,
                        0.690357139667,
                        0.0,
                        0.0,
                        -0.345178569834,
                        0.267277548335,
                        0.0,
                        -0.345178569834,
                        -0.267277548335,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459591068441,
                        0.0,
                        0.203726352234,
                        -0.229795534221,
                        0.0,
                        -0.203726352234,
                        -0.229795534221,
                    ],
                    [0.000015735182, 0.0, 0.0, 0.000205401177, 0.0, 0.0, -0.000221136359, 0.0, 0.0],
                    [
                        0.0,
                        -0.345178569834,
                        0.203726352234,
                        0.0,
                        0.377473575693,
                        -0.235501950285,
                        0.0,
                        -0.03229500586,
                        0.031775598051,
                    ],
                    [
                        0.0,
                        0.267277548335,
                        -0.229795534221,
                        0.0,
                        -0.235501950285,
                        0.218585504687,
                        0.0,
                        -0.031775598051,
                        0.011210029534,
                    ],
                    [0.000015735182, 0.0, 0.0, -0.000221136359, 0.0, 0.0, 0.000205401177, 0.0, 0.0],
                    [
                        0.0,
                        -0.345178569834,
                        -0.203726352234,
                        0.0,
                        -0.03229500586,
                        -0.031775598051,
                        0.0,
                        0.377473575693,
                        0.235501950285,
                    ],
                    [
                        0.0,
                        -0.267277548335,
                        -0.229795534221,
                        0.0,
                        0.031775598051,
                        0.011210029534,
                        0.0,
                        0.235501950285,
                        0.218585504687,
                    ],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.283304708235,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00003781932904],
                    [0.00000000000000, -0.00060845267210, -0.00001890966452],
                    [0.00000000000000, 0.00060845267210, -0.00001890966452],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000034308612, 0.0, 0.0, 0.000017154306, 0.0, 0.0, 0.000017154306, 0.0, 0.0],
                    [
                        0.0,
                        0.69034722208,
                        0.0,
                        0.0,
                        -0.34517361104,
                        0.267274828449,
                        0.0,
                        -0.34517361104,
                        -0.267274828449,
                    ],
                    [
                        0.0,
                        0.0,
                        0.459575612743,
                        0.0,
                        0.203725689988,
                        -0.229787806371,
                        0.0,
                        -0.203725689988,
                        -0.229787806371,
                    ],
                    [0.000017154306, 0.0, 0.0, 0.000203904093, 0.0, 0.0, -0.000221058399, 0.0, 0.0],
                    [
                        0.0,
                        -0.34517361104,
                        0.203725689988,
                        0.0,
                        0.377466284086,
                        -0.235500259218,
                        0.0,
                        -0.032292673046,
                        0.03177456923,
                    ],
                    [
                        0.0,
                        0.267274828449,
                        -0.229787806371,
                        0.0,
                        -0.235500259218,
                        0.218579787679,
                        0.0,
                        -0.03177456923,
                        0.011208018692,
                    ],
                    [0.000017154306, 0.0, 0.0, -0.000221058399, 0.0, 0.0, 0.000203904093, 0.0, 0.0],
                    [
                        0.0,
                        -0.34517361104,
                        -0.203725689988,
                        0.0,
                        -0.032292673046,
                        -0.03177456923,
                        0.0,
                        0.377466284086,
                        0.235500259218,
                    ],
                    [
                        0.0,
                        -0.267274828449,
                        -0.229787806371,
                        0.0,
                        0.03177456923,
                        0.011208018692,
                        0.0,
                        0.235500259218,
                        0.218579787679,
                    ],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.282650715222,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, -0.00033837753770],
                    [0.00000000000000, -0.00081713978219, 0.00016918876885],
                    [0.00000000000000, 0.00081713978219, 0.00016918876885],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.array(  # ecc by psi fd
                [
                    [0.000305098093, 0.0, 0.0, -0.000152549047, 0.0, 0.0, -0.000152549047, 0.0, 0.0],
                    [
                        0.0,
                        0.690395599827,
                        0.0,
                        0.0,
                        -0.345197799913,
                        0.267162476654,
                        0.0,
                        -0.345197799913,
                        -0.267162476654,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460115900113,
                        0.0,
                        0.203699000404,
                        -0.230057950057,
                        0.0,
                        -0.203699000404,
                        -0.230057950057,
                    ],
                    [-0.000152549047, 0.0, 0.0, 0.000361653884, 0.0, 0.0, -0.000209104837, 0.0, 0.0],
                    [
                        0.0,
                        -0.345197799913,
                        0.203699000404,
                        0.0,
                        0.377579483066,
                        -0.235430738529,
                        0.0,
                        -0.032381683152,
                        0.031731738125,
                    ],
                    [
                        0.0,
                        0.267162476654,
                        -0.230057950057,
                        0.0,
                        -0.235430738529,
                        0.218744030684,
                        0.0,
                        -0.031731738125,
                        0.011313919373,
                    ],
                    [-0.000152549047, 0.0, 0.0, -0.000209104837, 0.0, 0.0, 0.000361653884, 0.0, 0.0],
                    [
                        0.0,
                        -0.345197799913,
                        -0.203699000404,
                        0.0,
                        -0.032381683152,
                        -0.031731738125,
                        0.0,
                        0.377579483066,
                        0.235430738529,
                    ],
                    [
                        0.0,
                        -0.267162476654,
                        -0.230057950057,
                        0.0,
                        0.031731738125,
                        0.011313919373,
                        0.0,
                        0.235430738529,
                        0.218744030684,
                    ],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.28267436870285,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, -0.00031801749656],
                    [0.00000000000000, -0.00080724235455, 0.00015900874828],
                    [0.00000000000000, 0.00080724235455, 0.00015900874828],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [
                        2.86755305e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90367936e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45183968e-01,
                        2.67158588e-01,
                        0.00000000e00,
                        -3.45183968e-01,
                        -2.67158588e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60092567e-01,
                        0.00000000e00,
                        2.03699151e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        -2.03699151e-01,
                        -2.30046284e-01,
                    ],
                    [
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.53604923e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10227270e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45183968e-01,
                        2.03699151e-01,
                        0.00000000e00,
                        3.77567061e-01,
                        -2.35428870e-01,
                        0.00000000e00,
                        -3.23830932e-02,
                        3.17297185e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67158588e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        -2.35428870e-01,
                        2.18733436e-01,
                        0.00000000e00,
                        -3.17297185e-02,
                        1.13128480e-02,
                    ],
                    [
                        -1.43377653e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10227270e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.53604923e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45183968e-01,
                        -2.03699151e-01,
                        0.00000000e00,
                        -3.23830932e-02,
                        -3.17297185e-02,
                        0.00000000e00,
                        3.77567061e-01,
                        2.35428870e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67158588e-01,
                        -2.30046284e-01,
                        0.00000000e00,
                        3.17297185e-02,
                        1.13128480e-02,
                        0.00000000e00,
                        2.35428870e-01,
                        2.18733436e-01,
                    ],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.283033246143265,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    -0.000217363788816,
                    0.000000000000000,
                    -0.000742442561705,
                    0.000108681894407,
                    0.000000000000000,
                    0.000742442561705,
                    0.000108681894407,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        1.96117400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90148921e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45074461e-01,
                        2.67109270e-01,
                        0.00000000e00,
                        -3.45074461e-01,
                        -2.67109270e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60001543e-01,
                        0.00000000e00,
                        2.03664977e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        -2.03664977e-01,
                        -2.30000771e-01,
                    ],
                    [
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        3.08365200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10306500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45074461e-01,
                        2.03664977e-01,
                        0.00000000e00,
                        3.77473316e-01,
                        -2.35387123e-01,
                        0.00000000e00,
                        -3.23988558e-02,
                        3.17221463e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67109270e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        -2.35387123e-01,
                        2.18668768e-01,
                        0.00000000e00,
                        -3.17221463e-02,
                        1.13320029e-02,
                    ],
                    [
                        -9.80587000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -2.10306500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.08365200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45074461e-01,
                        -2.03664977e-01,
                        0.00000000e00,
                        -3.23988558e-02,
                        -3.17221463e-02,
                        0.00000000e00,
                        3.77473316e-01,
                        2.35387123e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67109270e-01,
                        -2.30000771e-01,
                        0.00000000e00,
                        3.17221463e-02,
                        1.13320029e-02,
                        0.00000000e00,
                        2.35387123e-01,
                        2.18668768e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000163881984112,  # ncc
            "(Q) CORRECTION ENERGY": -0.000383414612913,  # ncc
            "CCSDT(Q) TOTAL GRADIENT": np.array(
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.000216022413761,
                    0.000000000000000,
                    -0.000511432746023,
                    -0.000108011206892,
                    0.000000000000000,
                    0.000511432746023,
                    -0.000108011206892,
                ]
            ).reshape((-1, 3)),
            "CCSDTQ CORRELATION ENERGY": -0.283376955140291,  # ncc
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0013521561,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2800053174,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0582676514,  # occ
            "PBE TOTAL ENERGY": -76.38209089347507,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, -0.000000000000, 0.012770562551],
                    [-0.000000000000, 0.008604125876, -0.006385673964],
                    [0.000000000000, -0.008604125876, -0.006385673964],
                ]
            ),
            "B3LYP TOTAL ENERGY": -76.4684173811156711,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 1.57772181e-30, 2.02268338e-03],
                    [-9.31840056e-20, 3.04362060e-03, -1.01162676e-03],
                    [9.31840056e-20, -3.04362060e-03, -1.01162676e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -76.43134091764595,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-7.88860905e-31, 7.88860905e-31, 2.30650526e-03],
                    [-9.91741439e-20, 3.23927336e-03, -1.15353738e-03],
                    [9.91741439e-20, -3.23927336e-03, -1.15353738e-03],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05948928003552,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001918693775,
            "MP2 TOTAL GRADIENT": np.array(
                [  # occ findif-5 ae pk+conv
                    0.00000000000000,
                    0.00000000000000,
                    0.01250561195911,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01206536529299,
                    0.00000000000000,
                    0.01033165380573,
                    -0.00022012333306,
                    0.00000000000000,
                    -0.01033165380573,
                    -0.00022012333306,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        3.44512200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.01573440e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.35480990e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.87467360e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.59931900e-04,
                        4.32569100e-04,
                        0.00000000e00,
                        2.59931900e-04,
                        -4.32569100e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.15837144e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.13789867e-01,
                        0.00000000e00,
                        1.11748040e-03,
                        -1.02363850e-03,
                        0.00000000e00,
                        -1.11748040e-03,
                        -1.02363850e-03,
                    ],
                    [
                        -4.01573440e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.35926330e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.87467360e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.27499890e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.00162700e-04,
                        8.11818000e-05,
                        0.00000000e00,
                        -2.00162700e-04,
                        -8.11818000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.13789867e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.13964588e-01,
                        0.00000000e00,
                        1.98755000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -1.98755000e-05,
                        -8.73610000e-05,
                    ],
                    [
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41535090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.30180910e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.59931900e-04,
                        1.11748040e-03,
                        0.00000000e00,
                        -2.00162700e-04,
                        1.98755000e-05,
                        0.00000000e00,
                        4.11358061e-01,
                        -8.25553300e-04,
                        0.00000000e00,
                        -4.11417830e-01,
                        -3.11802500e-04,
                    ],
                    [
                        0.00000000e00,
                        4.32569100e-04,
                        -1.02363850e-03,
                        0.00000000e00,
                        8.11818000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -8.25553300e-04,
                        -4.66526560e-03,
                        0.00000000e00,
                        3.11802500e-04,
                        5.77626520e-03,
                    ],
                    [
                        2.85306200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.71764400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.30180910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41535090e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.59931900e-04,
                        -1.11748040e-03,
                        0.00000000e00,
                        -2.00162700e-04,
                        -1.98755000e-05,
                        0.00000000e00,
                        -4.11417830e-01,
                        3.11802500e-04,
                        0.00000000e00,
                        4.11358061e-01,
                        8.25553300e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.32569100e-04,
                        -1.02363850e-03,
                        0.00000000e00,
                        -8.11818000e-05,
                        -8.73610000e-05,
                        0.00000000e00,
                        -3.11802500e-04,
                        5.77626520e-03,
                        0.00000000e00,
                        8.25553300e-04,
                        -4.66526560e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.07481556,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235692,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.0093167821,
                    0.0,
                    0.0,
                    -0.00908998793,
                    0.0,
                    0.012725715572,
                    -0.000113397085,
                    0.0,
                    -0.012725715572,
                    -0.000113397085,
                ]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.07927852,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000248958942,  # vcc only
            "CISD CORRELATION ENERGY": -0.08014448073265,  # vcc
            "QCISD CORRELATION ENERGY": -0.08218197897917,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00062359,  # vcc only
            "LCCD CORRELATION ENERGY": -0.0834347185,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024003297,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [
                    -0.0,
                    0.0,
                    0.004644430473,
                    0.0,
                    0.0,
                    -0.004561891704,
                    -0.0,
                    0.015262723486,
                    -0.000041269384,
                    0.0,
                    -0.015262723486,
                    -0.000041269384,
                ]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.0848110820,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0024351,  # vcc
            "CCD CORRELATION ENERGY": -0.08113776652663,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.00235010,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [
                    [0.0, 0.0, 0.0064056106],
                    [0.0, 0.0, -0.0062159539],
                    [0.0, 0.0144496921, -0.0000948283],
                    [0.0, -0.0144496921, -0.0000948283],
                ]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        1.21745460e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.67345390e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.18297710e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.56709910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.92061000e-04,
                        4.57116300e-04,
                        0.00000000e00,
                        1.92061000e-04,
                        -4.57116300e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.08191655e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.06455377e-01,
                        0.00000000e00,
                        1.17641080e-03,
                        -8.68139000e-04,
                        0.00000000e00,
                        -1.17641080e-03,
                        -8.68139000e-04,
                    ],
                    [
                        -1.67345390e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.96873420e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -1.56709910e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.90289150e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.67896600e-04,
                        4.83546000e-05,
                        0.00000000e00,
                        -1.67896600e-04,
                        -4.83546000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.06455377e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.06585356e-01,
                        0.00000000e00,
                        1.26821500e-04,
                        -6.49859000e-05,
                        0.00000000e00,
                        -1.26821500e-04,
                        -6.49859000e-05,
                    ],
                    [
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.03317422e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.02513900e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.92061000e-04,
                        1.17641080e-03,
                        0.00000000e00,
                        -1.67896600e-04,
                        1.26821500e-04,
                        0.00000000e00,
                        4.08407145e-01,
                        -9.04348400e-04,
                        0.00000000e00,
                        -4.08431309e-01,
                        -3.98877500e-04,
                    ],
                    [
                        0.00000000e00,
                        4.57116300e-04,
                        -8.68139000e-04,
                        0.00000000e00,
                        4.83546000e-05,
                        -6.49859000e-05,
                        0.00000000e00,
                        -9.04348400e-04,
                        -7.77410400e-03,
                        0.00000000e00,
                        3.98877500e-04,
                        8.70722880e-03,
                    ],
                    [
                        2.27999700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.47647400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.02513900e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.03317422e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.92061000e-04,
                        -1.17641080e-03,
                        0.00000000e00,
                        -1.67896600e-04,
                        -1.26821500e-04,
                        0.00000000e00,
                        -4.08431309e-01,
                        3.98877500e-04,
                        0.00000000e00,
                        4.08407145e-01,
                        9.04348400e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.57116300e-04,
                        -8.68139000e-04,
                        0.00000000e00,
                        -4.83546000e-05,
                        -6.49859000e-05,
                        0.00000000e00,
                        -3.98877500e-04,
                        8.70722880e-03,
                        0.00000000e00,
                        9.04348400e-04,
                        -7.77410400e-03,
                    ],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.08217287869,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.002377557359,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005209606766,
                    0.0,
                    0.0,
                    -0.005071403517,
                    0.0,
                    0.014880198292,
                    -0.000069101625,
                    0.0,
                    -0.014880198292,
                    -0.000069101625,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        7.74047000e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.20954470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.31670700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.10104560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.84687400e-04,
                        4.38316100e-04,
                        0.00000000e00,
                        1.84687400e-04,
                        -4.38316100e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.06125475e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.04450486e-01,
                        0.00000000e00,
                        1.15794910e-03,
                        -8.37494800e-04,
                        0.00000000e00,
                        -1.15794910e-03,
                        -8.37494800e-04,
                    ],
                    [
                        -1.20954470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.49705710e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -1.10104560e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.43042210e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.64688300e-04,
                        5.23087000e-05,
                        0.00000000e00,
                        -1.64688300e-04,
                        -5.23087000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.04450486e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.04563594e-01,
                        0.00000000e00,
                        1.62442000e-04,
                        -5.65541000e-05,
                        0.00000000e00,
                        -1.62442000e-04,
                        -5.65541000e-05,
                    ],
                    [
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06352124e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.05612197e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.84687400e-04,
                        1.15794910e-03,
                        0.00000000e00,
                        -1.64688300e-04,
                        1.62442000e-04,
                        0.00000000e00,
                        4.07393462e-01,
                        -9.05508000e-04,
                        0.00000000e00,
                        -4.07413462e-01,
                        -4.14883200e-04,
                    ],
                    [
                        0.00000000e00,
                        4.38316100e-04,
                        -8.37494800e-04,
                        0.00000000e00,
                        5.23087000e-05,
                        -5.65541000e-05,
                        0.00000000e00,
                        -9.05508000e-04,
                        -8.15157550e-03,
                        0.00000000e00,
                        4.14883200e-04,
                        9.04562440e-03,
                    ],
                    [
                        2.17748900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.43756200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.05612197e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06352124e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.84687400e-04,
                        -1.15794910e-03,
                        0.00000000e00,
                        -1.64688300e-04,
                        -1.62442000e-04,
                        0.00000000e00,
                        -4.07413462e-01,
                        4.14883200e-04,
                        0.00000000e00,
                        4.07393462e-01,
                        9.05508000e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.38316100e-04,
                        -8.37494800e-04,
                        0.00000000e00,
                        -5.23087000e-05,
                        -5.65541000e-05,
                        0.00000000e00,
                        -4.14883200e-04,
                        9.04562440e-03,
                        0.00000000e00,
                        9.05508000e-04,
                        -8.15157550e-03,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.000630302926,  # vcc
            "(T) CORRECTION ENERGY": -0.00062614,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00469857108970],
                    [0.00000000000000, 0.00000000000000, -0.00464327797336],
                    [0.00000000000000, 0.01496002471717, -0.00002764655817],
                    [0.00000000000000, -0.01496002471717, -0.00002764655817],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        0.000605645748,
                        0.0,
                        0.0,
                        -0.001025922218,
                        0.0,
                        0.0,
                        0.000210138235,
                        0.0,
                        0.0,
                        0.000210138235,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000566396166,
                        0.0,
                        0.0,
                        -0.000920471148,
                        0.0,
                        0.0,
                        0.000177037491,
                        0.000434687941,
                        0.0,
                        0.000177037491,
                        -0.000434687941,
                    ],
                    [
                        0.0,
                        0.0,
                        0.205974349398,
                        0.0,
                        0.0,
                        -0.20443915744,
                        0.0,
                        0.001282151675,
                        -0.000767595979,
                        0.0,
                        -0.001282151675,
                        -0.000767595979,
                    ],
                    [
                        -0.001025922218,
                        0.0,
                        0.0,
                        0.001313504777,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.000920471148,
                        0.0,
                        0.0,
                        0.001248510511,
                        0.0,
                        0.0,
                        -0.000164019682,
                        0.000048912565,
                        0.0,
                        -0.000164019682,
                        -0.000048912565,
                    ],
                    [
                        0.0,
                        0.0,
                        -0.20443915744,
                        0.0,
                        0.0,
                        0.204583205785,
                        0.0,
                        0.000137691376,
                        -0.000072024172,
                        0.0,
                        -0.000137691376,
                        -0.000072024172,
                    ],
                    [
                        0.000210138235,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        -0.010688228885,
                        0.0,
                        0.0,
                        0.01062188193,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177037491,
                        0.001282151675,
                        0.0,
                        -0.000164019682,
                        0.000137691376,
                        0.0,
                        0.407323404956,
                        -0.000951721779,
                        0.0,
                        -0.407336422766,
                        -0.000468121273,
                    ],
                    [
                        0.0,
                        0.000434687941,
                        -0.000767595979,
                        0.0,
                        0.000048912565,
                        -0.000072024172,
                        0.0,
                        -0.000951721779,
                        -0.008268538433,
                        0.0,
                        0.000468121273,
                        0.009108158584,
                    ],
                    [
                        0.000210138235,
                        0.0,
                        0.0,
                        -0.000143791279,
                        0.0,
                        0.0,
                        0.01062188193,
                        0.0,
                        0.0,
                        -0.010688228885,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177037491,
                        -0.001282151675,
                        0.0,
                        -0.000164019682,
                        -0.000137691376,
                        0.0,
                        -0.407336422766,
                        0.000468121273,
                        0.0,
                        0.407323404956,
                        0.000951721779,
                    ],
                    [
                        0.0,
                        -0.000434687941,
                        -0.000767595979,
                        0.0,
                        -0.000048912565,
                        -0.000072024172,
                        0.0,
                        -0.000468121273,
                        0.009108158584,
                        0.0,
                        0.000951721779,
                        -0.008268538433,
                    ],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.082798380297,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.082796300732,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.082795702684,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.08280038799307,  # vcc
            "CCSDT CORRELATION ENERGY": -0.08311156413461,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0014842084,  # p4n
            "OLCCD CORRELATION ENERGY": -0.0847413506,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0024486744,  # occ
            "PBE TOTAL ENERGY": -26.04681636799003,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, -0.000000000000, -0.002915908298],
                    [0.000000000000, 0.000000000000, -0.000179663088],
                    [-0.000000000000, 0.026137498638, 0.001547791345],
                    [0.000000000000, -0.026137498638, 0.001547791345],
                ]
            ),
            "B3LYP TOTAL ENERGY": -26.1140057832643038,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [1.97215226e-31, 0.00000000e00, 3.32365409e-03],
                    [0.00000000e00, -1.05926147e-32, -5.83320285e-03],
                    [-5.87137508e-19, 1.91773664e-02, 1.25496529e-03],
                    [5.87137508e-19, -1.91773664e-02, 1.25496529e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -26.0893142594464784,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 9.86076132e-31, 3.05278616e-03],
                    [1.23259516e-32, 0.00000000e00, -5.57401307e-03],
                    [-5.95843028e-19, 1.94617102e-02, 1.26080391e-03],
                    [5.95843028e-19, -1.94617102e-02, 1.26080391e-03],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15485993330517828,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520162545964887,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.025490652204,
                    0.0,
                    0.013491755791,
                    -0.012745326102,
                    0.0,
                    -0.013491755791,
                    -0.012745326102,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.02198237, 0.0, 0.0, 0.01099118, 0.0, 0.0, 0.01099118, 0.0, 0.0],
                    [0.0, 0.62517441, 0.0, 0.0, -0.31258721, 0.24829043, 0.0, -0.31258721, -0.24829043],
                    [0.0, 0.0, 0.39893991, 0.0, 0.19093143, -0.19946995, 0.0, -0.19093143, -0.19946995],
                    [0.01099118, 0.0, 0.0, -0.00995947, 0.0, 0.0, -0.00103171, 0.0, 0.0],
                    [0.0, -0.31258721, 0.19093143, 0.0, 0.33968697, -0.21961093, 0.0, -0.02709976, 0.0286795],
                    [0.0, 0.24829043, -0.19946995, 0.0, -0.21961093, 0.19053107, 0.0, -0.0286795, 0.00893888],
                    [0.01099118, 0.0, 0.0, -0.00103171, 0.0, 0.0, -0.00995947, 0.0, 0.0],
                    [0.0, -0.31258721, -0.19093143, 0.0, -0.02709976, -0.0286795, 0.0, 0.33968697, 0.21961093],
                    [0.0, -0.24829043, -0.19946995, 0.0, 0.0286795, 0.00893888, 0.0, 0.21961093, 0.19053107],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.17091690,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533349,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027091204409, 0.0, 0.014414444742, -0.013545602205, 0.0, -0.014414444742, -0.013545602205]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.17311842,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.003268429219,  # vcc only
            "CISD CORRELATION ENERGY": -0.16787805277043,  # vcc
            "QCISD CORRELATION ENERGY": -0.17409647936869,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00371845,  # vcc only
            "LCCD CORRELATION ENERGY": -0.1770086091,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0341268118,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.030699807076, 0.0, 0.016626789229, -0.015349903538, 0.0, -0.016626789229, -0.015349903538]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1786081472,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0344530,  # vcc
            "CCD CORRELATION ENERGY": -0.17274172367784,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0337442,  # vcc
            "CCD TOTAL GRADIENT": np.array(
                [[0.0, 0.0, 0.02815376], [0.0, 0.01520857, -0.01407688], [0.0, -0.01520857, -0.01407688]]
            ),  # fcc findif in psi
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02427911, 0.0, 0.0, 0.01213956, 0.0, 0.0, 0.01213956, 0.0, 0.0],
                    [0.0, 0.61891369, 0.0, 0.0, -0.30945684, 0.2467696, 0.0, -0.30945684, -0.2467696],
                    [0.0, 0.0, 0.39863476, 0.0, 0.19000027, -0.19931738, 0.0, -0.19000027, -0.19931738],
                    [0.01213956, 0.0, 0.0, -0.01110176, 0.0, 0.0, -0.0010378, 0.0, 0.0],
                    [0.0, -0.30945684, 0.19000027, 0.0, 0.3374568, -0.21838493, 0.0, -0.02799995, 0.02838466],
                    [0.0, 0.2467696, -0.19931738, 0.0, -0.21838493, 0.18930325, 0.0, -0.02838466, 0.01001413],
                    [0.01213956, 0.0, 0.0, -0.0010378, 0.0, 0.0, -0.01110176, 0.0, 0.0],
                    [0.0, -0.30945684, -0.19000027, 0.0, -0.02799995, -0.02838466, 0.0, 0.3374568, 0.21838493],
                    [0.0, -0.2467696, -0.19931738, 0.0, 0.02838466, 0.01001413, 0.0, 0.21838493, 0.18930325],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.17387203707017695,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033935818857082,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.029278727285,
                    0.0,
                    0.015813927533,
                    -0.014639363642,
                    0.0,
                    -0.015813927533,
                    -0.014639363642,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02524909, 0.0, 0.0, 0.01262454, 0.0, 0.0, 0.01262454, 0.0, 0.0],
                    [0.0, 0.61582469, 0.0, 0.0, -0.30791234, 0.2459566, 0.0, -0.30791234, -0.2459566],
                    [0.0, 0.0, 0.39625423, 0.0, 0.18929354, -0.19812712, 0.0, -0.18929354, -0.19812712],
                    [0.01262454, 0.0, 0.0, -0.01154446, 0.0, 0.0, -0.00108008, 0.0, 0.0],
                    [0.0, -0.30791234, 0.18929354, 0.0, 0.3360106, -0.21762507, 0.0, -0.02809826, 0.02833153],
                    [0.0, 0.2459566, -0.19812712, 0.0, -0.21762507, 0.18819594, 0.0, -0.02833153, 0.00993118],
                    [0.01262454, 0.0, 0.0, -0.00108008, 0.0, 0.0, -0.01154446, 0.0, 0.0],
                    [0.0, -0.30791234, -0.18929354, 0.0, -0.02809826, -0.02833153, 0.0, 0.3360106, 0.21762507],
                    [0.0, -0.2459566, -0.19812712, 0.0, 0.02833153, 0.00993118, 0.0, 0.21762507, 0.18819594],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.003990106867,  # vcc
            "(T) CORRECTION ENERGY": -0.00384378,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03116000949177],
                    [0.00000000000000, 0.01681030698770, -0.01558000474588],
                    [0.00000000000000, -0.01681030698770, -0.01558000474588],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.026871482612, 0.0, 0.0, 0.013435741306, 0.0, 0.0, 0.013435741306, 0.0, 0.0],
                    [
                        0.0,
                        0.614216172202,
                        0.0,
                        0.0,
                        -0.307108086101,
                        0.245961925921,
                        0.0,
                        -0.307108086101,
                        -0.245961925921,
                    ],
                    [
                        0.0,
                        0.0,
                        0.39318444945,
                        0.0,
                        0.189186035018,
                        -0.196592224725,
                        0.0,
                        -0.189186035018,
                        -0.196592224725,
                    ],
                    [0.013435741306, 0.0, 0.0, -0.012279754441, 0.0, 0.0, -0.001155986865, 0.0, 0.0],
                    [
                        0.0,
                        -0.307108086101,
                        0.189186035018,
                        0.0,
                        0.334882090526,
                        -0.217573980469,
                        0.0,
                        -0.027774004425,
                        0.028387945452,
                    ],
                    [
                        0.0,
                        0.245961925921,
                        -0.196592224725,
                        0.0,
                        -0.217573980469,
                        0.18710085378,
                        0.0,
                        -0.028387945452,
                        0.009491370945,
                    ],
                    [0.013435741306, 0.0, 0.0, -0.001155986865, 0.0, 0.0, -0.012279754441, 0.0, 0.0],
                    [
                        0.0,
                        -0.307108086101,
                        -0.189186035018,
                        0.0,
                        -0.027774004425,
                        -0.028387945452,
                        0.0,
                        0.334882090526,
                        0.217573980469,
                    ],
                    [
                        0.0,
                        -0.245961925921,
                        -0.196592224725,
                        0.0,
                        0.028387945452,
                        0.009491370945,
                        0.0,
                        0.217573980469,
                        0.18710085378,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.00376395919,  # mrcc
            "CCSDT-1A CORRELATION ENERGY": -0.17782828147829832,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.0311852], [0.0, 0.01683584, -0.0155926], [0.0, -0.01683584, -0.0155926]]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.17782563826176556,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03117878], [0.0, 0.01683259, -0.01558939], [0.0, -0.01683259, -0.01558939]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.177555974326,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.17757120863770837,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03102835], [0.0, 0.01674637, -0.01551417], [0.0, -0.01674637, -0.01551417]]
            ),
            "CCSDT CORRELATION ENERGY": -0.17817199774450,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0011118724,  # p4n
            "OLCCD CORRELATION ENERGY": -0.1781057943,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0344689234,  # occ
            "PBE TOTAL ENERGY": -55.81473169701641,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-0.000000000000, 0.000000000000, 0.035829930902],
                    [-0.000000000000, 0.019748673246, -0.017915051582],
                    [0.000000000000, -0.019748673246, -0.017915051582],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.8900885216505259,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 5.91645679e-31, 2.68092690e-02],
                    [-4.57342645e-19, 1.49379444e-02, -1.34046506e-02],
                    [4.57342645e-19, -1.49379444e-02, -1.34046506e-02],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.8573454463695782,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 1.97215226e-30, 2.70819655e-02],
                    [-4.63472198e-19, 1.51381508e-02, -1.35409990e-02],
                    [4.63472198e-19, -1.51381508e-02, -1.35409990e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.195530391293,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161633,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.013731673196,
                    0.0,
                    0.005352105826,
                    -0.006865836598,
                    0.0,
                    -0.005352105826,
                    -0.006865836598,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.01184178, 0.0, 0.0, 0.00592089, 0.0, 0.0, 0.00592089, 0.0, 0.0],
                    [0.0, 0.59647691, 0.0, 0.0, -0.29823846, 0.23338967, 0.0, -0.29823846, -0.23338967],
                    [0.0, 0.0, 0.38876074, 0.0, 0.17693874, -0.19438037, 0.0, -0.17693874, -0.19438037],
                    [0.00592089, 0.0, 0.0, -0.00473124, 0.0, 0.0, -0.00118965, 0.0, 0.0],
                    [0.0, -0.29823846, 0.17693874, 0.0, 0.32637561, -0.20516421, 0.0, -0.02813715, 0.02822547],
                    [0.0, 0.23338967, -0.19438037, 0.0, -0.20516421, 0.18496249, 0.0, -0.02822547, 0.00941788],
                    [0.00592089, 0.0, 0.0, -0.00118965, 0.0, 0.0, -0.00473124, 0.0, 0.0],
                    [0.0, -0.29823846, -0.17693874, 0.0, -0.02813715, -0.02822547, 0.0, 0.32637561, 0.20516421],
                    [0.0, -0.23338967, -0.19438037, 0.0, 0.02822547, 0.00941788, 0.0, 0.20516421, 0.18496249],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.21084677,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130360,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.014436297067, 0.0, 0.005622513341, -0.007218148534, 0.0, -0.005622513341, -0.007218148534]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.21275859,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004646780601,  # vcc only
            "CISD CORRELATION ENERGY": -0.20576009250440,  # vcc
            "QCISD CORRELATION ENERGY": -0.21351003301667,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00505349,  # vcc only
            "LCCD CORRELATION ENERGY": -0.2167878305,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401306050,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.018147039011, 0.0, 0.007823362162, -0.009073519505, 0.0, -0.007823362162, -0.009073519505]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2185061347,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0404661,  # vcc
            "CCD CORRELATION ENERGY": -0.2119966470870,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0396775,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, 0.015575676], [0.0, 0.0064826668, -0.007787838], [0.0, -0.0064826668, -0.007787838]]
            ),
            "CCD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01343195, 0.0, 0.0, 0.00671598, 0.0, 0.0, 0.00671598, 0.0, 0.0],
                    [0.0, 0.58883226, 0.0, 0.0, -0.29441613, 0.23106683, 0.0, -0.29441613, -0.23106683],
                    [0.0, 0.0, 0.38825029, 0.0, 0.17519478, -0.19412514, 0.0, -0.17519478, -0.19412514],
                    [0.00671598, 0.0, 0.0, -0.00550282, 0.0, 0.0, -0.00121316, 0.0, 0.0],
                    [0.0, -0.29441613, 0.17519478, 0.0, 0.3235041, -0.20313082, 0.0, -0.02908797, 0.02793601],
                    [0.0, 0.23106683, -0.19412514, 0.0, -0.20313082, 0.1835696, 0.0, -0.02793601, 0.01055554],
                    [0.00671598, 0.0, 0.0, -0.00121316, 0.0, 0.0, -0.00550282, 0.0, 0.0],
                    [0.0, -0.29441613, -0.17519478, 0.0, -0.02908797, -0.02793601, 0.0, 0.3235041, 0.20313082],
                    [0.0, -0.23106683, -0.19412514, 0.0, 0.02793601, 0.01055554, 0.0, 0.20313082, 0.1835696],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.213298055172,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.039907245914335,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.016842165003,
                    0.0,
                    0.007150136873,
                    -0.008421082502,
                    0.0,
                    -0.007150136873,
                    -0.008421082502,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01452417, 0.0, 0.0, 0.00726209, 0.0, 0.0, 0.00726209, 0.0, 0.0],
                    [0.0, 0.58571417, 0.0, 0.0, -0.29285708, 0.23028954, 0.0, -0.29285708, -0.23028954],
                    [0.0, 0.0, 0.38565345, 0.0, 0.17448698, -0.19282672, 0.0, -0.17448698, -0.19282672],
                    [0.00726209, 0.0, 0.0, -0.00599674, 0.0, 0.0, -0.00126535, 0.0, 0.0],
                    [0.0, -0.29285708, 0.17448698, 0.0, 0.32200974, -0.20238826, 0.0, -0.02915266, 0.02790128],
                    [0.0, 0.23028954, -0.19282672, 0.0, -0.20238826, 0.18240136, 0.0, -0.02790128, 0.01042537],
                    [0.00726209, 0.0, 0.0, -0.00126535, 0.0, 0.0, -0.00599674, 0.0, 0.0],
                    [0.0, -0.29285708, -0.17448698, 0.0, -0.02915266, -0.02790128, 0.0, 0.32200974, 0.20238826],
                    [0.0, -0.23028954, -0.19282672, 0.0, 0.02790128, 0.01042537, 0.0, 0.20238826, 0.18240136],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005298735528,  # vcc
            "(T) CORRECTION ENERGY": -0.00516659,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01904285993124],
                    [0.00000000000000, 0.00824911606964, -0.00952142996562],
                    [0.00000000000000, -0.00824911606964, -0.00952142996562],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.01642193739, 0.0, 0.0, 0.008210968695, 0.0, 0.0, 0.008210968695, 0.0, 0.0],
                    [
                        0.0,
                        0.584003042573,
                        0.0,
                        0.0,
                        -0.292001521287,
                        0.230361256906,
                        0.0,
                        -0.292001521287,
                        -0.230361256906,
                    ],
                    [
                        0.0,
                        0.0,
                        0.38222875445,
                        0.0,
                        0.174417714376,
                        -0.191114377225,
                        0.0,
                        -0.174417714376,
                        -0.191114377225,
                    ],
                    [0.008210968695, 0.0, 0.0, -0.006834745201, 0.0, 0.0, -0.001376223494, 0.0, 0.0],
                    [
                        0.0,
                        -0.292001521287,
                        0.174417714376,
                        0.0,
                        0.320849017325,
                        -0.202389485641,
                        0.0,
                        -0.028847496039,
                        0.027971771265,
                    ],
                    [
                        0.0,
                        0.230361256906,
                        -0.191114377225,
                        0.0,
                        -0.202389485641,
                        0.181208918903,
                        0.0,
                        -0.027971771265,
                        0.009905458321,
                    ],
                    [0.008210968695, 0.0, 0.0, -0.001376223494, 0.0, 0.0, -0.006834745201, 0.0, 0.0],
                    [
                        0.0,
                        -0.292001521287,
                        -0.174417714376,
                        0.0,
                        -0.028847496039,
                        -0.027971771265,
                        0.0,
                        0.320849017325,
                        0.202389485641,
                    ],
                    [
                        0.0,
                        -0.230361256906,
                        -0.191114377225,
                        0.0,
                        0.027971771265,
                        0.009905458321,
                        0.0,
                        0.202389485641,
                        0.181208918903,
                    ],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.218598725534,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.218597572977,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.218253889761,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.21827269362849,  # vcc
            "CCSDT CORRELATION ENERGY": -0.21884856037681,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": 0.0012856903,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2180560836,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0405122800,  # occ
            "PBE TOTAL ENERGY": -55.83097248811423,  # psi 99,590
            "PBE TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.000000000000, 0.000000000000, 0.029614373344],
                    [-0.000000000000, 0.015644026348, -0.014807315411],
                    [0.000000000000, -0.015644026348, -0.014807315411],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.9072559501566744,  # psi 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [0.00000000e00, 3.15544362e-30, 1.99929859e-02],
                    [-3.19390336e-19, 1.04320800e-02, -9.99655055e-03],
                    [3.19390336e-19, -1.04320800e-02, -9.99655055e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.8744982972056121,  # psi 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # psi 99,590
                [
                    [-7.88860905e-31, 0.00000000e00, 2.02715602e-02],
                    [-3.25473520e-19, 1.06307719e-02, -1.01358376e-02],
                    [3.25473520e-19, -1.06307719e-02, -1.01358376e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.060478115157,
            "MP2 SINGLES ENERGY": -0.000694049865,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002004909679,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    # switches sign from unkn ref
                    0.000000000000000,
                    0.000000000000000,
                    0.013594741747853,
                    0.000000000000000,
                    0.000000000000000,
                    -0.013127629532095,
                    0.000000000000000,
                    0.010308255599051,
                    -0.000233556107879,
                    0.000000000000000,
                    -0.010308255599051,
                    -0.000233556107879,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [
                        3.87098460e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.45378253e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.77896598e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.31078241e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.65908217e-04,
                        4.34462348e-04,
                        0.00000000e00,
                        2.65908217e-04,
                        -4.34462348e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.17564784e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15517211e-01,
                        0.00000000e00,
                        1.14220963e-03,
                        -1.02378661e-03,
                        0.00000000e00,
                        -1.14220963e-03,
                        -1.02378661e-03,
                    ],
                    [
                        -4.45378253e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.80247034e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -4.31078241e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.71699407e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.03105828e-04,
                        8.16927983e-05,
                        0.00000000e00,
                        -2.03105828e-04,
                        -8.16927983e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.15517211e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.15737900e-01,
                        0.00000000e00,
                        -4.73512607e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        4.73512607e-05,
                        -1.10344685e-04,
                    ],
                    [
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.40034143e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.28328638e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.65908217e-04,
                        1.14220963e-03,
                        0.00000000e00,
                        -2.03105828e-04,
                        -4.73512607e-05,
                        0.00000000e00,
                        4.11395088e-01,
                        -8.05506756e-04,
                        0.00000000e00,
                        -4.11457890e-01,
                        -2.89351610e-04,
                    ],
                    [
                        0.00000000e00,
                        4.34462348e-04,
                        -1.02378661e-03,
                        0.00000000e00,
                        8.16927983e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        -8.05506756e-04,
                        -4.62479868e-03,
                        0.00000000e00,
                        2.89351610e-04,
                        5.75892997e-03,
                    ],
                    [
                        2.91398962e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74343907e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.28328638e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.40034143e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.65908217e-04,
                        -1.14220963e-03,
                        0.00000000e00,
                        -2.03105828e-04,
                        4.73512607e-05,
                        0.00000000e00,
                        -4.11457890e-01,
                        2.89351610e-04,
                        0.00000000e00,
                        4.11395088e-01,
                        8.05506756e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.34462348e-04,
                        -1.02378661e-03,
                        0.00000000e00,
                        -8.16927983e-05,
                        -1.10344685e-04,
                        0.00000000e00,
                        -2.89351610e-04,
                        5.75892997e-03,
                        0.00000000e00,
                        8.05506756e-04,
                        -4.62479868e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.075906990477,  # cfour
            "MP3 SINGLES ENERGY": -0.000916783719,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by psi fd
                [
                    [0.0, 0.0, 0.010131197027],
                    [0.0, 0.0, -0.009902977129],
                    [0.0, 0.012728956474, -0.000114109949],
                    [0.0, -0.012728956474, -0.000114109949],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.08034256,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000301596734,  # vcc only
            "CISD CORRELATION ENERGY": -0.08129169589155,  # vcc
            # "CISD CORRELATION ENERGY": -0.08142433,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.0834094914,  # p4n
            "LCCSD CORRELATION ENERGY": -0.0861427228,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.08620498,  # vcc != p4n
            "CCD CORRELATION ENERGY": -0.08115920440309,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.08357160616,
            "CCSD SINGLES ENERGY": -0.0011743271,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.00244892164,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005568141758,
                    0.0,
                    0.0,
                    -0.005430974166,
                    0.0,
                    0.014884143028,
                    -0.000068583796,
                    0.0,
                    -0.014884143028,
                    -0.000068583796,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.000713766189,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00481175567113],
                    [0.00000000000000, 0.00000000000000, -0.00475864799255],
                    [0.00000000000000, 0.01496337127168, -0.00002655383929],
                    [0.00000000000000, -0.01496337127168, -0.00002655383929],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.08463562959121,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0000399018,  # p4n
            "OLCCD CORRELATION ENERGY": -0.0862654609,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0024486744,  # occ
            "PBE TOTAL ENERGY": -26.046275584237,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [7.84827311e-37, -2.13273843e-19, -1.92100000e-03],
                    [4.63705881e-37, -1.26010313e-19, -1.13500000e-03],
                    [-1.12670315e-35, 2.60500000e-02, 1.52800000e-03],
                    [1.00184983e-35, -2.60500000e-02, 1.52800000e-03],
                ]
            ),
            "B3LYP TOTAL ENERGY": -26.113222717203,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.96104690e-36, 5.32907052e-19, 4.80000000e-03],
                    [2.94483876e-36, -8.00248756e-19, -7.20800000e-03],
                    [-8.25273903e-36, 1.89960000e-02, 1.20400000e-03],
                    [7.26894717e-36, -1.89960000e-02, 1.20400000e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -26.088528482841,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.85523208e-36, 5.04152275e-19, 4.54100000e-03],
                    [2.84351800e-36, -7.72715225e-19, -6.96000000e-03],
                    [-8.37121895e-36, 1.92800000e-02, 1.21000000e-03],
                    [7.38252447e-36, -1.92800000e-02, 1.21000000e-03],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15949744108346664,
            "MP2 SINGLES ENERGY": -0.0028296307982793997,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03541709278508698,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.025609525826,
                    0.0,
                    0.013506941035,
                    -0.012804762913,
                    0.0,
                    -0.013506941035,
                    -0.012804762913,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.02208471, 0.0, 0.0, 0.01104236, 0.0, 0.0, 0.01104236, 0.0, 0.0],
                    [0.0, 0.62483805, 0.0, 0.0, -0.31241902, 0.24820063, 0.0, -0.31241902, -0.24820063],
                    [0.0, 0.0, 0.39861928, 0.0, 0.19070358, -0.19930964, 0.0, -0.19070358, -0.19930964],
                    [0.01104236, 0.0, 0.0, -0.00999006, 0.0, 0.0, -0.0010523, 0.0, 0.0],
                    [0.0, -0.31241902, 0.19070358, 0.0, 0.33943262, -0.2194521, 0.0, -0.0270136, 0.02874852],
                    [0.0, 0.24820063, -0.19930964, 0.0, -0.2194521, 0.19041148, 0.0, -0.02874852, 0.00889816],
                    [0.01104236, 0.0, 0.0, -0.0010523, 0.0, 0.0, -0.00999006, 0.0, 0.0],
                    [0.0, -0.31241902, -0.19070358, 0.0, -0.0270136, -0.02874852, 0.0, 0.33943262, 0.2194521],
                    [0.0, -0.24820063, -0.19930964, 0.0, 0.02874852, 0.00889816, 0.0, 0.2194521, 0.19041148],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.17571623051,  # cfour
            "MP3 SINGLES ENERGY": -0.003236109562,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by psi fd
                [
                    [0.00000000000000, 0.00000000000000, 0.02731056234441],
                    [0.00000000000000, 0.01453084545981, -0.01365528117220],
                    [0.00000000000000, -0.01453084545981, -0.01365528117220],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.17767316,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.003373917694,  # vcc only
            "CISD CORRELATION ENERGY": -0.17186174301557,  # vcc
            # "CISD CORRELATION ENERGY": -0.1723668643052676,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.1791714105,  # p4n
            "LCCSD CORRELATION ENERGY": -0.1830545845,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.18308771, # vcc != p4n
            "CCD CORRELATION ENERGY": -0.17475513572639,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.178236032911,
            "CCSD SINGLES ENERGY": -0.00327524740575,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033982707798170,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.029273628227,
                    0.0,
                    0.015808308241,
                    -0.014636814114,
                    0.0,
                    -0.015808308241,
                    -0.014636814114,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.003901085777,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03117381407549],
                    [0.00000000000000, 0.01681897058340, -0.01558690703775],
                    [0.00000000000000, -0.01681897058340, -0.0155869070377],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.18258437583017,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0033018315,  # p4n
            "OLCCD CORRELATION ENERGY": -0.1825194982,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0344689234,  # occ
            "PBE TOTAL ENERGY": -55.813318056978,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.94106602e-19, 2.94106602e-19, 3.56590000e-02],
                    [1.47049177e-19, 1.96000000e-02, -1.78290000e-02],
                    [1.47049177e-19, -1.96000000e-02, -1.78290000e-02],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.888559899085,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.19134533e-19, 2.19134533e-19, 2.65690000e-02],
                    [1.09571390e-19, 1.47410000e-02, -1.32850000e-02],
                    [1.09571390e-19, -1.47410000e-02, -1.32850000e-02],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.855798924171,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.21336683e-19, 2.21336683e-19, 2.68360000e-02],
                    [1.10668341e-19, 1.49380000e-02, -1.34180000e-02],
                    [1.10668341e-19, -1.49380000e-02, -1.34180000e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.2005395272,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178535,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.0138883429,
                    0.0,
                    0.005389090661,
                    -0.00694417145,
                    0.0,
                    -0.005389090661,
                    -0.00694417145,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.01197655, 0.0, 0.0, 0.00598828, 0.0, 0.0, 0.00598828, 0.0, 0.0],
                    [0.0, 0.59600884, 0.0, 0.0, -0.29800442, 0.23326184, 0.0, -0.29800442, -0.23326184],
                    [0.0, 0.0, 0.38838238, 0.0, 0.17666871, -0.19419119, 0.0, -0.17666871, -0.19419119],
                    [0.00598828, 0.0, 0.0, -0.0047771, 0.0, 0.0, -0.00121118, 0.0, 0.0],
                    [0.0, -0.29800442, 0.17666871, 0.0, 0.32604541, -0.20496528, 0.0, -0.02804099, 0.02829657],
                    [0.0, 0.23326184, -0.19419119, 0.0, -0.20496528, 0.18480667, 0.0, -0.02829657, 0.00938452],
                    [0.00598828, 0.0, 0.0, -0.00121118, 0.0, 0.0, -0.0047771, 0.0, 0.0],
                    [0.0, -0.29800442, -0.17666871, 0.0, -0.02804099, -0.02829657, 0.0, 0.32604541, 0.20496528],
                    [0.0, -0.23326184, -0.19419119, 0.0, 0.02829657, 0.00938452, 0.0, 0.20496528, 0.18480667],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.215858882347,  # cfour
            "MP3 SINGLES ENERGY": -0.003348697377,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.014669237518],
                    [0.0, 0.005743113908, -0.007334618759],
                    [0.0, -0.005743113908, -0.007334618759],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.21748938,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004777946072,  # vcc only
            "CISD CORRELATION ENERGY": -0.20988111766307,  # vcc
            # "CISD CORRELATION ENERGY": -0.21038651,  # detci != vcc ???
            "LCCD CORRELATION ENERGY": -0.2191039411,  # p4n
            "LCCSD CORRELATION ENERGY": -0.2231241199,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.22316121, # vcc != p4n ???
            "CCD CORRELATION ENERGY": -0.21415436056607,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.217849506326,
            "CCSD SINGLES ENERGY": -0.00338286103325,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.039891470497466,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.016833254665,
                    0.0,
                    0.007144029475,
                    -0.008416627332,
                    0.0,
                    -0.007144029475,
                    -0.008416627332,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.005233938447,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01905797562047],
                    [0.00000000000000, 0.00825727647494, -0.00952898781024],
                    [0.00000000000000, -0.00825727647494, -0.00952898781024],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.22345631762464,  # vcc
            "OLCCD REFERENCE CORRECTION ENERGY": -0.0033240178,  # p4n
            "OLCCD CORRELATION ENERGY": -0.2226657917,  # p4n
            "OLCCD SAME-SPIN CORRELATION ENERGY": -0.0405122800,  # occ
            "PBE TOTAL ENERGY": -55.829414170216,  # nwc 99,590
            "PBE TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-2.43250961e-19, 2.43250961e-19, 2.94930000e-02],
                    [1.21621356e-19, 1.55340000e-02, -1.47460000e-02],
                    [1.21621356e-19, -1.55340000e-02, -1.47460000e-02],
                ]
            ),
            "B3LYP TOTAL ENERGY": -55.905608800925,  # nwc 99,590
            "B3LYP TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.63082808e-19, 1.63082808e-19, 1.97730000e-02],
                    [8.15372799e-20, 1.02630000e-02, -9.88600000e-03],
                    [8.15372799e-20, -1.02630000e-02, -9.88600000e-03],
                ]
            ),
            "B3LYP5 TOTAL ENERGY": -55.872832273661,  # nwc 99,590
            "B3LYP5 TOTAL GRADIENT": np.array(  # nwc 99,590
                [
                    [-1.65334444e-19, 1.65334444e-19, 2.00460000e-02],
                    [8.26672220e-20, 1.04580000e-02, -1.00230000e-02],
                    [8.26672220e-20, -1.04580000e-02, -1.00230000e-02],
                ]
            ),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-CONV  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.201627516796,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0534895025483176,
            "MP2 TOTAL GRADIENT": np.array(
                [  # fnocc findif-5 fc pk+conv
                    0.00000000000000,
                    0.00000000000000,
                    0.00317450456474,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00317450456474,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.00183193, 0.0, 0.0, 0.00183193, 0.0, 0.0],
                    [0.0, -0.00183193, 0.0, 0.0, 0.00183193, 0.0],
                    [0.0, 0.0, 0.64212587, 0.0, 0.0, -0.64212587],
                    [0.00183193, 0.0, 0.0, -0.00183193, 0.0, 0.0],
                    [0.0, 0.00183193, 0.0, 0.0, -0.00183193, 0.0],
                    [0.0, 0.0, -0.64212587, 0.0, 0.0, 0.64212587],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.20453611,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918710,  # fnocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000579954803, 0.0, 0.0, 0.000579954803]).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.20670909,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.002027930587,  # vcc
            "CISD CORRELATION ENERGY": -0.20039201434963,  # vcc
            "QCISD CORRELATION ENERGY": -0.20699674383631,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00180789,  # vcc
            "FCI CORRELATION ENERGY": -0.2092292951,  # detci
            "LCCD CORRELATION ENERGY": -0.2079585027,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.047635656759,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002535502057, 0.0, 0.0, -0.002535502057]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2087915976,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.047754723454,  # fnocc
            "CCD CORRELATION ENERGY": -0.20622552201818,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04781602,  # vcc
            "CCD TOTAL GRADIENT": np.array([[0.0, 0.0, 0.0011776783], [0.0, 0.0, -0.0011776783]]),  # vcc
            "CCSD CORRELATION ENERGY": -0.2068152041,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.0478712079,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.002335204281,
                    0.0,
                    0.0,
                    -0.002335204281,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.00134759, 0.0, 0.0, 0.00134759, 0.0, 0.0],
                    [0.0, -0.00134759, 0.0, 0.0, 0.00134759, 0.0],
                    [0.0, 0.0, 0.63866268, 0.0, 0.0, -0.63866268],
                    [0.00134759, 0.0, 0.0, -0.00134759, 0.0, 0.0],
                    [0.0, 0.00134759, 0.0, 0.0, -0.00134759, 0.0],
                    [0.0, 0.0, -0.63866268, 0.0, 0.0, 0.63866268],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.002040855770,  # vcc
            "(T) CORRECTION ENERGY": -0.0019205007159748158,
            "CCSD(T) TOTAL GRADIENT": np.array([[0.0, 0.0, 0.00342858327943], [0.0, 0.0, -0.00342858327943]]),  # ncc fd
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [-0.001978535996, 0.0, 0.0, 0.001978535996, 0.0, 0.0],
                    [0.0, -0.001978535996, 0.0, 0.0, 0.001978535996, 0.0],
                    [0.0, 0.0, 0.637319868539, 0.0, 0.0, -0.637319868539],
                    [0.001978535996, 0.0, 0.0, -0.001978535996, 0.0, 0.0],
                    [0.0, 0.001978535996, 0.0, -0.0, -0.001978535996, 0.0],
                    [0.0, 0.0, -0.637319868539, 0.0, 0.0, 0.637319868539],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.001944208736203,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [[0.00000000000, 0.00000000000, 0.00341625320310], [0.00000000000, 0.00000000000, -0.00341625320310]]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.208822533905,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00350581687383], [0.0, 0.0, -0.00350581687383]]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-2.02328225e-03, 1.66833702e-08, 2.17609176e-08, 2.02328225e-03, -1.66833702e-08, -2.17609176e-08],
                    [1.66833702e-08, -2.02328515e-03, 1.74087341e-08, -1.66833702e-08, 2.02328515e-03, -1.74087341e-08],
                    [2.17609176e-08, 1.74087341e-08, 6.37250530e-01, -2.17609176e-08, -1.74087341e-08, -6.37250530e-01],
                    [2.02328225e-03, -1.66833702e-08, -2.17609176e-08, -2.02328225e-03, 1.66833702e-08, 2.17609176e-08],
                    [-1.66833702e-08, 2.02328515e-03, -1.74087341e-08, 1.66833702e-08, -2.02328515e-03, 1.74087341e-08],
                    [-2.17609176e-08, -1.74087341e-08, -6.37250530e-01, 2.17609176e-08, 1.74087341e-08, 6.37250530e-01],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.208825939324,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00350694748535], [0.0, 0.0, -0.00350694748535]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.208693156430,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fc STAR
                [[0.0, 0.0, 0.00345167960987], [0.0, 0.0, -0.00345167960987]]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.20869193974177,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [[0.0, 0.0, 0.00344911055208], [0.0, 0.0, -0.00344911055208]]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.001990549473, 0.0, 0.0, 0.001990549473, -0.0, -0.0],
                    [0.0, -0.001990539318, 0.0, -0.0, 0.001990539318, -0.0],
                    [0.0, 0.0, 0.637308538355, -0.0, -0.0, -0.637308538355],
                    [0.001990549473, -0.0, -0.0, -0.001990549473, 0.0, 0.0],
                    [-0.0, 0.001990539318, -0.0, 0.0, -0.001990539318, 0.0],
                    [-0.0, -0.0, -0.637308538355, 0.0, 0.0, 0.637308538355],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.208825198347707,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.003519024898117,
                    0.000000000000000,
                    0.000000000000000,
                    -0.003519024898117,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [-0.00203074, 0.0, 0.0, 0.00203074, 0.0, 0.0],
                    [0.0, -0.00203074, 0.0, 0.0, 0.00203074, 0.0],
                    [0.0, 0.0, 0.63717213, 0.0, 0.0, -0.63717213],
                    [0.00203074, 0.0, 0.0, -0.00203074, 0.0, 0.0],
                    [0.0, 0.00203074, 0.0, 0.0, -0.00203074, 0.0],
                    [0.0, 0.0, -0.63717213, 0.0, 0.0, 0.63717213],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000379506649993,  # ncc
            "(Q) CORRECTION ENERGY": -0.000413051703749,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.209218171097884,  # ncc
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.2194081478,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.055833980855745646,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.010245839621,
                    0.0,
                    0.005893268945,
                    -0.00512291981,
                    0.0,
                    -0.005893268945,
                    -0.00512291981,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -9.24437570e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.20231256e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.60115628e-01,
                        2.82410191e-01,
                        0.00000000e00,
                        -3.60115628e-01,
                        -2.82410191e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.67582875e-01,
                        0.00000000e00,
                        2.16920929e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        -2.16920929e-01,
                        -2.33791438e-01,
                    ],
                    [
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.36961760e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.52570200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.60115628e-01,
                        2.16920929e-01,
                        0.00000000e00,
                        3.88438352e-01,
                        -2.49665560e-01,
                        0.00000000e00,
                        -2.83227242e-02,
                        3.27446311e-02,
                    ],
                    [
                        0.00000000e00,
                        2.82410191e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        -2.49665560e-01,
                        2.24169862e-01,
                        0.00000000e00,
                        -3.27446311e-02,
                        9.62157610e-03,
                    ],
                    [
                        4.62218780e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.52570200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.36961760e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.60115628e-01,
                        -2.16920929e-01,
                        0.00000000e00,
                        -2.83227242e-02,
                        -3.27446311e-02,
                        0.00000000e00,
                        3.88438352e-01,
                        2.49665560e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.82410191e-01,
                        -2.33791438e-01,
                        0.00000000e00,
                        3.27446311e-02,
                        9.62157610e-03,
                        0.00000000e00,
                        2.49665560e-01,
                        2.24169862e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.224155670726,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.051069118695,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004593346123, 0.0, 0.002925016535, -0.002296673061, 0.0, -0.002925016535, -0.002296673061]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.22737281,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.005605447265,  # vcc
            "CISD CORRELATION ENERGY": -0.21764746560900,  # vcc
            "QCISD CORRELATION ENERGY": -0.22775040212176,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00485806,  # vcc
            "LCCD CORRELATION ENERGY": -0.2296135965,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.049154543318,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.009143629163, 0.0, 0.005604924417, -0.004571814581, 0.0, -0.005604924417, -0.004571814581]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2318316308,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.049659952324,  # fnocc
            "CCD CORRELATION ENERGY": -0.22578168712640,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.04913759,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc findif in psi
                [[0.0, 0.0, 0.00677294], [0.0, 0.00429381, -0.00338647], [0.0, -0.00429381, -0.00338647]]
            ),
            "CCSD CORRELATION ENERGY": -0.22717334532791256,  # mrcc
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.049398348010672,
            "CCSD TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.00811816], [0.0, 0.00498838, -0.00405908], [0.0, -0.00498838, -0.00405908]]
            ),
            "CCSD TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00732388, 0.0, 0.0, 0.00366194, 0.0, 0.0, 0.00366194, 0.0, 0.0],
                    [0.0, 0.71380971, 0.0, 0.0, -0.35690486, 0.27918049, 0.0, -0.35690486, -0.27918049],
                    [0.0, 0.0, 0.47324568, 0.0, 0.21526997, -0.23662284, 0.0, -0.21526997, -0.23662284],
                    [0.00366194, 0.0, 0.0, -0.00357321, 0.0, 0.0, -0.00008873, 0.0, 0.0],
                    [0.0, -0.35690486, 0.21526997, 0.0, 0.3871942, -0.24722523, 0.0, -0.03028935, 0.03195526],
                    [0.0, 0.27918049, -0.23662284, 0.0, -0.24722523, 0.22465158, 0.0, -0.03195526, 0.01197127],
                    [0.00366194, 0.0, 0.0, -0.00008873, 0.0, 0.0, -0.00357321, 0.0, 0.0],
                    [0.0, -0.35690486, -0.21526997, 0.0, -0.03028935, -0.03195526, 0.0, 0.3871942, 0.24722523],
                    [0.0, -0.27918049, -0.23662284, 0.0, 0.03195526, 0.01197127, 0.0, 0.24722523, 0.22465158],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005613685607,  # vcc
            "(T) CORRECTION ENERGY": -0.00521238,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01079337838744],
                    [0.00000000000000, 0.00642184654717, -0.00539668919372],
                    [0.00000000000000, -0.00642184654717, -0.00539668919372],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -0.009738397928,
                        0.0,
                        0.0,
                        0.004869198964,
                        0.0,
                        0.0,
                        0.004869198964,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.711718022073,
                        0.0,
                        0.0,
                        -0.355859011036,
                        0.279305472354,
                        0.0,
                        -0.355859011036,
                        -0.279305472354,
                    ],
                    [
                        0.0,
                        0.0,
                        0.469065457749,
                        0.0,
                        0.215200115829,
                        -0.234532728875,
                        0.0,
                        -0.215200115829,
                        -0.234532728875,
                    ],
                    [
                        0.004869198964,
                        0.0,
                        0.0,
                        -0.004677737322,
                        0.0,
                        0.0,
                        -0.000191461642,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.355859011036,
                        0.215200115829,
                        0.0,
                        0.385717829378,
                        -0.247252794091,
                        0.0,
                        -0.029858818342,
                        0.032052678263,
                    ],
                    [
                        0.0,
                        0.279305472354,
                        -0.234532728875,
                        0.0,
                        -0.247252794091,
                        0.223154182027,
                        0.0,
                        -0.032052678263,
                        0.011378546847,
                    ],
                    [
                        0.004869198964,
                        0.0,
                        0.0,
                        -0.000191461642,
                        0.0,
                        0.0,
                        -0.004677737322,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.355859011036,
                        -0.215200115829,
                        0.0,
                        -0.029858818342,
                        -0.032052678263,
                        0.0,
                        0.385717829378,
                        0.247252794091,
                    ],
                    [
                        0.0,
                        -0.279305472354,
                        -0.234532728875,
                        0.0,
                        0.032052678263,
                        0.011378546847,
                        0.0,
                        0.247252794091,
                        0.223154182027,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.005209699406797,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [
                    [0.00000000000000, 0.00000000000000, 0.01072309503834],
                    [0.00000000000000, 0.00639691786956, -0.00536154751917],
                    [0.00000000000000, -0.00639691786956, -0.00536154751917],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.2327434892835214,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01100789973884],
                    [0.00000000000000, 0.00656715463790, -0.00550394986942],
                    [0.00000000000000, -0.00656715463790, -0.00550394986942],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00993141, 0.0, 0.0, 0.00496571, 0.0, 0.0, 0.00496571, 0.0, 0.0],
                    [0.0, 0.71162074, 0.0, 0.0, -0.35581037, 0.27934255, 0.0, -0.35581037, -0.27934255],
                    [0.0, 0.0, 0.4686399, 0.0, 0.21523949, -0.23431995, 0.0, -0.21523949, -0.23431995],
                    [0.00496571, 0.0, 0.0, -0.00477664, 0.0, 0.0, -0.00018907, 0.0, 0.0],
                    [0.0, -0.35581037, 0.21523949, 0.0, 0.38585314, -0.24729102, 0.0, -0.03004277, 0.03205153],
                    [0.0, 0.27934255, -0.23431995, 0.0, -0.24729102, 0.22301133, 0.0, -0.03205153, 0.01130862],
                    [0.00496571, 0.0, 0.0, -0.00018907, 0.0, 0.0, -0.00477664, 0.0, 0.0],
                    [0.0, -0.35581037, -0.21523949, 0.0, -0.03004277, -0.03205153, 0.0, 0.38585314, 0.24729102],
                    [0.0, -0.27934255, -0.23431995, 0.0, 0.03205153, 0.01130862, 0.0, 0.24729102, 0.22301133],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.23274736711556443,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01100335606981],
                    [0.00000000000000, 0.00656630481721, -0.00550167803491],
                    [0.00000000000000, -0.00656630481721, -0.00550167803491],
                ]
            ),
            "CCSDT-1B TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00992728, 0.0, 0.0, 0.00496364, 0.0, 0.0, 0.00496364, 0.0, 0.0],
                    [0.0, 0.71160797, 0.0, 0.0, -0.35580398, 0.27933599, 0.0, -0.35580398, -0.27933599],
                    [0.0, 0.0, 0.46863418, 0.0, 0.21524634, -0.23431709, 0.0, -0.21524634, -0.23431709],
                    [0.00496364, 0.0, 0.0, -0.0047753, 0.0, 0.0, -0.00018834, 0.0, 0.0],
                    [0.0, -0.35580398, 0.21524634, 0.0, 0.38559464, -0.24729116, 0.0, -0.02979065, 0.03204483],
                    [0.0, 0.27933599, -0.23431709, 0.0, -0.24729116, 0.22300767, 0.0, -0.03204483, 0.01130942],
                    [0.00496364, 0.0, 0.0, -0.00018834, 0.0, 0.0, -0.0047753, 0.0, 0.0],
                    [0.0, -0.35580398, -0.21524634, 0.0, -0.02979065, -0.03204483, 0.0, 0.38559464, 0.24729116],
                    [0.0, -0.27933599, -0.23431709, 0.0, 0.03204483, 0.01130942, 0.0, 0.24729116, 0.22300767],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.232250104269,  # ecc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01067153233521],
                    [0.00000000000000, 0.00637295285347, -0.00533576616760],
                    [0.00000000000000, -0.00637295285347, -0.00533576616760],
                ]
            ),
            "CCSDT-2 TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
            "CCSDT-3 CORRELATION ENERGY": -0.23227192730338686,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.01069509383085],
                    [0.00000000000000, 0.00638565480574, -0.00534754691542],
                    [0.00000000000000, -0.00638565480574, -0.00534754691542],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.00964914, 0.0, 0.0, 0.00482457, 0.0, 0.0, 0.00482457, 0.0, 0.0],
                    [0.0, 0.7116635, 0.0, 0.0, -0.35583175, 0.27924979, 0.0, -0.35583175, -0.27924979],
                    [0.0, 0.0, 0.46915589, 0.0, 0.21525082, -0.23457795, 0.0, -0.21525082, -0.23457795],
                    [0.00482457, 0.0, 0.0, -0.00464265, 0.0, 0.0, -0.00018191, 0.0, 0.0],
                    [0.0, -0.35583175, 0.21525082, 0.0, 0.38572295, -0.24725031, 0.0, -0.0298912, 0.03199949],
                    [0.0, 0.27924979, -0.23457795, 0.0, -0.24725031, 0.22316784, 0.0, -0.03199949, 0.01141011],
                    [0.00482457, 0.0, 0.0, -0.00018191, 0.0, 0.0, -0.00464265, 0.0, 0.0],
                    [0.0, -0.35583175, -0.21525082, 0.0, -0.0298912, -0.03199949, 0.0, 0.38572295, 0.24725031],
                    [0.0, -0.27924979, -0.23457795, 0.0, 0.03199949, 0.01141011, 0.0, 0.24725031, 0.22316784],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.232619797476334,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.010883725577495,
                    0.000000000000000,
                    0.006489002186983,
                    -0.005441862788749,
                    0.000000000000000,
                    -0.006489002186983,
                    -0.005441862788749,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -9.81991250e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.11412515e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.55706258e-01,
                        2.79218906e-01,
                        0.00000000e00,
                        -3.55706258e-01,
                        -2.79218906e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.68988233e-01,
                        0.00000000e00,
                        2.15192437e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        -2.15192437e-01,
                        -2.34494116e-01,
                    ],
                    [
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.72159190e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.88364300e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55706258e-01,
                        2.15192437e-01,
                        0.00000000e00,
                        3.85602165e-01,
                        -2.47205672e-01,
                        0.00000000e00,
                        -2.98959074e-02,
                        3.20132343e-02,
                    ],
                    [
                        0.00000000e00,
                        2.79218906e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        -2.47205672e-01,
                        2.23077632e-01,
                        0.00000000e00,
                        -3.20132343e-02,
                        1.14164840e-02,
                    ],
                    [
                        4.90995620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.88364300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -4.72159190e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.55706258e-01,
                        -2.15192437e-01,
                        0.00000000e00,
                        -2.98959074e-02,
                        -3.20132343e-02,
                        0.00000000e00,
                        3.85602165e-01,
                        2.47205672e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.79218906e-01,
                        -2.34494116e-01,
                        0.00000000e00,
                        3.20132343e-02,
                        1.14164840e-02,
                        0.00000000e00,
                        2.47205672e-01,
                        2.23077632e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000288788062660,  # ncc
            "(Q) CORRECTION ENERGY": -0.000503838444143,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.233074721244323,  # ncc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24515185206,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126410,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00033347691,
                    0.0,
                    -0.00056224437,
                    -0.000166738455,
                    0.0,
                    0.00056224437,
                    -0.000166738455,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        -3.00880800e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        7.00166972e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.50083486e-01,
                        2.71180080e-01,
                        0.00000000e00,
                        -3.50083486e-01,
                        -2.71180080e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.59237517e-01,
                        0.00000000e00,
                        2.06356325e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        -2.06356325e-01,
                        -2.29618758e-01,
                    ],
                    [
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.21172300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.71612700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.50083486e-01,
                        2.06356325e-01,
                        0.00000000e00,
                        3.80842059e-01,
                        -2.38768202e-01,
                        0.00000000e00,
                        -3.07585729e-02,
                        3.24118773e-02,
                    ],
                    [
                        0.00000000e00,
                        2.71180080e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        -2.38768202e-01,
                        2.19990798e-01,
                        0.00000000e00,
                        -3.24118773e-02,
                        9.62796070e-03,
                    ],
                    [
                        1.50440400e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.71612700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.21172300e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.50083486e-01,
                        -2.06356325e-01,
                        0.00000000e00,
                        -3.07585729e-02,
                        -3.24118773e-02,
                        0.00000000e00,
                        3.80842059e-01,
                        2.38768202e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.71180080e-01,
                        -2.29618758e-01,
                        0.00000000e00,
                        3.24118773e-02,
                        9.62796070e-03,
                        0.00000000e00,
                        2.38768202e-01,
                        2.19990798e-01,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.24747734,  # fnocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.05558309,  # fnocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.006163797026, 0.0, -0.004106253706, 0.003081898513, 0.0, 0.004106253706, 0.003081898513]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.25063260,  # vcc
            "MP4(T) CORRECTION ENERGY": -0.007755518242,  # vcc
            "CISD CORRELATION ENERGY": -0.24016600239402,  # vcc
            "QCISD CORRELATION ENERGY": -0.25077731041751,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00687323,  # vcc
            "LCCD CORRELATION ENERGY": -0.2531942099,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.053842594884,  # fnocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.001471171285, 0.0, -0.001391395837, 0.000735585643, 0.0, 0.001391395837, 0.000735585643]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2553008820,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.054321637599,  # fnocc
            "CCD CORRELATION ENERGY": -0.24885097949989,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.05375108,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, -0.0038684502], [0.0, -0.0026251298, 0.0019342251], [0.0, 0.0026251298, 0.0019342251]]
            ),
            "CCSD CORRELATION ENERGY": -0.250330548844,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.054051928864870,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.002486174824,
                    0.0,
                    -0.001923330621,
                    0.001243087412,
                    0.0,
                    0.001923330621,
                    0.001243087412,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        2.24316860e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.92694865e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.46347433e-01,
                        2.67302412e-01,
                        0.00000000e00,
                        -3.46347433e-01,
                        -2.67302412e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.64489488e-01,
                        0.00000000e00,
                        2.03993780e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        -2.03993780e-01,
                        -2.32244744e-01,
                    ],
                    [
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.23261320e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.11028900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46347433e-01,
                        2.03993780e-01,
                        0.00000000e00,
                        3.79028763e-01,
                        -2.35648096e-01,
                        0.00000000e00,
                        -3.26813308e-02,
                        3.16543160e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67302412e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        -2.35648096e-01,
                        2.20278013e-01,
                        0.00000000e00,
                        -3.16543160e-02,
                        1.19667307e-02,
                    ],
                    [
                        -1.12158430e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.11028900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.23261320e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.46347433e-01,
                        -2.03993780e-01,
                        0.00000000e00,
                        -3.26813308e-02,
                        -3.16543160e-02,
                        0.00000000e00,
                        3.79028763e-01,
                        2.35648096e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67302412e-01,
                        -2.32244744e-01,
                        0.00000000e00,
                        3.16543160e-02,
                        1.19667307e-02,
                        0.00000000e00,
                        2.35648096e-01,
                        2.20278013e-01,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.007366570060,  # vcc
            "(T) CORRECTION ENERGY": -0.007096579721,
            "CCSD(T) TOTAL GRADIENT": np.array(  # ncc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00065323185711],
                    [0.00000000000000, -0.00035236679634, -0.00032661592856],
                    [0.00000000000000, 0.00035236679634, -0.00032661592856],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # ncc fd
                [
                    [
                        -0.000589405884,
                        0.0,
                        0.0,
                        0.000294702942,
                        0.0,
                        0.0,
                        0.000294702942,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.690892000177,
                        0.0,
                        0.0,
                        -0.345446000089,
                        0.267700971726,
                        0.0,
                        -0.345446000089,
                        -0.267700971726,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460262946323,
                        0.0,
                        0.204108329476,
                        -0.230131473162,
                        0.0,
                        -0.204108329476,
                        -0.230131473162,
                    ],
                    [
                        0.000294702942,
                        0.0,
                        0.0,
                        -0.000024281097,
                        0.0,
                        0.0,
                        -0.000270421845,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.345446000089,
                        0.204108329476,
                        0.0,
                        0.377790533687,
                        -0.235904650601,
                        0.0,
                        -0.032344533598,
                        0.031796321125,
                    ],
                    [
                        0.0,
                        0.267700971726,
                        -0.230131473162,
                        0.0,
                        -0.235904650601,
                        0.218826959667,
                        0.0,
                        -0.031796321125,
                        0.011304513494,
                    ],
                    [
                        0.000294702942,
                        0.0,
                        0.0,
                        -0.000270421845,
                        0.0,
                        0.0,
                        -0.000024281097,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.345446000089,
                        -0.204108329476,
                        0.0,
                        -0.032344533598,
                        -0.031796321125,
                        0.0,
                        0.377790533687,
                        0.235904650601,
                    ],
                    [
                        0.0,
                        -0.267700971726,
                        -0.230131473162,
                        0.0,
                        0.031796321125,
                        0.011304513494,
                        0.0,
                        0.235904650601,
                        0.218826959667,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.007017918524544,  # ncc
            "A-CCSD(T) TOTAL GRADIENT": np.array(  # ccenergy fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00057233082116],
                    [0.00000000000000, -0.00038770405767, -0.00028616541058],
                    [0.00000000000000, 0.00038770405767, -0.00028616541058],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.257751267313,  # ecc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00086210387602],
                    [0.00000000000000, -0.00023280605874, -0.00043105193801],
                    [0.00000000000000, 0.00023280605874, -0.00043105193801],
                ]
            ),
            "CCSDT-1A TOTAL HESSIAN": np.array(  # mrcc
                [
                    [
                        -7.78035412e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90855814e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45427907e-01,
                        2.67759669e-01,
                        0.00000000e00,
                        -3.45427907e-01,
                        -2.67759669e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.59961661e-01,
                        0.00000000e00,
                        2.04153847e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        -2.04153847e-01,
                        -2.29980830e-01,
                    ],
                    [
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.13247422e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.75770284e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45427907e-01,
                        2.04153847e-01,
                        0.00000000e00,
                        3.77721655e-01,
                        -2.35956758e-01,
                        0.00000000e00,
                        -3.22937484e-02,
                        3.18029110e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67759669e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        -2.35956758e-01,
                        2.18732769e-01,
                        0.00000000e00,
                        -3.18029110e-02,
                        1.12480613e-02,
                    ],
                    [
                        3.89017706e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.75770284e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.13247422e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45427907e-01,
                        -2.04153847e-01,
                        0.00000000e00,
                        -3.22937484e-02,
                        -3.18029110e-02,
                        0.00000000e00,
                        3.77721655e-01,
                        2.35956758e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67759669e-01,
                        -2.29980830e-01,
                        0.00000000e00,
                        3.18029110e-02,
                        1.12480613e-02,
                        0.00000000e00,
                        2.35956758e-01,
                        2.18732769e-01,
                    ],
                ]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.257755250971,  # ecc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00086535505446],
                    [0.00000000000000, -0.00023045568589, -0.00043267752723],
                    [0.00000000000000, 0.00023045568589, -0.00043267752723],
                ]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.257108846844,  # vcc
            "CCSDT-2 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00049220649666],
                    [0.00000000000000, -0.00043758137368, -0.00024610324833],
                    [0.00000000000000, 0.00043758137368, -0.00024610324833],
                ]
            ),
            "CCSDT-3 CORRELATION ENERGY": -0.25713217826377,  # vcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # ecc by psi fd STAR
                [
                    [0.00000000000000, 0.00000000000000, 0.00051263693668],
                    [0.00000000000000, -0.00042772129404, -0.00025631846834],
                    [0.00000000000000, 0.00042772129404, -0.00025631846834],
                ]
            ),
            "CCSDT-3 TOTAL HESSIAN": np.array(  # mrcc
                [
                    [-0.000462718541, 0.0, 0.0, 0.00023135927, 0.0, 0.0, 0.00023135927, 0.0, 0.0],
                    [
                        0.0,
                        0.690863579709,
                        0.0,
                        0.0,
                        -0.345431789855,
                        0.26764066777,
                        0.0,
                        -0.345431789855,
                        -0.26764066777,
                    ],
                    [
                        0.0,
                        0.0,
                        0.460459119187,
                        0.0,
                        0.204125984085,
                        -0.230229559594,
                        0.0,
                        -0.204125984085,
                        -0.230229559594,
                    ],
                    [0.00023135927, 0.0, 0.0, 0.000033666172, 0.0, 0.0, -0.000265025443, 0.0, 0.0],
                    [
                        0.0,
                        -0.345431789855,
                        0.204125984085,
                        0.0,
                        0.377813669112,
                        -0.235883325927,
                        0.0,
                        -0.032381879257,
                        0.031757341842,
                    ],
                    [
                        0.0,
                        0.26764066777,
                        -0.230229559594,
                        0.0,
                        -0.235883325927,
                        0.218879113252,
                        0.0,
                        -0.031757341842,
                        0.011350446342,
                    ],
                    [0.00023135927, 0.0, 0.0, -0.000265025443, 0.0, 0.0, 0.000033666172, 0.0, 0.0],
                    [
                        0.0,
                        -0.345431789855,
                        -0.204125984085,
                        0.0,
                        -0.032381879257,
                        -0.031757341842,
                        0.0,
                        0.377813669112,
                        0.235883325927,
                    ],
                    [
                        0.0,
                        -0.26764066777,
                        -0.230229559594,
                        0.0,
                        0.031757341842,
                        0.011350446342,
                        0.0,
                        0.235883325927,
                        0.218879113252,
                    ],
                ]
            ),
            "CCSDT CORRELATION ENERGY": -0.257494892048411,  # ncc
            "CCSDT TOTAL GRADIENT": np.array(
                [  # ecc
                    0.000000000000000,
                    0.000000000000000,
                    0.000614598836608,
                    0.000000000000000,
                    -0.000362082670346,
                    -0.000307299418304,
                    0.000000000000000,
                    0.000362082670346,
                    -0.000307299418304,
                ]
            ).reshape((-1, 3)),
            "CCSDT TOTAL HESSIAN": np.array(  # ecc
                [
                    [
                        -5.54525900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.90642317e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -3.45321159e-01,
                        2.67590889e-01,
                        0.00000000e00,
                        -3.45321159e-01,
                        -2.67590889e-01,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        4.60364322e-01,
                        0.00000000e00,
                        2.04091988e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        -2.04091988e-01,
                        -2.30182161e-01,
                    ],
                    [
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.21557000e-05,
                        0.00000000e00,
                        0.00000000e00,
                        -2.65107200e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45321159e-01,
                        2.04091988e-01,
                        0.00000000e00,
                        3.77718239e-01,
                        -2.35841439e-01,
                        0.00000000e00,
                        -3.23970809e-02,
                        3.17494501e-02,
                    ],
                    [
                        0.00000000e00,
                        2.67590889e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        -2.35841439e-01,
                        2.18813058e-01,
                        0.00000000e00,
                        -3.17494501e-02,
                        1.13691031e-02,
                    ],
                    [
                        2.77262900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -2.65107200e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.21557000e-05,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.45321159e-01,
                        -2.04091988e-01,
                        0.00000000e00,
                        -3.23970809e-02,
                        -3.17494501e-02,
                        0.00000000e00,
                        3.77718239e-01,
                        2.35841439e-01,
                    ],
                    [
                        0.00000000e00,
                        -2.67590889e-01,
                        -2.30182161e-01,
                        0.00000000e00,
                        3.17494501e-02,
                        1.13691031e-02,
                        0.00000000e00,
                        2.35841439e-01,
                        2.18813058e-01,
                    ],
                ]
            ),
            "[Q] CORRECTION ENERGY": -0.000151213601440,  # ncc
            "(Q) CORRECTION ENERGY": -0.000368016662584,  # ncc
            "CCSDTQ CORRELATION ENERGY": -0.257824320323573,  # ncc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.058423513790,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001767468898,
            "MP2 TOTAL GRADIENT": np.array(
                # switched sign from unkn origin
                [
                    0.000000000000000,
                    0.000000000000000,
                    0.012305278627642,
                    0.000000000000000,
                    0.000000000000000,
                    -0.011851332672482,
                    0.000000000000000,
                    0.010327045553422,
                    -0.000226972977580,
                    0.000000000000000,
                    -0.010327045553422,
                    -0.000226972977580,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [
                        3.34735620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.92156600e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.25884470e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -3.78117870e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.61167000e-04,
                        4.35841400e-04,
                        0.00000000e00,
                        2.61167000e-04,
                        -4.35841400e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.15996143e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.13938386e-01,
                        0.00000000e00,
                        1.11245210e-03,
                        -1.02887840e-03,
                        0.00000000e00,
                        -1.11245210e-03,
                        -1.02887840e-03,
                    ],
                    [
                        -3.92156600e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.26586180e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -3.78117870e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.18188010e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.00350700e-04,
                        8.01285000e-05,
                        0.00000000e00,
                        -2.00350700e-04,
                        -8.01285000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.13938386e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.14114521e-01,
                        0.00000000e00,
                        1.83716000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -1.83716000e-05,
                        -8.80676000e-05,
                    ],
                    [
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41277320e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.29781620e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.61167000e-04,
                        1.11245210e-03,
                        0.00000000e00,
                        -2.00350700e-04,
                        1.83716000e-05,
                        0.00000000e00,
                        4.11361597e-01,
                        -8.23396800e-04,
                        0.00000000e00,
                        -4.11422413e-01,
                        -3.07426900e-04,
                    ],
                    [
                        0.00000000e00,
                        4.35841400e-04,
                        -1.02887840e-03,
                        0.00000000e00,
                        8.01285000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -8.23396800e-04,
                        -4.65210960e-03,
                        0.00000000e00,
                        3.07426900e-04,
                        5.76905560e-03,
                    ],
                    [
                        2.87104900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.72147900e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.29781620e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.41277320e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.61167000e-04,
                        -1.11245210e-03,
                        0.00000000e00,
                        -2.00350700e-04,
                        -1.83716000e-05,
                        0.00000000e00,
                        -4.11422413e-01,
                        3.07426900e-04,
                        0.00000000e00,
                        4.11361597e-01,
                        8.23396800e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.35841400e-04,
                        -1.02887840e-03,
                        0.00000000e00,
                        -8.01285000e-05,
                        -8.80676000e-05,
                        0.00000000e00,
                        -3.07426900e-04,
                        5.76905560e-03,
                        0.00000000e00,
                        8.23396800e-04,
                        -4.65210960e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.07375810,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220392,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009100318483,
                    0.0,
                    0.0,
                    -0.008860556313,
                    0.0,
                    0.012722119397,
                    -0.000119881085,
                    0.0,
                    -0.012722119397,
                    -0.000119881085,
                ]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.07825006,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000237348957,  # vcc only
            "CISD CORRELATION ENERGY": -0.07917581082828,  # vcc
            "QCISD CORRELATION ENERGY": -0.08117897504733,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00060269,  # vcc only
            "LCCD CORRELATION ENERGY": -0.0824313452,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022532590,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004402112418,
                    0.0,
                    0.0,
                    -0.004307902536,
                    0.0,
                    0.015259701356,
                    -0.000047104941,
                    0.0,
                    -0.015259701356,
                    -0.000047104941,
                ]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.0837903430,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0022897,  # vcc
            "CCD CORRELATION ENERGY": -0.08014654379229,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0022032,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [
                    [0.0, 0.0, 0.0061586096],
                    [0.0, 0.0, -0.0059576188],
                    [0.0, 0.0144468804, -0.0001004954],
                    [0.0, -0.0144468804, -0.0001004954],
                ]
            ),
            "CCSD CORRELATION ENERGY": -0.08117105566,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.002231965267,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00496423512,
                    0.0,
                    0.0,
                    -0.004814203262,
                    0.0,
                    0.014877060204,
                    -0.000075015929,
                    0.0,
                    -0.014877060204,
                    -0.000075015929,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [
                        6.58519500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.09716090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        6.18973300e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -9.89916500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.85471600e-04,
                        4.42369000e-04,
                        0.00000000e00,
                        1.85471600e-04,
                        -4.42369000e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.06191545e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.04510095e-01,
                        0.00000000e00,
                        1.15490300e-03,
                        -8.40725200e-04,
                        0.00000000e00,
                        -1.15490300e-03,
                        -8.40725200e-04,
                    ],
                    [
                        -1.09716090e-03,
                        0.00000000e00,
                        0.00000000e00,
                        1.38536200e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -9.89916500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.31932370e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.64703600e-04,
                        5.07376000e-05,
                        0.00000000e00,
                        -1.64703600e-04,
                        -5.07376000e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.04510095e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.04625444e-01,
                        0.00000000e00,
                        1.61035900e-04,
                        -5.76744000e-05,
                        0.00000000e00,
                        -1.61035900e-04,
                        -5.76744000e-05,
                    ],
                    [
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06335912e-02,
                        0.00000000e00,
                        0.00000000e00,
                        1.05583709e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.85471600e-04,
                        1.15490300e-03,
                        0.00000000e00,
                        -1.64703600e-04,
                        1.61035900e-04,
                        0.00000000e00,
                        4.07395303e-01,
                        -9.04522700e-04,
                        0.00000000e00,
                        -4.07416071e-01,
                        -4.11416200e-04,
                    ],
                    [
                        0.00000000e00,
                        4.42369000e-04,
                        -8.40725200e-04,
                        0.00000000e00,
                        5.07376000e-05,
                        -5.76744000e-05,
                        0.00000000e00,
                        -9.04522700e-04,
                        -8.14010510e-03,
                        0.00000000e00,
                        4.11416200e-04,
                        9.03850480e-03,
                    ],
                    [
                        2.19320700e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.44100500e-04,
                        0.00000000e00,
                        0.00000000e00,
                        1.05583709e-02,
                        0.00000000e00,
                        0.00000000e00,
                        -1.06335912e-02,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        1.85471600e-04,
                        -1.15490300e-03,
                        0.00000000e00,
                        -1.64703600e-04,
                        -1.61035900e-04,
                        0.00000000e00,
                        -4.07416071e-01,
                        4.11416200e-04,
                        0.00000000e00,
                        4.07395303e-01,
                        9.04522700e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.42369000e-04,
                        -8.40725200e-04,
                        0.00000000e00,
                        -5.07376000e-05,
                        -5.76744000e-05,
                        0.00000000e00,
                        -4.11416200e-04,
                        9.03850480e-03,
                        0.00000000e00,
                        9.04522700e-04,
                        -8.14010510e-03,
                    ],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.000606865286,  # vcc
            "(T) CORRECTION ENERGY": -0.00060401,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.00446268840416],
                    [0.00000000000000, 0.00000000000000, -0.00439493294278],
                    [0.00000000000000, 0.01495635291373, -0.00003387773069],
                    [0.00000000000000, -0.01495635291373, -0.00003387773069],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        0.000493717382,
                        0.0,
                        0.0,
                        -0.00091720077,
                        0.0,
                        0.0,
                        0.000211741694,
                        0.0,
                        0.0,
                        0.000211741694,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000457061127,
                        0.0,
                        0.0,
                        -0.000812917213,
                        0.0,
                        0.0,
                        0.000177928043,
                        0.000438798747,
                        0.0,
                        0.000177928043,
                        -0.000438798747,
                    ],
                    [
                        0.0,
                        0.0,
                        0.206046933798,
                        0.0,
                        0.0,
                        -0.204504540896,
                        0.0,
                        0.001278605784,
                        -0.000771196451,
                        0.0,
                        -0.001278605784,
                        -0.000771196451,
                    ],
                    [
                        -0.00091720077,
                        0.0,
                        0.0,
                        0.001205452004,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.000812917213,
                        0.0,
                        0.0,
                        0.001141063782,
                        0.0,
                        0.0,
                        -0.000164073285,
                        0.000047332929,
                        0.0,
                        -0.000164073285,
                        -0.000047332929,
                    ],
                    [
                        0.0,
                        0.0,
                        -0.204504540896,
                        0.0,
                        0.0,
                        0.204650897031,
                        0.0,
                        0.000135987404,
                        -0.000073178067,
                        0.0,
                        -0.000135987404,
                        -0.000073178067,
                    ],
                    [
                        0.000211741694,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        -0.010686239837,
                        0.0,
                        0.0,
                        0.01061862376,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177928043,
                        0.001278605784,
                        0.0,
                        -0.000164073285,
                        0.000135987404,
                        0.0,
                        0.407325954394,
                        -0.000950362432,
                        0.0,
                        -0.407339809152,
                        -0.000464230756,
                    ],
                    [
                        0.0,
                        0.000438798747,
                        -0.000771196451,
                        0.0,
                        0.000047332929,
                        -0.000073178067,
                        0.0,
                        -0.000950362432,
                        -0.008256200737,
                        0.0,
                        0.000464230756,
                        0.009100575256,
                    ],
                    [
                        0.000211741694,
                        0.0,
                        0.0,
                        -0.000144125617,
                        0.0,
                        0.0,
                        0.01061862376,
                        0.0,
                        0.0,
                        -0.010686239837,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.000177928043,
                        -0.001278605784,
                        0.0,
                        -0.000164073285,
                        -0.000135987404,
                        0.0,
                        -0.407339809152,
                        0.000464230756,
                        0.0,
                        0.407325954394,
                        0.000950362432,
                    ],
                    [
                        0.0,
                        -0.000438798747,
                        -0.000771196451,
                        0.0,
                        -0.000047332929,
                        -0.000073178067,
                        0.0,
                        -0.000464230756,
                        0.009100575256,
                        0.0,
                        0.000950362432,
                        -0.008256200737,
                    ],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.081774257938,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.081772270576,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.081772292290,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.08177701734273,  # vcc
            "CCSDT CORRELATION ENERGY": -0.08208821205578,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15242755400188052,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445360441348938,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.026279427993,
                    0.0,
                    0.013998590506,
                    -0.013139713997,
                    0.0,
                    -0.013998590506,
                    -0.013139713997,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.02266258, 0.0, 0.0, 0.01133129, 0.0, 0.0, 0.01133129, 0.0, 0.0],
                    [0.0, 0.62640007, 0.0, 0.0, -0.31320003, 0.24902164, 0.0, -0.31320003, -0.24902164],
                    [0.0, 0.0, 0.39927018, 0.0, 0.19168818, -0.19963509, 0.0, -0.19168818, -0.19963509],
                    [0.01133129, 0.0, 0.0, -0.01029722, 0.0, 0.0, -0.00103408, 0.0, 0.0],
                    [0.0, -0.31320003, 0.19168818, 0.0, 0.3403478, -0.22035491, 0.0, -0.02714777, 0.02866673],
                    [0.0, 0.24902164, -0.19963509, 0.0, -0.22035491, 0.19072649, 0.0, -0.02866673, 0.0089086],
                    [0.01133129, 0.0, 0.0, -0.00103408, 0.0, 0.0, -0.01029722, 0.0, 0.0],
                    [0.0, -0.31320003, -0.19168818, 0.0, -0.02714777, -0.02866673, 0.0, 0.3403478, 0.22035491],
                    [0.0, -0.24902164, -0.19963509, 0.0, 0.02866673, 0.0089086, 0.0, 0.22035491, 0.19072649],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.16862935,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03463618,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027862081662, 0.0, 0.014906293724, -0.013931040831, 0.0, -0.014906293724, -0.013931040831]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.17088317,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.003237304648,  # vcc only
            "CISD CORRELATION ENERGY": -0.16574719763576,  # vcc
            "QCISD CORRELATION ENERGY": -0.17187325792329,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00368626,  # vcc only
            "LCCD CORRELATION ENERGY": -0.1747537294,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0334378864,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.031468252099, 0.0, 0.017115261595, -0.015734126049, 0.0, -0.017115261595, -0.015734126049]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1763496376,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0337624,  # vcc
            "CCD CORRELATION ENERGY": -0.17052107838019,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.033058,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc by psi findif
                [[0.0, 0.0, 0.02892979], [0.0, 0.01570148, -0.01446489], [0.0, -0.01570148, -0.01446489]]
            ),
            "CCSD CORRELATION ENERGY": -0.1716495276047496,  # mrcc
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033248190929062,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.030055915902,
                    0.0,
                    0.016307167756,
                    -0.015027957951,
                    0.0,
                    -0.016307167756,
                    -0.015027957951,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.02591931, 0.0, 0.0, 0.01295966, 0.0, 0.0, 0.01295966, 0.0, 0.0],
                    [0.0, 0.6170188, 0.0, 0.0, -0.3085094, 0.24667188, 0.0, -0.3085094, -0.24667188],
                    [0.0, 0.0, 0.39656797, 0.0, 0.19003804, -0.19828399, 0.0, -0.19003804, -0.19828399],
                    [0.01295966, 0.0, 0.0, -0.01187521, 0.0, 0.0, -0.00108444, 0.0, 0.0],
                    [0.0, -0.3085094, 0.19003804, 0.0, 0.33664846, -0.21835496, 0.0, -0.02813906, 0.02831692],
                    [0.0, 0.24667188, -0.19828399, 0.0, -0.21835496, 0.1883856, 0.0, -0.02831692, 0.00989838],
                    [0.01295966, 0.0, 0.0, -0.00108444, 0.0, 0.0, -0.01187521, 0.0, 0.0],
                    [0.0, -0.3085094, -0.19003804, 0.0, -0.02813906, -0.02831692, 0.0, 0.33664846, 0.21835496],
                    [0.0, -0.24667188, -0.19828399, 0.0, 0.02831692, 0.00989838, 0.0, 0.21835496, 0.1883856],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.003956941783,  # vcc
            "(T) CORRECTION ENERGY": -0.00381116,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.03193430184375],
                    [0.00000000000000, 0.01730212130253, -0.01596715092187],
                    [0.00000000000000, -0.01730212130253, -0.01596715092187],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [
                        -0.027539040923,
                        0.0,
                        0.0,
                        0.013769520462,
                        0.0,
                        0.0,
                        0.013769520462,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        0.615442549333,
                        0.0,
                        0.0,
                        -0.307721274667,
                        0.246688641005,
                        0.0,
                        -0.307721274667,
                        -0.246688641005,
                    ],
                    [
                        0.0,
                        0.0,
                        0.39351580275,
                        0.0,
                        0.189939555413,
                        -0.196757901375,
                        0.0,
                        -0.189939555413,
                        -0.196757901375,
                    ],
                    [
                        0.013769520462,
                        0.0,
                        0.0,
                        -0.012609341037,
                        0.0,
                        0.0,
                        -0.001160179424,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.307721274667,
                        0.189939555413,
                        0.0,
                        0.335534419601,
                        -0.218314098209,
                        0.0,
                        -0.027813144934,
                        0.028374542796,
                    ],
                    [
                        0.0,
                        0.246688641005,
                        -0.196757901375,
                        0.0,
                        -0.218314098209,
                        0.187299796614,
                        0.0,
                        -0.028374542796,
                        0.009458104761,
                    ],
                    [
                        0.013769520462,
                        0.0,
                        0.0,
                        -0.001160179424,
                        0.0,
                        0.0,
                        -0.012609341037,
                        0.0,
                        0.0,
                    ],
                    [
                        0.0,
                        -0.307721274667,
                        -0.189939555413,
                        0.0,
                        -0.027813144934,
                        -0.028374542796,
                        0.0,
                        0.335534419601,
                        0.218314098209,
                    ],
                    [
                        0.0,
                        -0.246688641005,
                        -0.196757901375,
                        0.0,
                        0.028374542796,
                        0.009458104761,
                        0.0,
                        0.218314098209,
                        0.187299796614,
                    ],
                ]
            ),
            "A-(T) CORRECTION ENERGY": -0.00373191576,  # mrcc
            "CCSDT-1A CORRELATION ENERGY": -0.1755715323437954,  # mrcc
            "CCSDT-1A TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03195915], [0.0, 0.01732744, -0.01597957], [0.0, -0.01732744, -0.01597957]]
            ),
            "CCSDT-1B CORRELATION ENERGY": -0.17556886143889727,  # mrcc
            "CCSDT-1B TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03195282], [0.0, 0.01732422, -0.01597641], [0.0, -0.01732422, -0.01597641]]
            ),
            "CCSDT-2 CORRELATION ENERGY": -0.175300994962,  # ecc
            "CCSDT-3 CORRELATION ENERGY": -0.1753162719579464,  # mrcc
            "CCSDT-3 TOTAL GRADIENT": np.array(  # mrcc
                [[0.0, 0.0, 0.03180282], [0.0, 0.01723825, -0.01590141], [0.0, -0.01723825, -0.01590141]]
            ),
            "CCSDT CORRELATION ENERGY": -0.17591978591647,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.171184123093,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822454,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.014740098324,
                    0.0,
                    0.005852228009,
                    -0.007370049162,
                    0.0,
                    -0.005852228009,
                    -0.007370049162,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # cfour
                [
                    [-0.01271141, 0.0, 0.0, 0.00635571, 0.0, 0.0, 0.00635571, 0.0, 0.0],
                    [0.0, 0.59697724, 0.0, 0.0, -0.29848862, 0.23391528, 0.0, -0.29848862, -0.23391528],
                    [0.0, 0.0, 0.3890832, 0.0, 0.17739783, -0.1945416, 0.0, -0.17739783, -0.1945416],
                    [0.00635571, 0.0, 0.0, -0.00511412, 0.0, 0.0, -0.00124159, 0.0, 0.0],
                    [0.0, -0.29848862, 0.17739783, 0.0, 0.32667986, -0.20565656, 0.0, -0.02819123, 0.02825872],
                    [0.0, 0.23391528, -0.1945416, 0.0, -0.20565656, 0.18507929, 0.0, -0.02825872, 0.00946231],
                    [0.00635571, 0.0, 0.0, -0.00124159, 0.0, 0.0, -0.00511412, 0.0, 0.0],
                    [0.0, -0.29848862, -0.17739783, 0.0, -0.02819123, -0.02825872, 0.0, 0.32667986, 0.20565656],
                    [0.0, -0.23391528, -0.1945416, 0.0, 0.02825872, 0.00946231, 0.0, 0.20565656, 0.18507929],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.18575628,  # occ
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792248,  # occ
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.015431661276, 0.0, 0.006103353029, -0.007715830638, 0.0, -0.006103353029, -0.007715830638]
            ).reshape((-1, 3)),
            "MP4(SDQ) CORRELATION ENERGY": -0.18773117,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004470767049,  # vcc only
            "CISD CORRELATION ENERGY": -0.18202095532782,  # vcc
            "QCISD CORRELATION ENERGY": -0.18852342173162,  # vcc
            "QCISD(T) CORRECTION ENERGY": -0.00487511,  # vcc only
            "LCCD CORRELATION ENERGY": -0.1917024115,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367596656,  # occ
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.019152642358, 0.0, 0.008310773449, -0.009576321179, 0.0, -0.008310773449, -0.009576321179]
            ).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.1933416962,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.0370625,  # vcc
            "CCD CORRELATION ENERGY": -0.18708520686363,  # vcc
            "CCD SINGLES ENERGY": 0.0,
            "CCD SAME-SPIN CORRELATION ENERGY": -0.0363299,  # vcc
            "CCD TOTAL GRADIENT": np.array(  # vcc
                [[0.0, 0.0, 0.0166234857], [0.0, 0.0069994421, -0.0083117428], [0.0, -0.0069994421, -0.0083117428]]
            ),
            "CCSD CORRELATION ENERGY": -0.188317222733,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.036526852874970,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.017883390799,
                    0.0,
                    0.00765987541,
                    -0.0089416954,
                    0.0,
                    -0.00765987541,
                    -0.0089416954,
                ]
            ).reshape((-1, 3)),
            "CCSD TOTAL HESSIAN": np.array(  # vcc
                [
                    [-0.01542209, 0.0, 0.0, 0.00771105, 0.0, 0.0, 0.00771105, 0.0, 0.0],
                    [0.0, 0.58612421, 0.0, 0.0, -0.2930621, 0.23079136, 0.0, -0.2930621, -0.23079136],
                    [0.0, 0.0, 0.38586066, 0.0, 0.17492242, -0.19293033, 0.0, -0.17492242, -0.19293033],
                    [0.00771105, 0.0, 0.0, -0.00638987, 0.0, 0.0, -0.00132118, 0.0, 0.0],
                    [0.0, -0.2930621, 0.17492242, 0.0, 0.32224469, -0.20285689, 0.0, -0.02918259, 0.02793447],
                    [0.0, 0.23079136, -0.19293033, 0.0, -0.20285689, 0.18247704, 0.0, -0.02793447, 0.01045329],
                    [0.00771105, 0.0, 0.0, -0.00132118, 0.0, 0.0, -0.00638987, 0.0, 0.0],
                    [0.0, -0.2930621, -0.17492242, 0.0, -0.02918259, -0.02793447, 0.0, 0.32224469, 0.20285689],
                    [0.0, -0.23079136, -0.19293033, 0.0, 0.02793447, 0.01045329, 0.0, 0.20285689, 0.18247704],
                ]
            ),
            "T(CCSD) CORRECTION ENERGY": -0.005108963391,  # vcc
            "(T) CORRECTION ENERGY": -0.00498265,
            "CCSD(T) TOTAL GRADIENT": np.array(  # vcc fd
                [
                    [0.00000000000000, 0.00000000000000, 0.02003159253303],
                    [0.00000000000000, 0.00872844491897, -0.01001579626652],
                    [0.00000000000000, -0.00872844491897, -0.01001579626652],
                ]
            ),
            "CCSD(T) TOTAL HESSIAN": np.array(  # vcc fd
                [
                    [-0.01727460991, 0.0, 0.0, 0.008637304955, 0.0, 0.0, 0.008637304955, 0.0, 0.0],
                    [
                        0.0,
                        0.584492978582,
                        0.0,
                        0.0,
                        -0.292246489291,
                        0.230876480089,
                        0.0,
                        -0.292246489291,
                        -0.230876480089,
                    ],
                    [
                        0.0,
                        0.0,
                        0.382520093487,
                        0.0,
                        0.174867793441,
                        -0.191260046743,
                        0.0,
                        -0.174867793441,
                        -0.191260046743,
                    ],
                    [0.008637304955, 0.0, 0.0, -0.007206558163, 0.0, 0.0, -0.001430746792, 0.0, 0.0],
                    [
                        0.0,
                        -0.292246489291,
                        0.174867793441,
                        0.0,
                        0.321124482646,
                        -0.202872136765,
                        0.0,
                        -0.028877993355,
                        0.028004343324,
                    ],
                    [
                        0.0,
                        0.230876480089,
                        -0.191260046743,
                        0.0,
                        -0.202872136765,
                        0.181320766941,
                        0.0,
                        -0.028004343324,
                        0.009939279803,
                    ],
                    [0.008637304955, 0.0, 0.0, -0.001430746792, 0.0, 0.0, -0.007206558163, 0.0, 0.0],
                    [
                        0.0,
                        -0.292246489291,
                        -0.174867793441,
                        0.0,
                        -0.028877993355,
                        -0.028004343324,
                        0.0,
                        0.321124482646,
                        0.202872136765,
                    ],
                    [
                        0.0,
                        -0.230876480089,
                        -0.191260046743,
                        0.0,
                        0.028004343324,
                        0.009939279803,
                        0.0,
                        0.202872136765,
                        0.181320766941,
                    ],
                ]
            ),
            "CCSDT-1A CORRELATION ENERGY": -0.193424330972,  # ecc
            "CCSDT-1B CORRELATION ENERGY": -0.193423371134,  # ecc
            "CCSDT-2 CORRELATION ENERGY": -0.193087540038,  # vcc
            "CCSDT-3 CORRELATION ENERGY": -0.19310599643349,  # vcc
            "CCSDT CORRELATION ENERGY": -0.19368177447948,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.059407254257,
            "MP2 SINGLES ENERGY": -0.000688368657,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001851937488,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    # switched sign from unkn ref
                    0.000000000000000,
                    0.000000000000000,
                    0.013388410166131,
                    0.000000000000000,
                    0.000000000000000,
                    -0.012907368096590,
                    0.000000000000000,
                    0.010303507439169,
                    -0.000240521034770,
                    0.000000000000000,
                    -0.010303507439169,
                    -0.000240521034770,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [
                        3.77039407e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.35686969e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        3.68005099e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -4.21445800e-03,
                        0.00000000e00,
                        0.00000000e00,
                        2.67203501e-04,
                        4.37851629e-04,
                        0.00000000e00,
                        2.67203501e-04,
                        -4.37851629e-04,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        2.17707108e-01,
                        0.00000000e00,
                        0.00000000e00,
                        -2.15648507e-01,
                        0.00000000e00,
                        1.13660067e-03,
                        -1.02930031e-03,
                        0.00000000e00,
                        -1.13660067e-03,
                        -1.02930031e-03,
                    ],
                    [
                        -4.35686969e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.70636057e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        -4.21445800e-03,
                        0.00000000e00,
                        0.00000000e00,
                        4.62108124e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -2.03311621e-04,
                        8.06365754e-05,
                        0.00000000e00,
                        -2.03311621e-04,
                        -8.06365754e-05,
                    ],
                    [
                        0.00000000e00,
                        0.00000000e00,
                        -2.15648507e-01,
                        0.00000000e00,
                        0.00000000e00,
                        2.15870333e-01,
                        0.00000000e00,
                        -4.83601569e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        4.83601569e-05,
                        -1.10913000e-04,
                    ],
                    [
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -7.39767359e-03,
                        0.00000000e00,
                        0.00000000e00,
                        7.27918122e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.67203501e-04,
                        1.13660067e-03,
                        0.00000000e00,
                        -2.03311621e-04,
                        -4.83601569e-05,
                        0.00000000e00,
                        4.11398494e-01,
                        -8.03364358e-04,
                        0.00000000e00,
                        -4.11462386e-01,
                        -2.84876153e-04,
                    ],
                    [
                        0.00000000e00,
                        4.37851629e-04,
                        -1.02930031e-03,
                        0.00000000e00,
                        8.06365754e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        -8.03364358e-04,
                        -4.61144770e-03,
                        0.00000000e00,
                        2.84876153e-04,
                        5.75166101e-03,
                    ],
                    [
                        2.93237813e-04,
                        0.00000000e00,
                        0.00000000e00,
                        -1.74745438e-04,
                        0.00000000e00,
                        0.00000000e00,
                        7.27918122e-03,
                        0.00000000e00,
                        0.00000000e00,
                        -7.39767359e-03,
                        0.00000000e00,
                        0.00000000e00,
                    ],
                    [
                        0.00000000e00,
                        2.67203501e-04,
                        -1.13660067e-03,
                        0.00000000e00,
                        -2.03311621e-04,
                        4.83601569e-05,
                        0.00000000e00,
                        -4.11462386e-01,
                        2.84876153e-04,
                        0.00000000e00,
                        4.11398494e-01,
                        8.03364358e-04,
                    ],
                    [
                        0.00000000e00,
                        -4.37851629e-04,
                        -1.02930031e-03,
                        0.00000000e00,
                        -8.06365754e-05,
                        -1.10913000e-04,
                        0.00000000e00,
                        -2.84876153e-04,
                        5.75166101e-03,
                        0.00000000e00,
                        8.03364358e-04,
                        -4.61144770e-03,
                    ],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.074845418046,  # cfour
            "MP3 SINGLES ENERGY": -0.000911987215,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.0099090428],
                    [0.0, 0.0, -0.009667824547],
                    [0.0, 0.012725362696, -0.000120609126],
                    [0.0, -0.012725362696, -0.000120609126],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.07930873,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.000289639941,  # vcc only
            "CISD CORRELATION ENERGY": -0.08032199708731,  # vcc
            "LCCD CORRELATION ENERGY": -0.0824056198,  # p4n
            "LCCSD CORRELATION ENERGY": -0.0851177481,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.08517989, # vcc != p4n
            "CCD CORRELATION ENERGY": -0.08016689737891,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.08256719,
            "CCSD SINGLES ENERGY": -0.00117001688,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.00230304,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.005323074361,
                    0.0,
                    0.0,
                    -0.005174249172,
                    0.0,
                    0.014881203442,
                    -0.000074412594,
                    0.0,
                    -0.014881203442,
                    -0.000074412594,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00068823,  # cfour only
            "CCSDT CORRELATION ENERGY": -0.08361110233142,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15702660833165538,
            "MP2 SINGLES ENERGY": -0.0028059971624814647,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466304269235235,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.026398091851,
                    0.0,
                    0.014012163884,
                    -0.013199045925,
                    0.0,
                    -0.014012163884,
                    -0.013199045925,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.0227647, 0.0, 0.0, 0.01138235, 0.0, 0.0, 0.01138235, 0.0, 0.0],
                    [0.0, 0.62606906, 0.0, 0.0, -0.31303453, 0.24893382, 0.0, -0.31303453, -0.24893382],
                    [0.0, 0.0, 0.39895549, 0.0, 0.1914611, -0.19947774, 0.0, -0.1914611, -0.19947774],
                    [0.01138235, 0.0, 0.0, -0.01032718, 0.0, 0.0, -0.00105517, 0.0, 0.0],
                    [0.0, -0.31303453, 0.1914611, 0.0, 0.34009582, -0.22019746, 0.0, -0.02706129, 0.02873636],
                    [0.0, 0.24893382, -0.19947774, 0.0, -0.22019746, 0.1906097, 0.0, -0.02873636, 0.00886804],
                    [0.01138235, 0.0, 0.0, -0.00105517, 0.0, 0.0, -0.01032718, 0.0, 0.0],
                    [0.0, -0.31303453, -0.1914611, 0.0, -0.02706129, -0.02873636, 0.0, 0.34009582, 0.22019746],
                    [0.0, -0.24893382, -0.19947774, 0.0, 0.02873636, 0.00886804, 0.0, 0.22019746, 0.1906097],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.173406218947,  # cfour
            "MP3 SINGLES ENERGY": -0.003217654547,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [[0.0, 0.0, 0.0280810549], [0.0, 0.0150218165, -0.0140405275], [0.0, -0.0150218165, -0.0140405275]]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.17541828,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.00334215131,  # vcc only
            "CISD CORRELATION ENERGY": -0.16970933003145,  # vcc
            "LCCD CORRELATION ENERGY": -0.1769020687,  # p4n
            "LCCSD CORRELATION ENERGY": -0.1807707740,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.18080394, # vcc != p4n
            "CCD CORRELATION ENERGY": -0.17252087862035,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.175988485854028,
            "CCSD SINGLES ENERGY": -0.003256808469230,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.033291143258924,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.030051791297,
                    0.0,
                    0.016301545337,
                    -0.015025895649,
                    0.0,
                    -0.016301545337,
                    -0.015025895649,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.003863167899,  # cfour only
            "CCSDT CORRELATION ENERGY": -0.18030677104047,  # vcc (different orbs: -0.18031166502580)
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.1761163066,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837483,
            "MP2 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.014894057335,
                    0.0,
                    0.005886660707,
                    -0.007447028667,
                    0.0,
                    -0.005886660707,
                    -0.007447028667,
                ]
            ).reshape((-1, 3)),
            "MP2 TOTAL HESSIAN": np.array(  # vcc by psi fd
                [
                    [-0.0128439, 0.0, 0.0, 0.00642195, 0.0, 0.0, 0.00642195, 0.0, 0.0],
                    [0.0, 0.59650839, 0.0, 0.0, -0.29825419, 0.23378625, 0.0, -0.29825419, -0.23378625],
                    [0.0, 0.0, 0.38870359, 0.0, 0.17712488, -0.1943518, 0.0, -0.17712488, -0.1943518],
                    [0.00642195, 0.0, 0.0, -0.00515858, 0.0, 0.0, -0.00126337, 0.0, 0.0],
                    [0.0, -0.29825419, 0.17712488, 0.0, 0.32634865, -0.20545557, 0.0, -0.02809446, 0.02833069],
                    [0.0, 0.23378625, -0.1943518, 0.0, -0.20545557, 0.18492354, 0.0, -0.02833069, 0.00942826],
                    [0.00642195, 0.0, 0.0, -0.00126337, 0.0, 0.0, -0.00515858, 0.0, 0.0],
                    [0.0, -0.29825419, -0.17712488, 0.0, -0.02809446, -0.02833069, 0.0, 0.32634865, 0.20545557],
                    [0.0, -0.23378625, -0.1943518, 0.0, 0.02833069, 0.00942826, 0.0, 0.20545557, 0.18492354],
                ]
            ),
            "MP3 CORRELATION ENERGY": -0.190735179201,  # cfour
            "MP3 SINGLES ENERGY": -0.003322599961,  # cfour
            "MP3 TOTAL GRADIENT": np.array(  # vcc by fd psi
                [
                    [0.0, 0.0, 0.015664483926],
                    [0.0, 0.006224027206, -0.007832241963],
                    [0.0, -0.006224027206, -0.007832241963],
                ]
            ),
            "MP4(SDQ) CORRELATION ENERGY": -0.19243169,  # vcc only
            "MP4(T) CORRECTION ENERGY": -0.004597066465,  # vcc only
            "CISD CORRELATION ENERGY": -0.18613466061896,  # vcc
            "LCCD CORRELATION ENERGY": -0.1939920915,  # p4n
            "LCCSD CORRELATION ENERGY": -0.1979175937,  # p4n
            # "LCCSD CORRELATION ENERGY": -0.19795428,  # vcc != p4n
            "CCD CORRELATION ENERGY": -0.18921914785643,  # vcc nyi ???
            "CCSD CORRELATION ENERGY": -0.19282621471297376,
            "CCSD SINGLES ENERGY": -0.003354603508621,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.036502859698546,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.017873897449,
                    0.0,
                    0.007653541045,
                    -0.008936948724,
                    0.0,
                    -0.007653541045,
                    -0.008936948724,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00504351,  # cfour only
            "CCSDT CORRELATION ENERGY": -0.19824510672649,  # vcc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-AE-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.203778449,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.20648138,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20990784,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20874537,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00193646,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.22188844,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.22643352,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.23188996,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22941330,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00523874,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.27018509,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.27294541,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.27869144,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.27570541,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00726403,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.059477703268,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001918940186,
            "MP3 CORRELATION ENERGY": -0.07481148,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235724,  # dfocc
            "LCCD CORRELATION ENERGY": -0.08343267,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00240067,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15485159,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520588,
            "MP3 CORRELATION ENERGY": -0.17091968,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533767,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17701281,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03413088,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.19552518,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04162160,
            "MP3 CORRELATION ENERGY": -0.21084703,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130865,  # dfocc
            "LCCD CORRELATION ENERGY": -0.21678793,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.04013550,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.0604664810,
            "MP2 SINGLES ENERGY": -0.000694049858,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002005152902,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15948893,
            "MP2 SINGLES ENERGY": -0.00282963,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03542136,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.20053428,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04179065,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20162439774,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.20453779,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20796060,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.20681721,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00192057,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.21939933,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.22415842,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.22961687,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.22717646,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00521255,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24514540,
            "MP2 SINGLES ENERGY": 0.0,
            "MP3 CORRELATION ENERGY": -0.24747829,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.25319438,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "CCSD CORRELATION ENERGY": -0.25033052,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00709694,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05841222894,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0017676971,
            "MP3 CORRELATION ENERGY": -0.07375426,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220422,  # dfocc
            "LCCD CORRELATION ENERGY": -0.08242955,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00225358,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15241971,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445776,
            "MP3 CORRELATION ENERGY": -0.16863252,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03464025,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17475833,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03344184,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.17117906,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822980,
            "MP3 CORRELATION ENERGY": -0.18575660,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792752,  # dfocc
            "LCCD CORRELATION ENERGY": -0.19170259,  # dfocc
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03676455,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.059395907176,
            "MP2 SINGLES ENERGY": -0.00068836865,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001852162877,
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15701860,
            "MP2 SINGLES ENERGY": -0.00280600,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466721,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.17611121,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03838011,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20377997248921056,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05431321036920538,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.0027998, 0.0, 0.0, -0.0027998]  # dfmp2 findif-5 ae pk+df
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20657694,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000926859678, 0.0, 0.0, 0.000926859678]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2100497124,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002190589954, 0.0, 0.0, -0.002190589954]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20888438,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00193859,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.22188894,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674808,
            "MP3 CORRELATION ENERGY": -0.22652962,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2320261414,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22954333,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00524393,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.27018057,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530212,
            "MP3 CORRELATION ENERGY": -0.27294176,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2786878429,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.27570207,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00726375,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.05945820694747983,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0019203155958724552,
            "MP3 CORRELATION ENERGY": -0.07486724,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235854,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0835080983,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024018298,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15484736,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523329,
            "MP3 CORRELATION ENERGY": -0.17098385,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530465,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1771107929,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340809591,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.19551918,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161696,
            "MP3 CORRELATION ENERGY": -0.21084349,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130382,  # dfocc
            "LCCD CORRELATION ENERGY": -0.2167841215,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401306929,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.0604460449537298,
            "MP2 SINGLES ENERGY": -0.0006940498589629459,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020066877639503184,
            "LCCD CORRELATION ENERGY": -0.0834825821,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15948289,
            "MP2 SINGLES ENERGY": -0.00282963,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544835,
            "LCCD CORRELATION ENERGY": -0.1792713801,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.20052829,
            "MP2 SINGLES ENERGY": -0.00298375,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178599,
            "LCCD CORRELATION ENERGY": -0.2191002183,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CONV-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_hf_dz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.20162566806258586,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05352569481658172,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00315485, 0.0, 0.0, -0.00315485]  # dfmp2 findif-5 fc pk+df
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20463304,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2081020566,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.20695586,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00192267,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_adz_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.21939942,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05588210,
            "MP3 CORRELATION ENERGY": -0.22425409,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2297524911,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22730597,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00521769,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_pk_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_pk_rhf,
            "HF TOTAL HESSIAN": _hess_scf_h2o_qz2p_pk_rhf,
            "MP2 CORRELATION ENERGY": -0.24514425,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126481,
            "MP3 CORRELATION ENERGY": -0.24747778,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2531939249,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.25033030,
            "CCSD SINGLES ENERGY": 0.0,
            "(T) CORRECTION ENERGY": -0.00709666,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.058392397606538686,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0017690135626491292,
            "MP3 CORRELATION ENERGY": -0.07380965,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220548,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0825046579,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022547041,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.15241501,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448519,
            "MP3 CORRELATION ENERGY": -0.16869625,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460729,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1748557523,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333918420,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_uhf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_uhf,
            "MP2 CORRELATION ENERGY": -0.17117615,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822512,
            "MP3 CORRELATION ENERGY": -0.18575590,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792262,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1917015960,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367596684,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_bh3p_dz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.05937514348825628,
            "MP2 SINGLES ENERGY": -0.0006883686516107368,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018536363586657242,
            "LCCD CORRELATION ENERGY": -0.0824786458,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_adz_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.15701209,
            "MP2 SINGLES ENERGY": -0.00280600,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469422,
            "LCCD CORRELATION ENERGY": -0.1770018748,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "pk",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_pk_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_pk_rohf,
            "HF TOTAL HESSIAN": _hess_scf_nh2_qz2p_pk_rohf,
            "MP2 CORRELATION ENERGY": -0.17610830,
            "MP2 SINGLES ENERGY": -0.00294339,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837541,
            "LCCD CORRELATION ENERGY": -0.1939912613,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-AE-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.20377328786815951,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427570,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [0.0, 0.0, 0.00281146, 0.0, 0.0, -0.00281146]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20647618,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20990226,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.20873986012771106,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857381,
            "(T) CORRECTION ENERGY": -0.0019363109218456449,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.22188817,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05670210,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    0.009643414073,
                    0.0,
                    0.005501440694,
                    -0.004821707036,
                    0.0,
                    -0.005501440694,
                    -0.004821707036,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22643303,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.23188949,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22941290,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05017955,
            "(T) CORRECTION ENERGY": -0.00523867,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.27018399,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530655,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    -0.000546229785,
                    0.0,
                    -0.000967320028,
                    0.000273114892,
                    0.0,
                    0.000967320028,
                    0.000273114892,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.27294416,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.27869015,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.27570421,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05801141,
            "(T) CORRECTION ENERGY": -0.00726395,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.059476326350818454,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0019188791023,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [
                    0.0,
                    0.0,
                    0.0125029,
                    0.0,
                    0.0,
                    -0.01205882,
                    0.0,
                    0.01033888,
                    -0.00022204,
                    0.0,
                    -0.01033888,
                    -0.00022204,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07480952,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235716,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009315056264,
                    0.0,
                    0.0,
                    -0.009084753984,
                    0.0,
                    0.012696375212,
                    -0.00011515114,
                    0.0,
                    -0.012696375212,
                    -0.00011515114,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.08343038,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00240059,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004643130768,
                    0.0,
                    0.0,
                    -0.004557062257,
                    0.0,
                    0.015239807714,
                    -0.000043034255,
                    0.0,
                    -0.015239807714,
                    -0.000043034255,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15485101,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03520580,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.025470063809,
                    0.0,
                    0.013535107677,
                    -0.012735031905,
                    0.0,
                    -0.013535107677,
                    -0.012735031905,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.17091879,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03533750,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17701192,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03413070,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.19552441,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04162127,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.013727424376,
                    0.0,
                    0.005348487843,
                    -0.006863712188,
                    0.0,
                    -0.005348487843,
                    -0.006863712188,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.21084618,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130830,  # dfocc
            "LCCD CORRELATION ENERGY": -0.21678706,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.04013515,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.06046475293245379,
            "MP2 SINGLES ENERGY": -0.00069387098844,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.002005071400,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 ae cd+cd
                [0.0, 0.0, 0.01359215, 0.0, 0.0, -0.01312116, 0.0, 0.01031541, -0.0002355, 0.0, -0.01031541, -0.0002355]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15948823,
            "MP2 SINGLES ENERGY": -0.00282948,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03542128,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.025588961002,
                    0.0,
                    0.013550360249,
                    -0.012794480501,
                    0.0,
                    -0.013550360249,
                    -0.012794480501,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.20053352,
            "MP2 SINGLES ENERGY": -0.00298373,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04179032,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 ae cd+cd
                    0.0,
                    0.0,
                    0.013884053665,
                    0.0,
                    0.005385412795,
                    -0.006942026833,
                    0.0,
                    -0.005385412795,
                    -0.006942026833,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-FC-CD  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.201619244596,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05348825,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [0.0, 0.0, 0.00316665, 0.0, 0.0, -0.00316665]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20453260,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.20795503,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.2068117080298787,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04787083,
            "(T) CORRECTION ENERGY": -0.0019204203743072874,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.21939907,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05583617,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.010264703011,
                    0.0,
                    0.00588885358,
                    -0.005132351506,
                    0.0,
                    -0.00588885358,
                    -0.005132351506,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22415794,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.22961642,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22717607,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04939986,
            "(T) CORRECTION ENERGY": -0.00521248,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.24514436,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126931,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.000318778691,
                    0.0,
                    -0.000569356625,
                    -0.000159389346,
                    0.0,
                    0.000569356625,
                    -0.000159389346,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.24747710,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.25319315,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.25032939,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05405638,
            "(T) CORRECTION ENERGY": -0.00709686,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.058410863785614,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001767642489,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.01230315,
                    0.0,
                    0.0,
                    -0.01184537,
                    0.0,
                    0.01033427,
                    -0.00022889,
                    0.0,
                    -0.01033427,
                    -0.00022889,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07375232,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220415,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009099123102,
                    0.0,
                    0.0,
                    -0.008855857257,
                    0.0,
                    0.012692779327,
                    -0.000121632922,
                    0.0,
                    -0.012692779327,
                    -0.000121632922,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.08242726,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.00225350,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004401363786,
                    0.0,
                    0.0,
                    -0.004303629299,
                    0.0,
                    0.015236787233,
                    -0.000048867243,
                    0.0,
                    -0.015236787233,
                    -0.000048867243,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15241915,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03445770,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.026258239074,
                    0.0,
                    0.01404196652,
                    -0.013129119537,
                    0.0,
                    -0.01404196652,
                    -0.013129119537,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.16863165,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03464009,  # dfocc
            "LCCD CORRELATION ENERGY": -0.17475747,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0334416820,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.17117831,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822948,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.014735846129,
                    0.0,
                    0.005848618964,
                    -0.007367923065,
                    0.0,
                    -0.005848618964,
                    -0.007367923065,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.18575577,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792717,  # dfocc
            "LCCD CORRELATION ENERGY": -0.19170174,  # dfocc
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.03676422,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.05939419492939635,
            "MP2 SINGLES ENERGY": -0.0006881934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018520881544,
            "MP2 TOTAL GRADIENT": np.array(
                # dfocc findif-5 fc cd+cd
                [
                    0.0,
                    0.0,
                    0.01338641,
                    0.0,
                    0.0,
                    -0.01290149,
                    0.0,
                    0.01031066,
                    -0.00024246,
                    0.0,
                    -0.01031066,
                    -0.00024246,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15701792,
            "MP2 SINGLES ENERGY": -0.00280584,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03466715,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.026376923581,
                    0.0,
                    0.014055606253,
                    -0.01318846179,
                    0.0,
                    -0.014055606253,
                    -0.01318846179,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "cd",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.17611046,
            "MP2 SINGLES ENERGY": -0.00294336,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837979,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfocc findif-5 fc cd+cd
                    0.0,
                    0.0,
                    0.014889762324,
                    0.0,
                    0.00588299146,
                    -0.007444881162,
                    0.0,
                    -0.00588299146,
                    -0.007444881162,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2037748110768,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.054311870576,
            "MP2 TOTAL GRADIENT": np.array(
                # dfmp2 findif-5 ae cd+df
                [0.0, 0.0, 0.00279182, 0.0, 0.0, -0.00279182]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20657173,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000934631868, 0.0, 0.0, 0.000934631868]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2100441271,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.00218256974, 0.0, 0.0, -0.00218256974]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20887885,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04845784,
            "(T) CORRECTION ENERGY": -0.00193844,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.22188866,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674801,
            "MP3 CORRELATION ENERGY": -0.22652912,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2320256729,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22954292,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05010092,
            "(T) CORRECTION ENERGY": -0.00524386,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.27017947,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06530177,
            "MP3 CORRELATION ENERGY": -0.27294050,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2786865554,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.27570087,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05800702,
            "(T) CORRECTION ENERGY": -0.00726367,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.059456828193,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.00192025457659,
            "MP3 CORRELATION ENERGY": -0.07486528,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235846,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0835057932,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024017496,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15484678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523320,
            "MP3 CORRELATION ENERGY": -0.17098295,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530448,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1771099018,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340807883,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.19551841,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161663,
            "MP3 CORRELATION ENERGY": -0.21084263,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130347,  # dfocc
            "LCCD CORRELATION ENERGY": -0.2167832515,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401303480,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.06044431529,
            "MP2 SINGLES ENERGY": -0.00069387098844,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020066063,
            "LCCD CORRELATION ENERGY": -0.0834800819,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15948219,
            "MP2 SINGLES ENERGY": -0.00282948,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544828,
            "LCCD CORRELATION ENERGY": -0.1792705171,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.20052752,
            "MP2 SINGLES ENERGY": -0.00298373,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178566,
            "LCCD CORRELATION ENERGY": -0.2190993784,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  CD-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2016205147678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0535243575,
            "MP2 TOTAL GRADIENT": np.array(
                # dfmp2 findif-5 fc cd+df
                [0.0, 0.0, 0.00314686, 0.0, 0.0, -0.00314686]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20462784,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2080964757,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.20695033,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04775464,
            "(T) CORRECTION ENERGY": -0.00192252,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.21939916,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05588204,
            "MP3 CORRELATION ENERGY": -0.22425360,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2297520405,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.22730558,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04932106,
            "(T) CORRECTION ENERGY": -0.00521762,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.24514320,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126448,
            "MP3 CORRELATION ENERGY": -0.24747659,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "LCCD CORRELATION ENERGY": -0.2531926943,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "CCSD CORRELATION ENERGY": -0.25032917,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05405189,
            "(T) CORRECTION ENERGY": -0.00709658,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.05839103061,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.00176895897,
            "MP3 CORRELATION ENERGY": -0.07380770,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220541,  # dfocc
            "LCCD CORRELATION ENERGY": -0.0825023638,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022546311,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.15241445,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448512,
            "MP3 CORRELATION ENERGY": -0.16869538,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460714,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1748548876,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333916888,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_uhf,
            "MP2 CORRELATION ENERGY": -0.17117540,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822480,
            "MP3 CORRELATION ENERGY": -0.18575507,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792228,  # dfocc
            "LCCD CORRELATION ENERGY": -0.1917007514,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367593319,  # dfocc
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.05937342969795,
            "MP2 SINGLES ENERGY": -0.0006881934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001853561678,
            "LCCD CORRELATION ENERGY": -0.0824761581,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.15701141,
            "MP2 SINGLES ENERGY": -0.00280584,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469416,
            "LCCD CORRELATION ENERGY": -0.1770010376,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "cd",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_cd_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_cd_rohf,
            "MP2 CORRELATION ENERGY": -0.17610756,
            "MP2 SINGLES ENERGY": -0.00294336,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837509,
            "LCCD CORRELATION ENERGY": -0.1939904460,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  DF-AE-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.2037649370559149,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05430875283333263,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.00279211492833,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00279211492833,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20656153,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000933466293, 0.0, 0.0, 0.000933466293]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2100337333,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002183232102, 0.0, 0.0, -0.002183232102]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20886884012911314,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04845491,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.001970675302,
                    0.0,
                    0.0,
                    -0.001970675302,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.0019380186429220421,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.22187976,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05674571,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.00962182765,
                    0.0,
                    0.005498317937,
                    -0.004810913825,
                    0.0,
                    -0.005498317937,
                    -0.004810913825,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22651881,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004009128141, 0.0, 0.002552803039, -0.00200456407, 0.0, -0.002552803039, -0.00200456407]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2320149229,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.008561006838, 0.0, 0.005236802973, -0.004280503419, 0.0, -0.005236802973, -0.004280503419]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.22953289,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05009877,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.007518759967,
                    0.0,
                    0.004613106602,
                    -0.003759379983,
                    0.0,
                    -0.004613106602,
                    -0.003759379983,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00524345,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_df_rhf,
            "MP2 CORRELATION ENERGY": -0.27016105,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06529808,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    -0.000566657943,
                    0.0,
                    -0.000968877215,
                    0.000283328971,
                    0.0,
                    0.000968877215,
                    0.000283328971,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.27292175,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.007037747646, 0.0, -0.004494031919, 0.003518873823, 0.0, 0.004494031919, 0.003518873823]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2786671617,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.002344345036, 0.0, -0.001783728285, 0.001172172518, 0.0, 0.001783728285, 0.001172172518]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.27568236,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05800380,
            "CCSD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.003408844165, 0.0, -0.002343169064, 0.001704422083, 0.0, 0.002343169064, 0.001704422083]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00726213,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.0594557966607590,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001920220330437888,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01252024755551,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01207773525598,
                    0.00000000000000,
                    0.01032204616770,
                    -0.00022125614977,
                    0.00000000000000,
                    -0.01032204616770,
                    -0.00022125614977,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07486350,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00235837,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.009311184271,
                    0.0,
                    0.0,
                    -0.009082434632,
                    0.0,
                    0.012724190366,
                    -0.00011437482,
                    0.0,
                    -0.012724190366,
                    -0.00011437482,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0835030877,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0024016379,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004636606273,
                    0.0,
                    0.0,
                    -0.004551154292,
                    0.0,
                    0.015266304532,
                    -0.00004272599,
                    0.0,
                    -0.015266304532,
                    -0.00004272599,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.15483909,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03523134,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.025476049585,
                    0.0,
                    0.013480567736,
                    -0.012738024793,
                    0.0,
                    -0.013480567736,
                    -0.012738024793,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.17097347,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03530244,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027082665714, 0.0, 0.014406579724, -0.013541332857, 0.0, -0.014406579724, -0.013541332857]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1770997033,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0340788149,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.030686886723, 0.0, 0.016619966665, -0.015343443362, 0.0, -0.016619966665, -0.015343443362]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_uhf,
            "MP2 CORRELATION ENERGY": -0.19550726,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04161470,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.013708831104,
                    0.0,
                    0.005340400162,
                    -0.006854415552,
                    0.0,
                    -0.005340400162,
                    -0.006854415552,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.21083057,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04130147,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.014412459072, 0.0, 0.005610368822, -0.007206229536, 0.0, -0.005610368822, -0.007206229536]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2167706529,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0401283617,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.018122428755, 0.0, 0.007810952273, -0.009061214377, 0.0, -0.007810952273, -0.009061214377]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.0604436327328384,
            "MP2 SINGLES ENERGY": -0.0006940750313001934,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0020065676314032863,
            "MP2 TOTAL GRADIENT": np.array(
                [  # occ findif-5 ae df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01361287313486,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01314329502424,
                    0.00000000000000,
                    0.01029838165151,
                    -0.00023478905531,
                    0.00000000000000,
                    -0.01029838165151,
                    -0.00023478905531,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0834776542,  # p4n
            "LCCD TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.15947485,
            "MP2 SINGLES ENERGY": -0.00282982,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03544639,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.025593521597,
                    0.0,
                    0.013495283342,
                    -0.012796760798,
                    0.0,
                    -0.013495283342,
                    -0.012796760798,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1792603912,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "ae",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_rohf,
            "MP2 CORRELATION ENERGY": -0.20051655,
            "MP2 SINGLES ENERGY": -0.00298400,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.04178365,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 ae df+df
                    0.0,
                    0.0,
                    0.013865245912,
                    0.0,
                    0.005377216253,
                    -0.006932622956,
                    0.0,
                    -0.005377216253,
                    -0.006932622956,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2190866990,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  DF-FC-DF  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.201610660387,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0535212487451535,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.00314716362539,
                    0.00000000000000,
                    0.00000000000000,
                    -0.00314716362539,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20461763,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000595617648, 0.0, 0.0, 0.000595617648]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2080860831,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002514968877, 0.0, 0.0, -0.002514968877]).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.20694032546082639,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04775171,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.002316563628,
                    0.0,
                    0.0,
                    -0.002316563628,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.001922093564526723,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_adz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_adz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.21939028,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05587974,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.010243193827,
                    0.0,
                    0.005885789424,
                    -0.005121596913,
                    0.0,
                    -0.005885789424,
                    -0.005121596913,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.22424330,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.004605219936, 0.0, 0.002923029606, -0.002302609968, 0.0, -0.002923029606, -0.002302609968]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2297412879,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.009150916979, 0.0, 0.005603501036, -0.00457545849, 0.0, -0.005603501036, -0.00457545849]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.22729554,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04931891,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.008124347934,
                    0.0,
                    0.004987676555,
                    -0.004062173967,
                    0.0,
                    -0.004987676555,
                    -0.004062173967,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00521721,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "h2o",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_h2o_qz2p_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_h2o_qz2p_df_rhf,
            "MP2 CORRELATION ENERGY": -0.24512893,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.06126089,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.000298272081,
                    0.0,
                    -0.000570968013,
                    -0.00014913604,
                    0.0,
                    0.000570968013,
                    -0.00014913604,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.24746227,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.006198368687, 0.0, -0.004115138427, 0.003099184344, 0.0, 0.004115138427, 0.003099184344]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2531777549,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, -0.001506644114, 0.0, -0.001400658245, 0.000753322057, 0.0, 0.001400658245, 0.000753322057]
            ).reshape((-1, 3)),
            "CCSD CORRELATION ENERGY": -0.25031508,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.05404876,
            "CCSD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    -0.002520920562,
                    0.0,
                    -0.001932133533,
                    0.001260460281,
                    0.0,
                    0.001932133533,
                    0.001260460281,
                ]
            ).reshape((-1, 3)),
            "(T) CORRECTION ENERGY": -0.00709505,
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.058390006825,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.001768919072594215,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.00000000000000,
                    0.00000000000000,
                    0.01231996225662,
                    0.00000000000000,
                    0.00000000000000,
                    -0.01186374280678,
                    0.00000000000000,
                    0.01031743020277,
                    -0.00022810972492,
                    0.00000000000000,
                    -0.01031743020277,
                    -0.00022810972492,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.07380592,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.00220531,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.00909477332,
                    0.0,
                    0.0,
                    -0.008853050692,
                    0.0,
                    0.012720593477,
                    -0.000120861314,
                    0.0,
                    -0.012720593477,
                    -0.000120861314,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0824996438,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0022545103,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [
                    0.0,
                    0.0,
                    0.004394350773,
                    0.0,
                    0.0,
                    -0.004297223912,
                    0.0,
                    0.0152632844,
                    -0.000048563431,
                    0.0,
                    -0.0152632844,
                    -0.000048563431,
                ]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_uhf,
            "MP2 CORRELATION ENERGY": -0.15240678,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03448325,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.026264866471,
                    0.0,
                    0.013987430104,
                    -0.013132433236,
                    0.0,
                    -0.013987430104,
                    -0.013132433236,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.16868589,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03460509,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.027853586058, 0.0, 0.014898421791, -0.013926793029, 0.0, -0.014898421791, -0.013926793029]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1748446809,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0333897039,  # dfocc
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.031455324295, 0.0, 0.017108389967, -0.015727662148, 0.0, -0.017108389967, -0.015727662148]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "uhf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_uhf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_uhf,
            "MP2 CORRELATION ENERGY": -0.17116675,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03822296,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.01471721142,
                    0.0,
                    0.005840479593,
                    -0.00735860571,
                    0.0,
                    -0.005840479593,
                    -0.00735860571,
                ]
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.18574570,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.03792036,  # dfocc
            "MP3 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.015407701186, 0.0, 0.006091123422, -0.007703850593, 0.0, -0.006091123422, -0.007703850593]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1916908596,  # p4n
            "LCCD SINGLES ENERGY": 0.0000000000,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.0367574293,
            "LCCD TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.019127907852, 0.0, 0.00829827557, -0.009563953926, 0.0, -0.00829827557, -0.009563953926]
            ).reshape((-1, 3)),
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "bh3p",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_bh3p_dz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.059372748391,
            "MP2 SINGLES ENERGY": -0.000688391888527046,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.0018535174789756292,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.01340658,
                    0.0,
                    0.0,
                    -0.01292306,
                    0.0,
                    0.01029363,
                    -0.00024176,
                    0.0,
                    -0.01029363,
                    -0.00024176,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.0824737155,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(12).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(144).reshape((12, 12)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "aug-cc-pvdz",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_adz_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_adz_df_rohf,
            "MP2 CORRELATION ENERGY": -0.15700408,
            "MP2 SINGLES ENERGY": -0.00280619,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03469227,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.026382129796,
                    0.0,
                    0.014000533629,
                    -0.013191064898,
                    0.0,
                    -0.014000533629,
                    -0.013191064898,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1769909051,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    {
        "meta": {
            "system": "nh2",
            "basis": "cfour-qz2p",
            "scf_type": "df",
            "reference": "rohf",
            "fcae": "fc",
            "corl_type": "df",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_nh2_qz2p_df_rohf,
            "HF TOTAL GRADIENT": _grad_scf_nh2_qz2p_df_rohf,
            "MP2 CORRELATION ENERGY": -0.17609909,
            "MP2 SINGLES ENERGY": -0.00294363,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.03837317,
            "MP2 TOTAL GRADIENT": np.array(
                [  # dfmp2 findif-5 fc df+df
                    0.0,
                    0.0,
                    0.014870916178,
                    0.0,
                    0.00587474124,
                    -0.007435458089,
                    0.0,
                    -0.00587474124,
                    -0.007435458089,
                ]
            ).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.1939804718,  # p4n
            # "XXX TOTAL GRADIENT": np.zeros(9).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(81).reshape((9, 9)),
        },
    },
    # <<<  lopsided SCF/CORL algorithms  >>>
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.201612517228,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05348507322421174,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00316682, 0.0, 0.0, -0.00316682]  # occ findif-5 fc df+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20452073,
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918306,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "fc",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.20162236483,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.053488165399,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00316652, 0.0, 0.0, -0.00316652]  # occ findif-5 fc cd+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20453091,
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04918609,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    # {
    #    "meta": {
    #        "system": "hf",
    #        "basis": "cc-pvdz",
    #        "scf_type": "df",
    #        "reference": "rhf",
    #        "fcae": "fc",
    #        "corl_type": "cd",
    #    },
    #    "data": {
    #        "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
    #        "MP2 CORRELATION ENERGY": -0.201609396752,
    #        #            "MP2 TOTAL ENERGY": -100.221010002381,
    #        "MP2 SINGLES ENERGY": 0.0,
    #        "MP2 SAME-SPIN CORRELATION ENERGY": -0.4,
    #    },
    # },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "fc",
    #            "corl_type": "conv",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -0.058421122206,
    #            #            "MP2 TOTAL ENERGY": -26.003551681354,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -0.5,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "fc",
    #            "corl_type": "cd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -0.058409837177,
    #            #            "MP2 TOTAL ENERGY": -26.003540396324,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -0.7,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "fc",
    #            "corl_type": "conv",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -0.060939211739,
    #            #            "MP2 TOTAL ENERGY": -26.004545733768,
    #            "MP2 SINGLES ENERGY": 1.1,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -1.1,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "fc",
    #            "corl_type": "cd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -0.059393510962,
    #            #            "MP2 TOTAL ENERGY": -26.003000032991,
    #            "MP2 SINGLES ENERGY": 1.3,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -1.3,
    #        },
    #    },
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "df",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_df_rhf,
            "MP2 CORRELATION ENERGY": -0.2037668844651997,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427252944164894,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00281165, 0.0, 0.0, -0.00281165]  # occ findif-5 ae df+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20646457,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04988633,  # dfocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.000924553428, 0.0, 0.0, 0.000924553428]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2098900858,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048336089041,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002196305414, 0.0, 0.0, -0.002196305414]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107275173,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048456320034,  # fnocc
            "CCSD CORRELATION ENERGY": -0.20872812,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857038,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    {
        "meta": {
            "system": "hf",
            "basis": "cc-pvdz",
            "scf_type": "cd",
            "reference": "rhf",
            "fcae": "ae",
            "corl_type": "conv",
        },
        "data": {
            "HF TOTAL ENERGY": _scf_hf_dz_cd_rhf,
            "HF TOTAL GRADIENT": _grad_scf_hf_dz_cd_rhf,
            "MP2 CORRELATION ENERGY": -0.2037767503537,
            "MP2 SINGLES ENERGY": 0.0,
            "MP2 SAME-SPIN CORRELATION ENERGY": -0.05427563053,
            "MP2 TOTAL GRADIENT": np.array(
                [0.0, 0.0, 0.00281136, 0.0, 0.0, -0.00281136]  # occ findif-5 ae cd+conv
            ).reshape((-1, 3)),
            "MP3 CORRELATION ENERGY": -0.20647475,  # dfocc
            "MP3 SINGLES ENERGY": 0.0,
            "MP3 SAME-SPIN CORRELATION ENERGY": -0.04988936,  # dfocc
            "MP3 TOTAL GRADIENT": np.array([0.0, 0.0, -0.00092571521, 0.0, 0.0, 0.00092571521]).reshape((-1, 3)),
            "LCCD CORRELATION ENERGY": -0.2099004485,  # p4n
            "LCCD SINGLES ENERGY": 0.0,
            "LCCD SAME-SPIN CORRELATION ENERGY": -0.048339111990,  # fnocc
            "LCCD TOTAL GRADIENT": np.array([0.0, 0.0, 0.002195646447, 0.0, 0.0, -0.002195646447]).reshape((-1, 3)),
            "LCCSD CORRELATION ENERGY": -0.2107380019,  # p4n
            "LCCSD SINGLES ENERGY": 0.0,
            "LCCSD SAME-SPIN CORRELATION ENERGY": -0.048459381537,  # fnocc
            "CCSD CORRELATION ENERGY": -0.20873814,
            "CCSD SINGLES ENERGY": 0.0,
            "CCSD SAME-SPIN CORRELATION ENERGY": -0.04857333,
            # "XXX TOTAL GRADIENT": np.zeros(6).reshape((-1, 3)),
            # "XXX TOTAL HESSIAN": np.zeros(36).reshape((6, 6)),
        },
    },
    #    {
    #        "meta": {
    #            "system": "hf",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rhf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_hf_dz_df_rhf,
    #            "MP2 CORRELATION ENERGY": -2.3,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.3,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "ae",
    #            "corl_type": "conv",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -2.4,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.4,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "uhf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_uhf,
    #            "MP2 CORRELATION ENERGY": -2.5,
    #            "MP2 SINGLES ENERGY": 0.0,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.5,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "ae",
    #            "corl_type": "conv",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -2.7,
    #            "MP2 SINGLES ENERGY": -2.7,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.7,
    #        },
    #    },
    #    {
    #        "meta": {
    #            "system": "bh3p",
    #            "basis": "cc-pvdz",
    #            "scf_type": "df",
    #            "reference": "rohf",
    #            "fcae": "ae",
    #            "corl_type": "cd",
    #        },
    #        "data": {
    #            "HF TOTAL ENERGY": _scf_bh3p_dz_df_rohf,
    #            "MP2 CORRELATION ENERGY": -2.8,
    #            "MP2 SINGLES ENERGY": -2.8,
    #            "MP2 SAME-SPIN CORRELATION ENERGY": -2.8,
    #        },
    #    },
]


for calc in _std_suite:
    if calc["data"]:
        if "MP2 CORRELATION ENERGY" in calc["data"]:
            calc["data"]["MP2 TOTAL ENERGY"] = calc["data"]["MP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            if "MP2 SINGLES ENERGY" in calc["data"]:
                calc["data"]["MP2 DOUBLES ENERGY"] = (
                    calc["data"]["MP2 CORRELATION ENERGY"] - calc["data"]["MP2 SINGLES ENERGY"]
                )
                if "MP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["MP2 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["MP2 CORRELATION ENERGY"]
                        - calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["MP2 SINGLES ENERGY"]
                    )
                    calc["data"]["SCS-MP2 CORRELATION ENERGY"] = (
                        (1 / 3) * calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                        + (6 / 5) * calc["data"]["MP2 OPPOSITE-SPIN CORRELATION ENERGY"]
                        + calc["data"]["MP2 SINGLES ENERGY"]
                    )
                    calc["data"]["SCS-MP2 TOTAL ENERGY"] = (
                        calc["data"]["SCS-MP2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                    )

        if "MP3 CORRELATION ENERGY" in calc["data"]:
            calc["data"]["MP3 TOTAL ENERGY"] = calc["data"]["MP3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            if "MP2 CORRELATION ENERGY" in calc["data"]:
                calc["data"]["MP2.5 CORRELATION ENERGY"] = 0.5 * (
                    calc["data"]["MP3 CORRELATION ENERGY"] + calc["data"]["MP2 CORRELATION ENERGY"]
                )
                calc["data"]["MP2.5 TOTAL ENERGY"] = (
                    calc["data"]["MP2.5 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

            if "MP3 SINGLES ENERGY" in calc["data"]:
                calc["data"]["MP3 DOUBLES ENERGY"] = (
                    calc["data"]["MP3 CORRELATION ENERGY"] - calc["data"]["MP3 SINGLES ENERGY"]
                )
                if "MP2 SINGLES ENERGY" in calc["data"]:
                    calc["data"]["MP2.5 SINGLES ENERGY"] = 0.5 * (
                        calc["data"]["MP3 SINGLES ENERGY"] + calc["data"]["MP2 SINGLES ENERGY"]
                    )
                    calc["data"]["MP2.5 DOUBLES ENERGY"] = (
                        calc["data"]["MP2.5 CORRELATION ENERGY"] - calc["data"]["MP2.5 SINGLES ENERGY"]
                    )
                if "MP3 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["MP3 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["MP3 CORRELATION ENERGY"]
                        - calc["data"]["MP3 SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["MP3 SINGLES ENERGY"]
                    )
                    if "MP2 SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                        calc["data"]["MP2.5 SAME-SPIN CORRELATION ENERGY"] = 0.5 * (
                            calc["data"]["MP3 SAME-SPIN CORRELATION ENERGY"]
                            + calc["data"]["MP2 SAME-SPIN CORRELATION ENERGY"]
                        )
                        calc["data"]["MP2.5 OPPOSITE-SPIN CORRELATION ENERGY"] = (
                            calc["data"]["MP2.5 CORRELATION ENERGY"]
                            - calc["data"]["MP2.5 SAME-SPIN CORRELATION ENERGY"]
                            - calc["data"]["MP2.5 SINGLES ENERGY"]
                        )

        if (
            "MP3 TOTAL GRADIENT" in calc["data"]
            and "MP2 TOTAL GRADIENT" in calc["data"]
            and "HF TOTAL GRADIENT" in calc["data"]
        ):
            calc["data"]["MP2.5 TOTAL GRADIENT"] = 0.5 * (
                calc["data"]["MP3 TOTAL GRADIENT"] + calc["data"]["MP2 TOTAL GRADIENT"]
            )

        if "MP4(SDQ) CORRELATION ENERGY" in calc["data"]:
            calc["data"]["MP4(SDQ) TOTAL ENERGY"] = (
                calc["data"]["MP4(SDQ) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            if "MP4(T) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["MP4 CORRELATION ENERGY"] = (
                    calc["data"]["MP4(SDQ) CORRELATION ENERGY"] + calc["data"]["MP4(T) CORRECTION ENERGY"]
                )
                calc["data"]["MP4 TOTAL ENERGY"] = (
                    calc["data"]["MP4 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )
                if "MP3 CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["MP4 CORRECTION ENERGY"] = (
                        calc["data"]["MP4 CORRELATION ENERGY"] - calc["data"]["MP3 CORRELATION ENERGY"]
                    )

        if "CISD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CISD TOTAL ENERGY"] = (
                calc["data"]["CISD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "QCISD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["QCISD TOTAL ENERGY"] = (
                calc["data"]["QCISD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            if "QCISD(T) CORRECTION ENERGY" in calc["data"]:
                calc["data"]["QCISD(T) CORRELATION ENERGY"] = (
                    calc["data"]["QCISD CORRELATION ENERGY"] + calc["data"]["QCISD(T) CORRECTION ENERGY"]
                )
                calc["data"]["QCISD(T) TOTAL ENERGY"] = (
                    calc["data"]["QCISD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
                )

        if "FCI CORRELATION ENERGY" in calc["data"]:
            calc["data"]["FCI TOTAL ENERGY"] = calc["data"]["FCI CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]

        if "LCCD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["LCCD TOTAL ENERGY"] = (
                calc["data"]["LCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            if "LCCD SINGLES ENERGY" in calc["data"]:
                calc["data"]["LCCD DOUBLES ENERGY"] = (
                    calc["data"]["LCCD CORRELATION ENERGY"] - calc["data"]["LCCD SINGLES ENERGY"]
                )
                if "LCCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["LCCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["LCCD CORRELATION ENERGY"]
                        - calc["data"]["LCCD SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["LCCD SINGLES ENERGY"]
                    )

        if "LCCSD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["LCCSD TOTAL ENERGY"] = (
                calc["data"]["LCCSD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            if "LCCSD SINGLES ENERGY" in calc["data"]:
                calc["data"]["LCCSD DOUBLES ENERGY"] = (
                    calc["data"]["LCCSD CORRELATION ENERGY"] - calc["data"]["LCCSD SINGLES ENERGY"]
                )
                if "LCCSD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["LCCSD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["LCCSD CORRELATION ENERGY"]
                        - calc["data"]["LCCSD SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["LCCSD SINGLES ENERGY"]
                    )

        if "CCD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCD TOTAL ENERGY"] = calc["data"]["CCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            if "CCD SINGLES ENERGY" in calc["data"]:
                calc["data"]["CCD DOUBLES ENERGY"] = (
                    calc["data"]["CCD CORRELATION ENERGY"] - calc["data"]["CCD SINGLES ENERGY"]
                )
                if "CCD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["CCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["CCD CORRELATION ENERGY"]
                        - calc["data"]["CCD SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["CCD SINGLES ENERGY"]
                    )

        if "CCSD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSD TOTAL ENERGY"] = (
                calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            if "CCSD SINGLES ENERGY" in calc["data"]:
                calc["data"]["CCSD DOUBLES ENERGY"] = (
                    calc["data"]["CCSD CORRELATION ENERGY"] - calc["data"]["CCSD SINGLES ENERGY"]
                )
                if "CCSD SAME-SPIN CORRELATION ENERGY" in calc["data"]:
                    calc["data"]["CCSD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                        calc["data"]["CCSD CORRELATION ENERGY"]
                        - calc["data"]["CCSD SAME-SPIN CORRELATION ENERGY"]
                        - calc["data"]["CCSD SINGLES ENERGY"]
                    )

        if "T(CCSD) CORRECTION ENERGY" in calc["data"]:
            calc["data"]["CCSD+T(CCSD) CORRELATION ENERGY"] = (
                calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["T(CCSD) CORRECTION ENERGY"]
            )
            calc["data"]["CCSD+T(CCSD) TOTAL ENERGY"] = (
                calc["data"]["CCSD+T(CCSD) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "(T) CORRECTION ENERGY" in calc["data"]:
            calc["data"]["CCSD(T) CORRELATION ENERGY"] = (
                calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["(T) CORRECTION ENERGY"]
            )
            calc["data"]["CCSD(T) TOTAL ENERGY"] = (
                calc["data"]["CCSD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "A-(T) CORRECTION ENERGY" in calc["data"]:
            calc["data"]["A-CCSD(T) CORRELATION ENERGY"] = (
                calc["data"]["CCSD CORRELATION ENERGY"] + calc["data"]["A-(T) CORRECTION ENERGY"]
            )
            calc["data"]["A-CCSD(T) TOTAL ENERGY"] = (
                calc["data"]["A-CCSD(T) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDT-1A CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDT-1A TOTAL ENERGY"] = (
                calc["data"]["CCSDT-1A CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDT-1B CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDT-1B TOTAL ENERGY"] = (
                calc["data"]["CCSDT-1B CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDT-2 CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDT-2 TOTAL ENERGY"] = (
                calc["data"]["CCSDT-2 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDT-3 CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDT-3 TOTAL ENERGY"] = (
                calc["data"]["CCSDT-3 CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDT CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDT TOTAL ENERGY"] = (
                calc["data"]["CCSDT CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "(Q) CORRECTION ENERGY" in calc["data"]:
            calc["data"]["CCSDT(Q) CORRELATION ENERGY"] = (
                calc["data"]["CCSDT CORRELATION ENERGY"] + calc["data"]["(Q) CORRECTION ENERGY"]
            )
            calc["data"]["CCSDT(Q) TOTAL ENERGY"] = (
                calc["data"]["CCSDT(Q) CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "CCSDTQ CORRELATION ENERGY" in calc["data"]:
            calc["data"]["CCSDTQ TOTAL ENERGY"] = (
                calc["data"]["CCSDTQ CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )

        if "OLCCD CORRELATION ENERGY" in calc["data"]:
            calc["data"]["OLCCD TOTAL ENERGY"] = (
                calc["data"]["OLCCD CORRELATION ENERGY"] + calc["data"]["HF TOTAL ENERGY"]
            )
            calc["data"]["OLCCD OPPOSITE-SPIN CORRELATION ENERGY"] = (
                calc["data"]["OLCCD CORRELATION ENERGY"]
                - calc["data"]["OLCCD REFERENCE CORRECTION ENERGY"]
                - calc["data"]["OLCCD SAME-SPIN CORRELATION ENERGY"]
            )

    calc["data"].update(_std_generics[f"{calc['meta']['system']}_{calc['meta']['basis']}_{calc['meta']['fcae']}"])


def answer_hash(**kwargs):
    system = kwargs.pop("system")
    basis = kwargs.pop("basis")
    scf_type = kwargs.pop("scf_type")
    reference = kwargs.pop("reference")
    fcae = kwargs.pop("fcae")
    corl_type = kwargs.pop("corl_type")

    return "_".join([system, basis, scf_type, reference, fcae, corl_type])


std_suite = {answer_hash(**calc["meta"]): calc["data"] for calc in _std_suite}
