% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/access.R, R/coercion.R, R/constructors.R,
%   R/dataframe.R, R/printsum.R
\name{network}
\alias{network}
\alias{is.network}
\alias{as.network.network}
\alias{print.summary.network}
\alias{$<-.network}
\alias{<-.network}
\alias{as.network}
\alias{network.copy}
\alias{as.network.data.frame}
\alias{print.network}
\alias{summary.network}
\title{Network Objects}
\usage{
is.network(x)

as.network(x, ...)

network(
  x,
  vertex.attr = NULL,
  vertex.attrnames = NULL,
  directed = TRUE,
  hyper = FALSE,
  loops = FALSE,
  multiple = FALSE,
  bipartite = FALSE,
  ...
)

network.copy(x)

\method{as.network}{data.frame}(
  x,
  directed = TRUE,
  vertices = NULL,
  hyper = FALSE,
  loops = FALSE,
  multiple = FALSE,
  bipartite = FALSE,
  bipartite_col = "is_actor",
  ...
)

\method{print}{network}(
  x,
  matrix.type = which.matrix.type(x),
  mixingmatrices = FALSE,
  na.omit = TRUE,
  print.adj = FALSE,
  ...
)

\method{summary}{network}(object, na.omit = TRUE, mixingmatrices = FALSE, print.adj = TRUE, ...)
}
\arguments{
\item{x}{for \code{network}, a matrix giving the network structure in
adjacency, incidence, or edgelist form; otherwise, an object of class
\code{network}.}

\item{...}{additional arguments.}

\item{vertex.attr}{optionally, a list containing vertex attributes.}

\item{vertex.attrnames}{optionally, a list containing vertex attribute
names.}

\item{directed}{logical; should edges be interpreted as directed?}

\item{hyper}{logical; are hyperedges allowed?}

\item{loops}{logical; should loops be allowed?}

\item{multiple}{logical; are multiplex edges allowed?}

\item{bipartite}{count; should the network be interpreted as bipartite? If
present (i.e., non-NULL, non-FALSE) it is the count of the number of actors
in the bipartite network. In this case, the number of nodes is equal to the
number of actors plus the number of events (with all actors preceeding all
events). The edges are then interpreted as nondirected. Values of
bipartite==0 are permited, indicating a bipartite network with zero-sized
first partition.}

\item{vertices}{If \code{x} is a \code{data.frame}, \code{vertices} is an optional
\code{data.frame} containing the vertex attributes. The first column is assigned
to the \code{"vertex.names"} and additional columns are used to set vertex attributes
using their column names. If \code{bipartite} is \code{TRUE}, a \code{logical} column
named \code{"is_actor"} (or the name of a column specified using the
\code{bipartite_col} parameter) can be provided indicating which vertices
should be considered as actors. If not provided, vertices referenced in the
first column of \code{x} are assumed to be the network's actors. If your
network has isolates (i.e. there are vertices referenced in \code{vertices}
that are not referenced in \code{x}), the \code{"is_actor"} column is required.}

\item{bipartite_col}{\code{character(1L)}, default: \code{"is_actor"}.
The name of the \code{logical} column indicating which vertices should be
considered as actors in bipartite networks.}

\item{matrix.type}{one of \code{"adjacency"}, \code{"edgelist"},
\code{"incidence"}. See \code{\link{edgeset.constructors}} for details and
optional additional arguments}

\item{mixingmatrices}{logical; print the mixing matrices for the discrete
attributes?}

\item{na.omit}{logical; omit summarization of missing attributes in
\code{network}?}

\item{print.adj}{logical; print the network adjacency structure?}

\item{object}{an object of class \code{network}.}
}
\value{
\code{network}, \code{as.network}, and \code{print.network} all
return a network class object; \code{is.network} returns TRUE or FALSE.
}
\description{
Construct, coerce to, test for and print \code{network} objects.
}
\details{
\code{network} constructs a \code{network} class object from a matrix
representation. If the \code{matrix.type} parameter is not specified, it
will make a guess as to the intended \code{edgeset.constructors} function to
call based on the format of these input matrices. If the class of \code{x}
is not a matrix, network construction can be dispatched to other methods.
For example, If the \code{ergm} package is loaded, \code{network()} can
function as a shorthand for \code{as.network.numeric} with
\code{x} as an integer specifying the number of nodes to be created in the
random graph.

If the \code{ergm} package is loaded, \code{network} can function as a
shorthand for \code{as.network.numeric} if \code{x} is an integer specifying
the number of nodes. See the help page for
\code{as.network.numeric} in \code{ergm} package for details.

\code{network.copy} creates a new \code{network} object which duplicates its
supplied argument.  (Direct assignment with \code{<-} should be used rather
than \code{network.copy} in most cases.)

\code{as.network} tries to coerce its argument to a network, using the
\code{as.network.matrix} functions if \code{x} is a matrix. (If the argument
is already a network object, it is returned as-is and all other arguments
are ignored.)

\code{is.network} tests whether its argument is a network (in the sense that
it has class \code{network}).

\code{print.network} prints a network object in one of several possible
formats.  It also prints the list of global attributes of the network.

\code{summary.network} provides similar information.
}
\note{
Between versions 0.5 and 1.2, direct assignment of a network object
created a pointer to the original object, rather than a copy.  As of version
1.2, direct assignment behaves in the same manner as \code{network.copy}.
Direct use of the latter is thus superfluous in most situations, and is
discouraged.

Many of the network package functions modify their network object arguments
in-place. For example, \code{set.network.attribute(net,"myVal",5)} will have
the same effect as \code{net<-set.network.attribute(net,"myVal",5)}.
Unfortunately, the current implementation of in-place assignment breaks when
the network argument is an element of a list or a named part of another
object. So \code{set.network.attribute(myListOfNetworks[[1]],"myVal",5)}
will silently fail to modify its network argument, likely leading to
incorrect output.
}
\examples{

m <- matrix(rbinom(25,1,.4),5,5)
diag(m) <- 0
g <- network(m, directed=FALSE)
summary(g)

h <- network.copy(g)       #Note: same as h<-g
summary(h)

# networks from data frames ===========================================================
#* simple networks ====================================================================
simple_edge_df <- data.frame(
  from = c("b", "c", "c", "d", "a"),
  to = c("a", "b", "a", "a", "b"),
  weight = c(1, 1, 2, 2, 3),
  stringsAsFactors = FALSE
)
simple_edge_df

as.network(simple_edge_df)

# simple networks with vertices =======================================================
simple_vertex_df <- data.frame(
  name = letters[1:5],
  residence = c("urban", "rural", "suburban", "suburban", "rural"),
  stringsAsFactors = FALSE
)
simple_vertex_df

as.network(simple_edge_df, vertices = simple_vertex_df)

as.network(simple_edge_df,
  directed = FALSE, vertices = simple_vertex_df,
  multiple = TRUE
)

#* splitting multiplex data frames into multiple networks =============================
simple_edge_df$relationship <- c(rep("friends", 3), rep("colleagues", 2))
simple_edge_df

lapply(split(simple_edge_df, f = simple_edge_df$relationship),
  as.network,
  vertices = simple_vertex_df
)

#* bipartite networks without isolates ================================================
bip_edge_df <- data.frame(
  actor = c("a", "a", "b", "b", "c", "d", "d", "e"),
  event = c("e1", "e2", "e1", "e3", "e3", "e2", "e3", "e1"),
  actor_enjoyed_event = rep(c(TRUE, FALSE), 4),
  stringsAsFactors = FALSE
)
bip_edge_df

bip_node_df <- data.frame(
  node_id = c("a", "e1", "b", "e2", "c", "e3", "d", "e"),
  node_type = c(
    "person", "event", "person", "event", "person",
    "event", "person", "person"
  ),
  color = c(
    "red", "blue", "red", "blue", "red", "blue",
    "red", "red"
  ),
  stringsAsFactors = FALSE
)
bip_node_df

as.network(bip_edge_df, directed = FALSE, bipartite = TRUE)
as.network(bip_edge_df, directed = FALSE, vertices = bip_node_df, bipartite = TRUE)

#* bipartite networks with isolates ===================================================
bip_nodes_with_isolates <- rbind(
  bip_node_df,
  data.frame(
    node_id = c("f", "e4"),
    node_type = c("person", "event"),
    color = c("red", "blue"),
    stringsAsFactors = FALSE
  )
)
# indicate which vertices are actors via a column named `"is_actor"`
bip_nodes_with_isolates$is_actor <- bip_nodes_with_isolates$node_type == "person"
bip_nodes_with_isolates

as.network(bip_edge_df,
  directed = FALSE, vertices = bip_nodes_with_isolates,
  bipartite = TRUE
)

#* hyper networks from data frames ====================================================
hyper_edge_df <- data.frame(
  from = c("a/b", "b/c", "c/d/e", "d/e"),
  to = c("c/d", "a/b/e/d", "a/b", "d/e"),
  time = 1:4,
  stringsAsFactors = FALSE
)
tibble::as_tibble(hyper_edge_df)

# split "from" and "to" at `"/"`, coercing them to list columns
hyper_edge_df$from <- strsplit(hyper_edge_df$from, split = "/")
hyper_edge_df$to <- strsplit(hyper_edge_df$to, split = "/")
tibble::as_tibble(hyper_edge_df)

as.network(hyper_edge_df,
  directed = FALSE, vertices = simple_vertex_df,
  hyper = TRUE, loops = TRUE
)

# convert network objects back to data frames =========================================
simple_g <- as.network(simple_edge_df, vertices = simple_vertex_df)
as.data.frame(simple_g)
as.data.frame(simple_g, unit = "vertices")

bip_g <- as.network(bip_edge_df,
  directed = FALSE, vertices = bip_node_df,
  bipartite = TRUE
)
as.data.frame(bip_g)
as.data.frame(bip_g, unit = "vertices")

hyper_g <- as.network(hyper_edge_df,
  directed = FALSE, vertices = simple_vertex_df,
  hyper = TRUE, loops = TRUE
)
as.data.frame(hyper_g)
as.data.frame(hyper_g, unit = "vertices")
}
\references{
Butts, C. T.  (2008).  \dQuote{network: a Package for Managing
Relational Data in R.} \emph{Journal of Statistical Software}, 24(2).
\url{https://www.jstatsoft.org/v24/i02/}
}
\seealso{
\code{\link{network.initialize}}, \code{\link{attribute.methods}},
\code{\link{as.network.matrix}}, \code{\link{as.matrix.network}},
\code{\link{deletion.methods}}, \code{\link{edgeset.constructors}},
\code{\link{network.indicators}}, \code{\link{plot.network}}
}
\author{
Carter T. Butts \email{buttsc@uci.edu} and David Hunter
\email{dhunter@stat.psu.edu}
}
\keyword{classes}
\keyword{graphs}
