# ifndef _SKIT_BICGSTAB_H
# define _SKIT_BICGSTAB_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:bicgstab
NAME: @code{bicgstab} - bi-conjugate gradient stabilized method 
@findex bicgstab
@cindex stabilized conjugate gradient
@cindex iterative solver
SYNOPSIS:
 @example 
    int bicgstab (const Matrix &A, Vector &x, const Vector &b,
                   const Preconditioner &M, int &max_iter, Real &tol);
 @end example 
EXAMPLE:
 @noindent
 The simplest call to 'bicgstab' has the folling form:
 @example 
    int status = bicgstab(a, x, b, EYE, 100, 1e-7);
 @end example 
DESCRIPTION:       
 @noindent
 @code{bicgstab} solves the unsymmetric linear system Ax = b 
 using the preconditioned bi-conjugate gradient stabilized method

 @noindent
 The return value indicates convergence within max_iter (input)
 iterations (0), or no convergence within max_iter iterations (1).
 Upon successful return, output arguments have the following values:
  @table @code
    @itemx x
    	approximate solution to Ax = b

    @itemx max_iter
	the number of iterations performed before the tolerance was reached

    @itemx tol
    	the residual after the final iteration
  @end table
SEE ALSO:
  "cg", class "csr", class "vec"
NOTE: 
  @noindent
  @code{bicgstab} is an iterative template routine.

  @noindent
  @code{bicgstab} follows the algorithm described on p. 24 in
  @quotation
        Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods,
        2nd Edition,
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
        R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994,
        @url{ftp.netlib.org/templates/templates.ps}.
  @end quotation

  @noindent
  The present implementation is inspired from @code{IML++ 1.2} iterative method library,
  @url{http://math.nist.gov/iml++}.
AUTHOR: 
    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    12 march 1997

METHODS: @bicgstab
End:
*/
// ========== [ method implementation ] ====================================

namespace rheolef { 
template < class Matrix, class Vector, class Preconditioner, class Real >
int 
bicgstab(const Matrix &A, Vector &x, const Vector &b,
         const Preconditioner &M, int &max_iter, Real &tol)
{
  Real resid;
  Real rho_1, rho_2=0, alpha, beta, omega;
  Vector p, phat, s, shat, t, v;

  Real normb = norm(b);
  Vector r = b - A * x;
  Vector rtilde = r;

  if (normb == Real(0))
    normb = 1;
  
  if ((resid = norm(r) / normb) <= tol) {
    tol = resid;
    max_iter = 0;
    return 0;
  }

  for (int i = 1; i <= max_iter; i++) {
    rho_1 = dot(rtilde, r);
    if (rho_1 == Real(0)) {
      tol = norm(r) / normb;
      return 2;
    }
    if (i == 1)
      p = r;
    else {
      beta = (rho_1/rho_2) * (alpha/omega);
      p = r + beta * (p - omega * v);
    }
    phat = M.solve(p);
    v = A * phat;
    alpha = rho_1 / dot(rtilde, v);
    s = r - alpha * v;
    if ((resid = norm(s)/normb) < tol) {
      x += alpha * phat;
      tol = resid;
      return 0;
    }
    shat = M.solve(s);
    t = A * shat;
    omega = dot(t,s) / dot(t,t);
    x += alpha * phat + omega * shat;
    r = s - omega * t;

    rho_2 = rho_1;
    if ((resid = norm(r) / normb) < tol) {
      tol = resid;
      max_iter = i;
      return 0;
    }
    if (omega == Real(0)) {
      tol = norm(r) / normb;
      return 3;
    }
  }
  tol = resid;
  return 1;
}
}// namespace rheolef
#endif // _SKIT_BICGSTAB_H
