""" Python module for properties across a normal shock in perfect gases

Assumptions:
1) One dimensional flow
2) Constant area duct
3) Perfect gas with constant specific heat and molecular weights
4) Adiabatic flow

"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.2
"""


from scipy import optimize
from math import *
from Isentropic import Isentropic

class NormalShock:
    def __init__(self, g=1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = 1.4
        self.Isentropic = Isentropic(self.g)

    def get_My_from_Mx(self, Mx):
        """Returns Mach No My behind a normal shock from Mach No Mx ahead of the shock"""
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            return sqrt((2/(self.g-1) + Mx**2)/(2*self.g/(self.g-1)*Mx**2 -1))

    def get_Py_by_Px_from_Mx(self, Mx):
        """Returns py/px across a normal shock from  Mx """
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            return 2*self.g/(self.g+1)* Mx**2 - (self.g-1)/(self.g+1)

    def get_Ty_by_Tx_from_Mx(self, Mx):
        """Returns Ty/Tx across a normal shock from  Mx """
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            return (1 + (self.g-1)*0.5* Mx**2)*(2*self.g/(self.g-1)*Mx**2 - 1)/((self.g+1)**2/(self.g-1)*0.5* Mx**2)

    def get_rhoy_by_rhox_from_Mx(self, Mx):
        """Returns rhoy/rhox across a normal shock from Mx"""
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            return self.get_Py_by_Px_from_Mx(Mx) / self.get_Ty_by_Tx_from_Mx(Mx)
    
    def get_Poy_by_Pox_from_Mx(self, Mx):
        """Returns poy/pox across a normal shock from Mx"""
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            My = self.get_My_from_Mx(Mx)
            return self.Isentropic.get_P_by_Po_from_M(Mx) * self.get_Py_by_Px_from_Mx(Mx) / self.Isentropic.get_P_by_Po_from_M(My)

    def get_Poy_by_Px_from_Mx(self, Mx):
        """Returns poy/px across a normal shock from Mx"""
        if Mx <= 1:
            raise Exception("Mach No ahead of ths shock Mx should be greater than 1")
        else:
            My = self.get_My_from_Mx(Mx)
            return self.get_Py_by_Px_from_Mx(Mx) / self.Isentropic.get_P_by_Po_from_M(My)
    
    def get_Mx_from_My(self, My):
        """Returns Mach No Mx ahead of a normal shock from Mach No My behind the shock"""
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else:
            return sqrt((1 + (self.g-1)* 0.5 * My**2)/(self.g * My**2 - (self.g-1)*0.5))

    def get_Py_by_Px_from_My(self, My):
        """Returns py/px across a normal shock from  My """
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else :
            Mx = self.get_Mx_from_My(My)
            return self.get_Py_by_Px_from_Mx(Mx)
    
    def get_Ty_by_Tx_from_My(self, My):
        """Returns Ty/Tx across a normal shock from  My """
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else :
            Mx = self.get_Mx_from_My(My)
            return self.get_Ty_by_Tx_from_Mx(Mx)
    
    def get_rhoy_by_rhox_from_My(self, My):
        """Returns rhoy/rhox across a normal shock from My"""
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else :
            Mx = self.get_Mx_from_My(My)
            return self.get_rhoy_by_rhox_from_Mx(Mx)
     
    def get_Poy_by_Pox_from_My(self, My):
        """Returns poy/pox across a normal shock from My"""
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else :
            Mx = self.get_Mx_from_My(My)
            return self.get_Poy_by_Pox_from_Mx(Mx)

    def get_Poy_by_Px_from_My(self, My):
        """Returns poy/px across a normal shock from My"""
        if My >= 1.0 or My <= sqrt((self.g-1)/self.g * 0.5):
            raise Exception("Mach No behind a normal shock My should be between %s and 1.0" % str(sqrt((self.g-1)/self.g * 0.5)))
        else :
            Mx = self.get_Mx_from_My(My)
            return self.get_Poy_by_Px_from_Mx(Mx)
    

    def get_Mx_from_Ty_by_Tx(self, Ty_by_Tx):
        """Returns Mx from Ty/Tx equal to Ty_by_Tx"""
        if Ty_by_Tx <= 1.0:
            raise Exception("Ty/Tx must be greater than 1")
        else:
            a = 2*self.g*(self.g-1)
            b = 4*self.g - (self.g-1)**2 - Ty_by_Tx * (self.g+1)**2
            c = -2*(self.g-1)
            return sqrt((-b + sqrt(b**2 - 4*a*c))/a *0.5)
    
    def get_My_from_Ty_by_Tx(Self, Ty_by_Tx):
        """Returns Mx from Ty/Tx equal to Ty_by_Tx"""
        if Ty_by_Tx <= 1.0:
            raise Exception("Ty/Tx must be greater than 1")
        else:
            Mx = self.get_Mx_from_Ty_by_Tx(Ty_by_Tx)
            return self.get_My_from_Mx(Mx)

    def get_Py_by_Px_from_Ty_by_Tx(self, Ty_by_Tx):
        """Returns Py/Px from Ty/Tx equal to Ty_by_Tx"""
        if Ty_by_Tx <= 1.0:
            raise Exception("Ty/Tx must be greater than 1")
        else:
            Mx = self.get_Mx_from_Ty_by_Tx(Ty_by_Tx)
            return self.get_Py_by_Px_from_Mx(Mx)

    def get_Poy_by_Pox_from_Ty_by_Tx(self, Ty_by_Tx):
        """Returns poy/pox from Ty/Tx equal to Ty_by_Tx"""
        if Ty_by_Tx <= 1.0:
            raise Exception("Ty/Tx must be greater than 1")
        else:
            Mx = self.get_Mx_from_Ty_by_Tx(Ty_by_Tx)
            return self.get_Poy_by_Pox_from_Mx(Mx)

    def get_Poy_by_Px_from_Ty_by_Tx(self, Ty_by_Tx):
        """Returns poy/px from Ty/Tx equal to Ty_by_Tx"""
        if Ty_by_Tx <= 1.0:
            raise Exception("Ty/Tx must be greater than 1")
        else:
            Mx = self.get_Mx_from_Ty_by_Tx(Ty_by_Tx)
            return self.get_Poy_by_Px_from_Mx(Mx)

    def get_Mx_from_Py_by_Px(self, Py_by_Px):
        """Returns Mx from py/px equal to Py_by_Px"""
        if Py_by_Px <= 1:
            raise Exception("py/px must be greater than 1")
        else:
            return sqrt((Py_by_Px - 1)* (self.g+1)/self.g * 0.5 + 1)

    def get_My_from_Py_by_Px(self, Py_by_Px):
        """Returns My from py/px equal to Py_by_Px"""
        if Py_by_Px <= 1:
            raise Exception("py/px must be greater than 1")
        else:
            Mx = self.get_Mx_from_Py_by_Px(Py_by_Px)
            return self.get_My_from_Mx(Mx)

    def get_Ty_by_Tx_from_Py_by_Px(self, Py_by_Px):
        """Returns Ty/Tx from py/px equal to Py_by_Px"""
        if Py_by_Px <= 1:
            raise Exception("py/px must be greater than 1")
        else:
            Mx = self.get_Mx_from_Py_by_Px(Py_by_Px)
            return self.get_Ty_by_Tx_from_Mx(Mx)

    def get_Poy_by_Pox_from_Py_by_Px(self, Py_by_Px):
        """Returns poy/pox from Py/Px equal to Py_by_Px"""
        if Py_by_Px <= 1:
            raise Exception("py/px must be greater than 1")
        else:
            Mx = self.get_Mx_from_Py_by_Px(Py_by_Px)
            return self.get_Poy_by_Pox_from_Mx(Mx)

    def get_Poy_by_Px_from_Py_by_Px(self, Py_by_Px):
        """Returns Poy/Px from Py/Px equal to Py_by_Px"""
        if Py_by_Px <= 1:
            raise Exception("py/px must be greater than 1")
        else:
            Mx = self.get_Mx_from_Py_by_Px(Py_by_Px)
            return self.get_Poy_by_Px_from_Mx(Mx)

    def get_Mx_from_rhoy_by_rhox(self, rhoy_by_rhox):
        """Returns Mx from rhoy/rhox equal to rhoy_by_rhox"""
        if rhoy_by_rhox <=1.0 or rhoy_by_rhox >= (self.g+1)/(self.g-1):
            raise Exception("rhoy/rhox should be between 1 and ",(self.g+1)/(self.g-1))
        else:
            return sqrt(2*rhoy_by_rhox/(self.g+1- rhoy_by_rhox*(self.g-1)))

    def get_My_from_rhoy_by_rhox(self, rhoy_by_rhox):
        """Returns My from rhoy/rhox equal to rhoy_by_rhox"""
        if rhoy_by_rhox <=1.0 or rhoy_by_rhox >= (self.g+1)/(self.g-1):
            raise Exception("rhoy/rhox should be between 1 and ",(self.g+1)/(self.g-1))
        else:
            Mx = self.get_Mx_from_rhoy_by_rhox(rhoy_by_rhox)
            return self.get_My_from_Mx(Mx)

    def get_Ty_by_Tx_from_rhoy_by_rhox(self, rhoy_by_rhox):
        """Returns Ty/Tx from rhoy/rhox equal to rhoy_by_rhox"""
        if rhoy_by_rhox <=1.0 or rhoy_by_rhox >= (self.g+1)/(self.g-1):
            raise Exception("rhoy/rhox should be between 1 and ",(self.g+1)/(self.g-1))
        else:
            Mx = self.get_Mx_from_rhoy_by_rhox(rhoy_by_rhox)
            return self.get_Ty_by_Tx_from_Mx(Mx)

    def get_Poy_by_Pox_from_rhoy_by_rhox(self, rhoy_by_rhox):
        """Returns poy/pox from rhoy/rhox equal to rhoy_by_rhox"""
        if rhoy_by_rhox <=1.0 or rhoy_by_rhox >= (self.g+1)/(self.g-1):
            raise Exception("rhoy/rhox should be between 1 and ",(self.g+1)/(self.g-1))
        else:
            Mx = self.get_Mx_from_rhoy_by_rhox(rhoy_by_rhox)
            return self.get_Poy_by_Pox_from_Mx(Mx)

    def get_Poy_by_Px_from_rhoy_by_rhox(self, rhoy_by_rhox):
        """Returns poy/px from rhoy/rhox equal to rhoy_by_rhox"""
        if rhoy_by_rhox <=1.0 or rhoy_by_rhox >= (self.g+1)/(self.g-1):
            raise Exception("rhoy/rhox should be between 1 and ",(self.g+1)/(self.g-1))
        else:
            Mx = self.get_Mx_from_rhoy_by_rhox(rhoy_by_rhox)
            return self.get_Poy_by_Px_from_Mx(Mx)


    def get_Mx_from_Poy_by_Pox(self, Poy_by_Pox):
        """Returns Mx from poy/pox equal to Poy_by_Pox"""
        if Poy_by_Pox <= 0 or Poy_by_Pox >= 1:
            raise Exception("Poy/Pox must be between 0 and 1")
        else:
            return optimize.fsolve(lambda Mx: self.get_Poy_by_Pox_from_Mx(Mx) - Poy_by_Pox, 2.0)

    def get_My_from_Poy_by_Pox(self, Poy_by_Pox):
        """Returns My from poy/pox equal to Poy_by_Pox"""
        if Poy_by_Pox <= 0 or Poy_by_Pox >= 1:
            raise Exception("Poy/Pox must be between 0 and 1")
        else:
            Mx = self.get_Mx_from_Poy_by_Pox(Poy_by_Pox)
            return self.get_My_from_Mx(Mx)

    def get_Ty_by_Tx_from_Poy_by_Pox(self, Poy_by_Pox):
        """Returns Ty/Tx from poy/pox equal to Poy_by_Pox"""
        if Poy_by_Pox <= 0 or Poy_by_Pox >= 1:
            raise Exception("Poy/Pox must be between 0 and 1")
        else:
            Mx = self.get_Mx_from_Poy_by_Pox(Poy_by_Pox)
            return self.get_Ty_by_Tx_from_Mx(Mx)

    def get_Py_by_Px_from_Poy_by_Pox(self, Poy_by_Pox):
        """Returns py/px from poy/pox equal to Poy_by_Pox"""
        if Poy_by_Pox <= 0 or Poy_by_Pox >= 1:
            raise Exception("Poy/Pox must be between 0 and 1")
        else:
            Mx = self.get_Mx_from_Poy_by_Pox(Poy_by_Pox)
            return self.get_Py_by_Px_from_Mx(Mx)

    def get_Poy_by_Px_from_Poy_by_Pox(self, Poy_by_Pox):
        """Returns poy/px from poy/pox equal to Poy_by_Pox"""
        if Poy_by_Pox <= 0 or Poy_by_Pox >= 1:
            raise Exception("Poy/Pox must be between 0 and 1")
        else:
            Mx = self.get_Mx_from_Poy_by_Pox(Poy_by_Pox)
            return self.get_Poy_by_Px_from_Mx(Mx)


    def get_Mx_from_Poy_by_Px(self, Poy_by_Px):
        """Returns Mx from poy/px equal to Poy_by_Px"""
        if Poy_by_Px <= ((self.g+1)*0.5)**(self.g/(self.g-1)):
            raise Exception("Poy/Px should be greater than",((self.g+1)*0.5)**(self.g/(self.g-1)))
        else:
            return optimize.fsolve(lambda Mx: self.get_Poy_by_Px_from_Mx(Mx) - Poy_by_Px, 2.0)

    def get_My_from_Poy_by_Px(self, Poy_by_Px):
        """Returns My from poy/px equal to Poy_by_Px"""
        if Poy_by_Px <= ((self.g+1)*0.5)**(self.g/(self.g-1)):
            raise Exception("Poy/Px should be greater than",((self.g+1)*0.5)**(self.g/(self.g-1)))
        else:
            Mx = self.get_Mx_from_Poy_by_Px(Poy_by_Px)
            return self.get_My_from_Mx(Mx)

    def get_Ty_by_Tx_from_Poy_by_Px(self, Poy_by_Px):
        """Returns Mx from poy/px equal to Poy_by_Px"""
        if Poy_by_Px <= ((self.g+1)*0.5)**(self.g/(self.g-1)):
            raise Exception("Poy/Px should be greater than",((self.g+1)*0.5)**(self.g/(self.g-1)))
        else:
            Mx = self.get_Mx_from_Poy_by_Px(Poy_by_Px)
            return self.get_Ty_by_Tx_from_Mx(Mx)

    def get_Py_by_Px_from_Poy_by_Px(self, Poy_by_Px):
        """Returns py/px from poy/px equal to Poy_Px"""
        if Poy_by_Px <= ((self.g+1)*0.5)**(self.g/(self.g-1)):
            raise Exception("Poy/Px should be greater than",((self.g+1)*0.5)**(self.g/(self.g-1)))
        else:
            Mx = self.get_Mx_from_Poy_by_Px(Poy_by_Px)
            return self.get_Py_by_Px_from_Mx(Mx)

    def get_Poy_by_Pox_from_Poy_by_Px(self, Poy_by_Px):
        """Returns poy/pox from poy/px equal to Poy_by_Px """
        if Poy_by_Px <= ((self.g+1)*0.5)**(self.g/(self.g-1)):
            raise Exception("Poy/Px should be greater than",((self.g+1)*0.5)**(self.g/(self.g-1)))
        else:
            Mx = self.get_Mx_from_Poy_by_Px(Poy_by_Px)
            return self.get_Poy_by_Pox_from_Mx(Mx)

