/***********************************************************************************

    Copyright (C) 2007-2019 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_ENTRY_PARSER_HEADER
#define LIFEOGRAPH_ENTRY_PARSER_HEADER


#include <gtkmm.h>

#include "helpers.hpp"


namespace LIFEO
{

using namespace HELPERS;

typedef unsigned int CharClass;

static const CharClass
    CC_NOT_SET          = 0,
    CC_NOTHING          = 0x1,
    CC_NEWLINE          = 0x2,
    CC_ALPHA            = 0x3,
    CC_NUMBER           = 0x4,
    CC_PUNCT            = 0x5,
    CC_SPACE            = 0x8,          // space that will come eventually
    CC_TAB              = 0x10,

    CC_ASTERISK         = 0x40,         // bold
    CC_UNDERSCORE       = 0x41,         // italic
    CC_EQUALS           = 0x42,         // strikethrough
    CC_HASH             = 0x43,         // highlight

    CC_SLASH            = 0x800,
    CC_X                = 0x1010,
    CC_DASH             = 0x1014,
    CC_PLUS             = 0x1015,
    CC_TILDE            = 0x4040,
    CC_AT               = 0x4000,       // email

    CC_DOT              = 0x4200,
    CC_COLON            = 0x4300,

    CC_LESS             = 0x5000,       // tagging
    CC_MORE             = 0x5100,
    CC_SBB              = 0x5200,       // square bracket begin: comments
    CC_SBE              = 0x5300,       // square bracket end: comments

    // CLASSES (THESE WORK AS FLAGS)
    // every char must have one of these
    CF_ALPHA            = 0x10000,
    CF_NUMBER           = 0x20000,
    CF_PUNCTUATION      = 0x40000,
    CF_SPACE            = 0x80000,
    CF_NEWLINE          = 0x100000,

    // ADDITIONAL CLASSES
    CA_DATE_SEPARATOR   = 0x200000,
    CA_TODO_STATUS      = 0x400000,
    CA_MARKUP           = 0x800000,

    // SPECIAL CLASSES
    CS_MULTIPLE         = 0x20000000,   // char can occur multiple times
    CS_SPELLCHECK       = 0x80000000,

    // MULTIPLE CLASS COMBINATIONS (Do not use in recipes)
    CM_BLANK            = CF_SPACE|CF_NEWLINE,
    CM_NONSPACE         = CF_ALPHA|CF_NUMBER|CF_PUNCTUATION,
    CM_ANY_BUT_NEWLINE  = ( 0xFFFF0000 & ( ~CF_NEWLINE ) ),
    CM_CHAR_MASK        = 0xFFFF,
    CM_FLAG_MASK        = ( 0xFFFF0000 & ( ~CS_MULTIPLE ) );


class EntryParser
{
    public:
        typedef void ( EntryParser::*FPtr_void )();

        struct AbsChar  // abstract char
        {
            AbsChar( CharClass c, FPtr_void a ) : flags( c ), applier( a ) {}
            AbsChar( CharClass c ) : flags( c ), applier( nullptr ) {}

            CharClass       flags;
            FPtr_void       applier;
        };

        class Recipe
        {
            public:
                typedef std::vector< AbsChar > Contents;
                typedef unsigned int Id;
                typedef unsigned int State;
                static const State
                        RS_NOT_SET = 0x1, RS_IN_PROGRESS = 0x2, RS_REJECTED = 0x4,
                        RS_ACCEPTED = 0x8, RS_BLOCK = 0x1000;

                // order is important: first CharClass has to be from a recipe as it...
                // ...determines the type of the comparison
                static bool                 cmp_chars( CharClass cc1, CharClass cc2 )
                {
                    return( ( !( cc1 & CM_CHAR_MASK ) && ( cc1 & cc2 ) ) ||     // as class
                            ( cc1 & CM_CHAR_MASK ) == ( cc2 & CM_CHAR_MASK ) ); // as concrete char
                }

                // for creating main recipes
                Recipe( Id id, EntryParser* parent, const Contents* c, Id b )
                : m_id( id ), m_parent( parent ), m_contents( c ),
                  m_blocks( b ) { }

                // for copying main recipes to active
                Recipe( const Recipe* oth )
                : m_id( oth->m_id ), m_parent( oth->m_parent ),
                  m_contents( oth->m_contents ), m_blocks( oth->m_blocks ),
                  m_index( oth->m_index ), m_pos_start( oth->m_pos_start ),
                  m_pos_middle( oth->m_pos_middle ), m_int_value( oth->m_int_value ),
                  m_state( oth->m_state ) { }

                // for 2nd part recipes
                Recipe( Id id, EntryParser* parent, const Contents* c, Id b,
                        Ustring::size_type ps, Ustring::size_type pm )
                : m_id( id ), m_parent( parent ), m_contents( c ),
                  m_blocks( b ), m_index( 0 ), m_pos_start( ps ), m_pos_middle( pm ),
                  m_int_value( m_parent->m_int_last ) { }

                bool                        starts_with();
                State                       process_char();

            //private:
                Id                          m_id{ 0 };
                EntryParser* const          m_parent;
                const Contents*             m_contents;
                Id                          m_blocks;
                unsigned int                m_index{ 0 };
                Ustring::size_type          m_pos_start{ 0 };
                Ustring::size_type          m_pos_middle{ 0 };  // when needed
                unsigned int                m_int_value{ 0 };   // when needed
                State                       m_state{ RS_NOT_SET };

        };
        typedef std::set< Recipe* > RecipeSet;
        typedef std::list< Recipe* > RecipeList;

                                    EntryParser();
        virtual                     ~EntryParser() {}

        void                        parse( Ustring::size_type,
                                           Ustring::size_type );

        void                        set_search_str( const Ustring& );

    protected:
        virtual Wchar               get_char_at( int ) = 0;

        void                        process_char();

        // JUNCTIONS & CHECKS
        void                        check_date();
        void                        junction_subheading();
        void                        junction_markup();
        void                        junction_markup2();
        void                        junction_date_dotym();   // dot between year and month
        void                        junction_date_dotmd();   // dot between month and day
        void                        junction_todo();
        void                        junction_colon();
        void                        junction_at();
        void                        junction_link();

        // HELPERS
        void                        handle_number();
        void                        set_start();
        void                        set_middle();
        void                        add_block();

        // APPLIERS (TO BE OVERRIDEN)
        virtual void                apply_heading( bool ) { }
        virtual void                apply_subheading() { }
        virtual void                apply_subsubheading() { }
        virtual void                apply_bold() { }
        virtual void                apply_italic() { }
        virtual void                apply_strikethrough() { }
        virtual void                apply_highlight() { }
        virtual void                apply_comment() { }
        virtual void                apply_ignore() { }
        virtual void                apply_date() { }
        virtual void                apply_link() { }
        virtual void                apply_link_hidden() { }
        virtual void                apply_check_unf() { }
        virtual void                apply_check_prg() { }
        virtual void                apply_check_fin() { }
        virtual void                apply_check_ccl() { }
        virtual void                apply_chart() { }

        virtual void                apply_indent() { }
        virtual void                apply_match() { }

        virtual void                check_word() { } // for spell-checking

        void                        reset( Ustring::size_type, Ustring::size_type );

        Ustring::size_type          m_pos_end{ 0 };     // position of last char (constant)
        Ustring::size_type          m_pos_curr{ 0 };
        Ustring::size_type          m_pos_blank{ 0 };
        int                         m_pos_search{ 0 };

        // from new to old: curr, last
        CharClass                   m_cf_curr{ CC_NOT_SET };
        CharClass                   m_cf_last{ CC_NOT_SET };

        Wchar                       m_char_curr{ 0 };
        Wchar                       m_char_last{ 0 };

        Ustring                     m_word_curr;    // last word consisting purely of letters

        Recipe*                     m_recipe_curr{ nullptr };
        unsigned int                m_word_count{ 0 };
        unsigned int                m_int_last{ 0 };
        Date                        m_date_last{ 0 };
        Recipe::Id                  m_link_type_last{ 0 };
        bool                        m_flag_check_word{ false };

        Ustring                     m_search_str;

        RecipeSet                   m_all_recipes;
        RecipeList                  m_active_recipes;
        Recipe::Id                  m_blocked_flags{ 0 };

        static const Recipe::Id RID_HEADING         = 0x1;
        static const Recipe::Id RID_SUBHEADING      = 0x2;
        static const Recipe::Id RID_MARKUP          = 0x4;
        static const Recipe::Id RID_BOLD            = 0x8;
        static const Recipe::Id RID_ITALIC          = 0x10;
        static const Recipe::Id RID_HIGHLIGHT       = 0x20;
        static const Recipe::Id RID_STRIKETHROUGH   = 0x40;
        static const Recipe::Id RID_MARKUP_B_END    = 0x80;
        static const Recipe::Id RID_MARKUP_I_END    = 0x100;
        static const Recipe::Id RID_MARKUP_H_END    = 0x200;
        static const Recipe::Id RID_MARKUP_S_END    = 0x400;
        static const Recipe::Id RID_COMMENT         = 0x800;
        static const Recipe::Id RID_IGNORE          = 0x1000;
        static const Recipe::Id RID_TODO            = 0x2000;
        static const Recipe::Id RID_DATE            = 0x4000;
        static const Recipe::Id RID_LINK            = 0x8000;
        static const Recipe::Id RID_LINK_END        = 0x10000;
        static const Recipe::Id RID_EMAIL_END       = 0x20000;
        static const Recipe::Id RID_COLON           = 0x40000;
        static const Recipe::Id RID_AT              = 0x80000;
        static const Recipe::Id RID_ALL             = 0xFFFFFFFF;

        static const Recipe::Id RID_URL             = 0x100000; // only in m_link_type_last
        static const Recipe::Id RID_ID              = 0x200000; // only in m_link_type_last
        static const Recipe::Id RID_CHART           = 0x400000;

        static const Recipe::Id RID_DONE            = 0x10000000;   // only in format map
        static const Recipe::Id RID_SUBSUBHEADING   = 0x20000000;   // only in format map
        static const Recipe::Id RID_MONOSPACE       = 0x40000000;   // only in format map
        static const Recipe::Id RID_IMAGE           = 0x80000000;   // only in format map

        static Recipe::Contents     m_rc_subheading;
        static Recipe::Contents     m_rc_markup;
        static Recipe::Contents     m_rc_markup_b_end;
        static Recipe::Contents     m_rc_markup_i_end;
        static Recipe::Contents     m_rc_markup_h_end;
        static Recipe::Contents     m_rc_markup_s_end;
        static Recipe::Contents     m_rc_comment;
        static Recipe::Contents     m_rc_ignore;
        static Recipe::Contents     m_rc_todo;
        static Recipe::Contents     m_rc_date;
        static Recipe::Contents     m_rc_link_file;
        static Recipe::Contents     m_rc_link_email;
        static Recipe::Contents     m_rc_link_id;
        static Recipe::Contents     m_rc_link_end;
        static Recipe::Contents     m_rc_email_end;
        static Recipe::Contents     m_rc_chart;
        static Recipe::Contents     m_rc_colon;
        static Recipe::Contents     m_rc_at;
        static Recipe::Contents     m_rc_indent;

    private:
        Ustring::size_type          i_search{ 0 };
        Ustring::size_type          i_search_end{ 0 };
};

}   // end of namespace LIFEO

#endif
