/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2017 Martin Brehm
                  2012-2017 Martin Thomas
                  2016-2017 Sascha Gehrke

    This file was written by Sascha Gehrke.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/


// This must always be the first include directive
#include "config.h"

#include "ionpair.h"
#include "luzar.h"
#include "math.h"

#include "globalvar.h"
#include "maintools.h"


static CxObArray g_IonPairObserv;


bool gatherIonPair()
{
	CIonPairObservation *o;

	try { o = new CIonPairObservation(); } catch(...) { o = NULL; }
	if (o == NULL) NewException((double)sizeof(CIonPairObservation),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	g_IonPairObserv.Add(o);

	return true;
}


bool initializeIonPair() 
{
        int i;

        for(i = 0; i < g_IonPairObserv.GetSize(); i++) 
	{
                ((CIonPairObservation *)g_IonPairObserv[i])->initialize();
        }
        return true;
}


bool processIonPair(CTimeStep* ts) 
{
        int i;
 
	for(i = 0; i < g_IonPairObserv.GetSize(); i++) 
	{
                ((CIonPairObservation *)g_IonPairObserv[i])->process(ts);
        }
        return true;
}


bool finalizeIonPair() 
{
        int i;

        for(i = 0; i < g_IonPairObserv.GetSize(); i++) 
	{
                ((CIonPairObservation *)g_IonPairObserv[i])->finalize();
        }
        return true;
}


CIonPairObservation::CIonPairObservation()
{
	int z;
	CxString buf, buf2;

	try { m_oaAnion = new CxObArray(); } catch(...) { m_oaAnion = NULL; }
        if (m_oaAnion == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_oaCation = new CxObArray(); } catch(...) { m_oaCation = NULL; }
        if (m_oaCation == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_oaAnAtoms = new CxObArray(); } catch(...) { m_oaAnAtoms = NULL; }
        if (m_oaAnAtoms == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { m_oaCatAtoms = new CxObArray(); } catch(...) { m_oaCatAtoms = NULL; }
        if (m_oaCatAtoms == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	if (g_oaMolecules.GetSize() > 1)
	{
       		buf.sprintf("\n    Which of the molecules is the cation (");
        	for (z=0;z<g_oaMolecules.GetSize();z++)
        	{
                	buf2.sprintf("%s=%d",((CMolecule*)g_oaMolecules[z])->m_sName,z+1);
                	buf.strcat(buf2);
                	if (z < g_oaMolecules.GetSize()-1)
                        	buf.strcat(", ");
        	}
        	buf.strcat(")? ");
        	m_iShowMol = AskRangeInteger_ND("%s",1,g_oaMolecules.GetSize(),(const char*)buf) - 1;
		mprintf(WHITE,"\n    %s is the cation.\n\n",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
		Initiate_Offsets(m_iShowMol, m_oaCation);
		Initiate_Offsets(m_iShowMol, m_oaCatAtoms);

                buf.sprintf("\n    Which of the molecules is the anion (");
                for (z=0;z<g_oaMolecules.GetSize();z++)
                {
                        buf2.sprintf("%s=%d",((CMolecule*)g_oaMolecules[z])->m_sName,z+1);
                        buf.strcat(buf2);
                        if (z < g_oaMolecules.GetSize()-1)
                                buf.strcat(", ");
                }
                buf.strcat(")? ");
                m_iShowMol2 = AskRangeInteger_ND("%s",1,g_oaMolecules.GetSize(),(const char*)buf) - 1;
		mprintf(WHITE,"\n    %s is the anion.\n\n",((CMolecule*)g_oaMolecules[m_iShowMol2])->m_sName);
		Initiate_Offsets(m_iShowMol2, m_oaAnion);
		Initiate_Offsets(m_iShowMol2, m_oaAnAtoms);
	} 
	else 
	{
		mprintf("\n    There is only one molecule, which is handled as cation AND as anion. \n");
		m_iShowMol = 0;		
		m_iShowMol2 = 0;		
	}

	while ( ((CxIntArray*)(((CxObArray*)m_oaCatAtoms)->GetAt(0)))->GetSize() == 0 )
	{
        	mprintf("    Which atom(s) from %s should be handled as cationic for the ion pairing (e.g. \"C1,O3-5,N\")? [#2] ",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
        	inpprintf("! Which atom(s) from %s should be handled as cationic for the ion pairing (e.g. \"C1,O3-5,N\")? [#2] \n",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
        	myget(&buf);
		if (buf.GetLength() == 0)
		{
			buf.sprintf("#2");
		}
        	checkAtomChoice((CMolecule*)g_oaMolecules[m_iShowMol],buf,((CxObArray*)m_oaCatAtoms)); 
	}

	while ( ((CxIntArray*)(((CxObArray*)m_oaAnAtoms)->GetAt(0)))->GetSize() == 0 )
	{
	        mprintf("    Which atom(s) from %s should be handled as anionic for the ion pairing (e.g. \"C1,O3-5,N\")? [#2] ",((CMolecule*)g_oaMolecules[m_iShowMol2])->m_sName);
        	inpprintf("! Which atom(s) from %s should be handled as anionic for the ion pairing (e.g. \"C1,O3-5,N\")? [#2] \n",((CMolecule*)g_oaMolecules[m_iShowMol2])->m_sName);
	        myget(&buf);
		if (buf.GetLength() == 0)
       		{ 
			buf.sprintf("#2");
		}
		checkAtomChoice((CMolecule*)g_oaMolecules[m_iShowMol2],buf,((CxObArray*)m_oaAnAtoms));
	}

	while ( ((CxIntArray*)(((CxObArray*)m_oaCation)->GetAt(0)))->GetSize() == 0 )
	{
        	mprintf("    Which atom(s) from %s should be considered for the diffusion correction (e.g. \"C1,O3-5,N\")? [#2] ",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
        	inpprintf("! Which atom(s) from %s should be considered for the diffusion correction (e.g. \"C1,O3-5,N\")? [#2] \n",((CMolecule*)g_oaMolecules[m_iShowMol])->m_sName);
        	myget(&buf);
		if (buf.GetLength() == 0)
		{
			buf.sprintf("#2");
		}
        	checkAtomChoice((CMolecule*)g_oaMolecules[m_iShowMol],buf,((CxObArray*)m_oaCation)); 
	}
	  
	while ( ((CxIntArray*)(((CxObArray*)m_oaAnion)->GetAt(0)))->GetSize() == 0 )
	{
	        mprintf("    Which atom(s) from %s should be considered for the diffusion correction (e.g. \"C1,O3-5,N\")? [#2] ",((CMolecule*)g_oaMolecules[m_iShowMol2])->m_sName);
        	inpprintf("! Which atom(s) from %s should be considered for the diffusion correction (e.g. \"C1,O3-5,N\")? [#2] \n",((CMolecule*)g_oaMolecules[m_iShowMol2])->m_sName);
	        myget(&buf);
		if (buf.GetLength() == 0)
       		{ 
			buf.sprintf("#2");
		}
		checkAtomChoice((CMolecule*)g_oaMolecules[m_iShowMol2],buf,((CxObArray*)m_oaAnion));
	}

	m_fDist = AskFloat("    Maximum distance between cation and anion in pm? [350] ",350.0);
}


CIonPairObservation::~CIonPairObservation()
{
}


void CIonPairObservation::initialize()
{
	int n, m;

       	try { m_oaLastNeigh = new CxObArray(); } catch(...) { m_oaLastNeigh = NULL; }
       	if (m_oaLastNeigh == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

       	try { m_laLastPair = new CxIntArray(); } catch(...) { m_laLastPair = NULL; }
       	if (m_laLastPair == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

       	try { m_oaNeigh = new CxObArray(); } catch(...) { m_oaNeigh = NULL; }
       	if (m_oaNeigh == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

       	try { m_oaPair = new CxObArray(); } catch(...) { m_oaPair = NULL; }
       	if (m_oaPair == NULL) NewException((double)sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

        if ( g_iMaxStep != -1 )
        {
                m_iDepth = g_iMaxStep * 3 / 4 / g_iStride;
        }
        else
        {
                m_iDepth = g_iTrajSteps * 3 / 4 / g_iStride;
        }
        m_iDepth = AskUnsignedInteger("    Enter the resolution (=depth) of the ACF (in time steps): [%d] ", m_iDepth, m_iDepth);
        if ( m_iDepth > g_iTrajSteps )
        {
                mprintf("    Correlation Depth is exceeding number of total timesteps. Truncating ....");
                m_iDepth = g_iTrajSteps;
        }

	m_iTrunc = m_iDepth;
	while ( m_iTrunc >= m_iDepth )
	{
		if ( m_iTrunc > m_iDepth )
		{
			mprintf("    Truncation must be smaller than correlation depth!\n");
		}
		m_iTrunc = AskUnsignedInteger("    Enter the truncation for the fitting procedure (in time steps): [%d] ", (int)m_iDepth/10, (int)m_iDepth/10);
	}
				
	for ( n=0; n < m_oaCation->GetSize(); n++)		// Walks over all Cations  ............... building Arrays
	{
		m_laLastPair->Add(-1);	

                CxIntArray* IncludeME3;

		try { IncludeME3 = new CxIntArray(); } catch(...) { IncludeME3 = NULL; }
		if (IncludeME3 == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		for ( m=0; m < m_oaAnion->GetSize();m++)	// Walks over all Anions
		{
		        CxIntArray* IncludeME;
                        CxIntArray* IncludeME2;

                        try { IncludeME = new CxIntArray(); } catch(...) { IncludeME = NULL; }
                        if (IncludeME == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

                        try { IncludeME2 = new CxIntArray(); } catch(...) { IncludeME2 = NULL; }
                        if (IncludeME2 == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

                        m_oaNeigh->Add(IncludeME);
                        m_oaPair->Add(IncludeME2);
		
			IncludeME3->Add(0);
		}

		m_oaLastNeigh->Add(IncludeME3);
	}

	g_fTimestepLength = AskFloat("    Enter the length of one trajectory time step in fs: [0.5] ",0.5f);
}


void CIonPairObservation::process(CTimeStep* ts)
{
	int Cur_Pair = -1;
	double Min_Pair;
	int n,m,k,i;
	CxDVector3 Vec;

	for ( n=0; n < m_oaCation->GetSize(); n++ ) // Walk over Cations
	{
		Min_Pair = 2 * g_fBoxX;

		for ( m=0; m < m_oaAnion->GetSize(); m++ ) // Walk over Anion
		{
                        if ( n == m )
                        {
                                if ( m_iShowMol == m_iShowMol2 ) // Cation and Anion are same molecule. We don't want to check this pair
                                {
                                        continue;
                                }
                        }
			
			for ( k=0; k < ((CxIntArray*)(*m_oaCation)[n])->GetSize(); k++ ) //Walk over all Atoms of the Cation
			{
				for ( i=0; i < ((CxIntArray*)(*m_oaAnion)[m])->GetSize(); i++ )	// Walk over all Atoms of the Anion
				{
					Vec = ts->m_vaCoords[((CxIntArray*)(*m_oaCation)[n])->GetAt(k)] - ts->m_vaCoords[((CxIntArray*)(*m_oaAnion)[m])->GetAt(i)];
					CheckBoundary((CxDVector3*)&Vec);	
					
					if ( Vec.GetLength() <= m_fDist ) 	// In cutoff range -> Neighbor
					{
						if ( ((CxIntArray*)(*m_oaLastNeigh)[n])->GetAt(m) == 0 ) 	// Was not a neighbor ... hmmmm ....
						{
							((CxIntArray*)(*m_oaNeigh)[n*m_oaAnion->GetSize()+m])->Add(g_iSteps);
							((CxIntArray*)(*m_oaLastNeigh)[n])->SetAt(m,1);
						}
					}
					if ( Vec.GetLength() > m_fDist ) 	// Out of cutoff range -> No Neighbor
					{
						if ( ((CxIntArray*)(*m_oaLastNeigh)[n])->GetAt(m) == 1 ) 	// Was a neighbor ... hmmmm ....
						{
							((CxIntArray*)(*m_oaNeigh)[n*m_oaAnion->GetSize()+m])->Add(g_iSteps);
							((CxIntArray*)(*m_oaLastNeigh)[n])->SetAt(m,0);
						}
					}
				}
			}	
			
			for ( k=0; k < ((CxIntArray*)(*m_oaCatAtoms)[n])->GetSize(); k++ ) //Walk over all Atoms of the Cation
			{
				for ( i=0; i < ((CxIntArray*)(*m_oaAnAtoms)[m])->GetSize(); i++ )	// Walk over all Atoms of the Anion
				{
					Vec = ts->m_vaCoords[((CxIntArray*)(*m_oaCatAtoms)[n])->GetAt(k)] - ts->m_vaCoords[((CxIntArray*)(*m_oaAnAtoms)[m])->GetAt(i)];
					CheckBoundary((CxDVector3*)&Vec);	
					
			                if ( Vec.GetLength() < Min_Pair )
                        		{
                                		Min_Pair = Vec.GetLength();
                                		Cur_Pair = m;
                        		}
				}
			}
		}		// Cur_XXX contains the current partner
		
		if ( Cur_Pair != (*m_laLastPair)[n] )
		{
			if ( (*m_laLastPair)[n] != -1 )
			{
//				((CxIntArray*)(*m_oaPair)[ ( n * m_oaAnion->GetSize() ) + (*m_laLastPair)[n] - 1 ])->Add(g_iSteps);
				((CxIntArray*)(*m_oaPair)[ ( n * m_oaAnion->GetSize() ) + (*m_laLastPair)[n] ])->Add(g_iSteps);
			}
//			((CxIntArray*)(*m_oaPair)[ ( n * m_oaAnion->GetSize() ) + Cur_Pair - 1 ])->Add(g_iSteps);
			((CxIntArray*)(*m_oaPair)[ ( n * m_oaAnion->GetSize() ) + Cur_Pair ])->Add(g_iSteps);
			m_laLastPair->SetAt(n,Cur_Pair);
		}
	}
}


void CIonPairObservation::finalize()	// reconstruct functions -> autocorrelate -> sum results -> normalize -> fit
{
	long n,i,interval;
	double PairCor, DiffCor;
	CxString buf,buf2;
	unsigned long t0,eta;

	CxDoubleArray* NeighFunction;	// Input
	CxDoubleArray* PairFunction;
	
	CxDoubleArray* temp1Function;	// Temp
	CxDoubleArray* temp2Function;

	CLuzarCorrelation* luzar;

	try { PairFunction = new CxDoubleArray(); } catch(...) { PairFunction = NULL; }
	if (PairFunction == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { NeighFunction = new CxDoubleArray(); } catch(...) { NeighFunction = NULL; }
	if (NeighFunction == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { PairDynamic = new CxDoubleArray(); } catch(...) { PairDynamic = NULL; }
	if (PairDynamic == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { DiffusionDynamic = new CxDoubleArray(); } catch(...) { DiffusionDynamic = NULL; }
	if (DiffusionDynamic == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { k_t = new CxDoubleArray(); } catch(...) { k_t = NULL; }
	if (k_t == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { a = new CxDoubleArray(); } catch(...) { a = NULL; }
	if (a == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { b = new CxDoubleArray(); } catch(...) { b = NULL; }
	if (b == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { temp1Function = new CxDoubleArray(); } catch(...) { temp1Function = NULL; }
	if (temp1Function == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	try { temp2Function = new CxDoubleArray(); } catch(...) { temp2Function = NULL; }
	if (temp2Function == NULL) NewException((double)sizeof(CxDoubleArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	for ( i=0; i < m_iDepth; i++ )
	{
		PairDynamic->Add(0.0);
		DiffusionDynamic->Add(0.0);
	}

	try { luzar = new CLuzarCorrelation(); } catch(...) { luzar = NULL; }
	if (luzar == NULL) NewException((double)sizeof(CLuzarCorrelation),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	NeighFunction->SetSize(g_iSteps / g_iStride);
	PairFunction->SetSize(g_iSteps / g_iStride);

	luzar->Init(g_iSteps / g_iStride,m_iDepth);		// Inititalize Luzar-Correlation

	mprintf("    Finishing %6i Functions ... \nFunc %6li ",m_oaNeigh->GetSize(),(long)0);
        g_iDotCounter = 0;
	t0 = (unsigned long)time(NULL);
	interval = 2;

	for ( n=0; n < m_oaNeigh->GetSize(); n++) 		// Walk over all pairs
	{
		if ( n % interval == 0 )
		{
			if ( g_iDotCounter >= 50 )
			{
				if ( interval == 2 && n > 100 )
				{
					interval = 20;
				}
				else if ( interval == 20 && n > 2000 )
				{
					interval = 200;
				}
			        g_iDotCounter = 0;
		                if ((time(NULL) - t0) > 5)
		                {
		                        eta = (unsigned long)(((double)time(NULL) - t0) / n * ((double)MAX((long)0,(long)m_oaNeigh->GetSize() - (long)n)));
		                        FormatTime(eta,&buf);
		                        mprintf(" ETA %s",(const char*)buf);
		                }
		                mprintf("\nFunc %6li ",n);
			}
			g_iDotCounter++;
			mprintf(".");
		}

		if ( ((CxIntArray*)m_oaNeigh->GetAt(n))->GetSize() > 0 )
		{
			RestoreFunction(((CxIntArray*)m_oaNeigh->GetAt(n)),((CxDoubleArray*)NeighFunction));	// Restore function of pair n as NeighFunction and BondFunction
			RestoreFunction(((CxIntArray*)m_oaPair->GetAt(n)),((CxDoubleArray*)PairFunction));	

			luzar->Correlate(PairFunction,NeighFunction,temp1Function,temp2Function);
			
	        	for ( i=0; i < m_iDepth; i++ )		// Sum results to Output-array
	        	{
	        	        (*PairDynamic)[i] += (*temp1Function)[i];
	        	        (*DiffusionDynamic)[i] += (*temp2Function)[i];
	        	}
		}
	}
	mprintf("\n\n");

	if ( (*PairDynamic)[0] == 0 || (*DiffusionDynamic)[0] == 0 )
	{
		eprintf("First correlation value is zero ... something is wrong ...\n");
		abort();
	}

	PairCor = (double)(*PairDynamic)[0];
	DiffCor = (double)(*DiffusionDynamic)[0];

	for ( i=0; i < m_iDepth; i++ )			// Normalize
	{
		(*PairDynamic)[i] /= PairCor;
		(*DiffusionDynamic)[i] /= DiffCor;
		(*DiffusionDynamic)[i] -= (*PairDynamic)[i];
	}

	luzar->Fit(PairDynamic, DiffusionDynamic, k_t, a, b, m_iTrunc);

	WriteOutput();
}


void CIonPairObservation::RestoreFunction(CxIntArray* input, CxDoubleArray* output)
{
	unsigned int n;
	bool eins = false;

	for ( n=g_iStride; n <= g_iSteps; n+=g_iStride )		// Walk over all processed steps
	{
		if ( input->Contains(n) )
		{
			eins = !eins;
		}
		if (!eins)
		{
			output->SetAt(n,0.0);
		}
		else
		{
			output->SetAt(n,1.0);
		}
	}
}


bool CIonPairObservation::includeAtom(int atom, int number, int mol, CxObArray* target)  // atom: element number, number: atomsort number, mol m_iShowMol, target: in which array to include
{
	int n;

	for ( n=0; n < ((CMolecule*)g_oaMolecules[mol])->m_laSingleMolIndex.GetSize(); n++)
	{
		((CxIntArray*)target->GetAt(n))->Add(((CxIntArray*)(((CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[mol])->m_laSingleMolIndex[n]])->m_oaAtomOffset[atom]))->GetAt(number));
	}

	return true;
}


bool CIonPairObservation::includeALL(int mol, CxObArray* target)  // atom: element number, number: atomsort number, mol m_iShowMol, target: in which array to include
{
	CxIntArray* includeME;
	CxIntArray temparray;
	int m,i;
        try { includeME = new CxIntArray(); } catch(...) { includeME = NULL; }
        if (includeME == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
mprintf(RED,"CIonPairObservation::includeALL is not working at the moment!\nAborting ...\n");
abort();
	for ( m = 0; m < (((CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[mol])->m_laSingleMolIndex[0]])->m_oaAtomOffset).GetSize()-1; m++ ) // Walks over all atomtypes in Molecule 
	{
		temparray.CopyFrom((CxIntArray*)(((CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[mol])->m_laSingleMolIndex[0]])->m_oaAtomOffset[m]));
		for ( i = 0; i < temparray.GetSize(); i++ ) // Walks over all atoms of type
		{
			includeAtom(m,i,mol,target);
		}
	}

	return true;
}


bool CIonPairObservation::checkAtomChoice(CMolecule *mol, const char *s, CxObArray* m_oaInclude)
{
	const char *p, *q;
        char buf[32];
	const char *allowed = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890,-#_ ";
	int inclu, la = -1, i = -1, i2 = -1, atom = -1;	
	bool m = false, fin = false;

	p = s;

        while (*p != 0)
        {
		fin = false;
                while (*p == ' ')
                        p++;
                if (strchr(allowed,*p) == NULL)
                {
                        eprintf("Error: Character \"%c\" not allowed.\n",*p);
                        return false;
                }
                q = p;
                if (isalpha(*q) || (*q == '#'))
                {
                        if (m)
                        {
                                eprintf("Error: Only digit allowed after \"-\".\n");
                                return false;
                        }
                        while (isalpha(*q) || (*q == '#'))
                                q++;
                        if (q-p >= 32)
                        {
                                eprintf("Internal Error (%ld >= 32).\n",q-p);
                                return false;
                        }
                        memcpy(buf,p,q-p);
                        buf[q-p] = 0;
                        atom = mol->FindAtomInMol(buf); 	// atom holds number of element
                        if (atom == -1)
                        {
                                eprintf("Error: Atom \"%s\" not in the molecule.\n",buf);
                                return false;
                        } 
                } 
		else if (isdigit(*q))
                {
                        if (atom == -1)
                                atom = la;
			if (atom == -1)
                        {
                                eprintf("Error: Number in the beginning not possible.\n");
                                return false;
                        }
                        while (isdigit(*q)) 
				q++;
                        if ((*q != '-') && (*q != ',') && (*q != ' ') && (*q != 0))
                        {
                                eprintf("Error: Only \",\" or \"-\" may follow after a number.\n");
                                return false;
                        }
                        if (q-p >= 32)
                        {
                                eprintf("Internal Error (%ld >= 32).\n",q-p);
                                return false;
                        }
                        memcpy(buf,p,q-p);
                        buf[q-p] = 0;
                        if (atoi(buf)-1 >= mol->m_waAtomCount[atom])
                        {
                                eprintf("Error: Only %d %s atoms in the molecule (requested: %d)\n",mol->m_waAtomCount[atom],(const char*)((CAtom*)g_oaAtoms[mol->m_baAtomIndex[atom]])->m_sName,atoi(buf));
                                return false;
                        }
                        if (m)	// Range: Include all atoms from i to i2              	case: X1-3	A
                        {
                                i2 = atoi(buf)-1;
                                if (i2 < i) 	// If range from high to low -> wrong input
                                {
                                        eprintf("Error: Invalid atom range, %d < %d.\n",i2+1,i+1);
                                        return false;
                                }
				else
				{
					for (inclu = i; inclu <= i2; inclu++)
					{
						includeAtom(atom, inclu, mol->m_iIndex, m_oaInclude);	 //hier
					}
					fin = true;
				}
			} 
			else 	
			{
				i = atoi(buf)-1;
			}
                } 
		else if (*q == '-')
                {
                        if (i == -1)
                        {
                                eprintf("Error: \"-\" without preceding number.\n");
                                return false;
                        } 
                        m = true;
                        q++;
                } 
		else if (*q == ',')   
                {
                        if (atom == -1)
                        {
                                eprintf("Error: Comma without atom.\n");
                                return false;
                        }
			if ( i == -1 )  // include all atoms of kind         		case: X,	C
			{
				for (inclu = 0; inclu < mol->m_waAtomCount[atom]; inclu++ )
				{
					includeAtom(atom, inclu, mol->m_iIndex, m_oaInclude);	 //hier
				}
			}
			else if (!m)  // include single atom                    		case: X1,	B
			{
				includeAtom(atom, i, mol->m_iIndex, m_oaInclude);	// hier
			}
                        la = atom;
                        m = false;
                        i = -1;
                        i2 = -1;
                        atom = -1;
                        q++;
			fin = true;
                }
		p = q;
	}
	if ( !fin )
	{
		if ( i == -1 )	// case: X
	{
		for (inclu = 0; inclu < mol->m_waAtomCount[atom]; inclu++ )
		{
			includeAtom(atom, inclu, mol->m_iIndex, m_oaInclude);	 //hier
		}
	}
		else 			// case: X1
		{
			includeAtom(atom, i, mol->m_iIndex, m_oaInclude); // hier
		}
	}

	return true;
}

void CIonPairObservation::WriteOutput()
{
	CGrace* gc;
	int n,m;
	char buf[256];

	mprintf("    Writing \"Correlations.agr\"...\n");
	
	try { gc = new CGrace(); } catch(...) { gc = NULL; }
	if (gc == NULL) NewException((double)sizeof(CGrace),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	gc->SetRangeX(0,m_iDepth*g_fTimestepLength/1000*g_iStride);
	gc->SetRangeY(0,1);
	gc->MakeTicks();
	gc->SetLabelX("tau [ps]");
	gc->SetLabelY("f(tau)");
	gc->CurrentGraph()->m_bInvertXAxis = false;
	gc->CurrentGraph()->m_bLegend = true;

	gc->AddDataset();
	gc->SetDatasetName(0, "c(t)");
	gc->AddDataset();
	gc->SetDatasetName(1, "n(t)");
	for ( n = 0; n < PairDynamic->GetSize(); n++)
	{
		gc->AddXYTupel(0,n*g_fTimestepLength/1000*g_iStride,PairDynamic->GetAt(n));
		gc->AddXYTupel(1,n*g_fTimestepLength/1000*g_iStride,DiffusionDynamic->GetAt(n));
	}

	gc->WriteAgr("Correlations.agr",false);
	gc->WriteCSV("Correlations.csv");
	delete gc;

	mprintf("    Writing \"Fitted.agr\"...\n");

	CalcMinMax();

	try { gc = new CGrace(); } catch(...) { gc = NULL; }
	if (gc == NULL) NewException((double)sizeof(CGrace),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	gc->SetRangeX(0,m_dGraceMax);
	gc->SetRangeY(0,m_dGraceMax);
	gc->MakeTicks();
	gc->SetLabelX("kf c(t) - kd n(t) [1/ps]");
	gc->SetLabelY("-dc/dt [1/ps]");
	gc->CurrentGraph()->m_bInvertXAxis = false;
	gc->CurrentGraph()->m_bLegend = true;
		
	for ( n=0; n<6; n++ )
	{
		gc->AddDataset();
		sprintf(buf,"Truncation: %d",(int)(m_iTrunc - (n* (m_iTrunc/5))));		
        	gc->SetDatasetName(n, buf);

 		for ( m=0; m < k_t->GetSize(); m++)
		{
			gc->AddXYTupel(n,(*a)[n] * (*PairDynamic)[m] - (*b)[n] * (*DiffusionDynamic)[m], -(*k_t)[m]);
		}
	}

	gc->WriteCSV("Fitted.csv");
        gc->AddDataset();
        gc->SetDatasetName(6, "Unity");
	gc->AddXYTupel(6,m_dGraceMin,m_dGraceMin);
	gc->AddXYTupel(6,m_dGraceMax,m_dGraceMax);
        gc->WriteAgr("Fitted.agr",false);
        delete gc;
}

void CIonPairObservation::CalcMinMax()
{
	double temp;
	int n,m;	

	m_dGraceMin =  1E20;
	m_dGraceMax = -1E20;

	for ( m=0; m<6; m++ )
	{
		for ( n = m_iTrunc; n < PairDynamic->GetSize(); n++)
		{
			temp = ((*a)[m] * (*PairDynamic)[n] - (*b)[m] * (*DiffusionDynamic)[n]);
			if ( m_dGraceMin >= temp )
			{
				m_dGraceMin = temp;
			}
			if ( m_dGraceMax <= temp )
        	        {
        	                m_dGraceMax = temp;
        	        }                
		}
	}
	for ( n = m_iTrunc; n < k_t->GetSize(); n++)
	{
		temp = -k_t->GetAt(n);
                if ( m_dGraceMin >= temp )
                {
                        m_dGraceMin = temp;
                }
                if ( m_dGraceMax <= temp )
                {
                        m_dGraceMax = temp;
                }
	} 
}


void CIonPairObservation::CheckBoundary(CxDVector3* m_vChk)
{
	double RefX, RefY, RefZ;

	RefX = g_fBoxX / 2;
	RefY = g_fBoxY / 2;
	RefZ = g_fBoxZ / 2;

	if ( m_vChk->GetAt(0) < -RefX )
	{
		m_vChk->GetAt(0) += g_fBoxX;
 	}
	else if ( m_vChk->GetAt(0) > RefX )
	{
		m_vChk->GetAt(0) -= g_fBoxX;
 	}
 
        if ( m_vChk->GetAt(1) < -RefY )
        {
                m_vChk->GetAt(1) += g_fBoxY;
        }
        else if ( m_vChk->GetAt(1) > RefX )
        {
                m_vChk->GetAt(1) -= g_fBoxY;
        }
 
        if ( m_vChk->GetAt(2) < -RefZ )
        {
                m_vChk->GetAt(2) += g_fBoxZ;
        }
        else if ( m_vChk->GetAt(2) > RefZ )
        {
                m_vChk->GetAt(2) -= g_fBoxZ;
        }
}

void CIonPairObservation::Initiate_Offsets(int mol, CxObArray* array)
{
	int n;

	for ( n=0; n<((CMolecule*)g_oaMolecules[mol])->m_laSingleMolIndex.GetSize(); n++ )
	{
		CxIntArray* temp;

		try { temp = new CxIntArray(); } catch(...) { temp = NULL; }
        	if (temp == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		array->Add(temp);
	}
}
